/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include <gnome.h>
#include <gtk/gtkadjustment.h>
#include <gtk/gtkhscrollbar.h>
#include <gtk/gtkvscrollbar.h>
#include <libgnomevfs/gnome-vfs-types.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include "image-viewer.h"
#include "cursors.h"
#include "pixbuf-utils.h"

#define COLOR_GRAY_00   0x00000000
#define COLOR_GRAY_33   0x00333333
#define COLOR_GRAY_66   0x00666666
#define COLOR_GRAY_99   0x00999999
#define COLOR_GRAY_CC   0x00cccccc
#define COLOR_GRAY_FF   0x00ffffff

#define DRAG_THRESHOLD  1     /* When dragging the image ignores movements
			       * smaller than this value. */
 
#define MINIMUM_DELAY   10    /* When an animation frame has a 0 milli seconds
			       * delay use this delay instead. */

#define STEP_INCREMENT  20.0  /* Scroll increment. */

#define FLOAT_EQUAL(a,b) (fabs (a - b) < 1e-6)

enum {
	CLICKED,
	IMAGE_LOADED,
	ZOOM_CHANGED,
	LAST_SIGNAL
};

static void image_viewer_class_init           (ImageViewerClass *class);

static void image_viewer_init                 (ImageViewer *viewer);

static gint image_viewer_expose               (GtkWidget *widget, 
					       GdkEventExpose *event,
					       ImageViewer *viewer);

static gint image_viewer_button_press         (GtkWidget *widget, 
					       GdkEventButton *event,
					       ImageViewer *viewer);

static gint image_viewer_button_release       (GtkWidget *widget, 
					       GdkEventButton *event,
					       ImageViewer *viewer);

static gint image_viewer_motion_notify        (GtkWidget *widget, 
					       GdkEventMotion *event,
					       ImageViewer *viewer);

static void image_viewer_destroy              (GtkObject *viewer);

static void image_viewer_realize              (GtkWidget *widget);

static void image_viewer_size_allocate        (GtkWidget *widget, 
					       GtkAllocation *allocation);

static void get_zoomed_size                   (ImageViewer *viewer,
					       gint * width,
					       gint * height,
					       gdouble zoom_level);

static void scroll_to                         (ImageViewer *viewer,
					       gint x_offset,
					       gint y_offset);

static GdkPixbuf* get_current_pixbuf          (ImageViewer *viewer);

static void set_scroll_adjustments            (GtkWidget *widget,
                                               GtkAdjustment *hadj, 
					       GtkAdjustment *vadj);

static GtkHBoxClass *parent_class = NULL;
static guint image_viewer_signals[LAST_SIGNAL] = { 0 };


static void
image_viewer_class_init (ImageViewerClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = gtk_type_class (gtk_hbox_get_type ());
	widget_class = (GtkWidgetClass *) class;
	object_class = (GtkObjectClass*) class;

	image_viewer_signals[CLICKED] =
                gtk_signal_new ("clicked",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ImageViewerClass, clicked),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE, 0);
	image_viewer_signals[IMAGE_LOADED] =
                gtk_signal_new ("image_loaded",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ImageViewerClass, image_loaded),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE, 0);
	image_viewer_signals[ZOOM_CHANGED] =
                gtk_signal_new ("zoom_changed",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ImageViewerClass, zoom_changed),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE, 0);
	class->set_scroll_adjustments = set_scroll_adjustments;
        widget_class->set_scroll_adjustments_signal =
                gtk_signal_new ("set_scroll_adjustments",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ImageViewerClass, set_scroll_adjustments),
                                gtk_marshal_NONE__POINTER_POINTER,
                                GTK_TYPE_NONE, 2,
                                GTK_TYPE_ADJUSTMENT,
                                GTK_TYPE_ADJUSTMENT);

	gtk_object_class_add_signals (object_class, image_viewer_signals, 
                                      LAST_SIGNAL);

	object_class->destroy = image_viewer_destroy;

	widget_class->realize = image_viewer_realize;
	widget_class->size_allocate = image_viewer_size_allocate;

	class->clicked = NULL;
	class->image_loaded = NULL;
	class->zoom_changed = NULL;
}


static void image_loaded (ImageLoader* il, gpointer data);
static void image_error (ImageLoader *il, gpointer data);


static void 
hadj_value_changed (GtkObject *adj, 
		    ImageViewer *viewer)
{
	scroll_to (viewer, 
		   (gint) GTK_ADJUSTMENT (adj)->value, 
		   viewer->y_offset);
}


static void 
vadj_value_changed (GtkObject *adj, 
		    ImageViewer *viewer)
{
	scroll_to (viewer, 
		   viewer->x_offset,
		   (gint) GTK_ADJUSTMENT (adj)->value);
}


static void
image_viewer_init (ImageViewer *viewer)
{
/*	GTK_WIDGET_SET_FLAGS (viewer, GTK_NO_WINDOW);  FIXME */

	/* Initialize data. */

	viewer->check_type = CHECK_TYPE_MIDTONE;
	viewer->check_size = CHECK_SIZE_LARGE;
	viewer->check_color1 = COLOR_GRAY_66;
	viewer->check_color2 = COLOR_GRAY_99;

	viewer->is_animation = FALSE;
	viewer->running_animation = FALSE;
	viewer->rendering = FALSE;
	viewer->cursor_visible = TRUE;

	viewer->first_frame = NULL;
	viewer->current_frame = NULL;
	viewer->n_frames = 0;
	viewer->frame_pixbuf = NULL;
	viewer->anim_id = 0;

	viewer->loader = IMAGE_LOADER (image_loader_new (NULL, TRUE));
	gtk_signal_connect (GTK_OBJECT (viewer->loader), "done",
			    image_loaded,
			    viewer);
	gtk_signal_connect (GTK_OBJECT (viewer->loader), "error",
			    image_error,
			    viewer);

	viewer->zoom_level = 1.0;
	viewer->zoom_quality = ZOOM_QUALITY_HIGH;
	viewer->zoom_change = ZOOM_CHANGE_KEEP_PREV;
	viewer->zoom_fit = FALSE;
	viewer->doing_zoom_fit = FALSE;
	viewer->doing_change_frame = FALSE;

	viewer->is_void = TRUE;
	viewer->x_offset = 0;
	viewer->y_offset = 0;
	viewer->gdk_width = 0;
	viewer->gdk_height = 0;
	viewer->dragging = FALSE;

	viewer->black_bg = FALSE;

	viewer->area_pixbuf = NULL;
	viewer->area_max_width = 0;
	viewer->area_max_height = 0;
	viewer->area_bps = 0;
	viewer->area_color_space = GDK_COLORSPACE_RGB;

	viewer->cursor = NULL;
	viewer->cursor_void = NULL;

	/* Create the widget. */

	viewer->hadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 
							   0.0, 0.0, 0.0));
	viewer->vadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 
							   0.0, 0.0, 0.0));

	gtk_object_ref (GTK_OBJECT (viewer->hadj));
	gtk_object_sink (GTK_OBJECT (viewer->hadj));
	gtk_object_ref (GTK_OBJECT (viewer->vadj));
	gtk_object_sink (GTK_OBJECT (viewer->vadj));

	gtk_signal_connect (GTK_OBJECT (viewer->hadj), 
			    "value_changed",
			    GTK_SIGNAL_FUNC (hadj_value_changed), 
			    viewer);
	gtk_signal_connect (GTK_OBJECT (viewer->vadj), 
			    "value_changed",
			    GTK_SIGNAL_FUNC (vadj_value_changed), 
			    viewer);

	gtk_widget_push_visual (gdk_rgb_get_visual ());
        gtk_widget_push_colormap (gdk_rgb_get_cmap ());

	viewer->drawing_area = gtk_drawing_area_new ();

	gtk_widget_pop_visual ();
        gtk_widget_pop_colormap ();

	gtk_widget_set_events (viewer->drawing_area, 
			       GDK_EXPOSURE_MASK | 
			       GDK_BUTTON_PRESS_MASK |
			       GDK_BUTTON_RELEASE_MASK |
			       GDK_POINTER_MOTION_MASK |
			       GDK_POINTER_MOTION_HINT_MASK |
			       GDK_BUTTON_MOTION_MASK);

	gtk_signal_connect (GTK_OBJECT (viewer->drawing_area),
			    "expose_event",
                            GTK_SIGNAL_FUNC (image_viewer_expose), 
			    viewer);

	gtk_signal_connect (GTK_OBJECT (viewer->drawing_area),
			    "button_press_event",
                            GTK_SIGNAL_FUNC (image_viewer_button_press), 
			    viewer);
	gtk_signal_connect (GTK_OBJECT (viewer->drawing_area),
			    "button_release_event",
                            GTK_SIGNAL_FUNC (image_viewer_button_release), 
			    viewer);
	gtk_signal_connect (GTK_OBJECT (viewer->drawing_area),
			    "motion_notify_event",
                            GTK_SIGNAL_FUNC (image_viewer_motion_notify), 
			    viewer);

	gtk_widget_show (viewer->drawing_area);
	gtk_container_add (GTK_CONTAINER (viewer), viewer->drawing_area);
}


static void
paint (ImageViewer *viewer,
       gint src_x, 
       gint src_y,
       gint dest_x,
       gint dest_y,
       gint width,
       gint height,
       gint interp_type) 
{
	GdkPixbuf *pixbuf;
	double zoom_level;
	int bits_per_sample;
	GdkColorspace color_space;

	pixbuf = get_current_pixbuf (viewer);
	zoom_level = viewer->zoom_level;

	color_space = gdk_pixbuf_get_colorspace (pixbuf); 
	bits_per_sample = gdk_pixbuf_get_bits_per_sample (pixbuf);

	if ((viewer->area_pixbuf == NULL)
	    || (viewer->area_max_width < width) 
	    || (viewer->area_max_height < height)
	    || (viewer->area_bps != bits_per_sample)
	    || (viewer->area_color_space != color_space)) {
		if (viewer->area_pixbuf != NULL)
			gdk_pixbuf_unref (viewer->area_pixbuf);
		viewer->area_pixbuf = gdk_pixbuf_new (color_space,
						      FALSE, 
						      bits_per_sample,
						      width,
						      height);
		g_return_if_fail (viewer->area_pixbuf != NULL);

		viewer->area_max_width = width;
		viewer->area_max_height = height;
		viewer->area_color_space = color_space;
		viewer->area_bps = bits_per_sample;
	}

	if (gdk_pixbuf_get_has_alpha (pixbuf))   
		gdk_pixbuf_composite_color (pixbuf,
					    viewer->area_pixbuf,
					    0, 0,
					    width, height,
					    (double) -src_x,
					    (double) -src_y,
					    zoom_level, 
					    zoom_level,
					    interp_type,
					    255,
					    src_x, src_y,
					    viewer->check_size,
					    viewer->check_color1,
					    viewer->check_color2);
	else {
		gdk_pixbuf_scale (pixbuf,
				  viewer->area_pixbuf,
				  0, 0,
				  width, height,
				  (double) -src_x,
				  (double) -src_y,
				  zoom_level, 
				  zoom_level,
				  interp_type);
	}

	gdk_draw_rgb_image_dithalign (viewer->drawing_area->window,
				      viewer->drawing_area->style->black_gc,
				      dest_x, dest_y,
				      width, height,
				      GDK_RGB_DITHER_MAX,
				      gdk_pixbuf_get_pixels (viewer->area_pixbuf),
				      gdk_pixbuf_get_rowstride (viewer->area_pixbuf),
				      dest_x, dest_y);

#if 0
	gdk_draw_rectangle (viewer->drawing_area->window,
			    viewer->drawing_area->style->black_gc,
			    FALSE,
			    dest_x, dest_y,
			    width, height);
#endif

/*	viewer->rendering = FALSE; */
}


static gint change_frame_cb (gpointer data);
static gint get_frame_delay (GdkPixbufFrame *frame);

static gint
image_viewer_expose (GtkWidget *widget, 
		     GdkEventExpose *event,
		     ImageViewer *viewer)
{
	gint src_x, src_y;            /* Start to draw the image from this
					 point. */
	gint pixbuf_width;            /* Zoomed size of the image. */
	gint pixbuf_height;
	GdkRectangle image_area;      /* Inside this rectangle there is the
				       * visible part of the image. */
	GdkRectangle paint_area;      /* The intersection between image_area
				       * and event->area. */
	GtkWidget *drawing_area;

	if (viewer->rendering)
		return TRUE;

	drawing_area = viewer->drawing_area;
	viewer->rendering = TRUE;

	get_zoomed_size (viewer, &pixbuf_width, &pixbuf_height, 
			 viewer->zoom_level);

	src_x = viewer->x_offset;
	src_y = viewer->y_offset;

	image_area.x = MAX (0, (viewer->gdk_width - pixbuf_width) / 2);
	image_area.y = MAX (0, (viewer->gdk_height - pixbuf_height) / 2);
	image_area.width = MIN (pixbuf_width, viewer->gdk_width);
	image_area.height = MIN (pixbuf_height, viewer->gdk_height);

	/* Draw the background in four phases to avoid flickering. */
	if ((image_area.x != 0) 
	    || (image_area.y != 0) 
	    || (image_area.width < viewer->gdk_width)
	    || (image_area.height < viewer->gdk_height)) {
		GdkGC *gc;
		gint rx, ry, rw, rh;

		if (viewer->black_bg)
			gc = drawing_area->style->black_gc;
		else
			gc = drawing_area->style->bg_gc[GTK_STATE_NORMAL];

		/* Top rectangle. */
		rx = 0;
		ry = 0;
		rw = viewer->gdk_width;
		rh = image_area.y;
		gdk_draw_rectangle (drawing_area->window,
				    gc,
				    TRUE,
				    rx, ry,
				    rw, rh);

		/* Bottom rectangle. */
		rx = 0;
		ry = image_area.y + image_area.height;
		rw = viewer->gdk_width;
		rh = image_area.y + 1;
		gdk_draw_rectangle (drawing_area->window,
				    gc,
				    TRUE,
				    rx, ry,
				    rw, rh);

		/* Left rectangle. */
		rx = 0;
		ry = image_area.y;
		rw = image_area.x;
		rh = image_area.height;
		gdk_draw_rectangle (drawing_area->window,
				    gc,
				    TRUE,
				    rx, ry,
				    rw, rh);

		/* Right rectangle. */
		rx = image_area.x + image_area.width;
		ry = image_area.y;
		rw = image_area.x + 1;
		rh = image_area.height;
		gdk_draw_rectangle (drawing_area->window,
				    gc,
				    TRUE,
				    rx, ry,
				    rw, rh);
	}

	if ((get_current_pixbuf (viewer) != NULL) 
	    && gdk_rectangle_intersect (&event->area, 
					&image_area, 
					&paint_area)) {
		gint interp_type;

		if (viewer->zoom_quality == ZOOM_QUALITY_LOW)
			interp_type = GDK_INTERP_NEAREST;
		else
			interp_type = GDK_INTERP_BILINEAR;
/*			interp_type = GDK_INTERP_TILES;*/
/*			interp_type = GDK_INTERP_HYPER;*/

		if (FLOAT_EQUAL (viewer->zoom_level, 1.0))
			interp_type = GDK_INTERP_NEAREST;

		src_x += paint_area.x - image_area.x;
		src_y += paint_area.y - image_area.y;

		paint (viewer,
		       src_x, 
		       src_y,
		       paint_area.x, 
		       paint_area.y,
		       paint_area.width, 
		       paint_area.height,
		       interp_type);
	}

	viewer->rendering = FALSE;

	/* FIXME : where to put this ? */
	if (viewer->running_animation && (viewer->anim_id == 0))
		viewer->anim_id = gtk_timeout_add (
			get_frame_delay (viewer->current_frame->data), 
			change_frame_cb, 
			viewer);

	return TRUE;
}


static void
expose_area (ImageViewer *viewer,
	     gint x,
	     gint y,
	     gint width,
	     gint height)
{
	GdkEventExpose event;

	event.area.x = x;
	event.area.y = y;
	event.area.width = width;
	event.area.height = height;

	image_viewer_expose (viewer->drawing_area, &event, viewer);
}


void
image_viewer_scroll_to (ImageViewer *viewer,
			gint x_offset,
			gint y_offset)
{
	g_return_if_fail (viewer != NULL);

	if (get_current_pixbuf (viewer) == NULL)
		return;

	if (viewer->rendering) 
		return;

	scroll_to (viewer, x_offset, y_offset);

	gtk_signal_handler_block_by_data (GTK_OBJECT (viewer->hadj), viewer);
	gtk_signal_handler_block_by_data (GTK_OBJECT (viewer->vadj), viewer);
	gtk_adjustment_set_value (viewer->hadj, viewer->x_offset);
	gtk_adjustment_set_value (viewer->vadj, viewer->y_offset);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (viewer->hadj), viewer);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (viewer->vadj), viewer);
}


static void
scroll_to (ImageViewer *viewer,
	   gint x_offset,
	   gint y_offset)
{
	gint width, height;
	gint delta_x, delta_y;
	GdkEvent *event;
	GdkDrawable *drawable = viewer->drawing_area->window;
	gboolean rerun_animation;

	g_return_if_fail (viewer != NULL);

	if (get_current_pixbuf (viewer) == NULL)
		return;

	if (viewer->rendering) 
		return;

	get_zoomed_size (viewer, &width, &height, viewer->zoom_level);

	if (width > viewer->gdk_width) 
		x_offset = CLAMP (x_offset, 0, width - viewer->gdk_width);
	else
		x_offset = viewer->x_offset;

	if (height > viewer->gdk_height)
		y_offset = CLAMP (y_offset, 0, height - viewer->gdk_height);
	else
		y_offset = viewer->y_offset;

	if ((x_offset == viewer->x_offset) && (y_offset == viewer->y_offset))
		return;

	delta_x = x_offset - viewer->x_offset;
	delta_y = y_offset - viewer->y_offset;

	if ((delta_x != 0) || (delta_y != 0)) {
		GdkGC *gc = viewer->drawing_area->style->black_gc;
		gint src_x, dest_x;
		gint src_y, dest_y;

		if (delta_x < 0) {
			src_x = 0;
			dest_x = -delta_x;
		} else {
			src_x = delta_x;
			dest_x = 0;
		}

		if (delta_y < 0) {
			src_y = 0;
			dest_y = -delta_y;
		} else {
			src_y = delta_y;
			dest_y = 0;
		}

		gc = gdk_gc_new (drawable);
		gdk_gc_set_exposures (gc, TRUE);

		gdk_window_copy_area (drawable,
				      gc,
				      dest_x, dest_y,
				      drawable,
				      src_x, src_y,
				      viewer->gdk_width - abs (delta_x),
				      viewer->gdk_height - abs (delta_y));

		gdk_gc_destroy (gc);
	}

	viewer->x_offset = x_offset;
	viewer->y_offset = y_offset;

	expose_area (viewer, 
		     0, 
		     (delta_y < 0) ? 0 : viewer->gdk_height - abs (delta_y),
		     viewer->gdk_width,
		     abs (delta_y));

	expose_area (viewer, 
		     (delta_x < 0) ? 0 : viewer->gdk_width - abs (delta_x),
		     0,
		     abs (delta_x),
		     viewer->gdk_height);

	/* Process graphics exposures */

	rerun_animation = viewer->running_animation;
	viewer->running_animation = FALSE;
        while ((event = gdk_event_get_graphics_expose (drawable)) != NULL) {
		GdkEventExpose *expose = (GdkEventExpose*) event;

		expose_area (viewer, 
			     expose->area.x,
			     expose->area.y,
			     expose->area.width,
			     expose->area.height);

                if (expose->count == 0) {
                        gdk_event_free (event);
                        break;
                }
                gdk_event_free (event);
        }
	viewer->running_animation = rerun_animation;
}


static gint
image_viewer_button_press (GtkWidget *widget, 
			   GdkEventButton *event,
			   ImageViewer *viewer)
{
	if (viewer->dragging)
		return FALSE;

	if (event->button == 1) {
		GdkCursor *cursor;
		gint retval;

		cursor = cursor_get (widget->window, CURSOR_HAND_CLOSED);
		retval = gdk_pointer_grab (widget->window,
					   FALSE,
					   (GDK_POINTER_MOTION_MASK
					    | GDK_POINTER_MOTION_HINT_MASK
					    | GDK_BUTTON_RELEASE_MASK),
					   NULL,
					   cursor,
					   event->time);
		gdk_cursor_destroy (cursor);

		if (retval != 0)
			return FALSE;

		viewer->drag_realx = viewer->drag_x = event->x;
		viewer->drag_realy = viewer->drag_y = event->y;
		viewer->pressed = TRUE;
		return TRUE;
	}

	return FALSE;
}


static gint
image_viewer_button_release  (GtkWidget *widget, 
			      GdkEventButton *event,
			      ImageViewer *viewer)
{
	if (event->button != 1)
		return FALSE;

	if (! viewer->dragging)
		gtk_signal_emit (GTK_OBJECT (viewer), image_viewer_signals[CLICKED]);

	viewer->pressed = FALSE;
	viewer->dragging = FALSE;
	gdk_pointer_ungrab (event->time);
	
	return FALSE;
}


static gint
image_viewer_motion_notify (GtkWidget *widget, 
			    GdkEventMotion *event,
			    ImageViewer *viewer)
{
	gint x, y;
	GdkModifierType mods;

	if (! viewer->pressed)
		return FALSE;

	if (viewer->rendering)
		return FALSE;

	viewer->dragging = TRUE;

	if (event->is_hint)
                gdk_window_get_pointer (widget->window, &x, &y, &mods);
        else 
                return FALSE;

	viewer->drag_realx = x;
	viewer->drag_realy = y;

	if ((abs (viewer->drag_realx - viewer->drag_x) < DRAG_THRESHOLD)
	    && (abs (viewer->drag_realy - viewer->drag_y) < DRAG_THRESHOLD))
		return TRUE;

	x = viewer->x_offset + (viewer->drag_x - viewer->drag_realx);
	y = viewer->y_offset + (viewer->drag_y - viewer->drag_realy);

	scroll_to (viewer, x, y);

	gtk_signal_handler_block_by_data (GTK_OBJECT (viewer->hadj), viewer);
	gtk_signal_handler_block_by_data (GTK_OBJECT (viewer->vadj), viewer);
	gtk_adjustment_set_value (viewer->hadj, x);
	gtk_adjustment_set_value (viewer->vadj, y);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (viewer->hadj), viewer);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (viewer->vadj), viewer);

	viewer->drag_x = viewer->drag_realx;
	viewer->drag_y = viewer->drag_realy;

	return TRUE;
}


static GdkPixbuf *
create_pixbuf_from_frame (ImageViewer *viewer, GdkPixbufFrame *frame)
{
	GdkPixbuf *new_pixbuf, *pixbuf;
	GdkPixbufAnimation *anim;
	gint dest_width, dest_height;
	gint x_offset, y_offset;
	gint new_width, new_height;
	gint rowstride;
	guchar *pixels;
	
	anim = image_loader_get_animation (viewer->loader);
	g_assert (anim != NULL);

	pixbuf = gdk_pixbuf_frame_get_pixbuf (frame);
	gdk_pixbuf_ref (pixbuf);

	dest_width = gdk_pixbuf_get_width (pixbuf);
	dest_height = gdk_pixbuf_get_height (pixbuf);
	x_offset = gdk_pixbuf_frame_get_x_offset (frame);
	y_offset = gdk_pixbuf_frame_get_y_offset (frame);

	new_width = gdk_pixbuf_animation_get_width (anim);
	new_height = gdk_pixbuf_animation_get_height (anim);

	new_pixbuf = gdk_pixbuf_new (gdk_pixbuf_get_colorspace (pixbuf),
				     gdk_pixbuf_get_has_alpha (pixbuf),
				     gdk_pixbuf_get_bits_per_sample (pixbuf),
				     new_width,
				     new_height);

	/* Blank image. */
	rowstride = gdk_pixbuf_get_rowstride (new_pixbuf);
	pixels = gdk_pixbuf_get_pixels (new_pixbuf);
	bzero (pixels, rowstride * new_height);

	/* Copy the frame. */
	gdk_pixbuf_copy_area (pixbuf,
			      0, 0,
			      dest_width, dest_height,
			      new_pixbuf,
			      x_offset, y_offset);
	gdk_pixbuf_unref (pixbuf);

	return new_pixbuf;
}


static void
create_first_frame (ImageViewer *viewer)
{
	g_return_if_fail (viewer != NULL);

	if (viewer->frame_pixbuf)
		gdk_pixbuf_unref (viewer->frame_pixbuf);
	viewer->frame_pixbuf = NULL;

	viewer->frame_pixbuf = create_pixbuf_from_frame (viewer, viewer->first_frame->data);
}


static gint
change_frame_cb (gpointer data)
{
	ImageViewer *viewer = data;
	GdkPixbufFrameAction action;
	GdkPixbuf *pixbuf;
	gint dest_width, dest_height;
	gint x_offset, y_offset;

	if (viewer->anim_id != 0) {
		gtk_timeout_remove (viewer->anim_id);
		viewer->anim_id = 0;
	}

	viewer->current_frame = viewer->current_frame->next;

	if (viewer->current_frame == NULL) { /* Restart animation. */
		viewer->current_frame = viewer->first_frame;

		create_first_frame (viewer);
		viewer->doing_change_frame = TRUE;
		image_viewer_update_view (viewer);
		return FALSE;
	}

	action = gdk_pixbuf_frame_get_action (viewer->current_frame->data);
	switch (action) {
	case GDK_PIXBUF_FRAME_RETAIN:
		/*g_print ("retain (combine)\t\n"); */

		pixbuf = gdk_pixbuf_frame_get_pixbuf (viewer->current_frame->data);
		gdk_pixbuf_ref (pixbuf);

		dest_width = gdk_pixbuf_get_width (pixbuf);
		dest_height = gdk_pixbuf_get_height (pixbuf);
		x_offset = gdk_pixbuf_frame_get_x_offset (viewer->current_frame->data);
		y_offset = gdk_pixbuf_frame_get_y_offset (viewer->current_frame->data);

		gdk_pixbuf_composite (pixbuf,
				      viewer->frame_pixbuf,
				      x_offset, 
				      y_offset,
				      dest_width, 
				      dest_height,
				      (double) x_offset, 
				      (double) y_offset,
				      1.0, 1.0,
				      GDK_INTERP_NEAREST,
				      255);
		gdk_pixbuf_unref (pixbuf);
		break;

	case GDK_PIXBUF_FRAME_REVERT:
	case GDK_PIXBUF_FRAME_DISPOSE:
		/*g_print ("dispose (replace)\t\n"); */

		gdk_pixbuf_unref (viewer->frame_pixbuf);
		viewer->frame_pixbuf = create_pixbuf_from_frame (viewer, viewer->current_frame->data);
		break;
	}

	viewer->doing_change_frame = TRUE;
	image_viewer_update_view (viewer);

	return FALSE;
}


static void
image_viewer_destroy (GtkObject *object)
{
        ImageViewer* viewer;

        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_IMAGE_VIEWER (object));
  
        viewer = IMAGE_VIEWER (object);

	if (viewer->anim_id != 0) {
		gtk_timeout_remove (viewer->anim_id);
		viewer->anim_id = 0;
	}

	if (viewer->frame_pixbuf) {
		gdk_pixbuf_unref (viewer->frame_pixbuf);
		viewer->frame_pixbuf = NULL;
	}

	if (viewer->loader) {
		gtk_object_unref (GTK_OBJECT (viewer->loader));
		viewer->loader = NULL;
	}
		
	if (viewer->cursor) {
		gdk_cursor_destroy (viewer->cursor);
		viewer->cursor = NULL;
	}

	if (viewer->cursor_void) {
		gdk_cursor_destroy (viewer->cursor_void);
		viewer->cursor_void = NULL;
	}

	if (viewer->hadj) {
		gtk_signal_disconnect_by_data (GTK_OBJECT (viewer->hadj), 
					       viewer);
		gtk_object_unref (GTK_OBJECT (viewer->hadj));
		viewer->hadj = NULL;
	}
	if (viewer->vadj) {
		gtk_signal_disconnect_by_data (GTK_OBJECT (viewer->vadj), 
					       viewer);
		gtk_object_unref (GTK_OBJECT (viewer->vadj));
		viewer->vadj = NULL;
	}
	
	if (viewer->area_pixbuf != NULL)
		gdk_pixbuf_unref (viewer->area_pixbuf);
	
        /* Chain up */
	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void 
image_viewer_realize (GtkWidget *widget)
{
	ImageViewer *viewer;
	GdkWindow   *window;

	/* Chain up */
	(* GTK_WIDGET_CLASS (parent_class)->realize) (widget);

	viewer = IMAGE_VIEWER (widget);

	gtk_widget_realize (viewer->drawing_area);
	window = viewer->drawing_area->window;

	viewer->cursor = cursor_get (window, CURSOR_HAND_OPEN);
	viewer->cursor_void = cursor_get (window, CURSOR_VOID);
	gdk_window_set_cursor (window, viewer->cursor);
}


static GdkPixbuf*
get_current_pixbuf (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, NULL);

	if (viewer->is_void)
		return NULL;

	if (! viewer->is_animation)
		return image_loader_get_pixbuf (viewer->loader);

	return viewer->frame_pixbuf;
}


static void
zoom_to_fit (ImageViewer *viewer)
{
	GdkPixbuf *buf;
	double x_level, y_level;
	double new_zoom_level;		
	
	buf = get_current_pixbuf (viewer);
	
	x_level = (double) viewer->gdk_width / gdk_pixbuf_get_width (buf);
	y_level = (double) viewer->gdk_height / gdk_pixbuf_get_height (buf);
		
	new_zoom_level = (x_level < y_level) ? x_level : y_level;
	if (new_zoom_level > 0.0) {
		viewer->doing_zoom_fit = TRUE;
		image_viewer_set_zoom (viewer, new_zoom_level);
		viewer->doing_zoom_fit = FALSE;
	}
}


static void 
image_viewer_size_allocate (GtkWidget *widget, 
			    GtkAllocation  *allocation)
{
	ImageViewer * viewer;

	viewer = IMAGE_VIEWER (widget);
	viewer->gdk_width = allocation->width; 
	viewer->gdk_height = allocation->height;

	/* If zoom_fit is active update the zoom level. */

	if (! viewer->is_void 
	    && viewer->zoom_fit
	    && (get_current_pixbuf (viewer) != NULL)) 
		zoom_to_fit (viewer);

	/* If zoom_fit_if_larger is active update the zoom level. */

	if (! viewer->is_void 
	    && viewer->zoom_fit_if_larger
	    && (get_current_pixbuf (viewer) != NULL)) {
		GdkPixbuf *buf;
		
		buf = get_current_pixbuf (viewer);
		
		if ((viewer->gdk_width < gdk_pixbuf_get_width (buf))
		    || (viewer->gdk_height < gdk_pixbuf_get_height (buf))) 
			zoom_to_fit (viewer);
		else {
			viewer->doing_zoom_fit = TRUE;
			image_viewer_set_zoom (viewer, 1.0);
			viewer->doing_zoom_fit = FALSE;
		}
	}


	/* Check whether the offset is still valid. */

	if (get_current_pixbuf (viewer) != NULL) {
		gint width, height;
		
		get_zoomed_size (viewer, &width, &height, viewer->zoom_level);

		if (width > viewer->gdk_width)
			viewer->x_offset = CLAMP (viewer->x_offset, 
						  0, 
						  width - viewer->gdk_width);
		else
			viewer->x_offset = 0;

		if (height > viewer->gdk_height)
			viewer->y_offset = CLAMP (viewer->y_offset, 
						  0, 
						  height - viewer->gdk_height);
		else
			viewer->y_offset = 0;

		/* Change adjustment values. */

		viewer->hadj->lower = 0.0;
		viewer->hadj->upper = width;
		viewer->hadj->value = viewer->x_offset;
		viewer->hadj->step_increment = STEP_INCREMENT;
		viewer->hadj->page_increment = viewer->gdk_width / 2;
		viewer->hadj->page_size      = viewer->gdk_width;

		viewer->vadj->lower = 0.0;
		viewer->vadj->upper = height;
		viewer->vadj->value = viewer->y_offset;
		viewer->vadj->step_increment = STEP_INCREMENT;
		viewer->vadj->page_increment = viewer->gdk_height / 2;
		viewer->vadj->page_size      = viewer->gdk_height;
	} else {
		viewer->hadj->lower = 0.0;
		viewer->hadj->upper = 0.0;
		viewer->hadj->value = 0.0;

		viewer->vadj->lower = 0.0;
		viewer->vadj->upper = 0.0;
		viewer->vadj->value = 0.0;
	}

	gtk_signal_handler_block_by_data (GTK_OBJECT (viewer->hadj), viewer);
	gtk_signal_handler_block_by_data (GTK_OBJECT (viewer->vadj), viewer);
	gtk_adjustment_changed (viewer->hadj);
	gtk_adjustment_changed (viewer->vadj);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (viewer->hadj), viewer);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (viewer->vadj), viewer);

	/* Chain up. */

	GTK_WIDGET_CLASS (parent_class)->size_allocate (widget, allocation);
}





GtkType
image_viewer_get_type ()
{
	static guint image_viewer_type = 0;

	if (! image_viewer_type) {
		GtkTypeInfo image_viewer_info =	{
			"ImageViewer",
			sizeof (ImageViewer),
			sizeof (ImageViewerClass),
			(GtkClassInitFunc) image_viewer_class_init,
			(GtkObjectInitFunc) image_viewer_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		image_viewer_type = gtk_type_unique (gtk_hbox_get_type (), 
						     &image_viewer_info);
	}
	
	return image_viewer_type;
}


GtkWidget*     
image_viewer_new (void)
{
	return GTK_WIDGET (gtk_type_new (image_viewer_get_type()));
}


void
image_viewer_load_image (ImageViewer *viewer, 
			 const gchar *path)
{
	g_return_if_fail (viewer != NULL);
	g_return_if_fail (path != NULL);

	viewer->is_void = FALSE;

	image_viewer_stop_animation (viewer);
	image_loader_stop (viewer->loader);

	image_loader_set_path (viewer->loader, path);
	image_loader_start (viewer->loader);
}


void
image_viewer_load_from_pixbuf_loader (ImageViewer *viewer, 
				      GdkPixbufLoader *pixbuf_loader)
{
	g_return_if_fail (viewer != NULL);
	g_return_if_fail (pixbuf_loader != NULL);

	viewer->is_void = FALSE;

	image_viewer_stop_animation (viewer);
	image_loader_stop (viewer->loader);

	image_loader_load_from_pixbuf_loader (viewer->loader, pixbuf_loader);
}


void
image_viewer_set_void (ImageViewer *viewer)
{
	g_return_if_fail (viewer != NULL);

	viewer->is_void = TRUE;

	image_viewer_stop_animation (viewer);

	if (viewer->frame_pixbuf) {
		gdk_pixbuf_unref (viewer->frame_pixbuf);
		viewer->frame_pixbuf = NULL;
	}

	viewer->x_offset = 0;
	viewer->y_offset = 0;

	gtk_widget_queue_resize (GTK_WIDGET (viewer));
}


gboolean
image_viewer_is_void (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, TRUE);
	return viewer->is_void;
}


void
image_viewer_update_view (ImageViewer *viewer)
{
	g_return_if_fail (viewer != NULL);

	if (viewer->zoom_fit)
		image_viewer_zoom_to_fit (viewer);
	else if (viewer->zoom_fit_if_larger)
		image_viewer_zoom_to_fit_if_larger (viewer);
	else
		image_viewer_set_zoom (viewer, viewer->zoom_level);
}


gchar*
image_viewer_get_image_filename (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, NULL);
	return image_loader_get_path (viewer->loader);
}


gint
image_viewer_get_image_width (ImageViewer *viewer)
{
	GdkPixbufAnimation *anim;
	GdkPixbuf *pixbuf;

	g_return_val_if_fail (viewer != NULL, 0);

	anim = image_loader_get_animation (viewer->loader);
	if (anim != NULL) 
		return gdk_pixbuf_animation_get_width (anim);
	
	pixbuf = image_loader_get_pixbuf (viewer->loader);

	if (pixbuf != NULL)
		return gdk_pixbuf_get_width (pixbuf);

	return 0;
}


gint
image_viewer_get_image_height (ImageViewer *viewer)
{
	GdkPixbufAnimation *anim;
	GdkPixbuf *pixbuf;

	g_return_val_if_fail (viewer != NULL, 0);

	anim = image_loader_get_animation (viewer->loader);
	if (anim != NULL) 
		return gdk_pixbuf_animation_get_height (anim);
	
	pixbuf = image_loader_get_pixbuf (viewer->loader);
	if (pixbuf != NULL)
		return gdk_pixbuf_get_height (pixbuf);

	return 0;
}


gint
image_viewer_get_image_bps (ImageViewer *viewer)
{
	GdkPixbufAnimation *anim;
	GdkPixbuf *pixbuf;

	g_return_val_if_fail (viewer != NULL, 0);

	anim = image_loader_get_animation (viewer->loader);
	if (anim != NULL) 
		pixbuf = gdk_pixbuf_frame_get_pixbuf (
			gdk_pixbuf_animation_get_frames (anim)->data); 
	else
		pixbuf = image_loader_get_pixbuf (viewer->loader);

	if (pixbuf != NULL)
		return gdk_pixbuf_get_bits_per_sample (pixbuf);

	return 0;
}


gboolean
image_viewer_get_has_alpha (ImageViewer *viewer)
{
	GdkPixbufAnimation *anim;
	GdkPixbuf *pixbuf;

	g_return_val_if_fail (viewer != NULL, 0);

	anim = image_loader_get_animation (viewer->loader);
	if (anim != NULL) 
		pixbuf = gdk_pixbuf_frame_get_pixbuf (
			gdk_pixbuf_animation_get_frames (anim)->data); 
	else
		pixbuf = image_loader_get_pixbuf (viewer->loader);

	if (pixbuf != NULL)
		return gdk_pixbuf_get_has_alpha (pixbuf);

	return FALSE;
}


static void
init_animation (ImageViewer *viewer)
{
	GdkPixbufAnimation *anim;

	g_return_if_fail (viewer != NULL);

	if (!viewer->is_animation) return;
	if (viewer->running_animation) return;

	anim = image_loader_get_animation (viewer->loader);
	if (anim == NULL) {
		viewer->is_animation = FALSE;
		return;
	}

	viewer->n_frames = gdk_pixbuf_animation_get_num_frames (anim);
	viewer->first_frame = gdk_pixbuf_animation_get_frames (anim);
	viewer->current_frame = viewer->first_frame;

	create_first_frame (viewer);

	if (viewer->n_frames <= 1)
		return;
	
	viewer->running_animation = TRUE;
}


void
image_viewer_start_animation (ImageViewer *viewer)
{
	g_return_if_fail (viewer != NULL);

	if (viewer->current_frame == NULL)
		init_animation (viewer);

	if (viewer->n_frames <= 1)
		return;

	viewer->running_animation = TRUE;

	image_viewer_update_view (viewer);
}


void
image_viewer_stop_animation (ImageViewer *viewer)
{
	g_return_if_fail (viewer != NULL);

	if (viewer->anim_id == 0) 
		return;

	viewer->running_animation = FALSE;

	gtk_timeout_remove (viewer->anim_id);
	viewer->anim_id = 0;
}


void
image_viewer_step_animation (ImageViewer *viewer)
{
	g_return_if_fail (viewer != NULL);

	if (!viewer->is_animation) return;
	if (viewer->running_animation) return;
	if (viewer->n_frames == 1) return;
	if (viewer->rendering) return;

	change_frame_cb (viewer);
}


gboolean
image_viewer_is_animation (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);
	return viewer->is_animation;
}


gboolean
image_viewer_is_running_animation (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);
	return viewer->running_animation;
}


static void
image_error (ImageLoader *il,
	     gpointer data)
{
	ImageViewer *viewer = data;

	image_viewer_set_void (viewer);
	gtk_signal_emit (GTK_OBJECT (viewer), image_viewer_signals[IMAGE_LOADED]);
}


static void
image_loaded (ImageLoader *il,
	      gpointer data)
{
	ImageViewer *viewer = data;
	GdkPixbufAnimation *anim;

	image_viewer_stop_animation (viewer);

	anim = image_loader_get_animation (viewer->loader);
	viewer->is_animation = (anim != NULL);

	if (viewer->is_animation)
		init_animation (viewer);

	switch (viewer->zoom_change) {
	case ZOOM_CHANGE_ACTUAL_SIZE:
		image_viewer_set_zoom (viewer, 1.0);
		break;

	case ZOOM_CHANGE_FIT:
		image_viewer_zoom_to_fit (viewer);
		break;

	case ZOOM_CHANGE_KEEP_PREV:
		image_viewer_update_view (viewer);
		break;

	case ZOOM_CHANGE_FIT_IF_LARGER:
		image_viewer_zoom_to_fit_if_larger (viewer);
		break;
	}

	gtk_signal_emit (GTK_OBJECT (viewer), image_viewer_signals[IMAGE_LOADED]);
}


static gint
get_frame_delay (GdkPixbufFrame *frame)
{
	return MAX (MINIMUM_DELAY, 
		    gdk_pixbuf_frame_get_delay_time (frame) * 10);
}


void
image_viewer_set_zoom (ImageViewer *viewer, 
		       gdouble zoom_level)
{
	gdouble zoom_ratio;

	g_return_if_fail (viewer != NULL);
	g_return_if_fail (viewer->loader != NULL);

	/* try to keep the center of the view visible. */
	zoom_ratio = zoom_level / viewer->zoom_level;

	viewer->x_offset = ((viewer->x_offset + viewer->gdk_width / 2) 
			    * zoom_ratio - viewer->gdk_width / 2);
	viewer->y_offset = ((viewer->y_offset + viewer->gdk_height / 2) 
			    * zoom_ratio - viewer->gdk_height / 2);

 	/* if we are performing a zoom to fit do not reset zoom_fit. */
	if (! viewer->doing_zoom_fit) {
		viewer->zoom_fit = FALSE;
		viewer->zoom_fit_if_larger = FALSE;
	}
	viewer->zoom_level = zoom_level;

	if (! viewer->doing_zoom_fit)
		gtk_widget_queue_resize (GTK_WIDGET (viewer));

	if (! viewer->doing_change_frame) 
		gtk_signal_emit (GTK_OBJECT (viewer), 
				 image_viewer_signals[ZOOM_CHANGED]);

	else
		viewer->doing_change_frame = FALSE;
}


gdouble
image_viewer_get_zoom (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);

	return viewer->zoom_level;
}


void
image_viewer_set_zoom_quality (ImageViewer *viewer,
			       ZoomQuality quality)
{
	g_return_if_fail (viewer != NULL);
	viewer->zoom_quality = quality;
}


ZoomQuality
image_viewer_get_zoom_quality (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, -1);
	return viewer->zoom_quality;
}


void
image_viewer_set_zoom_change (ImageViewer *viewer,
			      ZoomChange zoom_change)
{
	g_return_if_fail (viewer != NULL);

	if (zoom_change == viewer->zoom_change)
		return;

	viewer->zoom_change = zoom_change;
}


ZoomChange
image_viewer_get_zoom_change (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, -1);
	return viewer->zoom_change;
}


static gdouble zooms[] = {                  0.05, 0.07, 0.10, 
			  0.15, 0.20, 0.30, 0.50, 0.75, 1.0,
			  1.5 , 2.0 , 3.0 , 5.0 , 7.5,  10.0, 
			  15.0, 20.0, 30.0, 50.0, 75.0, 100.0}; 

static const gint nzooms = sizeof (zooms) / sizeof (gdouble);


static gdouble
get_next_zoom (gdouble zoom)
{
	gint i;

	i = 0;
	while ((i < nzooms) && (zooms[i] <= zoom))
		i++;

	i = CLAMP (i, 0, nzooms - 1);

	return zooms[i];
}


static gdouble
get_prev_zoom (gdouble zoom)
{
	gint i;

	i = nzooms - 1;
	while ((i >= 0) && (zooms[i] >= zoom))
		i--;

	i = CLAMP (i, 0, nzooms - 1);

	return zooms[i];
}


void
image_viewer_zoom_in (ImageViewer *viewer)
{
	gdouble new_zoom;

	g_return_if_fail (viewer != NULL);
	g_return_if_fail (viewer->loader != NULL);

	if (get_current_pixbuf (viewer) == NULL)
		return;

	new_zoom = get_next_zoom (viewer->zoom_level);
	
	image_viewer_set_zoom (viewer, new_zoom);
}


void
image_viewer_zoom_out (ImageViewer *viewer)
{
	gdouble new_zoom;

	g_return_if_fail (viewer != NULL);
	g_return_if_fail (viewer->loader != NULL);

	if (get_current_pixbuf (viewer) == NULL)
		return;

	new_zoom = get_prev_zoom (viewer->zoom_level);
	image_viewer_set_zoom (viewer, new_zoom);
}


void
image_viewer_zoom_to_fit (ImageViewer *viewer)
{
	g_return_if_fail (viewer != NULL);
	g_return_if_fail (viewer->loader != NULL);

	viewer->zoom_fit = TRUE;
	viewer->zoom_fit_if_larger = FALSE;

	if (viewer->is_void)
		return;

	gtk_widget_queue_resize (GTK_WIDGET (viewer));
}


gboolean
image_viewer_is_zoom_to_fit (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);
	return viewer->zoom_fit;
}


void
image_viewer_zoom_to_fit_if_larger (ImageViewer *viewer)
{
	g_return_if_fail (viewer != NULL);
	g_return_if_fail (viewer->loader != NULL);

	viewer->zoom_fit = FALSE;
	viewer->zoom_fit_if_larger = TRUE;

	if (viewer->is_void)
		return;

	gtk_widget_queue_resize (GTK_WIDGET (viewer));
}


gboolean
image_viewer_is_zoom_to_fit_if_larger (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);
	return viewer->zoom_fit_if_larger;
}


void           
image_viewer_set_transp_type (ImageViewer *viewer, 
			      TranspType transp_type)
{
	g_return_if_fail (viewer != NULL);

	viewer->transp_type = transp_type;

	switch (transp_type) {
        case TRANSP_TYPE_BLACK:
		viewer->check_color1 = COLOR_GRAY_00;
		viewer->check_color2 = COLOR_GRAY_00;
		break;

        case TRANSP_TYPE_GRAY:
		viewer->check_color1 = COLOR_GRAY_99;
		viewer->check_color2 = COLOR_GRAY_99;
		break;

        case TRANSP_TYPE_WHITE:
		viewer->check_color1 = COLOR_GRAY_FF;
		viewer->check_color2 = COLOR_GRAY_FF;
		break;

        case TRANSP_TYPE_CHECKED:
		switch (viewer->check_type) {
		case CHECK_TYPE_DARK:
			viewer->check_color1 = COLOR_GRAY_00;
			viewer->check_color2 = COLOR_GRAY_33;
			break;
			
		case CHECK_TYPE_MIDTONE:
			viewer->check_color1 = COLOR_GRAY_66;
			viewer->check_color2 = COLOR_GRAY_99;
			break;
			
		case CHECK_TYPE_LIGHT:
			viewer->check_color1 = COLOR_GRAY_CC;
			viewer->check_color2 = COLOR_GRAY_FF;
			break;
		}
		break;
	}
}


TranspType
image_viewer_get_transp_type (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);

	return viewer->transp_type;
}


void           
image_viewer_set_check_type (ImageViewer *viewer, 
			     CheckType check_type)
{
	g_return_if_fail (viewer != NULL);

	viewer->check_type = check_type;
}


CheckType
image_viewer_get_check_type (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);

	return viewer->check_type;
}


void
image_viewer_set_check_size (ImageViewer *viewer, 
			     CheckSize check_size)
{
	g_return_if_fail (viewer != NULL);

	viewer->check_size = check_size;
}


CheckSize
image_viewer_get_check_size (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);

	return viewer->check_size;
}


void
image_viewer_alter (ImageViewer *viewer, 
		    ImageAlterType type)
{
	GdkPixbuf *pixbuf;
	GdkPixbuf *orig_pixbuf;

	g_return_if_fail (viewer != NULL);
	
	if (viewer->is_void) return;
	if (viewer->is_animation && (viewer->n_frames > 1)) return;
	if (viewer->rendering) return;
	
	orig_pixbuf = image_loader_get_pixbuf (viewer->loader);
	if (orig_pixbuf == NULL) return;

	pixbuf = NULL;
	switch (type) {
	case ALTER_NONE:
		pixbuf = gdk_pixbuf_copy (orig_pixbuf);
		return;

	case ALTER_ROTATE_90:
		pixbuf = pixbuf_copy_rotate_90 (orig_pixbuf, FALSE);
		break;

	case ALTER_ROTATE_90_CC:
		pixbuf = pixbuf_copy_rotate_90 (orig_pixbuf, TRUE);
		break;

	case ALTER_ROTATE_180:
		pixbuf = pixbuf_copy_mirror (orig_pixbuf, TRUE, TRUE);
		break;

	case ALTER_MIRROR:
		pixbuf = pixbuf_copy_mirror (orig_pixbuf, TRUE, FALSE);
		break;

	case ALTER_FLIP:
		pixbuf = pixbuf_copy_mirror (orig_pixbuf, FALSE, TRUE);
		break;
	}

	image_loader_set_pixbuf (viewer->loader, pixbuf);
	gdk_pixbuf_unref (pixbuf);

	image_viewer_update_view (viewer);
}


void
image_viewer_clicked (ImageViewer *viewer)
{
	g_return_if_fail (viewer != NULL);
	g_return_if_fail (IS_IMAGE_VIEWER (viewer));

	gtk_signal_emit (GTK_OBJECT (viewer), image_viewer_signals[CLICKED]);
}


static void 
get_zoomed_size (ImageViewer *viewer,
		 gint * width,
		 gint * height,
		 gdouble zoom_level)
{
	gint w, h;
	GdkPixbuf *pixbuf;

	pixbuf = get_current_pixbuf (viewer);

	if (pixbuf == NULL) {
		*width = 0;
		*height = 0;
	} else {
		w = image_viewer_get_image_width (viewer);
		h = image_viewer_get_image_height (viewer);

		*width  = (gint) floor ((double) w * zoom_level);
		*height = (gint) floor ((double) h * zoom_level);
	}
}


/* set_scroll_adjustments handler for the image view */
static void
set_scroll_adjustments (GtkWidget *widget,
			GtkAdjustment *hadj,
			GtkAdjustment *vadj)
{
        ImageViewer *viewer;

        g_return_if_fail (widget != NULL);
        g_return_if_fail (IS_IMAGE_VIEWER (widget));

        viewer = IMAGE_VIEWER (widget);

        if (hadj)
                g_return_if_fail (GTK_IS_ADJUSTMENT (hadj));
        else
                hadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 
							   0.0, 0.0, 0.0));

        if (vadj)
                g_return_if_fail (GTK_IS_ADJUSTMENT (vadj));
        else
                vadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 
							   0.0, 0.0, 0.0));

        if (viewer->hadj && viewer->hadj != hadj) {
                gtk_signal_disconnect_by_data (GTK_OBJECT (viewer->hadj), 
					       viewer);
                gtk_object_unref (GTK_OBJECT (viewer->hadj));
		viewer->hadj = NULL;
        }

        if (viewer->vadj && viewer->vadj != vadj) {
                gtk_signal_disconnect_by_data (GTK_OBJECT (viewer->vadj), 
					       viewer);
                gtk_object_unref (GTK_OBJECT (viewer->vadj));
		viewer->vadj = NULL;
        }

        if (viewer->hadj != hadj) {
                viewer->hadj = hadj;
                gtk_object_ref (GTK_OBJECT (viewer->hadj));
                gtk_object_sink (GTK_OBJECT (viewer->hadj));

                gtk_signal_connect (GTK_OBJECT (viewer->hadj), "value_changed",
                                    GTK_SIGNAL_FUNC (hadj_value_changed),
                                    viewer);
        }

        if (viewer->vadj != vadj) {
                viewer->vadj = vadj;
                gtk_object_ref (GTK_OBJECT (viewer->vadj));
                gtk_object_sink (GTK_OBJECT (viewer->vadj));

                gtk_signal_connect (GTK_OBJECT (viewer->vadj), "value_changed",
                                    GTK_SIGNAL_FUNC (vadj_value_changed),
                                    viewer);
        }
}


void
scroll_relative (ImageViewer *viewer, 
		 gint delta_x, 
		 gint delta_y)
{
	image_viewer_scroll_to (viewer, 			
				viewer->x_offset + delta_x,	
				viewer->y_offset + delta_y);
}


void
image_viewer_scroll_step_x (ImageViewer *viewer,
			    gboolean increment)
{
	g_return_if_fail (IS_IMAGE_VIEWER (viewer));
	scroll_relative (viewer, 
			 (increment ? 1 : -1) * viewer->hadj->step_increment,
			 0);
}


void
image_viewer_scroll_step_y (ImageViewer *viewer,
			    gboolean increment)
{
	g_return_if_fail (IS_IMAGE_VIEWER (viewer));
	scroll_relative (viewer, 
			 0,
			 (increment ? 1 : -1) * viewer->vadj->step_increment);
}


void
image_viewer_scroll_page_x (ImageViewer *viewer,
			    gboolean increment)
{
	g_return_if_fail (IS_IMAGE_VIEWER (viewer));
	scroll_relative (viewer, 
			 (increment ? 1 : -1) * viewer->hadj->page_increment,
			 0);
}


void
image_viewer_scroll_page_y (ImageViewer *viewer,
			    gboolean increment)
{
	g_return_if_fail (IS_IMAGE_VIEWER (viewer));
	scroll_relative (viewer, 
			 0,
			 (increment ? 1 : -1) * viewer->vadj->page_increment);
}


void
image_viewer_size (ImageViewer *viewer,
		   gint width,
		   gint height)
{
	g_return_if_fail (IS_IMAGE_VIEWER (viewer));
	gtk_drawing_area_size (GTK_DRAWING_AREA (viewer->drawing_area), 
			       width, height);
}


void
image_viewer_show_cursor (ImageViewer *viewer)
{
	g_return_if_fail (IS_IMAGE_VIEWER (viewer));

	viewer->cursor_visible = TRUE;
	gdk_window_set_cursor (viewer->drawing_area->window, 
			       viewer->cursor);
}


void
image_viewer_hide_cursor (ImageViewer *viewer)
{
	g_return_if_fail (IS_IMAGE_VIEWER (viewer));

	viewer->cursor_visible = FALSE;
	gdk_window_set_cursor (viewer->drawing_area->window, 
			       viewer->cursor_void);
}


gboolean
image_viewer_is_cursor_visible (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);
	return viewer->cursor_visible;
}


void
image_viewer_set_black_background (ImageViewer *viewer,
				   gboolean set_black)
{
	g_return_if_fail (IS_IMAGE_VIEWER (viewer));

	viewer->black_bg = set_black;
	gtk_widget_queue_draw (GTK_WIDGET (viewer));
}


gboolean
image_viewer_is_black_background (ImageViewer *viewer)
{
	g_return_val_if_fail (viewer != NULL, FALSE);

	return viewer->black_bg;
}
