/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqview.h"
#include "view_file_icon.h"

#include "collect.h"
#include "collect-io.h"
#include "collect-table.h"
#include "dnd.h"
#include "img-view.h"
#include "info.h"
#include "filelist.h"
#include "layout.h"
#include "layout_image.h"
#include "menu.h"
#include "thumb.h"
#include "utilops.h"
#include "ui_clist_edit.h"
#include "ui_fileops.h"
#include "ui_menu.h"


#include <gdk/gdkkeysyms.h> /* for keyboard values */


#define VFICON_MAX_COLUMNS 32
#define VFICON_TEXT_COLUMNS 33	/* this is ridiculous */

/* quicky note:
 *   I wanted this to simply be a wrapper around a collection table, and add
 *   the needed stuff there to make this work - but after wasting considerable time
 *   I decided it would be faster to re-work a copy of the collection table here, as
 *   many features of collection tables are not needed (and are annoying to circumvent).
 */

/*
 *-------------------------------------------------------------------
 * wrappers (FileData needs to also keep pixmap and mask)
 * the point is only use iconlist_read/free, and all the
 * other filelist_XXX functions (such as sort) will work.
 *-------------------------------------------------------------------
 */

#define ICON_DATA(x) ((IconData *)x)
#define FILE_DATA(x) ((FileData *)x)

typedef struct _IconData IconData;
struct _IconData
{
	FileData fd;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
};

static gint iconlist_read(const gchar *path, GList **list)
{
	GList *temp;
	GList *work;

	if (!filelist_read(path, &temp, NULL)) return FALSE;

	work = temp;
	while (work)
		{
		FileData *fd;
		IconData *id;

		fd = work->data;
		id = g_new(IconData, 1);
		
		memcpy(id, fd, sizeof(FileData));
		id->pixmap = NULL;
		id->mask = NULL;

		work->data = id;

		g_free(fd);

		work = work->next;
		}

	*list = temp;

	return TRUE;
}

static void iconlist_free(GList *list)
{
	GList *work = list;

	while (work)
		{
		IconData *id = work->data;
		work = work->next;

		g_free(FILE_DATA(id)->path);

		if (id->pixmap) gdk_pixmap_unref(id->pixmap);
		if (id->mask) gdk_bitmap_unref(id->mask);

		g_free(id);
		}

	g_list_free(list);
}

static void icondata_set_thumb(IconData *id, GdkPixmap *pixmap, GdkBitmap *mask)
{
	if (!id) return;

	if (id->pixmap) gdk_pixmap_unref(id->pixmap);
	if (id->mask) gdk_bitmap_unref(id->mask);

	id->pixmap = pixmap;
	id->mask = mask;

	if (id->pixmap) gdk_pixmap_ref(id->pixmap);
	if (id->mask) gdk_bitmap_ref(id->mask);
}


static void vficon_cell_draw_active(ViewFileIcon *vfi, FileData *fd, gint active);
static void vficon_move_focus(ViewFileIcon *vfi, gint row, gint col, gint relative);
static gint vficon_is_selected(ViewFileIcon *vfi, FileData *fd);
static void vficon_thumb_update(ViewFileIcon *vfi);


/*
 *-----------------------------------------------------------------------------
 * pop-up menu
 *-----------------------------------------------------------------------------
 */

static GList *vficon_pop_menu_file_list(ViewFileIcon *vfi)
{
	if (vficon_is_selected(vfi, vfi->click_fd))
		{
		return vficon_selection_get_list(vfi);
		}

	if (!vfi->click_fd) return NULL;

	return g_list_append(NULL, g_strdup(vfi->click_fd->path));
}

static void vficon_pop_menu_edit_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi;
	gint n;
	GList *list;

	vfi = submenu_item_get_data(widget);
	n = GPOINTER_TO_INT(data);

	if (!vfi) return;

	list = vficon_pop_menu_file_list(vfi);
	start_editor_from_path_list(n, list);
	path_list_free(list);
}

static void vficon_pop_menu_info_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	info_window_new(NULL, vficon_pop_menu_file_list(vfi));
}

static void vficon_pop_menu_view_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	if (vfi->click_fd) view_window_new(vfi->click_fd->path);
}

static void vficon_pop_menu_copy_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	file_util_copy(NULL, vficon_pop_menu_file_list(vfi), NULL);
}

static void vficon_pop_menu_move_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	file_util_move(NULL, vficon_pop_menu_file_list(vfi), NULL);
}

static void vficon_pop_menu_rename_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	file_util_rename(NULL, vficon_pop_menu_file_list(vfi));
}

static void vficon_pop_menu_delete_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	file_util_delete(NULL, vficon_pop_menu_file_list(vfi));
}

static void vficon_pop_menu_sort_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi;
	SortType type;
	
	vfi = submenu_item_get_data(widget);
	if (!vfi) return;

	type = (SortType)GPOINTER_TO_INT(data);

	if (vfi->layout)
		{
		layout_sort_set(vfi->layout, type, vfi->sort_ascend);
		}
	else
		{
		vficon_sort_set(vfi, type, vfi->sort_ascend);
		}
}

static void vficon_pop_menu_sort_ascend_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	if (vfi->layout)
		{
		layout_sort_set(vfi->layout, vfi->sort_method, !vfi->sort_ascend);
		}
	else
		{
		vficon_sort_set(vfi, vfi->sort_method, !vfi->sort_ascend);
		}
}

static void vficon_pop_menu_list_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	if (vfi->layout) layout_views_set(vfi->layout, vfi->layout->tree_view, FALSE);
}

static void vficon_pop_menu_refresh_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	vficon_refresh(vfi);
}

static GtkWidget *vficon_pop_menu(ViewFileIcon *vfi, gint active)
{
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *submenu;

	menu = popup_menu_short_lived();

	submenu_add_edit(menu, &item, GTK_SIGNAL_FUNC(vficon_pop_menu_edit_cb), vfi);
	gtk_widget_set_sensitive(item, active);

	item = menu_item_add(menu, _("Properties"), GTK_SIGNAL_FUNC(vficon_pop_menu_info_cb), vfi);
	gtk_widget_set_sensitive(item, active);

	item = menu_item_add(menu, _("View in new window"), GTK_SIGNAL_FUNC(vficon_pop_menu_view_cb), vfi);
	gtk_widget_set_sensitive(item, active);

	menu_item_add_divider(menu);
	item = menu_item_add(menu, _("Copy..."), GTK_SIGNAL_FUNC(vficon_pop_menu_copy_cb), vfi);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Move..."), GTK_SIGNAL_FUNC(vficon_pop_menu_move_cb), vfi);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Rename..."), GTK_SIGNAL_FUNC(vficon_pop_menu_rename_cb), vfi);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Delete..."), GTK_SIGNAL_FUNC(vficon_pop_menu_delete_cb), vfi);
	gtk_widget_set_sensitive(item, active);

	menu_item_add_divider(menu);

	submenu = submenu_add_sort(NULL, GTK_SIGNAL_FUNC(vficon_pop_menu_sort_cb), vfi,
				   FALSE, FALSE, TRUE, vfi->sort_method);
	menu_item_add_divider(submenu);
	menu_item_add_check(submenu, _("Ascending"), vfi->sort_ascend,
			    GTK_SIGNAL_FUNC(vficon_pop_menu_sort_ascend_cb), vfi);

	item = menu_item_add(menu, _("Sort"), NULL, NULL);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

	menu_item_add_check(menu, _("View as icons"), TRUE,
			    GTK_SIGNAL_FUNC(vficon_pop_menu_list_cb), vfi);
	menu_item_add(menu, _("Refresh"), GTK_SIGNAL_FUNC(vficon_pop_menu_refresh_cb), vfi);

	return menu;
}

/*
 *-------------------------------------------------------------------
 * signals
 *-------------------------------------------------------------------
 */

static void vficon_send_update(ViewFileIcon *vfi)
{
	if (vfi->func_status) vfi->func_status(vfi, vfi->data_status);
}

static void vficon_send_layout_select(ViewFileIcon *vfi, FileData *fd)
{
	const gchar *read_ahead_path = NULL;

	if (!vfi->layout || !fd) return;

	if (enable_read_ahead)
		{
		FileData *fd_n;
		gint row;

		row = g_list_index(vfi->list, fd);
		if (row > vficon_index_by_path(vfi, layout_image_get_path(vfi->layout)) &&
		    row + 1 < vficon_count(vfi, NULL))
			{
			fd_n = vficon_index_get_data(vfi, row + 1);
			}
		else if (row > 0)
			{
			fd_n = vficon_index_get_data(vfi, row - 1);
			}
		else
			{
			fd_n = NULL;
			}
		if (fd_n) read_ahead_path = fd_n->path;
		}

	layout_image_set_with_ahead(vfi->layout, fd->path, read_ahead_path);
}

/*
 *-------------------------------------------------------------------
 * misc utils
 *-------------------------------------------------------------------
 */

static gint vficon_find_position(ViewFileIcon *vfi, FileData *fd, gint *row, gint *col)
{
	gint n;

	n = g_list_index(vfi->list, fd);

	if (n < 0) return FALSE;

	*row = n / vfi->columns;
	*col = n - (*row * vfi->columns);

	return TRUE;
}

static FileData *vficon_find_info(ViewFileIcon *vfi, gint row, gint col)
{
	GList *list;

	list = gtk_clist_get_row_data(GTK_CLIST(vfi->clist), row);

	if (!list) return NULL;

	return g_list_nth_data(list, col);
}

/*
 *-------------------------------------------------------------------
 * tooltip type window
 *-------------------------------------------------------------------
 */

static void tip_show(ViewFileIcon *vfi)
{
	GtkWidget *label;
	gint x, y;
	gint row, col;

	if (vfi->tip_window) return;

	gdk_window_get_pointer(GTK_CLIST(vfi->clist)->clist_window, &x, &y, NULL);
	gtk_clist_get_selection_info(GTK_CLIST(vfi->clist), x, y, &row, &col);

	vfi->tip_fd = vficon_find_info(vfi, row, col);
	if (!vfi->tip_fd) return;

	vfi->tip_window = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_window_set_policy(GTK_WINDOW(vfi->tip_window), FALSE, FALSE, TRUE);
	gtk_container_border_width(GTK_CONTAINER(vfi->tip_window), 2);

	label = gtk_label_new(vfi->tip_fd->name);

	gtk_object_set_data(GTK_OBJECT(vfi->tip_window), "tip_label", label);
	gtk_container_add(GTK_CONTAINER(vfi->tip_window), label);
	gtk_widget_show(label);

	gdk_window_get_pointer(NULL, &x, &y, NULL);

	gtk_widget_popup(vfi->tip_window, x + 16, y + 16);
}

static void tip_hide(ViewFileIcon *vfi)
{
	if (vfi->tip_window) gtk_widget_destroy(vfi->tip_window);
	vfi->tip_window = NULL;
}

static gint tip_schedule_cb(gpointer data)
{
	ViewFileIcon *vfi = data;

	if (vfi->tip_delay_id == -1) return FALSE;

	if (GTK_WIDGET_SENSITIVE(gtk_widget_get_toplevel(vfi->clist)))
		{
		tip_show(vfi);
		}

	vfi->tip_delay_id = -1;
	return FALSE;
}

static void tip_schedule(ViewFileIcon *vfi)
{
	tip_hide(vfi);

	if (vfi->tip_delay_id != -1) gtk_timeout_remove(vfi->tip_delay_id);

	vfi->tip_delay_id = gtk_timeout_add(1000, tip_schedule_cb, vfi);
}

static void tip_unschedule(ViewFileIcon *vfi)
{
	tip_hide(vfi);

	if (vfi->tip_delay_id != -1) gtk_timeout_remove(vfi->tip_delay_id);
	vfi->tip_delay_id = -1;
}

static void tip_update(ViewFileIcon *vfi, FileData *fd)
{
	if (vfi->tip_window)
		{
		gint x, y;

		gdk_window_get_pointer(NULL, &x, &y, NULL);
		gdk_window_move(vfi->tip_window->window, x + 16, y + 16);

		if (fd != vfi->tip_fd)
			{
			GtkWidget *label;

			vfi->tip_fd = fd;

			if (!vfi->tip_fd)
				{
				tip_hide(vfi);
				tip_schedule(vfi);
				return;
				}

			label = gtk_object_get_data(GTK_OBJECT(vfi->tip_window), "tip_label");
			gtk_label_set_text(GTK_LABEL(label), vfi->tip_fd->name);
			}
		}
	else
		{
		tip_schedule(vfi);
		}
}

/*
 *-------------------------------------------------------------------
 * dnd
 *-------------------------------------------------------------------
 */

static void vficon_dnd_get(GtkWidget *widget, GdkDragContext *context,
			   GtkSelectionData *selection_data, guint info,
			   guint time, gpointer data)
{
	ViewFileIcon *vfi = data;
	gchar *uri_text = NULL;
	gint total;

	if (!vfi->click_fd) return;

	if (vficon_is_selected(vfi, vfi->click_fd))
		{
		GList *list;

		list = vficon_selection_get_list(vfi);
		if (!list) return;

		uri_text = make_uri_file_list(list, (info == TARGET_TEXT_PLAIN), &total);
		path_list_free(list);
		}
	else
		{
		const gchar *path = vfi->click_fd->path;

		switch (info)
			{
			case TARGET_URI_LIST:
				uri_text = g_strconcat("file:", path, "\r\n", NULL);
				break;
			case TARGET_TEXT_PLAIN:
				uri_text = g_strdup(path);
				break;
			}
		total = strlen(uri_text);
		}

	if (uri_text) gtk_selection_data_set(selection_data, selection_data->target,
					     8, uri_text, total);
	g_free(uri_text);
}

static void vficon_dnd_begin(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	ViewFileIcon *vfi = data;

	vficon_cell_draw_active(vfi, vfi->click_fd, TRUE);
	tip_unschedule(vfi);
}

static void vficon_dnd_end(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	ViewFileIcon *vfi = data;

	vficon_cell_draw_active(vfi, vfi->click_fd, FALSE);

	if (context->action == GDK_ACTION_MOVE)
		{
		vficon_refresh(vfi);
		}

	tip_unschedule(vfi);
}

static void vficon_dnd_init(ViewFileIcon *vfi)
{
	gtk_drag_source_set(vfi->clist, GDK_BUTTON2_MASK,
			    dnd_file_drag_types, dnd_file_drag_types_count,
			    GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);
	gtk_signal_connect(GTK_OBJECT(vfi->clist), "drag_data_get",
			   GTK_SIGNAL_FUNC(vficon_dnd_get), vfi);
	gtk_signal_connect(GTK_OBJECT(vfi->clist), "drag_begin",
			   GTK_SIGNAL_FUNC(vficon_dnd_begin), vfi);
	gtk_signal_connect(GTK_OBJECT(vfi->clist), "drag_end",
			   GTK_SIGNAL_FUNC(vficon_dnd_end), vfi);
}

/*
 *-------------------------------------------------------------------
 * cell updates
 *-------------------------------------------------------------------
 */

static void vficon_cell_draw_active(ViewFileIcon *vfi, FileData *fd, gint active)
{
	gint row, col;

	if (!fd || fd == vfi->focus_fd || vficon_is_selected(vfi, fd)) return;

	if (!vficon_find_position(vfi, fd, &row, &col)) return;

	if (active)
		{
		GtkStyle *style;

		style = gtk_style_copy(GTK_WIDGET(vfi->clist)->style);

		style->base[GTK_STATE_NORMAL] = style->bg[GTK_STATE_ACTIVE];
		style->fg[GTK_STATE_NORMAL] = style->fg[GTK_STATE_ACTIVE];

		gtk_clist_set_cell_style(GTK_CLIST(vfi->clist), row, col, style);
		gtk_style_unref(style);
		}
	else
		{
		gtk_clist_set_cell_style(GTK_CLIST(vfi->clist), row, col, NULL);
		}
}

static void vficon_update_cell(ViewFileIcon *vfi, gint row, gint col, gint selected)
{
	if (selected || (row == vfi->focus_row && col == vfi->focus_column))
		{
		GtkStyle *style;

		style = gtk_style_copy(GTK_WIDGET(vfi->clist)->style);

		if (selected)
			{
			style->base[GTK_STATE_NORMAL] = style->bg[GTK_STATE_SELECTED];
			style->fg[GTK_STATE_NORMAL] = style->fg[GTK_STATE_SELECTED];
			}
		else
			{
			style->base[GTK_STATE_NORMAL] = style->bg[GTK_STATE_ACTIVE];
			style->fg[GTK_STATE_NORMAL] = style->fg[GTK_STATE_ACTIVE];
			}

		if (row == vfi->focus_row && col == vfi->focus_column)
			{
			clist_edit_shift_color(style);
			}

		gtk_clist_set_cell_style(GTK_CLIST(vfi->clist), row, col, style);
		gtk_style_unref(style);
		}
        else
		{
		gtk_clist_set_cell_style(GTK_CLIST(vfi->clist), row, col, NULL);
		}
}

static void vficon_update_cell_by_info(ViewFileIcon *vfi, FileData *fd, gint selected)
{
	gint r, c;

	if (!vficon_find_position(vfi, fd, &r, &c)) return;

	vficon_update_cell(vfi, r, c, selected);
}

/*
 *-------------------------------------------------------------------
 * selections
 *-------------------------------------------------------------------
 */

static void vficon_verify_selections(ViewFileIcon *vfi)
{
	GList *work;

	work = vfi->selection;
	while (work)
		{
		FileData *fd = work->data;
		work = work->next;
		if (!g_list_find(vfi->list, fd))
			{
			vfi->selection = g_list_remove(vfi->selection, fd);
			}
		}
}

static gint vficon_is_selected(ViewFileIcon *vfi, FileData *fd)
{
	return (g_list_find(vfi->selection, fd) != NULL);
}

void vficon_select_all(ViewFileIcon *vfi)
{
	GList *work;

	g_list_free(vfi->selection);
	vfi->selection = NULL;

	work = vfi->list;
	while (work)
		{
		vfi->selection = g_list_append(vfi->selection, work->data);
		vficon_update_cell_by_info(vfi, work->data, TRUE);
		work = work->next;
		}

	vficon_send_update(vfi);
}

void vficon_select_none(ViewFileIcon *vfi)
{
	GList *work;

	work = vfi->selection;
	while (work)
		{
		vficon_update_cell_by_info(vfi, work->data, FALSE);
		work = work->next;
		}

	g_list_free(vfi->selection);
	vfi->selection = NULL;

	vficon_send_update(vfi);
}

static void vficon_select(ViewFileIcon *vfi, FileData *fd)
{
	vfi->prev_selection = fd;

	if (!fd || vficon_is_selected(vfi, fd)) return;

	vfi->selection = g_list_append(vfi->selection, fd);

	vficon_update_cell_by_info(vfi, fd, TRUE);

	vficon_send_update(vfi);
}

static void vficon_unselect(ViewFileIcon *vfi, FileData *fd)
{
	vfi->prev_selection = fd;

	if (!fd || !vficon_is_selected(vfi, fd)) return;

	vfi->selection = g_list_remove(vfi->selection, fd);

	vficon_update_cell_by_info(vfi, fd, FALSE);

	vficon_send_update(vfi);
}

static void vficon_select_util(ViewFileIcon *vfi, FileData *fd, gint select)
{
	if (select)
		{
		vficon_select(vfi, fd);
		}
	else
		{
		vficon_unselect(vfi, fd);
		}
}

static void vficon_select_region_util(ViewFileIcon *vfi, FileData *start, FileData *end, gint select)
{
	gint row1, col1;
	gint row2, col2;
	gint t;
	gint i, j;

	if (!vficon_find_position(vfi, start, &row1, &col1) ||
	    !vficon_find_position(vfi, end, &row2, &col2) ) return;

	if (!collection_rectangular_selection)
		{
		GList *work;
		FileData *fd;

		if (g_list_index(vfi->list, start) > g_list_index(vfi->list, end))
			{
			fd = start;
			start = end;
			end = fd;
			}

		work = g_list_find(vfi->list, start);
		while (work)
			{
			fd = work->data;
			vficon_select_util(vfi, fd, select);
			
			if (work->data != end)
				work = work->next;
			else
				work = NULL;
			}
		return;
		}

	if (row2 < row1)
		{
		t = row1;
		row1 = row2;
		row2 = t;
		}
	if (col2 < col1)
		{
		t = col1;
		col1 = col2;
		col2 = t;
		}

	if (debug) printf("table: %d x %d to %d x %d\n", row1, col1, row2, col2);

	for (i = row1; i <= row2; i++)
		{
		for (j = col1; j <= col2; j++)
			{
			FileData *fd = vficon_find_info(vfi, i, j);
			if (fd) vficon_select_util(vfi, fd, select);
			}
		}

	vfi->prev_selection = end;
}

gint vficon_index_is_selected(ViewFileIcon *vfi, gint row)
{
	FileData *fd = g_list_nth_data(vfi->list, row);

	if (!fd) return FALSE;

	return (g_list_find(vfi->selection, fd) != NULL);
}

gint vficon_selection_count(ViewFileIcon *vfi, gint *bytes)
{
	if (bytes)
		{
		gint b = 0;
		GList *work;

		work = vfi->selection;
		while (work)
			{
			FileData *fd = work->data;
			b += fd->size;

			work = work->next;
			}

		*bytes = b;
		}

	return g_list_length(vfi->selection);
}

GList *vficon_selection_get_list(ViewFileIcon *vfi)
{
	GList *list = NULL;
	GList *work;

	work = vfi->selection;
	while (work)
		{
		FileData *fd = work->data;

		list = g_list_prepend(list, g_strdup(fd->path));

		work = work->next;
		}

	list = g_list_reverse(list);

	return list;
}

GList *vficon_selection_get_list_by_index(ViewFileIcon *vfi)
{
	GList *list = NULL;
	GList *work;

	work = vfi->selection;	
	while (work)
		{
		list = g_list_prepend(list, GINT_TO_POINTER(g_list_index(vfi->list, work->data)));
		work = work->next;
		}

	return g_list_reverse(list);
}

void vficon_select_by_path(ViewFileIcon *vfi, const gchar *path)
{
	FileData *fd;
	GList *work;
	gint row, col;

	if (!path) return;

	fd = NULL;
	work = vfi->list;
	while (work && !fd)
		{
		FileData *chk = work->data;
		work = work->next;
		if (strcmp(chk->path, path) == 0) fd = chk;
		}

	if (!fd) return;

	if (!vficon_is_selected(vfi, fd))
		{
		vficon_select_none(vfi);
		vficon_select(vfi, fd);
		}

	if (vficon_find_position(vfi, fd, &row, &col))
		{
		vficon_move_focus(vfi, row, col, FALSE);
		}
}

/*
 *-------------------------------------------------------------------
 * focus
 *-------------------------------------------------------------------
 */

static void vficon_move_focus(ViewFileIcon *vfi, gint row, gint col, gint relative)
{
	gint new_row;
	gint new_col;

	if (relative)
		{
		new_row = vfi->focus_row;
		new_col = vfi->focus_column;

		new_row += row;
		if (new_row < 0) new_row = 0;
		if (new_row >= vfi->rows) new_row = vfi->rows - 1;

		while(col != 0)
			{
			if (col < 0)
				{
				new_col--;
				col++;
				}
			else
				{
				new_col++;
				col--;
				}

			if (new_col < 0)
				{
				if (new_row > 0)
					{
					new_row--;
					new_col = vfi->columns - 1;
					}
				else
					{
					new_col = 0;
					}
				}
			if (new_col >= vfi->columns)
				{
				if (new_row < vfi->rows - 1)
					{
					new_row++;
					new_col = 0;
					}
				else
					{
					new_col = vfi->columns - 1;
					}
				}
			}
		}
	else
		{
		new_row = row;
		new_col = col;

		if (new_row >= vfi->rows)
			{
			if (vfi->rows > 0)
				new_row = vfi->rows - 1;
			else
				new_row = 0;
			new_col = vfi->columns - 1;
			}
		if (new_col >= vfi->columns) new_col = vfi->columns - 1;
		}

	if (new_row == vfi->rows - 1)
		{
		gint l;

		/* if we moved beyond the last image, go to the last image */

		l = g_list_length(vfi->list);
		if (vfi->rows > 1) l -= (vfi->rows - 1) * vfi->columns;
		if (new_col >= l) new_col = l - 1;
		}

	if (vfi->list && (new_row != vfi->focus_row || new_col != vfi->focus_column) )
		{
		gint old_row;
		gint old_col;

		if (debug) printf("view_file_icon.c: highlight updated\n");

		GTK_CLIST(vfi->clist)->focus_row = new_row;

		old_row = vfi->focus_row;
		old_col = vfi->focus_column;
		vfi->focus_row = -1;
		vfi->focus_column = -1;
		vficon_update_cell(vfi, old_row, old_col,
			vficon_is_selected(vfi, vficon_find_info(vfi, old_row, old_col)) );

		vfi->focus_row = new_row;
		vfi->focus_column = new_col;
		vficon_update_cell(vfi, new_row, new_col,
			vficon_is_selected(vfi, vficon_find_info(vfi, new_row, new_col)) );

		/* scroll, if needed */
		if (!vfi->mouse_in_drag &&
		    gtk_clist_row_is_visible(GTK_CLIST(vfi->clist), new_row) != GTK_VISIBILITY_FULL)
			{
			if (new_row < old_row)
				gtk_clist_moveto(GTK_CLIST(vfi->clist), new_row, new_col, 0.0, -1);
			else
				gtk_clist_moveto(GTK_CLIST(vfi->clist), new_row, new_col, 1.0, -1);
			}
		}

	if (debug) printf("view_file_icon.c: highlight pos %d x %d\n", vfi->focus_column, vfi->focus_row);
	vfi->focus_fd = vficon_find_info(vfi, vfi->focus_row, vfi->focus_column);
}

static void vficon_update_focus(ViewFileIcon *vfi)
{
	gint new_row = 0;
	gint new_col = 0;

	if (vfi->focus_fd && vficon_find_position(vfi, vfi->focus_fd, &new_row, &new_col))
		{
		/* first find the old focus, if it exists and is valid */
		}
	else
		{
		/* (try to) stay where we were */
		new_row = vfi->focus_row;
		new_col = vfi->focus_column;
		}

	vficon_move_focus(vfi, new_row, new_col, FALSE);
}

/* used to figure the page up/down distances */
static gint clist_row_height(GtkCList *clist)
{
	GtkAdjustment* adj;
	gint ret;

	adj = gtk_clist_get_vadjustment(clist);

	if (!adj) return 1;

	ret = adj->page_size / adj->step_increment;
	if (ret < 1)
		ret = 1;
	else if (ret > 1)
		ret--;

	return ret;
}

/*
 *-------------------------------------------------------------------
 * keyboard
 *-------------------------------------------------------------------
 */

static gint vficon_keypress_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	ViewFileIcon *vfi = data;
	gint stop_signal = FALSE;
	gint focus_row = 0;
	gint focus_col = 0;
	FileData *fd;

	switch (event->keyval)
		{
		case GDK_Left: case GDK_KP_Left:
			focus_col = -1;
			stop_signal = TRUE;
			break;
		case GDK_Right: case GDK_KP_Right:
			focus_col = 1;
			stop_signal = TRUE;
			break;
		case GDK_Up: case GDK_KP_Up:
			focus_row = -1;
			stop_signal = TRUE;
			break;
		case GDK_Down: case GDK_KP_Down:
			focus_row = 1;
			stop_signal = TRUE;
			break;
		case GDK_Page_Up: case GDK_KP_Page_Up:
			focus_row = -clist_row_height(GTK_CLIST(vfi->clist));
			stop_signal = TRUE;
			break;
		case GDK_Page_Down: case GDK_KP_Page_Down:
			focus_row = clist_row_height(GTK_CLIST(vfi->clist));
			stop_signal = TRUE;
			break;
		case GDK_Home: case GDK_KP_Home:
			focus_row = -vfi->focus_row;
			focus_col = -vfi->focus_column;
			stop_signal = TRUE;
			break;
		case GDK_End: case GDK_KP_End:
			focus_row = vfi->rows - 1 - vfi->focus_row;
			focus_col = vfi->columns - 1 - vfi->focus_column;
			stop_signal = TRUE;
			break;
		case GDK_space:
			fd = vficon_find_info(vfi, vfi->focus_row, vfi->focus_column);
			if (fd)
				{
				vfi->mouse_start = fd;
				if (event->state & GDK_CONTROL_MASK)
					{
					gint selected = !vficon_is_selected(vfi, fd);
					vficon_select_util(vfi, fd, selected);
					if (selected) vficon_send_layout_select(vfi, fd);
					}
				else
					{
					vficon_select_none(vfi);
					vficon_select(vfi, fd);
					vficon_send_layout_select(vfi, fd);
					}
				}
			stop_signal = TRUE;
			break;
		default:
			break;
		}

	if (focus_row != 0 || focus_col != 0)
		{
		FileData *new_fd;
		FileData *old_fd;

		old_fd = vficon_find_info(vfi, vfi->focus_row, vfi->focus_column);
		vficon_move_focus(vfi, focus_row, focus_col, TRUE);
		new_fd = vficon_find_info(vfi, vfi->focus_row, vfi->focus_column);

		if (new_fd != old_fd)
			{
			if (event->state & GDK_SHIFT_MASK)
				{
				gtk_clist_freeze(GTK_CLIST(vfi->clist));
				if (!collection_rectangular_selection)
					{
					/* faster */
					vficon_select_region_util(vfi, old_fd, new_fd, FALSE);
					}
				else
					{
					/* slower */
					vficon_select_region_util(vfi, vfi->mouse_start, old_fd, FALSE);
					}
				vficon_select_region_util(vfi, vfi->mouse_start, new_fd, TRUE);
				gtk_clist_thaw(GTK_CLIST(vfi->clist));
				vficon_send_layout_select(vfi, new_fd);
				}
			else if (event->state & GDK_CONTROL_MASK)
				{
				vfi->mouse_start = new_fd;
				}
			else
				{
				vfi->mouse_start = new_fd;
				vficon_select_none(vfi);
				vficon_select(vfi, new_fd);
				vficon_send_layout_select(vfi, new_fd);
				}
			}
		}

	if (stop_signal)
		{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tip_unschedule(vfi);
		}

	return stop_signal;
}

/*
 *-------------------------------------------------------------------
 * mouse
 *-------------------------------------------------------------------
 */

static gint vficon_button_motion(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	ViewFileIcon *vfi = data;
	gint row, col;
	FileData *fd;

	if (bevent->window != GTK_CLIST(vfi->clist)->clist_window) return TRUE;

	gtk_clist_get_selection_info(GTK_CLIST(vfi->clist), bevent->x, bevent->y, &row, &col);

	fd = vficon_find_info(vfi, row, col);

	if (!vfi->mouse_in_drag)
		{
		tip_update(vfi, fd);
		return TRUE;
		}

	if (!fd || fd == vfi->mouse_current) return TRUE;

	gtk_clist_freeze(GTK_CLIST(vfi->clist));

	if (vfi->mouse_add)
		{
		vficon_select_region_util(vfi, vfi->mouse_start, vfi->mouse_current, FALSE);
		}

	vfi->mouse_current = fd;
	vficon_select_region_util(vfi, vfi->mouse_start, vfi->mouse_current, vfi->mouse_add);

	gtk_clist_thaw(GTK_CLIST(vfi->clist));

	return TRUE;
}

static void vficon_popup_destroy_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;
	vficon_cell_draw_active(vfi, vfi->click_fd, FALSE);
	vfi->click_fd = NULL;
	vfi->popup = NULL;
}

static gint vficon_button_press(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	ViewFileIcon *vfi = data;
	gint row, col;
	FileData *fd;

	tip_unschedule(vfi);

	if (bevent->window != GTK_CLIST(vfi->clist)->clist_window) return TRUE;

	gtk_clist_get_selection_info(GTK_CLIST(vfi->clist), bevent->x, bevent->y, &row, &col);
	fd = vficon_find_info(vfi, row, col);

	switch (bevent->button)
		{
		case 1:
			if (!fd) return TRUE;

			vfi->mouse_in_drag = TRUE;

			vficon_move_focus(vfi, row, col, FALSE);
			GTK_CLIST(vfi->clist)->focus_row = vfi->focus_row;

			gdk_pointer_grab(GTK_CLIST(vfi->clist)->clist_window, FALSE,
					 GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
					 NULL, NULL, bevent->time);
                        gtk_grab_add(vfi->clist);

			gtk_clist_freeze(GTK_CLIST(vfi->clist));

			if (bevent->state & GDK_CONTROL_MASK)
				{
				vfi->mouse_add = !vficon_is_selected(vfi, fd);
				if ((bevent->state & GDK_SHIFT_MASK) && vfi->prev_selection)
					{
					vficon_select_region_util(vfi, vfi->prev_selection, fd, vfi->mouse_add);
					}
				else
					{
					vficon_select_util(vfi, fd, vfi->mouse_add);
					}
				}
			else if (bevent->state & GDK_SHIFT_MASK)
				{
				vfi->mouse_add = TRUE;
				vficon_select_none(vfi);

				if (vfi->prev_selection)
					{
					vficon_select_region_util(vfi, vfi->prev_selection, fd, TRUE);
					}
				else
					{
					vficon_select_util(vfi, fd, TRUE);
					}
				}
			else
				{
				vfi->mouse_add = TRUE;
				vficon_select_none(vfi);

				vficon_select_util(vfi, fd, TRUE);
				}

			gtk_clist_thaw(GTK_CLIST(vfi->clist));

			vfi->mouse_start = fd;
			vfi->mouse_current = fd;

			break;
		case 2:
		case 3:
			vfi->click_fd = fd;
			if (bevent->button == 3)
				{
				vficon_cell_draw_active(vfi, vfi->click_fd, TRUE);
				tip_unschedule(vfi);

				vfi->popup = vficon_pop_menu(vfi, (fd != NULL));
				gtk_signal_connect(GTK_OBJECT(vfi->popup), "destroy",
						   GTK_SIGNAL_FUNC(vficon_popup_destroy_cb), vfi);
				gtk_menu_popup(GTK_MENU(vfi->popup), NULL, NULL, NULL, NULL, bevent->button, bevent->time);
				}
			break;
		default:
			break;
		}
	return TRUE;
}

static gint vficon_button_release(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	ViewFileIcon *vfi = data;

	tip_schedule(vfi);

	if (!vfi->mouse_in_drag) return FALSE;

	gtk_grab_remove(vfi->clist);
	gdk_pointer_ungrab(bevent->time);

	vfi->mouse_in_drag = FALSE;

	if (vficon_is_selected(vfi, vfi->focus_fd)) vficon_send_layout_select(vfi, vfi->focus_fd);

	return TRUE;
}

static gint vficon_button_leave(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	ViewFileIcon *vfi = data;

	tip_unschedule(vfi);
	return FALSE;
}

/*
 *-------------------------------------------------------------------
 * population
 *-------------------------------------------------------------------
 */

static void vficon_icon_update(ViewFileIcon *vfi, FileData *fd, gint row, gint col)
{
	IconData *id = ICON_DATA(fd);

	if (id && id->pixmap)
		{
		gint hs, vs;

		gdk_window_get_size(id->pixmap, &hs, &vs);
		hs = (thumb_max_width + 10 - hs) / 2;
		vs = (thumb_max_height + 6 - vs) / 2;
		gtk_clist_set_shift(GTK_CLIST(vfi->clist), row, col, 0, hs);
		gtk_clist_set_pixmap(GTK_CLIST(vfi->clist), row, col, id->pixmap, id->mask);
		}
	else
		{
		gtk_clist_set_text(GTK_CLIST(vfi->clist), row, col, NULL);
		}
}

static void vficon_add_icon(ViewFileIcon *vfi, FileData *fd, gint row, gint col)
{
	GList *temp;

	vficon_icon_update(vfi, fd, row, col);

	temp = gtk_clist_get_row_data(GTK_CLIST(vfi->clist), row);
	temp = g_list_nth(temp, col);
	temp->data = fd;

	vficon_update_cell(vfi, row, col, vficon_is_selected(vfi, fd));
}

static gint vficon_add_row(ViewFileIcon *vfi)
{
	gchar *text[VFICON_TEXT_COLUMNS];
	GList *temp = NULL;
	gint i;
	gint row;

	for (i = 0; i < VFICON_TEXT_COLUMNS; i++) text[i] = NULL;

	row = gtk_clist_append(GTK_CLIST(vfi->clist), text);

	for (i = 0; i < vfi->columns; i++) temp = g_list_prepend(temp, NULL);
			
	gtk_clist_set_row_data_full(GTK_CLIST(vfi->clist), row, temp, (GtkDestroyNotify)g_list_free);

	return row;
}

#define CLIST_FRAME_SIZE 4

static void vficon_populate(ViewFileIcon *vfi, gint resize, gint keep_position)
{
	GtkCList *clist;
	gint col, row;
	GList *work;
	gint visible_row = -1;
	gfloat visible_offset = 0.0;

	vficon_verify_selections(vfi);

	clist = GTK_CLIST(vfi->clist);

	if (keep_position)
		{
		gtk_clist_get_selection_info(clist, 0, 0, &visible_row, NULL);
		if (clist->row_height > 0 && clist->clist_window_height > 0 && visible_row < clist->rows - 1)
			{
			gint icon_offset;

			/* ugly, but it maintains exact scroll offset when refreshing */

			icon_offset = clist->row_height + (clist->voffset % (clist->row_height + 1));
			visible_offset = (gfloat)icon_offset / (clist->clist_window_height - clist->row_height - CLIST_FRAME_SIZE);

			visible_row++;
			}
		}

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);

	if (resize)
		{
		gint i;
		gtk_clist_set_row_height(clist, thumb_max_height + 6);
		for (i = 0; i < VFICON_MAX_COLUMNS; i++)
			{
			gtk_clist_set_column_resizeable(clist, i, FALSE);
			gtk_clist_set_column_justification(clist, i, GTK_JUSTIFY_LEFT);
			if (i < vfi->columns)
				{
				gtk_clist_set_column_visibility(clist, i, TRUE);
				gtk_clist_set_column_auto_resize(clist, i, FALSE);
				gtk_clist_set_column_width(clist, i, thumb_max_width + 10);
				}
			else
				{
				gtk_clist_set_column_visibility(clist, i, FALSE);
				}
			}
		}

	col = 0;
	row = -1;
	work = vfi->list;
	while (work)
		{
		FileData *fd = work->data;

		if (col == 0)
			{
			row = vficon_add_row(vfi);
			}

		vficon_add_icon(vfi, fd, row, col);

		work = work->next;
		col++;
		if (col >= vfi->columns) col = 0;
		}

	gtk_clist_thaw(clist);

	if (keep_position && visible_row > 0)
		{
		gtk_clist_moveto(clist, visible_row, -1, visible_offset, 0.0);
		}

	vfi->rows = row + 1;

	vficon_send_update(vfi);

	vficon_thumb_update(vfi);
}

static void vficon_populate_at_new_size(ViewFileIcon *vfi, gint w, gint h)
{
	gint new_cols;

	new_cols = (w  - 6) / (thumb_max_width + 10 + 6 + 1);
	if (new_cols < 1) new_cols = 1;

	if (new_cols == vfi->columns) return;

	vfi->columns = new_cols;

	vficon_populate(vfi, TRUE, TRUE);

	if (debug) printf("col tab pop cols=%d rows=%d\n", vfi->columns, vfi->rows);
}

static void vficon_sync(ViewFileIcon *vfi)
{
	GList *work;
	gint r, c;

	if (vfi->frozen || vfi->rows == 0) return;

	r = c = 0;

	gtk_clist_freeze(GTK_CLIST(vfi->clist));
	work = vfi->list;
	while (work)
		{
		FileData *fd;

		fd = work->data;
		
		if (vficon_find_position(vfi, fd, &r, &c))
			{
			vficon_add_icon(vfi, fd, r, c);
			c++;
			}
		work = work->next;
		}

	/* sanity checks, left over cells need to be NULL'd and excess rows removed */
	if (r == 0 && c == 0 && GTK_CLIST(vfi->clist)->rows > 0)
		{
		vfi->rows = 0;
		gtk_clist_remove(GTK_CLIST(vfi->clist), r);
		}
	else if (c < vfi->columns)
		{
		while (c < vfi->columns)
			{
			vficon_add_icon(vfi, NULL, r, c);
			c++;
			}
		}
	if (r < GTK_CLIST(vfi->clist)->rows - 1)
		{
		r++;
		vfi->rows = r;
		while (r < GTK_CLIST(vfi->clist)->rows)
			{
			gtk_clist_remove(GTK_CLIST(vfi->clist), r);
			r++;
			}
		}

	vficon_update_focus(vfi);
	gtk_clist_thaw(GTK_CLIST(vfi->clist));
}

static void vficon_select_row_cb(GtkCList *clist, gint row, gint column,
				 GdkEvent *event, gpointer data)
{
	gtk_clist_unselect_row(clist, row, column);
}

static void vficon_sized_cb(GtkWidget *widget, GtkAllocation *allocation, gpointer data)
{
	ViewFileIcon *vfi = data;

	vficon_populate_at_new_size(vfi, allocation->width, allocation->height);
}

/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */

void vficon_sort_set(ViewFileIcon *vfi, SortType type, gint ascend)
{
	if (vfi->sort_method == type && vfi->sort_ascend == ascend) return;

	vfi->sort_method = type;
	vfi->sort_ascend = ascend;

	if (!vfi->list) return;

	vfi->list = filelist_sort(vfi->list, vfi->sort_method, vfi->sort_ascend);

	gtk_clist_freeze(GTK_CLIST(vfi->clist));
	vficon_sync(vfi);
	gtk_clist_thaw(GTK_CLIST(vfi->clist));
}

/*
 *-----------------------------------------------------------------------------
 * thumb updates
 *-----------------------------------------------------------------------------
 */

static gint vficon_thumb_next(ViewFileIcon *vfi);

static void vficon_thumb_status(ViewFileIcon *vfi, gfloat val, const gchar *text)
{
	if (vfi->func_thumb_status)
		{
		vfi->func_thumb_status(vfi, val, text, vfi->data_thumb_status);
		}
}

static void vficon_thumb_cleanup(ViewFileIcon *vfi)
{
	vficon_thumb_status(vfi, 0.0, NULL);

	g_list_free(vfi->thumbs_list);
	vfi->thumbs_list = NULL;
	vfi->thumbs_count = 0;
	vfi->thumbs_running = FALSE;

	thumb_loader_free(vfi->thumbs_loader);
	vfi->thumbs_loader = NULL;

	vfi->thumbs_fd = NULL;
}

static void vficon_thumb_stop(ViewFileIcon *vfi)
{
	if (vfi->thumbs_running) vficon_thumb_cleanup(vfi);
}

static void vficon_thumb_do(ViewFileIcon *vfi, ThumbLoader *tl, FileData *fd)
{
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	gint r, c;

	if (!fd) return;

	if (thumb_loader_to_pixmap(tl, &pixmap, &mask) < 0)
		{
		/* unknown */
		thumb_from_xpm_d(img_unknown, thumb_max_width, thumb_max_height, &pixmap, &mask);
		}

	icondata_set_thumb(ICON_DATA(fd), pixmap, mask);
	if (pixmap) gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);

	if (vficon_find_position(vfi, fd, &r, &c))
		{
		vficon_add_icon(vfi, fd, r, c);
		}

	vficon_thumb_status(vfi, (gfloat)(vfi->thumbs_count) / g_list_length(vfi->list), _("Loading thumbs..."));
}

static void vficon_thumb_error_cb(ThumbLoader *tl, gpointer data)
{
	ViewFileIcon *vfi = data;

	if (vfi->thumbs_fd && vfi->thumbs_loader == tl)
		{
		vficon_thumb_do(vfi, NULL, vfi->thumbs_fd);
		}

	while (vficon_thumb_next(vfi));
}

static void vficon_thumb_done_cb(ThumbLoader *tl, gpointer data)
{
	ViewFileIcon *vfi = data;

	if (vfi->thumbs_fd && vfi->thumbs_loader == tl)
		{
		vficon_thumb_do(vfi, tl, vfi->thumbs_fd);
		}

	while (vficon_thumb_next(vfi));
}

static gint vficon_thumb_next(ViewFileIcon *vfi)
{
	FileData *fd;
	gint p = -1;
	gint r = -1;
	gint c = -1;

	gtk_clist_get_selection_info(GTK_CLIST(vfi->clist), 1, 1, &r, &c);

	/* find first visible undone */

	if (r != -1)
		{
		GList *work;
		gint n, col;

		col = 0;
		n = r * vfi->columns;

		work = g_list_nth(vfi->thumbs_list, n);
		while (work)
			{
			if (gtk_clist_row_is_visible(GTK_CLIST(vfi->clist), r) != GTK_VISIBILITY_NONE)
				{
				if (!GPOINTER_TO_INT(work->data))
					{
					work->data = GINT_TO_POINTER(TRUE);
					p = n;
					work = NULL;
					}
				else
					{
					col++;
					if (col >= vfi->columns)
						{
						col = 0;
						r++;
						}
					work = work->next;
					}
				}
			else
				{
				work = NULL;
				}
			n++;
			}
		}

	/* then find first undone */

	if (p == -1)
		{
		GList *work = vfi->thumbs_list;
		r = 0;
		while (work && p == -1)
			{
			if (!GPOINTER_TO_INT(work->data))
				{
				p = r;
				work->data = GINT_TO_POINTER(TRUE);
				}
			if (p == -1)
				{
				r++;
				work = work->next;
				if (!work)
					{
					vficon_thumb_cleanup(vfi);
					return FALSE;
					}
				}
			}
		}

	vfi->thumbs_count++;

	fd = g_list_nth_data(vfi->list, p);

	if (fd)
		{
		if (ICON_DATA(fd)->pixmap != NULL) return TRUE;

		vfi->thumbs_fd = fd;

		thumb_loader_free(vfi->thumbs_loader);

		vfi->thumbs_loader = thumb_loader_new(fd->path, thumb_max_width, thumb_max_height);
		thumb_loader_set_error_func(vfi->thumbs_loader, vficon_thumb_error_cb, vfi);

		if (!thumb_loader_start(vfi->thumbs_loader, vficon_thumb_done_cb, vfi))
			{
			/* set icon to unknown, continue */
			if (debug) printf("thumb loader start failed %s\n", vfi->thumbs_loader->path);
			vficon_thumb_do(vfi, NULL, fd);
			return TRUE;
			}
		return FALSE;
		}
	else
		{
		/* well, this should not happen, but handle it */
		if (debug) printf("thumb walked past end\n");
		vficon_thumb_cleanup(vfi);
		return FALSE;
		}

	return TRUE;
}

static void vficon_thumb_update(ViewFileIcon *vfi)
{
	gint i;
	GtkCList *clist;
	gint count;

	vficon_thumb_stop(vfi);

	clist = GTK_CLIST(vfi->clist);

	vficon_thumb_status(vfi, 0.0, _("Loading thumbs..."));

	count = vficon_count(vfi, NULL);
	for (i = 0; i < count; i++)
		{
		vfi->thumbs_list = g_list_prepend(vfi->thumbs_list, GINT_TO_POINTER(FALSE));
		}

	vfi->thumbs_running = TRUE;

	while (vficon_thumb_next(vfi));
}

/*
 *-----------------------------------------------------------------------------
 * row stuff
 *-----------------------------------------------------------------------------
 */

FileData *vficon_index_get_data(ViewFileIcon *vfi, gint row)
{
	return FILE_DATA(g_list_nth_data(vfi->list, row));
}

gchar *vficon_index_get_path(ViewFileIcon *vfi, gint row)
{
	FileData *fd;

	fd = g_list_nth_data(vfi->list, row);

	return (fd ? fd->path : NULL);
}

gint vficon_index_by_path(ViewFileIcon *vfi, const gchar *path)
{
	gint p = 0;
	GList *work;

	if (!path) return -1;

	work = vfi->list;
	while (work)
		{
		FileData *fd = work->data;
		if (strcmp(path, fd->path) == 0) return p;
		work = work->next;
		p++;
		}

	return -1;
}

gint vficon_count(ViewFileIcon *vfi, gint *bytes)
{
	if (bytes)
		{
		gint b = 0;
		GList *work;

		work = vfi->list;
		while (work)
			{
			FileData *fd = work->data;
			work = work->next;
			b += fd->size;
			}

		*bytes = b;
		}

	return g_list_length(vfi->list);
}

GList *vficon_get_list(ViewFileIcon *vfi)
{
	GList *list = NULL;
	GList *work;

	work = vfi->list;
	while (work)
		{
		FileData *fd = work->data;
		work = work->next;

		list = g_list_prepend(list, g_strdup(fd->path));
		}

	return g_list_reverse(list);
}

/*
 *-----------------------------------------------------------------------------
 *
 *-----------------------------------------------------------------------------
 */

static gint vficon_refresh_real(ViewFileIcon *vfi, gint keep_position)
{
	gint ret = TRUE;
	GList *old_list;
	GList *work;
	FileData *focus_fd;
	gint fr, fc;

	focus_fd = vficon_find_info(vfi, vfi->focus_row, vfi->focus_column);

	old_list = vfi->list;
	vfi->list = NULL;

	if (vfi->path)
		{
		ret = iconlist_read(vfi->path, &vfi->list);
		}

	/* check for same files from old_list */
	work = old_list;
	while (work)
		{
		FileData *fd;
		GList *needle;

		fd = work->data;
		needle = vfi->list;
		while (needle)
			{
			FileData *fdn = needle->data;
			if (strcmp(fd->name, fdn->name) == 0 && fd->date == fdn->date)
				{
				/* swap, to retain old thumbs */
				needle->data = fd;
				work->data = fdn;
				needle = NULL;
				}
			else
				{
				needle = needle->next;
				}
			}

		work = work->next;
		}

	vfi->list = filelist_sort(vfi->list, vfi->sort_method, vfi->sort_ascend);

	vficon_populate(vfi, TRUE, keep_position);

	/* attempt to keep focus on same icon when refreshing */
	if (focus_fd && vficon_find_position(vfi, focus_fd, &fr, &fc))
		{
		vficon_move_focus(vfi, fr, fc, FALSE);
		}

	iconlist_free(old_list);

	return ret;
}

gint vficon_refresh(ViewFileIcon *vfi)
{
	return vficon_refresh_real(vfi, TRUE);
}

/*
 *-----------------------------------------------------------------------------
 * base
 *-----------------------------------------------------------------------------
 */

gint vficon_set_path(ViewFileIcon *vfi, const gchar *path)
{
	gint ret;

	if (!path) return FALSE;
	if (vfi->path && strcmp(path, vfi->path) == 0) return TRUE;

	g_free(vfi->path);
	vfi->path = g_strdup(path);

	g_list_free(vfi->selection);
	vfi->selection = NULL;

	iconlist_free(vfi->list);
	vfi->list = NULL;

	/* NOTE: populate will clear the clist for us */
	ret = vficon_refresh_real(vfi, FALSE);

	vfi->focus_fd = NULL;
	vficon_move_focus(vfi, 0, 0, FALSE);

	return ret;
}

static void vficon_destroy_cb(GtkWidget *widget, gpointer data)
{
	ViewFileIcon *vfi = data;

	if (vfi->popup)
		{
		gtk_signal_disconnect_by_data(GTK_OBJECT(vfi->popup), vfi);
		}

	tip_unschedule(vfi);

	vficon_thumb_cleanup(vfi);

	g_free(vfi->path);

	iconlist_free(vfi->list);
	g_list_free(vfi->selection);
	g_free(vfi);
}

ViewFileIcon *vficon_new(const gchar *path)
{
	ViewFileIcon *vfi;

	vfi = g_new0(ViewFileIcon, 1);

	vfi->path = NULL;
	vfi->sort_method = SORT_NAME;
	vfi->sort_ascend = TRUE;

	vfi->selection = NULL;
	vfi->prev_selection = NULL;

	vfi->tip_window = NULL;
	vfi->tip_delay_id = -1;

	vfi->focus_row = 0;
	vfi->focus_column = 0;
	vfi->focus_fd = NULL;

	vfi->scroll_timeout_id = -1;

	vfi->frozen = FALSE;

	vfi->popup = NULL;

	vfi->widget = gtk_scrolled_window_new(NULL, NULL);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(vfi->widget),
				       GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_signal_connect(GTK_OBJECT(vfi->widget), "destroy",
			   GTK_SIGNAL_FUNC(vficon_destroy_cb), vfi);

	vfi->clist = gtk_clist_new(VFICON_MAX_COLUMNS);

	gtk_clist_set_button_actions(GTK_CLIST(vfi->clist), 0, GTK_BUTTON_SELECTS);
	gtk_clist_set_button_actions(GTK_CLIST(vfi->clist), 1, GTK_BUTTON_IGNORED);
	gtk_clist_set_button_actions(GTK_CLIST(vfi->clist), 2, GTK_BUTTON_IGNORED);

	/* we roll our own selection, connect to select row, then unselect it */
	gtk_clist_set_selection_mode(GTK_CLIST(vfi->clist), GTK_SELECTION_SINGLE);
	gtk_signal_connect(GTK_OBJECT(vfi->clist), "select_row",
			   GTK_SIGNAL_FUNC(vficon_select_row_cb), vfi);

	gtk_container_add(GTK_CONTAINER(vfi->widget), vfi->clist);
	gtk_widget_show(vfi->clist);
	
	gtk_signal_connect(GTK_OBJECT(vfi->clist), "size_allocate",
			   GTK_SIGNAL_FUNC(vficon_sized_cb), vfi);

	gtk_widget_set_events(vfi->clist, GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK |
			      GDK_BUTTON_PRESS_MASK | GDK_LEAVE_NOTIFY_MASK);
	gtk_signal_connect(GTK_OBJECT(vfi->clist),"motion_notify_event",
			   GTK_SIGNAL_FUNC(vficon_button_motion), vfi);
	gtk_signal_connect(GTK_OBJECT(vfi->clist),"button_press_event",
                           GTK_SIGNAL_FUNC(vficon_button_press), vfi);
	gtk_signal_connect(GTK_OBJECT(vfi->clist),"button_release_event",
			   GTK_SIGNAL_FUNC(vficon_button_release), vfi);
	gtk_signal_connect(GTK_OBJECT(vfi->clist),"leave_notify_event",
			   GTK_SIGNAL_FUNC(vficon_button_leave), vfi);

	gtk_signal_connect(GTK_OBJECT(vfi->clist),"key_press_event",
			   GTK_SIGNAL_FUNC(vficon_keypress_cb), vfi);

	vficon_dnd_init(vfi);

	/* force vfi->columns to be at least 1 (sane) - this will be corrected in the size_cb */
	vficon_populate_at_new_size(vfi, 1, 1);

	if (path) vficon_set_path(vfi, path);

	return vfi;
}

void vficon_set_status_func(ViewFileIcon *vfi,
			    void (*func)(ViewFileIcon *vfi, gpointer data), gpointer data)
{
	vfi->func_status = func;
	vfi->data_status = data;
}

void vficon_set_thumb_status_func(ViewFileIcon *vfi,
				  void (*func)(ViewFileIcon *vfi, gfloat val, const gchar *text, gpointer data),
				  gpointer data)
{
	vfi->func_thumb_status = func;
	vfi->data_thumb_status = data;
}

void vficon_set_layout(ViewFileIcon *vfi, LayoutWindow *layout)
{
	vfi->layout = layout;
}

/*
 *-----------------------------------------------------------------------------
 * maintenance (for rename, move, remove)
 *-----------------------------------------------------------------------------
 */

static gint vficon_maint_find_closest(ViewFileIcon *vfi, gint row, gint count, GList *ignore_list)
{
	GList *list = NULL;
	GList *work;
	gint rev = row - 1;
	row ++;

	work = ignore_list;
	while (work)
		{
		gint f = vficon_index_by_path(vfi, work->data);
		if (f >= 0) list = g_list_prepend(list, GINT_TO_POINTER(f));
		work = work->next;
		}

	while (list)
		{
		gint c = TRUE;
		work = list;
		while (work && c)
			{
			gpointer p = work->data;
			work = work->next;
			if (row == GPOINTER_TO_INT(p))
				{
				row++;
				c = FALSE;
				}
			if (rev == GPOINTER_TO_INT(p))
				{
				rev--;
				c = FALSE;
				}
			if (!c) list = g_list_remove(list, p);
			}
		if (c && list)
			{
			g_list_free(list);
			list = NULL;
			}
		}
	if (row > count - 1)
		{
		if (rev < 0)
			return -1;
		else
			return rev;
		}
	else
		{
		return row;
		}
}

void vficon_maint_renamed(ViewFileIcon *vfi, const gchar *source, const gchar *dest)
{
	gint row;
	gchar *source_base;
	gchar *dest_base;
	GList *work;
	FileData *fd;

	row = vficon_index_by_path(vfi, source);
	if (row < 0) return;

	source_base = remove_level_from_path(source);
	dest_base = remove_level_from_path(dest);

	work = g_list_nth(vfi->list, row);
	fd = work->data;

	if (strcmp(source_base, dest_base) == 0)
		{
		vfi->list = g_list_remove(vfi->list, fd);
		g_free(fd->path);

		fd->path = g_strdup(dest);
		fd->name = filename_from_path(fd->path);

		vfi->list = filelist_insert_sort(vfi->list, fd, vfi->sort_method, vfi->sort_ascend);

		vficon_sync(vfi);
		}
	else
		{
		vficon_maint_removed(vfi, source, NULL);
		}

	g_free(source_base);
	g_free(dest_base);
}

void vficon_maint_removed(ViewFileIcon *vfi, const gchar *path, GList *ignore_list)
{
	FileData *fd;
	gint row;
	gint new_row = -1;

	row = vficon_index_by_path(vfi, path);
	if (row < 0) return;

	fd = g_list_nth_data(vfi->list, row);
	if (!fd) return;

	if (vficon_index_is_selected(vfi, row) &&
	    layout_image_get_collection(vfi->layout, NULL) == NULL)
		{
		vficon_unselect(vfi, fd);

		if (!vfi->selection)
			{
			gint n;

			n = vficon_count(vfi, NULL);
			if (ignore_list)
				{
				new_row = vficon_maint_find_closest(vfi, row, n, ignore_list);
				if (debug) printf("row = %d, closest is %d\n", row, new_row);
				}
			else
				{
				if (row + 1 < n)
					{
					new_row = row + 1;
					}
				else if (row > 0)
					{
					new_row = row - 1;
					}
				}
			}
		else if (ignore_list)
			{
			GList *work;

			work = vfi->selection;
			while (work)
				{
				FileData *ignore_fd;
				GList *tmp;
				gint match = FALSE;

				ignore_fd = work->data;
				work = work->next;

				tmp = ignore_list;
				while (tmp && !match)
					{
					const gchar *ignore_path;

					ignore_path = tmp->data;
					tmp = tmp->next;

					if (strcmp(ignore_fd->path, ignore_path) == 0)
						{
						match = TRUE;
						}
					}
				if (!match)
					{
					new_row = g_list_index(vfi->list, ignore_fd);
					work = NULL;
					}
				}
			if (new_row == -1)
				{
				/* selection all ignored, use closest */
				new_row = vficon_maint_find_closest(vfi, row, vficon_count(vfi, NULL), ignore_list);
				}
			}
		else
			{
			new_row = g_list_index(vfi->list, vfi->selection->data);
			}
		if (new_row >= 0)
			{
			FileData *fdn;

			fdn = g_list_nth_data(vfi->list, new_row);
			vficon_select(vfi, fdn);
			vficon_send_layout_select(vfi, fdn);
			}
		}

	vfi->list = g_list_remove(vfi->list, fd);
	icondata_set_thumb(ICON_DATA(fd), NULL, NULL);
	file_data_free(fd);

	vficon_sync(vfi);
	vficon_send_update(vfi);
}

void vficon_maint_moved(ViewFileIcon *vfi, const gchar *source, const gchar *dest, GList *ignore_list)
{
	gchar *buf;

	if (!source || !vfi->path) return;

	buf = remove_level_from_path(source);

	if (strcmp(buf, vfi->path) == 0)
		{
		vficon_maint_removed(vfi, source, ignore_list);
		}

	g_free(buf);
}

