/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif
#include "intl.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "ui_help.h"

/*
 * well, the GtkText widget sucks, and I hate line wrapping,
 * so if you use less than 640x480 you've got problems.
 * these defaults are my preferences, they should be ok for most.
 */

#define HELP_WINDOW_WIDTH 600
#define HELP_WINDOW_HEIGHT 350

#define HELP_WINDOW_FONT "-adobe-courier-medium-r-normal--*-120-*-*-*-*-*-*"


/*
 *-----------------------------------------------------------------------------
 * 'help' window
 *-----------------------------------------------------------------------------
 */

#define SEARCH_BUFFER_LENGTH 1024

static gint text_buffer_count_lines(const gchar *buf)
{
	const gchar *ptr = buf;
	gint l = 0;

	if (!buf) return 0;

	while(*ptr != '\0')
		{
		if (*ptr == '\n') l++;
		ptr++;
		}
	return l;
}

static gint text_lines(GtkText *text)
{
	gint l;
	gint i;
	gint line_count = 1;

	l = gtk_text_get_length(text);

	for (i = 0; i < l; i += SEARCH_BUFFER_LENGTH)
		{
		gint end;

		end = i + SEARCH_BUFFER_LENGTH;
		if (end > l - 1) end = l - 1;

		if (end - i > 0)
			{
			gchar *buf;
			buf = gtk_editable_get_chars(GTK_EDITABLE(text), i, end);
			line_count += text_buffer_count_lines(buf);
			g_free(buf);
			}
		}
	return line_count;
}

/* NULL needle return 0, but counts the line total */
static gint text_find(GtkText *haystack, const gchar *needle, gint *line)
{
	gint l;
	gint i;
	gint n = 0;
	gint line_count = 1;

	if (needle) n = strlen(needle);

	l = gtk_text_get_length(haystack);

	for (i = 0; i < l; i += SEARCH_BUFFER_LENGTH)
		{
		gint end;
		gint found = -1;

		end = i + SEARCH_BUFFER_LENGTH + n;
		if (end > l - 1) end = l - 1;

		if (end - i > 0)
			{
			gchar *buf;
			gchar *s = NULL;

			buf = gtk_editable_get_chars(GTK_EDITABLE(haystack), i, end);
			if (buf)
				{
				if (needle) s = strstr(buf, needle);
				if (line) line_count += text_buffer_count_lines(buf);
				}
			if (s) found = i + (s - buf);

			g_free(buf);
			}
		if (found >= 0)
			{
			if (line) *line = line_count;
			return found;
			}
		}

	if (line) *line = line_count;
	return 0;
}

static void text_set_position_from_lines(GtkText *text, gint line, gint line_count)
{
	GtkAdjustment *adj;
        gfloat value;
        
	if (line_count < 2 || line >= line_count) return;

	adj = text->vadj;

	value = adj->upper * ((float)(line - 1) / (float)(line_count - 1));
	gtk_adjustment_set_value(adj, value);
}

static void help_window_scroll(GtkWidget *text, const gchar *key)
{
	gchar *needle;
	gint p;
	gint line;

	if (!key) return;

	needle = g_strdup_printf("[section:%s]", key);

	p = text_find(GTK_TEXT(text), needle, &line);

	if (p > 0 && p != gtk_editable_get_position(GTK_EDITABLE(text)))
		{
		gtk_text_freeze(GTK_TEXT(text));
		/* well, GtkText broken, freeze does not work for set_position :( */
		text_set_position_from_lines(GTK_TEXT(text), line, text_lines(GTK_TEXT(text)));

		/* above not entirely accurate :( do the set anyway */
		gtk_editable_set_position(GTK_EDITABLE(text), p);

		gtk_text_thaw(GTK_TEXT(text));
		}

	g_free(needle);
}

static void help_window_load_text(GtkWidget *text, const gchar *path)
{
	FILE *f;
	gchar s_buf[1024];
	gint l;
	GdkFont *font;

	if (!text || !path) return;

	gtk_text_freeze(GTK_TEXT(text));
	l = gtk_text_get_length(GTK_TEXT(text));
	if (l > 0) gtk_editable_delete_text(GTK_EDITABLE(text), 0, l);

	font = gdk_font_load(HELP_WINDOW_FONT);

	f = fopen(path, "r");
	if (!f)
		{
		gchar *buf;
		buf = g_strdup_printf(_("Unable to load:\n%s"), path);
		gtk_text_insert(GTK_TEXT(text), font, NULL, NULL, buf, strlen(buf));
		g_free(buf);
		}
	else
		{
		while (fgets(s_buf, sizeof(s_buf), f))
			{
			gtk_text_insert(GTK_TEXT(text), font, NULL, NULL, s_buf, strlen(s_buf));
			}
		fclose(f);
		}


	if (font) gdk_font_unref(font);

	gtk_text_thaw(GTK_TEXT(text));
}

static gint help_window_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	gtk_widget_destroy(widget);
	return TRUE;
}

static void help_window_close(GtkWidget *widget, gpointer data)
{
	GtkWidget *window = data;
	gtk_widget_destroy(window);
}

void help_window_set_key(GtkWidget *window, const gchar *key)
{
	GtkWidget *text;

	if (!window) return;

	text = gtk_object_get_data(GTK_OBJECT(window), "text_widget");
	if (!text) return;

	gdk_window_raise(window->window);

	if (key) help_window_scroll(text, key);
}

void help_window_set_file(GtkWidget *window, const gchar *path, const gchar *key)
{
	GtkWidget *text;

	if (!window || !path) return;

	text = gtk_object_get_data(GTK_OBJECT(window), "text_widget");
	if (!text) return;

	gdk_window_raise(window->window);

	help_window_load_text(text, path);
	if (key) help_window_scroll(text, key);
}

GtkWidget *help_window_new(const gchar *title,
			   const gchar *wmclass, const gchar *subclass,
			   const gchar *path, const gchar *key)
{
	GtkWidget *window;
	GtkWidget *text;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *button;
	GtkWidget *scroll;

	/* window */

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(window), FALSE, TRUE, FALSE);
	gtk_container_border_width(GTK_CONTAINER(window), 5);
	gtk_window_set_wmclass(GTK_WINDOW(window), subclass, wmclass);

	gtk_widget_set_usize(window, HELP_WINDOW_WIDTH, HELP_WINDOW_HEIGHT);

	gtk_window_set_title(GTK_WINDOW(window), title);

	gtk_signal_connect(GTK_OBJECT(window), "delete_event",
			   GTK_SIGNAL_FUNC(help_window_delete_cb), NULL);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(window), vbox);
	gtk_widget_show(vbox);

	gtk_object_set_data(GTK_OBJECT(window), "text_vbox", vbox);

	/* text window */

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show(hbox);

	text = gtk_text_new(NULL, NULL);
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), text, TRUE, TRUE, 0);
	gtk_widget_show(text);

	scroll = gtk_vscrollbar_new(GTK_TEXT(text)->vadj);
	gtk_box_pack_start(GTK_BOX(hbox), scroll, FALSE, FALSE, 0);
	gtk_widget_show(scroll);

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 0);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbox), 95, 40);
	gtk_box_pack_end(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	button = gtk_button_new_with_label(_("Close"));
	gtk_signal_connect (GTK_OBJECT (button), "clicked", help_window_close, window);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);
	gtk_widget_show(button);

	gtk_object_set_data(GTK_OBJECT(window), "text_widget", text);

	help_window_load_text(text, path);

	gtk_widget_show(window);

	if (key) help_window_scroll(text, key);

	return window;
}

GtkWidget *help_window_get_box(GtkWidget *window)
{
	return gtk_object_get_data(GTK_OBJECT(window), "text_vbox");
}

