/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "layout_util.h"

#include "cache_maint.h"
#include "collect.h"
#include "collect-dlg.h"
#include "dupe.h"
#include "layout_image.h"
#include "preferences.h"
#include "utilops.h"
#include "ui_fileops.h"
#include "ui_menu.h"
#include "ui_tabcomp.h"

#include <gdk/gdkkeysyms.h> /* for keyboard values */

#include "icons/icon_thumb.xpm"
#include "icons/icon_home.xpm"
#include "icons/icon_reload.xpm"
#include "icons/icon_zoom_in.xpm"
#include "icons/icon_zoom_out.xpm"
#include "icons/icon_zoom_fit.xpm"
#include "icons/icon_zoom_norm.xpm"
#include "icons/icon_config.xpm"
#include "icons/icon_float.xpm"

#include "icons/folder_close.xpm"
#include "icons/folder_open.xpm"
#include "icons/folder_deny.xpm"


#define MENU_EDIT_ACTION_OFFSET 16


/*
 *-----------------------------------------------------------------------------
 * keyboard handler
 *-----------------------------------------------------------------------------
 */

static gint layout_key_press_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	LayoutWindow *lw = data;
	gint stop_signal = FALSE;
	gint x = 0;
	gint y = 0;

	if (lw->path_entry && GTK_WIDGET_HAS_FOCUS(lw->path_entry))
		{
		if (event->keyval == GDK_Escape && lw->path)
			{
			gtk_entry_set_text(GTK_ENTRY(lw->path_entry), lw->path);
			}
		return stop_signal;
		}

	if (lw->image && GTK_WIDGET_HAS_FOCUS(lw->image->widget))
		{
		switch (event->keyval)
			{
			case GDK_Left: case GDK_KP_Left:
				x -= 1;
				stop_signal = TRUE;
				break;
			case GDK_Right: case GDK_KP_Right:
				x += 1;
				stop_signal = TRUE;
				break;
			case GDK_Up: case GDK_KP_Up:
				y -= 1;
				stop_signal = TRUE;
				break;
			case GDK_Down: case GDK_KP_Down:
				y += 1;
				stop_signal = TRUE;
				break;
			case GDK_BackSpace:
			case 'B': case 'b':
				layout_image_prev(lw);
				stop_signal = TRUE;
				break;
			case GDK_space:
				layout_image_next(lw);
				stop_signal = TRUE;
				break;
			}
		}

	if ( !(event->state & GDK_CONTROL_MASK) )
	    switch (event->keyval)
		{
		case '+': case GDK_KP_Add:
			layout_image_zoom_adjust(lw, get_zoom_increment());
			stop_signal = TRUE;
			break;
		case GDK_KP_Subtract:
			layout_image_zoom_adjust(lw, -get_zoom_increment());
			stop_signal = TRUE;
			break;
		case GDK_KP_Multiply:
			layout_image_zoom_set(lw, 0.0);
			stop_signal = TRUE;
			break;
		case GDK_KP_Divide: case '1':
			layout_image_zoom_set(lw, 1.0);
			stop_signal = TRUE;
			break;
		case '2':
			layout_image_zoom_set(lw, 2.0);
			stop_signal = TRUE;
			break;
		case '3':
			layout_image_zoom_set(lw, 3.0);
			stop_signal = TRUE;
			break;
		case '4':
			layout_image_zoom_set(lw, 4.0);
			stop_signal = TRUE;
			break;
		case '7':
			layout_image_zoom_set(lw, -4.0);
			stop_signal = TRUE;
			break;
		case '8':
			layout_image_zoom_set(lw, -3.0);
			stop_signal = TRUE;
			break;
		case '9':
			layout_image_zoom_set(lw, -2.0);
			stop_signal = TRUE;
			break;
		case GDK_Page_Up: case GDK_KP_Page_Up:
			layout_image_prev(lw);
			stop_signal = TRUE;
			break;
		case GDK_Page_Down: case GDK_KP_Page_Down:
			layout_image_next(lw);
			stop_signal = TRUE;
			break;
		case GDK_Home: case GDK_KP_Home:
			layout_image_first(lw);
			stop_signal = TRUE;
			break;
		case GDK_End: case GDK_KP_End:
			layout_image_last(lw);
			stop_signal = TRUE;
			break;
		case GDK_Delete: case GDK_KP_Delete:
			if (enable_delete_key)
				{
				file_util_delete(NULL, layout_selection_list(lw));
				stop_signal = TRUE;
				}
			break;
		case GDK_Escape:
			printf("FIXME:interrupting thumbs no longer allowed\n");
#if 0
			interrupt_thumbs();
#endif
			stop_signal = TRUE;
			break;
		case 'P': case 'p':
			layout_image_slideshow_pause_toggle(lw);
			break;
		case 'Q': case 'q':
			if (event->state == 0 || (event->state & GDK_MODIFIER_MASK) == GDK_LOCK_MASK)
				{
				exit_gqview();
				return FALSE;
				}
			break;
		}

	if (event->state & GDK_SHIFT_MASK)
		{
		x *= 3;
		y *= 3;
		}

	if (x != 0 || y!= 0)
		{
		keyboard_scroll_calc(&x, &y, event);
		layout_image_scroll(lw, x, y);
		}

	if (stop_signal) gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");

	return stop_signal;
}

void layout_keyboard_init(LayoutWindow *lw)
{
	gtk_signal_connect(GTK_OBJECT(lw->window), "key_press_event",
			   GTK_SIGNAL_FUNC(layout_key_press_cb), lw);
}

/*
 *-----------------------------------------------------------------------------
 * menu callbacks
 *-----------------------------------------------------------------------------
 */ 

static void layout_menu_new_cb(gpointer data, guint action, GtkWidget *widget)
{
	collection_window_new(NULL);
}

static void layout_menu_open_cb(gpointer data, guint action, GtkWidget *widget)
{
	collection_dialog_load(NULL);
}

static void layout_menu_dupes_cb(gpointer data, guint action, GtkWidget *widget)
{
	dupe_window_new(DUPE_MATCH_NAME);
}

static void layout_menu_dir_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_create_dir(lw->path);
}

static void layout_menu_copy_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_copy(NULL, layout_selection_list(lw), NULL);
}

static void layout_menu_move_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_move(NULL, layout_selection_list(lw), NULL);
}

static void layout_menu_rename_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_rename(NULL, layout_selection_list(lw));
}

static void layout_menu_delete_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	file_util_delete(NULL, layout_selection_list(lw));
}

static void layout_menu_exit_cb(gpointer data, guint action, GtkWidget *widget)
{
	exit_gqview();
}

static void layout_menu_alter_90_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_ROTATE_90);
}

static void layout_menu_alter_90cc_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_ROTATE_90_CC);
}

static void layout_menu_alter_180_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_ROTATE_180);
}

static void layout_menu_alter_mirror_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_MIRROR);
}

static void layout_menu_alter_flip_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_alter(lw, ALTER_FLIP);
}

static void layout_menu_select_all_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_select_all(lw);
}

static void layout_menu_unselect_all_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_select_none(lw);
}

static void layout_menu_config_cb(gpointer data, guint action, GtkWidget *widget)
{
	show_config_window();
}

static void layout_menu_remove_thumb_cb(gpointer data, guint action, GtkWidget *widget)
{
	cache_maintain_home(FALSE);
}

static void layout_menu_wallpaper_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_to_root(lw);
}

static void layout_menu_zoom_in_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_zoom_adjust(lw, get_zoom_increment());
}

static void layout_menu_zoom_out_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_zoom_adjust(lw, -get_zoom_increment());
}

static void layout_menu_zoom_1_1_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_zoom_set(lw, 1.0);
}

static void layout_menu_zoom_fit_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_zoom_set(lw, 0.0);
}

static void layout_menu_full_screen_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_full_screen_start(lw);
}

static void layout_menu_thumb_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_thumb_set(lw, GTK_CHECK_MENU_ITEM(widget)->active);
}

static void layout_menu_refresh_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_refresh(lw);
}

static void layout_menu_float_cb(gpointer data, guint action, GtkWidget *widget)
{
#if 0
	LayoutWindow *lw = data;
#endif

	printf("FIXME: implement float\n");
}

static void layout_menu_hide_cb(gpointer data, guint action, GtkWidget *widget)
{
#if 0
	LayoutWindow *lw = data;
#endif

	printf("FIXME: implement hide\n");
}

static void layout_menu_slideshow_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;

	layout_image_slideshow_toggle(lw);
}

static void layout_menu_help_cb(gpointer data, guint action, GtkWidget *widget)
{
	help_window_show("documentation");
}

static void layout_menu_notes_cb(gpointer data, guint action, GtkWidget *widget)
{
	help_window_show("release_notes");
}

static void layout_menu_about_cb(gpointer data, guint action, GtkWidget *widget)
{
	show_about_window();
}

/*
 *-----------------------------------------------------------------------------
 * edit menu
 *-----------------------------------------------------------------------------
 */ 

static void layout_menu_edit_cb(gpointer data, guint action, GtkWidget *widget)
{
	LayoutWindow *lw = data;
	gint n;

	n = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget), "edit_index"));

	start_editor_from_file(n, layout_image_get_path(lw));
}

static void layout_menu_edit_update(LayoutWindow *lw)
{
	gint i;

	/* main edit menu */

	for(i = 0; i < 8; i++)
		{
		GtkWidget *item;

		item = gtk_item_factory_get_item_by_action(lw->menu_fact, MENU_EDIT_ACTION_OFFSET + i);
		gtk_object_set_data(GTK_OBJECT(item), "edit_index", GINT_TO_POINTER(i));

		if (editor_command[i] && strlen(editor_command[i]) > 0)
			{
			gchar *text;
			if (editor_name[i] && strlen(editor_name[i]) > 0)
				{
				text = g_strdup_printf(_("in %s..."), editor_name[i]);
				}
			else
				{
				text = g_strdup(_("in (unknown)..."));
				}
			gtk_label_set_text(GTK_LABEL(GTK_BIN(item)->child), text);
			g_free(text);
			gtk_widget_show(item);
			}
		else
			{
			gtk_widget_hide(item);
			}
		}
}

void layout_edit_update_all(void)
{
	GList *work;

	work = layout_window_list;
	while (work)
		{
		LayoutWindow *lw = work->data;
		work = work->next;

		layout_menu_edit_update(lw);
		}
}

/*
 *-----------------------------------------------------------------------------
 * recent menu
 *-----------------------------------------------------------------------------
 */

static void layout_menu_recent_cb(GtkWidget *widget, gpointer data)
{
	gint n;
	gchar *path;

	n = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget), "recent_index"));

	path = g_list_nth_data(history_list_get_by_key("recent"), n);

	if (!path) return;

	/* make a copy of it */
	path = g_strdup(path);
	collection_window_new(path);
	g_free(path);
}

static void layout_menu_recent_update(LayoutWindow *lw)
{
	GtkWidget *menu;
	GtkWidget *recent;
	GtkWidget *item;
	GList *list;
	gint n;

	list = history_list_get_by_key("recent");
	n = 0;

	menu = gtk_menu_new();

	while (list)
		{
		item = menu_item_add(menu, filename_from_path((gchar *)list->data),
				     layout_menu_recent_cb, lw);
		gtk_object_set_data(GTK_OBJECT(item), "recent_index", GINT_TO_POINTER(n));
		list = list->next;
		n++;
		}

	if (n == 0)
		{
		menu_item_add(menu, _("Empty"), NULL, NULL);
		}

	recent = gtk_item_factory_get_item(lw->menu_fact, "/File/Open recent");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(recent), menu);
	gtk_widget_set_sensitive(recent, (n != 0));
}

void layout_recent_update_all(void)
{
	GList *work;

	work = layout_window_list;
	while (work)
		{
		LayoutWindow *lw = work->data;
		work = work->next;

		layout_menu_recent_update(lw);
		}
}

void layout_recent_add_path(const gchar *path)
{
	if (!path) return;

	history_list_add_to_key("recent", path, recent_list_max);

	layout_recent_update_all();
}

/*
 *-----------------------------------------------------------------------------
 * menu
 *-----------------------------------------------------------------------------
 */ 

static GtkItemFactoryEntry gqview_menu_items[] =
{
	{ N_("/_File"),				NULL,		NULL,				0, "<Branch>" },
	{ N_("/File/tear1"),			NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/File/_New collection"),		"C",		layout_menu_new_cb,		0, NULL },
	{ N_("/File/_Open collection..."),	"O",		layout_menu_open_cb,		0, NULL },
	{ N_("/File/Open _recent"),		NULL,		NULL,				0, NULL },
	{ N_("/File/sep1"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/File/_Find duplicates..."),	"D",		layout_menu_dupes_cb,		0, NULL },
	{ N_("/File/sep2"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/File/Create _Dir..."),		"<control>N",	layout_menu_dir_cb,		0, NULL },
	{ N_("/File/sep3"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/File/_Copy..."),			"<control>C",	layout_menu_copy_cb,		0, NULL },
	{ N_("/File/_Move..."),			"<control>M",	layout_menu_move_cb,		0, NULL },
	{ N_("/File/_Rename..."),		"<control>R",	layout_menu_rename_cb,		0, NULL },
	{ N_("/File/_Delete..."),		"<control>D",	layout_menu_delete_cb,		0, NULL },
	{ N_("/File/sep4"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/File/E_xit"),			"<control>Q",	layout_menu_exit_cb,		0, NULL },

	{ N_("/_Edit"),				NULL,		NULL,				0, "<Branch>" },
	{ N_("/Edit/tear1"),			NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/Edit/editor1"),			"<control>1",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 0, NULL },
	{ N_("/Edit/editor2"),			"<control>2",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 1, NULL },
	{ N_("/Edit/editor3"),			"<control>3",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 2, NULL },
	{ N_("/Edit/editor4"),			"<control>4",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 3, NULL },
	{ N_("/Edit/editor5"),			"<control>5",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 4, NULL },
	{ N_("/Edit/editor6"),			"<control>6",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 5, NULL },
	{ N_("/Edit/editor7"),			"<control>7",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 6, NULL },
	{ N_("/Edit/editor8"),			"<control>8",	layout_menu_edit_cb,		MENU_EDIT_ACTION_OFFSET + 7, NULL },
	{ N_("/Edit/sep1"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Edit/_Adjust"),			NULL,		NULL,				0, "<Branch>" },
	{ N_("/Edit/Adjust/tear1"),		NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/Edit/Adjust/_Rotate clockwise"),		"bracketright",	layout_menu_alter_90_cb,	0, NULL },
	{ N_("/Edit/Adjust/Rotate _counterclockwise"),	"bracketleft",	layout_menu_alter_90cc_cb,	0, NULL },
	{ N_("/Edit/Adjust/Rotate 1_80"),	"<shift>R",	layout_menu_alter_180_cb,	0, NULL },
	{ N_("/Edit/Adjust/_Mirror"),		"<shift>M",	layout_menu_alter_mirror_cb,	0, NULL },
	{ N_("/Edit/Adjust/_Flip"),		"<shift>F",	layout_menu_alter_flip_cb,	0, NULL },
	{ N_("/Edit/sep2"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Edit/Select _all"),		"<control>A",	layout_menu_select_all_cb,	0, NULL },
	{ N_("/Edit/Select _none"),	"<control><shift>A",	layout_menu_unselect_all_cb,	0, NULL },
	{ N_("/Edit/sep3"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Edit/_Options..."),		"<control>O",	layout_menu_config_cb,		0, NULL },
	{ N_("/Edit/_Remove old thumbnails"),	"<control>T",	layout_menu_remove_thumb_cb,	0, NULL },
	{ N_("/Edit/sep4"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Edit/Set as _wallpaper"),	"<control>W",	layout_menu_wallpaper_cb,	0, NULL },

	{ N_("/_View"),				NULL,		NULL,				0, "<Branch>" },
	{ N_("/View/tear1"),			NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/View/Zoom _in"),			"equal",	layout_menu_zoom_in_cb,		0, NULL },
	{ N_("/View/Zoom _out"),		"minus",	layout_menu_zoom_out_cb,	0, NULL },
	{ N_("/View/Zoom _1:1"),		"Z",		layout_menu_zoom_1_1_cb,	0, NULL },
	{ N_("/View/_Zoom to fit"),		"X",		layout_menu_zoom_fit_cb,	0, NULL },
	{ N_("/View/sep1"),			NULL,		NULL,				0, "<Separator>" },

	{ N_("/View/F_ull screen"),		"V",		layout_menu_full_screen_cb,	0, NULL },
	{ N_("/View/_Thumbnails"),		"T",		layout_menu_thumb_cb,		0, "<CheckItem>" },

	{ N_("/View/sep2"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/View/_Refresh Lists"),		"R",		layout_menu_refresh_cb,		0, NULL },
	{ N_("/View/(Un)_Float file list"),	"F",		layout_menu_float_cb,		0, NULL },
	{ N_("/View/(Un)_Hide file list"),	"H",		layout_menu_hide_cb,		0, NULL },
	{ N_("/View/sep3"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/View/Toggle _slideshow"),	"S",		layout_menu_slideshow_cb,	0, NULL },

	{ N_("/_Help"),				NULL,		NULL,				0, "<Branch>" },
	{ N_("/Help/tear1"),			NULL,		NULL,				0, "<Tearoff>" },
	{ N_("/Help/_Keyboard shortcuts"),	"F1",		layout_menu_help_cb,		0, NULL},
	{ N_("/Help/_Release notes"),		NULL,		layout_menu_notes_cb,		0, NULL},
	{ N_("/Help/sep1"),			NULL,		NULL,				0, "<Separator>" },
	{ N_("/Help/_About"),			NULL,		layout_menu_about_cb,		0, NULL },
};

static int gqview_nmenu_items = sizeof (gqview_menu_items) / sizeof (gqview_menu_items[0]);

static gchar *menu_translate(const gchar *path, gpointer data)
{
	return _(path);
}

GtkWidget *layout_menu_bar(LayoutWindow *lw)
{
	lw->menu_fact = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", lw->accel_grp);
	gtk_item_factory_set_translate_func(lw->menu_fact, menu_translate, NULL, NULL);

	gtk_item_factory_create_items(lw->menu_fact, gqview_nmenu_items, gqview_menu_items, lw);

	return gtk_item_factory_get_widget(lw->menu_fact, "<main>");
}

/*
 *-----------------------------------------------------------------------------
 * toolbar
 *-----------------------------------------------------------------------------
 */ 

static void layout_button_thumb_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_thumb_set(lw, GTK_TOGGLE_BUTTON(widget)->active);
}

static void layout_button_home_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;
	const gchar *path = homedir();

	if (path) layout_set_path(lw, path);
}

static void layout_button_refresh_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_refresh(lw);
}

static void layout_button_zoom_in_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_image_zoom_adjust(lw, get_zoom_increment());
}

static void layout_button_zoom_out_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_image_zoom_adjust(lw, -get_zoom_increment());
}

static void layout_button_zoom_fit_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_image_zoom_set(lw, 0.0);
}

static void layout_button_zoom_1_1_cb(GtkWidget *widget, gpointer data)
{
	LayoutWindow *lw = data;

	layout_image_zoom_set(lw, 1.0);
}

static void layout_button_config_cb(GtkWidget *widget, gpointer data)
{
	show_config_window();
}

static void layout_button_float_cb(GtkWidget *widget, gpointer data)
{
#if 0
	LayoutWindow *lw = data;
#endif

	printf("FIXME: implement float\n");
}

GtkWidget *layout_button(GtkWidget *box, gchar **pixmap_data, gint toggle,
			 GtkTooltips *tooltips, const gchar *tip_text,
			 GtkSignalFunc func, gpointer data)
{
	GtkWidget *button;
	GtkWidget *icon;
	GdkPixbuf *pixbuf;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	if (toggle)
		{
		button = gtk_toggle_button_new();
		}
	else
		{
		button = gtk_button_new();
		}

	gtk_signal_connect(GTK_OBJECT(button), "clicked", func, data);
	gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 0);
	gtk_widget_show(button);
	gtk_tooltips_set_tip(tooltips, button, tip_text, NULL);

	pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)pixmap_data);
	gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pixmap, &mask, 128);
	gdk_pixbuf_unref(pixbuf);

	icon = gtk_pixmap_new(pixmap, mask);
	gtk_widget_show(icon);
	gtk_container_add(GTK_CONTAINER(button), icon);

	gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);

	return button;
}

GtkWidget *layout_button_bar(LayoutWindow *lw)
{
	GtkWidget *box;
	GtkTooltips *tooltips;

	tooltips = lw->tooltips;

	box = gtk_hbox_new(FALSE, 0);

	lw->thumb_button = layout_button(box, (gchar **)icon_thumb_xpm, TRUE,
		      tooltips, _("Show thumbnails"), layout_button_thumb_cb, lw);
	layout_button(box, (gchar **)icon_home_xpm, FALSE,
		      tooltips, _("Change to home directory"), layout_button_home_cb, lw);
	layout_button(box, (gchar **)icon_reload_xpm, FALSE,
		      tooltips, _("Refresh file list"), layout_button_refresh_cb, lw);
	layout_button(box, (gchar **)icon_zoom_in_xpm, FALSE,
		      tooltips, _("Zoom in"), layout_button_zoom_in_cb, lw);
	layout_button(box, (gchar **)icon_zoom_out_xpm, FALSE,
		      tooltips, _("Zoom out"), layout_button_zoom_out_cb, lw);
	layout_button(box, (gchar **)icon_zoom_fit_xpm, FALSE,
		      tooltips, _("Fit image to window"), layout_button_zoom_fit_cb, lw);
	layout_button(box, (gchar **)icon_zoom_norm_xpm, FALSE,
		      tooltips, _("Set zoom 1:1"), layout_button_zoom_1_1_cb, lw);
	layout_button(box, (gchar **)icon_config_xpm, FALSE,
		      tooltips, _("Configure options"), layout_button_config_cb, lw);
	layout_button(box, (gchar **)icon_float_xpm, FALSE,
		      tooltips, _("Float Controls"), layout_button_float_cb, lw);

	return box;
}

/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */ 

void layout_util_sync_thumb(LayoutWindow *lw)
{
	GtkWidget *item;

	item = gtk_item_factory_get_item(lw->menu_fact, "/View/Thumbnails");
	gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(item), lw->thumbs_enabled);

	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(lw->thumb_button), lw->thumbs_enabled);
}

void layout_util_sync(LayoutWindow *lw)
{
	layout_util_sync_thumb(lw);
	layout_menu_recent_update(lw);
	layout_menu_edit_update(lw);
}

/*
 *-----------------------------------------------------------------------------
 * icons (since all the toolbar icons are included here, best place as any)
 *-----------------------------------------------------------------------------
 */

PixmapFolders *folder_icons_new(void)
{
	PixmapFolders *pf;
	GdkPixbuf *pixbuf;

	pf = g_new0(PixmapFolders, 1);

	pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)folder_close_xpm);
	gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pf->close_pixmap, &pf->close_mask, 128);
	gdk_pixbuf_unref(pixbuf);

	pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)folder_open_xpm);
	gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pf->open_pixmap, &pf->open_mask, 128);
	gdk_pixbuf_unref(pixbuf);

	pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)folder_deny_xpm);
	gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pf->deny_pixmap, &pf->deny_mask, 128);
	gdk_pixbuf_unref(pixbuf);

	return pf;
}

static void folder_icon_unref(GdkPixmap *pixmap)
{
	if (pixmap) gdk_pixmap_unref(pixmap);
}

void folder_icons_free(PixmapFolders *pf)
{
	if (!pf) return;

	folder_icon_unref(pf->close_pixmap);
	folder_icon_unref(pf->close_mask);

	folder_icon_unref(pf->open_pixmap);
	folder_icon_unref(pf->open_mask);

	folder_icon_unref(pf->deny_pixmap);
	folder_icon_unref(pf->deny_mask);

	g_free(pf);
}

