/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "img-view.h"

#include "collect.h"
#include "dnd.h"
#include "fullscreen.h"
#include "image.h"
#include "menu.h"
#include "slideshow.h"
#include "utilops.h"
#include "ui_menu.h"

#include <gdk/gdkkeysyms.h> /* for keyboard values */

#include "icons/view.xpm"


typedef struct _ViewWindow ViewWindow;
struct _ViewWindow
{
	GtkWidget *window;
	ImageWindow *imd;
	FullScreenData *fs;
	SlideShowData *ss;
};


static GtkWidget *view_popup_menu(ViewWindow *vw);
static void view_fullscreen_toggle(ViewWindow *vw, gint force_off);

static void view_slideshow_next(ViewWindow *vw);
static void view_slideshow_prev(ViewWindow *vw);
static void view_slideshow_start(ViewWindow *vw);
static void view_slideshow_stop(ViewWindow *vw);

static void view_window_close(ViewWindow *vw);


/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */ 

static ImageWindow *view_window_active_image(ViewWindow *vw)
{
	if (vw->fs) return vw->fs->imd;

	return vw->imd;
}

static gint view_window_contains_collection(ViewWindow *vw)
{
	CollectionData *cd;
	CollectInfo *info;

	cd = image_get_collection(view_window_active_image(vw), &info);

	return (cd && info);
}

static void view_collection_step(ViewWindow *vw, gint next)
{
	ImageWindow *imd = view_window_active_image(vw);
	CollectionData *cd;
	CollectInfo *info;
	CollectInfo *read_ahead_info = NULL;

	cd = image_get_collection(imd, &info);

	if (!cd || !info) return;

	if (next)
		{
		info = collection_next_by_info(cd, info);
		if (enable_read_ahead)
			{
			read_ahead_info = collection_next_by_info(cd, info);
			if (!read_ahead_info) read_ahead_info = collection_prev_by_info(cd, info);
			}
		}
	else
		{
		info = collection_prev_by_info(cd, info);
		if (enable_read_ahead)
			{
			read_ahead_info = collection_prev_by_info(cd, info);
			if (!read_ahead_info) read_ahead_info = collection_next_by_info(cd, info);
			}
		}

	if (info)
		{
		image_change_from_collection(imd, cd, info, image_zoom_get_default(imd, zoom_mode));

		if (read_ahead_info) image_prebuffer_set(imd, read_ahead_info->path);
		}
	
}

static void view_collection_step_to_end(ViewWindow *vw, gint last)
{
	ImageWindow *imd = view_window_active_image(vw);
	CollectionData *cd;
	CollectInfo *info;
	CollectInfo *read_ahead_info = NULL;

	cd = image_get_collection(imd, &info);

	if (!cd || !info) return;

	if (last)
		{
		info = collection_get_last(cd);
		if (enable_read_ahead) read_ahead_info = collection_prev_by_info(cd, info);
		}
	else
		{
		info = collection_get_first(cd);
		if (enable_read_ahead) read_ahead_info = collection_next_by_info(cd, info);
		}

	if (info)
		{
		image_change_from_collection(imd, cd, info, image_zoom_get_default(imd, zoom_mode));
		if (read_ahead_info) image_prebuffer_set(imd, read_ahead_info->path);
		}
	
}

static void view_step_next(ViewWindow *vw)
{
	if (vw->ss)
		{
		view_slideshow_next(vw);
		}
	else
		{
		view_collection_step(vw, TRUE);
		}
}

static void view_step_prev(ViewWindow *vw)
{
	if (vw->ss)
		{
		view_slideshow_prev(vw);
		}
	else
		{
		view_collection_step(vw, FALSE);
		}
}

/*
 *-----------------------------------------------------------------------------
 * view window keyboard
 *-----------------------------------------------------------------------------
 */

static gint view_window_key_press_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	ViewWindow *vw = data;
	ImageWindow *imd;
	gint stop_signal = FALSE;
	gint x = 0;
	gint y = 0;

	imd = view_window_active_image(vw);

	if ( !(event->state & GDK_CONTROL_MASK) )
	    switch (event->keyval)
		{
		case GDK_Left: case GDK_KP_Left:
			x -= 1;
			stop_signal = TRUE;
			break;
		case GDK_Right: case GDK_KP_Right:
			x += 1;
			stop_signal = TRUE;
			break;
		case GDK_Up: case GDK_KP_Up:
			y -= 1;
			stop_signal = TRUE;
			break;
		case GDK_Down: case GDK_KP_Down:
			y += 1;
			stop_signal = TRUE;
			break;
		case GDK_Page_Up: case GDK_KP_Page_Up:
		case GDK_BackSpace:
		case 'B': case 'b':
			view_step_prev(vw);
			stop_signal = TRUE;
			break;
		case GDK_Page_Down: case GDK_KP_Page_Down:
		case GDK_space:
			view_step_next(vw);
			stop_signal = TRUE;
			break;
		case GDK_Home: case GDK_KP_Home:
			view_collection_step_to_end(vw, FALSE);
			stop_signal = TRUE;
			break;
		case GDK_End: case GDK_KP_End:
			view_collection_step_to_end(vw, TRUE);
			stop_signal = TRUE;
			break;
		case '+': case '=': case GDK_KP_Add:
			image_zoom_adjust(imd, get_zoom_increment());
			break;
		case '-': case GDK_KP_Subtract:
			image_zoom_adjust(imd, -get_zoom_increment());
			break;
		case 'X': case 'x': case GDK_KP_Multiply:
			image_zoom_set(imd, 0.0);
			break;
		case 'Z': case 'z': case GDK_KP_Divide: case '1':
			image_zoom_set(imd, 1.0);
			break;
		case '2':
			image_zoom_set(imd, 2.0);
			break;
		case '3':
			image_zoom_set(imd, 3.0);
			break;
		case '4':
			image_zoom_set(imd, 4.0);
			break;
		case '7':
			image_zoom_set(imd, -4.0);
			break;
		case '8':
			image_zoom_set(imd, -3.0);
			break;
		case '9':
			image_zoom_set(imd, -2.0);
			break;
		case 'S': case 's':
			if (vw->ss)
				view_slideshow_stop(vw);
			else
				view_slideshow_start(vw);
			stop_signal = TRUE;
			break;
		case 'P': case 'p':
			slideshow_pause_toggle(vw->ss);
			break;
		case 'V': case 'v':
			view_fullscreen_toggle(vw, FALSE);
			stop_signal = TRUE;
			break;
		case ']':
			image_alter(imd, ALTER_ROTATE_90);
			stop_signal = TRUE;
			break;
		case '[':
			image_alter(imd, ALTER_ROTATE_90_CC);
			stop_signal = TRUE;
			break;
		case GDK_Delete: case GDK_KP_Delete:
			if (enable_delete_key)
				{
				view_fullscreen_toggle(vw, TRUE);
				file_util_delete(image_get_path(vw->imd), NULL);
				stop_signal = TRUE;
				}
			break;
		case GDK_Escape:
			if (vw->fs)
				{
				view_fullscreen_toggle(vw, TRUE);
				}
			else
				{
				gtk_widget_destroy(vw->window);
				}
			stop_signal = TRUE;
			break;
		}

	if (event->state & GDK_CONTROL_MASK)
		{
		gint n = -1;
		switch (event->keyval)
			{
			case '1':
				n = 0;
				break;
			case '2':
				n = 1;
				break;
			case '3':
				n = 2;
				break;
			case '4':
				n = 3;
				break;
			case '5':
				n = 4;
				break;
			case '6':
				n = 5;
				break;
			case '7':
				n = 6;
				break;
			case '8':
				n = 7;
				break;
			case 'C': case 'c':
				view_fullscreen_toggle(vw, TRUE);
				file_util_copy(image_get_path(vw->imd), NULL, NULL);
				stop_signal = TRUE;
				break;
			case 'M': case 'm':
				view_fullscreen_toggle(vw, TRUE);
				file_util_move(image_get_path(vw->imd), NULL, NULL);
				stop_signal = TRUE;
				break;
			case 'R': case 'r':
				view_fullscreen_toggle(vw, TRUE);
				file_util_rename(image_get_path(vw->imd), NULL);
				stop_signal = TRUE;
				break;
			case 'D': case 'd':
				view_fullscreen_toggle(vw, TRUE);
				file_util_delete(image_get_path(vw->imd), NULL);
				stop_signal = TRUE;
				break;
			case 'W': case 'w':
				view_window_close(vw);
				break;
			}
		if (n != -1)
			{
			view_fullscreen_toggle(vw, TRUE);
			start_editor_from_file(n, image_get_path(imd));
			}
		}
	else if (event->state & GDK_SHIFT_MASK)
		{
		switch (event->keyval)
			{
			case 'R': case 'r':
				image_alter(imd, ALTER_ROTATE_180);
				stop_signal = TRUE;
				break;
			case 'M': case 'm':
				image_alter(imd, ALTER_MIRROR);
				stop_signal = TRUE;
				break;
			case 'F': case 'f':
				image_alter(imd, ALTER_FLIP);
				stop_signal = TRUE;
				break;
			default:
				break;
			}
		x *= 3;
		y *= 3;
		}

	if (x != 0 || y!= 0)
		{
		keyboard_scroll_calc(&x, &y, event);
		image_scroll(imd, x, y);
		}

	return stop_signal;
}

/*
 *-----------------------------------------------------------------------------
 * view window main routines
 *-----------------------------------------------------------------------------
 */ 

static void button1_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	ViewWindow *vw = data;

	view_step_next(vw);
}

static void button2_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	ViewWindow *vw = data;

	view_step_prev(vw);
}

static void button3_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	ViewWindow *vw = data;
	GtkWidget *menu;

	menu = view_popup_menu(vw);
	gtk_menu_popup (GTK_MENU(menu), NULL, NULL, NULL, NULL,
		bevent->button, bevent->time);
}

static void button4_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
#if 0
	ViewWindow *vw = data;
#endif

	if (bevent->state & GDK_CONTROL_MASK)
		{
		image_zoom_adjust(imd, get_zoom_increment());
		}
	else if ( (bevent->state & GDK_SHIFT_MASK) != (mousewheel_scrolls))
		{
		image_scroll(imd, 0, -MOUSEWHEEL_SCROLL_SIZE);
		}
	else
		{
		/* FIXME: implement moving among image in view windows */
		}
}

static void button5_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
#if 0
	ViewWindow *vw = data;
#endif

	if (bevent->state & GDK_CONTROL_MASK)
		{
		image_zoom_adjust(imd, -get_zoom_increment());
		}
	else if ( (bevent->state & GDK_SHIFT_MASK) != (mousewheel_scrolls))
		{
		image_scroll(imd, 0, MOUSEWHEEL_SCROLL_SIZE);
		}
	else
		{
		/* FIXME: implement moving among image in view windows */
		}
}

static void view_image_set_buttons(ViewWindow *vw, ImageWindow *imd)
{
	image_set_button_func(imd, 1, button1_cb, vw);
	image_set_button_func(imd, 2, button2_cb, vw);
	image_set_button_func(imd, 3, button3_cb, vw);
	/* for wheel mice */
	image_set_button_func(imd, 4, button4_cb, vw);
	image_set_button_func(imd, 5, button5_cb, vw);
}

static void view_fullscreen_stop_func(FullScreenData *fs, gpointer data)
{
	ViewWindow *vw = data;

	vw->fs = NULL;

	if (vw->ss) vw->ss->imd = vw->imd;
}

static void view_fullscreen_toggle(ViewWindow *vw, gint force_off)
{
	if (force_off && !vw->fs) return;

	if (vw->fs)
		{
		fullscreen_stop(vw->fs);
		}
	else
		{
		vw->fs = fullscreen_start(vw->window, vw->imd, view_fullscreen_stop_func, vw);

		view_image_set_buttons(vw, vw->fs->imd);
		gtk_signal_connect(GTK_OBJECT(vw->fs->window), "key_press_event", GTK_SIGNAL_FUNC(view_window_key_press_cb), vw);

		if (vw->ss) vw->ss->imd = vw->fs->imd;
		}
}

static void view_slideshow_next(ViewWindow *vw)
{
	if (vw->ss) slideshow_next(vw->ss);
}

static void view_slideshow_prev(ViewWindow *vw)
{
	if (vw->ss) slideshow_prev(vw->ss);
}

static void view_slideshow_stop_func(SlideShowData *fs, gpointer data)
{
	ViewWindow *vw = data;

	vw->ss = NULL;
}

static void view_slideshow_start(ViewWindow *vw)
{
	if (!vw->ss)
		{
		CollectionData *cd;
		CollectInfo *info;

		cd = image_get_collection(view_window_active_image(vw), &info);
		if (cd && info)
			{
			vw->ss = slideshow_start_from_collection(view_window_active_image(vw), cd,
								 view_slideshow_stop_func, vw, info);
			}
		}
}

static void view_slideshow_stop(ViewWindow *vw)
{
	if (vw->ss) slideshow_free(vw->ss);
}

static void view_window_close(ViewWindow *vw)
{
	view_slideshow_stop(vw);
	view_fullscreen_toggle(vw, TRUE);
	gtk_widget_destroy(vw->window);
	g_free(vw);
}

static gint view_window_delete_cb(GtkWidget *w, GdkEventAny *event, gpointer data)
{
	ViewWindow *vw = data;

	view_window_close(vw);
	return TRUE;
}

static void real_view_window_new(const gchar *path, CollectionData *cd, CollectInfo *info)
{
	ViewWindow *vw;
	GtkAllocation req_size;
	gint w, h;

	if (!path && (!cd || !info)) return;

	vw = g_new0(ViewWindow, 1);
	vw->fs = NULL;
	vw->ss = NULL;

	vw->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(vw->window), TRUE, TRUE, FALSE);
	gtk_window_set_title (GTK_WINDOW (vw->window), "GQview");
        gtk_window_set_wmclass(GTK_WINDOW (vw->window), "view", "GQview");
        gtk_container_border_width (GTK_CONTAINER (vw->window), 0);

	window_set_icon(vw->window, (const gchar **)view_xpm, NULL);

	vw->imd = image_new(FALSE);
	image_attach_window(vw->imd, vw->window, NULL, " - GQview", TRUE);

	image_top_window_set_sync(vw->imd, TRUE);

	gtk_container_add(GTK_CONTAINER(vw->window), vw->imd->widget);
	gtk_widget_show(vw->imd->widget);

	image_dnd_init(vw->imd);

	view_image_set_buttons(vw, vw->imd);

	gtk_signal_connect(GTK_OBJECT(vw->window), "delete_event", (GtkSignalFunc) view_window_delete_cb, vw);
	gtk_signal_connect(GTK_OBJECT(vw->window), "key_press_event", GTK_SIGNAL_FUNC(view_window_key_press_cb), vw);

	if (cd && info)
		{
		image_change_from_collection(vw->imd, cd, info, image_zoom_get_default(NULL, zoom_mode));
		if (enable_read_ahead)
			{
			CollectInfo * r_info = collection_next_by_info(cd, info);
			if (!r_info) r_info = collection_prev_by_info(cd, info);
			if (r_info) image_prebuffer_set(vw->imd, r_info->path);
			}
		}
	else
		{
		image_change_path(vw->imd, path, image_zoom_get_default(NULL, zoom_mode));
		}

	if (image_zoom_get(vw->imd) == 0.0)
		{
		w = vw->imd->image_width;
		h = vw->imd->image_height;
		}
	else
		{
		w = vw->imd->width;
		h = vw->imd->height;
		}
	if (limit_window_size)
		{
		gint mw = gdk_screen_width() * max_window_size / 100;
		gint mh = gdk_screen_height() * max_window_size / 100;

		if (w > mw) w = mw;
		if (h > mh) h = mh;
		}

	gtk_window_set_default_size (GTK_WINDOW(vw->window), w, h);
	req_size.x = req_size.y = 0;
	req_size.width = w;
	req_size.height = h;
	gtk_widget_size_allocate(GTK_WIDGET(vw->window), &req_size);

	gtk_widget_set_usize(vw->imd->eventbox, w, h);

	gtk_widget_show(vw->window);
}

void view_window_new(const gchar *path)
{
	real_view_window_new(path, NULL, NULL);
}

void view_window_new_from_collection(CollectionData *cd, CollectInfo *info)
{
	real_view_window_new(NULL, cd, info);
}

/*
 *-----------------------------------------------------------------------------
 * view window menu routines and callbacks
 *-----------------------------------------------------------------------------
 */ 

static void view_new_window_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;
	CollectionData *cd;
	CollectInfo *info;

	cd = image_get_collection(vw->imd, &info);

	if (cd && info)
		{
		view_window_new_from_collection(cd, info);
		}
	else
		{
		view_window_new(image_get_path(vw->imd));
		}
}

static void view_edit_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw;
	gint n;

	vw = submenu_item_get_data(widget);
	n = GPOINTER_TO_INT(data);
	if (!vw) return;

	view_fullscreen_toggle(vw, TRUE);
	start_editor_from_file(n, image_get_path(vw->imd));
}

static void view_alter_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw;
	AlterType type;

	vw = submenu_item_get_data(widget);
	type = GPOINTER_TO_INT(data);

	if (!vw) return;
	image_alter(vw->imd, type);
}

static void view_wallpaper_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;
	ImageWindow *imd;

	imd = view_window_active_image(vw);

	image_to_root_window(imd, (image_zoom_get(imd) == 0.0));
}

static void view_zoom_in_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	image_zoom_adjust(view_window_active_image(vw), get_zoom_increment());
}

static void view_zoom_out_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	image_zoom_adjust(view_window_active_image(vw), -get_zoom_increment());
}

static void view_zoom_1_1_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	image_zoom_set(view_window_active_image(vw), 1.0);
}

static void view_zoom_fit_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	image_zoom_set(view_window_active_image(vw), 0.0);
}

static void view_copy_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, TRUE);
	file_util_copy(image_get_path(vw->imd), NULL, NULL);
}

static void view_move_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, TRUE);
	file_util_move(image_get_path(vw->imd), NULL, NULL);
}

static void view_rename_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, TRUE);
	file_util_rename(image_get_path(vw->imd), NULL);
}

static void view_delete_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, TRUE);
	file_util_delete(image_get_path(vw->imd), NULL);
}

static void view_fullscreen_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, FALSE);
}

static void view_slideshow_start_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_slideshow_start(vw);
}

static void view_slideshow_stop_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_slideshow_stop(vw);
}

static void view_slideshow_pause_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	slideshow_pause_toggle(vw->ss);
}

static void view_close_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_window_close(vw);
}

static GtkWidget *view_popup_menu(ViewWindow *vw)
{
	GtkWidget *menu;
	GtkWidget *item;

	menu = popup_menu_short_lived();

	menu_item_add(menu, _("Zoom in"), view_zoom_in_cb, vw);
	menu_item_add(menu, _("Zoom out"), view_zoom_out_cb, vw);
	menu_item_add(menu, _("Zoom 1:1"), view_zoom_1_1_cb, vw);
	menu_item_add(menu, _("Fit image to window"), view_zoom_fit_cb, vw);
	menu_item_add_divider(menu);

	item = submenu_add_edit(menu, NULL, view_edit_cb, vw);
	menu_item_add_divider(item);
	menu_item_add(item, _("Set as wallpaper"), view_wallpaper_cb, vw);

	submenu_add_alter(menu, view_alter_cb, vw);

	menu_item_add(menu, _("View in new window"), view_new_window_cb, vw);

	menu_item_add_divider(menu);
	menu_item_add(menu, _("Copy..."), view_copy_cb, vw);
	menu_item_add(menu, _("Move..."), view_move_cb, vw);
	menu_item_add(menu, _("Rename..."), view_rename_cb, vw);
	menu_item_add(menu, _("Delete..."), view_delete_cb, vw);

	menu_item_add_divider(menu);

	if (vw->ss)
		{
		menu_item_add(menu, _("Stop slideshow"), view_slideshow_stop_cb, vw);
		if (slideshow_paused(vw->ss))
			{
			item = menu_item_add(menu, _("Continue slideshow"), view_slideshow_pause_cb, vw);
			}
		else
			{
			item = menu_item_add(menu, _("Pause slideshow"), view_slideshow_pause_cb, vw);
			}
		}
	else
		{
		item = menu_item_add(menu, _("Start slideshow"), view_slideshow_start_cb, vw);
		gtk_widget_set_sensitive(item, view_window_contains_collection(vw));
		item = menu_item_add(menu, _("Pause slideshow"), view_slideshow_pause_cb, vw);
		gtk_widget_set_sensitive(item, FALSE);
		}

	if (vw->fs)
		{
		menu_item_add(menu, _("Exit full screen"), view_fullscreen_cb, vw);
		}
	else
		{
		menu_item_add(menu, _("Full screen"), view_fullscreen_cb, vw);
		}

	menu_item_add_divider(menu);
	menu_item_add(menu, _("Close window"), view_close_cb, vw);

	return menu;
}

