/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "image.h"


#include "image-load.h"
#include "collect.h"
#include "pixbuf_util.h"
#include "ui_fileops.h"

#include <math.h>
#include <gdk/gdkx.h>


/* define this to enable debug printing (sorry, peppered throughout file) */
/* #define IMAGE_DEBUG_ENABLE 1 */


#define IMAGE_TILE_SIZE 128
#define IMAGE_ZOOM_MIN -32.0
#define IMAGE_ZOOM_MAX 32.0

/* size of the image loader buffer (512 bytes x defined number) */
#define IMAGE_LOAD_BUFFER_COUNT 8

/* distance to drag mouse to disable image flip */
#define IMAGE_DRAG_SCROLL_THRESHHOLD 4

/* alpha channel checkerboard background (same as gimp) */
#define IMAGE_ALPHA_CHECK1 0x00999999
#define IMAGE_ALPHA_CHECK2 0x00666666
#define IMAGE_ALPHA_CHECK_SIZE 16


typedef enum {
	TILE_RENDER_NONE = 0,	/* do nothing */
	TILE_RENDER_AREA,	/* render an area of the tile */
	TILE_RENDER_ALL		/* render the whole tile */
} TileRenderType;

typedef struct _ImageTile ImageTile;
struct _ImageTile
{
	GdkPixmap *pixmap;	/* off screen buffer */
	GdkPixbuf *pixbuf;	/* pixbuf area for zooming */
	gint x;			/* x offset into image */
	gint y;			/* y offset into image */
	gint w;			/* width that is visible (may be less if at edge of image) */
	gint h;			/* height '' */

	gint blank;

/* render_todo: (explanation)
	NONE	do nothing
	AREA	render area of tile, usually only used when loading an image
		note: will jump to an ALL if render_done is not ALL.
	ALL	render entire tile, if never done before w/ ALL, for expose events *only*
*/

	TileRenderType render_todo;	/* what to do (see above) */
	TileRenderType render_done;	/* highest that has been done before on tile */
};

typedef struct _QueueData QueueData;
struct _QueueData
{
	ImageTile *it;
	gint x;
	gint y;
	gint w;
	gint h;
	gint new_data;
};

typedef struct _CacheData CacheData;
struct _CacheData
{
	GdkPixmap *pixmap;
	GdkPixbuf *pixbuf;
	ImageTile *it;
	guint size;
};


static void image_queue_clear(ImageWindow *imd);

static void image_update_title(ImageWindow *imd);
static void image_update_util(ImageWindow *imd);

static void image_button_do(ImageWindow *imd, GdkEventButton *event, gint button);


/*
 *-------------------------------------------------------------------
 * tile cache
 *-------------------------------------------------------------------
 */

static gint pixmap_calc_size(GdkPixmap *pixmap)
{
	gint w, h, d;

	gdk_window_get_geometry(pixmap, NULL, NULL, &w, &h, &d);
	return w * h * (d / 8);
}

static void image_tile_cache_remove(ImageWindow *imd, ImageTile *it)
{
	GList *work;

	work = imd->tile_cache;
	while(work)
		{
		CacheData *cd = work->data;
		work = work->next;

		if (cd->it == it)
			{
			imd->tile_cache = g_list_remove(imd->tile_cache, cd);
			imd->tile_cache_size -= cd->size;
			g_free(cd);
			}
		}
}

static void image_tile_cache_free(ImageWindow *imd, CacheData *cd)
{
	imd->tile_cache = g_list_remove(imd->tile_cache, cd);
	if (cd->pixmap)
		{
		gdk_pixmap_unref(cd->it->pixmap);
		cd->it->pixmap = NULL;
		cd->it->render_done = TILE_RENDER_NONE;
		}
	if (cd->pixbuf)
		{

#ifdef IMAGE_DEBUG_ENABLE
		printf("tile pixbuf free: %p\n", cd->it->pixbuf);
#endif

		gdk_pixbuf_unref(cd->it->pixbuf);
		cd->it->pixbuf = NULL;
		}
	imd->tile_cache_size -= cd->size;
	g_free(cd);
}

static void image_tile_cache_free_space(ImageWindow *imd, gint space, ImageTile *it)
{
	GList *work = g_list_last(imd->tile_cache);

	while (work && imd->tile_cache_size > 0 && imd->tile_cache_size + space > tile_cache_max * 1048576)
		{
		CacheData *cd = work->data;
		work = work->prev;
		if (cd->it != it) image_tile_cache_free(imd, cd);
		}
}

static void image_tile_cache_add(ImageWindow *imd, ImageTile *it,
				 GdkPixmap *pixmap, GdkPixbuf *pixbuf, guint size)
{
	CacheData *cd;

	cd = g_new(CacheData, 1);
	cd->pixmap = pixmap;
	cd->pixbuf = pixbuf;
	cd->it = it;
	cd->size = size;

	imd->tile_cache = g_list_prepend(imd->tile_cache, cd);

	imd->tile_cache_size += cd->size;
}

static void image_tile_prepare(ImageWindow *imd, ImageTile *it)
{
	if (!it->pixmap)
		{
		GdkPixmap *pixmap;
		guint size;

		pixmap = gdk_pixmap_new(imd->image->window, imd->tile_width, imd->tile_height, -1);

		size = pixmap_calc_size(pixmap);
		image_tile_cache_free_space(imd, size, it);

		it->pixmap = pixmap;
		image_tile_cache_add(imd, it, pixmap, NULL, size);
		}
	
	if ((imd->zoom != 1.0 || gdk_pixbuf_get_has_alpha(imd->pixbuf)) &&
	    !it->pixbuf)
		{
		GdkPixbuf *pixbuf;
		guint size;

		pixbuf = gdk_pixbuf_new(gdk_pixbuf_get_colorspace(imd->pixbuf),
					gdk_pixbuf_get_has_alpha(imd->pixbuf),
					gdk_pixbuf_get_bits_per_sample(imd->pixbuf),
					imd->tile_width, imd->tile_height);

		size = gdk_pixbuf_get_rowstride(pixbuf) * imd->tile_height;
		image_tile_cache_free_space(imd, size, it);

#ifdef IMAGE_DEBUG_ENABLE
		printf("tile pixbuf new : %p\n", pixbuf);
#endif

		it->pixbuf = pixbuf;
		image_tile_cache_add(imd, it, NULL, pixbuf, size);
		}
}

/*
 *-------------------------------------------------------------------
 * tiles
 *-------------------------------------------------------------------
 */

static ImageTile *image_tile_new(gint w, gint h)
{
	ImageTile *it;

	it = g_new0(ImageTile, 1);
	it->w = w;
	it->h = h;
	it->pixmap = NULL;
	it->pixbuf = NULL;
	it->blank = TRUE;
	it->render_todo = TILE_RENDER_NONE;
	it->render_done = TILE_RENDER_NONE;

	return it;
}

static void image_tile_free(ImageTile *it)
{
	if (!it) return;

	if (it->pixbuf) gdk_pixbuf_unref(it->pixbuf);
	if (it->pixmap) gdk_pixmap_unref(it->pixmap);

	g_free(it);
}

static void image_tile_sync_count(ImageWindow *imd, gint n)
{
	gint l;

	l = g_list_length(imd->tiles);

	if (l == n) return;

	if (l < n)
		{
		while (l < n)
			{
			imd->tiles = g_list_prepend(imd->tiles, image_tile_new(imd->tile_width, imd->tile_height));
			l++;
			}
		}
	else
		{
		/* This should remove from the tail of the GList, but with large images there are many tiles,
		 * making this significantly faster for those cases.
		 */
		while (l > n && imd->tiles)
			{
			ImageTile *it = imd->tiles->data;
			imd->tiles = g_list_remove(imd->tiles, it);
			image_tile_cache_remove(imd, it);
			image_tile_free(it);
			l--;
			}
		}
}

static void image_tile_sync(ImageWindow *imd, gint width, gint height, gint blank)
{
	gint rows;
	gint x, y;
	GList *work;

	imd->tile_cols = (width + imd->tile_width - 1) / imd->tile_width;

	rows = (height + imd->tile_height - 1) / imd->tile_height;

	image_tile_sync_count(imd, imd->tile_cols * rows);

	x = y = 0;
	work = imd->tiles;
	while(work)
		{
		ImageTile *it = work->data;
		work = work->next;

		it->x = x;
		it->y = y;
		if (x + imd->tile_width > width)
			{
			it->w = width - x;
			}	
		else
			{
			it->w = imd->tile_width;
			}
		if (y + imd->tile_height > height)
			{
			it->h = height - y;
			}
		else
			{
			it->h = imd->tile_height;
			}

		it->blank = blank;
		it->render_todo = TILE_RENDER_NONE;
		it->render_done = TILE_RENDER_NONE;

		x += imd->tile_width;
		if (x >= width)
			{
			x = 0;
			y += imd->tile_height;
			}
		}

	/* all it's are now useless in queue */
	image_queue_clear(imd);
}

static void image_tile_render(ImageWindow *imd, ImageTile *it, gint x, gint y, gint w, gint h, gint new_data)
{
	gint has_alpha;
	gint draw = FALSE;

	if (it->render_todo == TILE_RENDER_NONE && it->pixmap) return;

	if (it->render_done != TILE_RENDER_ALL)
		{
		x = 0;
		y = 0;
		w = it->w;
		h = it->h;
		it->render_done = TILE_RENDER_ALL;
		}
	else if (it->render_todo != TILE_RENDER_AREA)
		{
		it->render_todo = TILE_RENDER_NONE;
		return;
		}

	if (new_data) it->blank = FALSE;

	image_tile_prepare(imd, it);
	has_alpha = gdk_pixbuf_get_has_alpha(imd->pixbuf);

#ifdef IMAGE_DEBUG_ENABLE
	printf("image.c: rendering tile at %d, %d (%d x %d) [dim: %d x %d] nd:%d\n", it->x + x, it->y + y, w, h, imd->width, imd->height, new_data);
	printf("image.c: rendered  tile is %d, %d (%d x %d)\n", it->x, it->y, it->w, it->h);
#endif

	/* FIXME checker colors for alpha should be configurable,
	 * also should be drawn for blank = TRUE
	 */

	if (it->blank)
		{
		/* no data, do fast rect fill */
		gdk_draw_rectangle(it->pixmap, imd->image->style->black_gc, TRUE,
				   0, 0, it->w, it->h);
		}
	else if (imd->zoom == 1.0 || imd->scale == 1.0)
		{
		if (has_alpha)
			{
			gdk_pixbuf_composite_color(imd->pixbuf, it->pixbuf, x, y, w, h,
					 (double) 0.0 - it->x,
					 (double) 0.0 - it->y,
					 1.0, 1.0, GDK_INTERP_NEAREST,
					 255, it->x + x, it->y + y,
					 IMAGE_ALPHA_CHECK_SIZE, IMAGE_ALPHA_CHECK1, IMAGE_ALPHA_CHECK2);
			draw = TRUE;
			}
		else
			{
			/* faster, simple */
			gdk_pixbuf_render_to_drawable(imd->pixbuf, it->pixmap,
        		                              imd->image->style->fg_gc[GTK_WIDGET_STATE(imd->image)],
        	        	                      it->x + x, it->y + y,
        	                	              x, y,
        	                        	      w, h,
		                                      (GdkRgbDither)dither_quality, it->x + x, it->y + y);
			}
		}
	else
		{
		double scale_x, scale_y;

		if (imd->image_width == 0 || imd->image_height == 0) return;
		scale_x = (double)imd->width / imd->image_width;
		scale_y = (double)imd->height / imd->image_height;

#ifdef IMAGE_DEBUG_ENABLE
		printf("image.c (render scaled): %d , %d (%d x %d) offset %d, %d (%f : %f) [%d, %d]\n",
			x, y, w, h, 0 - it->x, 0 - it->y, scale_x, scale_y, it->x, it->y);
		printf("image.c (extras): src %d x %d dst %d x %d\n",
			gdk_pixbuf_get_width(imd->pixbuf), gdk_pixbuf_get_height(imd->pixbuf),
			gdk_pixbuf_get_width(it->pixbuf), gdk_pixbuf_get_height(it->pixbuf));
#endif

/* I don't know, this just seems to break whenever the tile is for the right side of the image
 * (note that GDK_INTERP_NEAREST does not show the brokenness ?
 *
 * UPDATE:
 * Ok, here is the deal: gdk_pixbuf_scale() is broken for anything but INTERP_NEAREST,
 * and gdk_pixbuf_composite_color() is only broken for INTERP_TILES (my favorite render type, btw)
 * so here is what we are gonna do: use gdk_pixbuf_composite_color(), and complain to gdk-pixbuf authors.
 *
 * UPDATE2: even gdk_pixbuf_composite_color() can crash under certain circumstances, maybe I have a bug
 * in here, somewhere. (Although GDK_INTERP_NEAREST never crashes with the same image and zoom/scale).
 *
 * UPDATE3: well, with gdk-pixbuf 0.10.0 it looks like some of the problems were fixed, only TILES
 * still appears to crash, in fact it now crashes in composite_color() too, so INTERP_TILES support
 * is dropped till it is fixed.
 *
 * UPDATE4: SCRATCH-THAT! gdk_pixbuf_scale is still broke! BAAH!
 */

#ifdef GDK_PIXBUF_TILES_BROKEN
		if (!has_alpha && (GdkInterpType)zoom_quality == GDK_INTERP_NEAREST)
#else
		if (!has_alpha)
#endif
			{
			gdk_pixbuf_scale(imd->pixbuf, it->pixbuf, x, y, w, h,
					 (double) 0.0 - it->x,
					 (double) 0.0 - it->y,
					 scale_x, scale_y, (GdkInterpType)zoom_quality);
			}
		else
			{
			gdk_pixbuf_composite_color(imd->pixbuf, it->pixbuf, x, y, w, h,
					 (double) 0.0 - it->x,
					 (double) 0.0 - it->y,
					 scale_x, scale_y, (GdkInterpType)zoom_quality,
					 255, it->x + x, it->y + y,
					 IMAGE_ALPHA_CHECK_SIZE, IMAGE_ALPHA_CHECK1, IMAGE_ALPHA_CHECK2);
			}
		draw = TRUE;
		}

	if (draw && it->pixbuf && !it->blank)
		{
		gdk_pixbuf_render_to_drawable(it->pixbuf, it->pixmap,
					      imd->image->style->fg_gc[GTK_WIDGET_STATE(imd->image)],
					      x, y,
					      x, y,
					      w, h,
					      (GdkRgbDither)dither_quality, it->x + x, it->y + y);
		}

	it->render_todo = TILE_RENDER_NONE;
}

static void image_tile_expose(ImageWindow *imd, ImageTile *it, gint x, gint y, gint w, gint h, gint new_data)
{
	image_tile_render(imd, it, x, y, w, h, new_data);

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf("expose (%d, %d): +%d, +%d (%d x %d)\n", it->x, it->y, x, y, w, h);
#endif

	gdk_draw_pixmap (imd->image->window, imd->image->style->fg_gc[GTK_WIDGET_STATE(imd->image)],
			 it->pixmap, x, y,
			 imd->x_offset + (it->x - imd->x_scroll) + x, imd->y_offset + (it->y - imd->y_scroll) + y, w, h);
}

static gint image_tile_is_visible(ImageWindow *imd, ImageTile *it)
{
	return (it->x + it->w >= imd->x_scroll && it->x <= imd->x_scroll + imd->window_width - imd->x_offset * 2 &&
		it->y + it->h >= imd->y_scroll && it->y <= imd->y_scroll + imd->window_height - imd->y_offset * 2);
}

/*
 *-------------------------------------------------------------------
 * render queue
 *-------------------------------------------------------------------
 */


static gint image_queue_draw_idle_cb(gpointer data)
{
	ImageWindow *imd = data;
	QueueData *qd;

	if (!imd->pixbuf || !imd->draw_queue || imd->draw_idle_id == -1)
		{

#ifdef IMAGE_DEBUG_ENABLE
		if (debug) printf("queue: abort\n");
#endif

		imd->draw_idle_id = -1;
		return FALSE;
		}

	qd = imd->draw_queue->data;

	if (GTK_WIDGET_REALIZED(imd->image))
		{
		if (image_tile_is_visible(imd, qd->it))
			{
			image_tile_expose(imd, qd->it, qd->x, qd->y, qd->w, qd->h, qd->new_data);
			}
		else if (qd->new_data)
			{
			/* if new pixel data, and we already have a pixmap, update the tile */
			qd->it->blank = FALSE;
			if (qd->it->pixmap && qd->it->render_done == TILE_RENDER_ALL)
				{
				image_tile_render(imd, qd->it, qd->x, qd->y, qd->w, qd->h, qd->new_data);
				}
			}
		}

	imd->draw_queue = g_list_remove(imd->draw_queue, qd);
	g_free(qd);

	if (!imd->draw_queue)
		{

#ifdef IMAGE_DEBUG_ENABLE
		if (debug) printf("queue: done\n");
#endif

		imd->draw_idle_id = -1;
		return FALSE;
		}

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf("queue: continue\n");
#endif

	return TRUE;
}

static QueueData *image_queue_combine(ImageWindow *imd, QueueData *qd)
{
	QueueData *found = NULL;
	GList *work;

#ifdef IMAGE_DEBUG_ENABLE
	if (qd->it->x + qd->x + qd->w > imd->width || qd->it->y + qd->y + qd->h > imd->height)
		{
		printf("WARNING: queue exceeds image bounds!\n");
		}
#endif

	work = imd->draw_queue;
	while(work && !found)
		{
		found = work->data;
		work = work->next;

		if (found->it != qd->it) found = NULL;
		}

	if (found)
		{
		if (found->x + found->w < qd->x + qd->w) found->w += (qd->x + qd->w) - (found->x + found->w);
		if (found->x > qd->x)
			{
			found->w += found->x - qd->x;
			found->x = qd->x;
			}

		if (found->y + found->h < qd->y + qd->h) found->h += (qd->y + qd->h) - (found->y + found->h);
		if (found->y > qd->y)
			{
			found->h += found->y - qd->y;
			found->y = qd->y;
			}
		found->new_data |= qd->new_data;
		}

	return found;
}

static gint image_clamp_to_visible(ImageWindow *imd, gint *x, gint *y, gint *w, gint *h)
{
	gint nx, ny;
	gint nw, nh;
	gint vx, vy;
	gint vw, vh;

	vw = imd->vis_width;
	vh = imd->vis_height;

	vx = imd->x_scroll;
	vy = imd->y_scroll;

	if (*x + *w < vx || *x > vx + vw || *y + *h < vy || *y > vy + vh) return FALSE;

	/* now clamp it */
	nx = CLAMP(*x, vx, vx + vw);
	nw = CLAMP(*w - (nx - *x), 1, vw);

	ny = CLAMP(*y, vy, vy + vh);
	nh = CLAMP(*h - (ny - *y), 1, vh);

	*x = nx;
	*y = ny;
	*w = nw;
	*h = nh;

	return TRUE;
}

static gint image_queue_to_tiles(ImageWindow *imd, gint x, gint y, gint w, gint h,
				 gint clamp, TileRenderType render, gint new_data)
{
	gint i, j;
	gint x1, x2;
	gint y1, y2;
	GList *work;

	if (clamp && !image_clamp_to_visible(imd, &x, &y, &w, &h)) return FALSE;

	x1 = (gint)floor(x / imd->tile_width) * imd->tile_width;
	x2 = (gint)ceil((x + w) / imd->tile_width) * imd->tile_width;

	y1 = (gint)floor(y / imd->tile_height) * imd->tile_height;
	y2 = (gint)ceil((y + h) / imd->tile_height) * imd->tile_height;

	work = g_list_nth(imd->tiles, y1 / imd->tile_height * imd->tile_cols + (x1 / imd->tile_width));
	for (j = y1; j <= y2; j += imd->tile_height)
		{
		GList *tmp;
		tmp = work;
		for (i = x1; i <= x2; i += imd->tile_width)
			{
			if (tmp)
				{
				ImageTile *it = tmp->data;
				QueueData *qd;

				if ((render == TILE_RENDER_ALL && it->render_done != TILE_RENDER_ALL) ||
				    (render == TILE_RENDER_AREA && it->render_todo != TILE_RENDER_ALL))
					{
					it->render_todo = render;
					}

				qd = g_new(QueueData, 1);
				qd->it = it;
				qd->new_data = new_data;

				if (i < x)
					{
					qd->x = x - i;
					}
				else
					{
					qd->x = 0;
					}
				qd->w = x + w - i - qd->x;
				if (qd->x + qd->w > imd->tile_width) qd->w = imd->tile_width - qd->x;


				if (j < y)
					{
					qd->y = y - j;
					}
				else
					{
					qd->y = 0;
					}
				qd->h = y + h - j - qd->y;
				if (qd->y + qd->h > imd->tile_height) qd->h = imd->tile_height - qd->y;

#ifdef IMAGE_DEBUG_ENABLE
				printf("add tile: [#%3d] [%3d, %3d (%3d x %3d)] %3d, %3d (%3d x %3d) new data:%d\n",
					g_list_index(imd->tiles, it), it->x, it->y, it->w, it->h,
					qd->x, qd->y, qd->w, qd->h, qd->new_data);
#endif

				if (qd->w < 1 || qd->h < 1 || /* <--- sanity checks, rare cases cause this */
				    image_queue_combine(imd, qd))
					{

#ifdef IMAGE_DEBUG_ENABLE
					if (debug) printf("queue combined (%d, %d): +%d, +%d (%d x %d)\n", i, j, qd->x, qd->y, qd->w, qd->h);
#endif

					g_free(qd);
					}
				else
					{
					imd->draw_queue = g_list_append(imd->draw_queue, qd);

#ifdef IMAGE_DEBUG_ENABLE
					if (debug) printf("queue added (%d, %d): +%d, +%d (%d x %d)\n", i, j, qd->x, qd->y, qd->w, qd->h);
#endif

					}

#ifdef IMAGE_DEBUG_ENABLE
				if (debug) printf("queued it: (%d, %d)\n", it->x, it->y);
#endif

				tmp = tmp->next;
				}
			}
		work = g_list_nth(work, imd->tile_cols);	/* step 1 row */
		}

	return TRUE;
}

static void image_queue(ImageWindow *imd, gint x, gint y, gint w, gint h,
			gint clamp, TileRenderType render, gint new_data)
{
	gint nx, ny;

	nx = CLAMP(x, 0, imd->width - 1);
	ny = CLAMP(y, 0, imd->height - 1);
	w -= (nx - x);
	h -= (nx - x);
	w = CLAMP(w, 0, imd->width - x);
	h = CLAMP(h, 0, imd->height - y);
	if (w < 1 || h < 1) return;

	if (image_queue_to_tiles(imd, x, y, w, h, clamp, render, new_data) &&
	    (!imd->draw_queue || imd->draw_idle_id == -1))
		{
		imd->draw_idle_id = gtk_idle_add_priority(G_PRIORITY_HIGH_IDLE, image_queue_draw_idle_cb, imd);
		}
}

static void image_queue_clear(ImageWindow *imd)
{
	GList *work;

	work = imd->draw_queue;
	while(work)
		{
		QueueData *qd;

		qd = work->data;
		work = work->next;
		g_free(qd);
		}

	g_list_free(imd->draw_queue);
	imd->draw_queue = NULL;

	if (imd->draw_idle_id != -1) gtk_idle_remove(imd->draw_idle_id);
	imd->draw_idle_id = -1;
}

/*
 *-------------------------------------------------------------------
 * core calculations
 *-------------------------------------------------------------------
 */

static gint image_top_window_sizable(ImageWindow *imd)
{
	if (!imd->top_window) return FALSE;
	if (!fit_window) return FALSE;
	if (!imd->top_window_sync) return FALSE;
	if (!imd->eventbox->window) return FALSE;

	return TRUE;
}

static gint image_size_top_window(ImageWindow *imd, gint w, gint h)
{
	gint ww, wh;

	if (!image_top_window_sizable(imd)) return FALSE;

	if (limit_window_size)
		{
		gint sw = gdk_screen_width() * max_window_size / 100;
		gint sh = gdk_screen_height() * max_window_size / 100;

		if (w > sw) w = sw;
		if (h > sh) h = sh;
		}

	w += (imd->top_window->allocation.width - imd->eventbox->allocation.width);
	h += (imd->top_window->allocation.height - imd->eventbox->allocation.height);

	gdk_window_get_size(imd->top_window->window, &ww, &wh);
	if (w == ww && h == wh) return FALSE;

#ifdef IMAGE_DEBUG_ENABLE
	printf("auto sized to %d x %d (was %d x %d)\n", w, h, ww, wh);
#endif

	gdk_window_resize(imd->top_window->window, w, h);

	return TRUE;
}

static void image_redraw(ImageWindow *imd, gint new_data)
{
	image_queue_clear(imd);
	image_queue(imd, 0, 0, imd->width, imd->height, TRUE, TILE_RENDER_ALL, new_data);
}

static void image_border_clear(ImageWindow *imd)
{
	if (!imd->image->window) return;

	if (imd->vis_width < imd->window_width)
		{
		if (imd->x_offset > 0)
			{
			gdk_window_clear_area(imd->image->window,
					      0, 0,
					      imd->x_offset, imd->window_height);
			}
		if (imd->window_width - imd->vis_width - imd->x_offset > 0)
			{
			gdk_window_clear_area(imd->image->window,
					      imd->vis_width + imd->x_offset, 0,
					      imd->window_width - imd->vis_width - imd->x_offset, imd->window_height);
			}
		}
	if (imd->vis_height < imd->window_height)
		{
		if (imd->y_offset > 0)
			{
			gdk_window_clear_area(imd->image->window,
					      imd->x_offset, 0,
					      imd->vis_width, imd->y_offset);
			}
		if (imd->window_height - imd->vis_height - imd->y_offset > 0)
			{
			gdk_window_clear_area(imd->image->window,
					      imd->x_offset, imd->vis_height + imd->y_offset,
					      imd->vis_width, imd->window_height - imd->vis_height - imd->y_offset);
			}
		}
}

static gint image_scroll_clamp(ImageWindow *imd)
{
	gint old_xs;
	gint old_ys;

	if (imd->zoom == 0.0)
		{
		imd->x_scroll = 0;
		imd->y_scroll = 0;
		return FALSE;
		}

	old_xs = imd->x_scroll;
	old_ys = imd->y_scroll;

	if (imd->x_offset > 0)
		{
		imd->x_scroll = 0;
		}
	else
		{
		imd->x_scroll = CLAMP(imd->x_scroll, 0, imd->width - imd->vis_width);
		}

	if (imd->y_offset > 0)
		{
		imd->y_scroll = 0;
		}
	else
		{
		imd->y_scroll = CLAMP(imd->y_scroll, 0, imd->height - imd->vis_height);
		}

	return (old_xs != imd->x_scroll || old_ys != imd->y_scroll);
}

static gint image_zoom_clamp(ImageWindow *imd, gfloat zoom, gint force, gint new)
{
	gint w, h;
	gfloat scale;

	zoom = CLAMP(zoom, IMAGE_ZOOM_MIN, IMAGE_ZOOM_MAX);

	if (imd->zoom == zoom && !force) return FALSE;

	w = imd->image_width;
	h = imd->image_height;

	if (zoom == 0.0)
		{
		gint max_w;
		gint max_h;
		gint sizeable;

		sizeable = (new && image_top_window_sizable(imd));

		if (sizeable)
			{
			max_w = gdk_screen_width();
			max_h = gdk_screen_height();

			if (limit_window_size)
				{
				max_w = max_w * max_window_size / 100;
				max_h = max_h * max_window_size / 100;
				}
			}
		else
			{
			max_w = imd->window_width;
			max_h = imd->window_height;
			}

		if ((zoom_to_fit_expands && !sizeable) || w > max_w || h > max_h)
			{
			if ((gfloat)max_w / w > (gfloat)max_h / h)
				{
				scale = (gfloat)max_h / h;
				h = max_h;
				w = w * scale + 0.5;
				if (w > max_w) w = max_w;
				}
			else
				{
				scale = (gfloat)max_w / w;
				w = max_w;
				h = h * scale + 0.5;
				if (h > max_h) h = max_h;
				}
			if (w < 1) w = 1;
			if (h < 1) h = 1;
			}
		else
			{
			scale = 1.0;
			}
		}
	else if (zoom > 0.0) /* zoom orig, in */
		{
		scale = zoom;
		w = w * scale;
		h = h * scale;
		}
	else /* zoom out */
		{
		scale = 1.0 / (0.0 - zoom);
		w = w * scale;
		h = h * scale;
		}

	imd->zoom = zoom;
	imd->width = w;
	imd->height = h;
	imd->scale = scale;

	return TRUE;
}

static gint image_size_clamp(ImageWindow *imd)
{
	gint old_vw, old_vh;

	old_vw = imd->vis_width;
	old_vh = imd->vis_height;

	if (imd->width < imd->window_width)
		{
		imd->vis_width = imd->width;
		imd->x_offset = (imd->window_width - imd->width) / 2;
		}
	else
		{
		imd->vis_width = imd->window_width;
		imd->x_offset = 0;
		}

	if (imd->height < imd->window_height)
		{
		imd->vis_height = imd->height;
		imd->y_offset = (imd->window_height - imd->height) / 2;
		}
	else
		{
		imd->vis_height = imd->window_height;
		imd->y_offset = 0;
		}

#ifdef IMAGE_DEBUG_ENABLE
	printf("size clamp: %d x %d (off: %d, %d)\n", imd->width, imd->height, imd->x_offset, imd->y_offset);
#endif

	return (old_vw != imd->vis_width || old_vh != imd->vis_height);
}

static void image_size_sync(ImageWindow *imd, gint new_width, gint new_height)
{
	if (imd->window_width == new_width && imd->window_height == new_height) return;

	imd->window_width = new_width;
	imd->window_height = new_height;

	if (imd->zoom == 0.0) image_zoom_clamp(imd, 0.0, TRUE, FALSE);

	image_size_clamp(imd);
	image_scroll_clamp(imd);

	gtk_widget_set_usize(imd->image, imd->window_width, imd->window_height);

	/* clear any borders */
	image_border_clear(imd);
	
	image_tile_sync(imd, imd->width, imd->height, FALSE);
	image_redraw(imd, FALSE);

	if (imd->title_show_zoom) image_update_title(imd);
	image_update_util(imd);
}

/*
 *-------------------------------------------------------------------
 * misc
 *-------------------------------------------------------------------
 */

static void image_update_title(ImageWindow *imd)
{
	gchar *title = NULL;
	gchar *zoom = NULL;
	gchar *collection = NULL;

	if (!imd->top_window) return;

	if (imd->collection && collection_to_number(imd->collection) >= 0)
		{
		const gchar *name;
		name = imd->collection->name;
		if (!name) name = _("Untitled");
		collection = g_strdup_printf(" (Collection %s)", name);
		}

	if (imd->title_show_zoom)
		{
		gchar *buf = image_zoom_get_as_text(imd);
		zoom = g_strconcat(" [", buf, "]", NULL);
		g_free(buf);
		}

	title = g_strdup_printf("%s%s%s%s%s",
		imd->title ? imd->title : "",
		imd->image_name ? imd->image_name : "",
		zoom ? zoom : "",
		collection ? collection : "",
		imd->title_right ? imd->title_right : "");

	gtk_window_set_title(GTK_WINDOW(imd->top_window), title);

	g_free(title);
	g_free(zoom);
	g_free(collection);
}

static void image_update_util(ImageWindow *imd)
{
	if (imd->update_func) imd->update_func(imd, imd->update_data);
}

static void image_scroll_real(ImageWindow *imd, gint x, gint y)
{
	gint old_x, old_y;
	gint x_off, y_off;
	gint w, h;

	old_x = imd->x_scroll;
	old_y = imd->y_scroll;

	imd->x_scroll += x;
	imd->y_scroll += y;

#ifdef IMAGE_DEBUG_ENABLE
	printf("scroll by %d, %d\n", x, y);
#endif

	image_scroll_clamp(imd);
	if (imd->x_scroll == old_x && imd->y_scroll == old_y) return;

	x_off = imd->x_scroll - old_x;
	y_off = imd->y_scroll - old_y;

	w = imd->vis_width - abs(x_off);
	h = imd->vis_height - abs(y_off);

	if (w < 1 || h < 1)
		{
		/* scrolled completely to new material */
		image_queue(imd, 0, 0, imd->width, imd->height, TRUE, TILE_RENDER_ALL, FALSE);
		return;
		}
	else
		{
		gint x1, y1;
		gint x2, y2;
		GdkGC *gc;
		GdkEvent *event;

		if (x_off < 0)
			{
			x1 = abs(x_off);
			x2 = 0;
			}
		else
			{
			x1 = 0;
			x2 = abs(x_off);
			}

		if (y_off < 0)
			{
			y1 = abs(y_off);
			y2 = 0;
			}
		else
			{
			y1 = 0;
			y2 = abs(y_off);
			}

		gc = gdk_gc_new(imd->image->window);
		gdk_gc_set_exposures(gc, TRUE);
		gdk_window_copy_area(imd->image->window, gc,
				     x1 + imd->x_offset, y1 + imd->y_offset,
				     imd->image->window,
				     x2 + imd->x_offset, y2 + imd->y_offset, w, h);
		gdk_gc_destroy(gc);

		/*
		 * RACE CONDITION avoidance for expose_event and scrolling (hack):
		 * [borrowed from eog - Eye of Gnome]
		 * this is really terrible to do, but the expose_events must be done right now,
		 * otherwise the signal may not be sent until after another scroll and that would
		 * cause portions of the image to never be redrawn (the part needing update now would
		 * then be in a different screen location).
		 */

		while ((event = gdk_event_get_graphics_expose(imd->image->window)) != NULL)
			{
			gtk_widget_event(imd->image, event);
			if (event->expose.count == 0)
				{
				gdk_event_free(event);
				break;
				}
			gdk_event_free(event);
			}

		w = imd->vis_width - w;
		h = imd->vis_height - h;

		if (w > 0)
			{
			image_queue(imd,
				    x_off > 0 ? imd->x_scroll + (imd->vis_width - w) : imd->x_scroll, imd->y_scroll,
				    w, imd->vis_height, TRUE, TILE_RENDER_ALL, FALSE);
			}
		if (h > 0)
			{
			/* FIXME, to optimize this, remove overlap */
			image_queue(imd,
				    imd->x_scroll, y_off > 0 ? imd->y_scroll + (imd->vis_height - h) : imd->y_scroll,
				    imd->vis_width, h, TRUE, TILE_RENDER_ALL, FALSE);
			}
		}

	
}

static void widget_set_cursor(GtkWidget *widget, gint icon)
{
	GdkCursor *cursor;

	if (!widget->window) return;

	if (icon == -1)
		{
		cursor = NULL;
		}
	else
		{
		cursor = gdk_cursor_new (icon);
		}

	gdk_window_set_cursor(widget->window, cursor);

	if (cursor) gdk_cursor_destroy(cursor);
}

/*
 *-------------------------------------------------------------------
 * image pixbuf handling
 *-------------------------------------------------------------------
 */

static void image_zoom_sync(ImageWindow *imd, gfloat zoom, gint force, gint blank, gint new)
{
	gfloat old_scale;
	gint old_cx, old_cy;
	gint sized;

	old_scale = imd->scale;
	old_cx = imd->x_scroll + imd->vis_width / 2;
	old_cy = imd->y_scroll + imd->vis_height / 2;

	if (!image_zoom_clamp(imd, zoom, force, new)) return;

	if (image_size_clamp(imd))
		{
		sized = image_size_top_window(imd, imd->width, imd->height);
		if (!sized) image_border_clear(imd);
		}
	else
		{
		sized = image_size_top_window(imd, imd->width, imd->height);
		}

	if (!force)
		{
		/* user zoom does not force, so keep visible center point */
		imd->x_scroll = old_cx / old_scale * imd->scale - (imd->vis_width / 2);
		imd->y_scroll = old_cy / old_scale * imd->scale - (imd->vis_height / 2);
		}
	image_scroll_clamp(imd);

	image_tile_sync(imd, imd->width, imd->height, blank);

	/* Only call redraw if the window size was not adjusted, since that will call the
	 * window size signal, which in turn will call a redraw with everything at a valid size.
	 * (imd->image is not adjusted until return to main loop)
	 */
	if (!sized) image_redraw(imd, FALSE);

	if (imd->title_show_zoom) image_update_title(imd);
	image_update_util(imd);
}

static void image_pixbuf_sync(ImageWindow *imd, gfloat zoom, gint blank, gint new)
{
#ifdef IMAGE_DEBUG_ENABLE
	printf("syncing pixbuf\n");
#endif

	if (!imd->pixbuf)
		{
		/* no pixbuf so just clear the window */
		imd->image_width = 0;
		imd->image_height = 0;

		if (imd->image->window) gdk_window_clear(imd->image->window);

		image_update_util(imd);
		
		return;
		}

	imd->image_width = gdk_pixbuf_get_width(imd->pixbuf);
	imd->image_height = gdk_pixbuf_get_height(imd->pixbuf);

	/* reset scrolling */
	imd->x_scroll = 0;
	imd->y_scroll = 0;

	image_zoom_sync(imd, zoom, TRUE, blank, new);
}

static void image_set_pixbuf(ImageWindow *imd, GdkPixbuf *pixbuf, gfloat zoom, gint new)
{
	if (pixbuf) gdk_pixbuf_ref(pixbuf);
	if (imd->pixbuf) gdk_pixbuf_unref(imd->pixbuf);
	imd->pixbuf = pixbuf;

	image_pixbuf_sync(imd, zoom, FALSE, new);
}

static void image_alter_real(ImageWindow *imd, AlterType type)
{
	GdkPixbuf *new = NULL;

	imd->delay_alter_type = ALTER_NONE;

	if (!imd->pixbuf) return;

	switch (type)
		{
		case ALTER_ROTATE_90:
			new = pixbuf_copy_rotate_90(imd->pixbuf, FALSE);
			break;
		case ALTER_ROTATE_90_CC:
			new = pixbuf_copy_rotate_90(imd->pixbuf, TRUE);
			break;
		case ALTER_ROTATE_180:
			new = pixbuf_copy_mirror(imd->pixbuf, TRUE, TRUE);
			break;
		case ALTER_MIRROR:
			new = pixbuf_copy_mirror(imd->pixbuf, TRUE, FALSE);
			break;
		case ALTER_FLIP:
			new = pixbuf_copy_mirror(imd->pixbuf, FALSE, TRUE);
			break;
		case ALTER_NONE:
		default:
			return;
			break;
		}

	if (new)
		{
		image_set_pixbuf(imd, new, imd->zoom, TRUE);
		gdk_pixbuf_unref(new);
		}
}

/*
 *-------------------------------------------------------------------
 * read ahead (prebuffer)
 *-------------------------------------------------------------------
 */

static void image_read_ahead_cancel(ImageWindow *imd)
{

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf("read ahead cancelled for :%s\n", imd->read_ahead_path);
#endif

	image_loader_free(imd->read_ahead_il);
	imd->read_ahead_il = NULL;

	if (imd->read_ahead_pixbuf) gdk_pixbuf_unref(imd->read_ahead_pixbuf);
	imd->read_ahead_pixbuf = NULL;

	g_free(imd->read_ahead_path);
	imd->read_ahead_path = NULL;
}

static void image_read_ahead_done_cb(ImageLoader *il, gpointer data)
{
	ImageWindow *imd = data;

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf("read ahead done for :%s\n", imd->read_ahead_path);
#endif

	imd->read_ahead_pixbuf = image_loader_get_pixbuf(imd->read_ahead_il);
	if (imd->read_ahead_pixbuf)
		{
		gdk_pixbuf_ref(imd->read_ahead_pixbuf);
		}
	else
		{
		imd->read_ahead_pixbuf = gdk_pixbuf_new_from_xpm_data(img_unknown);
		}
	image_loader_free(imd->read_ahead_il);
	imd->read_ahead_il = NULL;
}

static void image_read_ahead_error_cb(ImageLoader *il, gpointer data)
{
	/* we even treat errors as success, maybe at least some of the file was ok */
	image_read_ahead_done_cb(il, data);
}

static void image_read_ahead_start(ImageWindow *imd)
{
	/* already started ? */
	if (!imd->read_ahead_path || imd->read_ahead_il || imd->read_ahead_pixbuf) return;

	/* still loading ?, do later */
	if (imd->il) return;

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf("read ahead started for :%s\n", imd->read_ahead_path);
#endif

	imd->read_ahead_il = image_loader_new(imd->read_ahead_path);

	image_loader_set_error_func(imd->read_ahead_il, image_read_ahead_error_cb, imd);
	if (!image_loader_start(imd->read_ahead_il, image_read_ahead_done_cb, imd))
		{
		image_read_ahead_cancel(imd);
		}
}

static void image_read_ahead_set(ImageWindow *imd, const gchar *path)
{
	if (imd->read_ahead_path && path && strcmp(imd->read_ahead_path, path) == 0) return;

	image_read_ahead_cancel(imd);

	imd->read_ahead_path = g_strdup(path);

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf("read ahead set to :%s\n", imd->read_ahead_path);
#endif

	image_read_ahead_start(imd);
}

/*
 *-------------------------------------------------------------------
 * loading
 *-------------------------------------------------------------------
 */

static void image_load_pixbuf_ready(ImageWindow *imd)
{
	if (imd->pixbuf || !imd->il) return;

	imd->pixbuf = image_loader_get_pixbuf(imd->il);

	if (imd->pixbuf) gdk_pixbuf_ref(imd->pixbuf);

	image_pixbuf_sync(imd, imd->zoom, TRUE, TRUE);
}

static void image_load_area_cb(ImageLoader *il, guint x, guint y, guint w, guint h, gpointer data)
{
	ImageWindow *imd = data;

#ifdef IMAGE_DEBUG_ENABLE
	printf("image.c: area ready %d, %d (%d x %d)\n", (gint) x, (gint) y, (gint) w, (gint) h);
#endif

	if (!imd->pixbuf) image_load_pixbuf_ready(imd);

	if (imd->scale != 1.0)
		{
		x = (guint) floor((float)x * imd->scale);
		y = (guint) floor((float)y * imd->scale);
		w = (guint) ceil((float)w * imd->scale);
		h = (guint) ceil((float)h * imd->scale);

		if (w == 0) w = 1;
		if (h == 0) h = 1;

		if ((GdkInterpType)zoom_quality != GDK_INTERP_NEAREST)
			{
			/* some scaling types use surrounding pixels to smooth the image,
			 * this will expand the new area to cover up for faint black
			 * lines caused by previous renders with non-complete image
			 */
			y -= 1;
			h += 2;
			}

#ifdef IMAGE_DEBUG_ENABLE
		if (debug) printf("image.c: area ready adjusted %d, %d (%d x %d)\n", (gint) x, (gint) y, (gint) w, (gint) h);
#endif
		}

	image_queue(imd, (gint) x, (gint) y, (gint) w, (gint) h, FALSE, TILE_RENDER_AREA, TRUE);
}

static void image_load_done_cb(ImageLoader *il, gpointer data)
{
	ImageWindow *imd = data;

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf ("image done\n");
#endif

	image_loader_free(imd->il);
	imd->il = NULL;

	if (imd->delay_alter_type != ALTER_NONE)
		{
		image_alter_real(imd, imd->delay_alter_type);
		}

	image_read_ahead_start(imd);
}

static void image_load_error_cb(ImageLoader *il, gpointer data)
{

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf ("image error\n");
#endif

	/* even on error handle it like it was done,
	 * since we have a pixbuf with _something_ */

	image_load_done_cb(il, data);
}

/* this read ahead is located here merely for the callbacks, above */

static gint image_read_ahead_check(ImageWindow *imd)
{
	if (!imd->read_ahead_path) return FALSE;
	if (imd->il || imd->pixbuf) return FALSE;

	if (!imd->image_path || strcmp(imd->read_ahead_path, imd->image_path) != 0)
		{
		image_read_ahead_cancel(imd);
		return FALSE;
		}

	if (imd->read_ahead_il)
		{
		imd->il = imd->read_ahead_il;
		imd->read_ahead_il = NULL;

		/* override the old signals */
		image_loader_set_area_ready_func(imd->il, image_load_area_cb, imd);
		image_loader_set_error_func(imd->il, image_load_error_cb, imd);
		image_loader_set_buffer_size(imd->il, IMAGE_LOAD_BUFFER_COUNT);
		/* do this one directly (probably should add a set func) */
		imd->il->func_done = image_load_done_cb;

		imd->pixbuf = image_loader_get_pixbuf(imd->il);
		if (imd->pixbuf) gdk_pixbuf_ref(imd->pixbuf);

		image_read_ahead_cancel(imd);
		return TRUE;
		}
	else if (imd->read_ahead_pixbuf)
		{
		imd->pixbuf = imd->read_ahead_pixbuf;
		imd->read_ahead_pixbuf = NULL;

		image_read_ahead_cancel(imd);
		return TRUE;
		}

	image_read_ahead_cancel(imd);
	return FALSE;
}

static gint image_load_begin(ImageWindow *imd, const gchar *path)
{

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf ("image begin \n");
#endif

	if (imd->il) return FALSE;

	if (image_read_ahead_check(imd))
		{

#ifdef IMAGE_DEBUG_ENABLE
		if (debug) printf("from read ahead buffer: %s\n", imd->image_path);
#endif

		image_pixbuf_sync(imd, imd->zoom, FALSE, TRUE);
		return TRUE;
		}

	imd->il = image_loader_new(path);

	image_loader_set_area_ready_func(imd->il, image_load_area_cb, imd);
	image_loader_set_error_func(imd->il, image_load_error_cb, imd);
	image_loader_set_buffer_size(imd->il, IMAGE_LOAD_BUFFER_COUNT);

	if (!image_loader_start(imd->il, image_load_done_cb, imd))
		{

#ifdef IMAGE_DEBUG_ENABLE
		if (debug) printf("image start error\n");
#endif

		image_loader_free(imd->il);
		imd->il = NULL;
		return FALSE;
		}

	if (!imd->pixbuf) image_load_pixbuf_ready(imd);

	return TRUE;
}

static void image_reset(ImageWindow *imd)
{
	/* stops anything currently being done */

#ifdef IMAGE_DEBUG_ENABLE
	if (debug) printf("image reset\n");
#endif

	image_loader_free(imd->il);
	imd->il = NULL;

	image_queue_clear(imd);
	imd->delay_alter_type = ALTER_NONE;
}

/*
 *-------------------------------------------------------------------
 * image changer
 *-------------------------------------------------------------------
 */

static void image_change_complete(ImageWindow *imd, gfloat zoom, gint new)
{
	gint sync = TRUE;

	if (imd->pixbuf) gdk_pixbuf_unref(imd->pixbuf);
	imd->pixbuf = NULL;

	imd->zoom = zoom;	/* store the zoom, needed by the loader */

	if (imd->image_path && isfile(imd->image_path))
		{
		image_reset(imd);
		image_load_begin(imd, imd->image_path);
		if (imd->pixbuf)
			{
			imd->unknown = FALSE;
			sync = FALSE;
			}
		else
			{
			imd->pixbuf = gdk_pixbuf_new_from_xpm_data(img_unknown);
			imd->unknown = TRUE;
			}
		imd->size = filesize(imd->image_path);
		}
	else
		{
		if (imd->image_path)
			{
			imd->pixbuf = gdk_pixbuf_new_from_xpm_data(img_unknown);
			}
		else
			{
			imd->pixbuf = NULL;
			}
		imd->unknown = TRUE;
		imd->size = 0;
		}

	if (sync)
		{
		image_pixbuf_sync(imd, zoom, FALSE, new);
		}
	else
		{
		image_update_util(imd);
		}
}

static void image_change_real(ImageWindow *imd, const gchar *path,
			      CollectionData *cd, CollectInfo *info, gfloat zoom)
{
	imd->collection = cd;
	imd->collection_info = info;

	if (!path) zoom = 1.0;

	image_set_path(imd, path);
	image_change_complete(imd, zoom, TRUE);
}

/*
 *-------------------------------------------------------------------
 * callbacks
 *-------------------------------------------------------------------
 */

static void image_expose_cb(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	gint x, y;

	ImageWindow *imd = data;

	if (!imd->pixbuf) return;

	/* border */

	/* image */
	x = MAX(0, (gint)event->area.x - imd->x_offset + imd->x_scroll);
	y = MAX(0, (gint)event->area.y - imd->y_offset + imd->y_scroll);

#ifdef IMAGE_DEBUG_ENABLE
	printf("expose event: %d, %d (%d x %d) - [%d x %d (%d x %d)]\n",
		(gint)event->area.x, (gint)event->area.y,
		(gint)event->area.width, (gint)event->area.height,
		x, y, MIN((gint)event->area.width, imd->width - x), MIN((gint)event->area.height, imd->height - y));
#endif

	image_queue(imd, x, y,
		    MIN((gint)event->area.width, imd->width - x),
		    MIN((gint)event->area.height, imd->height - y),
		    FALSE, TILE_RENDER_ALL, FALSE);
}

static gint image_size_cb(GtkWidget *widget, GtkAllocation *allocation, gpointer data)
{
	ImageWindow *imd = data;

#ifdef IMAGE_DEBUG_ENABLE
	printf("Image window size event: %d x %d\n", allocation->width, allocation->height);
#endif

	image_size_sync(imd, allocation->width, allocation->height);

	return FALSE;
}

/*
 *-------------------------------------------------------------------
 * focus stuff
 *-------------------------------------------------------------------
 */

static void image_focus_paint(ImageWindow *imd)
{
	GtkWidget *widget;

	widget = imd->widget;
	if (!widget->window) return;

	gtk_paint_focus (widget->style, widget->window,
			 NULL, widget, "image_window",
			 widget->allocation.x, widget->allocation.y,
			 widget->allocation.width - 1, widget->allocation.height - 1);	
}

static gint image_focus_expose(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	ImageWindow *imd = data;

	if (!GTK_WIDGET_HAS_FOCUS(widget)) return FALSE;

	image_focus_paint(imd);
	return FALSE;
}

static gint image_focus_in_cb(GtkWidget *widget, GdkEventFocus *event, gpointer data)
{
	ImageWindow *imd = data;

	GTK_WIDGET_SET_FLAGS(imd->widget, GTK_HAS_FOCUS);

	gtk_widget_queue_draw(imd->widget);
	return FALSE;
}

static gint image_focus_out_cb(GtkWidget *widget, GdkEventFocus *event, gpointer data)
{
	ImageWindow *imd = data;

	GTK_WIDGET_UNSET_FLAGS(imd->widget, GTK_HAS_FOCUS);
	gtk_widget_queue_clear(imd->widget);
	gtk_frame_set_shadow_type(GTK_FRAME(imd->widget), GTK_SHADOW_IN);
	return FALSE;
}

/*
 *-------------------------------------------------------------------
 * mouse stuff
 *-------------------------------------------------------------------
 */

static void image_mouse_motion_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	ImageWindow *imd = data;

	if (!imd->in_drag || !gdk_pointer_is_grabbed()) return;

	if (imd->drag_moved < IMAGE_DRAG_SCROLL_THRESHHOLD)
		{
		imd->drag_moved++;
		}
	else
		{
		widget_set_cursor (imd->image, GDK_FLEUR);
		}

	/* do the scroll */
	image_scroll_real(imd, imd->drag_last_x - bevent->x, imd->drag_last_y - bevent->y);

	imd->drag_last_x = bevent->x;
	imd->drag_last_y = bevent->y;
}

static void image_mouse_press_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	ImageWindow *imd = data;

	switch (bevent->button)
		{
		case 1:
			imd->in_drag = TRUE;
			imd->drag_last_x = bevent->x;
			imd->drag_last_y = bevent->y;
			imd->drag_moved = 0;
			gdk_pointer_grab(imd->image->window, FALSE,
                                GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                                NULL, NULL, bevent->time);
			gtk_grab_add(imd->image);
			break;
		case 2:
			imd->drag_moved = 0;
			break;
		case 3:
		case 4:
		case 5:
			image_button_do(imd, bevent, bevent->button);
			break;
		default:
			break;
		}

	gtk_widget_grab_focus(imd->widget);
}

static void image_mouse_release_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	ImageWindow *imd = data;

	if (gdk_pointer_is_grabbed() && GTK_WIDGET_HAS_GRAB(imd->image))
		{
		gtk_grab_remove(imd->image);
		gdk_pointer_ungrab(bevent->time);
		widget_set_cursor(imd->image, -1);
		}

	if (bevent->button == 1 || bevent->button == 2)
		{
		if (imd->drag_moved < IMAGE_DRAG_SCROLL_THRESHHOLD) image_button_do(imd, bevent, bevent->button);
		}

	imd->in_drag = FALSE;
}

static void image_mouse_drag_cb(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	ImageWindow *imd = data;

	imd->drag_moved = IMAGE_DRAG_SCROLL_THRESHHOLD;
}

/*
 *-------------------------------------------------------------------
 * drag and drop
 *-------------------------------------------------------------------
 */

/*
 *-------------------------------------------------------------------
 * public interface
 *-------------------------------------------------------------------
 */

void image_attach_window(ImageWindow *imd, GtkWidget *window,
			 const gchar *title, const gchar *title_right, gint show_zoom)
{
	imd->top_window = window;
	g_free(imd->title);
	imd->title = g_strdup(title);
	g_free(imd->title_right);
	imd->title_right = g_strdup(title_right);
	imd->title_show_zoom = show_zoom;

	image_update_title(imd);
}

void image_set_update_func(ImageWindow *imd,
			   void (*update_func)(ImageWindow *imd, gpointer data),
			   gpointer data)
{
	imd->update_func = update_func;
	imd->update_data = data;
}

static void image_button_do(ImageWindow *imd, GdkEventButton *event, gint button)
{
	void (*func)(ImageWindow *, GdkEventButton *, gpointer);
	gpointer data;

	switch (button)
		{
		case 1:
			func = imd->func_btn1;
			data = imd->data_btn1;
			break;
		case 2:
			func = imd->func_btn2;
			data = imd->data_btn2;
			break;
		case 3:
			func = imd->func_btn3;
			data = imd->data_btn3;
			break;
		case 4:
			func = imd->func_btn4;
			data = imd->data_btn4;
			break;
		case 5:
			func = imd->func_btn5;
			data = imd->data_btn5;
			break;
		default:
			func = NULL;
			data = NULL;
			break;
		}

	if (func) func(imd, event, data);
}

void image_set_button_func(ImageWindow *imd, gint button,
			   void (*func)(ImageWindow *, GdkEventButton *, gpointer),
			   gpointer data)
{
	switch (button)
		{
		case 1:
			imd->func_btn1 = func;
			imd->data_btn1 = data;
			break;
		case 2:
			imd->func_btn2 = func;
			imd->data_btn2 = data;
			break;
		case 3:
			imd->func_btn3 = func;
			imd->data_btn3 = data;
			break;
		case 4:
			imd->func_btn4 = func;
			imd->data_btn4 = data;
			break;
		case 5:
			imd->func_btn5 = func;
			imd->data_btn5 = data;
			break;
		}
}

/* path, name */

const gchar *image_get_path(ImageWindow *imd)
{
	return imd->image_path;
}

const gchar *image_get_name(ImageWindow *imd)
{
	return imd->image_name;
}

/* merely changes path string, does not change the image! */
void image_set_path(ImageWindow *imd, const gchar *newpath)
{
	g_free(imd->image_path);
	imd->image_path = g_strdup(newpath);
	imd->image_name = filename_from_path(imd->image_path);

	image_update_title(imd);
}

/* load a new image */

void image_change_path(ImageWindow *imd, const gchar *path, gfloat zoom)
{
	if (imd->image_path == path ||
	    (path && imd->image_path && !strcmp(path, imd->image_path)) ) return;

	image_change_real(imd, path, NULL, NULL, zoom);
}

void image_change_pixbuf(ImageWindow *imd, GdkPixbuf *pixbuf, gfloat zoom)
{
	image_set_pixbuf(imd, pixbuf, zoom, TRUE);
}

void image_change_from_collection(ImageWindow *imd, CollectionData *cd, CollectInfo *info, gfloat zoom)
{
	if (!cd || !info || !g_list_find(cd->list, info)) return;

	image_change_real(imd, info->path, cd, info, zoom);
}

CollectionData *image_get_collection(ImageWindow *imd, CollectInfo **info)
{
	if (collection_to_number(imd->collection) >= 0)
		{
		if (g_list_find(imd->collection->list, imd->collection_info) != NULL)
			{
			if (info) *info = imd->collection_info;
			}
		else
			{
			if (info) *info = NULL;
			}
		return imd->collection;
		}

	if (info) *info = NULL;
	return NULL;
}

static void image_loader_sync_data(ImageLoader *il, gpointer data)
{
	/* change data for the callbacks directly */

	il->data_area_ready = data;
	il->data_error = data;
	il->data_done = data;
	il->data_percent = data;
}

/* this is more like a move function
 * it moves most data from source to imd, source does keep a ref on the pixbuf
 */
void image_change_from_image(ImageWindow *imd, ImageWindow *source)
{
	imd->unknown = source->unknown;

	image_set_pixbuf(imd, source->pixbuf, image_zoom_get(source), TRUE);

	imd->collection = source->collection;
	imd->collection_info = source->collection_info;
	imd->size = source->size;

	image_set_path(imd, image_get_path(source));

	image_loader_free(imd->il);
	imd->il = NULL;

	if (imd->pixbuf && source->il)
		{
		imd->il = source->il;
		source->il = NULL;

		image_loader_sync_data(imd->il, imd);

		imd->delay_alter_type = source->delay_alter_type;
		source->delay_alter_type = ALTER_NONE;
		}

	imd->read_ahead_il = source->read_ahead_il;
	source->read_ahead_il = NULL;
	if (imd->read_ahead_il) image_loader_sync_data(imd->read_ahead_il, imd);

	imd->read_ahead_pixbuf = source->read_ahead_pixbuf;
	source->read_ahead_pixbuf = NULL;

	imd->read_ahead_path = source->read_ahead_path;
	source->read_ahead_path = NULL;
}

/* manipulation */

void image_reload(ImageWindow *imd)
{
	image_change_complete(imd, imd->zoom, FALSE);
}

void image_scroll(ImageWindow *imd, gint x, gint y)
{
	image_scroll_real(imd, x, y);
}

void image_alter(ImageWindow *imd, AlterType type)
{
	if (imd->il)
		{
		/* still loading, wait till done */
		imd->delay_alter_type = type;
		return;
		}

	image_alter_real(imd, type);
}

/* zoom */

void image_zoom_adjust(ImageWindow *imd, gfloat increment)
{
	gfloat zoom = imd->zoom;

	if (increment == 0.0) return; /* avoid possible div by zero, a no-op anyway... */

	if (zoom == 0.0)
		{
		gfloat new_zoom;
		if (imd->scale < 1.0)
			{
			new_zoom = 0.0 - 1.0 / imd->scale;
			}
		else
			{
			new_zoom = imd->scale;
			}
		/* simply use the increment as the step size and lock (round) the new zoom to it */
		zoom = ceil(new_zoom / increment) * increment;
		}
	else if (increment < 0.0)
		{
		if (zoom >= 1.0 && zoom + increment < 1.0)
			{
			zoom = zoom + increment - 2.0;
			}
		else
			{
			zoom = zoom + increment;
			}
		}
	else
		{
		if (zoom <= -1.0 && zoom + increment > -1.0)
			{
			zoom = zoom + increment + 2.0;
			}
		else
			{
			zoom = zoom + increment;
			}
		}

	image_zoom_sync(imd, zoom, FALSE, FALSE, FALSE);
}

void image_zoom_set(ImageWindow *imd, gfloat zoom)
{
	image_zoom_sync(imd, zoom, FALSE, FALSE, FALSE);
}

gfloat image_zoom_get(ImageWindow *imd)
{
	return imd->zoom;
}

gfloat image_zoom_get_real(ImageWindow *imd)
{
	return imd->scale;
}

gchar *image_zoom_get_as_text(ImageWindow *imd)
{
	gfloat l = 1.0;
	gfloat r = 1.0;
	gint pl = 0;
	gint pr = 0;
	gchar *approx = " ";

	if (imd->zoom > 0.0)
		{
		l = imd->zoom;
		}
	else if (imd->zoom < 0.0)
		{
		r = 0.0 - imd->zoom;
		}
	else if (imd->zoom == 0.0 && imd->scale != 0.0)
		{
		if (imd->scale >= 1.0)
			{
			l = imd->scale;
			}
		else
			{
			r = 1.0 / imd->scale;
			}
		approx = " ~";
		}

	if (rint(l) != l) pl = 1;
	if (rint(r) != r) pr = 1;

	return g_strdup_printf("%.*f :%s%.*f", pl, l, approx, pr, r);
}

gfloat image_zoom_get_default(ImageWindow *imd, gint mode)
{
	gfloat zoom;

	if (mode == ZOOM_RESET_ORIGINAL)
		{
		zoom = 1.0;
		}
	else if (mode == ZOOM_RESET_FIT_WINDOW)
		{
		zoom = 0.0;
		}
	else
		{
		if (imd)
			{
			zoom = image_zoom_get(imd);
			}
		else
			{
			zoom = 1.0;
			}
		}

	return zoom;
}

/* read ahead */

void image_prebuffer_set(ImageWindow *imd, const gchar *path)
{
	if (path)
		{
		image_read_ahead_set(imd, path);
		}
	else
		{
		image_read_ahead_cancel(imd);
		}
}

/* allow top window to be resized ? */

void image_top_window_set_sync(ImageWindow *imd, gint allow_sync)
{
	imd->top_window_sync = allow_sync;
}

/* wallpaper util */

void image_to_root_window(ImageWindow *imd, gint scaled)
{
	GdkVisual *gdkvisual;
	GdkWindow *rootwindow;
	GdkPixmap *pixmap;
	GdkPixbuf *pb;

	if (!imd || !imd->pixbuf) return;


	rootwindow = (GdkWindow *) &gdk_root_parent;	/* hmm, don't know, correct? */
	gdkvisual = gdk_window_get_visual(rootwindow);
	if (gdkvisual != gdk_visual_get_system()) return;

	if (scaled)
		{
		pb = gdk_pixbuf_scale_simple(imd->pixbuf, gdk_screen_width(), gdk_screen_height(), (GdkInterpType)zoom_quality);
		}
	else
		{
		pb = gdk_pixbuf_scale_simple(imd->pixbuf, imd->width, imd->height, (GdkInterpType)zoom_quality);
		}

	gdk_pixbuf_render_pixmap_and_mask (pb, &pixmap, NULL, 128);
	gdk_window_set_back_pixmap(rootwindow, pixmap, FALSE);
	gdk_window_clear(rootwindow);
	gdk_pixbuf_unref(pb);
	gdk_pixmap_unref(pixmap);

	gdk_flush();
}


/*
 *-------------------------------------------------------------------
 * init / destroy
 *-------------------------------------------------------------------
 */

static void image_free(ImageWindow *imd)
{
	image_read_ahead_cancel(imd);

	g_free(imd->image_path);
	g_free(imd->title);
	g_free(imd->title_right);

	image_reset(imd);
	image_tile_sync_count(imd, 0);
	if (imd->pixbuf) gdk_pixbuf_unref(imd->pixbuf);

	g_free(imd);
}

static void image_destroy_cb(GtkObject *widget, gpointer data)
{
	ImageWindow *imd = data;
	image_free(imd);
}

ImageWindow *image_new(gint frame)
{
	ImageWindow *imd;

	imd = g_new0(ImageWindow, 1);
	imd->zoom = 1.0;
	imd->scale = 1.0;

	imd->draw_idle_id = -1;

	imd->tile_width = IMAGE_TILE_SIZE;
	imd->tile_height = IMAGE_TILE_SIZE;

	imd->top_window = NULL;
	imd->title = NULL;
	imd->title_right = NULL;
	imd->title_show_zoom = FALSE;

	imd->unknown = TRUE;

	imd->pixbuf = NULL;

	imd->has_frame = frame;
	imd->top_window_sync = FALSE;

	imd->tile_cache = NULL;
	imd->tile_cache_size = 0;

	imd->delay_alter_type = ALTER_NONE;

	imd->read_ahead_il = NULL;
	imd->read_ahead_pixbuf = NULL;
	imd->read_ahead_path = NULL;

	imd->eventbox = gtk_event_box_new();

	gtk_signal_connect_after(GTK_OBJECT(imd->eventbox), "size_allocate",
				 GTK_SIGNAL_FUNC(image_size_cb), imd);

	if (imd->has_frame)
		{
		imd->widget = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(imd->widget), GTK_SHADOW_IN);
		gtk_container_add(GTK_CONTAINER(imd->widget), imd->eventbox);
		gtk_widget_show(imd->eventbox);

		GTK_WIDGET_SET_FLAGS(imd->widget, GTK_CAN_FOCUS);
		gtk_signal_connect(GTK_OBJECT(imd->widget), "focus_in_event",
				   GTK_SIGNAL_FUNC(image_focus_in_cb), imd);
		gtk_signal_connect(GTK_OBJECT(imd->widget), "focus_out_event",
				   GTK_SIGNAL_FUNC(image_focus_out_cb), imd);

		gtk_signal_connect_after(GTK_OBJECT(imd->widget), "draw",
					 GTK_SIGNAL_FUNC(image_focus_expose), imd);
		gtk_signal_connect_after(GTK_OBJECT(imd->widget), "expose_event",
					 GTK_SIGNAL_FUNC(image_focus_expose), imd);
		}
	else
		{
		imd->widget = imd->eventbox;
		}

	imd->image = gtk_drawing_area_new();

	gtk_signal_connect(GTK_OBJECT(imd->image), "motion_notify_event",
			   GTK_SIGNAL_FUNC(image_mouse_motion_cb), imd);
	gtk_signal_connect(GTK_OBJECT(imd->image), "button_press_event",
			   GTK_SIGNAL_FUNC(image_mouse_press_cb), imd);
	gtk_signal_connect(GTK_OBJECT(imd->image), "button_release_event",
			   GTK_SIGNAL_FUNC(image_mouse_release_cb), imd);
	gtk_widget_set_events(imd->image, GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK);

	gtk_signal_connect(GTK_OBJECT(imd->image), "expose_event",
			   GTK_SIGNAL_FUNC(image_expose_cb), imd);
	gtk_signal_connect(GTK_OBJECT(imd->image), "drag_begin",
			   GTK_SIGNAL_FUNC(image_mouse_drag_cb), imd);

	gtk_container_add(GTK_CONTAINER(imd->eventbox), imd->image);
	gtk_widget_show(imd->image);

	gtk_signal_connect(GTK_OBJECT(imd->widget), "destroy",
			   GTK_SIGNAL_FUNC(image_destroy_cb), imd);

	return imd;
}

