/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "ui2_includes.h"
#include "ui2_typedefs.h"
#include "ui2_main.h"

#include "ui2_button.h"
#include "ui2_default.h"
#include "ui2_display.h"
#include "ui2_init.h"
#include "ui2_parse.h"
#include "ui2_skin.h"
#include "ui2_util.h"
#include "ui2_widget.h"
#include "ui_fileops.h"

#include "ui2_logo.xpm"

/*
 *-------------
 * yes, these are globals
 *-------------
 */

gint slik_smart_placement = TRUE;
gint slik_remember_position = FALSE;

gint slik_double_size = FALSE;

gint slik_colorshift_r = 0;
gint slik_colorshift_g = 0;
gint slik_colorshift_b = 0;
gint slik_colorshift_a = 64;
gint slik_colorshift_on = FALSE;

gint slik_transparency_force = FALSE;
gint slik_transparency_force_a = 192;

gint debug_mode = FALSE;
gint debug_skin = FALSE;

/* list of all windows created with ui_new() */
static GList *slik_ui_list = NULL;

/*
 *-------------
 * misc
 *-------------
 */

void window_set_icon(GtkWidget *window, const char **icon, const gchar *file)
{
	GdkPixbuf *pb;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	if (!icon && !file) return;

	if (icon)
		{
		pb = gdk_pixbuf_new_from_xpm_data(icon);
		}
	else
		{
		pb = gdk_pixbuf_new_from_file(file);
		}

	if (!pb) return;

	gdk_pixbuf_render_pixmap_and_mask(pb, &pixmap, &mask, 128);
	gdk_pixbuf_unref(pb);

	if (!GTK_WIDGET_REALIZED(window)) gtk_widget_realize(window);
	gdk_window_set_icon(window->window, NULL, pixmap, mask);
	/* apparently, gdk_window_set_icon does not ref the pixmap and mask, so don't unref it (leak?) */
}

void ui_set_icon(UIData *ui, const char **icon, const gchar *file)
{
	if (!ui || !ui->window) return;

	window_set_icon(ui->window, icon, file);
}

/*
 *-------------
 * these are the reserved widgets for window resizing and skin mode toggles.
 *-------------
 */

static void ui_skin_toggle_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	UIData *ui = data;
	const gchar *mode_key;

	/* luckily, the editor disables this button, no need to handle ui->edit */

	mode_key = ui_widget_get_data_by_widget(ui, button);

	if (debug_mode) printf("Setting skin mode key = \"%s\"\n", mode_key);

	ui_skin_mode_set(ui, mode_key);
}

static void ui_skin_expand_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	UIData *ui = data;
	const gchar *text;
	gint w, h;

	text = ui_widget_get_data_by_widget(ui, button);

	/* we must support the editor */
	if (!text && ui->edit && ui_widget_get_by_widget(ui, button) == NULL)
		{
		text = ui_widget_get_data_by_widget(ui->edit, button);
		ui = ui->edit;
		}

	if (debug_mode) printf("Skin expand data = \"%s\"\n", text);

	if (text && sscanf(text, "%i %i", &w, &h) == 2)
		{
		SkinData *skin = ui->skin;
		gint state;

		if (skin->width == skin->width_def && skin->height == skin->height_def)
			{
			/* do the expand (can be negative too) */
			skin_resize(ui, skin->width + w, skin->height + h);
			state = TRUE;
			}
		else
			{
			/* assume expanded, snap back to default */
			skin_resize(ui, skin->width_def, skin->height_def);
			state = FALSE;
			}
		button_state_set("skin_expand", ui, state);
		}
}

static void ui_add_reserved(UIData *ui)
{
	if (!ui_registered_key_exists(ui, "skin_toggle", button_type_id()))
		{
		button_register_key("skin_toggle", ui,
				     NULL, NULL,
				     ui_skin_toggle_click_cb, ui,
				     NULL, NULL,
				     NULL, NULL);
		}
	if (!ui_registered_key_exists(ui, "skin_size", button_type_id()))
		{
		/* this is really a dummy register, but it makes the key show in the editor */
		button_register_key("skin_size", ui,
				     NULL, NULL,
				     NULL, NULL,
				     NULL, NULL,
				     NULL, NULL);
		}
	if (!ui_registered_key_exists(ui, "skin_expand", button_type_id()))
		{
		button_register_key("skin_expand", ui,
				     NULL, NULL,
				     ui_skin_expand_click_cb, ui,
				     NULL, NULL,
				     NULL, NULL);
		}
}

/*
 *-------------
 * ui
 *-------------
 */

void ui_free(UIData *ui)
{
	slik_ui_list = g_list_remove(slik_ui_list, ui);

	if (ui->root_win_idle != -1) gtk_idle_remove(ui->root_win_idle);
	ui_register_free_all(ui);
	skin_free(ui->skin);
	g_free(ui->skin_path);
	g_free(ui->skin_mode_key);
	g_free(ui->key);
	g_free(ui);
}

UIData *ui_new(const gchar *class, const gchar *subclass, gint decorations, const gchar *title)
{
	UIData *ui;

	/* make sure widgets are registered */
	ui_init_types();

	ui = g_new0(UIData, 1);

	ui->decorations = decorations;
	ui->root_win_idle = -1;
	ui->edit = NULL;
	ui->skin = NULL;
	ui->key = g_strdup(subclass);
	ui->allow_move = TRUE;

	ui->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(ui->window), FALSE, FALSE, TRUE);
	gtk_container_border_width (GTK_CONTAINER(ui->window), 0);
	gtk_window_set_wmclass(GTK_WINDOW(ui->window), subclass, class);
	gtk_window_set_title(GTK_WINDOW(ui->window), title);

	gtk_widget_realize(ui->window);

	ui->display = gtk_drawing_area_new();
	ui_display_events_init(ui);
	gtk_container_add(GTK_CONTAINER(ui->window), ui->display);
	gtk_widget_show(ui->display);

	gtk_widget_realize(ui->display);

	if (!ui->decorations) gdk_window_set_decorations(ui->window->window, 0);

	/* add the default reserved keys */
	ui_add_reserved(ui);

	slik_ui_list = g_list_append(slik_ui_list, ui);

	return ui;
}

void ui_set_mouse_callback(UIData *ui, void (*func)(UIData *ui, gint button, guint32 time, gpointer data), gpointer data)
{
	ui->click_func = func;
	ui->click_data = data;
}

/*
 *-------------
 * app side keys
 *-------------
 */

RegisterData *ui_register_key(UIData *ui, const gchar *key, WidgetType type, gpointer callbacks, guint length)
{
	RegisterData *rd;

	rd = g_new(RegisterData, 1);
	rd->type = type;
	rd->key = g_strdup(key);
	rd->callbacks = callbacks;
	rd->callbacks_l = length;
	rd->private = FALSE;
	rd->private_widget = NULL;

	ui->register_list = g_list_prepend(ui->register_list, rd);

	return rd;
}

RegisterData *ui_register_key_private(UIData *ui, const gchar *key, WidgetType type,
				      gpointer callbacks, guint length, gpointer widget)
{
	RegisterData *rd;

	rd = ui_register_key(ui, key, type, callbacks, length);
	rd->private = TRUE;
	rd->private_widget = widget;
	return rd;
}

static void ui_register_free(RegisterData *rd)
{
	g_free(rd->callbacks);
	g_free(rd->key);
	g_free(rd);
}

void ui_register_free_all(UIData *ui)
{
	while(ui->register_list)
		{
		RegisterData *rd = ui->register_list->data;
		ui->register_list = g_list_remove(ui->register_list, rd);
		ui_register_free(rd);
		}
}

void ui_register_free_private(UIData *ui)
{
	GList *work = ui->register_list;

	while(work)
		{
		RegisterData *rd = work->data;
		work = work->next;

		if (rd->private)
			{
			ui->register_list = g_list_remove(ui->register_list, rd);
			ui_register_free(rd);
			}
		}
}

/* this will remove all keys with private_widget that match the widget,
 * needed only when removing a widget
 */
void ui_unregister_key_private_for_widget(UIData *ui, gpointer widget)
{
	GList *work;

	if (!widget) return;

	work = ui->register_list;
	while (work)
		{
		RegisterData *rd = work->data;
		work = work->next;

		if (rd->private && rd->private_widget == widget)
			{
			ui->register_list = g_list_remove(ui->register_list, rd);
			ui_register_free(rd);
			}
		}
}

static RegisterData *ui_registered_by_key(UIData *ui, const gchar *key, WidgetType type)
{
	GList *work;

	work = ui->register_list;
	while(work)
		{
		RegisterData *rd = work->data;
		if (rd->type == type && strcmp(rd->key, key) == 0) return rd;
		work = work->next;
		}
	return NULL;
}

gpointer ui_get_registered_callbacks(UIData *ui, const gchar *key, WidgetType type)
{
	GList *work;

	work = ui->register_list;
	while(work)
		{
		RegisterData *rd = work->data;
		if (rd->type == type && strcmp(rd->key, key) == 0) return rd->callbacks;
		work = work->next;
		}
	return NULL;
}

gint ui_registered_key_exists(UIData *ui, const gchar *key, WidgetType type)
{
	return (ui_registered_by_key(ui, key, type) != NULL);
}

gint ui_widget_exists(UIData *ui, const gchar *key, WidgetType type)
{
	if (!ui->skin) return FALSE;

	return (skin_widget_get_by_key(ui->skin, key, type) != NULL);
}

void ui_debug_print_registered_keys(UIData *ui)
{
	GList *work;

	printf("UI registered keys (%3d):\n", g_list_length(ui->register_list));
	printf("=========================\n");

	work = ui->register_list;
	while(work)
		{
		RegisterData *rd = work->data;
		work = work->next;

		printf("%-30s %-10s %s\n", rd->key, ui_widget_type_to_text(rd->type), rd->private ? "(widget)" : "");
		}
}

void ui_debug_print_all_keys(UIData *ui)
{
	ui_debug_print_registered_keys(ui);
	skin_debug_print_registered_keys(ui->skin);
}

/*
 *-------------
 * ui_misc
 *-------------
 */

gint ui_geometry_get(UIData *ui, gint *x, gint *y, gint *w, gint *h)
{
	gint rx, ry, rw, rh;

	if (!ui || !ui->window || !ui->window->window) return FALSE;

	gdk_window_get_position(ui->window->window, &rx, &ry);
	gdk_window_get_size(ui->window->window, &rw, &rh);

	if (x) *x = rx;
	if (y) *y = ry;
	if (w) *w = rw;
	if (h) *h = rh;

	ui_state_set(filename_from_path(ui->skin_path), ui->skin_mode_key, rx, ry, rw, rh);

	return TRUE;
}

void ui_title_set(UIData *ui, const gchar *text)
{
	gtk_window_set_title(GTK_WINDOW(ui->window), text);
}

void ui_skin_set(UIData *ui, SkinData *skin)
{
	/* remove old internal widget signals */
	ui_register_free_private(ui);

	skin_free(ui->skin);
	ui->skin = skin;
	skin->ui = ui;

	/* set up new internal widget signals */
	skin_widgets_init(skin, ui);

	/* restore skin states */
	if (slik_remember_position)
		{
		gint x;
		gint y;
		gint w;
		gint h;

		if (ui_state_get(filename_from_path(ui->skin_path), ui->skin_mode_key, &x, &y, &w, &h))
			{
			/* only do this if the skin is actually sizeable in some way */
			if ((skin->sizeable && ui_widget_exists(ui, "skin_size", button_type_id())) ||
			    ui_widget_exists(ui, "skin_expand", button_type_id()) )
				{
				w = CLAMP(w, skin->width_min, skin->width_max);
				h = CLAMP(h, skin->height_min, skin->height_max);
				skin_resize(ui, w, h);
				}
			}
		}

	ui_display_sync_all(ui);
}

gint ui_skin_load(UIData *ui, const gchar *path, const gchar *mode_key)
{
	SkinData *skin;
	gint success = FALSE;
	gchar *cpath;
	gchar *ckey;

	/* copy, since loading a new skin may free the source ? */
	cpath = g_strdup(path);
	ckey = g_strdup(mode_key);

	if (cpath)
		{
		if (!isdir(cpath) && isfile(cpath))
			{
			gchar *dirbuf = remove_level_from_path(path);

			g_free(ckey);
			ckey = g_strdup(filename_from_path(path));

			skin = skin_parse(dirbuf, cpath, FALSE);

			g_free(cpath);
			cpath = dirbuf;
			}
		else
			{
			gchar *datafile;

			datafile = g_strconcat(cpath, "/", ckey ? ckey : "skindata", NULL);
			skin = skin_parse(cpath, datafile, FALSE);
			g_free(datafile);
			}
		}
	else
		{
		skin = skin_load_default(ckey);
		}

	if (skin)
		{
		/* save current state of old skin */
		ui_geometry_get(ui, NULL, NULL, NULL, NULL);

		g_free(ui->skin_path);
		ui->skin_path = g_strdup(cpath);

		g_free(ui->skin_mode_key);
		ui->skin_mode_key = g_strdup(ckey);

		ui_skin_set(ui, skin);

		success = TRUE;
		}

	g_free(cpath);
	g_free(ckey);

	return success;
}

gint ui_skin_mode_set(UIData *ui, const gchar *mode_key)
{
	gint success;
	gint ox, oy, ow, oh;

	if (ui->window)
		{
		gdk_window_get_position(ui->window->window, &ox, &oy);
		}
	else
		{
		ox = oy = 0;
		}
	ow = ui->skin->width;
	oh = ui->skin->height;

	success = ui_skin_load(ui, ui->skin_path, mode_key);

	if (slik_smart_placement && success && ui->window)
		{
		gint x, y;
		gint move = FALSE;

		x = y = 0;

		if (oy + (oh / 2) > gdk_screen_height() / 2)
			{
			move = TRUE;
			if (oh > ui->skin->height)
				{
				x = ox;
				y = oy + oh - ui->skin->height;
				if (y > gdk_screen_height() - ui->skin->height)
					{
					y = gdk_screen_height() - ui->skin->height;
					}
				}
			else
				{
				x = ox;
				y = oy + oh - ui->skin->height;
				}
			}
		else if (oy < 0)
			{
			move = TRUE;
			x = ox;
			y = 0;
			}
		if (move) gdk_window_move(ui->window->window, x, y);
		}

	return success;
}

/*
 *-------------
 * ui_misc text based utils
 *-------------
 */

UIData *ui_find_by_key(const gchar *key)
{
	GList *work;

	if (!key) return NULL;

	work = slik_ui_list;
	while(work)
		{
		UIData *ui = work->data;

		if (ui->key && strcmp(ui->key, key) == 0) return ui;

		work = work->next;
		}
	return NULL;
}

/*
 *-------------
 * SLIK credits / about / logo
 *-------------
 */

const char **ui_slik_logo(void)
{
	return (const char **)slik_xpm;
}

static GtkWidget *slik_about = NULL;

static gint ui_slik_about_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	gtk_widget_destroy(slik_about);
	slik_about = NULL;
	return TRUE;
}

static void slik_about_toggle_cb(GtkWidget *button, gpointer data)
{
	gint *val = data;
	*val = GTK_TOGGLE_BUTTON(button)->active;

	if (val == &debug_mode)
		{
		printf("Debug set: %d\n", debug_mode);
		}
	else if (val == &debug_skin)
		{
		printf("Skin coordinates set: %d\n", debug_skin);
		}
}

static void ui_slik_about(void)
{
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *pixmap;

	GtkStyle *style;
	GdkPixmap *pmap;
	GdkBitmap *mask;

	gchar *buf;

	if (slik_about)
		{
		gdk_window_raise(slik_about->window);
		return;
		}

	slik_about = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(slik_about), "delete_event", GTK_SIGNAL_FUNC(ui_slik_about_delete_cb), NULL);
	gtk_window_set_wmclass(GTK_WINDOW(slik_about), "about", "SLIK");
	gtk_window_set_title(GTK_WINDOW(slik_about), _("About - SLIK"));
	gtk_container_border_width(GTK_CONTAINER(slik_about), 5);

	frame = gtk_frame_new(NULL);
	gtk_container_border_width(GTK_CONTAINER(frame), 5);
	gtk_container_add(GTK_CONTAINER(slik_about), frame);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);
	gtk_widget_realize(slik_about);

	style = gtk_widget_get_style(slik_about);
	pmap = gdk_pixmap_create_from_xpm_d(slik_about->window, &mask,
		&style->bg[GTK_STATE_NORMAL], (gchar **)ui_slik_logo());
	pixmap = gtk_pixmap_new(pmap, mask);
	gdk_pixmap_unref(pmap);
	if (mask) gdk_bitmap_unref(mask);

	gtk_box_pack_start(GTK_BOX(vbox), pixmap, FALSE, FALSE, 0);
	gtk_widget_show(pixmap);

	buf = g_strdup_printf(_("SLIK\nSimpLIstic sKin interface\nversion %s\nCopyright (c) 2001 by John Ellis\nhttp://gqmpeg.sourceforge.net\ngqview@email.com\n\nReleased under the GNU Public License"), SLIK_VERSION);
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_end(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	button = gtk_toggle_button_new();
	gtk_widget_set_usize(button, 8, 8);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), debug_mode);
	gtk_signal_connect(GTK_OBJECT(button), "toggled", slik_about_toggle_cb, &debug_mode);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 4);
	gtk_widget_show(button);

	button = gtk_toggle_button_new();
	gtk_widget_set_usize(button, 8, 8);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), debug_skin);
	gtk_signal_connect(GTK_OBJECT(button), "toggled", slik_about_toggle_cb, &debug_skin);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 4);
	gtk_widget_show(button);

	window_set_icon(slik_about, ui_slik_logo(), NULL);
	gtk_widget_show(slik_about);
}

GtkWidget *ui_slik_credit(void)
{
	GtkWidget *frame;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *pixmap;

	GdkPixbuf *pb;
	GdkPixmap *pmap;
	GdkBitmap *mask;

	gchar *buf;

	frame = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(frame), GTK_RELIEF_NONE);
	gtk_container_border_width(GTK_CONTAINER(frame), 10);
	gtk_signal_connect(GTK_OBJECT(frame), "clicked", GTK_SIGNAL_FUNC(ui_slik_about), NULL);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(frame), hbox);
	gtk_widget_show(hbox);

	pb = gdk_pixbuf_new_from_xpm_data(ui_slik_logo());
	gdk_pixbuf_render_pixmap_and_mask(pb, &pmap, &mask, 128);
	gdk_pixbuf_unref(pb);

	pixmap = gtk_pixmap_new(pmap, mask);
	gdk_pixmap_unref(pmap);
	if (mask) gdk_bitmap_unref(mask);

	gtk_box_pack_start(GTK_BOX(hbox), pixmap, FALSE, FALSE, 0);
	gtk_widget_show(pixmap);

	buf = g_strdup_printf("utilizes SLIK %s\nSimpLIstic sKin interface", SLIK_VERSION);
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);

	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	return frame;
}



