/*
 * GQmpeg
 * (C)1998, 1999 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License.
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at you own risk!
 */

#include "gqmpeg.h"

static gint display_frozen = FALSE;

static void calc_times(gint *h, gint *m, gint *s, gint t, NumberData *hour);

/*
 *-----------------------------------------------------------------------------
 * utility functions (private)
 *-----------------------------------------------------------------------------
 */

static void calc_times(gint *h, gint *m, gint *s, gint t, NumberData *hour)
{
	*h = t / 3600;
	*m = (t - (*h * 3600)) / 60;

	if (!hour && *h > 0)
		{
		*s = *m;
		*m = *h;
		}
	else
		{
		*s = t - ((*h *3600) + (*m * 60));
		}
}

/*
 *-----------------------------------------------------------------------------
 * display widget update functions (public)
 *-----------------------------------------------------------------------------
 */

void display_freeze()
{
	display_frozen = TRUE;
}

void display_thaw()
{
	display_frozen = FALSE;
}

void display_draw_song_count(gint n, gint redraw)
{
	gint h, m, s;

	if (!display_area || display_frozen) return;

	draw_number (skin->song_total, n);

	calc_times(&h, &m, &s, playlist_get_length(), skin->hour_total);
	draw_number(skin->hour_total, h);
	draw_number(skin->min_total, m);
	draw_number(skin->sec_total, s);

	playlist_window_length_update();

	if (redraw) display_flush();
}

void display_draw_song_number(gint n, gint redraw)
{
	if (display_frozen) return;

	draw_number (skin->song_current, n + 1);	

	if (redraw) display_flush();
}

void display_draw_shuffle(gint redraw)
{
	draw_item (skin->item_shuffle, shuffle_mode);
	change_button_light(skin->shuffle, shuffle_mode);

	if (redraw) display_flush();
}

void display_draw_repeat(gint redraw)
{
	draw_item (skin->item_repeat, repeat_mode);
	change_button_light(skin->repeat, repeat_mode);

	if (redraw) display_flush();
}

void display_draw_position(gint redraw)
{
	float pos;

	if (seconds_remaining == 0 && seconds == 0)
		pos = 0;
	else
		pos = (float)seconds / (seconds_remaining + seconds);
	draw_slider(skin->position, pos, FALSE, TRUE, FALSE, FALSE, FALSE);

	if (redraw) display_flush();
}

void display_draw_volume(gint redraw)
{
	if (skin->volume)
		draw_slider(skin->volume, ((float)get_volume() / 100 ), FALSE, TRUE, FALSE, FALSE, FALSE);
	else if (skin->volume_item)
		{
		gint section = (float)get_volume() / 100 * (skin->volume_item->sections - 1);
		draw_item(skin->volume_item, section);
		}

	if (redraw) display_flush();
}

void display_draw_balance(gint redraw)
{
	if (skin->balance)
		draw_slider(skin->balance, ((float)get_balance() / 100 ), FALSE, TRUE, FALSE, FALSE, FALSE);
	else if (skin->balance_item)
		{
		gint section = (float)get_balance() / 100 * (skin->balance_item->sections - 1);
		draw_item(skin->balance_item, section);
		}

	if (redraw) display_flush();
}

void display_draw_status(gint redraw)
{
	if (status == STATUS_STOP)
		{
		change_button_light(skin->play, FALSE);
		change_button_light(skin->stop, TRUE);
		change_button_light(skin->pause, FALSE);
		draw_item (skin->status, 0);
		}
	if (status == STATUS_PAUSE)
		{
		change_button_light(skin->play, FALSE);
		change_button_light(skin->stop, FALSE);
		change_button_light(skin->pause, TRUE);
		draw_item (skin->status, 1);
		}
	if (status == STATUS_PLAY)
		{
		change_button_light(skin->play, TRUE);
		change_button_light(skin->stop, FALSE);
		change_button_light(skin->pause, FALSE);
		draw_item (skin->status, 2);
		}

	if (redraw) display_flush();
}

void display_draw_frame_count(gint n, gint redraw)
{
	draw_number (skin->frame, n);

	if (redraw) display_flush();
}

void display_draw_cpu(gfloat percentf, gint redraw)
{
	draw_number (skin->cpu, (gint)percentf);

	if (redraw) display_flush();
}

void display_draw_time(gint count, gint remain, gint redraw)
{
	gint n;
	gint h,m,s;

	if (show_remaining_time)
		n = remain;
	else
		n = count;

	calc_times(&h, &m, &s, n, skin->hour);
	draw_number (skin->hour, h);
	draw_number (skin->min, m);
	draw_number (skin->sec, s);


	if (redraw) display_flush();
}

void display_draw_io_information(gint zero, gint redraw)
{
	if (!zero)
		{
		gint h, m, s;

		draw_number (skin->output_bits, output_bits);
		draw_number (skin->output_hz, output_hz / 1000);
		draw_number (skin->input_bits, input_bitrate);
		draw_number (skin->input_hz, input_hz / 1000);

		draw_number (skin->frame_total, frames + frames_remaining);
		calc_times(&h, &m, &s, seconds + seconds_remaining, NULL);
		draw_number (skin->song_min, m);
		draw_number (skin->song_sec, s);

		draw_item (skin->stereo, output_channels);
		draw_item (skin->mpegver, mpeg_version);
		draw_item (skin->layer, mpeg_layer);
		draw_item (skin->mode, mpeg_mode);
		}
	else
		{
		draw_number (skin->output_bits, output_bits);
		draw_number (skin->output_hz, output_hz / 1000);
		draw_number (skin->input_bits, 0);
		draw_number (skin->input_hz, 0);

		draw_number (skin->frame_total, 0);
		draw_number (skin->song_min, 0);
		draw_number (skin->song_sec, 0);

		draw_item (skin->stereo, output_channels);
		draw_item (skin->mpegver, 0);
		draw_item (skin->layer, 0);
		draw_item (skin->mode, 0);
		}

	if (redraw) display_flush();
}

void display_draw_song_text_info(gint n, gint current, gint redraw)
{
	if (current)
		{
		set_font_text (skin->artist, current_song_get_artist());
		set_font_text (skin->album, current_song_get_album());
		set_font_text (skin->genre, current_song_get_genre());
		set_font_text (skin->title, current_song_get_title());
		}
	else
		{
		set_font_text (skin->artist, playlist_get_artist(n));
		set_font_text (skin->album, playlist_get_album(n));
		set_font_text (skin->genre, playlist_get_genre(n));
		set_font_text (skin->title, playlist_get_title(n));
		}

	if (redraw) display_flush();
}

void display_draw_title(gchar *text, gint redraw)
{
	set_font_text(skin->title, text);

	if (redraw) display_flush();
}

/*
 *-----------------------------------------------------------------------------
 * display management functions (public)
 *-----------------------------------------------------------------------------
 */

void display_flush()
{
	redraw_skin();
}

/*
 *-----------------------------------------------------------------------------
 * display utility functions (public)
 *-----------------------------------------------------------------------------
 */

/*
 * this function assumes redraw of current song only,
 * and is mainly for initial song changes
 */
void display_draw_song_info(gint redraw)
{
	if (!skin) return;
	display_draw_song_text_info(-1, TRUE, FALSE);
	display_draw_io_information(TRUE, FALSE);
	display_draw_song_number(current_song_get_number(), FALSE);

	if (redraw) display_flush();
}


/*
 * this function forces a redraw of everything,
 * and is mainly for calling after the skin changes
 */
void display_draw_all(gint redraw)
{
	gint i;

	draw_button(skin->prev, FALSE, FALSE, TRUE);
	draw_button(skin->pause, FALSE, FALSE, TRUE);
	draw_button(skin->play, FALSE, FALSE, TRUE);
	draw_button(skin->stop, FALSE, FALSE, TRUE);
	draw_button(skin->next, FALSE, FALSE, TRUE);
	draw_button(skin->ff, FALSE, FALSE, TRUE);
	draw_button(skin->rw, FALSE, FALSE, TRUE);
	draw_button(skin->shuffle, FALSE, FALSE, TRUE);
	draw_button(skin->repeat, FALSE, FALSE, TRUE);
	draw_button(skin->playlist, FALSE, FALSE, TRUE);
	draw_button(skin->config, FALSE, FALSE, TRUE);
	draw_button(skin->exit, FALSE, FALSE, TRUE);
	draw_button(skin->iconify, FALSE, FALSE, TRUE);
	draw_button(skin->mixer, FALSE, FALSE, TRUE);
	draw_button(skin->skin_mode, FALSE, FALSE, TRUE);

	display_draw_shuffle(FALSE);
	display_draw_repeat(FALSE);
	display_draw_song_number(current_song_get_number(), FALSE);
	display_draw_song_count(playlist_get_count(), FALSE);
	display_draw_io_information(FALSE, FALSE);
	display_draw_cpu(0, FALSE);

	display_draw_status(FALSE);
	if (status == STATUS_PLAY || status == STATUS_PAUSE)
		{
		display_draw_time(seconds, seconds_remaining, FALSE);
		display_draw_frame_count(frames, FALSE);
		draw_slider(skin->position, 0, FALSE, FALSE, FALSE, FALSE, TRUE);
		}
	else
		{
		display_draw_time(0, 0, FALSE);
		display_draw_frame_count(0, FALSE);
		draw_slider(skin->position, 0, FALSE, TRUE, FALSE, FALSE, TRUE);
		}

	if (skin->volume)
		draw_slider(skin->volume, ((float)get_volume() / 100 ), FALSE, TRUE, FALSE, FALSE, TRUE);
	else if (skin->volume_item)
		{
		gint section = (float)get_volume() / 100 * (skin->volume_item->sections - 1);
		draw_item(skin->volume_item, section);
		}
	draw_button(skin->volume_up, FALSE, FALSE, TRUE);
	draw_button(skin->volume_dn, FALSE, FALSE, TRUE);

	if (skin->balance)
		draw_slider(skin->balance, ((float)get_balance() / 100 ), FALSE, TRUE, FALSE, FALSE, TRUE);
	else if (skin->balance_item)
		{
		gint section = (float)get_balance() / 100 * (skin->balance_item->sections - 1);
		draw_item(skin->balance_item, section);
		}
	draw_button(skin->balance_left, FALSE, FALSE, TRUE);
	draw_button(skin->balance_right, FALSE, FALSE, TRUE);

	for (i=0; i<10; i++)
		{
		draw_button(skin->preset[i], FALSE, FALSE, TRUE);
		}

	display_draw_song_text_info(-1, TRUE, FALSE);

	if (redraw) display_flush();
}

