/*
 * GQmpeg
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqmpeg.h"
#include "io_ogg123.h"

#include "playlist.h"
#include "songinfo.h"
#include "ui_clist_edit.h"
#include "ui_fileops.h"


typedef struct _SongInfoDialog SongInfoDialog;
struct _SongInfoDialog
{
	GtkWidget *vbox;
	GtkWidget *clist;
	GtkWidget *save_button;
	GtkWidget *remove_button;
	gchar *path;
	GList *comments;
};

static gchar *comments_reserved[] = { "TITLE", "ARTIST", "ALBUM", "DATE", "GENRE", "TRACKNUMBER", "DESCRIPTION",
				      "COPYRIGHT", "LICENSE", "ORGANIZATION", "CONTACT", "ISRC",
				      "PERFORMER", "LOCATION", "VERSION", NULL };
static gint comments_reserved_count = 15;

static void close_song_info_window(GtkWidget *widget, gpointer data)
{
	SongInfoDialog *s = data;
	g_free(s->path);
	path_list_free(s->comments);
	g_free(s);
}

static void comments_save_cb(GtkWidget *w, gpointer data)
{
	SongInfoDialog *s = data;

	ogg123_comment_set(s->path, s->comments);
	
	playlist_update_by_path(s->path);

	gtk_widget_set_sensitive(s->save_button, FALSE);
}

static void comment_remove_cb(GtkWidget *w, gpointer data)
{
	SongInfoDialog *s = data;
	gint row;
	GList *work;

	if (GTK_CLIST(s->clist)->selection == NULL) return;
	row = GPOINTER_TO_INT(GTK_CLIST(s->clist)->selection->data);

	if (row < comments_reserved_count) return;

	work = gtk_clist_get_row_data(GTK_CLIST(s->clist), row);
	if (work && work->next)
		{
		s->comments = g_list_remove(s->comments, work->next->data);
		s->comments = g_list_remove(s->comments, work->data);
		}
	gtk_clist_remove(GTK_CLIST(s->clist), row);

	gtk_widget_set_sensitive(s->save_button, TRUE);
}

static void comment_add_cb(GtkWidget *w, gpointer data)
{
	SongInfoDialog *s = data;
	gchar *buf[3];
	GList *work;
	gint row;

	buf[0] = _("Newkey");
	buf[1] = "";
	buf[2] = NULL;

	work = g_list_append(NULL, g_strdup(buf[0]));
	work = g_list_append(work, g_strdup(buf[1]));
	s->comments = g_list_concat(s->comments, work);

	row = gtk_clist_append(GTK_CLIST(s->clist), buf);
	gtk_clist_set_row_data(GTK_CLIST(s->clist), row, work);

	gtk_widget_set_sensitive(s->save_button, TRUE);
}

static gint list_row_rename_cb(ClistEditData *ced, const gchar *oldname, const gchar *newname, gpointer data)
{
	SongInfoDialog *s = data;
	GList *work;

	work = gtk_clist_get_row_data(GTK_CLIST(ced->clist), ced->row);
	if (!work)
		{
		if (ced->column == 0)
			{
			work = g_list_append(work, g_strdup(newname));
			work = g_list_append(work, g_strdup(""));
			}
		else
			{
			if (ced->row < comments_reserved_count)
				{
				work = g_list_append(work, g_strdup(comments_reserved[ced->row]));
				}
			else
				{
				work = g_list_append(work, g_strdup(""));
				}
			work = g_list_append(work, g_strdup(newname));
			}
		s->comments = g_list_concat(s->comments, work);
		gtk_clist_set_row_data(GTK_CLIST(ced->clist), ced->row, work);
		}

	if (ced->column == 1) work = work->next;
	g_free(work->data);
	work->data = g_strdup(newname);

	gtk_widget_set_sensitive(s->save_button, TRUE);

	return TRUE;
}

static void list_select_cb(GtkWidget *widget, gint row, gint column, GdkEvent *event, gpointer data)
{
	SongInfoDialog *s = data;

	if (row == -1 || column == -1) return;

	gtk_widget_set_sensitive(s->remove_button, (row >= comments_reserved_count));
	if (column == 0 && row < comments_reserved_count) return;

	clist_edit_by_row(GTK_CLIST(s->clist), row, column, list_row_rename_cb, s);
}

GtkWidget *ogg123_info_window(const gchar *file)
{
	SongInfoDialog *s;
	OGGInfo *info;
	GtkWidget *hbox;
	GtkWidget *vbox1;
	GtkWidget *frame;
	GtkWidget *scrolled;
	GtkWidget *button;
	GList *temp, *work;
	gchar *titles[] = {_("Key"), _("Description"), NULL};
	gint i;

	s = g_new0(SongInfoDialog, 1);
	info = ogg123_info_get(file);
	s->comments = ogg123_comment_get(file);

	s->path = g_strdup(file);

	s->vbox = gtk_vbox_new(FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(s->vbox), "destroy", (GtkSignalFunc)close_song_info_window, s);
	gtk_container_border_width(GTK_CONTAINER(s->vbox), 5);

	songinfo_add_label(s->vbox, _("Filename"), filename_from_path(file));

	frame = gtk_frame_new(_("Song info"));
	gtk_container_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(s->vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox1), 5);
        gtk_container_add(GTK_CONTAINER(frame),vbox1);
        gtk_widget_show(vbox1);

	if (info)
		{
		const gchar *text;

		songinfo_add_label(vbox1, _("Type"), _("ogg bitstream file"));
		songinfo_add_label(vbox1, _("Vendor"), (info->vendor) ? info->vendor : "");

		songinfo_add_label_bool(vbox1, _("Integrity check passed"), info->pass_integrity);
		songinfo_add_label_number(vbox1, _("Channels"), info->channels, NULL);

		text = _("Bit Rate (lowest)");
		if (info->bitrate_lower < 0)
			{
			songinfo_add_label(vbox1, text, _("none set"));
			}
		else
			{
			songinfo_add_label_number(vbox1, text, info->bitrate_lower / 1000, _("Kbits"));
			}
		text = _("Bit Rate (highest)");
		if (info->bitrate_upper < 0)
			{
			songinfo_add_label(vbox1, text, _("none set"));
			}
		else
			{
			songinfo_add_label_number(vbox1, text, info->bitrate_upper / 1000, _("Kbits"));
			}
	
		songinfo_add_label_number(vbox1, _("Bit Rate (encoded)"), info->bitrate_nominal / 1000, _("Kbits"));
		songinfo_add_label_number(vbox1, _("Bit Rate (average)"), info->bitrate_average / 1000, _("Kbits"));
		songinfo_add_label_number(vbox1, _("Length"), info->length, _("seconds"));
		songinfo_add_label_number(vbox1, _("File Size"), info->bytes, _("bytes"));
		}
	else
		{
		songinfo_add_label(vbox1, _("unable to read file info"), NULL);
		}

	frame = gtk_frame_new(_("Comments"));
	gtk_container_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(s->vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER (vbox1), 5);
        gtk_container_add(GTK_CONTAINER(frame), vbox1);
        gtk_widget_show(vbox1);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				       GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(vbox1), scrolled, TRUE, TRUE, 5);
	gtk_widget_show(scrolled);

	s->clist = gtk_clist_new_with_titles(2, titles);
	gtk_widget_set_usize(s->clist, 300, 150);
	gtk_clist_set_column_auto_resize(GTK_CLIST(s->clist), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(s->clist), 1, TRUE);
	gtk_clist_column_titles_passive(GTK_CLIST(s->clist));
	gtk_signal_connect(GTK_OBJECT(s->clist), "select_row",
			   GTK_SIGNAL_FUNC(list_select_cb), s);
	gtk_container_add(GTK_CONTAINER(scrolled), s->clist);
	gtk_widget_show(s->clist);

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 0);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbox), 95, 30);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	button = gtk_button_new_with_label(_("Add"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(comment_add_cb), s);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	gtk_widget_show(button);

	s->remove_button = gtk_button_new_with_label(_("Remove"));
	gtk_signal_connect(GTK_OBJECT(s->remove_button), "clicked",
			   GTK_SIGNAL_FUNC(comment_remove_cb), s);
	gtk_container_add(GTK_CONTAINER(hbox), s->remove_button);
	gtk_widget_set_sensitive(s->remove_button, FALSE);
	gtk_widget_show(s->remove_button);

	s->save_button = gtk_button_new_with_label(_("Save Comments"));
	gtk_signal_connect(GTK_OBJECT(s->save_button), "clicked",
			   GTK_SIGNAL_FUNC(comments_save_cb), s);
	gtk_container_add(GTK_CONTAINER(hbox), s->save_button);
	gtk_widget_set_sensitive(s->save_button, FALSE);
	gtk_widget_show(s->save_button);

	temp = g_list_copy(s->comments);
	i = 0;
	while (comments_reserved[i] != NULL)
		{
		gchar *buf[3];
		gint row;
		GList *found = NULL;

		work = s->comments;
		while (work && work->next && !found)
			{
			const gchar *key = work->data;
			if (strcasecmp(key, comments_reserved[i]) == 0) found = work;
			work = work->next;
			work = work->next;
			}

		buf[0] = comments_reserved[i];
		buf[1] = found ? found->next->data : "";
		buf[2] = NULL;

		row = gtk_clist_append(GTK_CLIST(s->clist), buf);
		gtk_clist_set_row_data(GTK_CLIST(s->clist), row, found);

		if (found)
			{
			temp = g_list_remove(temp, found->next->data);
			temp = g_list_remove(temp, found->data);
			}
		i++;
		}

	work = temp;
	while (work && work->next)
		{
		gchar *buf[3];
		gint row;

		buf[0] = work->data;
		buf[1] = work->next->data;
		buf[2] = NULL;

		row = gtk_clist_append(GTK_CLIST(s->clist), buf);
		gtk_clist_set_row_data(GTK_CLIST(s->clist), row, g_list_find(s->comments, work->data));

		work = work->next;
		work = work->next;
		}
	g_list_free(temp);

	ogg123_info_free(info);

	return s->vbox;
}


