/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif
#include "intl.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <gtk/gtk.h>

#include "ui_pathsel.h"
#include "ui_tabcomp.h"

#include "ui_clist_edit.h"
#include "ui_fileops.h"
#include "ui_menu.h"
#include "ui_utildlg.h"


#define DEST_WIDTH 250
#define DEST_HEIGHT 200

#define RENAME_PRESS_DELAY 333	/* 1/3 second, to allow double clicks */

/* #define PATH_SEL_USE_HEADINGS to enable list headings */

typedef struct _Dest_Data Dest_Data;
struct _Dest_Data
{
	GtkWidget *d_clist;
	GtkWidget *f_clist;
	GtkWidget *entry;
	gchar *filter;
	gchar *path;

	GList *filter_list;
	GList *filter_text_list;
	GtkWidget *filter_combo;

	gint show_hidden;
	GtkWidget *hidden_button;

	guint32 rename_time;
	gint right_click_row;

	void (*select_func)(const gchar *path, gpointer data);
	gpointer select_data;

	GenericDialog *gd;	/* any open confirm dialogs ? */
};

typedef struct _DestDel_Data DestDel_Data;
struct _DestDel_Data
{
	Dest_Data *dd;
	gchar *path;
};


static void dest_delete_dlg_cancel(GenericDialog *gd, gpointer data);


/*
 *-----------------------------------------------------------------------------
 * (private)
 *-----------------------------------------------------------------------------
 */ 

static void dest_free_data(GtkWidget *widget, gpointer data)
{
	Dest_Data *dd = data;

	if (dd->gd)
		{
		GenericDialog *gd = dd->gd;
		dest_delete_dlg_cancel(dd->gd, dd->gd->data);
		generic_dialog_close(gd);
		}

	g_free(dd->filter);
	g_free(dd->path);
	g_free(dd);
}

static gint dest_check_filter(const gchar *filter, const gchar *file)
{
	const gchar *f_ptr = filter;
	const gchar *strt_ptr;
	gint i;
	gint l;

	l = strlen(file);

	if (filter[0] == '*') return TRUE;
	while (f_ptr < filter + strlen(filter))
		{
		strt_ptr = f_ptr;
		i=0;
		while (*f_ptr != ';' && *f_ptr != '\0')
			{
			f_ptr++;
			i++;
			}
		if (*f_ptr != '\0' && f_ptr[1] == ' ') f_ptr++;	/* skip space immediately after separator */
		f_ptr++;
		if (l >= i && strncasecmp(file + l - i, strt_ptr, i) == 0) return TRUE;
		}
	return FALSE;
}

static gint dest_sort_cb(void *a, void *b)
{
	return strcmp((gchar *)a, (gchar *)b);
}

static gint is_hidden(const gchar *name)
{
	if (name[0] != '.') return FALSE;
	if (name[1] == '\0') return FALSE;
	if (name[1] == '.' && name[2] == '\0') return FALSE;
	return TRUE;
}

static void dest_populate(Dest_Data *dd, const gchar *path)
{
	DIR *dp;
	struct dirent *dir;
	struct stat ent_sbuf;
	gchar *buf[2];
	GList *path_list = NULL;
	GList *file_list = NULL;
	GList *list;

	buf[1] = NULL;

	if(!path || (dp = opendir(path)) == NULL)
		{
		/* dir not found */
		return;
		}
	while ((dir = readdir(dp)) != NULL)
		{
		/* skips removed files */
		if (dir->d_ino > 0 && (dd->show_hidden || !is_hidden(dir->d_name)) )
			{
			gchar *name = dir->d_name;
			gchar *filepath = g_strconcat(path, "/", name, NULL);
			if (stat(filepath, &ent_sbuf) >= 0 && S_ISDIR(ent_sbuf.st_mode))
				{
				path_list = g_list_prepend(path_list, g_strdup(name));
				}
			else if (dd->f_clist)
				{
				if (!dd->filter || (dd->filter && dest_check_filter(dd->filter, name)))
					file_list = g_list_prepend(file_list, g_strdup(name));
				}
			g_free(filepath);
			}
		}
	closedir(dp);

	path_list = g_list_sort(path_list, (GCompareFunc) dest_sort_cb);
	file_list = g_list_sort(file_list, (GCompareFunc) dest_sort_cb);

	gtk_clist_freeze(GTK_CLIST(dd->d_clist));
	gtk_clist_clear(GTK_CLIST(dd->d_clist));

	list = path_list;
	while (list)
		{
		gint row;
		gchar *filepath;
		if (strcmp(list->data, ".") == 0)
			{
			filepath = g_strdup(path);
			}
		else if (strcmp(list->data, "..") == 0)
			{
			gchar *p;
			filepath = g_strdup(path);
			p = (gchar *)filename_from_path(filepath);
			if (p - 1 != filepath) p--;
			p[0] = '\0';
			}
		else if (strcmp(path, "/") == 0)
			{
			filepath = g_strconcat("/", list->data, NULL);
			}
		else
			filepath = g_strconcat(path, "/", list->data, NULL);
		
		buf[0] = list->data;
		row = gtk_clist_append(GTK_CLIST(dd->d_clist),buf);
		gtk_clist_set_row_data_full(GTK_CLIST(dd->d_clist), row,
					    filepath, (GtkDestroyNotify) g_free);
		g_free(list->data);
		list = list->next;
		}

	g_list_free(path_list);

	gtk_clist_thaw(GTK_CLIST(dd->d_clist));

	if (dd->f_clist)
		{
		gtk_clist_freeze(GTK_CLIST(dd->f_clist));
		gtk_clist_clear(GTK_CLIST(dd->f_clist));

		list = file_list;
		while (list)
        	        {
			gint row;
			gchar *filepath;
			filepath = g_strconcat(path, "/", list->data, NULL);
		
			buf[0] = list->data;
			row = gtk_clist_append(GTK_CLIST(dd->f_clist),buf);
			gtk_clist_set_row_data_full(GTK_CLIST(dd->f_clist), row,
					    filepath, (GtkDestroyNotify) g_free);
			g_free(list->data);
			list = list->next;
			}

		g_list_free(file_list);

		gtk_clist_thaw(GTK_CLIST(dd->f_clist));
		}

	g_free(dd->path);
	dd->path = g_strdup(path);
}

static void dest_change_dir(Dest_Data *dd, const gchar *path, gint retain_name)
{
	gchar *old_name = NULL;
	gint s = 0;

	if (retain_name)
		{
		const gchar *buf = gtk_entry_get_text(GTK_ENTRY(dd->entry));
		if (!isdir(buf))
			{
			if (path && strcmp(path, "/") == 0)
				{
				old_name = g_strdup(filename_from_path(buf));
				}
			else
				{
				old_name = g_strconcat("/", filename_from_path(buf), NULL);
				s = 1;
				}
			}
		}

	gtk_entry_set_text(GTK_ENTRY(dd->entry), path);

	dest_populate(dd, path);

	/* remember filename */
	if (old_name)
		{
		gtk_entry_append_text(GTK_ENTRY(dd->entry), old_name);
		gtk_entry_select_region(GTK_ENTRY(dd->entry), strlen(path) + s, strlen(path) + strlen(old_name));
		g_free(old_name);
		}
}

/*
 *-----------------------------------------------------------------------------
 * destination widget file management utils
 *-----------------------------------------------------------------------------
 */ 

static gint dest_row_edit_rename_cb(ClistEditData *ced, const gchar *old, const gchar *new, gpointer data)
{
	const gchar *old_path;
	gchar *new_path;
	gchar *buf;

	old_path = gtk_clist_get_row_data(GTK_CLIST(ced->clist), ced->row);
	if (!old_path) return FALSE;

	buf = remove_level_from_path(old_path);
	new_path = g_strconcat(buf, "/", new, NULL);
	g_free(buf);

	if (isname(new_path))
		{
		buf = g_strdup_printf(_("File name %s already exists."), new);
		warning_dialog("Rename failed", buf);
		g_free(buf);
		}
	else if (rename(old_path, new_path) != 0)
		{
		buf = g_strdup_printf(_("Failed to rename %s to %s."), old, new);
		warning_dialog("Rename failed", buf);
		g_free(buf);
		}
	else
		{
		gtk_clist_set_row_data_full(GTK_CLIST(ced->clist), ced->row,
					    new_path, (GtkDestroyNotify) g_free);
		return TRUE;
		}

	g_free(new_path);
	return FALSE;
}

static void dest_delete_dlg_cancel(GenericDialog *gd, gpointer data)
{
	DestDel_Data *dl = data;

	dl->dd->gd = NULL;
	g_free(dl->path);
	g_free(dl);
}

static void dest_delete_dlg_ok_cb(GenericDialog *gd, gpointer data)
{
	DestDel_Data *dl = data;

	if (unlink (dl->path) < 0)
		{
		gchar *text = g_strdup_printf(_("Unable to delete file:\n%s"), dl->path);
		warning_dialog(_("File deletion failed"), text);
		g_free(text);
		}
	else if (dl->dd->path)
		{
		/* refresh list */
		gchar *path = g_strdup(dl->dd->path);
		dest_populate(dl->dd, path);
		g_free(path);
		}

	dest_delete_dlg_cancel(gd, data);
}

static void dest_popup_dir_rename_cb(GtkWidget *widget, gpointer data)
{
	Dest_Data *dd = data;
	clist_edit_by_row(GTK_CLIST(dd->d_clist), dd->right_click_row, 0, dest_row_edit_rename_cb, dd);
}

static void dest_popup_dir_done_cb(GtkWidget *widget, gpointer data)
{
	Dest_Data *dd = data;
	clist_edit_set_highlight(dd->d_clist, dd->right_click_row, FALSE);
}

static void dest_popup_file_rename_cb(GtkWidget *widget, gpointer data)
{
	Dest_Data *dd = data;
	clist_edit_by_row(GTK_CLIST(dd->f_clist), dd->right_click_row, 0, dest_row_edit_rename_cb, dd);
}

static void dest_popup_file_delete_cb(GtkWidget *widget, gpointer data)
{
	Dest_Data *dd = data;
	const gchar *path;
	gchar *text;
	DestDel_Data *dl;

	path = gtk_clist_get_row_data(GTK_CLIST(dd->f_clist), dd->right_click_row);
	if (!path) return;

	dl = g_new(DestDel_Data, 1);
	dl->dd = dd;
	dl->path = g_strdup(path);

	if (dd->gd)
		{
		GenericDialog *gd = dd->gd;
		dest_delete_dlg_cancel(dd->gd, dd->gd->data);
		generic_dialog_close(gd);
		}

	text = g_strdup_printf(_("About to delete the file:\n %s"), path);
	dd->gd = generic_dialog_new(_("Delete file"), text, "SLIK", "dlg_confirm", TRUE,
				    dest_delete_dlg_cancel, dl);
	g_free(text);

	generic_dialog_add(dd->gd, _("Delete"), dest_delete_dlg_ok_cb, TRUE);

	gtk_widget_show(dd->gd->dialog);
}

static void dest_popup_file_done_cb(GtkWidget *widget, gpointer data)
{
	Dest_Data *dd = data;
	clist_edit_set_highlight(dd->f_clist, dd->right_click_row, FALSE);
}

static gint dest_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	Dest_Data *dd = data;
	gint row = -1;
	gint col = -1;

	if (!event) return FALSE;
	gtk_clist_get_selection_info(GTK_CLIST(widget), event->x, event->y, &row, &col);
	if (row == -1 || col == -1) return FALSE;

	if (widget == dd->d_clist)
		{
		/* dir pressed */

		if (event->button == 3 && row > 1)
			{
			/* right click menu */

			GtkWidget *menu;

			dd->right_click_row = row;
			clist_edit_set_highlight(dd->d_clist, row, TRUE);

			menu = popup_menu_short_lived();
			menu_item_add(menu, _("Rename"), dest_popup_dir_rename_cb, dd);

			gtk_signal_connect(GTK_OBJECT(menu), "destroy",
				GTK_SIGNAL_FUNC(dest_popup_dir_done_cb), dd);
			gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, event->time);
			}
		
		return FALSE;
		}
	else
		{
		/* file pressed */

		if (event->button == 3)
			{
			/* right click menu */

			GtkWidget *menu;

			dd->right_click_row = row;
			clist_edit_set_highlight(dd->f_clist, row, TRUE);

			menu = popup_menu_short_lived();
			menu_item_add(menu, _("Rename"), dest_popup_file_rename_cb, dd);
			menu_item_add(menu, _("Delete"), dest_popup_file_delete_cb, dd);

			gtk_signal_connect(GTK_OBJECT(menu), "destroy",
				GTK_SIGNAL_FUNC(dest_popup_file_done_cb), dd);
			gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, event->time);

			return FALSE;
			}

		if (event->button != 1 ||
		    event->type != GDK_BUTTON_PRESS ||
		    event->time - dd->rename_time < RENAME_PRESS_DELAY)
			{
			dd->rename_time = event->time;
			return FALSE;
			}

		if (!GTK_CLIST(dd->f_clist)->selection ||
		    row != GPOINTER_TO_INT(GTK_CLIST(dd->f_clist)->selection->data))
			{
			dd->rename_time = event->time;
			return FALSE;
			}

		clist_edit_by_row(GTK_CLIST(dd->f_clist), row, 0, dest_row_edit_rename_cb, dd);
		}

	return FALSE;
}

static void dest_new_dir_cb(GtkWidget *widget, gpointer data)
{
	Dest_Data *dd = data;
	gchar *path;
	gchar *buf;
	const gchar *tmp;
	gint from_text = FALSE;

	tmp = gtk_entry_get_text(GTK_ENTRY(dd->entry));
	if (!isname(tmp))
		{
		path = g_strdup(tmp);
		from_text = TRUE;
		}
	else
		{
		buf = g_strconcat(dd->path, "/", _("New Directory"), NULL);
		path = unique_filename(buf, NULL, " ", FALSE);
		g_free(buf);
		}

	if (mkdir (path, 0755) < 0)
		{
		/* failed */
		gchar *text;

		text = g_strdup_printf(_("Unable to create directory:\n%s"), filename_from_path(path));
		warning_dialog(_("Error creating directory"), text);
		g_free(text);

		g_free(path);
		}
	else
		{
		gchar *text[2];
		gint row;

		if (from_text) gtk_entry_set_text(GTK_ENTRY(dd->entry), dd->path);

		text[0] = (gchar *)filename_from_path(path);
		text[1] = NULL;
		row = gtk_clist_append(GTK_CLIST(dd->d_clist), text);
		gtk_clist_set_row_data_full(GTK_CLIST(dd->d_clist), row,
					    path, (GtkDestroyNotify) g_free);

		clist_edit_by_row(GTK_CLIST(dd->d_clist), row, 0, dest_row_edit_rename_cb, dd);
		}
}

/*
 *-----------------------------------------------------------------------------
 * destination widget file selection, traversal, view options
 *-----------------------------------------------------------------------------
 */ 

static void dest_select_cb(GtkWidget *clist, gint row, gint column,
			   GdkEventButton *bevent, gpointer data)
{
	Dest_Data *dd = data;
	gchar *path = g_strdup(gtk_clist_get_row_data(GTK_CLIST(clist), row));

	if (clist == dd->d_clist)
		{
		dest_change_dir(dd, path, (dd->f_clist != NULL));
		}
	else
		{
		gtk_entry_set_text(GTK_ENTRY(dd->entry), path);
		if (bevent) dd->rename_time = bevent->time;

		if (bevent && bevent->button == 1 && bevent->type == GDK_2BUTTON_PRESS &&
		    dd->select_func)
			{
			dd->select_func(path, dd->select_data);
			}
		}

	g_free(path);
}

static void dest_home_cb(GtkWidget *widget, gpointer data)
{
	Dest_Data *dd = data;

	dest_change_dir(dd, homedir(), (dd->f_clist != NULL));
}

static void dest_show_hidden_cb(GtkWidget *widget, gpointer data)
{
	Dest_Data *dd = data;
	gchar *buf;

	dd->show_hidden = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dd->hidden_button));

	buf = g_strdup(dd->path);
	dest_populate(dd, buf);
	g_free(buf);
}

static void dest_entry_changed_cb(GtkEditable *editable, gpointer data)
{
	Dest_Data *dd = data;
	const gchar *path;
	gchar *buf;

	path = gtk_entry_get_text(GTK_ENTRY(dd->entry));
	if (!path) return;

	if (strcmp(path, dd->path) == 0) return;

	buf = remove_level_from_path(path);

	if (buf && strcmp(buf, dd->path) != 0)
		{
		gchar *tmp = remove_trailing_slash(path);
		if (isdir(tmp))
			{
			dest_populate(dd, tmp);
			}
		else if (isdir(buf))
			{
			dest_populate(dd, buf);
			}
		g_free(tmp);
		}
	g_free(buf);
}

static void dest_filter_list_sync(Dest_Data *dd)
{
	gchar *old_text;
	GList *fwork;
	GList *cwork;

	if (!dd->filter_list || !dd->filter_combo) return;

	old_text = g_strdup(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dd->filter_combo)->entry)));

	gtk_combo_set_popdown_strings(GTK_COMBO(dd->filter_combo), dd->filter_text_list);

	fwork = dd->filter_list;
	cwork = GTK_LIST(GTK_COMBO(dd->filter_combo)->list)->children;
	while (fwork && cwork)
		{
		gchar *filter = fwork->data;

		gtk_combo_set_item_string(GTK_COMBO(dd->filter_combo), GTK_ITEM(cwork->data), filter);

		if (strcmp(old_text, filter) == 0)
			{
			gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dd->filter_combo)->entry), filter);
			}

		fwork = fwork->next;
		cwork = cwork->next;
		}

	g_free(old_text);
}

static void dest_filter_add(Dest_Data *dd, const gchar *filter, const gchar *description, gint set)
{
	GList *work;
	gchar *buf;
	gint c = 0;

	if (!filter) return;

	work = dd->filter_list;
	while(work)
		{
		gchar *f = work->data;

		if (strcmp(f, filter) == 0)
			{
			if (set) gtk_combo_set_value_in_list(GTK_COMBO(dd->filter_combo), c, FALSE);
			return;
			}
		work = work->next;
		c++;
		}

	dd->filter_list = uig_list_insert_link(dd->filter_list, g_list_last(dd->filter_list), g_strdup(filter));

	if (description)
		{
		buf = g_strdup_printf("%s  ( %s )", description, filter);
		}
	else
		{
		buf = g_strdup_printf("( %s )", filter);
		}
	dd->filter_text_list = uig_list_insert_link(dd->filter_text_list, g_list_last(dd->filter_text_list), buf);

	if (set) gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dd->filter_combo)->entry), filter);
	dest_filter_list_sync(dd);
}

static void dest_filter_clear(Dest_Data *dd)
{
	path_list_free(dd->filter_list);
	dd->filter_list = NULL;

	path_list_free(dd->filter_text_list);
	dd->filter_text_list = NULL;

	dest_filter_add(dd, "*", _("All Files"), TRUE);
}

static void dest_filter_changed_cb(GtkEditable *editable, gpointer data)
{
	Dest_Data *dd = data;
	const gchar *buf;
	gchar *path;

	buf = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dd->filter_combo)->entry));

	g_free(dd->filter);
	dd->filter = NULL;
	if (strlen(buf) > 0) dd->filter = g_strdup(buf);

	path = g_strdup(dd->path);
	dest_populate(dd, path);
	g_free(path);
}

/*
 *-----------------------------------------------------------------------------
 * destination widget setup routines (public)
 *-----------------------------------------------------------------------------
 */ 

GtkWidget *path_selection_new_with_files(GtkWidget *entry, const gchar *path,
					 const gchar *filter, const gchar *filter_desc)
{
	GtkWidget *hbox2;
	Dest_Data *dd;
	GtkWidget *scrolled;
	GtkWidget *button;
	GtkWidget *table;
#ifdef PATH_SEL_USE_HEADINGS
	gchar *d_title[] = { "Directories", NULL };
	gchar *f_title[] = { "Files", NULL };
#endif

	dd = g_new0(Dest_Data, 1);
	dd->show_hidden = FALSE;
	dd->select_func = NULL;
	dd->select_data = NULL;
	dd->gd = NULL;

	table = gtk_table_new(4, 3, FALSE);
	gtk_table_set_col_spacing(GTK_TABLE(table), 1, 5);
	gtk_widget_show(table);

	dd->entry = entry;
	gtk_object_set_data(GTK_OBJECT(dd->entry), "destination_data", dd);

	hbox2 = gtk_hbox_new(FALSE, 5);
	gtk_table_attach(GTK_TABLE(table), hbox2, 0, 1, 0, 1,
			 GTK_EXPAND | GTK_FILL, FALSE, 0, 5);
	gtk_widget_show(hbox2);

	button = gtk_button_new_with_label(_("Home"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(dest_home_cb), dd);
	gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("New Directory"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(dest_new_dir_cb), dd);
	gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	dd->hidden_button = gtk_check_button_new_with_label(_("Show hidden"));
	gtk_signal_connect(GTK_OBJECT(dd->hidden_button), "clicked", GTK_SIGNAL_FUNC(dest_show_hidden_cb), dd);
	gtk_box_pack_end(GTK_BOX(hbox2), dd->hidden_button, FALSE, FALSE, 0);
	gtk_widget_show(dd->hidden_button);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_table_attach(GTK_TABLE(table), scrolled, 0, 1, 1, 2,
			 GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
	gtk_widget_show(scrolled);

#ifdef PATH_SEL_USE_HEADINGS
	dd->d_clist=gtk_clist_new_with_titles(1, d_title);
	gtk_clist_column_titles_passive(GTK_CLIST(dd->d_clist));
#else
	dd->d_clist=gtk_clist_new(1);
#endif
	gtk_clist_set_column_auto_resize(GTK_CLIST(dd->d_clist), 0, TRUE);
	gtk_signal_connect(GTK_OBJECT(dd->d_clist), "button_press_event", (GtkSignalFunc)dest_press_cb, dd);
        gtk_signal_connect(GTK_OBJECT(dd->d_clist), "select_row", (GtkSignalFunc)dest_select_cb, dd);
	gtk_signal_connect(GTK_OBJECT(dd->d_clist), "destroy", (GtkSignalFunc)dest_free_data, dd);
	gtk_widget_set_usize(dd->d_clist, DEST_WIDTH, DEST_HEIGHT);
	gtk_container_add (GTK_CONTAINER (scrolled), dd->d_clist);
        gtk_widget_show (dd->d_clist);

	if (filter)
		{
		GtkWidget *label;

		hbox2 = gtk_hbox_new(FALSE, 5);
		gtk_table_attach(GTK_TABLE(table), hbox2, 2, 3, 0, 1,
				 GTK_EXPAND | GTK_FILL, FALSE, 0, 5);
		gtk_widget_show(hbox2);

		label = gtk_label_new(_("Filter:"));
		gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);
		gtk_widget_show(label);

		dd->filter_combo = gtk_combo_new();
		gtk_box_pack_start(GTK_BOX(hbox2), dd->filter_combo, TRUE, TRUE, 0);
		gtk_widget_show(dd->filter_combo);

		scrolled = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
		gtk_table_attach(GTK_TABLE(table), scrolled, 2, 3, 1, 2,
				 GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
		gtk_widget_show(scrolled);

#ifdef PATH_SEL_USE_HEADINGS
		dd->f_clist=gtk_clist_new_with_titles(1, f_title);
		gtk_clist_column_titles_passive(GTK_CLIST(dd->f_clist));
#else
		dd->f_clist=gtk_clist_new(1);
#endif
		gtk_clist_set_column_auto_resize(GTK_CLIST(dd->f_clist), 0, TRUE);
		gtk_widget_set_usize(dd->f_clist, DEST_WIDTH, DEST_HEIGHT);
		gtk_signal_connect(GTK_OBJECT(dd->f_clist), "button_press_event", (GtkSignalFunc)dest_press_cb, dd);
		gtk_signal_connect(GTK_OBJECT(dd->f_clist), "select_row",(GtkSignalFunc) dest_select_cb, dd);
		gtk_container_add(GTK_CONTAINER (scrolled), dd->f_clist);
	        gtk_widget_show(dd->f_clist);

		dest_filter_clear(dd);
		dest_filter_add(dd, filter, filter_desc, TRUE);

		dd->filter = g_strdup(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dd->filter_combo)->entry)));
		}

	if (path && path[0] == '/' && isdir(path))
		{
		dest_populate(dd, path);
		}
	else
		{
		gchar *buf = remove_level_from_path(path);
		if (buf && buf[0] == '/' && isdir(buf))
			{
			dest_populate(dd, buf);
			}
		else
			{
			dest_populate(dd, (gchar *)homedir());
			if (path) gtk_entry_append_text(GTK_ENTRY(dd->entry), "/");
			if (path) gtk_entry_append_text(GTK_ENTRY(dd->entry), path);
			}
		g_free(buf);
		}

	if (dd->filter_combo) gtk_signal_connect(GTK_OBJECT(GTK_COMBO(dd->filter_combo)->entry), "changed", dest_filter_changed_cb, dd);
	gtk_signal_connect(GTK_OBJECT(dd->entry), "changed", dest_entry_changed_cb, dd);

	return table;
}

GtkWidget *path_selection_new(const gchar *path, GtkWidget *entry)
{
	return path_selection_new_with_files(entry, path, NULL, NULL);
}

void path_selection_sync_to_entry(GtkWidget *entry)
{
	Dest_Data *dd = gtk_object_get_data(GTK_OBJECT(entry), "destination_data");
	const gchar *path;

	if (!dd) return;

	path = gtk_entry_get_text(GTK_ENTRY(entry));
	
	if (isdir(path) && strcmp(path, dd->path) != 0)
		{
		dest_populate(dd, path);
		}
	else
		{
		gchar *buf = remove_level_from_path(path);
		if (isdir(buf) && strcmp(buf, dd->path) != 0)
			{
			dest_populate(dd, buf);
			}
		g_free(buf);
		}
}

void path_selection_add_select_func(GtkWidget *entry,
				    void (*func)(const gchar *, gpointer), gpointer data)
{
	Dest_Data *dd = gtk_object_get_data(GTK_OBJECT(entry), "destination_data");

	if (!dd) return;

	dd->select_func = func;
	dd->select_data = data;
}

void path_selection_add_filter(GtkWidget *entry, const gchar *filter, const gchar *description, gint set)
{
	Dest_Data *dd = gtk_object_get_data(GTK_OBJECT(entry), "destination_data");

	if (!dd) return;
	if (!filter) return;

	dest_filter_add(dd, filter, description, set);
}

void path_selection_clear_filter(GtkWidget *entry)
{
	Dest_Data *dd = gtk_object_get_data(GTK_OBJECT(entry), "destination_data");

	if (!dd) return;

	dest_filter_clear(dd);
}




