/*
 * GQmpeg
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


/*
 * the TODO for this module (using ogg123)
 * > All :)
 */

#include "gqmpeg.h"
#include "io_ogg123.h"

#include "players.h"
#include "ui_fileops.h"
#include "ui_utildlg.h"

#include <fcntl.h>
#include <time.h>

#define OGG123_BINARY "ogg123"

/* ----------------------------------------------------------
   input / output interface to ogg123
   ----------------------------------------------------------*/

gint ogg123_enabled = FALSE;

gchar *ogg123_device_options = NULL;
gchar *ogg123_extra_options = NULL;

/* internal stuff */
static SongData *ogg123_current_sd = NULL;
static gint ogg123_control_read_id = -1;
static gint ogg123_child_pid = -1;
static int ogg123_pipe[2];

/*
 *-----------------------------------------------------------------------------
 * output device utils
 *-----------------------------------------------------------------------------
 */

/* FIXME! this should be seriously fixed to only list available options */

enum {
	OGG123_DEV_OSS = 0,
	OGG123_DEV_ESD = 1,
	OGG123_DEV_IRIX = 2,
	OGG123_DEV_SUN = 3,
	OGG123_DEV_WAV = 4,
	OGG123_DEV_DEF = 5
};

static const gchar *ogg123_output_devices[] =
{
	"oss",
	"esd",
	"irix",
	"sun",
	"wav",
	"default",
	NULL
};
static gint ogg123_output_devices_count = 6;

/* sorry - it got ugly */
#if defined(linux) || defined(__FreeBSD__) || defined (__NetBSD__) || defined(__OpenBSD__)
  gint ogg123_device = OGG123_DEV_OSS;
#elif defined(sun)
  gint ogg123_device = OGG123_DEV_SUN;
#elif defined(SGI)
  gint ogg123_device = OGG123_DEV_IRIX;
#else
  gint ogg123_device = OGG123_DEV_DEF; /* use ogg123's default (from ogg123rc) */
#endif


const gchar *ogg123_get_device(gint device)
{
	if (device >= ogg123_output_devices_count || device < 0) device = 0;

	return ogg123_output_devices[device];
}

/* don't free the data in the list */
GList *ogg123_get_device_list(void)
{
	GList *list = NULL;
	gint i;

	for (i = 0; i < ogg123_output_devices_count; i++)
		{
		list = g_list_append(list, (gchar *)ogg123_get_device(i));
		}

	return list;
}


/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */

static GtkWidget *ogg123_info_window(const gchar *file)
{
	GtkWidget *vbox;
	GtkWidget *vbox1;
	GtkWidget *frame;
	GtkWidget *label;
	gchar *buf;

	vbox = gtk_vbox_new(FALSE, 0);
        gtk_widget_show(vbox);

	buf = g_strconcat(_("Filename: "), filename_from_path(file), NULL);
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	frame = gtk_frame_new(_("Song info"));
	gtk_container_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vbox1), 5);
	gtk_container_add(GTK_CONTAINER(frame), vbox1);
	gtk_widget_show(vbox1);

	label = gtk_label_new(_("Type: ogg bitstream file"));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	return vbox;
}

/*
 *-----------------------------------------------------------------------------
 * child control
 *-----------------------------------------------------------------------------
 */

static void ogg123_child_shutdown(void)
{

	if (ogg123_child_pid != -1)
		{
		pid_t t;

		if (kill(ogg123_child_pid, SIGINT) == -1)
			{
			printf(_("Failed to successfully send signal to pid = %d\n"), ogg123_child_pid);
			}
		
		t = waitpid(ogg123_child_pid, NULL, WNOHANG);
		/* maybe it just took a little bit */
		if (t != ogg123_child_pid && t != -1) waitpid(ogg123_child_pid, NULL, 0);
		}

	if (ogg123_control_read_id != -1) gdk_input_remove(ogg123_control_read_id);
	ogg123_control_read_id = -1;
	close(ogg123_pipe[0]);

	ogg123_child_pid = -1;
	pid = 0;
}

static void ogg123_error(void)
{
	ogg123_child_shutdown();
	module_playback_error(ogg123_current_sd);
}

static void ogg123_input_error(void)
{
	ogg123_error();
}

static void ogg123_input_end(void)
{
	waitpid(ogg123_child_pid, NULL, 0);
	ogg123_child_pid = -1;
}

/*
 *-----------------------------------------------------------------------------
 * input parsing
 *-----------------------------------------------------------------------------
 */

static gint parse_time(const gchar *text)
{
	gint m, s;

	if (sscanf(text, "%d:%d", &m, &s) < 2) return 0;

	return (m * 60 + s);
}

static void parse_bitrate(const gchar *text)
{
	static time_t old_t = 0;
	time_t new_t;
	const gchar *ptr;
	gint force = FALSE;

	ptr = strstr(text, "Bitrate: ");
	if (ptr)
		{
		gint new_rate;

		ptr += 9;
		new_rate = (gint)strtol(ptr, NULL, 10);
		if (input_bitrate != 0)
			{
			/* show a trend, ugly but it makes the rate change more smoothly */
			input_bitrate = ((input_bitrate * 4) + new_rate) / 5;
			}
		else
			{
			input_bitrate = new_rate;
			force = TRUE;
			}
		}

	/* we only update once per second */
	new_t = time(NULL);
	if (new_t > old_t || force)
		{
		module_playback_data_changed();
		old_t = new_t;
		}
}

static gint ogg123_input_parse(const gchar *buffer)
{
	if (debug_mode > 1) printf("ogg123 output:\"%s\"\n", buffer);

	if (strncmp(buffer, "Time: ", 6) == 0)
		{
		const gchar *ptr;

		/* time */
		ptr = buffer + 6;
		if (strlen(ptr) < 8) return FALSE;
		seconds = parse_time(ptr);
		frames = seconds;
		ptr += 8;
		ptr = strchr(ptr, '[');
		if (!ptr) return FALSE;
		ptr++;
		if (strlen(ptr) < 8) return FALSE;
		seconds_remaining = parse_time(ptr);
		frames_remaining = seconds_remaining;
		parse_bitrate(ptr);
		}
	else if (strncmp(buffer, "Bitstream is", 12) == 0)
		{
		sscanf(buffer, "Bitstream is %d channel, %d", &output_channels, &output_hz);
		/* FIXME these are broken, really. */
		input_channels = output_channels;
		input_hz = 44100;
		module_playback_data_changed();
		}
	/* look for end */
	else if (strncmp(buffer, "Done.", 5) == 0)
		{
		ogg123_input_end();
		ogg123_child_shutdown();
		module_playback_end();
		return FALSE;
		}
	else if (strncmp(buffer, "No such device", 14) == 0)
		{
		/* device error */
		gchar *buf;

		printf(_("Error, ogg123 reported:\"%s\")\n"), buffer);
		buf = g_strdup_printf("ogg123: %s", buffer);
		warning_dialog(_("GQmpeg: ogg123 error"), buf);
		g_free(buf);

		ogg123_error();
		return FALSE;
		}

	return TRUE;
}

/*
 *-----------------------------------------------------------------------------
 * input buffer
 *-----------------------------------------------------------------------------
 */

static char ogg123_read_buffer[1024];
static gint ogg123_read_buffer_pos = 0;

static void ogg123_input_read_cb(gpointer data, int fd, GdkInputCondition condition)
{
	gchar buf[1024];
	gint r;
	gint p;
	gint l;
	gchar *ptr;

	r = read(fd, buf, sizeof(buf));

	if (r == -1)
		{
		ogg123_input_error();
		return;
		}
	if (r == 0)
		{
		printf(_("ogg123 disappeared! (unexpected EOF)\n"));
		ogg123_input_end();
		ogg123_error();
		return;
		}

	ptr = ogg123_read_buffer + ogg123_read_buffer_pos;
	l = sizeof(ogg123_read_buffer) - 1;
	p = 0;
	while(p < r)
		{
		gchar c;
		c = buf[p];
		p++;

		if (c == '\b' && ogg123_read_buffer_pos > 0)
			{
			ogg123_read_buffer_pos--;
			}
		else if (c == '\r' || c == '\n')
			{
 			if (ogg123_read_buffer_pos > 0)
				{
				ogg123_read_buffer[ogg123_read_buffer_pos] = '\0';
				if (!ogg123_input_parse(ogg123_read_buffer)) return;
				ogg123_read_buffer_pos = 0;
				}
			}
		else if (ogg123_read_buffer_pos < l)
			{
			ogg123_read_buffer[ogg123_read_buffer_pos] = c;
			ogg123_read_buffer_pos++;
			}
		}

#if 0
	/* this is only needed if the input is not newline/return based */
	if (ogg123_read_buffer_pos > 0)
		{
		ogg123_read_buffer[ogg123_read_buffer_pos] = '\0';
		ogg123_input_parse(ogg123_read_buffer);
		}
#endif
}

static void ogg123_input_read_reset(void)
{
	ogg123_read_buffer_pos = 0;
}

/*
 *-----------------------------------------------------------------------------
 * child setup (fork)
 *-----------------------------------------------------------------------------
 */

#define OGG123_MAX_COMMANDS 32

static gint ogg123_child_run(SongData *sd, gint position)
{
	pid_t frk_pid;
	char cmd_arguments[OGG123_MAX_COMMANDS][512];
	char *cmd_ptr[OGG123_MAX_COMMANDS];
	int cmd_cnt = 0;
	gchar *exec_bin = OGG123_BINARY;

	if (!sd || ogg123_child_pid != -1) return FALSE;

	strcpy(cmd_arguments[cmd_cnt], OGG123_BINARY);
	cmd_ptr[cmd_cnt] = cmd_arguments[cmd_cnt];
	cmd_cnt++;

	strcpy(cmd_arguments[cmd_cnt], "-v");
	cmd_ptr[cmd_cnt] = cmd_arguments[cmd_cnt];
	cmd_cnt++;

	if (position > 0)
		{
		strcpy(cmd_arguments[cmd_cnt], "-k");
		cmd_ptr[cmd_cnt] = cmd_arguments[cmd_cnt];
		cmd_cnt++;

		sprintf(cmd_arguments[cmd_cnt], "%d", position);
		cmd_ptr[cmd_cnt] = cmd_arguments[cmd_cnt];
		cmd_cnt++;
		}

	if (strcmp(ogg123_get_device(ogg123_device), "default") != 0)
		{
		sprintf(cmd_arguments[cmd_cnt], "--device=%s", ogg123_get_device(ogg123_device));
		cmd_ptr[cmd_cnt] = cmd_arguments[cmd_cnt];
		cmd_cnt++;
		}

	if (ogg123_device_options && strlen(ogg123_device_options) > 0)
		{
		sprintf(cmd_arguments[cmd_cnt], "--device-option=%s", ogg123_device_options);
		cmd_ptr[cmd_cnt] = cmd_arguments[cmd_cnt];
		cmd_cnt++;
		}

	if (ogg123_extra_options &&
	    strlen(ogg123_extra_options) > 0 &&
	    cmd_cnt + 2 < OGG123_MAX_COMMANDS)
		{
		gchar **vector;
		gint i;

		vector = g_strsplit(ogg123_extra_options, " ", OGG123_MAX_COMMANDS - 2 - cmd_cnt);
		i = 0;

		while(vector && vector[i] != NULL)
			{
			strcpy(cmd_arguments[cmd_cnt], vector[i]);
			cmd_ptr[cmd_cnt] = cmd_arguments[cmd_cnt];
			cmd_cnt++;
			i++;
			}
		g_strfreev(vector);
		}

	strcpy(cmd_arguments[cmd_cnt], sd->path);
	cmd_ptr[cmd_cnt] = cmd_arguments[cmd_cnt];
	cmd_cnt++;

	strcpy(cmd_arguments[cmd_cnt], "");
	cmd_ptr[cmd_cnt] = NULL;
	cmd_cnt++;

	/* Create the pipe. */
	if (pipe(ogg123_pipe))
                {
                fprintf (stderr, _("Pipe failed.\n"));
                return FALSE;
                }

        if (debug_mode) printf("opening: %s\n", sd->path);

	/* Create the child process. */
	frk_pid = fork ();
	if (frk_pid == (pid_t) 0)
		{
		/* This is the child process. */
		dup2(ogg123_pipe[1], 2);
		close(ogg123_pipe[0]);

		execvp(exec_bin, cmd_ptr);
		printf(_("unable to run %s (in the path?)\n"), exec_bin);
		_exit(1);
		}
	else if (frk_pid < (pid_t) 0)
		{
		/* The fork failed. */
		fprintf (stderr, _("Fork failed.\n"));
		close(ogg123_pipe[0]);
		close(ogg123_pipe[1]);
		pid = 0;
		ogg123_child_pid = -1;
		return FALSE;
		}
	else
		{
		/* This is the parent process. */
		ogg123_child_pid = (int) frk_pid;
		pid = ogg123_child_pid;

		close(ogg123_pipe[1]);
		}

	if (debug_mode) printf("ogg123 pid = %d\n", ogg123_child_pid);

	ogg123_input_read_reset();
	ogg123_control_read_id = gdk_input_add (ogg123_pipe[0], GDK_INPUT_READ, ogg123_input_read_cb, NULL);

	return TRUE;
}

/*
 *----------------------------------------------------------------------------
 * module callback funcs
 *----------------------------------------------------------------------------
 */

static void ogg123_data_init(SongData *sd)
{
	sd->type_description = _("Ogg bitstream audio file");
}

static gint ogg123_data_set(SongData *sd, gint read_info)
{
	if (read_info && !sd->info_loaded)
		{
		sd->info_loaded = TRUE;

		/* FIXME well, this is currently nothing, nada, no extended info */
		}

	return TRUE;
}

/*
 *----------------------------------------------------------------------------
 * module play funcs
 *----------------------------------------------------------------------------
 */

static gint ogg123_start(SongData *sd, gint position)
{
	gint ret;

	if (debug_mode) printf("io_ogg123.c: play started at %d\n", position);
	ogg123_current_sd = sd;

	ret = ogg123_child_run(sd, position);

	playback_done_command(EXEC_PLAY, !ret);

	return ret;
}

static gint ogg123_stop(SongData *sd)
{
	if (debug_mode) printf("io_ogg123.c: play stopped\n");
	if (sd != ogg123_current_sd)
		{
		printf("io_ogg123.c warning: attempt to stop playback of non matching file\n");
		}
	ogg123_child_shutdown();
	ogg123_current_sd = NULL;

	playback_done_command(EXEC_STOP, FALSE);
	return TRUE;
}

static gint ogg123_pause(SongData *sd)
{
	if (debug_mode) printf("io_ogg123.c: play paused at %d\n", seconds);
	if (sd != ogg123_current_sd)
		{
		printf("io_ogg123.c warning: attempt to pause playback of non matching file\n");
		}

	ogg123_child_shutdown();

	playback_done_command(EXEC_PAUSE, FALSE);
        return TRUE;
}

static gint ogg123_continue(SongData *sd)
{
	gint ret;

	if (debug_mode) printf("io_ogg123.c: play restarted at %d\n", seconds);
	if (sd != ogg123_current_sd)
		{
		printf("io_ogg123.c warning: attempt to continue playback of non matching file\n");
		}
	ogg123_current_sd = sd;
	ret = ogg123_child_run(sd, seconds);

        playback_done_command(EXEC_CONTINUE, !ret);
        return ret;
}

static gint ogg123_seek(SongData *sd, gint position)
{
	if (debug_mode) printf("io_ogg123.c: play seeking to %d\n", position);
	if (sd != ogg123_current_sd)
		{
		printf("io_ogg123.c warning: attempt to seek in non matching file\n");
		}

	if (status == STATUS_PLAY)
		{
		gint ret;

		ogg123_child_shutdown();

		seconds_remaining += seconds - position;
		seconds = position;

		ret = ogg123_child_run(sd, position);

		playback_done_command(EXEC_SEEK, !ret);
		return ret;
		}
	else
		{
		playback_done_command(EXEC_SEEK, FALSE);
		return TRUE;
		}
}

/*
 *----------------------------------------------------------------------------
 * player module interface routines
 *----------------------------------------------------------------------------
 */

void ogg123_init(void)
{
	IO_ModuleData *imd;
	gchar *mod_desc = "OGG file";
	gint id;

	if (debug_mode) printf("ogg123 module initing...\n");

	ogg123_enabled = file_in_path(OGG123_BINARY);
	if (!ogg123_enabled)
		{
		printf(_("Failed to find %s in your path!\n"), OGG123_BINARY);
		printf(_("ogg123 player module disabled.\n"));
		}

	imd = g_new0(IO_ModuleData, 1);

	imd->title = "ogg123";
	imd->description = _("ogg player");

	if (ogg123_enabled)
		{
		imd->songdata_init_func = ogg123_data_init;
		imd->songdata_info_func = ogg123_data_set;

		imd->start_func = ogg123_start;
		imd->stop_func = ogg123_stop;
		imd->pause_func = ogg123_pause;
		imd->continue_func = ogg123_continue;
		imd->seek_func = ogg123_seek;

		imd->info_func = ogg123_info_window;
		}

	imd->config_load_func = ogg123_config_load;
	imd->config_save_func = ogg123_config_save;
	imd->config_init_func = ogg123_config_init;
	imd->config_apply_func = ogg123_config_apply;
	imd->config_close_func = ogg123_config_close;

	id = player_module_register(imd);

	module_register_file_suffix_type(".ogg", mod_desc, id);
}

