/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2003 Bert Vermeulen
    Copyright (C) 2002-2003 Peter Stamfest

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: mainwin.c,v 1.55 2003/11/03 21:07:57 stamfest Exp $ */

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <config.h>

#include "common.h"
#include "search.h"
#include "mainwin.h"
#include "configfile.h"
#include "browse.h"
#include "prefs.h"
#include "util.h"
#include "template.h"
#include "filter.h"
#include "COPYING.h"
#include "i18n.h"
#include "debug.h"
#include "input.h"
#include "state.h"
#include "errorchain.h"
#include "progress.h"

#include "../icons/logo.xpm"

#ifdef HAVE_LDAP_STR2OBJECTCLASS
#    include "schemabrowse.h"
#endif

struct mainwin_data mainwin;

static void create_license_window(void);
static GtkWidget *create_about_window(void);

static void close_current_tab(struct mainwin_data *win);
static void mainwin_destroyed(GtkWidget *widget, struct mainwin_data *win);
static void switchpage_refocus(GtkNotebook *notebook, GtkNotebookPage *page,
			       int pagenum, struct mainwin_data *win);

struct tab *get_last_of_mode(int mode)
{
     if(!mainwin.lastofmode)
	  return NULL;
     
     return g_hash_table_lookup(mainwin.lastofmode, GINT_TO_POINTER(mode));
}


void go_to_page(struct tab *tab)
{
     gtk_notebook_set_page(GTK_NOTEBOOK(mainwin.mainbook), 
			   gtk_notebook_page_num(GTK_NOTEBOOK(mainwin.mainbook),
						 tab->content));
}


void enter_last_of_mode(struct tab *tab)
{

     if(!mainwin.lastofmode)
	  mainwin.lastofmode = g_hash_table_new(g_direct_hash, g_direct_equal);
     
     g_hash_table_insert(mainwin.lastofmode, (gpointer) tab->type, tab);

}


void fill_serverlist_combo(GtkWidget *combo)
{
     GList *serverlist, *I;
     struct ldapserver *server;

     if(combo == NULL)
	  return;

     serverlist = NULL;
     for (I = config->servers ; I ; I = I->next) {
	  server = (struct ldapserver *) I->data;
	  serverlist = g_list_append(serverlist, server->name);
     }

     if(!serverlist)
	  /* all servers were deleted -- pass an empty string to the combo */
	  serverlist = g_list_append(serverlist, "");

     gtk_combo_set_popdown_strings(GTK_COMBO(combo), serverlist);

     g_list_free(serverlist);

}

static gboolean mainwin_restore_snapshot(struct mainwin_data *win)
{
     int i, type;
     char tmp[32];
     struct pbar_win *pw = NULL;

     if (!config->restore_tabs) return FALSE;
     if (!exists_entity("mainwin.tabs")) return FALSE;

     pw = create_progress_bar_in_window(_("Restoring last GUI state"));
     update_progress(pw, _("Restoring tabs"));

     for (i = 0 ; ; i++) {
	  snprintf(tmp, sizeof(tmp), "mainwin.tabs.%d", i);
	  if (!exists_entity(tmp)) break;
	  type = state_value_get_int(tmp, "type", -1);
	  if (type > 0) {
	       struct tab *tab = new_modetab(win, type);
	       int error_ctx = error_new_context(_(""), pw->win);

	       if (tab->vtab && tab->vtab->restore_snapshot) {
		    tab->vtab->restore_snapshot(error_ctx, tmp, tab, pw);
	       }

	       error_flush(error_ctx);
	  }
	  update_progress(pw, NULL);
	  if (pw->cancelled) break;
     }

     if (i > 0) {
	  type = state_value_get_int("mainwin.tabs", "active", -1);
	  gtk_notebook_set_page(GTK_NOTEBOOK(win->mainbook), type);
     } 

     update_progress(pw, _("Restoring tabs"));
     free_progress(pw);

     

     return i > 0;
}

static void mainwin_save_snapshot(struct mainwin_data *win)
{
     struct tab *tab = NULL; 
     int i;
     char tmp[32];

     rm_value("mainwin.tabs");

     if (!config->restore_tabs) return;
     
     for( i = 0 ; (tab = mainwin_get_tab_nth(win, i)) != NULL ; i++) {
	  snprintf(tmp, sizeof(tmp), "mainwin.tabs.%d", i);
	  state_value_set_int(tmp, "type", tab->type);
	  if (tab->vtab && tab->vtab->save_snapshot) {
	       int error_ctx = 
		    error_new_context(_("Saving main window snapshot"), NULL);
	       tab->vtab->save_snapshot(error_ctx, tmp, tab);
	       error_flush(error_ctx);
	  }
     }

     state_value_set_int("mainwin.tabs", "active", 
			 gtk_notebook_get_current_page(GTK_NOTEBOOK(win->mainbook)));
}


/* gtk2 checked (multiple destroy callbacks safety), confidence 0.7:
   cleanup_all_tabs semantics? */
static void mainwin_destroyed(GtkWidget *widget, struct mainwin_data *win)
{
     mainwin_save_snapshot(win);
     cleanup(win);
     gtk_main_quit();
}

void cleanup(struct mainwin_data *win)
{
     cleanup_all_tabs(win);
}

static gboolean ctrl_b_hack(GtkWidget *widget, GdkEventKey *event, gpointer obj)
{
     if(event && event->type == GDK_KEY_PRESS &&
	event->state & GDK_CONTROL_MASK && event->keyval == GDK_b) {
	  gtk_signal_emit_by_name(GTK_OBJECT(obj), "activate");
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
	  return(TRUE);
     }

#if GTK_MAJOR >= 2
     return(FALSE);
#else
     return(TRUE);  /* may be a bug to return TRUE unconditionally in gtk 1.2 */
#endif
}


static gboolean ctrl_w_hack(GtkWidget *widget, GdkEventKey *event, gpointer obj)
{
     if(event && event->type == GDK_KEY_PRESS &&
	event->state & GDK_CONTROL_MASK && event->keyval == GDK_w) {
	  gtk_signal_emit_by_name(GTK_OBJECT(obj), "activate");
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
	  return(TRUE);
     }

#if GTK_MAJOR >= 2
     return(FALSE);
#else
     return(TRUE);  /* may be a bug to return TRUE unconditionally in gtk 1.2 */
#endif
}

static void new_modetab_search(struct mainwin_data *win) 
{
     new_modetab(win, SEARCH_MODE);
}

static void new_modetab_browse(struct mainwin_data *win) 
{
     new_modetab(win, BROWSE_MODE);
}

static void new_modetab_schema(struct mainwin_data *win) 
{
     new_modetab(win, SCHEMA_MODE);
}


static GList *log_list = NULL;
static int log_list_len = 0;

void clear_message_history()
{
     if (log_list) {
	  g_list_foreach(log_list, (GFunc) g_free, NULL);
	  g_list_free(log_list);
	  log_list = NULL;
     }

     if (mainwin.ml_text) {
#if GTK_MAJOR >= 2
	  GtkTextIter start;
	  GtkTextIter end;
	  
	  gtk_text_buffer_get_start_iter(mainwin.ml_buffer, &start);
	  gtk_text_buffer_get_end_iter(mainwin.ml_buffer, &end);

	  gtk_text_buffer_delete(mainwin.ml_buffer, &start, &end);
#else
	  gtk_editable_delete_text(GTK_EDITABLE(mainwin.ml_text), 0, -1);
#endif
     }
}


void message_log_append(const char *buf)
{
     log_list = g_list_append(log_list, g_strdup(buf));
     log_list_len++;

     if (mainwin.ml_text) {
#if GTK_MAJOR >= 2
	  GtkTextIter iter;
	  gtk_text_buffer_get_end_iter(mainwin.ml_buffer, &iter);
	  gtk_text_buffer_insert(mainwin.ml_buffer, &iter,
				 buf, strlen(buf));
	  gtk_text_buffer_insert(mainwin.ml_buffer, &iter, "\n", 1);

	  gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(mainwin.ml_text),
				       gtk_text_buffer_create_mark(mainwin.ml_buffer,
								   NULL,
								   &iter,
								   FALSE),
				       0.0,
				       FALSE,
				       0.0, 0.0);
#else
	  int pos = gtk_text_get_length(GTK_TEXT(mainwin.ml_text));
	  gtk_editable_insert_text(GTK_EDITABLE(mainwin.ml_text), buf, 
				   strlen(buf), &pos);
	  gtk_editable_insert_text(GTK_EDITABLE(mainwin.ml_text), "\n", 
				   1, &pos);
#endif
     }
     while (log_list_len > MESSAGE_LOG_MAX) {
	  g_free(log_list->data);
	  log_list = g_list_remove(log_list, log_list->data);
	  log_list_len--;
     }
}

static void message_log_destroyed(GtkWidget *window,
				  struct mainwin_data *win)
{
     win->ml_window = NULL;
     win->ml_text   = NULL;
#if GTK_MAJOR >= 2
     win->ml_buffer = NULL;
#endif
}

static void clear_clicked(void)
{
     clear_message_history();
}

static void message_log(struct mainwin_data *win) 
{
     GtkWidget *window, *vbox0, *scrwin, *text, *bbox, *button;
#if GTK_MAJOR >= 2
     GtkTextBuffer *buffer;
     GtkTextIter iter;
#else
     int pos = 0;
#endif
     GList *I;

     assert(win);

     if (win->ml_window) {
	  gdk_beep(); /* Is this OK, philosophically? */
#if GTK_MAJOR >= 2
	  gtk_window_present(GTK_WINDOW(win->ml_window));
#else
	  gtk_window_activate_focus(GTK_WINDOW(win->ml_window));
#endif
	  return;
     }

#if GTK_MAJOR < 2
     window = stateful_gtk_window_new(GTK_WINDOW_DIALOG,
				      "statusbar-log", 500, 350); 
#else
     window = stateful_gtk_window_new(GTK_WINDOW_TOPLEVEL,
				      "statusbar-log", 500, 350); 
#endif
     win->ml_window = window;

     gtk_widget_realize(window);

     gtk_signal_connect(GTK_OBJECT(window), "destroy",
			GTK_SIGNAL_FUNC(message_log_destroyed), win);

     gtk_signal_connect(GTK_OBJECT(window), "key_press_event",
			GTK_SIGNAL_FUNC(close_on_esc),
			(gpointer) window);

/*      current_search_options_window = window; */
     gtk_window_set_title(GTK_WINDOW(window), _("Message Log"));
     gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);

     vbox0 = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox0), 
				CONTAINER_BORDER_WIDTH);
     gtk_widget_show(vbox0);
     gtk_container_add(GTK_CONTAINER(window), vbox0);

     /* scrolled window to hold the log */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_show(scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
#if GTK_MAJOR >= 2
     gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					 GTK_SHADOW_IN);
#endif
     gtk_box_pack_start(GTK_BOX(vbox0), scrwin, TRUE, TRUE, 0);

#if GTK_MAJOR >= 2
     text = gtk_text_view_new();
     buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
     win->ml_buffer = buffer;

     gtk_text_view_set_editable(GTK_TEXT_VIEW(text), FALSE);
     gtk_text_buffer_get_end_iter(buffer, &iter);
#else
     text = gtk_text_new(NULL, NULL);
     gtk_text_set_editable(GTK_TEXT(text), FALSE);
#endif

     win->ml_text = text;

     gtk_widget_show(text);
     gtk_container_add(GTK_CONTAINER(scrwin), text); 

     for (I = log_list ; I ; I = g_list_next(I) ) {
#if GTK_MAJOR >= 2
	  gtk_text_buffer_insert(buffer, &iter,
				 I->data, strlen(I->data));
	  gtk_text_buffer_insert(buffer, &iter, "\n", 1);
#else
	  gtk_editable_insert_text(GTK_EDITABLE(text), I->data, 
				   strlen(I->data), &pos);
	  gtk_editable_insert_text(GTK_EDITABLE(text), "\n", 1, &pos);
#endif
     }

#if GTK_MAJOR >= 2
     gtk_text_buffer_get_end_iter(buffer, &iter);
     gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(text),
				  gtk_text_buffer_create_mark(buffer, NULL,
							      &iter, FALSE),
				  0.0,
				  FALSE,
				  0.0, 0.0);
#endif

     bbox = gtk_hbutton_box_new();
     gtk_widget_show(bbox);

     gtk_box_pack_end(GTK_BOX(vbox0), bbox, FALSE, FALSE, 3);

#if GTK_MAJOR >= 2
     button = gtk_button_new_from_stock(GTK_STOCK_CLEAR);
#else
     button = gq_button_new_with_label(_("_Clear"));
#endif
     gtk_widget_show(button);
     gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(clear_clicked),
			(gpointer) win);
     gtk_box_pack_start(GTK_BOX(bbox), button, FALSE, TRUE, 10);

#if GTK_MAJOR >= 2
     button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
#else
     button = gq_button_new_with_label(_("_Close"));
#endif
     gtk_widget_show(button);
     gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			       GTK_SIGNAL_FUNC(gtk_widget_destroy),
			       (gpointer) window);
     gtk_box_pack_end(GTK_BOX(bbox), button, FALSE, TRUE, 10);

     GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(button);

     gtk_widget_show(window);
}

/* Callback function called when a tab gets removed from the
   notebook. */
static void remove_tab(GtkContainer *notebook,
		       GtkWidget *content,
		       struct mainwin_data *win)
{
     int thismode;
     struct tab *tab = NULL; 
     int i;

     tab = gtk_object_get_data(GTK_OBJECT(content), "tab");
     if (tab) {
	  thismode = tab->type;
	  g_hash_table_insert(win->lastofmode, (gpointer) thismode, NULL);
	  
	  /* try to find another tab with the same mode so we can put that
	     one into lastofmode... */
	  for( i = 0 ; (tab = mainwin_get_tab_nth(win, i)) != NULL ; i++) {
	       if (tab->type == thismode) {
		    /* found one! */
		    enter_last_of_mode(tab);
		    break;
	       }
	  }
     }

     if (gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), 0) == NULL) {
	  gtk_widget_destroy(win->mainwin);
     }
}


void mainwin_update_filter_menu(struct mainwin_data *win)
{
     GList *menuitems = gtk_container_children(GTK_CONTAINER(win->filtermenu));
     GList *I;

     /* Filters | list of filters */

     if (menuitems) {
	  for ( I = g_list_first(menuitems) ; I ; I = g_list_next(I) ) {
	       GtkWidget *item = GTK_WIDGET(I->data);
	       gpointer data = gtk_object_get_data(GTK_OBJECT(item), "filter");
	       
	       if (data) {
		    gtk_widget_destroy(item);
	       }
	  }
	  
	  g_list_free(menuitems);
     }

     for ( I = g_list_first(config->filters) ; I ; I = g_list_next(I) ) {
	  struct gq_filter *filter;
	  GtkWidget *menuitem;

	  filter = (struct gq_filter *) I->data;
	  menuitem = gtk_menu_item_new_with_label(filter->name);

	  gtk_object_set_data(GTK_OBJECT(menuitem), "filter", filter);
	  gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
				    GTK_SIGNAL_FUNC(filter_selected),
				    (GtkObject *) filter);

	  gtk_container_add(GTK_CONTAINER(win->filtermenu), menuitem);
	  gtk_widget_show(menuitem);
     }
}


void create_mainwin(struct mainwin_data *win)
{
     GtkWidget *outer_vbox, *main_vbox, *menubar, *menuitem, *submenu;
     GtkWidget *File, *menuFile, *New, *Close, *ShowM, *Quit;
     GtkWidget *Search, *Browse, *Schema;
     GtkWidget *menuHelp, *Help, *License, *About;
     GtkWidget *Filters, *menuFilters;
     GtkWidget *handlebox;
     GtkAccelGroup *accel_group;

     assert(win != NULL);
/*      mainwin = gtk_window_new(GTK_WINDOW_TOPLEVEL); */
/*      gtk_window_set_default_size(GTK_WINDOW(mainwin), 770, 478); */

     win->mainwin = stateful_gtk_window_new(GTK_WINDOW_TOPLEVEL,
					    "mainwin", 770, 478); 

     gtk_container_border_width(GTK_CONTAINER(win->mainwin), 0);
     gtk_signal_connect(GTK_OBJECT(win->mainwin), "destroy",
			GTK_SIGNAL_FUNC(mainwin_destroyed),
			win);
     gtk_window_set_title(GTK_WINDOW(win->mainwin), _("GQ"));
     gtk_window_set_policy(GTK_WINDOW(win->mainwin), FALSE, TRUE, FALSE);



     outer_vbox = gtk_vbox_new(FALSE, 2);
     gtk_container_border_width(GTK_CONTAINER(outer_vbox), 0);
     gtk_widget_show(outer_vbox);
     gtk_container_add(GTK_CONTAINER(win->mainwin), outer_vbox);

     accel_group = gtk_accel_group_new();

#if GTK_MAJOR < 2
     gtk_accel_group_attach(accel_group, GTK_OBJECT(win->mainwin));
#else 
     gtk_window_add_accel_group(GTK_WINDOW(win->mainwin), accel_group);
#endif


     handlebox = gtk_handle_box_new();
     gtk_widget_show(handlebox);
     gtk_box_pack_start(GTK_BOX(outer_vbox), handlebox, FALSE, TRUE, 0);

     menubar = gtk_menu_bar_new();
     gtk_widget_show(menubar);
     gtk_container_add(GTK_CONTAINER(handlebox), menubar);

     /* File menu */
     File = gq_menu_item_new_with_label(_("_File"));
     gtk_widget_show(File);
     gtk_container_add(GTK_CONTAINER(menubar), File);

     menuFile = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(File), menuFile);

     /* File | New */
     New = gq_menu_item_new_with_label(_("_New tab"));
     gtk_widget_show(New);
     gtk_container_add(GTK_CONTAINER(menuFile), New);
     submenu = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(New), submenu);

     /* File | New | Search */
     Search = gq_menu_item_new_with_label(_("_Search"));
     gtk_widget_show(Search);
     gtk_menu_append(GTK_MENU(submenu), Search);
     gtk_signal_connect_object(GTK_OBJECT(Search), "activate",
			       GTK_SIGNAL_FUNC(new_modetab_search),
			       (gpointer) win);
     gtk_widget_add_accelerator(Search, "activate", accel_group, 'S',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);

     /* File | New | Browse */
     Browse = gq_menu_item_new_with_label(_("_Browse"));
     gtk_widget_show(Browse);
     gtk_menu_append(GTK_MENU(submenu), Browse);
     gtk_signal_connect_object(GTK_OBJECT(Browse), "activate",
			       GTK_SIGNAL_FUNC(new_modetab_browse),
			       (gpointer) win);
     gtk_widget_add_accelerator(Browse, "activate", accel_group, 'B',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
     /* ctrl-b is used by text widgets, so the searchterm textbox that
	always has focus in search mode blocks the above accelerator...*/
     gtk_signal_connect(GTK_OBJECT(win->mainwin), "key_press_event",
			GTK_SIGNAL_FUNC(ctrl_b_hack),
			(gpointer) Browse);

     /* File | New | Schema */
     Schema = gq_menu_item_new_with_label(_("S_chema"));
     gtk_widget_show(Schema);
     gtk_menu_append(GTK_MENU(submenu), Schema);
     gtk_widget_add_accelerator(Schema, "activate", accel_group, 'Z',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     gtk_signal_connect_object(GTK_OBJECT(Schema), "activate",
			       GTK_SIGNAL_FUNC(new_modetab_schema),
			       (gpointer) win);
#else
     gtk_widget_set_sensitive(Schema, FALSE);
#endif

     /* File | Preferences */
     menuitem = gq_menu_item_new_with_label(_("_Preferences"));
     gtk_widget_show(menuitem);
     gtk_container_add(GTK_CONTAINER(menuFile), menuitem);
     gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
			       GTK_SIGNAL_FUNC(create_prefs_window),
			       (gpointer) win);
     gtk_widget_add_accelerator(menuitem, "activate", accel_group, 'P',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);

     /* File | Close */
     Close = gq_menu_item_new_with_label(_("_Close tab"));
     gtk_widget_show(Close);
     gtk_container_add(GTK_CONTAINER(menuFile), Close);
     gtk_signal_connect_object(GTK_OBJECT(Close), "activate",
			       GTK_SIGNAL_FUNC(close_current_tab), 
			       (gpointer) win);
     gtk_widget_add_accelerator(Close, "activate", accel_group, 'W',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
     /* :-( */
     gtk_signal_connect(GTK_OBJECT(win->mainwin), "key_press_event",
			GTK_SIGNAL_FUNC(ctrl_w_hack),
			(gpointer) Close);

     /* File | Show Message */

     ShowM = gq_menu_item_new_with_label(_("Show _Message Log"));
     gtk_widget_show(ShowM);
     gtk_container_add(GTK_CONTAINER(menuFile), ShowM);
     gtk_signal_connect_object(GTK_OBJECT(ShowM), "activate",
			       GTK_SIGNAL_FUNC(message_log), 
			       (gpointer) win);

     /* File | Quit */
     Quit = gq_menu_item_new_with_label(_("_Quit"));
     gtk_widget_show(Quit);
     gtk_container_add(GTK_CONTAINER(menuFile), Quit);
     gtk_signal_connect_object(GTK_OBJECT(Quit), "activate",
			       GTK_SIGNAL_FUNC(gtk_widget_destroy),
			       GTK_OBJECT(win->mainwin));
     gtk_widget_add_accelerator(Quit, "activate", accel_group, 'Q',
				GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);


     /* Filters menu */
     Filters = gq_menu_item_new_with_label(_("F_ilters"));
     gtk_widget_show(Filters);
     gtk_container_add(GTK_CONTAINER(menubar), Filters);

     menuFilters = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(Filters), menuFilters);
     win->filtermenu = menuFilters;

     /* Filters | New */
     New = gq_menu_item_new_with_label(_("_New filter"));
     gtk_widget_show(New);
     gtk_container_add(GTK_CONTAINER(menuFilters), New);
     submenu = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(New), submenu);

     /* Filters | New | From Search tab */
     menuitem = gq_menu_item_new_with_label(_("From _Search tab"));
     gtk_widget_show(menuitem);
     gtk_menu_append(GTK_MENU(submenu), menuitem);
     gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
			       GTK_SIGNAL_FUNC(add_filter),
			       NULL);

     /* Filters | New | Filter Editor */
     menuitem = gq_menu_item_new_with_label(_("Filter _editor"));
     gtk_widget_show(menuitem);
     gtk_menu_append(GTK_MENU(submenu), menuitem);
     gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
			       GTK_SIGNAL_FUNC(add_new_filter_callback),
			       NULL);

     /* Filters | Edit Filters */
     menuitem = gq_menu_item_new_with_label(_("_Edit Filters"));
     gtk_widget_show(menuitem);
     gtk_container_add(GTK_CONTAINER(menuFilters), menuitem);
     gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
			       GTK_SIGNAL_FUNC(show_filters),
			       NULL);

     /* Filters separator */
     menuitem = gtk_menu_item_new();
     gtk_widget_show(menuitem);
     gtk_container_add(GTK_CONTAINER(menuFilters), menuitem);

     mainwin_update_filter_menu(win);

     /* Help menu */
     Help = gq_menu_item_new_with_label(_("_Help"));
     gtk_widget_show(Help);
     gtk_container_add(GTK_CONTAINER(menubar), Help);
     gtk_menu_item_right_justify(GTK_MENU_ITEM(Help));

     menuHelp = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(Help), menuHelp);

     /* Help | License */
     License = gq_menu_item_new_with_label(_("_License"));
     gtk_widget_show(License);
     gtk_container_add(GTK_CONTAINER(menuHelp), License);
     gtk_signal_connect(GTK_OBJECT(License), "activate",
			GTK_SIGNAL_FUNC(create_license_window), NULL);

     /* Help | About */
     About = gq_menu_item_new_with_label(_("_About"));
     gtk_widget_show(About);
     gtk_container_add(GTK_CONTAINER(menuHelp), About);
     gtk_signal_connect(GTK_OBJECT(About), "activate",
			GTK_SIGNAL_FUNC(create_about_window), NULL);

     main_vbox = gtk_vbox_new(FALSE, 2);
     gtk_container_border_width(GTK_CONTAINER(main_vbox), 4);
     gtk_widget_show(main_vbox);
     gtk_box_pack_start(GTK_BOX(outer_vbox), main_vbox, TRUE, TRUE, 1);

     win->mainbook = gtk_notebook_new();
     gtk_widget_show(win->mainbook);
/*      GTK_WIDGET_UNSET_FLAGS(GTK_NOTEBOOK(mainbook), GTK_CAN_FOCUS); */
     gtk_box_pack_start(GTK_BOX(main_vbox), win->mainbook, TRUE, TRUE, 0);

     win->statusbar = gtk_statusbar_new();
     gtk_widget_show(win->statusbar);

     gtk_box_pack_end(GTK_BOX(outer_vbox), win->statusbar, FALSE, FALSE, 0);
     gtk_widget_set_sensitive(win->statusbar, TRUE);
     
     gtk_signal_connect(GTK_OBJECT(win->mainbook), "switch-page",
			GTK_SIGNAL_FUNC(switchpage_refocus), win);
     gtk_signal_connect(GTK_OBJECT(win->mainbook), "remove",
			GTK_SIGNAL_FUNC(remove_tab), win);

     gtk_widget_realize(win->mainwin);

     if (! mainwin_restore_snapshot(win)) {
	  new_modetab(win, SEARCH_MODE);
	  new_modetab(win, BROWSE_MODE | 32768);
	  new_modetab(win, SCHEMA_MODE | 32768);
     }

     gtk_widget_show(win->mainwin);
}

struct tab *mainwin_get_tab_nth(struct mainwin_data *win, int n)
{
     GtkWidget *content = 
	  gtk_notebook_get_nth_page(GTK_NOTEBOOK(win->mainbook), n);
     if (content == NULL) return NULL;

     return gtk_object_get_data(GTK_OBJECT(content), "tab");
}

struct tab *mainwin_get_current_tab(GtkWidget *notebook)
{
     int tabnum = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
     GtkWidget *content = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), 
						    tabnum);
     return gtk_object_get_data(GTK_OBJECT(content), "tab");
}

struct tab *new_modetab(struct mainwin_data *win, int mode)
{
     GtkWidget *label, *focusbox;
     struct tab *tab;
     int focus;

     /* hack, hack */
     focus = !(mode & 32768);
     mode &= 32767;

     switch(mode) {
     case SEARCH_MODE:
	  label = gq_label_new(_("_Search"));
	  tab = new_searchmode();
	  break;
     case BROWSE_MODE:
	  label = gq_label_new(_("_Browse"));
	  tab = new_browsemode();
	  break;
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     case SCHEMA_MODE:
	  label = gq_label_new(_("S_chema"));
	  tab = new_schemamode();
	  break;
#endif
     default:
	  return NULL;
     }

     gtk_object_set_data(GTK_OBJECT(tab->content), "tab", tab);

     tab->win = win;

     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(win->mainbook), 
			      tab->content,
			      label);

     if(focus) {
	  enter_last_of_mode(tab);

	  gtk_notebook_set_page(GTK_NOTEBOOK(win->mainbook), -1);

	  focusbox = tab->focus;
	  if(focusbox)
	       gtk_widget_grab_focus(focusbox);
     }
     return tab;
}


static void switchpage_refocus(GtkNotebook *notebook, GtkNotebookPage *page,
			       int pagenum, struct mainwin_data *win)
{
     GtkWidget *focusbox;
     struct tab *tab;

     tab = mainwin_get_tab_nth(win, pagenum);
     if(!tab)
	  return;

     /* retrieve mode, store this pane as the last one used for this mode */
     enter_last_of_mode(tab);
     
     focusbox = tab->focus;
     if(focusbox) {
	  gtk_widget_grab_focus(focusbox);
	  gtk_editable_select_region(GTK_EDITABLE(focusbox), 0, -1);
     }
}


void cleanup_all_tabs(struct mainwin_data *win)
{
     /* don't waste time refocusing on disappearing tabs */
     gtk_signal_disconnect_by_func(GTK_OBJECT(win->mainbook),
				   GTK_SIGNAL_FUNC(switchpage_refocus), win);
}


static void close_current_tab(struct mainwin_data *win)
{
     int tabnum;
     GtkWidget *content;

     tabnum = gtk_notebook_get_current_page(GTK_NOTEBOOK(win->mainbook));
     content = gtk_notebook_get_nth_page(GTK_NOTEBOOK(win->mainbook), tabnum);
     /* for whatever reason: gtk_notebook_remove_page does not call
	the remove signal on the notebook. I consider this to be a GTK
	bug */

/*      gtk_notebook_remove_page(GTK_NOTEBOOK(win->mainbook), tabnum); */

     gtk_widget_destroy(content);
}

void update_serverlist(struct mainwin_data *win)
{
     struct tab *tab;
     int i;

     for( i = 0 ; (tab = mainwin_get_tab_nth(win, i)) != NULL ; i++) {
	  switch(tab->type) {
	  case SEARCH_MODE:
	       fill_serverlist_combo(SEARCHTAB(tab)->serverlist_combo);
	       break;
	  case BROWSE_MODE:
	       update_browse_serverlist(tab);
	       break;
	  case SCHEMA_MODE:
	       update_schema_serverlist(tab);
	       break;
	  }
     }
}

static GtkWidget *license_window = NULL;

static void close_license_window(GtkWidget *window)
{
     gtk_widget_destroy(window);
}

static void destroy_license_window(GtkWidget *window)
{
     assert(window == license_window);
     license_window = NULL;
}

static void create_license_window(void)
{
#if GTK_MAJOR < 2
     GtkWidget *hbox;
     GtkStyle *license_style;
#else
     PangoFontDescription *font_desc;
#endif
     GtkWidget *window, *vbox1, *vbox2, *text, *hbox0, *w;
     GtkWidget *ok_button;

     if (license_window) {
#if GTK_MAJOR >= 2
	  gtk_window_present(GTK_WINDOW(license_window));
#else
	  gtk_window_activate_focus(GTK_WINDOW(license_window));
#endif
	  return;
     }

     license_window = window = gtk_dialog_new();
     gtk_widget_set_usize(window, 540, 400);
     gtk_window_set_title(GTK_WINDOW(window), _("License"));

     vbox1 = GTK_DIALOG(window)->vbox;
/*       gtk_container_border_width(GTK_CONTAINER(vbox1), 20); */
     gtk_container_border_width(GTK_CONTAINER(vbox1), CONTAINER_BORDER_WIDTH);
     gtk_widget_show(vbox1);

#if GTK_MAJOR < 2
     hbox = gtk_hbox_new(FALSE, 0);
/*      gtk_container_border_width(GTK_CONTAINER(hbox), CONTAINER_BORDER_WIDTH); */
     gtk_widget_show(hbox);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox, TRUE, TRUE, 0);

     license_style = gtk_style_new();
     license_style->font = gdk_font_load("fixed");

     text = gtk_text_new(NULL, NULL);
     gtk_widget_set_style(text, license_style);
     gtk_style_unref(license_style);

     gtk_text_set_point(GTK_TEXT(text), 0);
     gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL,
		     license, strlen(license));
     gtk_widget_show(text);
     gtk_text_set_editable(GTK_TEXT(text), FALSE);

     gtk_box_pack_start(GTK_BOX(hbox), text, TRUE, TRUE, 0);

     w = gtk_vscrollbar_new(GTK_TEXT(text)->vadj);
     gtk_box_pack_start(GTK_BOX(hbox), w, FALSE, FALSE, 0);
     gtk_widget_show(w);
#else
     text = gtk_text_view_new();

     gtk_text_buffer_set_text(gtk_text_view_get_buffer(GTK_TEXT_VIEW(text)),
			      license, strlen(license));

     /* Change default font throughout the widget */
     font_desc = pango_font_description_from_string("Monospace");
     gtk_widget_modify_font(text, font_desc);
     pango_font_description_free(font_desc);

     gtk_widget_show(text);

     w = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(w), 
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_container_add(GTK_CONTAINER(w), text); 
     gtk_widget_show(w);

     gtk_box_pack_start(GTK_BOX(vbox1), w, TRUE, TRUE, 0);
#endif

     vbox2 = GTK_DIALOG(window)->action_area;
     gtk_widget_show(vbox2);

     hbox0 = gtk_hbutton_box_new();
     gtk_widget_show(hbox0);
     gtk_box_pack_start(GTK_BOX(vbox2), hbox0, TRUE, FALSE, 0);

#if GTK_MAJOR >= 2
     ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
#else
     ok_button = gq_button_new_with_label(_("_OK"));
#endif
     gtk_widget_show(ok_button);
     gtk_box_pack_start(GTK_BOX(hbox0), ok_button,
			FALSE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(ok_button);
     gtk_signal_connect(GTK_OBJECT(window), "destroy",
			(GtkSignalFunc) destroy_license_window,
			NULL);
     gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked",
			       (GtkSignalFunc) close_license_window,
			       GTK_OBJECT(window));
     gtk_signal_connect_object(GTK_OBJECT(window), "key_press_event",
			       (GtkSignalFunc) close_on_esc,
			       (gpointer) window);

     gtk_widget_show(window);

}

static GtkWidget *about_window = NULL;

static void destroy_about_window(GtkWidget *window)
{
     assert(window == about_window);
     about_window = NULL;
}

static GtkWidget *create_about_window(void)
{
     GtkWidget *dialog_vbox2, *hbox0, *hbox;
     GtkWidget *about_label;
     GtkWidget *dialog_action_area2;
     GtkWidget *about_ok_button;
     GtkWidget *pixmap;
     GdkPixmap *gq_icon;
     GdkBitmap *gq_icon_mask;
     GString *about_text = NULL;

     if (about_window) {
#if GTK_MAJOR >= 2
	  gtk_window_present(GTK_WINDOW(about_window));
#else
	  gtk_window_activate_focus(GTK_WINDOW(about_window));
#endif
	  return about_window;
     }

     about_window = gtk_dialog_new();
     gtk_widget_realize(GTK_WIDGET(about_window));
/*       gtk_container_border_width(GTK_CONTAINER(about_window), 12); */

     gtk_window_set_title(GTK_WINDOW(about_window), _("About GQ"));
     gtk_window_set_policy(GTK_WINDOW(about_window), FALSE, FALSE, FALSE);

     dialog_vbox2 = GTK_DIALOG(about_window)->vbox;
     gtk_widget_show(dialog_vbox2);

     hbox = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox);
     gtk_box_pack_start(GTK_BOX(dialog_vbox2), hbox, TRUE, TRUE, 10);

     gq_icon = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(about_window)->window,
					    &gq_icon_mask,
					    &about_window->style->white,
					    logo_xpm);
     pixmap = gtk_pixmap_new(gq_icon, gq_icon_mask);
     gtk_widget_show(pixmap);
     gtk_box_pack_start(GTK_BOX(hbox), pixmap, TRUE, TRUE, 10);

     about_text = g_string_sized_new(512);

     g_string_sprintf(about_text, 
		      "GQ %s\n\n\n%s\n\n"
		      "GTK version %d.%d.%d\n"
		      "GLib version %d.%d.%d\n", 
		      VERSION, about_blurb,
		      gtk_major_version, gtk_minor_version, gtk_micro_version,
		      glib_major_version, glib_minor_version, glib_micro_version);

     about_label = gtk_label_new(about_text->str);
     g_string_free(about_text, TRUE);

     gtk_label_set_justify(GTK_LABEL(about_label), GTK_JUSTIFY_LEFT);
     gtk_widget_show(about_label);
     gtk_box_pack_start(GTK_BOX(hbox), about_label, FALSE, FALSE, 4);

     dialog_action_area2 = GTK_DIALOG(about_window)->action_area;
     gtk_container_border_width(GTK_CONTAINER(dialog_action_area2), 
				CONTAINER_BORDER_WIDTH);
     gtk_widget_show(dialog_action_area2);

     hbox0 = gtk_hbutton_box_new();
     gtk_widget_show(hbox0);
     gtk_box_pack_start(GTK_BOX(dialog_action_area2), hbox0,
			TRUE, FALSE, 0);

#if GTK_MAJOR >= 2
     about_ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
#else
     about_ok_button = gq_button_new_with_label(_("_OK"));
#endif
     gtk_widget_show(about_ok_button);
     gtk_box_pack_start(GTK_BOX(hbox0), about_ok_button,
			TRUE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(about_ok_button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(about_ok_button);
/*       gtk_container_border_width(GTK_CONTAINER(about_ok_button), 7); */
     gtk_signal_connect_object(GTK_OBJECT(about_ok_button), "clicked",
			       GTK_SIGNAL_FUNC(gtk_widget_destroy),
			       GTK_OBJECT(about_window));

     gtk_signal_connect_object(GTK_OBJECT(about_window), "key_press_event",
			       GTK_SIGNAL_FUNC(close_on_esc), (gpointer) about_window);

     gtk_signal_connect(GTK_OBJECT(about_window), "destroy",
			(GtkSignalFunc) destroy_about_window,
			NULL);

     gtk_widget_show(about_window);

     return(about_window);
}


/* 
   Local Variables:
   c-basic-offset: 5
   End:
*/
