/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: configfile.c,v 1.23 2002/07/03 18:49:45 stamfest Exp $ */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>
#include <signal.h>

#include <glib.h>

#include "configfile.h"
#include "common.h"
#include "util.h"
#include "template.h"
#include "filter.h"
#include "errorchain.h"
#include "debug.h"
#include "i18n.h"


struct gq_config config;


struct keywordlist configwords[] = {
     { "gq-config",	  T_GQ_CONFIG ,        NEEDS_CLOSE },

     /* global options */
     { "confirm-mod",     T_CONFIRM_MOD,       NEEDS_CLOSE },
     { "search-argument", T_SEARCH_ARGUMENT,   NEEDS_CLOSE },
     { "show-dn",         T_SHOWDN,            NEEDS_CLOSE },
     { "show-oc",         T_SHOWOC,            NEEDS_CLOSE },
     { "show-rdn-only",   T_SHOW_RDN_ONLY,     NEEDS_CLOSE },
     { "sort-search-mode",T_SORT_SEARCH,       NEEDS_CLOSE },
     { "sort-browse-mode",T_SORT_BROWSE,       NEEDS_CLOSE },
     { "ldif-format",     T_LDIF_FORMAT,       NEEDS_CLOSE },

     /* per-server options */
     { "ldapserver",	  T_LDAPSERVER,        NEEDS_CLOSE },
     { "name",		  T_NAME,              NEEDS_CLOSE|NEEDS_DATA },
     { "ldaphost",	  T_LDAPHOST,          NEEDS_CLOSE|NEEDS_DATA },
     { "ldapport",	  T_LDAPPORT,          NEEDS_CLOSE|NEEDS_DATA },
     { "basedn",	  T_BASEDN,            NEEDS_CLOSE|NEEDS_DATA },
     { "binddn",	  T_BINDDN,            NEEDS_CLOSE|NEEDS_DATA },
     { "bindpw",	  T_BINDPW,            NEEDS_CLOSE|NEEDS_DATA },
     { "bindtype",        T_BINDTYPE,          NEEDS_CLOSE|NEEDS_DATA },
     { "search-attribute",T_SEARCHATTR,        NEEDS_CLOSE|NEEDS_DATA },
     { "maxentries",      T_MAXENTRIES,        NEEDS_CLOSE|NEEDS_DATA },
     { "cache-connection",T_CACHECONN,         NEEDS_CLOSE|NEEDS_DATA },
     { "enable-tls",      T_ENABLETLS,         NEEDS_CLOSE|NEEDS_DATA },
     { "schema-server",   T_SCHEMASERVER,      NEEDS_CLOSE|NEEDS_DATA },
     { "local-cache-timeout", T_LOCAL_CACHE_TIMEOUT, NEEDS_CLOSE|NEEDS_DATA },
     { "ask-pw",	  T_ASK_PW,	       NEEDS_CLOSE|NEEDS_DATA },

     /* templates */
     { "template",        T_TEMPLATE,          NEEDS_CLOSE },
     { "objectclass",     T_OBJECTCLASS,       NEEDS_CLOSE|NEEDS_DATA },

     /* filters */
     { "filter",          T_FILTER,            NEEDS_CLOSE },
     { "ldapfilter",      T_LDAPFILTER,        NEEDS_CLOSE|NEEDS_DATA },
     { "servername",      T_SERVERNAME,        NEEDS_CLOSE|NEEDS_DATA },

     { "", 0 }
};


struct tokenlist token_searchargument[] = {
     { SEARCHARG_BEGINS_WITH, "Begins with" },
     { SEARCHARG_ENDS_WITH,   "Ends with" },
     { SEARCHARG_CONTAINS,    "Contains" },
     { SEARCHARG_EQUALS,      "Equals" },
     { 0, "" }
};

struct tokenlist token_bindtype[] = {
     { BINDTYPE_SIMPLE,    "Simple" },
     { BINDTYPE_KERBEROS,  "Kerberos"},
     { BINDTYPE_SASL,      "SASL"},
     { 0, "" }
};


struct tokenlist token_ldifformat[] = {
     { LDIF_UMICH,         "UMich" },
     { LDIF_V1,            "Version1" },
     { 0, "" }
};


/* malloc ldapserver struct, plug into list */
struct ldapserver *new_ldapserver(void)
{
     struct ldapserver *server, *newserver;

     newserver = MALLOC(sizeof(struct ldapserver), "struct ldapserver");

     if(config.ldapservers) {
	  server = config.ldapservers;
	  while(server->next)
	       server = server->next;
	  server->next = newserver;
     }
     else
	  config.ldapservers = newserver;
	  
     return(newserver);
}


/* malloc and init gq_template struct */
struct gq_template *new_template(void)
{
     struct gq_template *tmpl;

     tmpl = MALLOC(sizeof(struct gq_template), "struct gq_template");
     memset(tmpl, 0, sizeof(struct gq_template));
     tmpl->name[0] = '\0';
     tmpl->objectclasses = NULL;

     config.templates = g_list_append(config.templates, tmpl);

     return(tmpl);
}


/* malloc and init gq_template struct */
struct gq_filter *new_filter(void)
{
     struct gq_filter *filter;

     filter = MALLOC(sizeof(struct gq_filter), "struct gq_filter");
     memset(filter, 0, sizeof(struct gq_filter));
     filter->name[0] = '\0';
     filter->ldapfilter[0] = '\0';
     filter->servername[0] = '\0';
     filter->basedn[0] = '\0';

     return(filter);
}


/* bloody hell */
void delete_ldapserver(struct ldapserver *dserver)
{
     struct ldapserver *server, *prevserver;

     server = config.ldapservers;

     if(server == dserver)
	  config.ldapservers = dserver->next;
     else {
	  prevserver = server;
	  while(server && server != dserver) {
	       prevserver = server;
	       server = server->next;
	  }

	  if(!server) {
	       fprintf(stderr,
		       "delete_ldapserver failed: unable to locate server\n");
	       return;
	  }

	  if(prevserver == config.ldapservers)
	       config.ldapservers = server->next;
	  else
	       prevserver->next = server->next;
     }

     FREE(server, "struct ldapserver");

}


char *homedir(void)
{
     struct passwd *pwent;
     char *result;
   
     pwent = getpwuid(getuid());
     if(pwent && pwent->pw_dir)
	  result = strdup(pwent->pw_dir);
     else
	  result = NULL;
     endpwent();

     return result;
}


void config_skip_whitespace(struct configfile *f)
{
     char c;

     while( (c = f->rcfile[f->p]) )
	  switch(c) {
	  case '\n':
	       f->line++;
	  case ' ':
	  case '\t':
	       f->p++;
	       break;
	  default:
	       return;
	  }

}


/* takes 'true' or 'false', returns false if parse error */
int config_get_bool(struct configfile *f)
{
     char errstr[256];

     if(!strcasecmp(f->cur_string, "true"))
	  return(1);
     else if(!strcasecmp(f->cur_string, "false"))
	  return(0);

     snprintf(errstr, sizeof(errstr),
	      _("line %d: boolean value '%s' should be 'True' "
		"or 'False'"), f->line, f->cur_string);
     error_push(f->err_context, errstr);

     return(0);
}


void config_id_string(struct configfile *f)
{
     int t;

     f->cur_token = 0;

     if(!strncasecmp(f->cur_string, "!--", 3)) {
	  f->cur_token = T_COMMENT;
     }
     else if(!strncasecmp(f->cur_string, "?xml", 4)) {
	  f->cur_token = T_XML_VERSION;
     }
     else {
	  for(t = 0; configwords[t].token; t++) {
	       if(!strcasecmp(f->cur_string, configwords[t].attribute)) {
		    f->cur_token = configwords[t].token;
		    f->cur_token_flags = configwords[t].flags;
		    break;
	       }
	  }
     }

}


void config_get_token(struct configfile *f)
{
     int i;
     char c;

     i = 0;
     f->cur_entity_type = IGNORE;

     if(f->status == TOP)
	  config_skip_whitespace(f);

     switch(f->rcfile[f->p]) {
     case '<':
	  f->p++;
	  if(f->rcfile[f->p] == '/') {
	       f->p++;
	       f->cur_entity_type = CLOSE;
	  }
	  else
	       f->cur_entity_type = OPEN;

	  while(f->rcfile[f->p] != '>'
		&& i < MAX_ENTITY_LEN) {
	       f->cur_string[i++] = f->rcfile[f->p++];
	  }
	  f->cur_string[i] = '\0';
	  f->p++;

	  config_id_string(f);
	  break;

     default:
	  f->cur_token = T_DATA;
	  while( (c = f->rcfile[f->p]) && c != '<' && c != '>'
		 && i < MAX_DATA_LEN) {
	       /* escape next char */
	       if(c == '\\') {
		    if(f->rcfile[++(f->p)]) {
			 c = f->rcfile[f->p];
		    }
	       }
	       f->cur_string[i++] = c;
	       if(c == '\n')
		    f->line++;
	       f->p++;
	  }
	  f->cur_string[i++] = '\0';
	  break;
     }

}


void process_rcfile(char *rcfile)
{
     struct configfile *f;
     char errstr[256];

     /* DEBUG
	int i;
     */

     f = MALLOC(sizeof(struct configfile), "struct configfile");
     f->rcfile = rcfile;
     f->p = 0;
     f->line = 0;
     f->status = TOP;
     f->sp = 1;
     f->cur_ldapserver = NULL;
     f->cur_template = NULL;
     f->err_context = error_new_context(_("Error parsing configfile"));

     while(f->rcfile[f->p] != '\0') {
	  do {
	       config_get_token(f);
	  } while (f->cur_string[0] == '\n');

	  /* allow for whitespace at end of file */
	  if(strlen(f->cur_string) == 0)
	       continue;


	  if(f->cur_token == T_DATA) {
	       /* FIXME this is ugly */
	       if(f->sp < 3) {
		    snprintf(errstr, sizeof(errstr), 
			     _("line %d: unmarked data '%s'"),
			     f->line, f->cur_string);
		    error_push(f->err_context, errstr);
		    break;
	       }

	       /* FIXME ugly too */
	       if(f->stack[f->sp - 2] == T_GQ_CONFIG) {
		    /* global settings */
		    switch(f->stack[f->sp - 1]) {
		    case T_CONFIRM_MOD:
			 config.confirm_mod = config_get_bool(f);
			 break;
		    case T_SEARCH_ARGUMENT:
			 /* compatibility with older versions of gq -- accept
			    token number as well */
			 if(isdigit( (int) f->cur_string[0]))
			      config.search_argument = atoi(f->cur_string);
			 else
			      config.search_argument = tokenize(token_searchargument,
								  f->cur_string);
			 break;
		    case T_SHOWDN:
			 config.showdn = config_get_bool(f);
			 break;
		    case T_SHOWOC:
			 config.showoc = config_get_bool(f);
			 break;
		    case T_SHOW_RDN_ONLY:
			 config.show_rdn_only = config_get_bool(f);
			 break;
		    case T_SORT_SEARCH:
			 config.sort_search = config_get_bool(f);
			 break;
		    case T_SORT_BROWSE:
			 config.sort_browse = config_get_bool(f);
			 break;
		    case T_LDIF_FORMAT:
			 config.ldifformat = tokenize(token_ldifformat, f->cur_string);
			 break;
		    case T_SCHEMASERVER:
			 strncpy(config.schemaserver, f->cur_string, MAX_SERVERNAME_LEN - 1);
			 break;
		    }
	       }
	       else {
		    /* FIXME groan */
		    if(f->sp > 2 && f->stack[f->sp - 2] == T_LDAPSERVER) {
			 /* filling in an <ldapserver> struct? */
			 switch(f->stack[f->sp - 1]) {
			 case T_NAME:
			      strncpy(f->cur_ldapserver->name, f->cur_string,
				      sizeof(f->cur_ldapserver->name) - 1);
			      break;
			 case T_LDAPHOST:
			      strncpy(f->cur_ldapserver->ldaphost, f->cur_string,
				     sizeof(f->cur_ldapserver->ldaphost) - 1);
			      break;
			 case T_LDAPPORT:
			      f->cur_ldapserver->ldapport = atoi(f->cur_string);
			      break;
			 case T_BASEDN:
			      strncpy(f->cur_ldapserver->basedn, f->cur_string,
				      sizeof(f->cur_ldapserver->basedn) - 1);
			      break;
			 case T_BINDDN:
			      strncpy(f->cur_ldapserver->binddn, f->cur_string,
				      sizeof(f->cur_ldapserver->binddn) - 1);
			      break;
			 case T_BINDPW:
			      strncpy(f->cur_ldapserver->bindpw, f->cur_string,
				      sizeof(f->cur_ldapserver->bindpw) - 1);
			      break;
			 case T_BINDTYPE:
			      f->cur_ldapserver->bindtype = tokenize(token_bindtype, f->cur_string);
			 case T_SEARCHATTR:
			      strncpy(f->cur_ldapserver->searchattr, f->cur_string,
				      sizeof(f->cur_ldapserver->searchattr) - 1);
			      break;
			 case T_MAXENTRIES:
			      f->cur_ldapserver->maxentries = atoi(f->cur_string);
			      break;
			 case T_CACHECONN:
			      f->cur_ldapserver->cacheconn = config_get_bool(f);
			      break;
			 case T_ENABLETLS:
			      f->cur_ldapserver->enabletls = config_get_bool(f);
			      break;
			 case T_LOCAL_CACHE_TIMEOUT:
			      f->cur_ldapserver->local_cache_timeout = atoi(f->cur_string);
			      break;
			 case T_ASK_PW:
			      f->cur_ldapserver->ask_pw = config_get_bool(f);
			      break;

			 }

		    }
		    else if(f->sp > 2 && f->stack[f->sp - 2] == T_TEMPLATE) {
			 /* fill in gq_template struct */
			 switch(f->stack[f->sp - 1]) {
			 case T_NAME:
			      strncpy(f->cur_template->name, f->cur_string, 
				      sizeof(f->cur_template->name) - 1);
			      break;
			 case T_OBJECTCLASS:
			      f->cur_template->objectclasses =
				   g_list_append(f->cur_template->objectclasses, strdup(f->cur_string));
			      break;
			 }
		    }
		    else if(f->sp > 2 && f->stack[f->sp - 2] == T_FILTER) {
			 switch(f->stack[f->sp - 1]) {
			 case T_NAME:
			      strncpy(f->cur_filter->name, f->cur_string, 
				      sizeof(f->cur_filter->name) - 1);
			      break;
			 case T_LDAPFILTER:
			      strncpy(f->cur_filter->ldapfilter, f->cur_string,
				      sizeof(f->cur_filter->ldapfilter) - 1);
			      break;
			 case T_SERVERNAME:
			      strncpy(f->cur_filter->servername, f->cur_string,
				      MAX_SERVERNAME_LEN - 1);
			      break;
			 case T_BASEDN:
			      strncpy(f->cur_filter->basedn, f->cur_string,
				      MAX_DN_LEN - 1);
			      break;
			 }
		    }
	       }


	  }
	  else {
	       /* token is an entity */
	       switch(f->cur_entity_type) {
	       case OPEN:
		    if(f->cur_token_flags & NEEDS_CLOSE)
			 f->stack[f->sp++] = f->cur_token;
		    if(f->cur_token_flags & NEEDS_DATA)
			 f->status = IN_TAG;

		    if(f->cur_token == T_LDAPSERVER) {
			 f->cur_ldapserver = new_ldapserver();
			 init_ldapserver(f->cur_ldapserver);
		    }
		    else if(f->cur_token == T_TEMPLATE) {
			 f->cur_template = new_template();
		    }
		    else if(f->cur_token == T_FILTER) {
			 f->cur_filter = new_filter();
		    }

		    break;
	       case CLOSE:
		    if(f->cur_token == T_FILTER) {
			 config.filters = g_list_append(config.filters, f->cur_filter);
		    }
		    if(f->cur_token_flags & NEEDS_CLOSE) {
			 if(f->stack[(f->sp) - 1] == f->cur_token) {
			      f->sp--;
			 }
			 else {
			      snprintf(errstr, sizeof(errstr),
				       _("line %d: can't close %s here\n"),
				       f->line, f->cur_string);
			      error_push(f->err_context, errstr);
			 }
		    }
		    if(f->cur_token_flags & NEEDS_DATA)
			 f->status = TOP;
		    break;
	       case IGNORE:
		    /* yeah well */
		    break;
	       }

	  }

/* DEBUG
	  printf("%2d: st %d token %2d entity-type %2d string %25s",
		 f->line, f->status, f->cur_token, f->cur_entity_type,
		 f->cur_string);
	  if(f->sp > 1) {
	       printf(" stack: ");
	       for(i = 1; i < f->sp; i++)
		    printf("%d ", f->stack[i]);
	  }
	  printf("\n");
*/

     }

     error_flush(f->err_context);
     FREE(f, "struct configfile");

}


char *filename_config(int context)
{
     static char rcpath[128];
     char *home;

     home = homedir();
     if(home == NULL) {
	  error_push(context, _("you have no home directory!"));
	  return(NULL);
     }

     snprintf(rcpath, sizeof(rcpath), "%s/%s", home, RCFILE);
     free(home);

     return(rcpath);
}


void load_config(void)
{
     FILE *rcfd;
     struct stat sfile;
     int load_context, rcsize;
     char *rcfile, *rcpath, errstr[MAX_ERRMSG_SIZE];

     load_context = error_new_context(_("Error loading configfile"));

     rcpath = filename_config(load_context);
     if(!rcpath) {
	  error_flush(load_context);
	  return;
     }

     /* no config file is fine */
     if(stat(rcpath, &sfile) == -1 || !sfile.st_size) {
	  error_flush(load_context);
	  return;
     }

     /* refuse to read config file if world readable (bind passwords) */
     if( (sfile.st_mode & ~S_IFMT) != 00600 ) {
	  snprintf(errstr, sizeof(errstr), 
		   _("%s is group and/or world readable.\n"
		     "This file can contain passwords in cleartext,\n"
		     "and should be mode 0600.\n\n"
		     "Continuing with default settings...\n"),
		   rcpath);
	  error_push(load_context, errstr);
	  error_flush(load_context);
	  return;
     }

     rcsize = sfile.st_size;
     rcfile = MALLOC(sfile.st_size + 1, "configfile buffer");
     if( (rcfd = fopen(rcpath, "r")) ) {
	  fread(rcfile, sfile.st_size, 1, rcfd);
	  rcfile[rcsize] = '\0';
	  fclose(rcfd);
     }

     process_rcfile(rcfile);

     error_flush(load_context);
     FREE(rcfile, "configfile buffer");

}


/* just a pretty printer */
void config_write(struct writeconfig *wc, char *string)
{
     int i;

     for(i = 0; i < wc->indent; i++)
	  fprintf(wc->outfile, "%s", CONFIG_INDENT_STRING);

     fprintf(wc->outfile, "%s", string);

}


void config_write_bool(struct writeconfig *wc, int value, char *entity)
{
     char outstr[128];

     snprintf(outstr, sizeof(outstr),
	      "<%s>%s</%s>\n", entity, value ? "True" : "False", entity);
     config_write(wc, outstr);

}


void config_write_int(struct writeconfig *wc, int value, char *entity)
{
     char outstr[128];

     snprintf(outstr, sizeof(outstr), 
	      "<%s>%d</%s>\n", entity, value, entity);
     config_write(wc, outstr);

}


void config_write_string(struct writeconfig *wc, char *value, char *entity)
{
     int i, p;
     char outstr[1024];

     snprintf(outstr, sizeof(outstr), "<%s>", entity);

     /* quick-and-dirty escape < and > */
     for(i = 0; value[i]; i++) {
	  if(value[i] == '<' || value[i] == '>') {
	       strcat(outstr, "\\"); /* FIXME: Buffer overflow probs  */
	  }
	  p = strlen(outstr);
	  outstr[p++] = value[i];
	  outstr[p] = '\0';
     }

     strcat(outstr, "</");
     strcat(outstr, entity);
     strcat(outstr, ">\n");
     config_write(wc, outstr);
}


void config_write_string_ne(struct writeconfig *wc, char *value, char *entity)
{

     if(value && strlen(value))
	  config_write_string(wc, value, entity);

}


void save_config(void)
{
     GList *templatelist, *oclist, *filterlist;
     struct ldapserver *server;
     struct gq_template *template;
     struct gq_filter *filter;
     struct writeconfig *wc;
     int write_context;
     char *rcpath, errstr[256];

     write_context = error_new_context(_("Error writing configfile"));

     server = config.ldapservers;
     wc = MALLOC(sizeof(struct writeconfig), "struct writeconfig");
     wc->indent = 0;

     rcpath = filename_config(write_context);
     if(!rcpath) {
	  error_flush(write_context);
	  return;
     }

     wc->outfile = fopen(rcpath, "w");
     if(!wc->outfile) {
	  snprintf(errstr, sizeof(errstr), 
		   _("unable to open %s for writing:\n%s\n"), rcpath,
		   strerror(errno));
	  error_push(write_context, errstr);
	  error_flush(write_context);
	  return;
     }
     fchmod(fileno(wc->outfile), S_IRUSR|S_IWUSR);

     config_write(wc, "<?xml version=\"1.0\" standalone=\"yes\"?>\n");
     config_write(wc, "<gq-config>\n\n");

     /* global settings */
     wc->indent++;
     config_write_bool(wc, config.confirm_mod, "confirm-mod");
     config_write_string(wc, detokenize(token_searchargument, config.search_argument),
		      "search-argument");
     config_write_bool(wc, config.showdn, "show-dn");
     config_write_bool(wc, config.showoc, "show-oc");
     config_write_bool(wc, config.show_rdn_only, "show-rdn-only");
     config_write_bool(wc, config.sort_search, "sort-search-mode");
     config_write_bool(wc, config.sort_browse, "sort-browse-mode");
     config_write_string(wc, detokenize(token_ldifformat, config.ldifformat), "ldif-format");
     if(strlen(config.schemaserver))
	  config_write_string(wc, config.schemaserver, "schema-server");
     config_write(wc, "\n");

     /* ldapservers */
     while(server) {
	  config_write(wc, "<ldapserver>\n");
	  wc->indent++;

	  config_write_string(wc, server->name, "name");
	  config_write_string(wc, server->ldaphost, "ldaphost");
	  config_write_int(wc, server->ldapport, "ldapport");
	  config_write_string_ne(wc, server->basedn, "basedn");
	  config_write_string_ne(wc, server->binddn, "binddn");
	  config_write_string_ne(wc, server->bindpw, "bindpw");
	  if(server->bindtype != DEFAULT_BINDTYPE)
	       config_write_string_ne(wc, detokenize(token_bindtype, server->bindtype), "bindtype");
	  config_write_string_ne(wc, server->searchattr, "search-attribute");
	  if(server->maxentries != DEFAULT_MAXENTRIES)
	       config_write_int(wc, server->maxentries, "maxentries");
	  if(server->cacheconn != DEFAULT_CACHECONN)
	       config_write_bool(wc, server->cacheconn, "cache-connection");
	  if(server->enabletls != DEFAULT_ENABLETLS)
	       config_write_bool(wc, server->enabletls, "enable-tls");
	  if(server->local_cache_timeout != DEFAULT_LOCAL_CACHE_TIMEOUT)
	       config_write_int(wc, server->local_cache_timeout, "local-cache-timeout");
	  if(server->ask_pw != DEFAULT_ASK_PW)
	       config_write_bool(wc, server->ask_pw, "ask-pw");

	  wc->indent--;
	  config_write(wc, "</ldapserver>\n\n");

	  server = server->next;
     }

     /* templates */
     templatelist = config.templates;
     while(templatelist) {
	  template = (struct gq_template *) templatelist->data;
	  config_write(wc, "<template>\n");
	  wc->indent++;

	  config_write_string(wc, template->name, "name");
	  oclist = template->objectclasses;
	  while(oclist) {
	       config_write_string(wc, (char *) oclist->data, "objectclass");
	       oclist = oclist->next;
	  }

	  wc->indent--;
	  config_write(wc, "</template>\n\n");

	  templatelist = templatelist->next;
     }

     /* filters */
     filterlist = config.filters;
     while(filterlist) {
	  filter = (struct gq_filter *) filterlist->data;
	  config_write(wc, "<filter>\n");
	  wc->indent++;

	  config_write_string(wc, filter->name, "name");
	  config_write_string(wc, filter->ldapfilter, "ldapfilter");
	  if(filter->servername[0])
	       config_write_string(wc, filter->servername, "servername");
	  if(filter->basedn[0])
	       config_write_string(wc, filter->basedn, "basedn");

	  wc->indent--;
	  config_write(wc, "</filter>\n\n");

	  filterlist = filterlist->next;
     }

     wc->indent--;
     config_write(wc, "</gq-config>\n");

     fclose(wc->outfile);
     FREE(wc, "struct writeconfig");

     error_flush(write_context);

}


void init_ldapserver(struct ldapserver *server)
{
     /* intitialize everything to nul-bytes first. This should help
	when using strncpy to put stuff into the char arrays */
     memset(server, 0, sizeof(struct ldapserver));

     server->name[0] = '\0';
     server->ldaphost[0] = '\0';
     server->ldapport = 389;
     server->basedn[0] = '\0';
     server->binddn[0] = '\0';
     server->bindpw[0] = '\0';
     server->enteredpw[0] = '\0';
     server->bindtype = DEFAULT_BINDTYPE;
     strncpy(server->searchattr, DEFAULT_SEARCHATTR, MAX_ATTR_LEN - 1);
     server->maxentries = DEFAULT_MAXENTRIES;
     server->cacheconn = DEFAULT_CACHECONN;
     server->enabletls = DEFAULT_ENABLETLS;
     server->local_cache_timeout = DEFAULT_LOCAL_CACHE_TIMEOUT;
     server->ask_pw = DEFAULT_ASK_PW;
     server->connection = NULL;
     server->missing_closes = 0;
     server->ss = NULL;
     server->flags = 0;
     server->next = NULL;
     server->version = LDAP_VERSION2;
     server->server_down = 0;
}


void init_config(void)
{
     struct ldapserver *default_server;

     config.ldapservers = NULL;
     config.templates = NULL;
     config.filters = NULL;
     config.confirm_mod = 0;
     config.search_argument = SEARCHARG_BEGINS_WITH;
     config.showdn = DEFAULT_SHOWDN;
     config.showoc = DEFAULT_SHOWOC;
     config.show_rdn_only = DEFAULT_SHOW_RDN_ONLY;
     config.sort_search = 0;
     config.sort_browse = 0;
     config.ldifformat = DEFAULT_LDIFFORMAT;
     config.schemaserver[0] = '\0';

     load_config();
     if(!config.ldapservers) {
	  /* no ldapserver defined in configfile */
	  default_server = new_ldapserver();
	  init_ldapserver(default_server);
	  /* here we are happy with strcpy - what we copy is
	     definitely short enough */
	  strcpy(default_server->name, "localhost"); /* Flawfinder: ignore */
	  strcpy(default_server->ldaphost, "localhost"); /* Flawfinder: ignore */
	  strcpy(default_server->searchattr, "cn");/* Flawfinder: ignore */
	  default_server->ldapport = 389;
	  config.ldapservers = default_server;
     }

}

/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */
