/*
    GQ -- a GTK-based LDAP client is
    Copyright (C) 1998-2001 Bert Vermeulen
    
    This file (browse-dnd.c) is
    Copyright (C) 2002 by Peter Stamfest <peter@stamfest.at>

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: browse-dnd.c,v 1.6 2002/06/18 22:07:14 stamfest Exp $ */

/*  #define DEBUG_DND */
/*  typedef unsigned long ulong; */

#include "config.h"
#ifdef BROWSER_DND

#include <stdio.h>
#include <string.h>

#include <gtk/gtk.h>

#include "browse.h" 
#include "browse-dnd.h" 

#include "errorchain.h"
#include "ldapops.h"
#include "ldif.h"
#include "util.h" 
#include "i18n.h" 

typedef struct _dnd_refresh {
     GtkCTree *tree;
     struct ldapserver *server;
     int options;
     char *dn;
} dnd_refresh;

static GList *dnd_refresh_list = NULL;


static dnd_refresh *new_dnd_refresh_dn(GtkCTree *tree, 
				       struct ldapserver *server,
				       char *dn,
				       int options)
{
     dnd_refresh *dr = g_malloc(sizeof(dnd_refresh));

     dr->tree = tree;
     dr->options = options;
     dr->server = server;

     if (dn)
	  dr->dn = g_strdup(dn);
     else 
	  dr->dn = NULL;

     return dr;
}

static dnd_refresh *new_dnd_refresh_node(GtkCTree *tree, 
					 GtkCTreeNode *node,
					 int options)
{
     dn_browse_entry *entry;
     entry = (dn_browse_entry *) gtk_ctree_node_get_row_data(tree, node);

     if (IS_DN_ENTRY(entry)) {
	  return new_dnd_refresh_dn(tree, server_from_node(tree, node),
				    entry->dn, options);
     } else {
	  return new_dnd_refresh_dn(tree, server_from_node(tree, node),
				    NULL, options);
     }
}

static void dnd_refresh_free(dnd_refresh *dr) 
{
     if (dr) {
	  if (dr->dn) g_free(dr->dn);
	  g_free(dr);
     }
}

#if 0
static gboolean drag_drop(GtkWidget *ctreeroot,
			  GdkDragContext *drag_context,
			  gint x,
			  gint y,
			  guint time,
			  gpointer user_data)
{
     int rc, row, column;
     GNode *node;
     GtkCTreeNode *ctree_node;
     struct node_entry *target_entry;
     int answer;

#ifdef DEBUG_DND
     printf("dropped %08lx at x=%d, y=%d\n", (ulong) drag_context, x, y);
#endif

     rc = gtk_clist_get_selection_info(GTK_CLIST(ctreeroot), x, y,
				       &row, &column);
     if (rc == 0) return FALSE;
     
     ctree_node = gtk_ctree_node_nth(GTK_CTREE(ctreeroot), row);
     if (ctree_node == NULL) return FALSE;

     node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot),
						  GTK_CTREE_NODE(ctree_node));
     if (node == NULL) return FALSE;
     if ((target_entry = (struct node_entry *) node->data) == NULL) 
	  return FALSE;

     if (target_entry->status & IS_SERVER) return FALSE;

     answer = question_popup(_("Do you really want to move this entry?"),
			     _("Do you really want to move this entry?"));
     if (!answer) return FALSE;

     /* dropping OK */

     return TRUE;
}
#endif

typedef struct _motion_timer_data {
     GtkAdjustment *adj;
     gfloat change;
     int timer_id;
     int oldinterval, newinterval;
} motion_timer_data;

/* We want the CTree to scroll up or down during dragging if we get
   close to its border. We use a time for this. 

   Some Notes: 
   * The closer we are to the border the faster we scroll.
   * The timer gets started by a drag motion callback
 */
static gboolean drag_motion_timer(motion_timer_data *td) {
     gfloat nval = td->adj->value + td->change;
     gboolean rc = TRUE;

     if (td->oldinterval != td->newinterval) {
	  rc = FALSE; /* remove THIS timer */
	  td->timer_id = gtk_timeout_add(td->newinterval, 
					 (GtkFunction) drag_motion_timer,
					 td);
	  td->oldinterval = td->newinterval;  
     }

     if (nval < td->adj->lower) nval = td->adj->lower;
     if (nval > td->adj->upper - td->adj->page_size) nval = td->adj->upper - td->adj->page_size;

     gtk_adjustment_set_value(td->adj, nval);
     return rc;
}

/* stops/removes the scrolling timer of the ctree if the CTree get
   destroyed */
static void remove_drag_motion_timer(motion_timer_data *td) {
     gtk_timeout_remove(td->timer_id);
     g_free(td);
}

/* implements changing scroll speeds depending on distance to border */
static int find_scroll_interval(int distance) {
     static int intervalmap[] = { 10, 10, 20, 20, 25, 25, 50, 50, 
				  67, 100, 200 };

     if (distance >= (sizeof(intervalmap) / sizeof(intervalmap[0])) ||
	 distance < 0) {
	  return 200;
     } 
     return intervalmap[distance];
}

/* During dragging: check if we get close to the top or bottom
   borders. If we do start scrolling using a timer */
static gboolean drag_motion(GtkWidget *ctreeroot,
			    GdkDragContext *drag_context,
			    gint x,
			    gint y,
			    guint time,
			    gpointer user_data)
{
     motion_timer_data *td = gtk_object_get_data(GTK_OBJECT(ctreeroot),
						 "scroll-timer-data");

     /* 

	HACK Alert

	clist_window_height is internal to CList AND doesn't count the
	highlighting border... (thus the + 2) 

	HACK Alert

     */

     int height = GTK_CLIST(ctreeroot)->clist_window_height + 2;

     if (y < 10) {
	  if (!td) {
	       td = g_malloc(sizeof(motion_timer_data));
	       td->adj = gtk_clist_get_vadjustment(GTK_CLIST(ctreeroot));
	       td->change = -td->adj->step_increment;
	       td->oldinterval = td->newinterval = find_scroll_interval(y);
	       td->timer_id = gtk_timeout_add(td->oldinterval, 
                                              (GtkFunction) drag_motion_timer,
					      td);

	       gtk_object_set_data_full(GTK_OBJECT(ctreeroot),
					"scroll-timer-data", td,
					(GtkDestroyNotify)remove_drag_motion_timer);
	  } else {
	       td->newinterval = find_scroll_interval(y);
	       td->change = -td->adj->step_increment;
	  }
     } else if (height - y < 10) {
	  if (!td) {
	       td = g_malloc(sizeof(motion_timer_data));
	       td->adj = gtk_clist_get_vadjustment(GTK_CLIST(ctreeroot));
	       td->change = td->adj->step_increment;
	       td->oldinterval = td->newinterval = find_scroll_interval(height - y);
	       td->timer_id = gtk_timeout_add(td->oldinterval, 
                                              (GtkFunction) drag_motion_timer,
					      td);

	       gtk_object_set_data_full(GTK_OBJECT(ctreeroot),
					"scroll-timer-data", td,
					(GtkDestroyNotify)remove_drag_motion_timer);
	  } else {
	       td->newinterval = find_scroll_interval(height - y);
	       td->change = td->adj->step_increment;
	  }
     } else {
	  if (td) {
	       gtk_object_remove_data(GTK_OBJECT(ctreeroot),
				      "scroll-timer-data");
	  }
     }
     
     return TRUE;
}

/* Cleanup function for the data in the drag and selection data hash */
static gboolean drag_and_select_free_elem(gpointer key,
					  gpointer value,
					  gpointer user_data)
{
     g_free(key);
     g_free(value);
     return TRUE;
}

/* Cleanup function for the drag and selection data hash */
static void drag_and_select_data_free(GHashTable *hash) 
{
     g_hash_table_foreach_remove(hash, drag_and_select_free_elem, NULL);
     g_hash_table_destroy(hash);
}

static void drag_begin(GtkWidget *ctreeroot,
		       GdkDragContext *drag_context,
		       GHashTable *hash) 
{
     /* store data we began dragging with for drag_data_get later on */
     GtkCTreeNode *ctree_node;
     struct ldapserver *server;
     browse_entry *entry;
     GHashTable *seldata;
     char cbuf[20];

     ctree_node = g_hash_table_lookup(hash, "tree-row-selected");
/*       node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), */
/*  						  GTK_CTREE_NODE(ctree_node)); */
     server = server_from_node(GTK_CTREE(ctreeroot), ctree_node);
     entry = (browse_entry *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot),
							  ctree_node);

#ifdef DEBUG_DND
     printf("drag_begin start node=%08lx entry=%08lx\n",
	    (ulong) ctree_node,
	    (ulong) entry);
#endif

     if (IS_DN_ENTRY(entry)) {
	  dn_browse_entry *dn_entry = (dn_browse_entry *) entry;

	  seldata = g_hash_table_new(g_str_hash, g_str_equal);
	  g_hash_table_insert(seldata, g_strdup("dn"), g_strdup(dn_entry->dn));
	  g_hash_table_insert(seldata, g_strdup("nickname"), 
			      g_strdup(server->name));
	  g_hash_table_insert(seldata, g_strdup("server"), 
			      g_strdup(server->ldaphost));
	  snprintf(cbuf, sizeof(cbuf), "%d", server->ldapport);
	  g_hash_table_insert(seldata, g_strdup("port"), g_strdup(cbuf));
	  
	  /* dragging is always done recusively (for now) */
	  g_hash_table_insert(seldata, g_strdup("recursively"), g_strdup("TRUE"));
	  
	  gtk_object_set_data_full(GTK_OBJECT(ctreeroot), 
				   "drag-and-selection-data", 
				   seldata,
				   (GtkDestroyNotify) drag_and_select_data_free);

#ifdef DEBUG_DND
	  printf("drag_begin %08lx %08lx\n", (ulong) ctreeroot,
		 (ulong) seldata);
#endif
     }
}

static int dnd_refresh_sort_func(const dnd_refresh *a, const dnd_refresh *b)
{
     if (a == NULL || a->dn == NULL) return -1;
     if (b == NULL || b->dn == NULL) return 1;

     return strlen(a->dn) - strlen(b->dn);
}

static void do_refresh() {
     GList *l;
     dnd_refresh *at_end = NULL;

     /* do any queued refresh operations */
     
     /* sort refresh operations to work top-down. This way at the end
        everything will be visible */
     dnd_refresh_list = g_list_sort(dnd_refresh_list,
				    (GCompareFunc) dnd_refresh_sort_func);

     for (l = g_list_first(dnd_refresh_list) ; l ; l = g_list_next(l)) {
	  dnd_refresh *dr = l->data;
	  if (dr->dn == NULL) continue;

	  refresh_subtree_with_options(dr->tree, 
				       tree_node_from_server_dn(dr->tree, 
								dr->server,
								dr->dn),
				       dr->options); 

	  if (dr->options & SELECT_AT_END) {
	       at_end = dr;
	  }
     }

     if (at_end) {
	  GtkCTreeNode *node = tree_node_from_server_dn(at_end->tree, 
							at_end->server,
							at_end->dn);
/*  	  show_dn(at_end->tree, */


/*  		  , at_end->dn); */
	  if (node) 
	       gtk_ctree_select(at_end->tree, node);
     }
     
     for (l = g_list_first(dnd_refresh_list) ; l ; l = g_list_next(l)) {
	  dnd_refresh_free(l->data);
     }
     g_list_free(dnd_refresh_list);
     dnd_refresh_list = NULL;
}


static void drag_end(GtkWidget *ctreeroot,
		     GdkDragContext *drag_context,
		     gpointer user_data) 
{
#ifdef DEBUG_DND
     printf("drag_end %08lx\n", (ulong) drag_context);
#endif
     /* kill any motion timer ... */
     /* NOTE: This would not work for inter-widget drag/drop as the
        motion-related widget would be different from the one for
        which drag_end gets called */

     gtk_object_remove_data(GTK_OBJECT(ctreeroot),
			    "scroll-timer-data"); 

     gtk_object_remove_data(GTK_OBJECT(ctreeroot), 
			    "drag-and-selection-data");

     do_refresh();
}


static void drag_selection_data_prepare(gpointer key,
					gpointer value,
					GByteArray *buf)
{
     char nul = 0;
     g_byte_array_append(buf, key, strlen(key));
     g_byte_array_append(buf, "=", 1);
     g_byte_array_append(buf, value, strlen(value));
     g_byte_array_append(buf, &nul, 1); 
}

static void drag_data_get(GtkWidget *ctreeroot,
			  GdkDragContext *drag_context,
			  GtkSelectionData *data,
			  guint info,
			  guint time,
			  GHashTable *hash) 
{
     GHashTable *selhash = gtk_object_get_data(GTK_OBJECT(ctreeroot), 
					       "drag-and-selection-data");
     GByteArray *buf;
     char nul = 0;

     if (!selhash) {
	  statusbar_msg(_("Could not find data to drag - internal error"));
	  return;
     }
     buf = g_byte_array_new();

     g_hash_table_foreach(selhash, (GHFunc) drag_selection_data_prepare, buf);
     g_byte_array_append(buf, &nul, 1); 

#ifdef DEBUG_DND
     printf("drag_data_get ctx=%08lx data=%08lx\n", (ulong) drag_context,
	    (ulong) data);
#endif

     gtk_selection_data_set(data,
			    gdk_atom_intern("gq-browse-ctree", FALSE),
			    8, /* character data */
			    buf->data,
			    buf->len);

     g_byte_array_free(buf, TRUE);

#ifdef DEBUG_DND
     printf("data->data=%08lx data=%08lxx\n",
	    (ulong)(data->data), (ulong) data);
#endif
}

static void move_progress(const char *from_dn,
			  const char *to_dn,
			  const char *new_dn) 
{
     int len = strlen(from_dn) + strlen(to_dn) + 80;
     char *msg = g_malloc(len);
/*       snprintf(msg, len, "Moved %s to %s", from_dn, to_dn); */
     snprintf(msg, len, _("Created %s"), new_dn);
     statusbar_msg(msg);
     g_free(msg);
}

static GHashTable *drag_selection_data_unpack(const char *data, int len)
{
     GHashTable *selhash = NULL;
     const char *d = data, *eq;
     char *key, *value;
     int l, n;

     if (data == NULL) return NULL;

     selhash = g_hash_table_new(g_str_hash, g_str_equal);

     for ( d = data ; (l = strlen(d)) ; d = d + l + 1 ) {
	  if ((eq = strchr(d, '=')) != NULL) {
	       n = eq - d;

	       key = g_malloc(n + 1);
	       strncpy(key, d, n);
	       key[n] = 0;
	       
	       n = l - 1 - n;
	       eq++;
	       
	       value = g_malloc(n + 1);
	       strncpy(value, eq, n);
	       value[n] = 0;

#ifdef DEBUG_DND
	       printf("key=%s value=%s\n", key, value);
#endif
	       g_hash_table_insert(selhash, key, value);
	  } 
     }
     return selhash;
}


static void do_move_after_reception(GtkWidget *ctreeroot,
				    GHashTable *selhash,
				    GtkCTreeNode *target_ctree_node,
				    dn_browse_entry *target_entry,
				    struct ldapserver *target_server,
				    int flags)
{
     GtkCTreeNode *ctree_node;
     char *dn = NULL;
     struct ldapserver *source_server;
     char *newdn = NULL;

#ifdef DEBUG_DND
     printf("do_move_after_reception selhash=%08lx server=%s dn=%s\n",
	    (ulong) selhash, (char*)g_hash_table_lookup(selhash, "nickname"), dn);
#endif

     source_server = server_by_name(g_hash_table_lookup(selhash, "nickname"));
     if (source_server == NULL) {
	  statusbar_msg(_("Could not find source server by its nickname!"));
	  return;
     }

     dn = g_hash_table_lookup(selhash, "dn");
     
     /* Avoid infinite recursion when copying/dragging an entry on top
        of itself. */
     /* NOTE: THIS WILL NOT WORK IF MOVING ENTRIES BETWEEN DIFFERENT
        "servers" THAT ARE ACTUALLY POINTING TO THE SAME LDAP
        DIRECTORY!!!! - INFINITE RECURSION ALERT! - FIXME */
     /* Note: Pointer comparison, as we only store information about
	servers once.  */
     

     if ((source_server == target_server) &&
	 ( strcasecmp(dn, target_entry->dn) == 0)) {
	  statusbar_msg(_("Cannot move/copy entry onto itself!"));
          error_popup(_("Cannot move/copy entry onto itself!"), 
		      _("Cannot move/copy entry onto itself!"));

	  return;
     }
     
     if ((flags & MOVE_RECURSIVELY) &&
	 (source_server == target_server) &&
	 is_ancestor(target_entry->dn, dn)) {
	  statusbar_msg(_("Cannot recursively move/copy entry onto or below itself!"));
          error_popup(_("Cannot recursively move/copy entry onto or below itself!"), 
		      _("Cannot recursively move/copy entry onto or below itself!"));
	  
	  return;
     }
     


     if (IS_DN_ENTRY(target_entry)) {
	  newdn = move_entry(dn, source_server, 
			     target_entry->dn, target_server,
			     flags, move_progress);
	  
	  /* register that we have to refresh the target node */
	  dnd_refresh_list =
	       g_list_append(dnd_refresh_list,
			     new_dnd_refresh_dn(GTK_CTREE(ctreeroot),
						target_server, 
						target_entry->dn,
						REFRESH_FORCE_EXPAND)
			     );

	  /* arrange for selecting the final node */

	  dnd_refresh_list =
	       g_list_append(dnd_refresh_list,
			     new_dnd_refresh_dn(GTK_CTREE(ctreeroot),
						target_server, 
						newdn,
						SELECT_AT_END)
			     );

/*  	  refresh_subtree_with_options(GTK_CTREE(ctreeroot), target_ctree_node, */
/*  				       REFRESH_FORCE_EXPAND); */
	  
	  if (newdn) {
	       ctree_node = node_from_dn(GTK_CTREE(ctreeroot),
					 target_ctree_node, newdn);
	       if (ctree_node)
		    gtk_ctree_select(GTK_CTREE(ctreeroot),
				     ctree_node);
	       

#ifdef DEBUG_DND
	       printf("moved %s below %s, became %s\n",
		      dn,
		      target_entry->dn,
		      newdn);
#endif
	       g_free(newdn);
	  }
     }
}

/* handles data receiving and does the actual move */
static void drag_data_received(GtkWidget *ctreeroot,
			       GdkDragContext *drag_context,
			       gint x,
			       gint y,
			       GtkSelectionData *data,
			       guint info,
			       guint time,
			       gpointer user_data)
{
     GtkCTreeNode *ctree_node;
     struct ldapserver *target_server;
     int rc, row, column;
     GHashTable *selhash = NULL;
     dn_browse_entry *target_entry;
     int answer;

     if (data == NULL) {  /* PSt: I've seen this happen */
	  statusbar_msg(_("No selection data available"));
	  return;
     }

#ifdef DEBUG_DND
     printf("drag_data_received ctx=%08lx seldata=%08lx\n",
	    (ulong) drag_context, (ulong) data);
#endif

     rc = gtk_clist_get_selection_info(GTK_CLIST(ctreeroot), x, y,
				       &row, &column);
     if (rc == 0) return;
     
     ctree_node = gtk_ctree_node_nth(GTK_CTREE(ctreeroot), row);
     if (ctree_node == NULL) return;

     /* NOTE: type has not yet been checked !!! */
     target_entry = (dn_browse_entry *) 
	  gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot),
				      GTK_CTREE_NODE(ctree_node));

     if (! IS_DN_ENTRY(target_entry)) return;
     target_server = server_from_node(GTK_CTREE(ctreeroot), 
				      GTK_CTREE_NODE(ctree_node));

     /* For now we allow recursive moves, as we _believe_ it
	works even with a limited number of answers from
	servers... */

     answer = 
	  question_popup(_("Do you really want to move this entry?"),
			 _("Do you really want to move this entry recursively?\n"
			   "Note that you should only do this if you have\n"
			   "access to ALL attributes of the object to move\n"
			   "(the same holds for objects below it) as the\n"
			   "original object(s) WILL BE REMOVED!\n\n"
			   "USE AT YOUR OWN RISK!"));
     
     if (answer) {
	  selhash = drag_selection_data_unpack(data->data, 
					       data->length);
	  
	  if (selhash) {
	       do_move_after_reception(ctreeroot, selhash, ctree_node,
				       target_entry, target_server, 
				       MOVE_CROSS_SERVER | MOVE_RECURSIVELY |
				       MOVE_DELETE_MOVED);
	       
	       drag_and_select_data_free(selhash);


	  }
     }
#ifdef DEBUG_DND
     printf("dragged to %d/%d\n", x, y);
#endif
}

/*

  do sender side "deletion" after a move - IMHO gtk is buggy in having
  trouble to communicate to the sending side if the dropping actually
  worked... However, I do not know if the dnd protocols actually work
  like this...

  Luckily, we do EVERYTHING in the drag_data_received handler (only
  possible because we only do in-widget dnd)

*/

static void drag_data_delete(GtkWidget *widget,
			     GdkDragContext *drag_context,
			     GHashTable *hash)
{
     GtkCTreeNode *ctree_node;
     char *server = NULL, *dn = NULL;

     GHashTable *selhash = gtk_object_get_data(GTK_OBJECT(widget), 
					       "drag-and-selection-data");

     server = g_hash_table_lookup(selhash, "nickname");
     dn = g_hash_table_lookup(selhash, "dn");

#ifdef DEBUG_DND
     printf("drag_data_delete ctx=%08lx suggested=%d action=%d server=%s dn=%s\n", 
	    (ulong) drag_context,
	    drag_context->suggested_action,
	    drag_context->action,
	    server, dn
	    );
#endif

     ctree_node = tree_node_from_server_dn(GTK_CTREE(widget),
					   server_by_name(server), dn);
     
     if (ctree_node) {
	  GtkCTreeNode *parent = GTK_CTREE_ROW(ctree_node)->parent;
  	  if (parent) {
	       /* register that we have to refresh the parent node */

	       dnd_refresh_list =
		    g_list_append(dnd_refresh_list,
				  new_dnd_refresh_node(GTK_CTREE(widget),
						       parent, 0)
				  );
/*    	       refresh_subtree_with_options(GTK_CTREE(widget), parent, 0); */
	  }
     }
#ifdef DEBUG_DND
     printf("drag_data_delete done\n");
#endif
     gtk_object_remove_data(GTK_OBJECT(widget), 
			    "drag-and-selection-data");
}

#if 0
gboolean compare_drag_func(GtkCTree *ctreeroot,
			   GtkCTreeNode *source_node,
			   GtkCTreeNode *new_parent,
			   GtkCTreeNode *new_sibling)
{
    printf("compare_drag_func\n");
    return TRUE;
}
#endif


void copy_entry(GtkWidget *widget, GHashTable *hash)
{
     GtkCTree *ctree;
     int have_sel;
     struct ldapserver *server;
     dn_browse_entry *entry;
     GtkCTreeNode *ctree_node;
     GHashTable *seldata;
     char cbuf[20];

     ctree = g_hash_table_lookup(hash, "ctreeroot");

     ctree_node = g_hash_table_lookup(hash, "tree-row-popped-up");
     entry = (dn_browse_entry *) gtk_ctree_node_get_row_data(ctree,
							     ctree_node);
     if (entry == NULL) return;
     if (!IS_DN_ENTRY(entry)) return;

     server = server_from_node(ctree, ctree_node);

     seldata = g_hash_table_new(g_str_hash, g_str_equal);
     g_hash_table_insert(seldata, g_strdup("dn"), g_strdup(entry->dn));
     g_hash_table_insert(seldata, g_strdup("nickname"), 
			 g_strdup(server->name));
     g_hash_table_insert(seldata, g_strdup("server"), 
			 g_strdup(server->ldaphost));
     snprintf(cbuf, sizeof(cbuf), "%d", server->ldapport);
     g_hash_table_insert(seldata, g_strdup("port"), g_strdup(cbuf));

     gtk_object_set_data_full(GTK_OBJECT(ctree), 
			      "drag-and-selection-data", 
			      seldata,
			      (GtkDestroyNotify) drag_and_select_data_free);

     have_sel = gtk_selection_owner_set(GTK_WIDGET(ctree),
					GDK_SELECTION_PRIMARY,
					GDK_CURRENT_TIME);

#ifdef DEBUG_DND
     printf("copy_entry %08lx %d\n", (ulong) ctree, have_sel);
#endif
}

void copy_entry_all(GtkWidget *widget, GHashTable *hash)
{
     GHashTable *selhash = NULL;
     GtkWidget *ctreeroot;

     copy_entry(widget, hash);

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");

     selhash = gtk_object_get_data(GTK_OBJECT(ctreeroot), 
				   "drag-and-selection-data");
     
     g_hash_table_insert(selhash, g_strdup("recursively"), g_strdup("TRUE"));

#ifdef DEBUG_DND
     printf("copy_entry_all %08lx\n", (ulong) widget);
#endif
}

static void get_selection_string(GtkWidget *ctree,
				 GtkSelectionData *data,
				 GHashTable *selhash)
{
     struct ldapserver *server = 
	  server_by_name(g_hash_table_lookup(selhash, "nickname"));
     char *dn = g_hash_table_lookup(selhash, "dn");

     LDAP *ld;
     int rc;

     LDAPMessage *msg = NULL, *e;
     GString *out = NULL;
     gboolean ok = FALSE;

#ifdef DEBUG_DND
     printf("get_selection_string\n");
#endif

     /* retrieve from server ... */

     if( (ld = open_connection(server)) == NULL) goto fail;

     rc = ldap_search_s(ld, dn, 
			g_hash_table_lookup(selhash, "recursively") ? LDAP_SCOPE_SUBTREE : LDAP_SCOPE_BASE,
			"objectClass=*", NULL, 0, &msg);

     if (rc == LDAP_SERVER_DOWN) {
	  server->server_down = 1;
	  goto fail;
     }
     if (rc != LDAP_SUCCESS) goto fail;

     out = g_string_new("");

     for(e = ldap_first_entry(ld, msg); e ; e = ldap_next_entry(ld, e)) {
	  ldif_entry_out(out, ld, e);
     }
     
     gtk_selection_data_set(data,
			    GDK_SELECTION_TYPE_STRING,
			    8, /* character data */
			    out->str,
			    out->len
			    );
     ok = TRUE;

     /* I hate to do it like this, but sometimes labels are such
        helpful beasts */
 fail:
     if (!ok) {
	  gtk_selection_data_set(data,
				 GDK_SELECTION_TYPE_STRING,
				 8, /* character data */
				 NULL,
				 0
				 );
     }

     if (out) g_string_free(out, TRUE);
     if (msg) ldap_msgfree(msg);
     close_connection(server, FALSE);
     return;
}

static void get_selection_gq(GtkWidget *ctree,
			     GtkSelectionData *data,
			     GHashTable *selhash)
{
     GByteArray *buf = g_byte_array_new();
     char nul = 0;

     g_hash_table_foreach(selhash, (GHFunc) drag_selection_data_prepare, buf);
     g_byte_array_append(buf, &nul, 1); 

#ifdef DEBUG_DND
     printf("get_selection_gq data=%08lx\n", (ulong) data);
#endif

     gtk_selection_data_set(data,
			    gdk_atom_intern("gq-browse-ctree", FALSE),
			    8, /* character data */
			    buf->data,
			    buf->len);

#ifdef DEBUG_DND
     printf("data->data=%08lx data=%08lxx\n",
	    (ulong)(data->data), (ulong) data);
#endif

     g_byte_array_free(buf, TRUE);
}

static void get_selection(GtkWidget *ctree,
			  GtkSelectionData *data,
			  guint info,
			  guint time,
			  GHashTable *hash)
{
     GHashTable *selhash = gtk_object_get_data(GTK_OBJECT(ctree), 
					       "drag-and-selection-data");
     if (selhash) {
	  if (data->target == gdk_atom_intern("gq-browse-ctree", FALSE)) {
	       get_selection_gq(ctree, data, selhash);
	  }
	  if (data->target == GDK_SELECTION_TYPE_STRING) {
	       get_selection_string(ctree, data, selhash);
	  }
     }
}

void paste_entry(GtkWidget *widget, GHashTable *hash)
{
     GtkWidget *ctree = g_hash_table_lookup(hash, "ctreeroot");
     
#ifdef DEBUG_DND
     printf("paste_entry %08lx\n", (ulong) ctree);
#endif
     gtk_selection_convert(ctree,
			   GDK_SELECTION_PRIMARY,
			   gdk_atom_intern("gq-browse-ctree", FALSE),
			   GDK_CURRENT_TIME);
     do_refresh();
}

/* gets fired after the gtk_selection_convert triggered through
   "paste" finished. Similar to when dragging stops. */
static void selection_received(GtkWidget *ctree,
			       GtkSelectionData *data,
			       guint time,
			       GHashTable *hash) 
{

     GtkCTreeNode *ctree_node;
     dn_browse_entry *target_entry;
     struct ldapserver *target_server;
     GtkCTree *ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     GHashTable *selhash = NULL;

#ifdef DEBUG_DND
     printf("selection_received seldata=%08lx\n",
	    (ulong) data);
#endif

     ctree_node = g_hash_table_lookup(hash, "tree-row-popped-up");
     target_entry = 
	  (dn_browse_entry *) gtk_ctree_node_get_row_data(ctreeroot,
							  ctree_node);
     if (target_entry == NULL) return;
     if (!IS_DN_ENTRY(target_entry)) return;
     
     target_server = server_from_node(ctreeroot, ctree_node);

     /* For now we allow recursive moves, as we _believe_ it
	works even with a limited number of answers from
	servers... */
     
     selhash = drag_selection_data_unpack(data->data, 
					  data->length);

     if (selhash) {
	  int flags = MOVE_CROSS_SERVER;
	  char *msg = NULL;
	  int answer;


	  if (g_hash_table_lookup(selhash, "recursively") != NULL) {
	       flags |= MOVE_RECURSIVELY;
	  }

	  if (flags & MOVE_RECURSIVELY) {
	       msg = _("Do you really want to paste this entry recursively?\n"
		       "Note that you might not be able to really copy everything\n" 
		       "in case you do not have access to ALL attributes of the\n"
		       "object(s) to paste (the same holds for objects below it).\n"
		       "USE AT YOUR OWN RISK!");
	  } else {
	       msg = _("Do you really want to paste this entry?\n"
		       "Note that you might not be able to really paste\n"
		       "the entire object if you do not have\n"
		       "access to ALL attributes of the object.\n");
	  }
	  
	  answer = 
	       question_popup(_("Do you really want to move this entry?"),
			      msg);
	  
	  if (answer) {
	       do_move_after_reception(GTK_WIDGET(ctreeroot),
				       selhash, ctree_node,
				       target_entry, target_server,
				       flags);
	  }
     }
}

void browse_dnd_setup(GtkWidget *ctreeroot, GHashTable *hash) 
{
     GtkTargetEntry *target_entry;

     /* signal for copying entries out of the tree */
     gtk_selection_add_target(GTK_WIDGET(ctreeroot),
			      GDK_SELECTION_PRIMARY,
			      GDK_SELECTION_TYPE_STRING,
			      1);
     gtk_selection_add_target(GTK_WIDGET(ctreeroot),
			      GDK_SELECTION_PRIMARY,
			      gdk_atom_intern("gq-browse-ctree", FALSE),
			      1);

     gtk_signal_connect(GTK_OBJECT(ctreeroot), 
			"selection-get",
			(GtkSignalFunc) get_selection,
			(gpointer) hash);
     gtk_signal_connect(GTK_OBJECT(ctreeroot), 
			"selection-received",
			(GtkSignalFunc) selection_received,
			(gpointer) hash);

     target_entry = (GtkTargetEntry*) g_malloc(sizeof(GtkTargetEntry));
     target_entry->target = "gq-browse-ctree";
     target_entry->flags = 0; /* GTK_TARGET_SAME_WIDGET; */
     target_entry->info = 0;

     gtk_drag_dest_set(ctreeroot, GTK_DEST_DEFAULT_ALL,
		       target_entry, 1,
		       GDK_ACTION_MOVE
		       );
     gtk_drag_source_set(ctreeroot, 
			 GDK_BUTTON1_MASK | GDK_BUTTON2_MASK,
			 target_entry, 1,
			 GDK_ACTION_MOVE
			 );

     gtk_clist_set_button_actions(GTK_CLIST(ctreeroot), 1, 
				  GTK_BUTTON_SELECTS | GTK_BUTTON_DRAGS | GTK_BUTTON_EXPANDS);

     gtk_signal_connect(GTK_OBJECT(ctreeroot), "drag-data-received",
			GTK_SIGNAL_FUNC(drag_data_received),
			(gpointer) NULL);

     gtk_signal_connect(GTK_OBJECT(ctreeroot), "drag-data-get",
			 GTK_SIGNAL_FUNC(drag_data_get),
			 (gpointer) hash);
     
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "drag-data-delete",
			 GTK_SIGNAL_FUNC(drag_data_delete),
			 (gpointer) hash);

     gtk_signal_connect(GTK_OBJECT(ctreeroot), "drag-motion",
			GTK_SIGNAL_FUNC(drag_motion),
			(gpointer) NULL);

     gtk_signal_connect(GTK_OBJECT(ctreeroot), "drag-begin",
			GTK_SIGNAL_FUNC(drag_begin),
			(gpointer) hash);


     /* HACK alert */
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "drag-end",
			GTK_SIGNAL_FUNC(drag_end),
			(gpointer) NULL);
#if 0
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "drag-drop",
			GTK_SIGNAL_FUNC(drag_drop),
			(gpointer) target_entry);
#endif
/*       gtk_ctree_set_drag_compare_func(GTK_CTREE(ctreeroot), */
/*  				     compare_drag_func); */

     g_free(target_entry);
     target_entry = NULL;
}

#endif /* BROWSER_DND */

/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */
