/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: util.c,v 1.41 2002/07/03 20:13:05 stamfest Exp $ */

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <pwd.h>

#include <config.h>

#include <lber.h>
#include <ldap.h>
#ifdef HAVE_LDAP_STR2OBJECTCLASS
#    include <ldap_schema.h>
#endif
#ifdef HAVE_SASL
#    include <sasl.h>
#endif


#include "common.h"
#include "util.h"
#include "configfile.h"
#include "errorchain.h"
#include "template.h"
#include "debug.h"
#include "schema.h"
#include "encode.h"
#include "i18n.h"
#include "mainwin.h"

#include "../icons/warning.xpm"

#define TRY_VERSION3 1

#ifdef HAVE_SASL
static int util_ldap_sasl_interact(LDAP *ld, unsigned flags, void *defaults, void *in)
{
     sasl_interact_t *interact;
     
     for (interact = in; interact->id != SASL_CB_LIST_END; interact++) {
	  if (interact->id == SASL_CB_USER) {
	       interact->result = strdup("");
	       interact->len = 0;
	  } else return LDAP_OTHER;
     }
     
     return LDAP_SUCCESS;
}
#endif

static int do_ldap_connect(LDAP **ld_out, struct ldapserver *server,
			   int open_context, int flags)
{
     LDAP *ld = NULL;
     char message[128];
     char *binddn = NULL, *bindpw = NULL;
     int rc = LDAP_SUCCESS;

     *ld_out = NULL;

     ld = ldap_init(server->ldaphost, server->ldapport); 
     server->server_down = 0;

     if(!ld) {
	  snprintf(message, sizeof(message),
		   _("Failed to initialize LDAP structure."));
	  error_push(open_context, message);
     }
     else {
	  server->version = LDAP_VERSION2;

#ifndef HAVE_OPENLDAP12
	  if (flags & TRY_VERSION3) {
	       /* try to use LDAP Version 3 */
	       
	       int version = LDAP_VERSION3;
	       if (ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION,
				   &version) == LDAP_OPT_SUCCESS) {
		    server->version = LDAP_VERSION3;
/*  	       } else { */
/*  		    error_push(open_context, message); */
/*  		    push_ldap_addl_error(ld, open_context); */
	       }
	  }
#endif

	  if (server->enabletls) {
#if defined(HAVE_TLS)
	       {
		    if (server->version != LDAP_VERSION3) {
			 snprintf(message, sizeof(message),
				  _("Couldn't set protocol version to LDAPv3."));
			 error_push(open_context, message);
		    }
	       }

	       /* Let's turn on TLS */
	       rc = ldap_start_tls_s(ld, NULL, NULL);
	       if(rc != LDAP_SUCCESS) {
		    snprintf(message, sizeof(message),
			    _("Couldn't enable TLS on the LDAP connection: %s"),
			    ldap_err2string(rc));
		    error_push(open_context, message);
		    push_ldap_addl_error(ld, open_context);
		    ldap_unbind(ld);

		    return rc;
	       }
#else
	       snprintf(message, sizeof(message),
		       _("TLS was not found to be supported by your LDAP libraries.\n"
			 "See README.TLS for more information.\n"));
	       error_push(open_context, message);
#endif /* defined(HAVE_TLS) */
	  }

	  if(server->local_cache_timeout >= 0) {
#if HAVE_LDAP_CLIENT_CACHE
	       ldap_enable_cache(ld, server->local_cache_timeout, 
				 DEFAULT_LOCAL_CACHE_SIZE);
#else
	       snprintf(message, sizeof(message),
		       _("OpenLDAP client-side caching was not enabled.\n"
			 "See configure --help for more information.\n"));
	       error_push(open_context, message);
#endif
	  }
	       
	  if(server->binddn[0]) {
	       binddn = server->binddn;
	       bindpw = server->bindpw;
	  }
	  else if (server->enteredpw[0]) {
	       binddn = server->binddn;
	       bindpw = server->enteredpw;
	  } else {
	       binddn = NULL;
	       bindpw = NULL;
	  }

	  switch (server->bindtype) {
	       case BINDTYPE_KERBEROS:
#		    ifdef HAVE_KERBEROS
		    rc = ldap_bind_s(ld, binddn, bindpw, LDAP_AUTH_KRBV4);
#		    else
		    error_push(open_context, 
			       _("GQ was compiled without Kerberos support.\n"
				 "Run 'configure --help' for more information\n"));
		    statusbar_msg("");
		    ldap_unbind(ld);
		    ld = NULL;
#		    endif
		    break;
	       case BINDTYPE_SASL:
#		    ifdef HAVE_SASL
		    rc = ldap_sasl_interactive_bind_s(ld, binddn, NULL, NULL, NULL, LDAP_SASL_QUIET, util_ldap_sasl_interact, NULL);
#		    else
		    error_push(open_context, 
			       _("GQ was compiled without SASL support.\n"
				 "Run 'configure --help' for more information\n"));
		    statusbar_msg("");
		    ldap_unbind(ld);
		    ld = NULL;
#		    endif
		    break;
	       default:
		    rc = ldap_simple_bind_s(ld, binddn, bindpw);
		    break;
	  }

	  if (rc != LDAP_SUCCESS) {
	       /* Maybe we cannot use LDAPv3 ... try again */
		    


	       snprintf(message, sizeof(message),
			_("Couldn't bind LDAP connection: %s"),
			ldap_err2string(rc));
	       error_push(open_context, message);
	       push_ldap_addl_error(ld, open_context);
	       statusbar_msg("");
	       /* might as well clean this up */
	       ldap_unbind(ld);
	       ld = NULL;
	  }
	  else {
	       /* always store connection handle, regardless of connection
		  caching -- call close_connection() after each operation
		  to do the caching thing or not */
	       server->connection = ld;
	       server->missing_closes = 1;
	  }

     }
     *ld_out = ld;
     return rc;
}


/*
 * open connection to LDAP server, and store connection for caching
 */
LDAP *open_connection(struct ldapserver *server)
{
     LDAP *ld;
     int open_context;
     int rc;
     char message[256]; /* FIXME */
     int newpw = 0;

     if(!server) return NULL;

     server->missing_closes++;

     /* reuse previous connection if available */
     if(server->connection) {
	  if (server->server_down == 0)
	       return(server->connection);
	  else {
	       /* do not leak file descriptors in case we need to
                  "rebind" */
	       ldap_unbind(server->connection);
	       server->connection = NULL;
	  }
	  
     }

     if (server->ask_pw &&
	 server->binddn[0] != 0 && /* makes sense only if we bind as someone */
	 /*	 server->bindpw[0] == 0 &&  */
	 server->enteredpw[0] == 0) {
	 char *ret;
	 snprintf(message, sizeof(message),
		  _("Password for server %s"), server->ldaphost);
	 
	 if (query_popup(message, &ret, TRUE /* is_password */)) {
	     if (ret) {
		 strncpy(server->enteredpw, ret, 
			 sizeof(server->enteredpw) - 1);
		 g_free(ret);
		 newpw = 1;
	     } else {
		 server->bindpw[0] = 0;
	     }
	 } else {
	     return NULL;
	 }
     }

     if(server->ldapport == 389) {
	  snprintf(message, sizeof(message), 
		   _("connecting to %s"), server->ldaphost);
     } else {
	  snprintf(message, sizeof(message),
		   _("connecting to %s port %d"), 
		   server->ldaphost, server->ldapport);
     }
     statusbar_msg(message);

     ld = NULL;
     open_context = error_new_context(_("Error connecting to server"));

     rc = do_ldap_connect(&ld, server, open_context, 
			  TRY_VERSION3);
     if (rc == LDAP_PROTOCOL_ERROR) {
	  /* this might be an indication that the server does not
	     understand LDAP_VERSION3 - retry without trying VERSION3 */
	  rc = do_ldap_connect(&ld, server, open_context, 0);
	  if (rc == LDAP_SUCCESS) error_clear(open_context);

/*       } else if (rc == LDAP_INVALID_CREDENTIALS) { */
     }

     error_flush(open_context);
     if (rc != LDAP_SUCCESS) {
	  if (newpw) server->bindpw[0] = 0; 
     }

     return(ld);
}


/*
 * called after every set of LDAP operations. This preserves the connection
 * if caching is enabled for the server. Set always to TRUE if you want to
 * close the connection regardless of caching.
 */
void close_connection(struct ldapserver *server, int always)
{
     server->missing_closes--;
/*      if (server->missing_closes < 0) abort(); */

     if( server->connection &&
	 ( (server->missing_closes == 0 && !server->cacheconn)
	   || always) ) {

#if 0
	  /*
	   * DO NOT clear the schema - we might be disconnecting
	   * because we are not caching the connection.
	   * 
	   * This fixes a bug reported by Stephan Duehr <stephand@suse.de>
	   *
	   * BUT: FIXME: Think about how to deal with schema changes
	   * due to server restarts (Popup menu on the schema top
	   * nodes [ie. resurrect the refresh popup already there]?)
	   * 
	   */
 
	  clear_server_schema(server);
#endif
	  /* definitely close this connection */
	  ldap_unbind(server->connection);
	  server->connection = NULL;
/*  	  server->missing_closes = 0; */
     }

}


/*
 * clear cached server schema
 */
void clear_server_schema(struct ldapserver *server)
{
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     GList *list;
     struct server_schema *ss;

     if(server->ss) {
	  ss = server->ss;

	  /* objectclasses */
	  list = ss->oc;
	  if(list) {
	       while(list) {
		    ldap_objectclass_free(list->data);
		    list = list->next;
	       }
	       g_list_free(ss->oc);
	  }

	  /* attribute types */
	  list = ss->at;
	  if(list) {
	       while(list) {
		    ldap_attributetype_free(list->data);
		    list = list->next;
	       }
	       g_list_free(ss->at);
	  }

	  /* matching rules */
	  list = ss->mr;
	  if(list) {
	       while(list) {
		    ldap_matchingrule_free(list->data);
		    list = list->next;
	       }
	       g_list_free(ss->mr);
	  }

	  /* syntaxes */
	  list = ss->s;
	  if(list) {
	       while(list) {
		    ldap_syntax_free(list->data);
		    list = list->next;
	       }
	       g_list_free(ss->s);
	  }

	  FREE(server->ss, "struct server_schema");
	  server->ss = NULL;
     }
     else
	  server->flags &= ~SERVER_HAS_NO_SCHEMA;
#endif

}


/*
 * delete entry
 */
gboolean delete_entry_full(struct ldapserver *server, char *dn,
			   gboolean recursive)
{
     LDAP *ld;
     int msg;
     char message[MAX_DN_LEN + 128];
     gboolean rc = TRUE;

     /* FIXME confirm-mod check here */

     set_busycursor();

     if( (ld = open_connection(server) ) == NULL) {
	  set_normalcursor();
	  return(FALSE);
     }

     if (recursive) {
	  int something_to_do = 1;
	  LDAPMessage *res, *e;
	  static char* attrs [] = {"dn", NULL};

	  while (something_to_do) {
	       something_to_do = 0;
	       msg = ldap_search_s (ld, dn, LDAP_SCOPE_ONELEVEL,
				    "(objectclass=*)", attrs, 1, &res);
	       if(msg == LDAP_SUCCESS) {
		    for (e = ldap_first_entry(ld, res); e ; 
			 e = ldap_next_entry(ld, e)) {
			 if (delete_entry_full(server, ldap_get_dn(ld, e), 
					       TRUE)) {
			      something_to_do = 1;
			 } else {
			      goto done;
			 }
		    }
		    ldap_msgfree(res);
	       } else if (msg == LDAP_SERVER_DOWN) {
		    server->server_down++;
		    goto done;
	       }
	  }
     }

     snprintf(message, sizeof(message), _("deleting: %s"), dn);
     statusbar_msg(message);

     msg = ldap_delete_s(ld, dn);
#if HAVE_LDAP_CLIENT_CACHE
     ldap_uncache_entry(ld, dn);
#endif

     if(msg != LDAP_SUCCESS) {
	  snprintf(message, sizeof(message),
		   "%s: %s", dn, ldap_err2string(msg));
	  error_popup(_("Error deleting entry"), message);
	  rc = FALSE;
     }
     else {
	  snprintf(message, sizeof(message),
		   _("deleted %s"), dn);
	  statusbar_msg(message);
     }

 done:
     set_normalcursor();
     close_connection(server, FALSE);

     return(rc);
}


gboolean delete_entry(struct ldapserver *server, char *dn) 
{
     return delete_entry_full(server, dn, FALSE);
}

gboolean do_recursive_delete(struct ldapserver *server, char* dn)
{
     return delete_entry_full(server, dn, TRUE);
}

/*
 * display hourglass cursor on mainwin
 */
void set_busycursor(void)
{
     GdkCursor *busycursor;

     busycursor = gdk_cursor_new(GDK_WATCH);
     gdk_window_set_cursor(mainwin->window, busycursor);
     gdk_cursor_destroy(busycursor);

}


/*
 * set mainwin cursor to default
 */
void set_normalcursor(void)
{

     gdk_window_set_cursor(mainwin->window, NULL);

}


void make_message(char *buffer, int bufsize, 
		  int cnt, char *singular, char *plural, char *suffix)
{

     switch(cnt) {
     case 0:
	  snprintf(buffer, bufsize, _("no %s %s"), plural, suffix);
	  break;
     case 1:
	  snprintf(buffer, bufsize, _("1 %s %s"), singular, suffix);
	  break;
     default:
	  snprintf(buffer, bufsize, "%d %s %s", cnt, plural, suffix);
	  break;
     }

}


/*
 * callback for key_press_event on a widget, destroys obj if key was esc
 */
int close_on_esc(GtkWidget *widget, GdkEventKey *event, gpointer obj)
{

     if(event && event->type == GDK_KEY_PRESS && event->keyval == GDK_Escape) {
	  gtk_widget_destroy(GTK_WIDGET(obj));
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
     }

     return(TRUE);
}


/*
 * callback for key_press_event on a widget, calls func if key was esc
 */
int func_on_esc(GtkWidget *widget, GdkEventKey *event, GtkWidget *window)
{
     void (*func)(GtkWidget *);

     if(event && event->type == GDK_KEY_PRESS && event->keyval == GDK_Escape) {
	  func = gtk_object_get_data(GTK_OBJECT(window), "close_func");
	  func(widget);
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
     }

     return(TRUE);
}


int tokenize(struct tokenlist *list, char *keyword)
{
     int i;

     for(i = 0; strlen(list[i].keyword); i++)
	  if(!strcasecmp(list[i].keyword, keyword))
	       return(list[i].token);

     return(0);
}


char *detokenize(struct tokenlist *list, int token)
{
     int i;

     for(i = 0; strlen(list[i].keyword); i++)
	  if(list[i].token == token)
	       return(list[i].keyword);

     return(list[0].keyword);
}

gpointer detokenize_data(struct tokenlist *list, int token)
{
     int i;

     for(i = 0; strlen(list[i].keyword); i++)
	  if(list[i].token == token)
	       return(list[i].data);

     return(list[0].data);
}

/*
 * return pointer to username (must be freed)
 */
char *get_username(void)
{
     struct passwd *pwent;
     char *username;

     username = NULL;
     pwent = getpwuid(getuid());
     if(pwent && pwent->pw_name)
	  username = strdup(pwent->pw_name);
     endpwent();

     return(username);
}


static guint context = 0, msgid = 0;

/*
 * display message in main window's statusbar, and flushes the
 * GTK event queue
 */
void statusbar_msg(char *message)
{
     if(!context)
	  context = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar),
						 "mainwin");
     if(msgid)
	  gtk_statusbar_remove(GTK_STATUSBAR(statusbar), context, msgid);

     msgid = gtk_statusbar_push(GTK_STATUSBAR(statusbar), context, message);

     /* make sure statusbar gets updated right away */
     while(gtk_events_pending())
	  gtk_main_iteration();
}

/*
 * display message in main window's statusbar, and flushes the
 * GTK event queue, UTF-8 version
 */
void statusbar_msg_utf8(char *message)
{
     int len = strlen(message);
     char *buf = g_malloc(len + 10);
     
     decode_string(buf, message, len);

     if(!context)
	  context = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar),
						 "mainwin");
     if(msgid)
	  gtk_statusbar_remove(GTK_STATUSBAR(statusbar), context, msgid);

     msgid = gtk_statusbar_push(GTK_STATUSBAR(statusbar), context, buf);

     /* make sure statusbar gets updated right away */
     while(gtk_events_pending())
	  gtk_main_iteration();

     g_free(buf);
}


/*
 * return pointer to (struct ldapserver *) matching name
 */
struct ldapserver *server_by_name(char *name)
{
     struct ldapserver *server;

     if(name == NULL || name[0] == '\0')
	  return(NULL);

     server = config.ldapservers;
     while(server) {
	  if(!strcmp(server->name, name))
	       return(server);
	  server = server->next;
     }

     /* XXX - This needs to be gotten rid of, and instead have the appropriate
               errors generated in the code that depends on server_by_name. 
	       -- a7r
     */
     error_popup(_("Error finding server"), _("Server not configured."));

     return(NULL);
}


/*
 * check if entry has a subtree
 */
int is_leaf_entry(struct ldapserver *server, char *dn)
{
     LDAP *ld;
     LDAPMessage *res;
     GString *gmessage;
     int msg, is_leaf;

     is_leaf = 0;

     set_busycursor();

     if( (ld = open_connection(server) ) == NULL) {
	  set_normalcursor();
	  return(-1);
     }

     gmessage = g_string_sized_new(128);
     g_string_sprintf(gmessage, _("checking subtree for %s"), dn);
     statusbar_msg(gmessage->str);
     g_string_free(gmessage, TRUE);

     msg = ldap_search(ld, dn, LDAP_SCOPE_ONELEVEL, "(objectclass=*)",
		       NULL, 0);
     if(msg != -1) {
	  if( (ldap_result(ld, msg, 0, NULL, &res) != LDAP_RES_SEARCH_ENTRY))
	       is_leaf = 1;
	  ldap_msgfree(res);
	  ldap_abandon(ld, msg);
     }

     close_connection(server, FALSE);
     set_normalcursor();
     statusbar_msg("");

     return(is_leaf);
}

/*
 * check if child is a direct subentry of possible_parent
 */
gboolean is_direct_parent(char *child, char *possible_parent) 
{
     /* SHOULD use gq_ldap_explode_dn for this */
     char *c = strchr(child, ',');
     if (c && (strcasecmp(c + 1, possible_parent) == 0)) return TRUE;
     return FALSE;
}

/*
 * check if child is a (possibly indirect) subentry of possible_ancestor
 */
gboolean is_ancestor(char *child, char *possible_ancestor) 
{
     char **rdn = gq_ldap_explode_dn(child, FALSE);
     GString *s;
     int n;
     gboolean rc = FALSE;

     for (n = 0 ; rdn[n] ; n++) {
     }

     s = g_string_new("");
     for (n-- ; n >= 0 ; n--) {
	  g_string_insert(s, 0, rdn[n]);
	  if ((strcasecmp(s->str, possible_ancestor) == 0)) {
	       rc = TRUE;
	       break;
	  }
	  g_string_insert(s, 0, ",");
     }

     g_string_free(s, TRUE);
     gq_exploded_free(rdn);

     return rc;
}

GList *ar2glist(char *ar[])
{
     GList *tmp;
     int i;

     if(ar == NULL) {
	  /* gtk_combo_set_popdown_strings() can't handle an
	     empty list, so give it a list with an empty entry */
	  tmp = g_list_append(NULL, "");
	  return(tmp);
     }

     tmp = NULL;
     i = 0;
     while(ar[i])
	  tmp = g_list_append(tmp, ar[i++]);

     return(tmp);
}


/*
 * pops up a warning dialog (with hand icon), and displays all messages in
 * the GList , one per line. The GList is freed here.
 */
void warning_popup(GList *messages)
{
     GList *list;
     GtkWidget *window, *vbox1, *vbox2, *vbox3, *label, *hbox0;
     GtkWidget *hbox1, *ok_button;
     GtkWidget *pixmap;
     GdkPixmap *warning;
     GdkBitmap *warning_mask;

     window = gtk_dialog_new();
     gtk_container_border_width(GTK_CONTAINER(window), 12);
     gtk_window_set_title(GTK_WINDOW(window), _("Warning"));
     gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
     vbox1 = GTK_DIALOG(window)->vbox;
     gtk_widget_show(vbox1);

     hbox1 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, FALSE, 10);
     warning = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(mainwin)->window,
					    &warning_mask,
					    &mainwin->style->white,
					    warning_xpm);
     pixmap = gtk_pixmap_new(warning, warning_mask);
     gtk_widget_show(pixmap);
     gtk_box_pack_start(GTK_BOX(hbox1), pixmap, TRUE, TRUE, 10);

     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox2, TRUE, TRUE, 10);

     list = messages;
     while(list) {
	  label = gtk_label_new((char *) list->data);
	  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	  gtk_widget_show(label);
	  gtk_box_pack_start(GTK_BOX(vbox2), label, TRUE, TRUE, 0);
	  list = list->next;
     }

     /* OK button */
     vbox3 = GTK_DIALOG(window)->action_area;
     gtk_widget_show(vbox3);

     hbox0 = gtk_hbutton_box_new();
     gtk_widget_show(hbox0);
     gtk_box_pack_start(GTK_BOX(vbox3), hbox0, TRUE, TRUE, 0);

     ok_button = gtk_button_new_with_label(_("OK"));
     gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked",
			       (GtkSignalFunc) gtk_widget_destroy,
			       (gpointer) window);
     gtk_box_pack_start(GTK_BOX(hbox0), ok_button, FALSE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(ok_button);
     gtk_widget_show(ok_button);

     gtk_signal_connect_object(GTK_OBJECT(window), "destroy",
			       (GtkSignalFunc) gtk_widget_destroy,
			       (gpointer) window);
     gtk_signal_connect_object(GTK_OBJECT(window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) window);

     gtk_widget_show(window);

     g_list_free(messages);

}


void single_warning_popup(char *message)
{
     GList *msglist;

     msglist = g_list_append(NULL, message);
     warning_popup(msglist);

}


#ifdef HAVE_LDAP_STR2OBJECTCLASS
GList *find_at_by_mr_oid(struct ldapserver *server, char *oid)
{
     GList *list, *srvlist;
     LDAPAttributeType *at;

     list = NULL;
     srvlist = server->ss->at;
     while(srvlist) {
	  at = (LDAPAttributeType *) srvlist->data;

	  if( (at->at_equality_oid && !strcasecmp(oid, at->at_equality_oid)) ||
	      (at->at_ordering_oid && !strcasecmp(oid, at->at_ordering_oid)) ||
	      (at->at_substr_oid && !strcasecmp(oid, at->at_substr_oid)) ) {
	       if(at->at_names && at->at_names[0])
		    list = g_list_append(list, at->at_names[0]);
	       else
		    list = g_list_append(list, at->at_oid);
	  }
	  srvlist = srvlist->next;
     }

     return(list);
}

LDAPAttributeType *find_canonical_at_by_at(struct server_schema *schema,
					   char *attr)
{
     GList *atlist;
     char **n;
     
     if (!schema) return NULL;
     
     for ( atlist = schema->at ; atlist ; atlist = atlist->next ) {
	  LDAPAttributeType *at = (LDAPAttributeType *) atlist->data;
	  if (!at) continue;
	  
	  for (n = at->at_names ; n && *n ; n++) {
/*  	       printf("%s ", *n); */
	       if (strcasecmp(*n, attr) == 0) {
		    /* found! */
		    return at;
	       }
	  }
     }
     return NULL;
}

GList *find_at_by_s_oid(struct ldapserver *server, char *oid)
{
     GList *list, *srvlist;
     LDAPAttributeType *at;

     list = NULL;
     srvlist = server->ss->at;
     while(srvlist) {
	  at = (LDAPAttributeType *) srvlist->data;

	  if(at->at_syntax_oid && !strcasecmp(oid, at->at_syntax_oid)) {
	       if(at->at_names && at->at_names[0])
		    list = g_list_append(list, at->at_names[0]);
	       else
		    list = g_list_append(list, at->at_oid);
	  }
	  srvlist = srvlist->next;
     }

     return(list);
}


GList *find_mr_by_s_oid(struct ldapserver *server, char *oid)
{
     GList *list, *srvlist;
     LDAPMatchingRule *mr;

     list = NULL;
     srvlist = server->ss->mr;
     while(srvlist) {
	  mr = (LDAPMatchingRule *) srvlist->data;

	  if(mr->mr_syntax_oid && !strcasecmp(oid, mr->mr_syntax_oid)) {
	       if(mr->mr_names && mr->mr_names[0])
		    list = g_list_append(list, mr->mr_names[0]);
	       else
		    list = g_list_append(list, mr->mr_oid);
	  }
	  srvlist = srvlist->next;
     }

     return(list);
}


GList *find_oc_by_at(struct ldapserver *server, char *atname)
{
     GList *list, *srvlist;
     LDAPObjectClass *oc;
     int i, found;

     list = NULL;
     srvlist = server->ss->oc;
     while(srvlist) {
	  oc = (LDAPObjectClass *) srvlist->data;

	  found = 0;

	  if(oc->oc_at_oids_must) {
	       i = 0;
	       while(oc->oc_at_oids_must[i] && !found) {
		    if(!strcasecmp(atname, oc->oc_at_oids_must[i])) {
			 found = 1;
			 if(oc->oc_names && oc->oc_names[0])
			      list = g_list_append(list, oc->oc_names[0]);
			 else
			      list = g_list_append(list, oc->oc_oid);
		    }
		    i++;
	       }
	  }

	  if(oc->oc_at_oids_may) {
	       i = 0;
	       while(oc->oc_at_oids_may[i] && !found) {
		    if(!strcasecmp(atname, oc->oc_at_oids_may[i])) {
			 found = 1;
			 if(oc->oc_names && oc->oc_names[0])
			      list = g_list_append(list, oc->oc_names[0]);
			 else
			      list = g_list_append(list, oc->oc_oid);
		    }
		    i++;
	       }
	  }

	  srvlist = srvlist->next;
     }

     return(list);
}

const char *find_s_by_at_oid(struct ldapserver *server, const char *oid)
{
     GList *srvlist;
     LDAPAttributeType *at;
     char **c;
     struct server_schema *ss = NULL;

     if (server == NULL) return NULL;
     ss = get_schema(server);

     if(ss == NULL || ss->at == NULL)
	  return(NULL);

     srvlist = ss->at;
     while(srvlist) {
	  at = (LDAPAttributeType *) srvlist->data;

	  for (c = at->at_names ; c && *c ; c++) {
	      if (!strcasecmp(oid, *c)) {
		  return at->at_syntax_oid;
	      }
	  }
	  srvlist = srvlist->next;
     }

     return NULL;
}

#else /* HAVE_LDAP_STR2OBJECTCLASS */


/* fall back to attributeName to find syntax. */

struct oid2syntax_t {
     const char *oid;
     const char *syntax;
};

static struct oid2syntax_t oid2syntax[] = {
     { "userPassword", "1.3.6.1.4.1.1466.115.121.1.40" },
     { "jpegPhoto", "1.3.6.1.4.1.1466.115.121.1.28" },
     { "audio", "1.3.6.1.4.1.1466.115.121.1.4" },
     { "photo", "1.3.6.1.4.1.1466.115.121.1.4" },
     { NULL, NULL },
};

const char *find_s_by_at_oid(struct ldapserver *server, const char *oid)
{
     struct oid2syntax_t *os;
     for (os = oid2syntax ; os->oid ; os++) {
	  if (strcasecmp(os->oid, oid) == 0) return os->syntax;
     }
     
     return "1.3.6.1.4.1.1466.115.121.1.3";
}


#endif /* HAVE_LDAP_STR2OBJECTCLASS */


struct gq_template *find_template_by_name(char *templatename)
{
     GList *templatelist;
     struct gq_template *template;

     template = NULL;
     templatelist = config.templates;
     while(templatelist) {
	  template = (struct gq_template *) templatelist->data;
	  if(!strcasecmp(templatename, template->name))
	       break;
	  else
	       template = NULL;
	  templatelist = templatelist->next;
     }

     return(template);
}



void dump_mods(LDAPMod **mods)
{
     LDAPMod *mod;
     int cmod, cval;

     cmod = 0;
     while(mods[cmod]) {
	  mod = mods[cmod];
	  switch(mod->mod_op) {
	  case LDAP_MOD_ADD: printf("LDAP_MOD_ADD"); break;
	  case LDAP_MOD_DELETE: printf("LDAP_MOD_DELETE"); break;
	  case LDAP_MOD_REPLACE: printf("LDAP_MOD_REPLACE"); break;
	  case LDAP_MOD_BVALUES: printf("LDAP_MOD_BVALUES"); break;
	  }
	  printf(" %s\n", mod->mod_type);
	  cval = 0;
	  while(mod->mod_values && mod->mod_values[cval]) {
	       printf("\t%s\n", mod->mod_values[cval]);
	       cval++;
	  }

	  cmod++;
     }


}

static void query_destroy(GtkWidget *button, GtkWidget *comm) {
     if (! gtk_object_get_data(GTK_OBJECT(comm), "ended"))
	  gtk_main_quit();
     
     gtk_object_set_data(GTK_OBJECT(comm), "destroyed", "1");
}

static void query_ok(GtkWidget *button, GtkObject *comm) {
     gtk_main_quit();
     gtk_object_set_data(GTK_OBJECT(comm), "rc", "1");
}

static void query_cancel(GtkWidget *button, GtkObject *comm) {
     gtk_main_quit();
     gtk_object_set_data(GTK_OBJECT(comm), "rc", (void*) NULL);
}

/* pops up a dialog to retrieve user data via a GtkEntry. This
   functions waits for the data and puts it into outbuf. */

int query_popup(const char *title, gchar **outbuf, gboolean is_password)
{
     GtkWidget *window, *vbox1, *vbox2, *label, *inputbox, *button, *hbox0;
     int rc;
     GtkWidget *f = gtk_grab_get_current();
     GtkWidget *comm; /* communication widget, not shown but used to
                         associate data with */
     
     /* This is a BAD hack - it solves a problem with the query popup
        dialog that locks up focus handling with all the
        window-managers I have been able to test this with. Actually,
        it should be sufficient to let go of the focus, but
        hiding/showing seems to work... (as I do not know how to
        release the focus in gtk) - Any gtk Hackers around? */
     if (f != NULL) {
	 gtk_widget_hide(f);
	 gtk_widget_show(f);
     }

     comm = gtk_entry_new();

     window = gtk_dialog_new();
     gtk_container_border_width(GTK_CONTAINER(window), 12);
     gtk_window_set_title(GTK_WINDOW(window), title);
     gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
     gtk_signal_connect(GTK_OBJECT(window), "destroy",
			(GtkSignalFunc) query_destroy,
			(gpointer) comm);
     gtk_signal_connect_object(GTK_OBJECT(window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) window);

     vbox1 = GTK_DIALOG(window)->vbox;
     gtk_widget_show(vbox1);

     label = gtk_label_new(title);
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(vbox1), label, TRUE, TRUE, 0);

     inputbox = gtk_entry_new();

     GTK_WIDGET_SET_FLAGS(inputbox, GTK_CAN_FOCUS);
     GTK_WIDGET_SET_FLAGS(inputbox, GTK_CAN_DEFAULT);
     if (is_password) {
	  gtk_entry_set_visibility(GTK_ENTRY(inputbox), FALSE);
     }
     gtk_widget_show(inputbox);
     gtk_signal_connect(GTK_OBJECT(inputbox), "activate",
			GTK_SIGNAL_FUNC(query_ok), GTK_OBJECT(comm));
     gtk_box_pack_start(GTK_BOX(vbox1), inputbox, TRUE, TRUE, 0);

     vbox2 = GTK_DIALOG(window)->action_area;
     gtk_widget_show(vbox2);

     hbox0 = gtk_hbutton_box_new();
     gtk_widget_show(hbox0);
     gtk_box_pack_start(GTK_BOX(vbox2), hbox0, TRUE, TRUE, 0);

     button = gtk_button_new_with_label(_("OK"));
     gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(query_ok), GTK_OBJECT(comm));
     gtk_box_pack_start(GTK_BOX(hbox0), button, FALSE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
     GTK_WIDGET_SET_FLAGS(button, GTK_RECEIVES_DEFAULT);
     gtk_widget_grab_default(button);
     gtk_widget_show(button);

     button = gtk_button_new_with_label(_("Cancel"));
     gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(query_cancel), 
			GTK_OBJECT(comm));

     gtk_box_pack_end(GTK_BOX(hbox0), button, FALSE, FALSE, 0);
     gtk_widget_show(button);

/*       gtk_window_set_transient_for(GTK_WINDOW(window),  */
/*  				  GTK_WINDOW(getMainWin())); */
     gtk_widget_grab_focus(GTK_WIDGET(window));
     gtk_window_set_modal(GTK_WINDOW(window), TRUE);

     gtk_widget_show(window);
     gtk_widget_grab_focus(inputbox);

     gtk_main();
     gtk_object_set_data(GTK_OBJECT(comm), "ended", "1");

     rc = gtk_object_get_data(GTK_OBJECT(comm), "rc") != NULL;

     if (! gtk_object_get_data(GTK_OBJECT(comm), "destroyed") && rc) {
	 *outbuf = gtk_editable_get_chars(GTK_EDITABLE(inputbox), 0, -1);
     } else {
	 rc = 0;
	 *outbuf = NULL;
     }

     if (! gtk_object_get_data(GTK_OBJECT(comm), "destroyed")) {
	 gtk_widget_destroy(window);
     }
     gtk_widget_unref(comm);

     return rc;
}


/* pops up a question dialog to ask the user a simple question. This
   functions waits for the answer and returns it. */

int question_popup(const char *title, const char *question)
{
     GtkWidget *window, *hbox0, *hbox1, *vbox1, *hbox2, *label, *button, *pixmap;
     int rc;
     GdkPixmap *warning;
     GdkBitmap *warning_mask;
     GtkWidget *f = gtk_grab_get_current();
     GtkWidget *comm; /* communication widget, not shown but used to
                         associate data with */
     
     /* This is a BAD hack - it solves a problem with the query popup
        dialog that locks up focus handling with all the
        window-managers I have been able to test this with. Actually,
        it should be sufficient to let go of the focus, but
        hiding/showing seems to work... (as I do not know how to
        release the focus in gtk) - Any gtk Hackers around? */
     if (f != NULL) {
	 gtk_widget_hide(f);
	 gtk_widget_show(f);
     }

     comm = gtk_entry_new();

     window = gtk_dialog_new();
     gtk_container_border_width(GTK_CONTAINER(window), 12);
     gtk_window_set_title(GTK_WINDOW(window), title);
     gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
     gtk_signal_connect(GTK_OBJECT(window), "destroy",
			(GtkSignalFunc) query_destroy,
			(gpointer) comm);
     gtk_signal_connect_object(GTK_OBJECT(window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) window);

     vbox1 = GTK_DIALOG(window)->vbox;
     gtk_widget_show(vbox1);

     hbox1 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, FALSE, 10);
     warning = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(mainwin)->window,
                                            &warning_mask,
                                            &mainwin->style->white,
                                            warning_xpm);
     pixmap = gtk_pixmap_new(warning, warning_mask);
     gtk_widget_show(pixmap);
     gtk_box_pack_start(GTK_BOX(hbox1), pixmap, TRUE, TRUE, 10);

     label = gtk_label_new(question);
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
     gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
     gtk_widget_show(label);
     gtk_box_pack_end(GTK_BOX(hbox1), label, TRUE, TRUE, 0);

     hbox2 = GTK_DIALOG(window)->action_area;
     gtk_widget_show(hbox2);

     hbox0 = gtk_hbutton_box_new();
     gtk_widget_show(hbox0);
     gtk_box_pack_start(GTK_BOX(hbox2), hbox0, TRUE, TRUE, 0);

     button = gtk_button_new_with_label(_("Yes"));
     gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(query_ok), GTK_OBJECT(comm));
     gtk_box_pack_start(GTK_BOX(hbox0), button, FALSE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
     GTK_WIDGET_SET_FLAGS(button, GTK_RECEIVES_DEFAULT);
     gtk_widget_grab_default(button);
     gtk_widget_show(button);

     button = gtk_button_new_with_label(_("No"));
     gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(query_cancel), 
			GTK_OBJECT(comm));

     gtk_box_pack_end(GTK_BOX(hbox0), button, FALSE, FALSE, 0);
     gtk_widget_show(button);

/*       gtk_window_set_transient_for(GTK_WINDOW(window),  */
/*  				  GTK_WINDOW(getMainWin())); */
     gtk_widget_grab_focus(GTK_WIDGET(window));
     gtk_window_set_modal(GTK_WINDOW(window), TRUE);

     gtk_widget_show(window);

     gtk_main();
     gtk_object_set_data(GTK_OBJECT(comm), "ended", "1");

     rc = gtk_object_get_data(GTK_OBJECT(comm), "rc") != NULL;

/*       if (! gtk_object_get_data(GTK_OBJECT(comm), "destroyed") && rc) { */
/*  	 *outbuf = gtk_editable_get_chars(GTK_EDITABLE(inputbox), 0, -1); */
/*       } else { */
/*  	 rc = 0; */
/*  	 *outbuf = NULL; */
/*       } */

     if (! gtk_object_get_data(GTK_OBJECT(comm), "destroyed")) {
	 gtk_widget_destroy(window);
     }
     gtk_widget_unref(comm);

     return rc;
}


/*
 * get all suffixes a server considers itself authorative for.
 */

GList *get_suffixes(struct ldapserver *server)
{
     LDAP *ld;
     LDAPMessage *res, *e;
     int msg, i;
     int num_suffixes = 0;
     char **vals, message[128];
     char *ldapv3_config[] = {
	  "namingcontexts",
	  NULL
     };
     
     GList *suffixes = NULL;
     
     set_busycursor();
     
     if( (ld = open_connection(server)) == NULL) {
	  set_normalcursor();
	  return NULL;
     }
     
     /* try LDAP V3 style config */
     statusbar_msg(_("base search on NULL DN"));
     msg = ldap_search_s(ld, "", LDAP_SCOPE_BASE, "(objectclass=*)",
			 ldapv3_config, 0, &res);
     if(msg == LDAP_SUCCESS) {
	  e = ldap_first_entry(ld, res);
	  while (e) {
	       vals = ldap_get_values(ld, e, "namingcontexts");
	       if (vals) {
		    for(i = 0; vals[i]; i++) {
			 suffixes = g_list_append(suffixes, g_strdup(vals[i]));
/*  			 add_suffix(entry, ctreeroot, node, vals[i]); */
			 num_suffixes++;
		    }
		    ldap_value_free(vals);
	       }
	       e = ldap_next_entry(ld, e);
	  }
	  ldap_msgfree(res);
     } else if (msg == LDAP_SERVER_DOWN) {
	  server->server_down++;
	  /* do not try V2 in case of server down problems */
     }

     if(num_suffixes == 0) {
	  /* try Umich style config */
	  statusbar_msg(_("base search on cn=config"));
	  msg = ldap_search_s(ld, "cn=config", LDAP_SCOPE_BASE,
			      "(objectclass=*)", NULL, 0, &res);
	  
	  if(msg == LDAP_SUCCESS) {
	       e = ldap_first_entry(ld, res);
	       while (e) {
		    char **valptr;
		    
		    vals = ldap_get_values(ld, e, "database");
		    if (vals) {
			 for (valptr = vals; *valptr; valptr++) {
			      char *p = *valptr;
				   
			      i = 0;
			      while (p[i] && p[i] != ':')
				   i++;
			      while (p[i] && (p[i] == ':' || p[i] == ' '))
				   i++;
			      if (p[i]) {
				   int len = strlen(p + i);
					
				   while (p[i + len - 1] == ' ')
					len--;
				   p[i + len] = '\0';
					
				   suffixes = g_list_append(suffixes,
							    g_strdup(p + i));
/*  				   add_suffix(entry, ctreeroot, node, p + i); */
				   num_suffixes++;
			      }
			 }
			 ldap_value_free(vals);
		    }
		    e = ldap_next_entry(ld, e);
	       }
	       ldap_msgfree(res);
	  } else if (msg == LDAP_SERVER_DOWN) {
	       server->server_down++;
	  }
     }

     if (num_suffixes == 0) {
	  
	  /* last resort: add the configured base DN and hope something comes out */
	  if(strlen(server->basedn)) {
	       suffixes = g_list_append(suffixes, 
					g_strdup(server->basedn));
/*  	       add_suffix(entry, ctreeroot, node, entry->server->basedn); */
	       num_suffixes++;
	  }
     }
     
     set_normalcursor();
     close_connection(server, FALSE);
     
     make_message(message, sizeof(message), num_suffixes,
		  _("suffix"), _("suffixes"), _("found"));
     statusbar_msg(message);

     return g_list_first(suffixes);
}

#ifdef HAVE_LDAP_STR2DN

/* OpenLDAP 2.1 both deprecated and broke ldap_explode_dn in one go (I
   won't comment on this).

   NOTE: this is just a first try to adapt code from Pierangelo
   Masarati <masarati@aero.polimi.it>. 
*/

char **gq_ldap_explode_dn(const char *dn, int dummy)
{
     int i;
     LDAPDN *parts;
     char **v = 0; 
     
     ldap_str2dn(dn, &parts, LDAP_DN_FORMAT_LDAP);
     /* count DN elements */
     for( i = 0 ; parts[i] ; i++ ) ;
     v = (char **) calloc((i + 2), sizeof(char*));

     for( i = 0 ; parts[i] ; i++ ) {
	  ldap_rdn2str(parts[0][i], &v[i],
		       LDAP_DN_FORMAT_LDAPV3 | LDAP_DN_PRETTY );
     } 
     return v;
}
     
void gq_exploded_free(char **exploded_dn)
{
     free(exploded_dn);
}

#endif



/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */
