/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: tinput.c,v 1.37 2002/06/18 22:07:14 stamfest Exp $ */

#include <config.h>

#ifdef HAVE_LDAP_STR2OBJECTCLASS

#include <glib.h>
#include <gtk/gtk.h>

#include <lber.h>
#include <ldap.h>
#include <ldap_schema.h>

#include <stdio.h>
#include <string.h>

#include "common.h"
#include "util.h"
#include "configfile.h"
#include "template.h"
#include "schema.h"
#include "errorchain.h"
#include "tinput.h"
#include "formfill.h"
#include "input.h"
#include "i18n.h"


/*
 * fetch objectclasses in template from server and build GList of formfill with it
 */
GList *formfill_from_template(struct ldapserver *server, struct gq_template *template)
{
     GList *formlist, *oclist;
     struct server_schema *ss;
     struct formfill *oc;

     /* make sure the server's schema is fetched, or
	find_oc_by_oc_name() will fail... */
     ss = get_schema(server);

     /* list all the objectclasses first, with values */
     oclist = template->objectclasses;
     formlist = add_attrs_by_oc(server, oclist);

     /* fill in objectClass attributes */

     oc = lookup_attribute_using_schema(formlist, "objectClass", ss, NULL);
     if (oc) {
	  /* new_formfill initializes num_inputfields with 1 */ 
	  if (!oc->values) oc->num_inputfields--;
	  while (oclist) {
	       GByteArray *gb = g_byte_array_new();
	       g_byte_array_append(gb,
				   oclist->data,
				   strlen(oclist->data));
	       oc->values = g_list_append(oc->values, gb);
	       oc->num_inputfields++;
	       
	       oclist = oclist->next;
	  }
     }

     return(formlist);
}


/*
 * fetch objectclasses in entry from server and build GList of formfill with it
 */
GList *formfill_from_entry_objectclass(struct ldapserver *server, char *dn)
{
     GList *formlist, *oclist, *tmplist;
     LDAP *ld;
     LDAPMessage *res, *entry;
     struct server_schema *ss;
     struct formfill *form;
     int msg, count;
     char **vals;
     char *oc_only[] = { "objectClass", NULL };

     formlist = NULL;

     set_busycursor();

     if( (ld = open_connection(server)) == NULL) {
          set_normalcursor();
          return(NULL);
     }

     msg = ldap_search_s(ld, dn, LDAP_SCOPE_BASE,
			 "(objectclass=*)", oc_only, 0, &res);

     if(msg != LDAP_SUCCESS) {
	  if (msg == LDAP_SERVER_DOWN) {
	       server->server_down++;
	  }
	  statusbar_msg(ldap_err2string(msg));
	  set_normalcursor();
	  close_connection(server, FALSE);
	  return(NULL);
     }

     entry = ldap_first_entry(ld, res);

     if(entry) {
	  count = 0;
	  oclist = NULL;
	  vals = ldap_get_values(ld, res, "objectClass");
	  if(vals) {
	       for(count = 0; vals[count]; count++)
		    oclist = g_list_append(oclist, g_strdup(vals[count]));
	       ldap_value_free(vals);
	  }

	  ss = get_schema(server);
	  formlist = add_attrs_by_oc(server, oclist);

	  /* add the values that were already in the "objectClass" attribute */
	  form = lookup_attribute_using_schema(formlist, "objectClass", ss, 
					       NULL);
	  if(form) {
	       tmplist = oclist;
	       while(tmplist) {
		    GByteArray *gb = g_byte_array_new();
		    g_byte_array_append(gb, tmplist->data, 
					strlen(tmplist->data));

		    g_free(tmplist->data);

		    form->values = g_list_append(form->values, gb);
		    form->num_inputfields++;
		    tmplist = tmplist->next;
	       }
	       /* got an off-by-one error someplace */
	       if(form->num_inputfields)
		    form->num_inputfields--;
	       g_list_free(oclist);
	  }

     }
     ldap_msgfree(res);

     close_connection(server, FALSE);
     set_normalcursor();

     return(formlist);
}


static GList *add_oc_and_superiors(GList *oc_list, struct server_schema *ss,
				   LDAPObjectClass *oc) 
{
     /* check if superior(s) has/have been added as well */
     if (oc->oc_sup_oids) {
	  int i;

	  for (i = 0 ; oc->oc_sup_oids[i] ; i++) {
	       LDAPObjectClass *soc = 
		    find_oc_by_oc_name(ss, oc->oc_sup_oids[i]);
	       if (soc) {
		    oc_list = add_oc_and_superiors(oc_list, ss, soc);
	       }
	  }
     }

     /* add oc itself possibly after its superiors... */

     if (oc_list && g_list_find(oc_list, oc)) {
	  /* already added */
     } else {
	  oc_list = g_list_append(oc_list, oc);
     }

     return oc_list;
}

GList *add_attrs_by_oc(struct ldapserver *server, GList *oclist)
{
     GList *formlist, *ocs_to_add = NULL, *l;
     LDAPObjectClass *oc;
     struct server_schema *ss;
     struct formfill *form;
     int i;


     if(oclist == NULL)
	  return(NULL);

     formlist = NULL;


     /* add the objectclass attribute in manually */
     if( (form = new_formfill()) == NULL) {
	  error_popup(_("Oops!"), _("Not enough memory to make form."));
	  return(NULL);
     }
     strcpy(form->attrname, "objectClass");  /* Flawfinder: ignore */
     form->flags |= FLAG_MUST_IN_SCHEMA;
     set_displaytype(server, form);

     formlist = g_list_append(formlist, form);

     /* schema functions below need this */
     ss = get_schema(server);

     for ( l = oclist ; l ; l = l->next ) {
	  oc = find_oc_by_oc_name(ss, (char *) l->data);
	  if(oc) {
	       ocs_to_add = add_oc_and_superiors(ocs_to_add, ss, oc);
	  }
     }

     for (l = g_list_first(ocs_to_add) ; l ; l = g_list_next(l)) {
	  oc = (LDAPObjectClass *) l->data;
	  if(oc) {
	       LDAPAttributeType *at;

	       /* required attributes */
	       i = 0;
	       while(oc->oc_at_oids_must && oc->oc_at_oids_must[i]) {
/*  	            if (strcasecmp(oc->oc_at_oids_must[i], "objectClass")) { */
	            if (lookup_attribute_using_schema(formlist,
						      oc->oc_at_oids_must[i],
						      ss, &at) == NULL) {
		         if( (form = new_formfill()) == NULL) {
			      error_popup(_("Oops!"), _("Not enough memory to make form."));
			      return(NULL);
		         }
		         strncpy(form->attrname, oc->oc_at_oids_must[i], MAX_ATTR_LEN);
			 form->flags |= FLAG_MUST_IN_SCHEMA;
			 if (at && at->at_single_value) {
			      form->flags |= FLAG_SINGLE_VALUE;
			 }
			 set_displaytype(server, form);
		         formlist = formlist_append(formlist, form);
		    }
		    i++;
	       }

	       /* allowed attributes */
	       i = 0;
	       while(oc->oc_at_oids_may && oc->oc_at_oids_may[i]) {
/*  		    if (strcasecmp(oc->oc_at_oids_may[i], "objectClass")) { */
	            if (lookup_attribute_using_schema(formlist, 
						      oc->oc_at_oids_may[i],
						      ss, &at) == NULL) {
		         if( (form = new_formfill()) == NULL) {
			      error_popup(_("Oops!"), _("Not enough memory to make form."));
			      return(NULL);
		         }
		         strncpy(form->attrname, oc->oc_at_oids_may[i], MAX_ATTR_LEN);
			 set_displaytype(server, form);
		         formlist = formlist_append(formlist, form);
			 if (at && at->at_single_value) {
			      form->flags |= FLAG_SINGLE_VALUE;
			 }
		    }
		    i++;
	       }

	  }

/*  	  oclist = oclist->next; */
     }

     return(formlist);
}


/*
 * adds all attributes allowed in schema to formlist
 */
GList *add_schema_attrs(struct ldapserver *server, GList *value_list)
{
     GList *oclist, *schema_list, *tmplist, *tmpvallist, *addendum;
     struct formfill *form, *oldform, *newform;

     struct server_schema *ss;

     form = lookup_attribute(value_list, "objectclass");
     if(form == NULL || form->values == NULL) {
	  /* not even an objectclass attribute type here... give up */
	  return(value_list);
     }

     /* build temporary list of object classes from formfill list */
     
     oclist = NULL;
     tmplist = form->values;
     while (tmplist) {
	  GByteArray *gb = (GByteArray*) tmplist->data;
	  /* we know this is plain text... */
	  oclist = g_list_append(oclist, g_strndup(gb->data, gb->len));
	  tmplist = tmplist->next;
     }

     ss = get_schema(server);
     schema_list = add_attrs_by_oc(server, oclist);
     g_list_foreach(oclist, (GFunc) g_free, NULL);
     g_list_free(oclist);

     /* merge value_list's values into schema_list */
     tmplist = schema_list;
     while(tmplist) {
	  form = (struct formfill *) tmplist->data;
	  oldform = lookup_attribute_using_schema(value_list, form->attrname,
						  ss, NULL);
	  if(oldform) {
	       form->num_inputfields = oldform->num_inputfields;
	       form->displaytype = oldform->displaytype;
	       form->dt_handler = oldform->dt_handler;
	       form->flags |= oldform->flags; /* to keep FLAG_MUST_IN_SCHEMA */
	       tmpvallist = oldform->values;
	       while(tmpvallist) {
		    GByteArray *oldgb = (GByteArray *) tmpvallist->data;
		    GByteArray *gb = g_byte_array_new();
		    g_byte_array_append(gb, oldgb->data, oldgb->len);

		    form->values = g_list_append(form->values, gb);
		    tmpvallist = tmpvallist->next;
	       }
	  }
	  tmplist = tmplist->next;
     }

     /* final inverse check for attrs that were in the original entry (value_list)
	but not allowed according to the schema (schema_list) */
     addendum = NULL;
     tmplist = value_list;
     while(tmplist) {
	  form = (struct formfill *) tmplist->data;
	  newform = lookup_attribute_using_schema(schema_list, form->attrname,
						  ss, NULL);
	  if(newform == NULL) {
	       /* attribute type not in schema */
	       newform = new_formfill();
	       strncpy(newform->attrname, form->attrname, 
		       sizeof(newform->attrname));
	       newform->num_inputfields = form->num_inputfields;
	       newform->displaytype = form->displaytype;
	       newform->dt_handler = form->dt_handler;
	       newform->flags = form->flags | FLAG_NOT_IN_SCHEMA;
	       tmpvallist = form->values;
	       while(tmpvallist) {
		    GByteArray *oldgb = (GByteArray *) tmpvallist->data;
		    GByteArray *gb = g_byte_array_new();
		    g_byte_array_append(gb, oldgb->data, oldgb->len);

		    newform->values = g_list_append(newform->values, gb);
		    tmpvallist = tmpvallist->next;
	       }
	       addendum = g_list_append(addendum, newform);
	  }
	  tmplist = tmplist->next;
     }
     if(addendum) {
	  while(addendum) {
	       schema_list = g_list_append(schema_list, addendum->data);
	       addendum = addendum->next;
	  }
	  g_list_free(addendum);
     }

     free_formlist(value_list);

     return(schema_list);
}


#endif

/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */
