/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2002 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: search.c,v 1.22 2002/06/18 22:07:14 stamfest Exp $ */

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#include "common.h"
#include "mainwin.h"
#include "configfile.h"
#include "util.h"
#include "search.h"
#include "formfill.h"
#include "input.h"
#include "template.h"
#include "tinput.h"
#include "filter.h"
#include "encode.h"
#include "i18n.h"


extern struct gq_config config;

static gboolean search_button_press_on_tree_item(GtkWidget *clist,
						 GdkEventButton *event,
						 GHashTable *hash);

void new_searchmode(GHashTable *hash)
{
     GtkWidget *main_clist, *searchmode_vbox, *hbox1, *scrwin;
     GtkWidget *searchtype_om, *searchtype_menu, *searchtype_mi;
     GtkWidget *stinput;
     GtkWidget *servcombo, *searchbase_combo;
     GtkWidget *findbutton;
     GSList *group;

     searchmode_vbox = gtk_vbox_new(FALSE, 0);

     hbox1 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(searchmode_vbox), hbox1, FALSE, FALSE, 3);

     /* search/filter dropdown box */
     searchtype_menu = gtk_menu_new();
     searchtype_om = gtk_option_menu_new();

     searchtype_mi = gtk_radio_menu_item_new_with_label(NULL, _("search"));
     gtk_object_set_data(GTK_OBJECT(searchtype_mi), "searchtype", GINT_TO_POINTER(ST_SEARCH));
     group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(searchtype_mi));
     gtk_menu_append(GTK_MENU(searchtype_menu), searchtype_mi);
     gtk_widget_show(searchtype_mi);
     g_hash_table_insert(hash, "searchtype_item", searchtype_mi);

     searchtype_mi = gtk_radio_menu_item_new_with_label(group, _("filter"));
     gtk_object_set_data(GTK_OBJECT(searchtype_mi), "searchtype", GINT_TO_POINTER(ST_FILTER));
     gtk_menu_append(GTK_MENU(searchtype_menu), searchtype_mi);
     gtk_widget_show(searchtype_mi);

     gtk_option_menu_set_menu(GTK_OPTION_MENU(searchtype_om), searchtype_menu);
     gtk_box_pack_start(GTK_BOX(hbox1), searchtype_om, FALSE, FALSE, 0);
     gtk_widget_show(searchtype_om);

     /* searchterm entry box */
     stinput = gtk_entry_new();
     gtk_widget_show(stinput);
     gtk_signal_connect_object(GTK_OBJECT(stinput), "activate",
			       GTK_SIGNAL_FUNC(findbutton_pressed_callback),
			       (gpointer) hash);
     gtk_box_pack_start(GTK_BOX(hbox1), stinput, TRUE, TRUE, 6);
     GTK_WIDGET_SET_FLAGS(stinput, GTK_CAN_FOCUS);
     GTK_WIDGET_SET_FLAGS(stinput, GTK_CAN_DEFAULT);
     GTK_WIDGET_SET_FLAGS(stinput, GTK_RECEIVES_DEFAULT);
     gtk_widget_grab_default(stinput);
     g_hash_table_insert(hash, "focus", stinput);

     /* LDAP server combo box */
     servcombo = gtk_combo_new();
     fill_serverlist_combo(servcombo);

     gtk_box_pack_start(GTK_BOX(hbox1), servcombo, FALSE, TRUE, 0);
     gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(servcombo)->entry), FALSE);
     gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(servcombo)->entry), "changed",
			       GTK_SIGNAL_FUNC(servername_changed_callback),
			       (gpointer) hash);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(GTK_COMBO(servcombo)->entry), GTK_CAN_FOCUS);
     GTK_WIDGET_UNSET_FLAGS(GTK_BUTTON(GTK_COMBO(servcombo)->button), GTK_CAN_FOCUS);
     gtk_widget_show(servcombo);
     g_hash_table_insert(hash, "serverlist_combo", servcombo);

     /* search base combo box */
     searchbase_combo = gtk_combo_new();

     if(config.ldapservers)
	  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(searchbase_combo)->entry), config.ldapservers->basedn);

     gtk_box_pack_start(GTK_BOX(hbox1), searchbase_combo, FALSE, TRUE, 0);
     gtk_signal_connect(GTK_OBJECT(GTK_COMBO(searchbase_combo)->button), "button_press_event",
			GTK_SIGNAL_FUNC(searchbase_button_pressed), hash);
     GTK_WIDGET_UNSET_FLAGS(GTK_ENTRY(GTK_COMBO(searchbase_combo)->entry), GTK_CAN_FOCUS);
     GTK_WIDGET_UNSET_FLAGS(GTK_BUTTON(GTK_COMBO(searchbase_combo)->button), GTK_CAN_FOCUS);
     gtk_widget_show(searchbase_combo);
     g_hash_table_insert(hash, "searchbase_combo", searchbase_combo);

     /* find button */
     findbutton = gtk_button_new_with_label(_(" Find "));
     GTK_WIDGET_UNSET_FLAGS(findbutton, GTK_CAN_FOCUS);
     gtk_widget_show(findbutton);
     gtk_box_pack_start(GTK_BOX(hbox1), findbutton, FALSE, TRUE, 6);
     gtk_container_border_width(GTK_CONTAINER (findbutton), 0);
     gtk_signal_connect_object(GTK_OBJECT(findbutton), "pressed",
			       GTK_SIGNAL_FUNC(findbutton_pressed_callback),
			       (gpointer) hash);

     /* dummy clist, gets replaced on first search */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_show(scrwin);
     g_hash_table_insert(hash, "scrwin", scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     main_clist = gtk_clist_new(1);
     gtk_clist_set_column_title(GTK_CLIST(main_clist), 0, "");
     gtk_clist_column_titles_show(GTK_CLIST(main_clist));
     gtk_widget_show(main_clist);
     gtk_clist_column_titles_passive(GTK_CLIST(main_clist));
     g_hash_table_insert(hash, "main_clist", main_clist);

     gtk_container_add(GTK_CONTAINER(scrwin), main_clist);
     gtk_box_pack_start(GTK_BOX(searchmode_vbox), scrwin, TRUE, TRUE, 0);

     gtk_widget_show(searchmode_vbox);
     g_hash_table_insert(hash, "vbox", searchmode_vbox);

}


void servername_changed_callback(GHashTable *hash)
{
     GList *searchbase_list;
     GtkWidget *servcombo, *searchbase_combo;
     struct ldapserver *cur_server;
     char *cur_servername;

     servcombo = g_hash_table_lookup(hash, "serverlist_combo");
     cur_servername = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(servcombo)->entry), 0, -1);
     cur_server = server_by_name(cur_servername);
     g_free(cur_servername);
     g_hash_table_insert(hash, "cur_server", cur_server);

     /* make sure searchbase gets refreshed next time the searchbase combo button is
	pressed. Just insert the server's default base DN for now */
     g_hash_table_remove(hash, "populated_searchbase");
     searchbase_combo = g_hash_table_lookup(hash, "searchbase_combo");
     if (cur_server) {
	  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(searchbase_combo)->entry), cur_server->basedn);
	  searchbase_list = g_list_append(NULL, cur_server->basedn);
	  gtk_combo_set_popdown_strings(GTK_COMBO(searchbase_combo), searchbase_list);
	  g_list_free(searchbase_list);
     }

}


gint searchbase_button_pressed(GtkWidget *widget, GdkEventButton *event, GHashTable *hash)
{
     GList *searchbase_list;
     GList *suffixes_list, *temp;
     GtkWidget *servcombo, *searchbase_combo;
     struct ldapserver *server;
     int found_default_searchbase;
     char *cur_servername;

     found_default_searchbase = 0;

     if (!g_hash_table_lookup(hash, "populated_searchbase") && event->button == 1) {
    
	  servcombo = g_hash_table_lookup(hash, "serverlist_combo");
	  cur_servername = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(servcombo)->entry), 0, -1);
	  server = server_by_name(cur_servername);
	  g_free(cur_servername);
	  if (!server)
	       return(FALSE);

	  searchbase_list = NULL;
	  suffixes_list = get_suffixes(server);
    
	  temp = suffixes_list;
	  while (temp) {
	       if (g_strcasecmp(server->basedn, temp->data) == 0)
		    found_default_searchbase = 1;
	       searchbase_list = g_list_append(searchbase_list, temp->data);
      	       temp = g_list_next(temp);
	  }

	  if (found_default_searchbase == 0)
	       searchbase_list = g_list_prepend(searchbase_list, server->basedn);

	  searchbase_combo = g_hash_table_lookup(hash, "searchbase_combo");
	  gtk_combo_set_popdown_strings(GTK_COMBO(searchbase_combo), searchbase_list);

	  temp = suffixes_list;
	  while (temp) {
	       g_free(temp->data);
	       temp = g_list_next(temp);
	  }
	  if (suffixes_list)
	       g_list_free(suffixes_list);
	  g_list_free(searchbase_list);

	  g_hash_table_insert(hash, "populated_searchbase", GINT_TO_POINTER(1));
     }

     return FALSE;
}


void findbutton_pressed_callback(GHashTable *hash)
{
     GtkWidget *focusbox;

     set_busycursor();
     query(hash);
     set_normalcursor();

     focusbox = g_hash_table_lookup(hash, "focus");
     gtk_widget_grab_focus(focusbox);
     gtk_widget_grab_default(focusbox);
     gtk_editable_select_region(GTK_EDITABLE(focusbox), 0, -1);

}


struct resultset *new_resultset(GHashTable *hash)
{
     struct resultset *cur_resultset, *newset, *set;

     newset = g_malloc(sizeof(struct resultset));
     newset->objectclass = NULL;
     newset->attributes = NULL;
     newset->num_attributes = 0;
     newset->next = NULL;

     cur_resultset = g_hash_table_lookup(hash, "cur_resultset");
     if(cur_resultset == NULL) {
	  g_hash_table_insert(hash, "cur_resultset", newset);
     }
     else {
	  for(set = cur_resultset; set->next; set = set->next)
	       ;
	  set->next = newset;
     }

     return(newset);
}


void free_resultset(struct resultset *set)
{
     struct resultset *nextset;

     while(set) {
	  nextset = set->next;
	  g_free(set);
	  set = nextset;
     }

}


int column_by_attr(struct attrs **attrlist, char *attribute)
{
     struct attrs *attr;

     attr = find_attr(*attrlist, attribute);

     if(!attr)
	  return(new_attr(attrlist, attribute));
     else
	  return(attr->column);
}


int new_attr(struct attrs **attrlist, char *attr)
{
     struct attrs *new_attr, *alist;
     int cnt;

     new_attr = g_malloc(sizeof(struct attrs));
     strncpy(new_attr->name, attr, MAX_ATTR_LEN);
     new_attr->next = NULL;

     cnt = 0;
     if(*attrlist) {
	  cnt++;
	  alist = *attrlist;
	  while(alist->next) {
	       cnt++;
	       alist = alist->next;
	  }
	  alist->next = new_attr;
     }
     else
	  *attrlist = new_attr;

     new_attr->column = cnt;

     return(cnt);
}


struct attrs *find_attr(struct attrs *attrlist, char *attribute)
{

     while(attrlist) {
	  if(!strcasecmp(attrlist->name, attribute))
	       return(attrlist);
	  attrlist = attrlist->next;
     }

     return(NULL);
}


void free_attrlist(struct attrs *attrlist)
{
     struct attrs *next_attrs;

     while(attrlist) {
	  next_attrs = attrlist->next;;
	  g_free(attrlist);
	  attrlist = next_attrs;
     }

}


void query(GHashTable *hash)
{
     GSList *group;
     LDAP *ld;
     LDAPMessage *res, *e;
     BerElement *berptr;
     GtkWidget *main_clist, *new_main_clist, *scrwin, *focusbox;
     GtkWidget *servcombo, *searchbase_combo, *searchtype_item;
     GtkRadioMenuItem *menuitem;
     struct ldapserver *server;
     struct resultset *set, *cur_resultset;
     struct attrs *attrlist;
     gchar *cur_servername, *cur_searchbase, *querystring, *cl[MAX_NUM_ATTRIBUTES];
     char tolist[MAX_NUM_ATTRIBUTES][128], message[MAX_LDAPFILTER_LEN + 128];
     char filter[MAX_LDAPFILTER_LEN];
     char *attr, *dn, **vals, *searchterm, *decoded_value;
     int searchtype, msg, rc, i, row;
     int cur_col, oc_col, columns_done[MAX_NUM_ATTRIBUTES];

     if(g_hash_table_lookup(hash, "search lock"))
	  return;

     g_hash_table_insert(hash, "search lock", "ongoing");

     focusbox = g_hash_table_lookup(hash, "focus");
     searchterm = gtk_editable_get_chars(GTK_EDITABLE(focusbox), 0, -1);
     querystring = encoded_string(searchterm);
     g_free(searchterm);

     for(i = 0; i < MAX_NUM_ATTRIBUTES; i++)
	  columns_done[i] = 0;

     servcombo = g_hash_table_lookup(hash, "serverlist_combo");
     cur_servername = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(servcombo)->entry), 0, -1);
     server = server_by_name(cur_servername);
     g_free(cur_servername);
     if(!server) {
	  statusbar_msg("Oops! No server found!");
	  g_hash_table_remove(hash, "search lock");
	  return;
     }

     searchtype = -1;
     searchtype_item = g_hash_table_lookup(hash, "searchtype_item");
     group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(searchtype_item));
     while(group) {
	  menuitem = group->data;
	  if(menuitem->check_menu_item.active) {
	       searchtype = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(menuitem), "searchtype"));
	       break;
	  }
	  group = group->next;
     }

     if(searchtype == ST_FILTER && querystring[0] == 0) {
	  statusbar_msg("Please enter a valid search filter");
	  g_hash_table_remove(hash, "search lock");
	  return;
     }

     if( (ld = open_connection(server)) == NULL) {
	  g_hash_table_remove(hash, "search lock");
	  return;
     }

     /* construct filter */
     switch(searchtype) {
     case ST_SEARCH:
	  switch(config.search_argument) {
	  case SEARCHARG_BEGINS_WITH:
	       snprintf(filter, MAX_LDAPFILTER_LEN - 1, "(%s=%s*)", server->searchattr, querystring);
	       break;
	  case SEARCHARG_ENDS_WITH:
	       snprintf(filter, MAX_LDAPFILTER_LEN - 1, "(%s=*%s)", server->searchattr, querystring);
	       break;
	  case SEARCHARG_CONTAINS:
	       snprintf(filter, MAX_LDAPFILTER_LEN - 1, "(%s=*%s*)", server->searchattr, querystring);
	       break;
	  case SEARCHARG_EQUALS:
	       snprintf(filter, MAX_LDAPFILTER_LEN - 1, "(%s=%s)", server->searchattr, querystring);
	       break;

	  };
	  break;
     case ST_FILTER:
	  strncpy(filter, querystring, MAX_LDAPFILTER_LEN - 1);
	  break;
     };

     free(querystring);

     snprintf(message, sizeof(message), "searching for %s", filter);
     statusbar_msg(message);

     searchbase_combo = g_hash_table_lookup(hash, "searchbase_combo");
     cur_searchbase = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(searchbase_combo)->entry), 0, -1);
     if(config.sort_search)
	  msg = ldap_search_s(ld, cur_searchbase, LDAP_SCOPE_SUBTREE,
			      filter, NULL, 0, &res);
     else
	  msg = ldap_search(ld, cur_searchbase, LDAP_SCOPE_SUBTREE,
			    filter, NULL, 0);
     g_free(cur_searchbase);

     if((config.sort_search && msg != LDAP_SUCCESS) || msg == -1) {
	  if (config.sort_search && msg != LDAP_SERVER_DOWN) {
	       server->server_down++;
	  }
	  statusbar_msg(ldap_err2string(msg == -1 ? LDAP_OTHER : msg));
	  g_hash_table_remove(hash, "search lock");
	  close_connection(server, FALSE);
	  return;
     }

     cur_resultset = g_hash_table_lookup(hash, "cur_resultset");
     if(cur_resultset) {
	  free_resultset(cur_resultset);
	  g_hash_table_remove(hash, "cur_resultset");
     }

     g_hash_table_insert(hash, "cur_resultset_server", server);

     /* build new clist */
     new_main_clist = gtk_clist_new(MAX_NUM_ATTRIBUTES);
     GTK_CLIST(new_main_clist)->button_actions[2] = GTK_BUTTON_SELECTS;
     GTK_WIDGET_UNSET_FLAGS(GTK_CLIST(new_main_clist), GTK_CAN_FOCUS);
     gtk_widget_show(new_main_clist);
     gtk_clist_column_titles_show(GTK_CLIST(new_main_clist));
     gtk_signal_connect(GTK_OBJECT(new_main_clist), "select_row",
                        GTK_SIGNAL_FUNC(select_entry_callback),
                        hash);
/*       gtk_signal_connect(GTK_OBJECT(new_main_clist), "unselect_row", */
/*                          GTK_SIGNAL_FUNC(unselect_entry_callback), */
/*                          hash); */
     gtk_signal_connect(GTK_OBJECT(new_main_clist), "button_press_event",
			GTK_SIGNAL_FUNC(search_button_press_on_tree_item),
			hash);

     main_clist = g_hash_table_lookup(hash, "main_clist");
     gtk_clist_clear(GTK_CLIST(main_clist));
     gtk_widget_destroy(main_clist);
     g_hash_table_insert(hash, "main_clist", new_main_clist);

     scrwin = g_hash_table_lookup(hash, "scrwin");
     gtk_container_add(GTK_CONTAINER(scrwin), new_main_clist);

     attrlist = NULL;

     /* reserve columns 0 & 1 for DN and objectClass, respectively */
     if(config.showdn) {
	  column_by_attr(&attrlist, "DN");
	  gtk_clist_set_column_title(GTK_CLIST(new_main_clist), 0, "DN");
	  gtk_clist_set_column_width(GTK_CLIST(new_main_clist), 0, 260);
     }
     oc_col = column_by_attr(&attrlist, "objectClass");
     gtk_clist_set_column_title(GTK_CLIST(new_main_clist), oc_col, "objectClass");
     gtk_clist_set_column_width(GTK_CLIST(new_main_clist), oc_col, 120);
     columns_done[oc_col] = 1;
     if(config.showoc == 0)
	  gtk_clist_set_column_visibility(GTK_CLIST(new_main_clist), oc_col, 0);


     row = 0;
     if(!config.sort_search)
	  rc = (ldap_result(ld, msg, 0, NULL, &res) == LDAP_RES_SEARCH_ENTRY);
     else
	  rc = (ldap_sort_entries(ld, &res, NULL, strcasecmp) == LDAP_SUCCESS);

     while(rc) {

	  for(e = ldap_first_entry(ld, res); e != NULL; e = ldap_next_entry(ld, e)) {

	       /* not every attribute necessarily comes back for every entry,
		* so clear this every time */
	       for(i = 0; i < MAX_NUM_ATTRIBUTES; i++) {
		    cl[i] = NULL;
		    tolist[i][0] = '\0';
	       }

	       dn = ldap_get_dn(ld, e);
	       /* store for later reference */
	       set = new_resultset(hash);
	       strncpy(set->dn, dn, MAX_DN_LEN);

	       if(config.showdn) {
		    decoded_value = decoded_string(dn);
		    strcpy(tolist[0], decoded_value);
		    free(decoded_value);
		    cl[0] = tolist[0];
	       }
#if defined(HAVE_LDAP_MEMFREE)
	       ldap_memfree(dn);
#else
               free(dn);
#endif

	       for(attr = ldap_first_attribute(ld, e, &berptr); attr != NULL;
		   attr = ldap_next_attribute(ld, e, berptr)) {
		    /* This should now work for ;binary as well */
		    cur_col = column_by_attr(&attrlist, attr);
		    if(cur_col == MAX_NUM_ATTRIBUTES) {
			 ldap_memfree(attr);
			 break;
		    }

		    if(!columns_done[cur_col]) {
			 char *c = attr_strip(attr);
			 gtk_clist_set_column_title(GTK_CLIST(new_main_clist), cur_col, c);
			 if (c) g_free(c);
			 gtk_clist_set_column_width(GTK_CLIST(new_main_clist), cur_col, 120);
			 columns_done[cur_col] = 1;
		    }

		    vals = ldap_get_values(ld, e, attr);
		    if(vals) {
			 for(i = 0; vals[i] != NULL; i++) {
			      /* that's an awful lot of mallocs in the innermost loop,
				 but then we don't want to count on max valuelen either */
			      decoded_value = decoded_string(vals[i]);
			      if(i == 0)
				   strcpy(tolist[cur_col], decoded_value);
			      else {
				   strcat(tolist[cur_col], " ");
				   strcat(tolist[cur_col], decoded_value);
			      }
			      free(decoded_value);
			 }
			 ldap_value_free(vals);
			 cl[cur_col] = tolist[cur_col];
		    }
		    ldap_memfree(attr);
	       }
#ifndef HAVE_OPENLDAP12
	       if(berptr)
		    ber_free(berptr, 0);
#endif

	       /* insert row into result window */
	       gtk_clist_append(GTK_CLIST(new_main_clist), cl);
	       gtk_clist_set_row_data(GTK_CLIST(new_main_clist), row, set);
	       row++;
	  }

	  if(config.sort_search)
	       rc = 0;
	  else
	       if(server->maxentries == 0 || row < server->maxentries) {
		    ldap_msgfree(res);
		    rc = (ldap_result(ld, msg, 0, NULL, &res) == LDAP_RES_SEARCH_ENTRY);
	       }
	       else
		    rc = 0;

     }

     if(rc == -1)
	  statusbar_msg(ldap_err2string(msg));
     else {
	  make_message(message, sizeof(message), 
		       row, "entry", "entries", "found");
	  statusbar_msg(message);
     }

     gtk_clist_column_titles_passive(GTK_CLIST(new_main_clist));
     free_attrlist(attrlist);
     ldap_msgfree(res);

     close_connection(server, FALSE);

     g_hash_table_remove(hash, "search lock");

}


void results_popup_menu(GHashTable *hash, GdkEventButton *event, struct resultset *set)
{
     GtkWidget *root_menu, *menu, *menu_item;

     /* this is a hack to pass the selected set under the menu to the callbacks.
	Each callback MUST clear this after use! */
     g_hash_table_insert(hash, "set", set);

     root_menu = gtk_menu_item_new_with_label("Root");
     gtk_widget_show(root_menu);

     menu = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(root_menu), menu);
     menu_item = gtk_tearoff_menu_item_new();
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_set_sensitive(menu_item, FALSE);
     gtk_widget_show(menu_item);

     /* Edit */
     menu_item = gtk_menu_item_new_with_label(_("Edit"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
			       GTK_SIGNAL_FUNC(search_edit_entry_callback),
			       (gpointer) hash);
     gtk_widget_show(menu_item);

     /* Use as template */
     menu_item = gtk_menu_item_new_with_label(_("Use as template"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
			       GTK_SIGNAL_FUNC(search_new_from_entry_callback),
			       (gpointer) hash);
     gtk_widget_show(menu_item);

     /* separator */
     menu_item = gtk_menu_item_new();
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);

     /* Delete */
     menu_item = gtk_menu_item_new_with_label(_("Delete"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
			       GTK_SIGNAL_FUNC(delete_search_entry),
			       (gpointer) hash);
     gtk_widget_show(menu_item);

     gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		    event->button, event->time);

}


void search_new_from_entry_callback(GHashTable *hash)
{
     struct ldapserver *server;
     struct resultset *set;

     set = g_hash_table_lookup(hash, "set");
     g_hash_table_remove(hash, "set");
     if(set == NULL || set->dn == NULL)
	  return;

     if( (server = g_hash_table_lookup(hash, "cur_resultset_server")) == NULL)
	  return;

     new_from_entry(server, set->dn);

}


void search_edit_entry_callback(GHashTable *hash)
{
     struct ldapserver *server;
     struct resultset *set;

     set = g_hash_table_lookup(hash, "set");
     g_hash_table_remove(hash, "set");
     if(set == NULL || set->dn == NULL)
	  return;

     if( (server = g_hash_table_lookup(hash, "cur_resultset_server")) == NULL)
	  return;

     edit_entry(server, set->dn);

}


void delete_search_entry(GHashTable *hash)
{
     struct ldapserver *server;
     struct resultset *set;

     set = g_hash_table_lookup(hash, "set");
     g_hash_table_remove(hash, "set");

     if(set == NULL || set->dn == NULL)
	  return;

     if( (server = g_hash_table_lookup(hash, "cur_resultset_server")) == NULL)
	  return;

     delete_entry(server, set->dn);

}


int select_entry_callback(GtkWidget *clist, gint row, gint column,
			   GdkEventButton *event, gpointer hash)
{
     struct resultset *set;

     if(event) {
	  set = gtk_clist_get_row_data(GTK_CLIST(clist), row);

	  if(event->button == 1 && event->type == GDK_2BUTTON_PRESS) {
	       g_hash_table_insert(hash, "set", set);
	       search_edit_entry_callback(hash);
	  }
     }

     return(TRUE);
}

#if 0   /* no longer needed */
int unselect_entry_callback(GtkWidget *clist, gint row, gint column,
			   GdkEventButton *event, gpointer hash)
{
     GtkCListRow *clistrow;
     GList *list;
     struct resultset *set;

     if(event && event->button == 3) {

	  list = g_list_nth(GTK_CLIST(clist)->row_list, row);
	  clistrow = list->data;
	  if(clistrow->state != GTK_STATE_SELECTED) {
	       set = gtk_clist_get_row_data(GTK_CLIST(clist), row);
	       results_popup_menu(hash, event, set);

	       /* why TF doesn't this work */
	       gtk_signal_emit_stop_by_name(GTK_OBJECT(clist), "unselect_row");
	  }

     }

     return(TRUE);
}
#endif

static gboolean search_button_press_on_tree_item(GtkWidget *clist,
						 GdkEventButton *event,
						 GHashTable *hash) 
{
     struct resultset *set;
     int rc, row, column;

     if (event->type == GDK_BUTTON_PRESS && event->button == 3
	 && event->window == GTK_CLIST(clist)->clist_window) {
	  rc = gtk_clist_get_selection_info(GTK_CLIST(clist), 
					    event->x, event->y, &row, &column);
	  if (rc == 0)
	       return TRUE;

	  set = gtk_clist_get_row_data(GTK_CLIST(clist), row);
	  results_popup_menu(hash, event, set);

	  gtk_signal_emit_stop_by_name(GTK_OBJECT(clist),
				       "button_press_event");
     }

     return TRUE;
}


void cleanup_search_mode(GHashTable *hash)
{
     GtkWidget *main_clist;
     struct resultset *cur_resultset;

     cur_resultset = g_hash_table_lookup(hash, "cur_resultset");
     if(cur_resultset) {
	  free_resultset(cur_resultset);
     }

     main_clist = g_hash_table_lookup(hash, "main_clist");
     if(main_clist) {
	  gtk_clist_clear(GTK_CLIST(main_clist));
	  gtk_widget_destroy(main_clist);
     }

}


/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */
