/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This file (dt_password.c) is
    Copyright (C) 2002 by Peter Stamfest and Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: dt_password.c,v 1.6 2002/04/16 08:28:23 stamfest Exp $ */

#include <config.h>

#ifdef HAVE_LIBCRYPTO

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include <openssl/rand.h>
#include <openssl/des.h>
#include <openssl/md5.h>
#include <openssl/sha.h>

#include "common.h"
#include "util.h"
#include "errorchain.h"
#include "formfill.h"
#include "input.h"
#include "tinput.h"
#include "browse.h"
#include "encode.h"
#include "ldif.h" /* for b64_decode */
#include "dt_password.h"

#if defined(HAVE_LIBCRYPTO)
static GByteArray *dt_password_encode_password_crypt(char *data, int len);
static GByteArray *dt_password_encode_password_md5(char *data, int len);
static GByteArray *dt_password_encode_password_sha1(char *data, int len);
static GByteArray *dt_password_encode_password_ssha1(char *data, int len);
static GByteArray *dt_password_encode_password_smd5(char *data, int len);

typedef GByteArray *(CryptFunc)(char *data, int len);
#endif

struct tokenlist cryptmap[] = {
     { 0,                 "Clear", NULL },
#if defined(HAVE_LIBCRYPTO)
     /* currently, we still need unique token codes, should drop this
        requirement */
     { FLAG_ENCODE_CRYPT, "Crypt", dt_password_encode_password_crypt },
     { FLAG_ENCODE_SSHA,  "SSHA",  dt_password_encode_password_ssha1 },
     { FLAG_ENCODE_SMD5,  "SMD5",  dt_password_encode_password_smd5  },
     { FLAG_ENCODE_MD5,   "MD5",   dt_password_encode_password_md5   },
     { FLAG_ENCODE_SHA,   "SHA",   dt_password_encode_password_sha1  },
#endif
     { 0,		  "",      NULL },
};



static dt_password_handler dt_password_handler_vtab = {
     {
	  dt_password_get_widget,
	  dt_password_get_data,
	  dt_password_set_data,
	  bervalLDAPMod			/* reuse method from dt_entry */
     },
     decode_utf8, /* encode method */	/* reuse method from dt_entry */
     encode_utf8, /* decode method */	/* reuse method from dt_entry */
};

display_type_handler *dt_password_get_handler() {
     return (display_type_handler *) &dt_password_handler_vtab;
}


#if defined(HAVE_LIBCRYPTO)

static GByteArray *dt_password_encode_password_crypt(char *data, int len)
{
     GString *salt;
     GByteArray *gb = g_byte_array_new();
     unsigned char *password, rand[16];

     gb = g_byte_array_set_size(gb, 64);

     password = (char*) g_malloc((len + 1) * sizeof(char));
     memcpy(password, data, len);
     password[len] = 0;

     salt = g_string_sized_new(32);
     RAND_pseudo_bytes(rand, 8);
     b64_encode(salt, rand, 8);
     /* crypt(3) says [a-zA-Z0-9./] while base64 has [a-zA-Z0-9+/] */
     if(salt->str[0] == '+') salt->str[0] = '.';
     if(salt->str[1] == '+') salt->str[1] = '.';
     salt->str[2] = 0;

     strcpy(gb->data, "{crypt}");
     des_fcrypt(password, salt->str, gb->data + strlen(gb->data));
     g_byte_array_set_size(gb, strlen(gb->data));

     g_string_free(salt, TRUE);
     g_free(password);

     return gb;
}

static GByteArray *dt_password_encode_password_md5(char *data, int len)
{
     char md5_out[MD5_DIGEST_LENGTH];
     GString *b64 = g_string_sized_new(MD5_DIGEST_LENGTH * 4 / 3 + 4);

     GByteArray *gb = g_byte_array_new();

     MD5(data, len, md5_out);
     b64_encode(b64, md5_out, MD5_DIGEST_LENGTH);

     gb = g_byte_array_set_size(gb, MD5_DIGEST_LENGTH + 10);

     strcpy(gb->data, "{MD5}");
     strcat(gb->data, b64->str);

     g_byte_array_set_size(gb, strlen(gb->data));

     g_string_free(b64, TRUE);

     return gb;
}

static GByteArray *dt_password_encode_password_sha1(char *data, int len)
{
     char sha_out[SHA_DIGEST_LENGTH];
     GString *b64 = g_string_sized_new(SHA_DIGEST_LENGTH * 4 / 3 + 4);

     GByteArray *gb = g_byte_array_new();

     SHA1(data, len, sha_out);
     b64_encode(b64, sha_out, SHA_DIGEST_LENGTH);

     gb = g_byte_array_set_size(gb, SHA_DIGEST_LENGTH + 10);

     strcpy(gb->data, "{SHA}");
     strcat(gb->data, b64->str);

     g_byte_array_set_size(gb, strlen(gb->data));

     g_string_free(b64, TRUE);

     return gb;
}


static GByteArray *dt_password_encode_password_ssha1(char *data, int len)
{
     unsigned char rand[4];
     unsigned char sha_out[SHA_DIGEST_LENGTH + sizeof(rand)];
     SHA_CTX  SHA1context;

     GString *b64 = g_string_sized_new(SHA_DIGEST_LENGTH * 4 / 3 + 4);
     GByteArray *gb = g_byte_array_new();
     RAND_pseudo_bytes(rand, sizeof(rand));

     SHA1_Init(&SHA1context);
     SHA1_Update(&SHA1context, data, len);
     SHA1_Update(&SHA1context, rand, sizeof(rand));
     SHA1_Final(sha_out, &SHA1context);

     memcpy(sha_out + SHA_DIGEST_LENGTH, rand, sizeof(rand));

     b64_encode(b64, sha_out, sizeof(sha_out));

     gb = g_byte_array_set_size(gb, SHA_DIGEST_LENGTH + 16);

     strcpy(gb->data, "{SSHA}");
     strcat(gb->data, b64->str);

     g_byte_array_set_size(gb, strlen(gb->data));

     g_string_free(b64, TRUE);

     return gb;
}

static GByteArray *dt_password_encode_password_smd5(char *data, int len)
{
     unsigned char rand[4];
     unsigned char md5_out[MD5_DIGEST_LENGTH + sizeof(rand)];
     MD5_CTX  MD5context;

     GString *b64 = g_string_sized_new(MD5_DIGEST_LENGTH * 4 / 3 + 4);
     GByteArray *gb = g_byte_array_new();
     RAND_pseudo_bytes(rand, sizeof(rand));

     MD5_Init(&MD5context);
     MD5_Update(&MD5context, data, len);
     MD5_Update(&MD5context, rand, sizeof(rand));
     MD5_Final(md5_out, &MD5context);

     memcpy(md5_out + MD5_DIGEST_LENGTH, rand, sizeof(rand));

     b64_encode(b64, md5_out, sizeof(md5_out));

     gb = g_byte_array_set_size(gb, MD5_DIGEST_LENGTH + 16);

     strcpy(gb->data, "{SMD5}");
     strcat(gb->data, b64->str);

     g_byte_array_set_size(gb, strlen(gb->data));

     g_string_free(b64, TRUE);

     return gb;
}

#endif /* HAVE_LIBCRYPTO */

GtkWidget *dt_password_get_widget(struct formfill *form, GByteArray *data,
				  GtkSignalFunc *activatefunc,
				  GHashTable *hash) 
{
     GtkWidget *hbox;
     GtkWidget *inputbox;
     GtkWidget *combo;
     GtkStyle *style;

     GList *cryptlist = NULL;
     int temp, max_width, i;


     hbox = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox);
    
     inputbox = gtk_entry_new();
     gtk_widget_show(inputbox);


     gtk_box_pack_start(GTK_BOX(hbox), inputbox, TRUE, TRUE, 0);
    
     /* XXX this will need an encoder wedge */
     if(activatefunc)
	  gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
				    GTK_SIGNAL_FUNC(activatefunc),
				    (gpointer) hash);
    
     combo = gtk_combo_new();
     GTK_WIDGET_UNSET_FLAGS(GTK_COMBO(combo)->entry, GTK_CAN_FOCUS);
     style = gtk_widget_get_style(GTK_COMBO(combo)->entry);

     cryptlist = NULL;
     temp = max_width = 0;

     for(i = 0 ; cryptmap[i].keyword[0] ; i++) {
	  temp = gdk_string_width(style->font, cryptmap[i].keyword);
	  if (temp > max_width)
	       max_width = temp;
	  cryptlist = g_list_append(cryptlist, cryptmap[i].keyword);
     }
     gtk_combo_set_popdown_strings(GTK_COMBO(combo), cryptlist);
     gtk_widget_set_usize(GTK_COMBO(combo)->entry, max_width + 20, -1);
     g_list_free(cryptlist);

     gtk_widget_show(combo);
     gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);

     dt_password_set_data(form, data, hbox);

     return hbox;
}


void dt_password_set_data(struct formfill *form, GByteArray *data, 
			  GtkWidget *hbox) 
{
     char *crypt_type = "Clear";
     GList *boxchildren;
     GtkWidget *entry, *combo;
     int i;

     boxchildren = GTK_BOX(hbox)->children;
     entry = ((struct _GtkBoxChild *) boxchildren->data)->widget;
     combo = ((struct _GtkBoxChild *) boxchildren->next->data)->widget;

     if(data) {
	  GByteArray *encoded = 
	      DT_PASSWORD(form->dt_handler)->encode(data->data, data->len);
	  gtk_entry_set_text(GTK_ENTRY(entry), encoded->data);
	  g_byte_array_free(encoded, TRUE);

	  if(data->data[0] == '{') {
	       char crypt_prefix[12];
	       for(i = 1; data->data[i] && data->data[i] != '}' && i < 10; i++)
		    crypt_prefix[i - 1] = data->data[i];
	       crypt_prefix[i - 1] = '\0';
	       crypt_type = detokenize(cryptmap, 
				       tokenize(cryptmap, crypt_prefix));
	  }
     }
     /* set the combo to the encoding type */
     gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), crypt_type);
}

GByteArray *dt_password_get_data(struct formfill *form, GtkWidget *hbox) 
{
     GByteArray *copy;
     GtkWidget *combo;
     GList *boxchildren;
     gchar *content;
     gchar *crypt_type;
     int l;

     boxchildren = GTK_BOX(hbox)->children;
     content = gtk_editable_get_chars(GTK_EDITABLE(((struct _GtkBoxChild *) boxchildren->data)->widget), 0, -1);

     if (!content) return NULL;

     l = strlen(content);
     if (l == 0) {
	  g_free(content);
	  return NULL;
     }

     copy = DT_PASSWORD(form->dt_handler)->decode(content, 
						  strlen(content));

     if(boxchildren->next) {
	  int cryptflag;
	  /* advance to crypt combo */
	  boxchildren = boxchildren->next;
	  combo = ((struct _GtkBoxChild *) boxchildren->data)->widget;
	  crypt_type = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combo)->entry), 0, -1);
	  

	  cryptflag = tokenize(cryptmap, crypt_type);

	  /* check to see if the content was already crypted.... */
	  if (copy && copy->data[0] != '{') {
	       /* need to crypt the stuff */
	       CryptFunc *cryptfunc = detokenize_data(cryptmap, cryptflag);
	       if (cryptfunc != NULL) {
		    GByteArray *crypted = cryptfunc(copy->data, copy->len);
		    g_byte_array_free(copy, TRUE);
		    copy = crypted;
	       }
	  }
	  
	  g_free(crypt_type);
     }

/*       printf("getting orgtext '%s' text '%s' orglen=%d len1=%d len2=%d\n", */
/*  	    content, */
/*  	    copy->data, */
/*  	    strlen(content), */
/*  	    copy->len, strlen(copy->data)); */

     g_free(content);

     return copy;
}

#endif /* HAVE_LIBCRYPTO */

/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */

