/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>

#include <glib.h>

#include <lber.h>
#include <ldap.h>

#include "common.h"
#include "util.h"
#include "formfill.h"

extern struct tokenlist cryptmap[];


struct formfill *new_formfill(void)
{
     struct formfill *form;

     if( (form = (struct formfill *) g_malloc(sizeof(struct formfill))) == NULL)
	  return(NULL);
     form->attrname[0] = '\0';
     form->num_inputfields = 1;
     form->displaytype = DISPLAYTYPE_ENTRY;
     form->flags = 0;
     form->values = NULL;

     return(form);
}


void free_formlist(GList *formlist)
{

     while(formlist) {
	  free_formfill( (struct formfill *) formlist->data);
	  formlist = formlist->next;
     }
     g_list_free(formlist);

}


void free_formfill(struct formfill *form)
{
     GList *vals;

     if((vals = form->values) != NULL) {
	  while(vals) {
	       g_free(vals->data);
	       vals = vals->next;
	  }
	  g_list_free(form->values);
     }

     g_free(form);

}


/*
 * add form to formlist, checking first if the attribute type isn't
 * already in the list
 */
GList *formlist_append(GList *formlist, struct formfill *form)
{
     GList *fl;
     struct formfill *tmp;

     fl = formlist;
     while(fl) {

	  tmp = (struct formfill *) fl->data;
	  if(!strcmp(tmp->attrname, form->attrname)) {
	       /* already have this attribute */
	       free_formfill(form);
	       return(formlist);
	  }

	  fl = fl->next;
     }

     formlist = g_list_append(formlist, form);

     return(formlist);
}


GList *formlist_from_entry(struct ldapserver *server, char *dn, int ocvalues_only)
{
     GList *formlist;
     GString *message;
     LDAP *ld;
     LDAPMessage *res, *entry;
     BerElement *ber;
     struct formfill *form;
     int msg, i;
     char *attr, **vals;

     formlist = NULL;

     set_busycursor();

     message = g_string_sized_new(128);
     g_string_sprintf(message, "fetching %s from %s", dn, server->name);
     statusbar_msg(message->str);
     g_string_free(message, TRUE);

     if( (ld = open_connection(server)) == NULL) {
          set_normalcursor();
          return(NULL);
     }

     msg = ldap_search_s(ld, dn, LDAP_SCOPE_BASE, "(objectclass=*)", NULL, 0, &res);

     if(msg == -1) {
	  statusbar_msg(ldap_err2string(msg));
	  set_normalcursor();
	  return(NULL);
     }

     if (res) {
	  entry = ldap_first_entry(ld, res);
	  if(entry) {
	       for(attr = ldap_first_attribute(ld, entry, &ber); attr; attr = ldap_next_attribute(ld, entry, ber)) {
		    if( (form = new_formfill()) == NULL)
			 break;
		    strncpy(form->attrname, attr, MAX_ATTR_LEN);
		    if(ocvalues_only == 0 || !strcasecmp(attr, "objectClass")) {
			 vals = ldap_get_values(ld, res, attr);
			 if(vals) {
			      for(i = 0; vals[i]; i++)
				   form->values = g_list_append(form->values, g_strdup(vals[i]));
			      form->num_inputfields = i;
			      ldap_value_free(vals);
			 }
		    }
		    set_displaytype(form);
		    formlist = g_list_append(formlist, form);
	       }
	       /* this bombs out with openldap 1.2.x libs... */
	       /*
		 if(ber)
		      ber_free(ber, 0);
	       */
	  }
	  ldap_msgfree(res);
     }

     close_connection(server, FALSE);
     set_normalcursor();

     return(formlist);
}


GList *dup_formlist(GList *oldlist)
{
     GList *newlist, *oldval, *newval;
     struct formfill *oldform, *newform;

     newlist = NULL;

     while(oldlist) {
	  oldform = (struct formfill *) oldlist->data;
	  if( (newform = new_formfill()) == NULL)
	       break;
	  strcpy(newform->attrname, oldform->attrname);
	  newform->num_inputfields = oldform->num_inputfields;
	  newform->displaytype = oldform->displaytype;
	  newform->flags = oldform->flags;
	  oldval = oldform->values;
	  newval = NULL;
	  while(oldval) {
	       if(oldval->data)
		    newval = g_list_append(newval, g_strdup((char *) (oldval->data)));

	       oldval = oldval->next;
	  }
	  newform->values = newval;
	  newlist = g_list_append(newlist, newform);

	  oldlist = oldlist->next;
     }

     return(newlist);
}


void dump_hash(GHashTable *hash)
{
     GList *formlist;
     char *dn;

     dn = g_hash_table_lookup(hash, "dn");
     if(dn)
	  printf("dn: %s\n", dn);

     formlist = g_hash_table_lookup(hash, "formlist");
     if(!formlist)
	  return;

     dump_formlist(formlist);

}


void dump_formlist(GList *formlist)
{
     GList *values;
     struct formfill *form;

     while(formlist) {
	  form = (struct formfill *) formlist->data;
	  printf("%s\n", form->attrname);
	  values = form->values;

	  while(values) {
	       if(values->data)
		    printf("\t%s\n", (char *) values->data);
	       values = values->next;
	  }

	  formlist = formlist->next;
     }

}


struct formfill *lookup_attribute(GList *formlist, char *attr)
{
     struct formfill *form;

     while(formlist) {
	  form = (struct formfill *) formlist->data;
	  if(form->attrname && !strcasecmp(form->attrname, attr))
	       return(form);

	  formlist = formlist->next;
     }

     return(NULL);
}


/*
 * sure would be nice to use serverschema for this
 *
 * hint, hint
 *
 */
void set_displaytype(struct formfill *form)
{
     int i;
     char *password, crypt_prefix[12];

     if(!strcasecmp(form->attrname, "userPassword")) {
	  form->displaytype = DISPLAYTYPE_PASSWORD;

	  /* see if we can find a {crypt prefix} in the password, and set flags accordingly */
	  form->flags &= !ENCODING_MASK;
	  if(form->values) {
	       password = form->values->data;
	       if(password && password[0] == '{') {
		    for(i = 1; password[i] && password[i] != '}' && i < 10; i++)
			 crypt_prefix[i - 1] = password[i];
		    crypt_prefix[i - 1] = '\0';

	       }
	       form->flags |= tokenize(cryptmap, crypt_prefix);
	  }

     }
     else {
	  form->displaytype = DISPLAYTYPE_ENTRY;
     }

}


/****************** unused functions *********************/



/*
 * arrow next to inputfield hit: add empty field below this one
 */
void add_field_to_formfill(GList *formlist, char *attr)
{
     struct formfill *form;

     while(formlist) {

	  form = (struct formfill *) formlist->data;
	  if(!strcmp(form->attrname, attr)) {
	       form->num_inputfields++;
	       break;
	  }

	  formlist = formlist->next;
     }

}


