/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <string.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include <config.h>

#include "common.h"
#include "util.h"
#include "errorchain.h"
#include "formfill.h"
#include "input.h"
#include "tinput.h"
#include "browse.h"
#include "encode.h"

#include "../icons/line.xpm"
#include "../icons/textarea.xpm"


struct tokenlist cryptmap[] = {
     { 0,                 "Clear" },
#if defined(HAVE_LIBCRYPTO)
     { FLAG_ENCODE_CRYPT, "Crypt" },
     { FLAG_ENCODE_MD5,   "MD5"   },
     { FLAG_ENCODE_SHA,   "SHA"   },
#endif
     { 0, "" }
};



void create_form_window(GHashTable *hash)
{
     GtkWidget *edit_window, *vbox;

     edit_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title(GTK_WINDOW(edit_window), "New entry");
     gtk_widget_set_usize(edit_window, 500, 450);
     gtk_widget_show(edit_window);
     gtk_signal_connect_object(GTK_OBJECT(edit_window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) edit_window);

     vbox = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox), 5);
     gtk_container_add(GTK_CONTAINER(edit_window), vbox);
     gtk_widget_show(vbox);

     g_hash_table_insert(hash, "parent window", edit_window);
     g_hash_table_insert(hash, "target_vbox", vbox);

}


void create_form_content(GHashTable *hash)
{
     gpointer edit;
     GdkPixmap *icon;
     GdkBitmap *icon_mask;
     GtkWidget *target_vbox, *target_window, *vbox2, *hbox1, *hbox2;
     GtkWidget *button, *linebutton, *textareabutton;
     GtkWidget *scwin, *table;
     GtkWidget *pixmap;
     int detail_context;

     detail_context = error_new_context("Error getting entry");
     target_vbox = g_hash_table_lookup(hash, "target_vbox");

     hbox1 = gtk_hbox_new(FALSE, 5);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(target_vbox), hbox1, FALSE, FALSE, 0);

     /* line button */
     linebutton = gtk_button_new();
     gtk_object_set_data(GTK_OBJECT(linebutton), "transform", "make entry");
     GTK_WIDGET_UNSET_FLAGS(linebutton, GTK_CAN_FOCUS);
     icon = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(target_vbox)->window,
                                         &icon_mask,
                                         &target_vbox->style->white,
                                         line_xpm);
     pixmap = gtk_pixmap_new(icon, icon_mask);
     gtk_widget_show(pixmap);
     gtk_container_add(GTK_CONTAINER(linebutton), pixmap);
     gtk_widget_show(linebutton);
     gtk_box_pack_start(GTK_BOX(hbox1), linebutton, FALSE, FALSE, 5);
     gtk_signal_connect(GTK_OBJECT(linebutton), "clicked",
                        GTK_SIGNAL_FUNC(change_displaytype),
                        (gpointer) hash);

     /* textarea button */
     textareabutton = gtk_button_new();
     gtk_object_set_data(GTK_OBJECT(textareabutton), "transform", "make text");
     GTK_WIDGET_UNSET_FLAGS(textareabutton, GTK_CAN_FOCUS);
     icon = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(target_vbox)->window,
                                         &icon_mask,
                                         &target_vbox->style->white,
                                         textarea_xpm);
     pixmap = gtk_pixmap_new(icon, icon_mask);
     gtk_widget_show(pixmap);
     gtk_container_add(GTK_CONTAINER(textareabutton), pixmap);
     gtk_widget_show(textareabutton);
     gtk_box_pack_start(GTK_BOX(hbox1), textareabutton, FALSE, FALSE, 0);
     gtk_signal_connect(GTK_OBJECT(textareabutton), "clicked",
                        GTK_SIGNAL_FUNC(change_displaytype),
                        (gpointer) hash);

     /* scrolled window with vbox2 inside */
     scwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_container_border_width(GTK_CONTAINER(scwin), 5);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(scwin);
     gtk_box_pack_start(GTK_BOX(target_vbox), scwin, TRUE, TRUE, 5);
     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scwin), vbox2);
     g_hash_table_insert(hash, "vbox_holding_table", vbox2);

     /* table inside vbox2, will self-expand */
     table = gtk_table_new(3, 2, FALSE);
     gtk_container_border_width(GTK_CONTAINER(table), 5);
     gtk_table_set_row_spacings(GTK_TABLE(table), 1);
     gtk_table_set_col_spacings(GTK_TABLE(table), 10);
     gtk_widget_show(table);
     gtk_box_pack_start(GTK_BOX(vbox2), table, FALSE, TRUE, 0);
     g_hash_table_insert(hash, "table", table);

     hbox2 = gtk_hbox_new(TRUE, 0);
     gtk_widget_show(hbox2);
     gtk_box_pack_end(GTK_BOX(target_vbox), hbox2, FALSE, TRUE, 10);

     edit = g_hash_table_lookup(hash, "edit");
     if(edit) {
	  button = gtk_button_new_with_label("  Apply  ");
          gtk_widget_show(button);
          gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
          gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                                    GTK_SIGNAL_FUNC(mod_entry_from_formlist),
                                    (gpointer) hash);
          GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
          GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
          gtk_widget_grab_default(button);

          button = gtk_button_new_with_label("  Refresh  ");
	  gtk_widget_set_sensitive(button, FALSE);
          GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
          gtk_widget_show(button);
          gtk_box_pack_end(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
/*
          gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                                    GTK_SIGNAL_FUNC(refresh_right_pane),
                                    NULL);
*/

	  /* this will be used as the callback to set on "activate" on
	     inputfields (i.e. hitting return in an entry box). */
	  g_hash_table_insert(hash, "activate", mod_entry_from_formlist);

     }
     else {
          button = gtk_button_new_with_label("  OK  ");
          gtk_widget_show(button);
          gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
          gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                                    GTK_SIGNAL_FUNC(add_entry_from_formlist),
                                    (gpointer) hash);
          GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
          gtk_widget_grab_default(button);

          button = gtk_button_new_with_label("  Cancel  ");
          gtk_widget_show(button);
          gtk_box_pack_end(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
	  target_window = g_hash_table_lookup(hash, "parent window");
          gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                                    GTK_SIGNAL_FUNC(destroy_editwindow),
                                    (gpointer) hash);

	  g_hash_table_insert(hash, "activate", add_entry_from_formlist);

     }

     statusbar_msg("");
     error_flush(detail_context);

}


void build_inputform(GHashTable *hash)
{
     GList *values, *formlist, *cryptlist;
     GdkColor red = { 0, 0xffff, 0x0000, 0x0000 };
     GtkStyle *style, *not_in_schema;
     GtkWidget *target_table, *hbox, *label, *inputbox, *arrowbutton, *combo;
     struct formfill *form, *focusform;
     void *activatefunc;
     int row, row_arrow, vcnt, i, temp, max_width;
     char *crypt_type, *dn;

     gdk_color_alloc(gdk_colormap_get_system(), &red);
     not_in_schema = gtk_style_copy(gtk_widget_get_default_style());
     not_in_schema->fg[GTK_STATE_NORMAL] = red;

     /* mind form->num_inputfields and len(GList * values) may not
	be the same -- if adding a field with an arrowbutton */

     target_table = g_hash_table_lookup(hash, "table");
     formlist = g_hash_table_lookup(hash, "formlist");
     focusform = g_hash_table_lookup(hash, "focusform");
     activatefunc = g_hash_table_lookup(hash, "activate");

     /* DN input box */
     label = gtk_label_new("dn");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(target_table), label, 0, 1, 0, 1,
		      0, 0, 0, 0);

     inputbox = gtk_entry_new();
     gtk_widget_show(inputbox);
     gtk_object_set_data(GTK_OBJECT(inputbox), "displaytype", GINT_TO_POINTER(DISPLAYTYPE_DN));
     gtk_table_attach(GTK_TABLE(target_table), inputbox, 1, 2, 0, 1,
		      GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
     if(activatefunc)
	  gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
				    GTK_SIGNAL_FUNC(activatefunc), (gpointer) hash);
     dn = g_hash_table_lookup(hash, "dn");
     if(dn)
	  gtk_entry_set_text(GTK_ENTRY(inputbox), dn);
     /* set focus on DN for now, will get overridden by focusform in hash if present */
     gtk_editable_set_position(GTK_EDITABLE(inputbox), 0);
     gtk_widget_grab_focus(inputbox);

     row = 1;
     while(formlist) {
	  form = (struct formfill *) formlist->data;

	  /* attribute name */
	  label = gtk_label_new(form->attrname);
	  gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
	  if(form->flags & FLAG_NOT_IN_SCHEMA)
	       gtk_widget_set_style(label, not_in_schema);
	  gtk_widget_show(label);
	  gtk_table_attach(GTK_TABLE(target_table), label, 0, 1, row, row + 1,
			   0, 0, 0, 0);

	  row_arrow = row;

	  vcnt = 0;
	  values = form->values;
	  while(vcnt < form->num_inputfields) {
	       inputbox = NULL;
	       switch(form->displaytype) {

	       case DISPLAYTYPE_ENTRY:
		    inputbox = gtk_entry_new();
		    gtk_widget_show(inputbox);
		    gtk_object_set_data(GTK_OBJECT(inputbox), "formfill", form);
		    gtk_object_set_data(GTK_OBJECT(inputbox), "displaytype", GINT_TO_POINTER(DISPLAYTYPE_ENTRY));
		    gtk_table_attach(GTK_TABLE(target_table), inputbox, 1, 2, row, row + 1,
				     GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
		    if(activatefunc)
			 gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
						   GTK_SIGNAL_FUNC(activatefunc), (gpointer) hash);
		    if(values) {
			 gtk_entry_set_text(GTK_ENTRY(inputbox), values->data);
			 values = values->next;
		    }
		    break;

	       case DISPLAYTYPE_TEXT:
		    inputbox = gtk_text_new(NULL, NULL);
		    gtk_widget_show(inputbox);
		    gtk_object_set_data(GTK_OBJECT(inputbox), "formfill", form);
		    gtk_object_set_data(GTK_OBJECT(inputbox), "displaytype", GINT_TO_POINTER(DISPLAYTYPE_TEXT));
		    gtk_table_attach(GTK_TABLE(target_table), inputbox, 1, 2, row, row + 1,
				     GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
		    gtk_text_set_editable(GTK_TEXT(inputbox), TRUE);
		    if(values) {
			 gtk_text_freeze(GTK_TEXT(inputbox));
			 gtk_text_set_point(GTK_TEXT(inputbox), 0);
			 gtk_text_insert(GTK_TEXT(inputbox), NULL, NULL, NULL, values->data, -1);
			 gtk_text_thaw(GTK_TEXT(inputbox));
			 values = values->next;
		    }
		    break;

	       case DISPLAYTYPE_PASSWORD:
		    hbox = gtk_hbox_new(FALSE, 0);
		    gtk_widget_show(hbox);
		    gtk_object_set_data(GTK_OBJECT(hbox), "formfill", form);
		    gtk_object_set_data(GTK_OBJECT(hbox), "displaytype", GINT_TO_POINTER(DISPLAYTYPE_PASSWORD));

		    inputbox = gtk_entry_new();
		    gtk_widget_show(inputbox);
		    gtk_box_pack_start(GTK_BOX(hbox), inputbox, TRUE, TRUE, 0);

		    /* XXX this will need an encoder wedge */
		    if(activatefunc)
			 gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
						   GTK_SIGNAL_FUNC(activatefunc), (gpointer) hash);

		    combo = gtk_combo_new();
		    GTK_WIDGET_UNSET_FLAGS(GTK_COMBO(combo)->entry, GTK_CAN_FOCUS);
		    style = gtk_widget_get_style(GTK_COMBO(combo)->entry);
		    cryptlist = NULL;
		    temp = max_width = 0;
		    for(i = 0; cryptmap[i].keyword[0]; i++) {
			 temp = gdk_string_width(style->font, cryptmap[i].keyword);
			 if (temp > max_width)
			      max_width = temp;
			 cryptlist = g_list_append(cryptlist, cryptmap[i].keyword);
		    }
		    gtk_combo_set_popdown_strings(GTK_COMBO(combo), cryptlist);
		    gtk_widget_set_usize(GTK_COMBO(combo)->entry, max_width + 20, -1);
		    g_list_free(cryptlist);

		    /* set the combo to the encoding type */
		    crypt_type = detokenize(cryptmap, form->flags & ENCODING_MASK);
		    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), crypt_type);

		    gtk_widget_show(combo);
		    gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);

		    gtk_table_attach(GTK_TABLE(target_table), hbox, 1, 2, row, row + 1,
				     GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);

		    if(values) {
			 gtk_entry_set_text(GTK_ENTRY(inputbox), values->data);
			 values = values->next;
		    }
		    break;

	       }
	       if(inputbox) {
		    if(form == focusform)
			 gtk_widget_grab_focus(inputbox);
	       }

	       vcnt++;
	       row_arrow = row++;
	  }

	  arrowbutton = gq_new_arrowbutton(hash);
	  gtk_object_set_data(GTK_OBJECT(arrowbutton), "formfill", form);
	  gtk_table_attach(GTK_TABLE(target_table), arrowbutton,
			   2, 3, row_arrow, row_arrow + 1,
			   0, GTK_EXPAND, 5, 0);

	  formlist = formlist->next;
     }

     gtk_style_unref(not_in_schema);

}


void edit_entry(struct ldapserver *server, char *dn)
{
     GHashTable *formhash;
     GList *oldlist, *newlist;
     GtkWidget *edit_window, *vbox;

     edit_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title(GTK_WINDOW(edit_window), dn);
     gtk_widget_set_usize(edit_window, 500, 450);
     gtk_widget_show(edit_window);
     gtk_signal_connect_object(GTK_OBJECT(edit_window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) edit_window);

     vbox = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox), 5);
     gtk_container_add(GTK_CONTAINER(edit_window), vbox);
     gtk_widget_show(vbox);

     formhash = g_hash_table_new(g_str_hash, g_str_equal);
     g_hash_table_insert(formhash, "parent window", edit_window);
     g_hash_table_insert(formhash, "target_vbox", vbox);

     oldlist = formlist_from_entry(server, dn, 0);
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     oldlist = add_schema_attrs(server, oldlist);
#endif

     if(oldlist) {
	  newlist = dup_formlist(oldlist);
	  g_hash_table_insert(formhash, "edit", GINT_TO_POINTER(1));
	  g_hash_table_insert(formhash, "close window", GINT_TO_POINTER(1));
	  g_hash_table_insert(formhash, "server", server);
          g_hash_table_insert(formhash, "dn", g_strdup(dn));
          g_hash_table_insert(formhash, "olddn", g_strdup(dn));
          g_hash_table_insert(formhash, "formlist", newlist);
          g_hash_table_insert(formhash, "oldlist", oldlist);

	  create_form_content(formhash);
	  build_inputform(formhash);
     }

}


void new_from_entry(struct ldapserver *server, char *dn)
{
     GHashTable *hash;
     GList *formlist;
     int i;
     char *newdn, **oldrdn;

     hash = g_hash_table_new(g_str_hash, g_str_equal);
     g_hash_table_insert(hash, "server", server);

#ifdef HAVE_LDAP_STR2OBJECTCLASS
     /* if schema stuff is available, construct formlist based on the objectclasses
	in the entry */
     formlist = formfill_from_entry_objectclass(server, dn);
#else
     /* no schema stuff: just use the current entry then, throw away all values
	except those in the objectclass attribute */
     formlist = formlist_from_entry(server, dn, 1);
#endif
     if(formlist) {
	  g_hash_table_insert(hash, "formlist", formlist);

	  /* don't need the RDN of the current entry */
	  newdn = g_malloc(strlen(dn));
	  newdn[0] = 0;
	  oldrdn = ldap_explode_dn(dn, 0);
	  for(i = 1; oldrdn[i]; i++) {
	       strcat(newdn, ",");
	       strcat(newdn, oldrdn[i]);
	  }
	  ldap_value_free(oldrdn);
	  g_hash_table_insert(hash, "dn", newdn);

	  create_form_window(hash);
	  create_form_content(hash);
	  build_inputform(hash);
     }
     else {
	  g_hash_table_destroy(hash);
     }

}


/*
 * update formlist from on-screen table
 */
void update_formlist(GHashTable *hash)
{
     GList *formlist, *fl, *children, *boxchildren;
     GtkTable *table;
     GtkTableChild *child;
     GtkBox *hbox;
     GtkWidget *combo;
     struct formfill *form;
     int displaytype;
     char *content, *crypt_type, *old_dn;

     formlist = (GList *) g_hash_table_lookup(hash, "formlist");
     if(formlist == NULL)
	  return;

     /* clear any values in formlist, they'll be overwritten shortly
	with more authorative (newer) data anyway */
     fl = formlist;
     while(fl) {
	  form = (struct formfill *) fl->data;
	  if(form->values) {
	       /* XXX leaking g_strdup()ed values from gtk_editable_get_chars here
		  should loop the GList first */
	       g_list_free(form->values);
	       form->values = NULL;
	  }
	  fl = fl->next;
     }

     table = (GtkTable *) g_hash_table_lookup(hash, "table");
     if(table == NULL)
	  return;

     /* g_tables store the list of children bottom-to-top for some reason */
     children = g_list_last(table->children);
     while(children) {
	  child = children->data;

	  form = (struct formfill *) gtk_object_get_data(GTK_OBJECT(child->widget), "formfill");
	  displaytype = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(child->widget), "displaytype"));
	  if(form && displaytype) {
	       content = NULL;

	       /* gtk_editable_get_chars does a g_strdup()... */
	       switch(displaytype) {
	       case DISPLAYTYPE_ENTRY:
		    content = gtk_editable_get_chars(GTK_EDITABLE(child->widget), 0, -1);
		    break;
	       case DISPLAYTYPE_TEXT:
		    content = gtk_editable_get_chars(GTK_EDITABLE(child->widget), 0, -1);
		    break;
	       case DISPLAYTYPE_PASSWORD:
		    hbox = (GtkBox *) child->widget;
		    boxchildren = hbox->children;
		    content = gtk_editable_get_chars(GTK_EDITABLE(((struct _GtkBoxChild *) boxchildren->data)->widget), 0, -1);
		    if(boxchildren->next) {
			 /* advance to crypt combo */
			 boxchildren = boxchildren->next;
			 combo = ((struct _GtkBoxChild *) boxchildren->data)->widget;
			 crypt_type = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combo)->entry), 0, -1);
			 form->flags &= !ENCODING_MASK;
			 form->flags |= tokenize(cryptmap, crypt_type);
		    }
		    break;
	       default:

	       }

	       /* don't bother adding in empty fields */
	       if(content && strlen(content))
		    form->values = g_list_append(form->values, content);
	  }
	  else if(displaytype == DISPLAYTYPE_DN) {
	       content = gtk_editable_get_chars(GTK_EDITABLE(child->widget), 0, -1);
	       old_dn = g_hash_table_lookup(hash, "dn");
	       if(old_dn)
		    g_free(old_dn);
	       g_hash_table_insert(hash, "dn", content);
	  }

	  children = g_list_previous(children);
     }

}


void clear_table(GHashTable *hash)
{
     GtkWidget *vbox, *old_table, *new_table;

     vbox = g_hash_table_lookup(hash, "vbox_holding_table");
     old_table = g_hash_table_lookup(hash, "table");

     gtk_widget_destroy(old_table);

     /* table inside vbox, will self-expand */
     new_table = gtk_table_new(3, 2, FALSE);
     gtk_container_border_width(GTK_CONTAINER(new_table), 5);
     gtk_table_set_row_spacings(GTK_TABLE(new_table), 1);
     gtk_table_set_col_spacings(GTK_TABLE(new_table), 10);
     gtk_widget_show(new_table);
     gtk_box_pack_start(GTK_BOX(vbox), new_table, FALSE, TRUE, 0);
     g_hash_table_insert(hash, "table", new_table);

}


void add_entry_from_formlist(GHashTable *hash)
{
     LDAP *ld;
     LDAPMod **mods, *mod;
     GList *formlist, *values;
     struct ldapserver *server;
     struct formfill *form;
     int add_context, res, cval, cmod;
     char *dn;

     add_context = error_new_context("Error adding entry");

     /* handle impossible errors first */
     formlist = g_hash_table_lookup(hash, "formlist");
     if(!formlist) {
	  error_push(add_context, "Hmmm, no formlist to build entry! Please notify bert@biot.com");
	  error_flush(add_context);
	  return;
     }
     server = g_hash_table_lookup(hash, "server");
     if(!server) {
	  error_push(add_context, "Hmmm, no server! Please notify bert@biot.com");
	  error_flush(add_context);
	  return;
     }

     update_formlist(hash);

     dn = g_hash_table_lookup(hash, "dn");
     if(!dn || dn[0] == 0) {
	  error_push(add_context, "You must enter a DN for a new entry.");
	  error_flush(add_context);
	  return;
     }

#if defined(HAVE_LIBCRYPTO)
     encode_password(hash);
#endif

     set_busycursor();

     ld = open_connection(server);
     if(!ld) {
	  set_normalcursor();
	  return;
     }

     /* build LDAPMod from formlist */
     cmod = 0;
     mods = g_malloc(sizeof(void *) * (g_list_length(formlist) + 1));
     while(formlist) {
	  form = (struct formfill *) formlist->data;
	  mod = g_malloc(sizeof(LDAPMod));
	  mod->mod_op = LDAP_MOD_ADD;
	  mod->mod_type = g_strdup(form->attrname);
	  mod->mod_values = g_malloc(sizeof(char *) * (g_list_length(form->values) + 1));
	  values = form->values;
	  cval = 0;
	  while(values) {
	       mod->mod_values[cval++] = g_strdup(values->data);
	       values = values->next;
	  }
	  mod->mod_values[cval] = NULL;

	  if(cval == 0) {
	       /* don't add an attribute without values */
	       g_free(mod->mod_values);
	       g_free(mod);
	  }
	  else {
	       mods[cmod++] = mod;
	  }

	  formlist = formlist->next;
     }
     mods[cmod] = NULL;

     res = ldap_add_s(ld, dn, mods);
     ldap_mods_free(mods, 1);
     if(res != LDAP_SUCCESS) {
	  error_push(add_context, ldap_err2string(res));
	  error_flush(add_context);
	  set_normalcursor();
	  return;
     }

     error_flush(add_context);

     destroy_editwindow(hash);

     set_normalcursor();

}


void mod_entry_from_formlist(GHashTable *hash)
{
     GList *newlist, *oldlist;
     GString *message;
     LDAPMod **mods;
     LDAP *ld;
     struct ldapserver *server;
     int mod_context, res;
     char *olddn, *dn;

     update_formlist(hash);

     mod_context = error_new_context("Problem modifying entry");

     olddn = g_hash_table_lookup(hash, "olddn");
     dn = g_hash_table_lookup(hash, "dn");

     if(strcasecmp(olddn, dn)) {
	  if(change_rdn(hash, mod_context) == 1) {
	       error_flush(mod_context);
	       return;
	  }
	  /* refresh visual if requested by browse mode */
	  if(g_hash_table_lookup(hash, "ctree_refresh"))
	       refresh_subtree(NULL, hash);
     }

     server = g_hash_table_lookup(hash, "server");
     if( (ld = open_connection(server)) == NULL)
	  return;

     oldlist = g_hash_table_lookup(hash, "oldlist");
     newlist = g_hash_table_lookup(hash, "formlist");

#if defined(HAVE_LIBCRYPTO)
     encode_password(hash);
#endif

     mods = formdiff_to_ldapmod(oldlist, newlist);
     if(mods == NULL || mods[0] == NULL)
	  return;

     res = ldap_modify_s(ld, dn, mods);
     if(res != LDAP_SUCCESS) {
	  error_push(mod_context, ldap_err2string(res));
     }
     else {
	  message = g_string_sized_new(256);
	  g_string_sprintfa(message, "modified %s", dn);
	  statusbar_msg(message->str);
	  g_string_free(message, 1);
	  if(g_hash_table_lookup(hash, "close window"))
	       destroy_editwindow(hash);
     }

     error_flush(mod_context);

}


int change_rdn(GHashTable *hash, int context)
{
     GString *message;
     LDAP *ld;
     struct ldapserver *server;
     int error, res, i;
     char *olddn, *dn;
     char **oldrdn, **rdn;

     server = g_hash_table_lookup(hash, "server");
     if( (ld = open_connection(server)) == NULL)
	  return(1);

     olddn = g_hash_table_lookup(hash, "olddn");
     dn = g_hash_table_lookup(hash, "dn");

     oldrdn = ldap_explode_dn(olddn, 0);
     rdn = ldap_explode_dn(dn, 0);

     message = g_string_sized_new(256);

     error = 0;
     for(i = 1; rdn[i]; i++) {
	  if(oldrdn[i] == NULL || rdn[i] == NULL || strcasecmp(oldrdn[i], rdn[i])) {
	       g_string_sprintf(message, "You can only change the RDN of the DN (%s)",
				 oldrdn[0]);
	       error_push(context, message->str);
	       error = 1;
	       break;
	  }
     }

     if(!error) {
	  g_string_sprintfa(message, "modifying RDN to %s", rdn[0]);
	  statusbar_msg(message->str);

	  res = ldap_modrdn2_s(ld, olddn, rdn[0], 1);
	  if(res != LDAP_SUCCESS) {
	       g_string_sprintf(message, "ldap_modrdn2_s: %s", ldap_err2string(res));
	       error_push(context, message->str);
	       error = 1;
	  }
     }

     g_string_free(message, TRUE);

     return(error);
}


/*
 * convert the differences between oldlist and newlist into
 * LDAPMod structures
 */
LDAPMod **formdiff_to_ldapmod(GList *oldlist, GList *newlist)
{
     GList *oldlist_tmp, *values, *deleted_values, *added_values;
     LDAPMod **mods, *mod;
     struct formfill *oldform, *newform;
     int old_l, new_l, modcnt;

     oldlist_tmp = oldlist;
     old_l = g_list_length(oldlist);
     new_l = g_list_length(newlist);
     mods = g_malloc( sizeof(void *) * (( old_l > new_l ? old_l : new_l ) * 4) );
     mods[0] = NULL;
     modcnt = 0;

     /* pass 1: deleted attributes, and deleted/added values */
     while(oldlist) {
	  oldform = (struct formfill *) oldlist->data;
	  newform = lookup_attribute(newlist, oldform->attrname);
	  /* oldform->values can come up NULL if the attribute was in the form
	     only because add_attrs_by_oc() added it. Not a delete in this case... */
	  if(oldform->values != NULL && (newform == NULL || newform->values == NULL)) {
	       /* attribute deleted */
	       mod = g_malloc(sizeof(LDAPMod));
	       mod->mod_op = LDAP_MOD_DELETE;
	       mod->mod_type = g_strdup(oldform->attrname);
	       mod->mod_values = NULL;
	       mods[modcnt++] = mod;
	  }
	  else {

	       /* pass 1.1: deleted values */
	       deleted_values = NULL;
	       values = oldform->values;
	       while(values) {
		    if(!find_value(newform->values, (char *) values->data)) {
			 deleted_values = g_list_append(deleted_values, values->data);
		    }
		    values = values->next;
	       }

	       /* pass 1.2: added values */
	       added_values = NULL;
	       values = newform->values;
	       while(values) {
		    if(!find_value(oldform->values, (char *) values->data))
			 added_values = g_list_append(added_values, values->data);
		    values = values->next;
	       }

	       if(deleted_values && added_values) {
		    /* values deleted and added -- likely a simple edit.
		       Optimize this into a MOD_REPLACE. This could be
		       more work for the server in case of a huge number
		       of values, but who knows...
		    */
		    mod = g_malloc(sizeof(LDAPMod));
		    mod->mod_op = LDAP_MOD_REPLACE;
		    mod->mod_type = g_strdup(oldform->attrname);
		    mod->mod_values = glist_to_mod_values(newform->values);
		    mods[modcnt++] = mod;
	       }
	       else {
		    if(deleted_values) {
			 mod = g_malloc(sizeof(LDAPMod));
			 mod->mod_op = LDAP_MOD_DELETE;
			 mod->mod_type = g_strdup(oldform->attrname);
			 mod->mod_values = glist_to_mod_values(deleted_values);
			 mods[modcnt++] = mod;
		    }
		    else if(added_values) {
			 mod = g_malloc(sizeof(LDAPMod));
			 mod->mod_op = LDAP_MOD_ADD;
			 mod->mod_type = g_strdup(oldform->attrname);
			 mod->mod_values = glist_to_mod_values(added_values);
			 mods[modcnt++] = mod;
		    }
	       }

	       g_list_free(deleted_values);
	       g_list_free(added_values);

	  }

	  oldlist = oldlist->next;
     }
     oldlist = oldlist_tmp;

     /* pass 2: added attributes */
     while(newlist) {
	  newform = (struct formfill *) newlist->data;
	  if(lookup_attribute(oldlist, newform->attrname) == NULL) {
	       /* new attribute was added */
	       if(newform->values) {
		    mod = g_malloc(sizeof(LDAPMod));
		    mod->mod_op = LDAP_MOD_ADD;
		    mod->mod_type = g_strdup(newform->attrname);
		    mod->mod_values = glist_to_mod_values(newform->values);
		    mods[modcnt++] = mod;
	       }
	  }

	  newlist = newlist->next;
     }

     mods[modcnt] = NULL;

     return(mods);
}


char **glist_to_mod_values(GList *values)
{
     int valcnt;
     char **od_values;

     valcnt = 0;
     od_values = g_malloc(sizeof(char *) * (g_list_length(values) + 1));
     while(values) {
	  if(values->data)
	       od_values[valcnt++] = g_strdup(values->data);
	  values = values->next;
     }
     od_values[valcnt] = NULL;

     return(od_values);
}


int find_value(GList *list, char *value)
{

     while(list) {
	  if(!strcmp( (char *) list->data, value))
	       return(1);

	  list = list->next;
     }

     return(0);
}


void destroy_editwindow(GHashTable *hash)
{
     GList *formlist;
     GtkWidget *window;
     char *dn;

     formlist = g_hash_table_lookup(hash, "formlist");
     free_formlist(formlist);

     dn = g_hash_table_lookup(hash, "dn");
     if(dn)
	  g_free(dn);

     window = g_hash_table_lookup(hash, "parent window");
     gtk_widget_destroy(window);

     g_hash_table_destroy(hash);

}


void add_row(GtkWidget *button, gpointer hash)
{
     struct formfill *form;

     update_formlist(hash);

     form = (struct formfill *) gtk_object_get_data(GTK_OBJECT(button), "formfill");
     if(!form)
	  return;

     form->num_inputfields++;

     clear_table(hash);
     build_inputform(hash);

}


GtkWidget *gq_new_arrowbutton(GHashTable *hash)
{
     GtkWidget *newabutton, *arrow;

     newabutton = gtk_button_new();
     GTK_WIDGET_UNSET_FLAGS(newabutton, GTK_CAN_FOCUS);
     gtk_signal_connect(GTK_OBJECT(newabutton), "clicked",
			GTK_SIGNAL_FUNC(add_row),
			(gpointer) hash);
     arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_OUT);
     gtk_widget_show(arrow);
     gtk_container_add(GTK_CONTAINER(newabutton), arrow);
     gtk_widget_show(newabutton);

     return(newabutton);
}



GtkWidget *find_focusbox(GtkTable *table)
{
     GList *children;
     GtkTableChild *child;
     GtkWidget *focusbox;

     if(!table)
	  return(NULL);

     focusbox = NULL;

     children = table->children;
     while(children) {

	  child = children->data;
	  if(GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(child->widget))) {
	       focusbox = child->widget;
	       break;
	  }

	  children = children->next;
     }

     return(focusbox);
}


/*
 * callback for entry or textbox buttons
 */
void change_displaytype(GtkWidget *button, gpointer hash)
{
     GtkWidget *focusbox;
     GtkTable *table;
     struct formfill *form, *focusform;
     char *transformtype;

     update_formlist(hash);

     table = (GtkTable *) g_hash_table_lookup(hash, "table");
     focusbox = find_focusbox(table);
     if(focusbox == NULL)
	  /* nothing focused */
	  return;

     focusform = (struct formfill *) gtk_object_get_data(GTK_OBJECT(focusbox), "formfill");
     if(focusform == NULL)
	  /* field can't be resized anyway */
	  return;
     g_hash_table_insert(hash, "focusform", focusform);

     transformtype = gtk_object_get_data(GTK_OBJECT(button), "transform");

     form = (struct formfill *) gtk_object_get_data(GTK_OBJECT(focusbox), "formfill");
     if(!form)
	  return;

     if(!strcmp(transformtype, "make entry")) {
	  if(GTK_IS_ENTRY(focusbox))
	       return;

	  form->displaytype = DISPLAYTYPE_ENTRY;
     }
     else if(!strcmp(transformtype, "make text")) {
	  if(GTK_IS_TEXT(focusbox))
	       return;

	  form->displaytype = DISPLAYTYPE_TEXT;
     }

     /* redraw */
     clear_table(hash);
     build_inputform(hash);

}
