/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"

/* number of steps to use for the sliding of windows */
#define SLIDESTEPS	10	/* XXX: make an option? */

/*
 * prepare workspace stuff for a shutdown/restart: simply
 * move the current workspace to 0,0; it doesn't matter which
 * desktop because 0,0 is always valid on any desktop.
 */
void workspace_shutdown() {
	screen_t *screen;

	/* do this for all screens */
	screen = screen_list;
	while (screen) {
		workspace_viewport_move(screen, screen->desktop,
			-screen->desktop->viewx, -screen->desktop->viewy);
		screen = screen->next;
	}
}

/*
 * add desktops to a screen.  on error we return -1, leaving as many
 * complete desktops as we could add there, and freeing up partially
 * added one.
 */
int workspace_add_desks(screen_t *screen, int width, int height, int count) {
	desktop_t *desktop;
	desktop_t *prev;
	int spaces, n;

	/* find the end of the list of desktops */
	prev = screen->desktop_list;
	while (prev && prev->next)
		prev = prev->next;

	/* add count desktops of dim width x height */
	while (count--) {
		desktop = calloc(1, sizeof(desktop_t));
		if (!desktop)
			return -1;

		desktop->num = screen->desktop_count++;
		desktop->width = width;
		desktop->height = height;
		desktop->workspaces = malloc(width * height * sizeof(workspace_t *));
		if (!desktop->workspaces)
			goto free1;

		/* get all the workspaces for this desktop */
		for (spaces = 0; spaces < desktop->width * desktop->height; spaces++) {
			desktop->workspaces[spaces] = calloc(1, sizeof(workspace_t));
			if (!desktop->workspaces[spaces])
				goto free2;
			desktop->workspaces[spaces]->desktop = desktop;
		}
		desktop->current_space = desktop->workspaces[0];

		/* link it up */
		desktop->prev = prev;
		if (prev)
			prev->next = desktop;
		else
			screen->desktop_list = desktop;
		prev = desktop;
	}

	return 0;

free2:
	/* free and exit with error */
	for (n = 0; n < spaces; n++)
		free(desktop->workspaces[n]);
free1:
	free(desktop);
	return -1;
}

/* remove the desktops from a screen */
void workspace_rm_desks(screen_t *screen) {
	desktop_t *desktop, *tmp;
	int i, numspaces;

	desktop = screen->desktop_list;
	while (desktop) {
		numspaces = desktop->width * desktop->height;
		for (i = 0; i < numspaces; i++)
			free(desktop->workspaces[i]);
		free(desktop->workspaces);
		
		tmp = desktop->next;
		free(desktop);
		desktop = tmp;
	}
}

/* switch the desktop for screen to desk number num */
void workspace_switch_desk(screen_t *screen, int num) {
	client_t *focusthis = NULL;
	client_t *client;
	desktop_t *desktop, *olddesk;

	/* find the desktop to switch to */
	desktop = screen->desktop_list;
	while (desktop) {
		if (desktop->num == num)
			goto gotdesk;
		desktop = desktop->next;
	}
	return;

gotdesk:
	if (screen->desktop == desktop)
		return;

	/* unmap clients on the current desktop, and map them for the new */
	client = client_list;
	while (client) {
		if (client->state == NormalState && !client->flags.internal) {
			if (client->workspace->desktop == screen->desktop) {
				/*
				 * handle windows that are stuck to the glass.  the idea is
				 * to preserve focus settings, so if a sticky window is
				 * focused it'll stay focused, and if it isn't focused it wont
				 * get the focus.  this means we don't need to check flags.nofocus,
				 * because if it's nofocus it will never have focus in the first
				 * place.
				 */
				if (client->flags.sticky) {
					if (client->screen->desktop->current_space->focused == client
							&& focusthis == NULL)
						focusthis = client;
					workspace_rm_client(client);
					workspace_add_client(desktop->current_space, client);
				} else
					XUnmapWindow(display, client->frame);
			} else if (client->workspace->desktop == desktop)
				XMapWindow(display, client->frame);
		}

		client = client->next;
	}

	/* set these up before doing focus stuff */
	olddesk = screen->desktop;
	screen->desktop = desktop;

	/* now set the focus */
	if (focusthis)
		focus_client(focusthis);
	else if (desktop->current_space->focused)
		focus_client(desktop->current_space->focused);
	else
		focus_none(screen);

	/* let plugins know about the desktop change */
	plugin_desktop_change(screen, olddesk);
}

/* add a client to a workspace */
void workspace_add_client(workspace_t *workspace, client_t *client) {
	client->workspace = workspace;
	if (!client->flags.nofocus)
		focus_list_add(client);
}

/* remove a client from the workspace it is on */
void workspace_rm_client(client_t *client) {
	if (!client->flags.nofocus) {
		focus_unfocus(client);
		focus_list_rm(client);
	}
	client->workspace = NULL;
}

/*
 * add a client to the appropriate workspace based on it's position.
 * we handle any removal from a workspace that the client is on here: to
 * allow us to not do anything if it's on the workspace that this would move
 * it to.
 */
void workspace_add_bypos(desktop_t *desktop, client_t *client) {
	workspace_t *newspace;
	int vx, vy;

	/*
	 * if the client is even partially visible on this workspace
	 * it must be in this workspace; otherwise it's  nonimportant which
	 * workspace it is in if it crosses over lines: we favor up and left;
	 * if it becomes visible on a current_space it isn't part of
	 * workspace_viewport_move will take care of switching it as neccessary.
	 */
	if (client->x > -(client->width + client->left_space + client->right_space)
			&& client->y > -(client->height + client->top_space + client->bottom_space)
			&& client->x < client->screen->width && client->y < client->screen->height) {
		newspace = desktop->current_space;
	} else {
		/* get the offsets in the workspace grid */	
		vx = (client->x + desktop->viewx * client->screen->width)
			/ client->screen->width;
		vy = (client->y + desktop->viewy * client->screen->height)
			/ client->screen->height;

		/* clip them to the desktop's size */
		if (vx < 0)
			vx = 0;
		else if (vx > desktop->width)
			vx = desktop->width - 1;
		if (vy < 0)
			vy = 0;
		else if (vy > desktop->height)
			vy = desktop->height - 1;

		/* get new workspace */
		newspace = desktop->workspaces[vx + (vy * desktop->width)];
	}

	/* make sure the client isn't already on this space */
	if (client->workspace == newspace)
		return;

	/* remove from the old, add to the new */
	if (client->workspace)
		workspace_rm_client(client);
	workspace_add_client(newspace, client);
}

/*
 * modify the effective viewport for a large workspace, xmove and ymove
 * are relative (i.e., 1 or -1 or 0, etc) and in terms of screens.
 *
 * we do virtual desktops by moving windows in the opposite direction of
 * the viewport.  to make stuff make more sense to the user we maintain
 * focus lists for each viewport; which means that here we need to handle
 * clients that'll end up still visible (i.e. are half between viewports)
 * by putting them into the new focuslist.
 */
int workspace_viewport_move(screen_t *screen, desktop_t *desktop, int xmove, int ymove) {
	client_t *focusthis = NULL;
	client_t *client;
	workspace_t *workspace;
	int tmpx, tmpy;
	int i, modx, mody;
	int moving;

	/* get new offsets */
	tmpx = desktop->viewx + xmove;
	if (tmpx >= desktop->width || tmpx < 0)
		return 0;
	tmpy = desktop->viewy + ymove;
	if (tmpy >= desktop->height || tmpy < 0)
		return 0;

	/* now set them */
	if (tmpx == desktop->viewx && tmpy == desktop->viewy)
		return 0;
	desktop->viewx = tmpx;
	desktop->viewy = tmpy;

	/* the new workspace */
	workspace = desktop->workspaces[tmpx + (tmpy * desktop->width)];

/* keep a client by putting it into the new workspace's focuslist */
#define KEEPCLIENT(c) do {					\
	if (desktop->current_space->focused == (c)) {		\
		if (focusthis == NULL)				\
			focusthis = (c);			\
	}							\
	workspace_rm_client((c));				\
	workspace_add_client(workspace, (c));			\
} while (0)

	/* slide the windows to their new position */
	if (desktop == screen->desktop) {
		moving = 0;
		modx = (xmove * screen->width) / SLIDESTEPS;
		mody = (ymove * screen->height) / SLIDESTEPS;
		for (i = 0; i < SLIDESTEPS; i++) {
			client = client_list;
			while (client) {
				if (client->state == NormalState && !client->flags.internal
						&& client->workspace->desktop == desktop
						&& !client->flags.sticky) {
					XMoveWindow(display, client->frame, client->x - (modx * i),
						client->y - (mody * i));
					moving = 1;
				}
				client = client->next;
			}

			/* only delay if we are actually sliding windows */
			if (moving) {
				XSync(display, 0);
				usleep(5);
			} else {
				break;
			}
		}
	}

	/* now switch to the new workspace */
	client = client_list;
	while (client) {
		if (client->state == NormalState && !client->flags.internal
				&& client->workspace->desktop == desktop) {
			if (!client->flags.sticky) {
				client->x -= (xmove * screen->width);
				client->y -= (ymove * screen->height);
				client_sizeframe(client);

				/*
				 * clients recieve a synthetic config, because they have moved, even
				 * though the illusion is that the 'viewport' is what moved.
				 */
				action_send_config(client);

				/* if the client is visible but thinks it's on a diff wspace it must be kept */
				if (client->workspace != workspace
						&& client->x > -(client->width + client->left_space + client->right_space)
						&& client->y > -(client->height + client->top_space + client->bottom_space)
						&& client->x < screen->width && client->y < screen->height)
					KEEPCLIENT(client);
			} else
				KEEPCLIENT(client);
		}
		client = client->next;
	}

#undef KEEPCLIENT

	/* set the new current_space */
	desktop->current_space = workspace;

	/* now set the focus */
	if (focusthis)
		focus_client(focusthis);
	else if (workspace->focused)
		focus_client(workspace->focused);
	else
		focus_none(screen);

	/* let plugins know */
	plugin_workspace_change(screen, desktop);

	return 1;
}
