/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"

/* our list of keys that we bind */
static keybind_t *keys_list	= NULL;

/* shutdown keys; free allocated memory and the like */
void keys_shutdown() {
	keybind_t *key, *tmp;

	/* free memory allocated in our list of keys */
	key = keys_list;
	while (key) {
		/* some keybinding types have allocated memory */
		switch (key->action) {
		case KEY_COMMAND:
			free(key->dat.cmd);
			break;
		}

		/* free the keybind_t */
		tmp = key->next;
		free(key);
		key = tmp;
	}
}

/* add a key binding */
keybind_t *keys_add(int keycode, int modifiers, int action, void* dat) {
	keybind_t *key;

	key = malloc(sizeof(keybind_t));
	if (!key)
		return NULL;
	key->keycode = keycode;
	key->modifiers = modifiers;
	key->action = action;
	key->dat.generic = dat;

	key->next = keys_list;
	keys_list = key;

	return key;
}

/* do grabs on the root window of screen */
void keys_grab(screen_t *screen) {
	keybind_t *key;

	key = keys_list;
	while (key) {
		XGrabKey(display, key->keycode, key->modifiers, screen->root, 1,
			GrabModeAsync, GrabModeAsync);
		key = key->next;
	}
}

/* move the viewport */
static void keys_moveport(screen_t *screen, int dir) {
	int vx, vy;

	switch (dir) {
	case MV_UP:		vx = 0; vy = -1;	break;
	case MV_DOWN:		vx = 0; vy = 1;		break;
	case MV_LEFT:		vx = -1; vy = 0;	break;
	case MV_RIGHT:		vx = 1; vy = 0;		break;
	case MV_UPRIGHT:	vx = 1; vy = -1;	break;
	case MV_DOWNRIGHT:	vx = 1; vy = 1;		break;
	case MV_DOWNLEFT:	vx = -1; vy = 1;	break;
	default:
	case MV_UPLEFT:		vx = -1; vy = -1;	break;
	}

	workspace_viewport_move(screen, screen->desktop, vx, vy);
}

/*
 * focus and raise a win, we don't use focus_setfocused because
 * when going across screens in the all or screen style cyclings
 * it would try to just set workspace->focused pointers.
 */
#define FOCUSRAISE(foc) do {						\
	focus_client((foc));						\
	XRaiseWindow(display, client_focused->frame);			\
} while (0)

/* forward loop through screens, check if foc to focus foc, for cycle */
#define SCREEN_FLOOP(foc, currentscr) do {				\
	screen = (currentscr)->next;					\
									\
	while (screen != (currentscr)) {				\
		if (!screen) {						\
			screen = screen_list;				\
			continue;					\
		}							\
									\
		if ((foc)) {						\
			FOCUSRAISE((foc));				\
			break;						\
		}							\
		screen = screen->next;					\
	}								\
} while (0)

/* backward loop through screens, like floop */
#define SCREEN_BLOOP(foc, currentscr) do {				\
	screen = (currentscr)->next;					\
									\
	while (screen != (currentscr)) {				\
		if (!screen) {						\
			screen = screen_list;				\
			while (screen->next)				\
				screen = screen->next;			\
			continue;					\
		}							\
									\
		if ((foc)) {						\
			FOCUSRAISE((foc));				\
			break;						\
		}							\
		screen = screen->prev;					\
	}								\
} while (0)

/* cycle screens */
static void keys_cyclescr(screen_t *keyscr, client_t *client, int cycletype) {
	screen_t *screen;
	screen_t *pointer_screen;
	Window pointer_root;
	Window dumwin;
	int dumint;

	/*
	 * screen-based cycling doesn't require visible client windows: it
	 * warps the pointer to new screens, and focused the focused window
	 * on the screen if there is one. The currently 'focused' screen
	 * is the one w/ the mouse pointer, because that is where keybinds
	 * will take effect.  So event if the focused window is on a different
	 * screen we treat the mouse pointer screen as the focused screen.
	 */
	XQueryPointer(display, keyscr->root, &pointer_root, &dumwin, &dumint, &dumint,
		&dumint, &dumint, &dumint);
	pointer_screen = screen_list;
	while (pointer_screen) {
		if (pointer_screen->root == pointer_root)
			break;
		pointer_screen = pointer_screen->next;
	}

	/* now we know which screen we're on; put it on the screen it needs to go to */
	switch (cycletype) {
	case CF_FSCR:
		screen = pointer_screen->next;

		if (!screen)
			screen = screen_list;

		/* only refocus if this isn't the screen of the focused window */
		if (screen == pointer_screen)
			goto warp;
		if (screen->desktop->current_space->focused)
			if (!client || client->screen != screen)
				FOCUSRAISE(screen->desktop->current_space->focused);
	case CF_BSCR:
	default:
		screen = pointer_screen->prev;

		/* last screen on the list */
		if (!screen) {
			screen = screen_list;
			while (screen->next)
				screen = screen->next;
		}

		/* only refocus if this isn't the screen of the focused window */
		if (screen == pointer_screen)
			goto warp;
		if (screen->desktop->current_space->focused)
			if (!client || client->screen != screen)
				FOCUSRAISE(screen->desktop->current_space->focused);
	}

warp:
	/* put pointer in the middle of the screen */
	XWarpPointer(display, None, screen->root, 0, 0, 1, 1, screen->width / 2, screen->height / 2);
}

/*
 * cycle input focus, slightly complicated becase we support several cycling
 * types here (for multiscreen), see keys.h for info on the types.
 */
static void keys_cyclefoc(screen_t *keyscr, client_t *client, int cycletype) {
	screen_t *screen;

	/* screen based cycling; not strictly for just focus cycling */
	if (cycletype == CF_FSCR || cycletype == CF_BSCR) {
		keys_cyclescr(keyscr, client, cycletype);
		return;
	}

	/* if no window is focused, this is easier */
	if (!client) {
		if (cycletype % 2 == 0) {
			if (keyscr->desktop->current_space->focus_list)
				FOCUSRAISE(keyscr->desktop->current_space->focus_list);
			else if (cycletype == CF_FALL)
				SCREEN_FLOOP(screen->desktop->current_space->focus_list, keyscr);
		} else {
			if (keyscr->desktop->current_space->focus_last)
				FOCUSRAISE(keyscr->desktop->current_space->focus_last);
			else if (cycletype == CF_BALL)
				SCREEN_BLOOP(screen->desktop->current_space->focus_last, keyscr);
		}
		return;
	}

	/* if the type is div by 2, it's a forward cycling type */
	if (cycletype % 2 == 0) {
		if (client->focus_next) {
			FOCUSRAISE(client->focus_next);
		} else {
			if (cycletype == CF_FALL)
				SCREEN_FLOOP(screen->desktop->current_space->focus_list, client->screen);
			else if (client->screen->desktop->current_space->focus_list)
				FOCUSRAISE(client->screen->desktop->current_space->focus_list);
		}
	} else {
		if (client->focus_prev) {
			FOCUSRAISE(client->focus_prev);
		} else {
			if (cycletype == CF_BALL)
				SCREEN_BLOOP(screen->desktop->current_space->focus_last, client->screen);
			else if (client->screen->desktop->current_space->focus_last)
				FOCUSRAISE(client->screen->desktop->current_space->focus_last);
		}
	}
}

#undef SCREEN_BLOOP
#undef SCREEN_FLOOP
#undef FOCUSRAISE

/* perform an action for a key */
static void keys_action(screen_t *screen, keybind_t *key) {
	client_t *client = client_focused;

	XUngrabKeyboard(display, CurrentTime);

	switch (key->action) {
	case KEY_ICONIFY:
		if (client)
			action_iconify(client);
		break;
	case KEY_ZOOM:
		if (client)
			action_zoom(client);
		break;
	case KEY_SWITCHDESK:
		workspace_switch_desk(screen, key->dat.num);
		break;
	case KEY_MOVEVIEWPORT:
		keys_moveport(screen, key->dat.dir);
		break;
	case KEY_COMMAND:
		action_exec(screen->num, key->dat.cmd);
		break;
	case KEY_DELETE:
		if (client && !client->flags.nodelete)
			action_sendcmesg(client->window, WM_DELETE_WINDOW, CurrentTime);
		break;
	case KEY_CYCLEFOCUS:
		keys_cyclefoc(screen, client, key->dat.cycletype);
		break;
	case KEY_RAISE:
		if (client)
			XRaiseWindow(display, client->frame);
		break;
	case KEY_LOWER:
		if (client)
			XLowerWindow(display, client->frame);
		break;
	case KEY_DGROUPSWITCH:
		if (client)
			dgroup_switch(client, NULL);
		break;
	case KEY_STICKY:
		if (client)
			client->flags.sticky = ~client->flags.sticky;
		break;
	case KEY_RESTART:
		restart_bin = binary_name;
		restart_flag = 1;
		break;
	}
}

/* handles when we get keypresses from our grabs */
void keys_press(screen_t *screen, XKeyEvent *e) {
	keybind_t *key;

	key = keys_list;
	while (key) {
		if (e->keycode == key->keycode
				&& e->state == key->modifiers) {
			keys_action(screen, key);
			return;
		}
		key = key->next;
	}
}
