/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"

/* our handler table */
typedef void (*event_handler_t)(XEvent *event);
static event_handler_t handlers[LASTEvent];

/* handler for DestroyNotify events */
static void event_destroy_notify(XEvent *event) {
	XDestroyWindowEvent *e = &event->xdestroywindow;
	client_t *client;

	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0)
		return;
	if (e->window != client->window)
		return;

	if (!client->flags.internal)
		XRemoveFromSaveSet(display, client->window);
	plugin_window_death(client);
	client_rm(client);
}

/* handler for ConfigureRequest */
static void event_configure_request(XEvent *event) {
	XConfigureRequestEvent *e = &event->xconfigurerequest;
	XWindowChanges wc;
	client_t *client;

	/* now we only update the structure if the window has one */
	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0)
		client = NULL;

	/* 
	 * first of all we need to respond to ConfigureRequests of windows that haven't yet been
	 * mapped, and thus don't have client_t structs in our list.  We just pass along the configure
	 * essentially.
	 */
	if (!client) {
		wc.x = e->x;
		wc.y = e->y;
		wc.width = e->width;
		wc.height = e->height;
		wc.border_width = e->border_width;
		wc.stack_mode = e->detail;
		XConfigureWindow(display, e->window, e->value_mask, &wc);
	} else {
		/* now for managed windows we do stuff to our struct */
		if (e->window != client->window)
			return;
	
		if (e->value_mask & CWX)
			client->x = e->x;
		if (e->value_mask & CWY)
			client->y = e->y;
		if (e->value_mask & CWWidth)
			client->width = e->width;
		if (e->value_mask & CWHeight)
			client->height = e->height;

		/* when a client moves itself, it may need to change workspaces */
		client_sizeframe(client);
		action_send_config(client);
		workspace_add_bypos(client->screen->desktop, client);
		plugin_geometry_change(client);
	}
}

/* handler for MapRequest events */
static void event_map_request(XEvent *event) {
	XMapRequestEvent *e = &event->xmaprequest;
	screen_t *screen;
	client_t *client;

	/* create the client_t structure */
	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->parent, root_context, (XPointer *) &screen) != 0)
			return;

		/* if a plugin takes the event we stop processing on it */
		if (plugin_map_request(screen, e))
			return;
		client = client_add(screen, e->window, NULL, NULL, NULL);
		if (!client) return;
	} else {
		/*
		 * for some reason some broken clients (netscape) seem to do more
		 * than one map request on top window...
		 */
		return;
	}

	client_map(client, 1);
}

/* handler for MapNotify events */
static void event_map_notify(XEvent *event) {
	XMapEvent *e = &event->xmap;
	client_t *client;
	plugin_t *plugin;

	/*
	 * look for the client context we created for it in event_map_request; and
	 * only continue if this is a notify from the client->window; if it's for the
	 * frame we don't care here.
	 */
	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
			PLUGIN_CALLBACK(plugin, map_notify, NULL, e);
		return;
	}
	if (e->window != client->window)
		return;

	/* give focus to new windows (if the options say to) */
	if (options.focus_new && client->workspace)
		focus_setfocused(client);

	/* let plugins know about the first mapping */
	if (client->plugin)
		PLUGIN_CALLBACK(client->plugin, map_notify, client, e);
}

/* handler for UnmapNotify events */
static void event_unmap_notify(XEvent *event) {
	XUnmapEvent *e = &event->xunmap;
	client_t *client;
	plugin_t *plugin;

	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
			PLUGIN_CALLBACK(plugin, unmap_notify, e);
		return;
	}
	if (e->window != client->window)
		return;

	/*
	 * see if the event came from reparenting
	 * (see screen.c, manage_existing_windows)
	 */
	if (client->flags.unmap_from_reparent) {
		client_map(client, 0);
		client->flags.unmap_from_reparent = 0;
		return;
	}

	/*
	 * get rid of our client structure, reparent it to the root,
	 * and remove from our saveset
	 */
	if (client->state == NormalState)
		XUnmapWindow(display, client->frame);
	plugin_window_death(client);
	client_setstate(client, WithdrawnState);
	if (!client->flags.internal)
		XRemoveFromSaveSet(display, client->window);
	XReparentWindow(display, client->window, client->screen->root, client->x, client->y);
	client_rm(client);
}

/* handler for ButtonPress events */
static void event_button_press(XEvent *event) {
	XButtonEvent *e = &event->xbutton;
	client_t *client;
	screen_t *screen;
	plugin_t *plugin;
	decor_t *decor;

	/* if not a client_t, check if it's a root click or a nonmanaged plugin window */
	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->window, root_context, (XPointer *) &screen) == 0)
			plugin_root_button_press(screen, e);
		else if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
			PLUGIN_CALLBACK(plugin, button_press, NULL, e);
		return;
	}

	/* there's a few ways to respond to button presses */
	if (e->window == client->frame) {
		/* in click focus modes, we need to give focus to the client */
		if (options.focus == FOCUS_CLICK)
			focus_setfocused(client);

		/* either move/resize on it, or just raise it */
		if (e->state == options.mouse_modifier) {
			if (e->button == Button3)
				action_resize(client);
			else
				action_move(client);
		} else {
			/*
			 * clicking to raise/focus a window; getting here can only happen
			 * when we're in clickfocus mode.
			 */
			XAllowEvents(display, ReplayPointer, CurrentTime);
			XRaiseWindow(display, client->frame);
		}

		return;
	} else if (e->window == client->window) {
		/* first attempt to give it focus if we're in that type of mode */
		if (options.focus == FOCUS_CLICK)
			focus_setfocused(client);

		/* give managed plugin windows their callback */
		if (client->plugin)
			PLUGIN_CALLBACK(client->plugin, button_press, client, e);
	} else if (XFindContext(display, e->window, decor_context, (XPointer *) &decor) == 0) {
		/* presses on decoration units: focus the client; then tell the decor */
		if (options.focus == FOCUS_CLICK)
			focus_setfocused(client);
		decor_handlepress(client, decor, e);

		return;
	}
}

/* handler for ButtonRelease events */
static void event_button_release(XEvent *event) {
	XButtonEvent *e = &event->xbutton;
	screen_t *screen;
	plugin_t *plugin;
	client_t *client;

	/* if not the root window, try for an unmanaged plugin */
	if (XFindContext(display, e->window, root_context, (XPointer *) &screen) != 0) {
		if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0) {
			PLUGIN_CALLBACK(plugin, button_release, NULL, e);
		} else if (XFindContext(display, e->window, client_context, (XPointer *) &client) == 0) {
			if (client->plugin)
				PLUGIN_CALLBACK(client->plugin, button_release, client, e);
		}
		return;
	}

	plugin_root_button_release(screen, e);

	/* workspace switches on a desktop */
	if (e->button == Button1) {
		if (e->y_root == 0) {
			if (workspace_viewport_move(screen, screen->desktop, 0, -1))
				XWarpPointer(display, None, screen->root, 0, 0, 1, 1, e->x_root,
					screen->height - 1);
		} else if (e->y_root == screen->height - 1) {
			if (workspace_viewport_move(screen, screen->desktop, 0, 1))
				XWarpPointer(display, None, screen->root, 0, 0, 1, 1,
					e->x_root, 0);
		} else if (e->x_root == 0) {
			if (workspace_viewport_move(screen, screen->desktop, -1, 0))
				XWarpPointer(display, None, screen->root, 0, 0, 1, 1,
					screen->width - 1, e->y_root);
		} else if (e->x_root == screen->width - 1) {
			if (workspace_viewport_move(screen, screen->desktop, 1, 0))
				XWarpPointer(display, None, screen->root, 0, 0, 1, 1,
					0, e->y_root);
		}
	}
}

/* handler for KeyPress events */
static void event_key_press(XEvent *event) {
	XKeyEvent *e = &event->xkey;
	screen_t *screen;
	plugin_t *plugin;
	client_t *client;

	if (XFindContext(display, e->root, root_context, (XPointer *) &screen) != 0) {
		if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
			if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
				PLUGIN_CALLBACK(plugin, key_press, NULL, e);
		} else {
			if (client->plugin)
				PLUGIN_CALLBACK(client->plugin, key_press, client, e);
		}
		return;
	}

	keys_press(screen, e);
}

/* handler for Expose events, this is exported so it may be called, such as during opaque movement */
void event_expose(XEvent *event) {
	XExposeEvent *e = &event->xexpose;
	client_t *client;
	decor_t *decor;
	plugin_t *plugin;

	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
			PLUGIN_CALLBACK(plugin, expose, NULL, e);
		return;
	}

	if (XFindContext(display, e->window, decor_context, (XPointer *) &decor) != 0) {
		if (client->plugin)
			PLUGIN_CALLBACK(client->plugin, expose, client, e);
		return;
	}

	decor_expose(client, decor, e);
}

/* handler for ClientMessage */
static void event_client_message(XEvent *event) {
	XClientMessageEvent *e = &event->xclient;
	client_t *client;

	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0)
		return;
	if (e->window != client->window)
		return;

	if (e->message_type == WM_CHANGE_STATE) {
		if (e->format == 32 && e->data.l[0] == IconicState 
				&& client->state != IconicState) {
			action_iconify(client);
		}
	}
}

/* handler for PropertyNotify */
static void event_property_notify(XEvent *event) {
	XPropertyEvent *e = &event->xproperty;
	client_t *client;
	long supplied;

	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0)
		return;
	if (e->window != client->window)
		return;

	if (e->state != PropertyNewValue)
		return;

	switch (e->atom) {
	case XA_WM_NAME:
		if (client->store_name)
			XFree(client->store_name);
		XFetchName(display, client->window, &client->store_name);

		/*
		 * XXX: need a callback to let plugins know
		 */
		if (client->state != IconicState)
			decor_titlechange(client);
		break;
	case XA_WM_NORMAL_HINTS:
		XGetWMNormalHints(display, client->window, &client->normalhints, &supplied);
		break;
	}
}

/* handler for EnterNotify */
static void event_enter_notify(XEvent *event) {
	XCrossingEvent *e = &event->xcrossing;
	client_t *client;
	plugin_t *plugin;

	/* we only get this for sloppy/pointer focus, or for plugin windows */
	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
			PLUGIN_CALLBACK(plugin, enter_notify, NULL, e);
		return;
	}
	if (e->window != client->frame)
		return;

	if (client->plugin)
		PLUGIN_CALLBACK(client->plugin, enter_notify, client, e);

	focus_setfocused(client);
}

/* handler for LeaveNotify */
static void event_leave_notify(XEvent *event) {
	XCrossingEvent *e = &event->xcrossing;
	client_t *client;
	plugin_t *plugin;

	/* only get this for pointer focus mode, or for plugin windows */
	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
			PLUGIN_CALLBACK(plugin, leave_notify, NULL, e);
		return;
	}
	if (e->window != client->frame)
		return;

	if (client->plugin)
		PLUGIN_CALLBACK(client->plugin, leave_notify, client, e);

	/*
	 * we check if the client has a workspace first, because we will get
	 * a leave notify if the user iconifies a window, and the window
	 * will have left it's workspace, and thus already be unfocused.
	 */
	if (client->workspace) 
		focus_unfocus(client);
}

/* handler for ShapeNotify */
static void event_shape_notify(XEvent *event) {
	client_t *client;

	if (XFindContext(display, event->xany.window, client_context, (XPointer *) &client) != 0)
		return;
	if (event->xany.window != client->window)
		return;

	client_shape(client);
}

/* handler for KeyRelease events */
static void event_key_release(XEvent *event) {
	XKeyEvent *e = &event->xkey;
	plugin_t *plugin;
	client_t *client;

	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
			PLUGIN_CALLBACK(plugin, key_release, NULL, e);
		return;
	}
	if (client->plugin)
		PLUGIN_CALLBACK(client->plugin, key_release, client, e);
}

/* handler for MotionNotify */
static void event_motion_notify(XEvent *event) {
	XMotionEvent *e = &event->xmotion;
	plugin_t *plugin;
	client_t *client;

	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
			PLUGIN_CALLBACK(plugin, pointer_motion, NULL, e);
		return;
	}
	if (client->plugin)
		PLUGIN_CALLBACK(client->plugin, pointer_motion, client, e);
}

/* handler for VisibilityNotify */
static void event_visibility_notify(XEvent *event) {
	XVisibilityEvent *e = &event->xvisibility;
	plugin_t *plugin;
	client_t *client;

	if (XFindContext(display, e->window, client_context, (XPointer *) &client) != 0) {
		if (XFindContext(display, e->window, plugin_context, (XPointer *) &plugin) == 0)
			PLUGIN_CALLBACK(plugin, visibility_notify, NULL, e);
		return;
	}
	if (client->plugin)
		PLUGIN_CALLBACK(client->plugin, visibility_notify, client, e);
}

/* setup our event handler table, etc */
static void event_init() {
	int i;

	for (i = 0; i < LASTEvent; i++)
		handlers[i] = NULL;

	handlers[DestroyNotify]		= event_destroy_notify;
	handlers[ConfigureRequest]	= event_configure_request;
	handlers[MapRequest]		= event_map_request;
	handlers[MapNotify]		= event_map_notify;
	handlers[UnmapNotify]		= event_unmap_notify;
	handlers[ButtonPress]		= event_button_press;
	handlers[ButtonRelease]		= event_button_release;
	handlers[KeyPress]		= event_key_press;
	handlers[Expose]		= event_expose;
	handlers[ClientMessage]		= event_client_message;
	handlers[PropertyNotify]	= event_property_notify;
	handlers[EnterNotify]		= event_enter_notify;
	handlers[LeaveNotify]		= event_leave_notify;

	/* 
	 * the following aren't used in the main course of events, but are
	 * here in case a plugin selected for them on a window it created.
	 *
	 * note; not all events are here right now, although they should be
	 * eventually (all the ones that makes sense for a plugin to possibly
	 * need that is)
	 *
	 * XXX: I've decided I don't like this way of doing it; there's going
	 * to be a generic "misc X11 event" callback for plugins, and they can
	 * just do a switch inside of that callback for the type of event.
	 */
	handlers[KeyRelease]		= event_key_release;
	handlers[MotionNotify]		= event_motion_notify;
	handlers[VisibilityNotify]	= event_visibility_notify;
}

/* main loop for X event processing */
void event_loop() {
	XEvent event;

	event_init();

	while (1) {
		XNextEvent(display, &event);

		/* handle the event */
		if (event.type == shape_base + ShapeNotify)
			event_shape_notify(&event);
		else if (handlers[event.type] != NULL) 
			handlers[event.type](&event);

		/*
		 * if something sets restart_flag, we leave this function
		 * and let main() take the appropriate action.
		 */
		if (restart_flag)
			return;
	}
}
