/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"
#include "plugutil.h"
#include "menu.h"
#include "parseparams.h"

/* the root menu */
static menu_t	*rootmenu	= NULL;

/* the button that opens the root menu */
static int	menu_button	= Button3;

/* plugin intialization */
int init(plugin_t *plugin) {
	param_t *rmparam;
	dgroup_t *dgroup;
	pixmap_t *submenu_bullet;
	char *fontname;
	char *dgident;
	int button;

	/* get parameters */
	OPTIONAL_PARAM(button, &plugin->params, int, "menu_button", Button3);
	OPTIONAL_PARAM(fontname, &plugin->params, string, "menu_font", NULL);
	OPTIONAL_PARAM(submenu_bullet, &plugin->params, pixmap, "submenu_bullet", NULL);
	OPTIONAL_PARAM(dgident, &plugin->params, string, "menu_dgroup", NULL);
	if (dgident) {
		dgroup = dgroup_ident(dgident);
		if (!dgroup)
			PWARN(plugin, "undeclared decoration group identifier");
		free(dgident);
	} else {
		dgroup = NULL;
	}

	/* get the menu system up */
	if (menu_init(fontname, dgroup, submenu_bullet) != 0)
		goto free;

	/* build the rootmenu */
	if ((rootmenu = menu_create()) == NULL)
		goto free;

	/* build the rootmenu from parameters */
	if ((rmparam = plugin_find_param(&plugin->params, "rootmenu")) == NULL)
		PERR(plugin, "required block 'rootmenu' not present");
	parseparams(plugin, rootmenu, rmparam);
	return PLUGIN_OK;

free:
	if (fontname)
		free(fontname);
	return PLUGIN_UNLOAD;
}

/* clean up */
void shutdown() {
	if (rootmenu)
		menu_delete(rootmenu);
	menu_shutdown();
}

/* realize the menus */
int start(plugin_t *plugin) {
	if (menu_realize(plugin, rootmenu) == -1)
		PERR(plugin, "unable to realize menus");

	return PLUGIN_OK;
}

/* root window presses */
int root_button_press(plugin_t *plugin, screen_t *screen, XButtonEvent *e) {
	if (e->button == menu_button)
		menu_use(rootmenu, screen);

	return PLUGIN_OK;
}

/* presses on an open window */
int button_press(plugin_t *plugin, client_t *client, XButtonEvent *e) {
	menu_t *menu;

	menu = menu_getmenu(client);
	if (!menu)
		PERR(plugin, "recieved button_press on a client we didn't create");
	menu_click(menu, client, e);

	return PLUGIN_OK;
}

/* handle exposures on our windows */
int expose(plugin_t *plugin, client_t *client, XExposeEvent *e) {
	menu_t *menu;

	menu = menu_getmenu(client);
	if (!menu)
		PERR(plugin, "recieved expose on a client we didn't create");
	menu_expose(menu, client, e);

	return PLUGIN_OK;
}
