/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"
#include "plugutil.h"
#include <sys/wait.h>

/* the external program to run for root setting */
static char	*rootset_bin		= NULL;
/* flag used to tell it to tile the image */
static char	*rootset_tileflag	= NULL;
/* flag used to tell it to scale the image */
static char	*rootset_scaleflag	= NULL;
/* if we should wait for the external program to finish */
static int	wait_complete		= 0;
/* scale or tile: default to tile */
static int	scaleimg		= 0;
/* the file containing the image */
static char	*imagefn		= NULL;

/* get parameters */
int init(plugin_t *plugin) {
	/* params for the program to use to set the root */
	REQUIRED_PARAM(rootset_bin, plugin, &plugin->params, string, "rootset_bin");
	OPTIONAL_PARAM(rootset_tileflag, &plugin->params, string, "rootset_tileflag", NULL);
	OPTIONAL_PARAM(rootset_scaleflag, &plugin->params, string, "rootset_scaleflag", NULL);
	OPTIONAL_PARAM(wait_complete, &plugin->params, bool, "wait_complete", 0);

	/* the name of the file to tell rootset_bin to use */
	REQUIRED_PARAM(imagefn, plugin, &plugin->params, string, "imagefn");
	OPTIONAL_PARAM(scaleimg, &plugin->params, bool, "scaleimg", 0);

	return PLUGIN_OK;
}

/* free up used memory and then die */
void shutdown() {
	if (rootset_bin)
		free(rootset_bin);
	if (rootset_tileflag)
		free(rootset_tileflag);
	if (rootset_scaleflag)
		free(rootset_scaleflag);
	if (imagefn)
		free(imagefn);
}

/* perform our operation (set the root bg) and unload ourselves */
int start(plugin_t *plugin) {
	char *cmd;
	pid_t pid;
	int screen, status, len;

	/* find how much space we need for the 'exec', and the cmd string */
	len = 5 + strlen(rootset_bin) + 1;
	if (scaleimg && rootset_scaleflag)
		len += strlen(rootset_scaleflag) + 1;
	else if (rootset_tileflag)
		len += strlen(rootset_tileflag) + 1;
	len += strlen(imagefn) + 1;

	/* get memory for the cmd string */
	cmd = malloc(len);
	if (!cmd)
		return PLUGIN_UNLOAD;

	/* set up the cmd string */
	strcpy(cmd, "exec ");
	strcat(cmd, rootset_bin);
	strcat(cmd, " ");
	if (scaleimg && rootset_scaleflag) {
		strcat(cmd, rootset_scaleflag);
		strcat(cmd, " ");
	} else if (rootset_tileflag) {
		strcat(cmd, rootset_tileflag);
		strcat(cmd, " ");
	}
	strcat(cmd, imagefn);

	/* set it for each screen */
	for (screen = 0; screen < screen_count; screen++) {
		pid = action_exec(screen, cmd);
		if (wait_complete)
			waitpid(pid, &status, 0);
	}

	free(cmd);
	return PLUGIN_UNLOAD;
}
