/*
 ** Copyright (C) 2001-2005 Dirk-Jan C. Binnema <djcb@djcbsoftware.nl>
 **  
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **  
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; woithout even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **  
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **  
 */

#ifdef HAVE_CONFIG
#include <config.h>
#endif /*HAVE_CONFIG*/

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "gnuvd.h"

#ifndef VERSION
#define VERSION "?.?"
#endif /*VERSION*/

#define S_KARS_MAX 10

/* internal functions */
static void show_results     (Buffer *results, int keep_entities);
static char get_special_char (char *str);
static void help             ();

int
main (int argc, char* argv[])
{
	int   keep_entities = 0;
	char *word = NULL;
	
	VDQuery *vd_query  = NULL;
	VDError vd_ex;

	while (1) {

		int c = getopt (argc, argv, "kh");

		switch  (c) {
		case -1:
			break;
		case 'k':
			keep_entities = 1;
			continue;
		case 'h':
			help();
			return 0;
		default:
			help();
			return 1;
		}

		if (optind >= argc) {
			help();
			return 1;
		}  else
			break;
	}
			
	word  = argv[optind];
	vd_ex = VD_ERR_OK;
	
	/* prepare query */
	vd_query = vd_query_new (word, &vd_ex);
	if (!vd_query) {
		if (vd_ex != VD_ERR_OK)
			fprintf (stderr, "error: %s\n", 
				 vd_error_string(vd_ex));
		else 	
			fprintf (stderr, "an unknown error occured\n");
		return 1;
	}

	/* do query */
	vd_query_perform (vd_query, &vd_ex);
	if (vd_ex == VD_ERR_NOT_FOUND) {
		fprintf (stderr, "'%s' was not found in the dictionary\n", 
			 word); 
		vd_query_destroy (vd_query);
		return 2;	
	} else if (vd_ex != VD_ERR_OK) {
		fprintf  (stderr, "error: %s\n", vd_error_string (vd_ex));
		vd_query_destroy (vd_query);
		return 1;	
	} 

	show_results (vd_query->_result_buffer, keep_entities);
	vd_query_destroy (vd_query);
	
	return 0;
}


static void
show_results (Buffer* buffer, int keep_entities)
{
	BufferIterator i;
	int j;
	char k;
	char s_kars[S_KARS_MAX + 2];
	int skip_spc     = 1;
	int stress_pos;

	FILE *out        = stdout;

	/* output to PAGER or to STDOUT? */
	if (isatty(fileno(stdin)) && isatty(fileno(stdout))) {
		char *pager = getenv("PAGER");
		if (pager && pager[0] && strcmp(pager, "-") != 0) {
			out = popen(pager, "w");
			if (!out)
				out = stdout;
		}
	}
	
	
	/* strip superfluous space after */
	for (i = buffer_end (buffer) - 1; i >= buffer_begin(buffer) + 3; --i) 
		if ((buffer_at (buffer, i-3) == 'D') &&
		    (buffer_at (buffer, i-2) == 'L') &&
		    (buffer_at (buffer, i-1) == '>'))
			break;

	buffer_erase (buffer, i, buffer_end(buffer) - i - 1);

	for (j = buffer_begin (buffer); j < buffer_end(buffer); ++j) {
		
		/* Keep <=> */
		if (buffer_end(buffer) - j > 3) {
			if ((strncmp (buffer_data_pos(buffer,j), "<=>", 3) == 0)) {
				fprintf (out, "<=>");
				j+= 2;
				continue;
			}
		}
		
		/* DL->\n */
		if (buffer_end(buffer) - j > 5 && strncmp(buffer_data_pos(buffer,j),
							  "<DL>", 4) == 0) {
			j += 4;
			continue;
		}
	

		/* SMALL ->  */
		if (buffer_end(buffer) - j > 7) {
			if ((strncmp (buffer_data_pos(buffer,j),"<SMALL>", 7) == 0)) {
				fprintf (out, " (");
				j += 6;
				continue;
			}
		}
		
		/* /SMALL ->  */
		if (buffer_end(buffer) - j > 8) {
			if ((strncmp (buffer_data_pos(buffer,j),"</SMALL>", 8) == 0)) {
				fprintf (out, ")");
				j += 7;
				continue;
			}
		}


		/* DD->\n */
		if (buffer_end(buffer) - j > 8 && strncmp(buffer_data_pos(buffer,j),
							  "<DD><B>", 7) == 0) {
			j += 6;
			fprintf (out, "\n");
			continue;
		}

		/* 
		 *  Insert a ` at the start of a stressed syllable
		 *  contributed by Berteun <berteun@dds.nl> 
		 */
		if (buffer_end(buffer) - j > 11) {

			/* is at syllable separator or the start of the word? */
			if ((strncmp (buffer_data_pos(buffer,j), "<BIG>", 5) == 0) 
			    || (strncmp (buffer_data_pos(buffer,j), "&#183;", 6) == 0)) { 

				int dd_pos, m_pos;
				
				stress_pos = buffer_find_offset (buffer,j,"<U>",   3); /* next stress */
				dd_pos     = buffer_find_offset (buffer,j,"<DD>",  4); 
				m_pos      = buffer_find_offset (buffer,j,"&#183;",6); 
				
				if (stress_pos!=buffer_end(buffer) && stress_pos < dd_pos && stress_pos < m_pos) {
					fprintf (out, "`");
					j += (strncmp(buffer_data_pos(buffer,j),"<BIG>", 5) == 0) ? 4 : 5;
					continue;
				}
			}
		}
		
		/* remove tags */
		if (buffer_at(buffer,j) == '<' || buffer_at(buffer,j) == '\t') {
			while (j < buffer_end(buffer) && buffer_at(buffer,j) != '>')
				++j;
			continue;
		}
		
		/* replace special chars */
		if (buffer_at(buffer,j) == '&' && !keep_entities) {
			int c = 0;
			while (j < buffer_end(buffer) 
			       && buffer_at(buffer,j) != ';' && c < S_KARS_MAX)
				s_kars [c++] = buffer_at(buffer,j++);
			if (buffer_at(buffer,j) == ';') {
				s_kars [c++] = ';';
				s_kars [c]   = '\0';

				k = get_special_char (s_kars);
				if (k)
					fprintf (out, "%c", k);
			}
			continue;
		}
		
		/* skip superfluous space at beginning */
		if (skip_spc) {
			if (isspace(buffer_at(buffer,j)))
				continue;
			else
				skip_spc = 0;
		}
		fprintf (out, "%c", buffer_at(buffer,j));
	}

	/* if we were outputing to PAGER, close it */
	if (out != stdout) {
		fflush (out);
		pclose (out);
	}
}


static char
get_special_char (char *str)
{
	int i;
	
	struct map_entry {
		char *entity;
		char ascii;
	};
	
	static struct map_entry symbol_map[]  = {
		/* special chars */
				
		{ "&#36;",     '$'},
		{ "&#37;",     '%'},
		{ "&#38;",     '&'},
		{ "&#39;",     '\''},
		{ "&#64;",     '@'},
		{ "&#126;",    '~'},
		{ "&#167;",    ''},
		{ "&#176;",    ''},
		{ "&#178;",    ''},
		{ "&#180;",    '0'},
		{ "&#183;",    '|'},
		{ "&#215;",    'x'},
		{ "&#229;",    ''},
		{ "&#956;",    ''},

		/* A/a */
		{ "&agrave;",  ''},
		{ "&acute;",   ''},
		{ "&acirc;",   ''},
		{ "&auml;",    ''},
		{ "&aring;",   ''},
		{ "&aelig;",   ''}, 
		
		{ "&Agrave;",  ''},
		{ "&Acute;",   ''},
		{ "&Acirc;",   ''},
		{ "&Auml;",    ''},
		{ "&Aring;",   ''},
		{ "&Aelig;",   ''}, 

		/* C/c */
		{ "&ccedil;",  ''},
		{ "&Ccdeil;",  ''},
		
		/* E/e */
		{ "&egrave;",  ''},
		{ "&ecute;",   ''},
		{ "&ecirc;",   ''},
		{ "&euml;",    ''},
		{ "&eacute;",  ''}, /* weird -> 'sao tom_' */ 
			
		{ "&Egrave;",  ''},
		{ "&Ecute;",   ''},
		{ "&Ecirc;",   ''},
		{ "&Euml;",    ''},

		/* I/i */
		{ "&igrave;",  ''},
		{ "&icute;",   ''},
		{ "&icirc;",   ''},
		{ "&iuml;",    ''},
			
		{ "&Igrave;",  ''},
		{ "&Icute;",   ''},
		{ "&Icirc;",   ''},
		{ "&Iuml;",    ''},

		/* n */
		{ "&ntilde;",  ''},
		{ "&Ntilde;",  ''},
		
		
		/* O/o */
		{ "&ograve;",  ''},
		{ "&ocute;",   ''},
		{ "&ocirc;",   ''},
		{ "&ouml;",    ''},
		{ "&oslash;",  ''},
		{ "&otilde;",  ''},
		
		{ "&Ograve;",  ''},
		{ "&Ocute;",   ''},
		{ "&Ocirc;",   ''},
		{ "&Ouml;",    ''},
		{ "&Oslash;",  ''},
		{ "&Otilde;",  ''},
		
		/* U/u */
		{ "&ugrave;",  ''},
		{ "&ucute;",   ''},
		{ "&ucirc;",   ''},
		{ "&uuml;",    ''},
			
		{ "&Ugrave;",  ''},
		{ "&Ucute;",   ''},
		{ "&Ucirc;",   ''},
		{ "&Uuml;",    ''},
		
		/* ij */
		{ "&yuml;",    ''}};
	



	for (i = 0; i != sizeof (symbol_map) / sizeof (struct map_entry); ++i)
		if (strcmp (str, symbol_map[i].entity) == 0)
			return symbol_map[i].ascii;
	return '?';
}


static void
help()
{
	printf ("gnuvd version " VERSION  ",\n"
		"(c) 2001-2005 Dirk-Jan C. Binnema <djcb@djcbsoftware.nl>\n\n"
		"usage: gnuvd [options] word\n"
		"where [options] are:\n"
		"\t-k: keep html entities (i.e. &egrave;)\n"
		"\t-h: show this help message\n");
}


