/* Josh Pieper, (c) 2000 */

/* This file is distributed under the GPL, see file COPYING for details */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <string.h>
#ifdef HAVE_REGEX_H
 #include <regex.h>
#endif

#include "blacklist.h"
#include "conf.h"
#include "gnut.h"
#include "host.h"
#include "list.h"
#include "qry.h"
#include "queue.h"

Gnut_Queue *push_list;   /* qreplies for which we are currently expecting
			    push connections */

Gnut_List *search_list;   /* qreplies resulting from our current search */

#ifndef PTHREADS_DRAFT4
 pthread_mutex_t query_mutex = PTHREAD_MUTEX_INITIALIZER;
 pthread_mutex_t push_mutex = PTHREAD_MUTEX_INITIALIZER;
#else
 pthread_mutex_t query_mutex;
 pthread_mutex_t push_mutex;
#endif

void fre_qrpl(query_resp ***x, int bugnum)
{
  yfre((void **) x, bugnum);
}

void fre_qrp(query_resp **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

/* %%% this routine is obsolete in uniname */
void query_kname(query_resp *a_qrp) /* 0.4.27.c13 */
{
  dqi(0x016a);
  if (a_qrp) {
    a_qrp->qr_ndata[0] = 0; /* 0.4.27.c13 */
  }
  dqi(0x0166);
}

void query_kill(query_resp **p_qrp, int tn1) /* 0.4.27.c13 */
{
  query_resp *a_qrp;

  if (p_qrp) {
    a_qrp = *p_qrp;
    if (a_qrp) {
      query_kname(a_qrp); /* 0.4.27.c13 */
      dqi(0x0167);
      fre_qrp(p_qrp, tn1);
    }
  }
}

/* Allocate a query.resp record and fill in all the fields, including
   copying the name. If the name parameter is null, creates a query.resp
   with a null name.
   0.4.27.c19
*/
query_resp *query_create(int tn1,
			 uint8 ip[4], uint8 port[2], uint32 ref, uint32 speed,
			 uint32 size, uint8 guid[16],
			 int16 htype, int16 rating, /* 0.4.27.c21 */
			 int32 uindex, char flag,
			 int dcache, int retry, uint32 crc, char *name)
{
  query_resp *rv;
  uint16 nlen;

  /* 0.4.27.c13 */
  nlen = 0;
  if (name) {
    nlen = strlen(name);
    if (nlen > QR_NAME_MAXLEN) {
      nlen = QR_NAME_MAXLEN;
    }
  }

  rv = (query_resp *) ymaloc(sizeof(query_resp) + nlen, tn1);
  SMCPY(rv->qr_ip, ip);
  SMCPY(rv->qr_port, port);
  rv->qr_ref = ref;
  rv->qr_speed = speed;
  rv->qr_size = size;
  SMCPY(rv->qr_guid, guid);
  rv->qr_htype = htype; /* 0.4.27.c21 */
  rv->qr_rating = rating; /* 0.4.27.c21 */
  rv->qr_uindex = uindex;
  rv->qr_flag = flag;
  rv->qr_crc = crc;
  rv->qr_dcache = dcache;
  rv->qr_retry = retry;
  rv->qr_nlen = nlen;
  if (name) {
    /* We use strncpy and explicitly set the null byte because we're
       paranoid -- we're afraid another thread might have changed the
       memory pointed to by "name" in the time that elapsed between
       the strlen call above and this call. 0.4.27.c13 */
    strncpy(rv->qr_ndata, name, nlen+1);
    rv->qr_ndata[nlen] = 0;
  }

  return rv;
}

/* Create a new query.resp containing all the same fields as an existing
   one, also allocates and copies the name
   Changed for 0.4.27.c19, calls query.create now */
query_resp *query_copy(query_resp *a_qrp, uint32 tn1)
{
  dqi(0x0114);
  return(query_create(tn1, a_qrp->qr_ip, a_qrp->qr_port, a_qrp->qr_ref,
		      a_qrp->qr_speed, a_qrp->qr_size, a_qrp->qr_guid,
		      a_qrp->qr_htype, a_qrp->qr_rating, /* 0.4.27.c21 */
		      a_qrp->qr_uindex, a_qrp->qr_flag, a_qrp->qr_dcache,
		      a_qrp->qr_retry, a_qrp->qr_crc, a_qrp->qr_ndata));
}

int query_init()
{
  push_list = gnut_queue_new(516);
  search_list = 0;
  return 0; 
}

/* Add a qreply to the push list, atomically within push_mutex */
int query_push_add(query_resp *a_qrp)
{
  int s;

  dqi(0x0140);

  gd_s(1, "query_push.add entering qr="); gd_p(1, a_qrp); gd_s(1, "\n");
  pthread_mutex_lock(&push_mutex);

  dqi(0x0141);
  gnut_queue_insert(push_list, a_qrp, 504);
  dqi(0x0142);

  /* Punt oldest one if necessary */
  if (push_list->size > MAX_OUTSTANDING_PUSHES) {
    query_resp *d_qrp;

    dqi(0x0143);
    d_qrp = gnut_queue_remove(push_list);
    dqi(0x0144);
    if (d_qrp) {
      /* qr-del */
      dqi(0x0145);
      query_kill(&d_qrp, 545); /* 0.4.27.c13 */
    }
  }

  dqi(0x0146);

  s = push_list->size;
  pthread_mutex_unlock(&push_mutex);
  gd_s(1, "query_push.add exit push.list->size="); gd_i(1, s); gd_s(1, "\n");

  dqi(0x0147);
  return 0;
}

/* Locates a qreply (by GUID and ref) in the push list, and if found,
   returns its pointer after removing it but not deallocating its data.
   This is done atomically within push_mutex */
query_resp * query_push_find(uchar guid[16], uint32 ref)
{
  Gnut_List *gltmp;
  query_resp *a_qrp;
  query_resp *r_qrp;

  r_qrp = 0;

  pthread_mutex_lock(&push_mutex);

  for(gltmp=push_list->head; gltmp && (r_qrp == 0); gltmp=gltmp->next) {
    a_qrp = gltmp->data;
    if ((memcmp(a_qrp->qr_guid, guid, sizeof(a_qrp->qr_guid)) == 0)
	&& (a_qrp->qr_ref == ref))
      {
	/* we have a match!!! */
	r_qrp = a_qrp;
      }
  }

  if (r_qrp) {
    gnut_queue_delete(push_list, r_qrp);
  }

  pthread_mutex_unlock(&push_mutex);

  return r_qrp;
}

#if 0
/* 0.4.27.c19 I removed this unused routine */
/* Hasn't been used since 0.3.29, and not necessary now that the qr.crc is
   used to filter out duplicates even more thoroughly before they get put in */
/* clears out dups in the query list, this can take a long time so it
   shouldn't be called very often */
int query_unique()
{
  Gnut_List *tmp;
  Gnut_List *cur, *prev;
  query_resp *a_qrp1, *a_qrp2;
  int i;

  for (i=0,tmp=search_list; tmp; tmp=tmp->next,i++) {
    /* now for each element, we need to go through the rest
     * of the list and unlink any that are copies of this... */
    /*    printf("%i\n",i); */
    /* this is to keep abormally large lists from taking forever */
    if (i>500) {
      return 0;
    }
    a_qrp1 = (query_resp *) tmp->data;
    for (prev=tmp,cur=tmp->next; cur; cur=cur->next) {
      a_qrp2 = cur->data;
      if (memcmp(a_qrp1->qr_guid, a_qrp2->qr_guid, 16)==0
	  && a_qrp1->qr_ref==a_qrp2->qr_ref)
	{
	  /* we need to unlink this one from the list and continue */
	  prev->next = cur->next;
	  query_kill(&a_qrp2, 168); /* 0.4.27.c13 */ /* qr-del */
	  fre_gl(&cur, 170);
	  cur = prev;
	}
      prev = cur;
    }
  }
  return 0;
}
#endif

uint32 query_num()
{
  uint32 ret;
  pthread_mutex_lock(&query_mutex);

  ret = gnut_list_size(search_list);

  pthread_mutex_unlock(&query_mutex);
  return ret;
}

int query_lock()
{
  pthread_mutex_lock(&query_mutex);
  return 0;
}

int query_unlock()
{
  pthread_mutex_unlock(&query_mutex);
  return 0;
}

Gnut_List *query_retrieve()
{
  return search_list;
}

/* query.add is used to add a query reply to the list for the 'find'
   and 'resp' commands. It allocates a new query.resp structure, you
   have to supply all the relevant data fields. */
int query_add(uint8 ip[4], uint8 port[2],
	      uint32 ref, uint32 speed, /* 0.4.27.c11 */
              uint32 size, /* 0.4.27.c11 */
	      char *name, uint8 guid[16]
	      , int16 htype, int16 rating /* 0.4.27.c21 */
	      )
{
  query_resp *a_qrp;
  int i;
  char *t;
  uint32 qcrc;
  int dup;

  /* If we're in a VPN, we ignore query results from other VPN's of
   * differing class, because they can't be downloaded even with a
   * push request. */
  if (host_ok_for_neither(ip)) {
    return 0;
  }

  /* If we already have max responses, forget this one */
  if (query_num() >= conf_get_int("max_responses")) {
    return 0;
  }

#if 0
  printf("add ");
  for(i=0; i<16; i++) {
    printf("%02x", guid[i]);
  }
  printf("\n");
#endif

  /* 0.4.27.c19 */
  a_qrp = query_create(298, /* qr-new */
		       ip, port, ref, speed, size, guid,
		       htype, rating, /* 0.4.27.c21 */
		       0, ' ', 0, 0,
		       0, name);

  /* Calculate CRC that identifies whether this result is new */
  crc32_start(&qcrc);
  for(i=0; i<4; i++) {
    crc32_add8(&qcrc, ip[i]);
  }
  for(i=0; i<2; i++) {
    crc32_add8(&qcrc, port[i]);
  }
  t = name;
  for(i=0; (i<100) && (*t); i++) {
    crc32_add8(&qcrc, *t);
    t++;
  }
  a_qrp->qr_crc = qcrc;

  /* Search list to see if it's there (no mutex needed, because
   * prepend is sufficiently atomic) */
  dup = 0;
  {
    Gnut_List *l;
    query_resp *a_qrp2;

    l = search_list;
    while(l) {
      a_qrp2 = l->data;
      if (a_qrp2->qr_crc == qcrc) {
	dup = 1;
      }
      l = l->next;
    }
  }

  if (dup == 0) {
    /* The result is new -- we can add it now. */
    pthread_mutex_lock(&query_mutex);
    search_list = gnut_list_prepend(search_list, a_qrp);
    pthread_mutex_unlock(&query_mutex);
  } else {
    /* We already have this one. Fr.ee up the stuff we al.located. */
    query_kill(&a_qrp, 237); /* qr-del */ /* 0.4.27.c13 */
  }

  return 0;
}

/* query.add.cache creates a query.resp structure and starts a download
   thread to get the file and put it in the file cache. */
int query_add_cache(uint8 ip[4], uint8 port[2],
		    uint32 ref, /* 0.4.27.c11 */
		    uint32 speed, uint32 size, /* 0.4.27.c11 */
		    char *name, uint8 guid[16]
		    , int16 htype, int16 rating /* 0.4.27.c21 */
		    )
{
  query_resp *a_qrp;
  int min_cache_size;
  char *lcname;
  char *s;

  min_cache_size = conf_get_int("min_cache_size");
  if (min_cache_size == 0) {
    min_cache_size = 15;
  }

  /* only cache files 1/4 the size of the cache or smaller
   * (limits thrashing, I hope - Ray) */
  /* 0.4.27.c11 Changed the math to avoid overflow, cache_size may now be
     2^31 x 1024 bytes (old math limited it to 2^31) */
  if ( ((size / 1024) < min_cache_size)
       || ((size / 1024) > (conf_get_int("cache_size") / 4)) ) {
    return 0;
  }

  /* Lowercase the name and check if it's blacklisted */
  lcname = ymaloc(strlen(name)+1, 517);
  strcpy(lcname, name);
  for(s=lcname; *s; s++) {
    *s = tolower(*s);
  }

  {
    int allow;

    allow = gnut_blacklist_allow(BLACKLIST_TYPE_SEARCH, lcname, 0, 0, 1);
    fre_str(&lcname, 518);
    if(! allow) {
      if (gc_verbose & 8) {
	printf("\n%s blocked from cache.\n", name);
      }
      return 0;
    }
  }

  /* 0.4.27.c19 */
  a_qrp = query_create(299, /* qr-new */
		       ip, port, ref, speed, size, guid, 
		       htype, rating, /* 0.4.27.c21 */
		       0, ' ', 1, 0, 0,
		       name);

  gnut_xfer_start(a_qrp);

  return 0;
}

/* query.clear deletes all the entries in the search list. */
int query_clear()
{
  Gnut_List *gltmp;
  query_resp *a_qrp;
  
  pthread_mutex_lock(&query_mutex);

  /* qr-del */
  for (gltmp = search_list; gltmp; gltmp = gltmp->next) {
    a_qrp = gltmp->data;
    if (a_qrp) {
      query_kname(a_qrp); /* 0.4.27.c13 */
    }
  }

  search_list = gnut_list_fre(search_list);

  pthread_mutex_unlock(&query_mutex);
  return 0;
}

#if 0
/* 0.4.27.c19 I removed this unused routine */
/* Not used, and not sure why Josh wrote it */
/* query.find takes a guid and refnum and looks for a matching entry in
   the search list, if found it returns a copy. */
query_resp *query_find(uchar guid[16], uint32 ref)
{
  Gnut_List *gltmp;
  query_resp *a_qrp1, *a_qrp2;

  pthread_mutex_lock(&query_mutex);
  
  for(gltmp=search_list; gltmp; gltmp=gltmp->next) {
    a_qrp1 = gltmp->data;
    if (memcmp(a_qrp1->qr_guid, guid, sizeof(guid))==0 && a_qrp1->qr_ref==ref) {
      /* we have a match!!! */
      a_qrp2 = query_copy(a_qrp1, 300); /* qr-new */ /* 0.4.27.c13 */
      pthread_mutex_unlock(&query_mutex);
      return a_qrp2;
    }
  }

  pthread_mutex_unlock(&query_mutex);
  return 0;
}
#endif

/* Locate an item in the search list with the given index, mark it
   with a flag, copy it and return a pointer to the copy.
   Pass 0 for the flag if you don't want to change the flag of the
   query.resp */
query_resp *query_index(int32 uindex, char flag)
{
  /* we need to actually make a copy, because as soon as the mutex
   * is unlocked, we aren't guaranteed of the original's existence */
  Gnut_List *gltmp;
  query_resp *a_qrp1, *a_qrp2;

  pthread_mutex_lock(&query_mutex); {
    for(gltmp = search_list; gltmp; gltmp = gltmp->next) {
      a_qrp1 = gltmp->data;
      if (a_qrp1->qr_uindex == uindex) {
	/* we have a match!!! */
	if (flag) {
	  a_qrp1->qr_flag = flag;
	}
	a_qrp2 = query_copy(a_qrp1, 301); /* qr-new */
	pthread_mutex_unlock(&query_mutex);
	return a_qrp2;
      }
    }
  } pthread_mutex_unlock(&query_mutex);
  return 0;
}

/* 0.4.27.c04 */
void query_tag_filter(char flag, int (*testfunc)(query_resp *))
{
  Gnut_List *gltmp;
  query_resp *a_qrp1;
  int flagme;

  pthread_mutex_lock(&query_mutex); {
    for(gltmp = search_list; gltmp; gltmp = gltmp->next) {
      a_qrp1 = gltmp->data;
      flagme = (*testfunc)(a_qrp1);
      if (flagme) {
	a_qrp1->qr_flag = flag;
      }
    }
  } pthread_mutex_unlock(&query_mutex);
}
