/* Jon Arney, (c) 2000 */

/* This file is distributed under the GPL, see file COPYING for details */

#ifndef GNUT_HTTP_H
#define GNUT_HTTP_H

#include "transfer.h"

#define GNUT_HTTP_DEBUG      2
#define GNUT_HTTP_INTERNALS_DEBUG  3
#define HTTP_REQUEST_DEBUG    3
#define HTTP_HEADER_DEBUG    3
#define HTTP_IO_DEBUG      3

/*
 * This type definition
 * is meant for handling HTTP
 * requests to the Gnutella-net
 * network.
 */
typedef struct gnut_http_query_st {
  /*
   * Where did this req come from?
   */
  uchar   ip[4];
  uint16   port;
  char    guid[16];
  time_t  last_used;
  Gnut_List *query_list;
} gnut_http_query_t;

typedef struct http_header_st {
  char *name;
  char *value;
} http_header_t;

/* This is a representation of an HTTP request. */
typedef struct http_request_st {
  char *method;
  char *version;
  char *url_file;
  int sock;
  Gnut_List *rq_headers;
  Gnut_List *response_headers;
} http_request_t;

typedef struct http_cookie_st {
  char *value;
  char *path;
  char *domain; /* Unused */
  time_t expiration; /*Unused */
} http_cookie_t;

#define HTTP_MAX_HEADER_SIZE 64

#ifdef __cplusplus
extern "C" {
#endif

  extern uint32 urate_history[10];  /* 0.4.27.c27 */
  extern uint32 best_urate; /* 0.4.27.c27 */
  extern int urate_clipped; /* 0.4.27.c27 */
  extern int urate_measured; /* 0.4.27.c27 */
  extern int gh_did_upload;  /* 0.4.27.c27 */

  void fre_ghq(gnut_http_query_t **x, int bugnum);

  void cgi_to_normal(char * news, char * olds);

  void update_urate(void);

  /* Serve a share file. */
  int gnut_http_serve_file(gnut_transfer *gt, http_request_t *req);

  /* Get a list of GNUT files... */
  int gnut_http_file_list(gnut_transfer *gt, http_request_t *req);

  /* Provide a gateway into the GNUT network. */
  int gnut_http_file_search(gnut_transfer *gt, http_request_t *req);

  /* Make a search form... */
  int gnut_http_file_search_form(gnut_transfer *gt, http_request_t *req);

  /* Call this when attempting to find a query-er who requested this
   * result. */
  int gnut_http_result_append(uint8 ip[4], uint8 port[2],
			      uint32 ref, /* 0.4.27.c11 */
			      uint32 speed, uint32 size, /* 0.4.27.c11 */
			      char *name, uint8 guid[16]
			      , int16 htype, int16 rating /* 0.4.27.c21 */
			      ); 

  void gnut_http_write_top(http_request_t *req);
  void gnut_http_write_bottom(http_request_t *req);
  int gnut_http_write_form(http_request_t *req, int ttw, int rets);

  void gnut_http_template_scan();

  void fre_hth(http_header_t **x, int bugnum);

  /* A new header object to be parsed from 'header_line'. */
  http_header_t *http_header_new(void);

  /* Parse an HTTP header into the header object. */
  int http_header_parse(http_header_t *hdr, char *buffer);

  /* Set the contents of an HTTP header. */
  int http_header_set(http_header_t *hdr, char *name, char *value);

  /* Delete memory associated with a header. */
  void http_header_delete(http_header_t *hdr);

  void fre_hrt(http_request_t **x, int bugnum);
  void fre_hct(http_cookie_t **x, int bugnum);

  /* Allocate a new request object which talks on socket 'socket'. */
  http_request_t *http_request_new(int socket);

  /* Delete memory associated with request_delete. */
  void http_request_delete(http_request_t *req);

  /* Parse a new HTTP request from a request line. */
  int http_request_parse(http_request_t *req, char *request);

  /* Read a header from the input source. */
  int http_request_read_header(http_request_t *req);

  /* Examine the HTTP headers and find one of the values. */
  char *http_request_find_header(http_request_t *req, char *name);

  /* Write a raw string to the output source. */
  int http_write_str(http_request_t *req, char *a);
  int http_write_va(http_request_t *req, int maxlen, char *format, ...);

  /* Add a header to the response of this request. */
  int http_response_header_add(http_request_t *req, char *name, char *value);
  int http_response_header_add_va(http_request_t *req, char *name,
				  char *fmt, ...);

  /* Set a cookie. */
  http_cookie_t *http_cookie_new(void);
  void http_cookie_delete(http_cookie_t *cookp);
  int http_response_set_cookie(http_request_t *req, http_cookie_t *cookp);
  int http_header_get_cookie(http_request_t *req, http_cookie_t *cookp);

  /* Write the HTTP response haders from the response to this request. */
  int http_response_header_write(http_request_t *req);

#ifdef __cplusplus
}
#endif

#endif
