/* Josh Pieper, (c) 2000
   Robert Munafo, (c) 2001

   This file is distributed under the GPL, see file COPYING for details */

/* Proposed optimization of host cache:

         packets ---.
                     v
   full cache --> LRU cache --> new connections
    (50000)        (1000)

 - full cache is maintained just as before: hosts are added by host.restore()
   and by sniffing IP's from packets, and hosts are removed when a connection
   closes or fails to open.

 - LRU cache is filled two ways:
    - any IP seen in a packet, if not currently in the LRU cache,
      is copied and added. If this makes the LRU cache overfull,
      the oldest entries are deleted
    - if LRU cache is less than half-full, random hosts from the
      full cache are copied and added (but left in the full cache)
 - each time a new random connection is started, it gets its IP from
   the end (oldest) entry on the LRU cache, and that entry is deleted

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#ifndef WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#endif

#include "conf.h"
#include "hash.h"
#include "host.h"
#include "http.h"
#include "net.h"
#include "list.h"
#include "threads.h"

Gnut_Hash *host_list = 0;

uint hosts_num = 0;
float64 hosts_files = 0;
float64 hosts_kbytes = 0;

#define HLRUSIZE 1024
host_entry * host_lru;
int next_lrup;
uint32 hlru_num;

#ifndef PTHREADS_DRAFT4
 pthread_mutex_t host_mutex=PTHREAD_MUTEX_INITIALIZER;
#else
 pthread_mutex_t host_mutex;
#endif

/* 0.4.27.c21 */
eqhd_vendor evendors[] = {
  {"xxxx", "Unknown"},    /* This one must be first */

  {"BEAR", "BearShare"},  /* These two are first because        */
  {"LIME", "LimeWire"},   /*         they occur in more packets */

  /* The rest are in alphabetical order */
  {"CULT", "Cultiv8r"},
  {"GNOT", "Gnotella"},
  {"GNUC", "Gnucleus"},
  {"GNUT", "gnut"},       /* That's us! */
  {"GTKG", "Gtk-Gnutella"},
  {"HSLG", "Hagelslag"},
  {"MACT", "Mactella"},
  {"NAPS", "NapShare"},
  {"OCFG", "OpenCola"},
  {"TOAD", "ToadNode"},
  {0, 0}
};

void fre_hen(host_entry **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

int host_class(uchar *ip)
{
  /* The VPN filtering needs to be conditional: If a VPN host
   * is found, we can try connecting to it and if successful, can use it.
   * But we shouldn't blindly add all VPN addresses because a lot come
   * in from other VPNs elsewhere on the Internet and those should be
   * ignored */
  if (ip[0] == 10) {
    /* Class A virtual private network */
    return VPN_A;
  }

  if (ip[0] == 172) { /* tagok */
    if ((ip[1] >= 16) && (ip[1] <= 31)) {
      /* Class B virtual private network */
      return VPN_B;
    }
  }

  if ((ip[0] == 192) && (ip[1] == 168)) { /* tagok */
    /* Class C virtual private network */
    return VPN_C;
  }

  if ((ip[0] == 169) && (ip[1] == 254)) { /* tagok */
    /* APIPA address */
    return VPN_P;
  }

  if (ip[0] == 127) { /* tagok */
    /* Local loopback range */
    return BAD_ADDR;
  }

  if (ip[0] == 0) {
    /* class A network 0 -- old-style broadcast */
    return BAD_ADDR;
  }

  if ((ip[0] == 192) && (ip[1] == 0) && (ip[2] == 2)) { /* tagok */
    /* TEST-NET */
    return BAD_ADDR;
  }

  if (ip[0] >= 224) { /* tagok */
    /* Class D multicast and class E reserved
     * %%% not sure about this -- parts of this range might be another type
     * of VPN address within certain institutional networks */
    return BAD_ADDR;
  }

  return 0;
}

int can_connect(uchar *from_ip, uchar *to_ip)
{
  if (gc_no_rfc_1597) {
    return 1;
  }
  if (host_class(to_ip)) {
    /* to_ip is a VPN address: they're connectable only if in the same
     * VPN class as from_ip (this happens when both are within the same
     * institution's private IP network) */
    if (host_class(from_ip) & host_class(to_ip)) {
      return 1;
    }
    return 0;
  }
  /* Else: to_ip is a normal address */
  return 1;
}

int host_ok_for_get(uchar *ip)
{
  return(can_connect(net_local_ip(), ip));
}

int host_ok_for_push(uchar *ip)
{
  return(can_connect(ip, net_local_ip()));
}

int host_ok_for_either(uchar *ip)
{
  return(can_connect(net_local_ip(), ip) || can_connect(ip, net_local_ip()));
}

int host_ok_for_both(uchar *ip)
{
  return(can_connect(net_local_ip(), ip) && can_connect(ip, net_local_ip()));
}

int host_ok_for_neither(uchar *ip)
{
  if(can_connect(net_local_ip(), ip) || can_connect(ip, net_local_ip())) {
    return 0;
  }
  return 1;
}

uchar host_hash(void *a)
{
  uint32 crc; /* 0.4.27.c09 */
  host_entry *he;
  he = a;

  /* 0.4.27.c09 (all 6 following lines) */
  crc = crc32_block(&(he->ip[0]), 4);
  crc32_add8(&crc, he->port[0]);
  crc32_add8(&crc, he->port[1]);
  crc ^= (crc >> 16);
  crc ^= (crc >> 8);
  return (crc & 0xff);
}

int host_compare(void *a, void *b)
{
  host_entry *he1;
  host_entry *he2;
  he1 = (host_entry *) a;
  he2 = (host_entry *) b;

  if ((memcmp(he1->ip, he2->ip, 4) == 0)
      && (memcmp(he1->port, he2->port, 2) == 0)) {
    return 0;
  }
  return -1;
}

void clear_he(host_entry * he)
{
  he->ip[0] = 0;
  he->ip[1] = 0;
  he->ip[2] = 0;
  he->ip[3] = 0;
  he->port[0] = 0;
  he->port[1] = 0;
  he->files = 0;
  he->kbytes = 0;
}

int host_init()
{
  int i;

  gd_s(3, "host_init entering\n");
  host_list = gnut_hash_new(host_hash, host_compare);

  host_lru = ymaloc(HLRUSIZE * sizeof(host_entry), 154);
  hlru_num = 0;

  for(i=0; i<HLRUSIZE; i++) {
    clear_he(&(host_lru[i]));
  }

  return 0;
}

/* Gets a random host_entry from the recent-hosts array */
void host_get_lru(host_entry *he)
{
  int going;
  int countdown;

  going = 1;
  countdown = 20;
  while(going) {
    next_lrup = (next_lrup + (rand() & 0xff)) % HLRUSIZE;
    if (host_lru[next_lrup].ip[0]) {
      *he = host_lru[next_lrup];
      clear_he(&(host_lru[next_lrup]));
      if (hlru_num) {
	hlru_num--;
      }
      going = 0;
    }
    if (countdown <= 0) {
      clear_he(he);
      going = 0;
    }
    countdown--;
  }
}

/* Store a host_entry in the recent-hosts array */
void host_put_lru(host_entry *he)
{
  uint32 crc;
  int i;

  crc = crc32_block(he, 6);
  i = crc % HLRUSIZE;
  if (host_lru[i].ip[0] == 0) {
    if (hlru_num < HLRUSIZE) {
      hlru_num++;
    }
  }
  host_lru[i] = *he;
}

/* Internal routine to remove a host entry from the host hash. This
 * should only be called if the mutex is locked */
void host_rmv_core(host_entry *h)
{
  hosts_num--;
  hosts_files -= h->files;
  hosts_kbytes -= h->kbytes;
  gnut_hash_remove(host_list, h);
  fre_hen(&h, 497);
}

int host_remove(uchar ip[4], uchar port[2])
{
  host_entry he;
  host_entry *he2;

  gd_s(1,"host.remove entering ");
  gd_i(1, ip[0]); gd_s(1, "."); gd_i(1, ip[1]); gd_s(1, ".");
  gd_i(1, ip[2]); gd_s(1, "."); gd_i(1, ip[3]); gd_s(1, "\n");

  pthread_mutex_lock(&host_mutex);

  memcpy(he.ip, ip, 4);
  memcpy(he.port, port, 2);

  dqi(0x0154);
  he2 = gnut_hash_find(host_list, &he);
  if (!he2) {
    pthread_mutex_unlock(&host_mutex);
    /* This happens if the same host was connected twice and both
     * connections close */
    gd_s(1, "host.remove  host not found!\n");
    return -1;
  }

  host_rmv_core(he2);

  pthread_mutex_unlock(&host_mutex);

  return 0;
}

int purged;

host_entry * host_add(host_entry *he)
{
  host_entry * rv;

  rv = he;

  gd_s(3, "host.add entering host_list="); gd_p(3, host_list);
  gd_s(3, " he="); gd_p(3, he); gd_s(3, "\n");

  pthread_mutex_lock(&host_mutex);

  dqi(0x0155);
  if (gnut_hash_find(host_list, he)) {
    /* Already there */
    /* printf("host.add skip duplicate %i.%i.%i.%i\n", he->ip[0], he->ip[1],
       he->ip[2], he->ip[3]); */
  } else {
    /* It's new */
    int bs, bl;
    host_entry * lamb;

    /* Perform garbage collection on a per-bucket basis. This isn't optimal
       (we don't try to keep track of which hosts are actually active)
       but it doesn't matter because if the deleted host is active,
       it will send a PONG pretty soon and get put right back into the list,
       thereby bumping the next-oldest entry in this bucket and so on until
       a really inactive host is deleted. */
    bs = gnut_hash_bucketsize(host_list, he);

    /* Compute the limit for this hash bucket. We do it in a way that
       preserves the low 8 bits of the list_size setting. For example,
       if the list_size were 286. the first 30 buckets would
       get bl=2, and the other (256-30) would get bl=1               */
    bl = gnut_hash_bucketnum(host_list, he);
    bl = ((gc_host_list_size & 0xff) > bl) ? 1 : 0;
    bl += gc_host_list_size >> 8;
    while (bs >= bl) {
      lamb = gnut_hash_bucketoldest(host_list, he);
      if (lamb) {
	purged++;
        /* printf("N %d bs %d lamb %d\n", purged, bs, lamb->files); */
        host_rmv_core(lamb);
	bs = gnut_hash_bucketsize(host_list, he);
      } else {
	bs = 0;
      }
    }

    gnut_hash_insert(host_list, he, 499);
    hosts_num++;
    hosts_files += he->files;
    hosts_kbytes += he->kbytes;
    rv = 0;
  }

  pthread_mutex_unlock(&host_mutex);

  gd_s(3,"host.add returning\n");

  return(rv);
}

/* host.restore loads the host list file (usually ~/.gnut_hosts)
 * by calling host.add() on each entry to add the host IPs to the
 * host IP hashtable. */
int host_restore(char *fname)
{
  FILE *fp;
  host_entry *he;
  char bf[256];
  char *buf;
  int s_ip[4];
  int s_port;
  uint16 p16;
  int errf;
  int good;
  uint32 s_files;
  uint32 s_kbytes;
  int n1, n2;
  uint32 sr;
  int nserv;

  /* 0.4.27.c27 Since we restore the share upload rate history in this
     routine, we might as well init the other EQHD (extended QueryHits
     descriptor) values here too. */
  gh_did_upload = 0;
  gh_did_receive = 0;
  urate_measured = 0;

  errf = 0;

  gd_s(2, "host.restore entering\n");

  fp = fopen(fname, "r");

  if (fp == 0) {
    gd_s(1, "host.restore can't open: "); gd_s(1, fname); gd_s(1, "\n");
    return -1;
  }

  host_clear();

  gd_s(3, "host.restore host_list="); gd_p(3, host_list); gd_s(3, "\n");
  gd_s(2, "host.restore starting to read hosts...\n");

  n1 = n2 = 0;
  he = 0;
  nserv = 0; /* 0.4.27.c27 */
  while (fgets(bf, sizeof(bf), fp)) { /* 0.4.27.c27 */
    buf = bf; /* 0.4.27.c27 */

    /* 0.4.27.c27 Skip leading spaces */
    while(*buf == ' ') {
      buf++;
    }

    /* Purge any block allocated during last iteration of this loop */
    if (he) {
      fre_hen(&he, 495);
    }

    he = (host_entry *) ycaloc(sizeof(host_entry), 1,
			       330); /* There are two 330's on purpose! */

    good = 0;
    s_files = 0;
    s_kbytes = 0;
    if (buf[0] == '#') { /* 0.4.27.c27 */
      /* Comment -- ignore */
    } else if (sscanf(buf, "%i.%i.%i.%i:%i %u %u", /* 0.4.27.c11 */
               &s_ip[0], &s_ip[1], &s_ip[2], &s_ip[3],
	       &s_port, &s_files, &s_kbytes) >= 5) {
      good = 1;
    } else if (sscanf(buf, "%i.%i.%i.%i %i %u %u", /* 0.4.27.c11 */
               &s_ip[0], &s_ip[1], &s_ip[2], &s_ip[3],
               &s_port, &s_files, &s_kbytes) >= 5) {
      good = 1;
    } else if (sscanf(buf, "servrate %i", &sr) >= 1) { /* 0.4.27.c27 */
      if (nserv < 10) {
	urate_history[nserv] = sr;
      }
      nserv++;
    } else {
      errf++;
      if (errf < 4) {
        printf("In %s, bad format:\n  %s\n", fname, buf);
      }
    }

    if (good) {
      /* printf("got host %03d.%03d.%03d.%03d:%05d %5d %9d\n",
             s_ip[0], s_ip[1], s_ip[2], s_ip[3],
	     s_port, s_files, s_kbytes); */
      he->ip[0] = s_ip[0];
      he->ip[1] = s_ip[1];
      he->ip[2] = s_ip[2];
      he->ip[3] = s_ip[3];
      p16 = s_port;
      p16 = GUINT16_TO_LE(p16);
      memcpy(he->port, &p16, 2);
      he->files = s_files;
      he->kbytes = s_kbytes;

      if (host_ok_for_gnet(he->ip)) {
	he = host_add(he);
        n1++;
        if (n1 >= 1000) {
          if (n2 == 0) {
            printf("loading hosts");
          }
          printf(".");
          fflush(stdout);
          n2++;
          n1 = 0;
	}
      }
    }
  }
  if (n2) {
    printf("\n");
  }
  if (he) {
    fre_hen(&he, 496);
  }

  update_urate(); /* 0.4.27.c27 */

  if (errf >= 4) {
    printf("There are %d additional bad line(s) in %s.\n", errf - 3, fname);
  }

  fclose(fp);

  printf("Got %i addresses from hosts file.\n", hosts_num);

  gd_s(2,"host_restore returning\n");
  return 0;
}

int hs_n1, hs_n2;
int hwl_ignore_lru; /* 0.4.27.c10 */

int host_write_line(void *a, void *b)
{
  host_entry *he;
  FILE *fp;
  uint16 p16;
  char buf[100];
  int l1;
  uint32 crc; /* 0.4.27.c10 */
  int i; /* 0.4.27.c10 */
  int oktowrite; /* 0.4.27.c10 */
  uint32 avgsize;

  he = a;
  fp = b;

  /* 0.4.27.c10 */
  oktowrite = 1;
  if (hwl_ignore_lru) {
    crc = crc32_block(he, 6);
    i = crc % HLRUSIZE;
    if (memcmp(&(host_lru[i]), he, 6) == 0) {
      /* it's a match */
      oktowrite = 0;
    }
  }

  if (oktowrite) { /* 0.4.27.c10 */
    memcpy(&p16, he->port, 2);
    p16 = GUINT16_FROM_LE(p16);
    sprintf(buf, "%i.%i.%i.%i %i", he->ip[0], he->ip[1], he->ip[2], he->ip[3],
	    p16);
    l1 = strlen(buf);
    fprintf(fp, "%s", buf);
    while(l1 < ((3+1+3+1+3+1+3 +1+ 5 +1) & 0xf8)) {
      fprintf(fp, " ");
      l1++;
    }
    avgsize = he->files;
    if (avgsize) {
      avgsize = (he->kbytes / avgsize) << 10;
    }
    fprintf(fp, "\t%u\t%u\t%u\n", he->files, he->kbytes, avgsize); /* 0.4.27.c11 */

    hs_n1++;
    if (hs_n1 >= 1000) {
      if (hs_n2 == 0) {
	printf("saving hosts");
      }
      printf(".");
      fflush(stdout);
      hs_n2++;
      hs_n1 = 0;
    }
  }

  return 0;
}

int host_save(char *fname)
{
  FILE *fp;
  int i;
  host_entry *he;

  /* if nothing to save, we want to keep old list */
  if (!hosts_num) {
    return 0;
  }

  unlink(fname);

  fp = fopen(fname, "w");
  if (fp == 0) {
    return -1;
  }

  gd_s(3, "host_save opened file\n");

  /* 0.4.27.c27 Since I'm changing the hosts file format I might as well
     add comments to the host file explaining the format */
  fprintf(fp, "# Hosts file created by gnut %s\n# Also contains other important info needed when restarting.\n#\n# The first 10 lines are a record of the speed (in bytes per second)\n# of the 10 most recent successful uploads.\n", VERSION);

  /* 0.4.27.c27 Save the upload rates history */
  for(i=0; i<10; i++) {
    fprintf(fp, "servrate %i\n", urate_history[i]);
  }

  fprintf(fp, "#\n# The rest of this file contains hosts and their stats.\n# The 'num' column gives the number of shared files on that host, and the\n# 'size/1024' column gives the total size of those files divided by 1024.\n# If the sizes seem too big, remember that most shared files are about 5 to\n# 10 million bytes. The 'avg. size' column shows this average (in bytes).\n#\n# IP and port          num   size/1024 avg. size\n");

  hwl_ignore_lru = 0; /* 0.4.27.c10 */
  hs_n1 = hs_n2 = 0;
  for(i=0; i<HLRUSIZE; i++) {
    he = host_lru + i;
    if (he->ip[0]) {
      host_write_line(he, fp);
    }

#if 0
    if (he->ip[0]) {
      uint16 p16;
      memcpy(&p16, he->port, 2);
      p16 = GUINT16_FROM_LE(p16);
      fprintf(fp, "%i.%i.%i.%i %i\n", he->ip[0], he->ip[1],
	      he->ip[2], he->ip[3], p16);
    }
#endif
  }

  pthread_mutex_lock(&host_mutex);
  gd_s(3, "host_save locked mutex\n");

  hwl_ignore_lru = 1; /* 0.4.27.c10 */
  gnut_hash_foreach(host_list, host_write_line, fp);

  if (hs_n2) {
    printf("\n");
  }

  pthread_mutex_unlock(&host_mutex);

  fclose(fp);
  return 0;
}

int host_totals(uint *num, float64 *files, float64 *size)
{
  gd_s(2, "host.totals num="); gd_i(2, hosts_num); gd_s(2, " files=");
  gd_f64(2, hosts_files); gd_s(2, " size=");
  gd_f64(2, (hosts_kbytes * 1024.0)); gd_s(2, "\n");

  if (num) {
    *num = hosts_num;
  }
  if (files) {
    *files = hosts_files;
  }
  if (size) {
    /* We have to multiply by 1024 here because the Gnutella spec specifies
       that the host send their size in units of 1024 bytes. Everywhere else
       in gnut we follow the SI, NIST and CGMP conventions of using powers
       of 1000 to make calculations easier for users. Read FAQ question
       5.17 in the manual. */
    *size = hosts_kbytes * 1024.0;
  }
  return 0;
}

int host_find(uchar ip[4], uchar port[2])
{
  host_entry he,*ret;

  dqi(0x0139);

  gd_s(3,"host_find entering "); gd_i(3, ip[0]); gd_s(3, ".");
  gd_i(3, ip[1]); gd_s(3, "."); gd_i(3, ip[2]); gd_s(3, "."); gd_i(3, ip[3]);
  gd_s(3, "\n");

  pthread_mutex_lock(&host_mutex);

  memcpy(he.ip, ip, 4);
  memcpy(he.port, port, 2);

  dqi(0x0156);
  ret = gnut_hash_find(host_list,&he);

  pthread_mutex_unlock(&host_mutex);

  if (ret) {
    gd_s(2,"found ip "); gd_i(2, ret->ip[0]); gd_s(2, ".");
    gd_i(2, ret->ip[1]); gd_s(2, "."); gd_i(2, ret->ip[2]); gd_s(2, ".");
    gd_i(2, ret->ip[3]); gd_s(2, "\n");
    return 0;
  }
  dqi(0x013a);
  return -1;
}

int host_clear()
{
  pthread_mutex_lock(&host_mutex);

  gd_s(3, "host_clear entering\n");
  gnut_hash_fre(host_list);

  host_list = gnut_hash_new(host_hash, host_compare);

  gd_s(3, "host_clear new host_list="); gd_p(3, host_list); gd_s(3, "\n");

  hosts_num = 0;
  hosts_files = 0;
  hosts_kbytes = 0;

  pthread_mutex_unlock(&host_mutex);
  return 0;
}

int host_lock()
{
  pthread_mutex_lock(&host_mutex);
  return 0;
}

int host_unlock()
{
  pthread_mutex_unlock(&host_mutex);
  return 0;
}

Gnut_Hash * host_retrieve()
{
  return host_list;
}
