/* Josh Pieper, (c) 2000 */

/* This file is distributed under the GPL, see file COPYING for details */

#ifndef GNUT_CONNECTION_H
#define GNUT_CONNECTION_H

#include "lib.h"
#include "queue.h"
#include "host.h"
#include "protocol.h"
#include "threads.h"

/* Define the following to enable the packet_stats option */
/* #define MOREPACKETSTATS */

typedef struct _gcs gcs;

struct _gcs {
  int gc_sock;  /* fd of the sock belonging to this connection */

  pthread_mutex_t out_q_mutex;
  Gnut_Queue *out_queue;    /* packets queued up to be sent */
  
  ip_t    ip;
  uint16  port;
  
  int     sent_packets;
  int     received_packets;
  int     dropped_packets;
  int     bad_packets;
  
  float64 snt_bytes;
  float64 rcv_bytes;
  
  /* the next couple are for rate control */
  float64 rate_snt_bytes;
  float64 rate_rcv_bytes;
  time_t  rate_start_time;

  int     routing_errors;

  int     seen_pong;
  
  /* type indicates whether this is an outgoing, or incoming
   * connection */
  int     ctype;
  
  /* the state indicates what state the connection is in, see
   * defines below for possible values */
  int     cstate;
  
  /* this index is used for user-interface purposes, it should
   * only be set by the ui... */
  int32   uindex;
  
  pthread_t tid;
#ifdef MOREPACKETSTATS
  int     packet_count[256];
  int     packet_byte_count[256];
  int     max_packet_size[256];
  int     ttl_count[256];

  int     out_packet_count[256];
  int     out_packet_byte_count[256];
  int     out_max_packet_size[256];
  int     out_ttl_count[256];
#endif
};

extern pthread_mutex_t  query_packet_mutex;
extern pthread_mutex_t  qrepl_mutex;

#define TYPE_UNSPEC 0
#define TYPE_OUTGOING  1
#define TYPE_INCOMING  2

#define TYPE_UPLOAD 1
#define TYPE_DOWNLOAD 2
#define TYPE_SEND_PUSH 3
#define TYPE_RECV_PUSH 4

#define STATE_UNCON -1

#define STATE_CONNECTING 1
#define STATE_NEGOTIATING 2
#define STATE_CONNECTED 3
#define STATE_CLEANABLE 4
#define STATE_DEAD 5
#define STATE_ERROR 6
#define STATE_QUEUED 7
#define STATE_RETRY_WAIT 8

#define BACKLOG_MAX 100
#define BAD_PACKET_MAX 100

#ifdef __cplusplus
extern "C" {
#endif

  extern uint32 con_num_queries; /* 0.4.27.c28 */
  extern uint32 con_num_responses; /* 0.4.27.c28 */
  extern uint32 con_num_messages;
  extern uint32 con_num_sent;

  void fre_gc(gcs **x, int bugnum);
  int send_to_one(gcs *gc, gnutella_packet *gpa);
  int send_to_all(gnutella_packet *gpa);
  int send_to_all_except(gcs *gc, gnutella_packet *gpa,
			 int no_drop);

  int gnut_connection_net_totals(uint32 *messages, uint32 *sent,
				 float64 *rcv_bytes, float64 *snt_bytes,
				 float64 *blacklisted);

  gcs * gnut_connection_new();
  int gnut_connection_delete(gcs *gc);
  int connection_loop(gcs *gc);
  int gnut_connection_find(uchar ip[4], uint16 port);
  int gnut_connection_find_ptr(gcs *);
  /* need connection enumeration functions */

  int gnut_connection_enumerate( int (*a)(gcs *));
  int gnut_connection_num();
  int gnut_connection_kill_oldest();
  long droprate(long drop, long recv);
  int gnut_connection_kill_worst(int interact);

  int eqhd_flageval(uint8 validbyte, uint8 flagbyte, uint8 mask);
  void parse_eqhd(uint8 *einfo, int32 infolen,
		  uint8 *ip, uint32 speed,
		  int16 *type, int16 *rating); /* 0.4.27.c21 */
  char name_tok_cb(GnutSearch *s, void *data);
  int connection_handle_packet(gcs *gc, gnutella_packet *gpa);

#ifdef __cplusplus
}
#endif

#endif
