/* blacklist.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef WIN32
# include <netdb.h>
#endif

#ifdef HAVE_REGEX_H
# include <regex.h>
#endif

#include "lib.h"
#include "list.h"
#include "host.h"
#include "cli.h"
#include "blacklist.h"

static Gnut_List *blacklist;
static int blacklist_totals;

#define GNUT_BLACKLIST_DEBUG			5

int gnut_blacklist_totals(void)
{
  return blacklist_totals;
}

void gnut_blacklist_init(void)
{
  char *d;

  blacklist = 0;

#ifdef WIN32
  d = expand_path("blacklist.ini");
#else
  d = expand_path("~/.gnut_blacklist");
#endif
  if (d) {
    run_config_file(d, 1, 1, 1);
    fre_str(&d, 406);
  }
}

static int blacklist_writer(blacklist_t *bl, void *data)
{
  FILE *f;
  int type_all;
  
  f = (FILE *)data;
  
  fprintf(f, "blacklist ");
  if (bl->which & BLACKLIST_HAVE_SEARCH) {
    fprintf(f, "search %s ", bl->search_string);
  }
  if (bl->host_string) {
    fprintf(f, "host %s ", bl->host_string);
    if (bl->subnetbits != 0) {
      fprintf(f, "subnet %i ", bl->subnetbits);
    }
  }
  if (bl->which & BLACKLIST_HAVE_PORT) {
    fprintf(f, "port %i ", bl->port);
  }
  type_all = BLACKLIST_TYPE_INCOMING | 
    BLACKLIST_TYPE_OUTGOING |
    BLACKLIST_TYPE_SEARCH |
    BLACKLIST_TYPE_PING |
    BLACKLIST_TYPE_HTTP;
  
  if ((bl->type & type_all) == bl->type) {
    fprintf(f, "all ");
  } else {
    if (bl->type & BLACKLIST_TYPE_INCOMING) {
      fprintf(f, "incoming ");
    }
    if (bl->type & BLACKLIST_TYPE_OUTGOING) {
      fprintf(f, "outgoing ");
    }
    if (bl->type & BLACKLIST_TYPE_SEARCH) {
      fprintf(f, "search ");
    }
    if (bl->type & BLACKLIST_TYPE_PING) {
      fprintf(f, "ping ");
    }
    if (bl->type & BLACKLIST_TYPE_HTTP) {
      fprintf(f, "http ");
    }
  }
  fprintf(f, "\n");
  return 0;
}

void gnut_blacklist_save(void)
{
  FILE *f;
  char *d;
  
#ifdef WIN32
  f = fopen("blacklist.ini", "w");
#else
  d = expand_path("~/.gnut_blacklist");
  f = fopen(d, "w");
#endif
  if (f) {
    gnut_blacklist_foreach(blacklist_writer, (void *)f);
    fclose(f);
  }
}

static void make_subnet_filter(struct in_addr *network, struct in_addr *address, int subnetbits)
{
  int i;
  
  network->s_addr = address->s_addr;

  for (i = 1; i <= subnetbits; i++) {
    network->s_addr &= ~(1 << (32 - i));
  }
}

void gnut_blacklist_add(char *search_regex, char *hostname,
			char *subnetbits, char *port, int filter_incoming,
			int filter_outgoing, int filter_ping,
			int filter_searches, int filter_http)
{
  blacklist_t *bl;
  struct hostent *he;
  struct in_addr *addr;

  gd_s(GNUT_BLACKLIST_DEBUG, "gnut_blacklist_add IN\n");

  bl = (blacklist_t *)ymaloc(sizeof(blacklist_t), 363);

  bl->type = 0;
  bl->which = 0;
  bl->hits = 0;

  if (filter_incoming)
    bl->type |= BLACKLIST_TYPE_INCOMING;
  if (filter_outgoing)
    bl->type |= BLACKLIST_TYPE_OUTGOING;
  if (filter_ping)
    bl->type |= BLACKLIST_TYPE_PING;
#ifndef WIN32
  if (filter_searches)
    bl->type |= BLACKLIST_TYPE_SEARCH;
#endif
  if (filter_http)
    bl->type |= BLACKLIST_TYPE_HTTP;

  if (search_regex && strlen(search_regex)) {
    bl->search_string = ystdup(search_regex, 481);
    bl->which |= BLACKLIST_HAVE_SEARCH;
#ifndef WIN32
    regcomp(&bl->search, search_regex, REG_EXTENDED | REG_NEWLINE);
#endif
  }

  if (hostname && strlen(hostname)) {
    bl->host_string = ystdup(hostname, 482);
    if (!subnetbits) {
      bl->subnetbits = 0;
    } else {
      bl->subnetbits = atoi(subnetbits);
    }
    he = gethostbyname(hostname);
    if (he) {
      bl->which |= BLACKLIST_HAVE_ADDRESS;
      addr = (struct in_addr *)he->h_addr_list[0];
      if (addr) {
	make_subnet_filter(&bl->addr, addr, bl->subnetbits);
      }
    }
  } else {
    bl->host_string = 0;
  }
  
  if (port && strlen(port)) {
    bl->port = atoi(port);
    bl->which |= BLACKLIST_HAVE_PORT;
  }
  
  blacklist = gnut_list_prepend(blacklist, bl);
  gd_s(GNUT_BLACKLIST_DEBUG, "gnut_blacklist_add OUT\n");
}

void gnut_blacklist_remove(int index)
{
  gd_s(GNUT_BLACKLIST_DEBUG, "gnut_blacklist_remove\n");
}

/* Returns true if the operation is allowed, and false if it is not. */
int gnut_blacklist_allow(int operation, char *search, struct in_addr *addr,
 int port, int opt1)
{
  Gnut_List *list;
  blacklist_t *bl;
#ifndef WIN32
  regmatch_t pmatch;
  int nomatch;
#endif
  ip_t lc_addr;

  if (addr) { /* copy *addr to lc.a */
    int i;
    uchar *p;

    p = (uchar *) addr;
    for(i=0; i<sizeof(struct in_addr); i++) {
      lc_addr.b[i] = p[i];
    }
  }

  list = blacklist;

  gd_s(GNUT_BLACKLIST_DEBUG, "gnut_blacklist_allow IN ");
  gd_s(GNUT_BLACKLIST_DEBUG, search ? search : "N/A");
  gd_s(GNUT_BLACKLIST_DEBUG, " ");
  gd_x(GNUT_BLACKLIST_DEBUG, addr ? lc_addr.a.s_addr : 0);
  gd_s(GNUT_BLACKLIST_DEBUG, " ");
  gd_i(GNUT_BLACKLIST_DEBUG, port);
  gd_s(GNUT_BLACKLIST_DEBUG, "\n");

  while (1) {
    if (!list) {
      break;
    }

    bl = list->data;
    if (!bl) {
      break;
    }

    /* Find out if this operation applies to the requester filter */
    if ((operation & bl->type)) {
      /* Check to see if the search criteria matches the blacklist. */
#ifndef WIN32
      if (search && (bl->which & BLACKLIST_HAVE_SEARCH)) {
	nomatch = regexec(&bl->search, search, 1, &pmatch, 0);
	if (!nomatch) {
	  blacklist_totals++;
	  bl->hits++;
	  gd_s(GNUT_BLACKLIST_DEBUG, "gnut_blacklist_allow OUT1\n");
	  return 0;
	}
      }
#endif
      
      /* Check to see if the address is blacklisted. */
      if (addr && (bl->which & BLACKLIST_HAVE_ADDRESS)) {
	if ((bl->addr.s_addr & lc_addr.a.s_addr) == bl->addr.s_addr) {
	  blacklist_totals++;
	  bl->hits++;
	  gd_s(GNUT_BLACKLIST_DEBUG, "gnut_blacklist_allow OUT2\n");
	  return 0;
	}
      }
      
      if (port && (bl->which & BLACKLIST_HAVE_PORT)) {
	if (bl->port == port) {
	  blacklist_totals++;
	  bl->hits++;
	  gd_s(GNUT_BLACKLIST_DEBUG, "gnut_blacklist_allow OUT3\n");
	  return 0;
	}
      }
    }
    list = gnut_list_next(list);
  }
  gd_s(GNUT_BLACKLIST_DEBUG, "gnut_blacklist_allow OUT4\n");
  return 1;
}


int gnut_blacklist_foreach(int (*handle)(blacklist_t *bl, void *data), void *data)
{
  Gnut_List *list;
  blacklist_t *bl;
  
  list = blacklist;

  while (1) {
    if (!list)
      break;
    bl = list->data;
    if (!bl)
      break;

    handle(bl, data);
    list = gnut_list_next(list);
  }
  return 1;
}
