/* gMUDix -- MUDix for X windows
 * Copyright (c) 2002 Marko Boomstra (m.boomstra@chello.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "mudix.h"
#include "file.h"


/* first pointer to a list of ALL users */
USER *user_list = NULL;


USER *new_user(void)
{
    USER *user;
    int   i;

    /* get us some memory */
    if (!(user = malloc(sizeof(USER)))) 
    {
	fprintf(stdout, "No memory for user?!\n");
	exit_mudix();
    }

    if (!(user->net.rxbuf = malloc(sizeof(gchar) * RXBUF_LENGTH))) 
    {
	fprintf(stdout, "No memory for receive buffer?!\n");
	exit_mudix();
    }

    user->next                  = NULL;
    user->net.rxp               = user->net.rxbuf;
    user->net.stream            = NULL;
    user->net.mccp_out          = NULL;
    user->net.mccp_outp         = NULL;
    user->net.mccp_outsize      = 0;
    user->net.site              = NULL;
    user->net.host_name         = NULL;
    user->net.rx_tag            = 0;
    user->net.exc_tag           = 0;
    user->net.port              = 0;
    user->net.sock              = 0;
    user->net.thread_id         = 0;
    user->net.status            = NET_CLOSED;
    user->net.recon_timer_load  = 0;
    user->net.recon_timer       = 0;
    user->net.recon_id          = 0;
    user->net.charset           = strdup(DEFAULT_CHARSET);

    user->gui_user.g_window     = NULL;
    user->gui_user.g_view       = NULL;
    user->gui_user.g_buffer     = NULL;
    user->gui_user.g_status     = NULL;
    user->gui_user.g_input      = NULL;
    user->gui_user.g_mark       = NULL;
    user->gui_user.win_width    = DEFAULT_WIDTH;
    user->gui_user.win_height   = DEFAULT_HEIGHT;
    user->gui_user.win_pos_x    = 0;
    user->gui_user.win_pos_y    = 0;
    user->gui_user.lines_max    = DEFAULT_LINE_MAX;
    user->gui_user.ansi.attrib  = 0;
    user->gui_user.ansi.fg      = ANSI_COLOR_NONE;
    user->gui_user.ansi.bg      = ANSI_COLOR_NONE;
    user->gui_pref.g_preference = NULL;
    user->gui_pref.g_selecttree = NULL;
    user->gui_pref.pref_width   = DEFAULT_PREF_WIDTH;
    user->gui_pref.pref_height  = DEFAULT_PREF_HEIGHT;

    /* set all colors to their default */
    gui_color_set_def_all(user);

    /* clear all tags */
    for (i=0; i<NR_ALL_COLORS; i++)
    {
        user->gui_user.g_fg_color_tags[i] = NULL;
        user->gui_user.g_bg_color_tags[i] = NULL;
    }
    user->gui_user.g_underline_tag = NULL;

    /* setup default fonts */
    user->gui_user.fonts[FONT_USER_WINDOW]     = strdup(DEFAULT_FONT);
    user->gui_user.fonts[FONT_USER_SCROLLBACK] = strdup(DEFAULT_FONT);
    user->gui_user.fonts[FONT_USER_INPUT]      = strdup(DEFAULT_FONT_INPUT);
    user->gui_user.fonts[FONT_USER_CAPTURE]    = strdup(DEFAULT_FONT);

    /* clear all editors */
    for (i=0; i<NUM_SELECT; i++)
    {
        user->gui_pref.g_edit[i] = NULL;
    }

    user->alias_list   = NULL;
    user->macro_list   = NULL;
    user->path_list    = NULL;
    user->tabs_list    = NULL;
    user->timer_list   = NULL;
    user->trigger_list = NULL;
    user->vars_list    = NULL;
    user->logfile      = NULL;
    user->filename     = NULL;
    user->session      = NULL;
    user->flags        = 0;

    /* create a trigger buffer */
    if (!(user->trigger_buf = malloc(sizeof(gchar) * MAX_STRING))) 
    {
	fprintf(stdout, "No memory for trigger buffer?!\n");
	exit_mudix();
    }

    /* also set current trigger pointer */
    user->trigger = user->trigger_buf;

    /* set customizable characters to default */
    user->custom_chars[CUST_CMD_STACK]   = DEFAULT_CMD_STACK;
    user->custom_chars[CUST_VAR_SIGN]    = DEFAULT_VAR_SIGN;
    user->custom_chars[CUST_BLOCK_OPEN]  = DEFAULT_BLOCK_OPEN;
    user->custom_chars[CUST_BLOCK_CLOSE] = DEFAULT_BLOCK_CLOSE;
    user->custom_chars[CUST_CMD_CHAR]    = DEFAULT_CMD_CHAR;

    /* intialize the history buffers */
    init_history(user);

    /* create a 1 sec timer for the timers */
    user->timer_id = init_timer(user);

    /* give this user an ID */
    user->id = 0;
    while (get_user_id(user->id))
    {
        user->id++;
    }

    /* lock the mutex as we are going to change the user_list */
    g_mutex_lock(user_network_mutex);

    /* put user in the user_list */
    user->next = user_list;
    user_list  = user;

    /* unlock it again */
    g_mutex_unlock(user_network_mutex);

    return user;
}


void del_user(USER *user)
{
    int i;

    /* clean up the alias list */
    cleanup_alias_list(user);
    
    /* clean up the macro list */
    cleanup_macro_list(user);
    
    /* clean up the path list */
    cleanup_path_list(user);
    
    /* clean up the timer list */
    cleanup_timer_list(user);

    /* clean up the trigger list */
    cleanup_trigger_list(user);
    
    /* clean up the variable list */
    cleanup_vars_list(user);

    /* clean up the tab completion list */
    cleanup_tabs_list(user);

    /* clean up the history */
    for (i=0; i<MAX_HISTORY; i++) 
    {
	if (user->history[i])
        {
            free(user->history[i]);
        }
    }

    /* clean up the fonts */
    for (i=0; i<NR_FONTS; i++)
    {
        free(user->gui_user.fonts[i]);
    }

    if (user->trigger_buf)
    {
        free(user->trigger_buf);
    }

    /* clean up the filename */
    if (user->filename)
    {
        free(user->filename);
    }

    /* clean up the session name */
    if (user->session)
    {
        free(user->session);
    }

    /* remove the previously created timer */
    gtk_timeout_remove(user->timer_id);

    /* lock the mutex as we are going to remove the net data of a user */
    g_mutex_lock(user_network_mutex);

    /* clean up the site string */
    if (user->net.site)
    {
        free(user->net.site);
    }

    /* clean up the host name */
    if (user->net.host_name)
    {
        free(user->net.host_name);
    }

    /* clean up the MCCP buffer */
    if (user->net.mccp_out)
    {
        free(user->net.mccp_out);
    }

    /* clean up the MCCP stream */
    if (user->net.stream)
    {
        free(user->net.stream);
    }

    /* clean up the receive buffer */
    if (user->net.rxbuf)
    {
        free(user->net.rxbuf);
    }

    /* remove the user from the user_list */
    if (user == user_list)
    {
        user_list = user->next;
    }
    else
    {
        USER *iUser;

        for (iUser = user_list; iUser; iUser = iUser->next)
        {
            if (iUser->next == user)
            {
                iUser->next = user->next;
                break;
            }
        }
    }

    /* unlock it again */
    g_mutex_unlock(user_network_mutex);

    free(user);
}


void init_user_mutex(void)
{
    /* check if it was already initialized */
    if (user_network_mutex)
    {
        return;
    }

    /* create a mutex */
    user_network_mutex = g_mutex_new();
}


/* get the user data that belongs to an existing window */
USER *get_user_window(GtkWindow *window)
{
    USER *user;

    for (user = user_list; user; user = user->next)
    {
        if (GTK_WINDOW(user->gui_user.g_window) == window)
        {
            /* yes - this user owns the window :) */
            break;
        }
    }

    return user;
}


bool load_user(USER *user, char *file)
{
    FILE *fp;
    bool  found;
    int   i;

    if (!(fp = fopen(file, "r")))
    {
	return FALSE;
    }

    if (!(user->filename = strdup(file)))
    {
	fprintf(stdout, "No memory for filename?!\n");
	exit_mudix();
    }

    found = FALSE;
    while (1) 
    {
        char *word;

        word = fread_word(fp);
	if (*word == '\0')
        {
	    break;
        }

        if (!strcmp(word, "#USER")) 
        {
            /* USER block found :) */
        }
        else if (!strcmp(word, "#END")) 
	{
            /* entire user block read */
	    found = TRUE;
	    break;
        }
	else if (!strcmp(word, "name")) 
        {
	    user->session = fread_string(fp);
	}   
	else if (!strcmp(word, "site")) 
        {
	    user->net.site = fread_string(fp);
	}   
        /* OBSOLETE: al, but here to support old MUDix (only F1 to F12) */
	else if (!strcmp(word, "al"))
        {
	    MACRO *macro;
	    guint  key = atoi(fread_word(fp));

	    if (!(macro = new_macro_append(user)))
            {
		break;
            }

            if (key < 2)
            {
                /* F1 and F2 */
	        macro->key = GDK_F1+key;
            }
            else
            {
                /* F3 to F12 */
                macro->key = GDK_F3+(key-2);
            }
	    macro->state = 0;
	    macro->text  = fread_string(fp);
        }
	else if (!strcmp(word, "macro"))
        {
	    MACRO *macro;
	    guint  key   = atoi(fread_word(fp));
	    guint  state = atoi(fread_word(fp));

	    if (!(macro = new_macro_append(user)))
            {
		break;
            }

	    macro->key   = key;
	    macro->state = state;
	    macro->text  = fread_string(fp);
	}   
        /* support for old MUDix alias: nmal */
	else if (!strcmp(word, "nmal") ||
	         !strcmp(word, "alias"))
        {
	    ALIAS *alias;

	    if (!(alias = new_alias_append(user)))
            {
		break;
            }

	    alias->name   = fread_string(fp);
	    alias->string = fread_string(fp);
	}
	else if (!strcmp(word, "tab")) 
        {
	    TAB *tab;

	    if (!(tab = new_tab_append(user)))
            {
		break;
            }

	    tab->name = fread_string(fp);
	}
	else if (!strcmp(word, "path")) 
        {
	    PATH *path;

	    if (!(path = new_path_append(user)))
            {
		break;
            }

	    path->name = fread_string(fp);
	    path->path = fread_string(fp);
	}
	else if (!strcmp(word, "var")) 
        {
	    VAR *var;

	    if (!(var = new_var_append(user)))
            {
		break;
            }

	    var->name  = fread_string(fp);
	    var->value = fread_string(fp);
	}
        /* support for old MUDix trigger */
	else if (!strcmp(word, "trig")) 
        {
            TRIGGER *trigger;
	    int      level = atoi(fread_word(fp));

            if (!(trigger = new_trigger_append(user, level)))
            {
		break;
            }

	    trigger->input    = fread_string(fp);
	    trigger->response = fread_string(fp);

	    trigger->enabled  = TRUE;
	}   
	else if (!strcmp(word, "trigger")) 
        {
            TRIGGER *trigger;
	    int      level = atoi(fread_word(fp));

            if (!(trigger = new_trigger_append(user, level)))
            {
		break;
            }

            trigger->enabled  = atoi(fread_word(fp));
	    trigger->input    = fread_string(fp);
	    trigger->response = fread_string(fp);

            /* if it's a login trigger, then always enable it */
            if (trigger->level == TRG_LOGIN ||
                trigger->level == TRG_PASSWORD)
            {
                trigger->enabled = TRUE;
            }
	}   
	else if (!strcmp(word, "timer")) 
        {
            TIMER *timer;
            int    relval = atoi(fread_word(fp));
            int    relcnt = atoi(fread_word(fp));
	    int    time = atoi(fread_word(fp));

            if (!(timer = new_timer_append(user)))
            {
		break;
            }

            timer->response = fread_string(fp);
            timer->relval   = relval;
            timer->relcnt   = relcnt;

            /* time holds the time left when timer was saved */
            if (time)
            {
                /* reload the timer if the timer was running */
                timer->timer = relval;
            }
	}   
	else if (!strcmp(word, "port")) 
        {
            char *str = fread_word(fp);

            /* OLD port was read with fread_string, remove ~ */
            for (i=0; i<strlen(str); i++)
            {
                if (str[i] == '~')
                {
                    str[i] = '\0';
                    break;
                }
            }

	    user->net.port = atoi(str);
	}   
	else if (!strcmp(word, "winheight")) 
        {
            user->gui_user.win_height = atoi(fread_word(fp));
	}   
	else if (!strcmp(word, "winwidth"))
        {
            user->gui_user.win_width = atoi(fread_word(fp));
	}  
	else if (!strcmp(word, "winposx")) 
        {
            user->gui_user.win_pos_x = atoi(fread_word(fp));
	}   
	else if (!strcmp(word, "winposy"))
        {
            user->gui_user.win_pos_y = atoi(fread_word(fp));
	}  
	else if (!strcmp(word, "linemax")) 
        {
            user->gui_user.lines_max = atoi(fread_word(fp));
	}   
	else if (!strcmp(word, "prfheight")) 
        {
            user->gui_pref.pref_height = atoi(fread_word(fp));
	}   
	else if (!strcmp(word, "prfwidth"))
        {
            user->gui_pref.pref_width = atoi(fread_word(fp));
	}  
	else if (!strcmp(word, "fgcolor"))
        {
            user->gui_user.default_color[DEF_FG].red   = atoi(fread_word(fp));
            user->gui_user.default_color[DEF_FG].green = atoi(fread_word(fp));
            user->gui_user.default_color[DEF_FG].blue  = atoi(fread_word(fp));
	}  
	else if (!strcmp(word, "bgcolor"))
        {
            user->gui_user.default_color[DEF_BG].red   = atoi(fread_word(fp));
            user->gui_user.default_color[DEF_BG].green = atoi(fread_word(fp));
            user->gui_user.default_color[DEF_BG].blue  = atoi(fread_word(fp));
	}  
	else if (!strcmp(word, "colors"))
        {
            int nrcols = atoi(fread_word(fp));

            for (i=0; i<nrcols && i<NR_ALL_COLORS; i++)
            {
                user->gui_user.colors[i].red   = atoi(fread_word(fp));
                user->gui_user.colors[i].green = atoi(fread_word(fp));
                user->gui_user.colors[i].blue  = atoi(fread_word(fp));
	    }
        }
	else if (!strcmp(word, "fonts"))
        {
            int nrfonts = atoi(fread_word(fp));

            /* only set our "known" fonts */
            for (i=0; i<nrfonts && i<NR_FONTS; i++)
            {
                free(user->gui_user.fonts[i]);
                user->gui_user.fonts[i] = fread_string(fp);
	    }
        }
	else if (!strcmp(word, "flags"))
        {
            user->flags = atoi(fread_word(fp));
        }
	else if (!strcmp(word, "rectime"))
        {
            user->net.recon_timer_load = atoi(fread_word(fp));
        }
	else if (!strcmp(word, "charset"))
        {
            free(user->net.charset);
            user->net.charset = fread_string(fp);
        }
	else if (!strcmp(word, "custchars"))
        {
            int nrcust = atoi(fread_word(fp));

            /* only set our "known" custom chars */
            for (i=0; i<nrcust && i<NR_CUSTOM_CHARS; i++)
            {
                user->custom_chars[i] = atoi(fread_word(fp));
	    }
        }
        else
        {
            fprintf(stdout, "Error reading user file, unknown: %s\n", word);
            break;
        }
    }
  
    fclose(fp);

    return found;
}


bool save_user(USER *user)
{
    FILE    *fp;
    TRIGGER *trig;
    TIMER   *timer;
    ALIAS   *alias;
    MACRO   *macro;
    VAR     *var;
    PATH    *path;
    TAB     *tabs;
    int      i;

    if (!user->filename || !(fp = fopen(user->filename, "w")))
    {
        return FALSE;
    }

    /* grab the position of the window */
    if (user->gui_user.g_window)
    {
        gtk_window_get_position(GTK_WINDOW(user->gui_user.g_window),
                                &user->gui_user.win_pos_x,
                                &user->gui_user.win_pos_y);
    }

    /* save general user info */
    fprintf(fp, "#USER\n");
    fprintf(fp, "name      %s~\n",      smash_tilde(user->session));
    fprintf(fp, "site      %s~\n",      smash_tilde(user->net.site));
    fprintf(fp, "port      %d\n",       user->net.port);
    fprintf(fp, "charset   %s~\n",      smash_tilde(user->net.charset));
    fprintf(fp, "rectime   %d\n",       user->net.recon_timer_load);
    fprintf(fp, "flags     %d\n",       user->flags & FLG_SAVE_FLAGS_MASK);

    fprintf(fp, "winheight %d\n",       user->gui_user.win_height);
    fprintf(fp, "winwidth  %d\n",       user->gui_user.win_width);
    fprintf(fp, "winposx   %d\n",       user->gui_user.win_pos_x);
    fprintf(fp, "winposy   %d\n",       user->gui_user.win_pos_y);
    fprintf(fp, "linemax   %d\n",       user->gui_user.lines_max);
       
    fprintf(fp, "prfheight %d\n",       user->gui_pref.pref_height);
    fprintf(fp, "prfwidth  %d\n",       user->gui_pref.pref_width);

    /* save the customizable characters */
    fprintf(fp, "custchars %d",         NR_CUSTOM_CHARS);
    for (i=0; i<NR_CUSTOM_CHARS; i++)
    {
        fprintf(fp, " %d",              user->custom_chars[i]);
    }
    fprintf(fp, "\n");

    /* default text FG and BG color */
    fprintf(fp, "fgcolor   %d %d %d\n", user->gui_user.default_color[DEF_FG].red,
                                        user->gui_user.default_color[DEF_FG].green,
                                        user->gui_user.default_color[DEF_FG].blue);
    fprintf(fp, "bgcolor   %d %d %d\n", user->gui_user.default_color[DEF_BG].red,
                                        user->gui_user.default_color[DEF_BG].green,
                                        user->gui_user.default_color[DEF_BG].blue);

    /* save all colors */
    fprintf(fp, "colors    %d",         NR_ALL_COLORS);
    for (i=0; i<NR_ALL_COLORS; i++)
    {
        fprintf(fp, " %d %d %d",        user->gui_user.colors[i].red,
                                        user->gui_user.colors[i].green,
                                        user->gui_user.colors[i].blue);
    }
    fprintf(fp, "\n");

    /* save the fonts */
    fprintf(fp, "fonts     %d\n",       NR_FONTS);
    for (i=0; i<NR_FONTS; i++)
    {
        fprintf(fp, "%s~\n",            user->gui_user.fonts[i]);
    }

    /* save all the triggers */
    for (trig = user->trigger_list; trig; trig = trig->next) 
    {
        fprintf(fp, "trigger %d %d\n%s~\n%s~\n", trig->level, trig->enabled,
                                                 smash_tilde(trig->input), 
                                                 smash_tilde(trig->response));
    }

    /* save all the timers */
    for (timer = user->timer_list; timer; timer = timer->next) 
    {
        fprintf(fp, "timer   %d %d %d\n%s~\n", timer->relval, 
                                               timer->relcnt,
                                               timer->timer,
                                               smash_tilde(timer->response));
    }

    /* save aliases */
    for (alias = user->alias_list; alias; alias = alias->next) 
    {
        fprintf(fp, "alias\n%s~\n%s~\n", smash_tilde(alias->name), 
                                         smash_tilde(alias->string));
    }
	    
    /* save the macro keys */
    for (macro = user->macro_list; macro; macro = macro->next) 
    {
        fprintf(fp, "macro %d %d\n%s~\n", macro->key, macro->state,
                                          smash_tilde(macro->text));
    }

    /* save paths */
    for (path = user->path_list; path; path = path->next) 
    {
        fprintf(fp, "path\n%s~\n%s~\n", smash_tilde(path->name), 
                                        smash_tilde(path->path));
    }
	    
    /* save tab completions */
    for (tabs = user->tabs_list; tabs; tabs = tabs->next) 
    {
      	fprintf(fp, "tab\n%s~\n", smash_tilde(tabs->name));
    }
	    
    /* save variables */
    for (var = user->vars_list; var; var = var->next) 
    {
        fprintf(fp, "var\n%s~\n%s~\n", smash_tilde(var->name), 
                                       smash_tilde(var->value));
    }
	    
    fprintf(fp, "#END");
    fclose(fp);

    return TRUE;
}


bool user_read_info(char *path, char **name, char **site, guint *port)
{
    FILE *fp;
    bool  found;
    int   i;

    if (!(fp = fopen(path, "r")))
    {
	return FALSE;
    }

    found = FALSE;
    while (1) 
    {
        char *word;

        word = fread_word(fp);
	if (*word == '\0')
        {
	    break;
        }

        if (!strcmp(word, "#USER")) 
        {
            /* USER block found :) */
            found = TRUE;
        }
	else if (!strcmp(word, "name")) 
        {
	    *name = fread_string(fp);
	}   
	else if (!strcmp(word, "site")) 
        {
	    *site = fread_string(fp);
	}   
	else if (!strcmp(word, "port")) 
        {
            char *str = fread_word(fp);

            /* OLD port was read with fread_string, remove ~ */
            for (i=0; i<strlen(str); i++)
            {
                if (str[i] == '~')
                {
                    str[i] = '\0';
                    break;
                }
            }

	    *port = atoi(str);
            break;  /* we know that port is the LAST to read, so break */
	}   
        else
        {
            if (feof(fp))
            {
                break;
            }
            else
            {
                fread_to_eol(fp);
                continue;
            }
        }
    }
  
    fclose(fp);

    return found;
}


bool check_valid_user(USER *user)
{
    USER *list;

    for (list = user_list; list; list = list->next)
    {
        if (user == list)
        {
            /* yes, this user is valid */
            break;
        }
    }

    return (list)? TRUE: FALSE;
}


int user_count(void)
{
    USER *user;
    int   count = 0;

    for (user = user_list; user; user = user->next)
    {
        count++;
    }

    return count;
}


USER *get_user_with_color(GdkColor *color)
{
    USER *user;

    /* smart lookup: check if color address in range of user */
    for (user = user_list; user; user = user->next)
    {
        if ((gint)color < ((gint)user + sizeof(USER)))
        {
            /* address of color is within this user's range */
            return user;
        }
    }
    
    return NULL;
}


USER *get_user_id(guint id)
{
    USER *user;

    for (user = user_list; user; user = user->next)
    {
        if (user->id == id)
        {
            return user;
        }
    }
    
    return NULL;
}


USER *get_user_file(char *file)
{
    USER *user;

    for (user = user_list; user; user = user->next)
    {
        if (!strcmp(user->filename, file))
        {
            return user;
        }
    }
    
    return NULL;
}


/* look up the user that owns this session pointer */
USER *get_user_session(gchar *session)
{
    USER *user;

    for (user = user_list; user; user = user->next)
    {
        if (user->session == session)
        {
            return user;
        }
    }
    
    return NULL;
}
