/* gmpc-coveramazon (GMPC plugin)
 * Copyright (C) 2006-2009 Qball Cow <qball@sarine.nl>
 * Project homepage: http://gmpcwiki.sarine.nl/
 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libmpd/debug_printf.h>
#include <gmpc/plugin.h>
#include <gmpc/gmpc_easy_download.h>
#include <glib/gstdio.h>
#include <config.h>
#define AMAZONKEY "14TC04B24356BPHXW1R2"

static char * host = "http://ecs.amazonaws.%s/onca/xml?Service=AWSECommerceService&Operation=ItemSearch&SearchIndex=Music&ResponseGroup=Images,EditorialReview&AWSAccessKeyId=%s&%s=%s&%s=%s";

static int fetch_cover_art(mpd_Song *song,int type, char **url);

#define ENDPOINTS 6
static char *endpoints[ENDPOINTS][2] = 
{ 
	{"com", 	"United States"},
	{"co.uk",	"United Kingdom"},
	{"jp",		"Japan"},
	{"fr",		"France"},
	{"ca", 		"Canada"},
	{"de", 		"Germany"}
};

static GtkWidget *wp_pref_vbox = NULL;

typedef struct amazon_song_info {
	char *image_big;
	char *image_medium;
	char *image_small;
	char *album_info;

}amazon_song_info;

/**
 * Get Set enabled
 */
static int amazon_get_enabled()
{
	return cfg_get_single_value_as_int_with_default(config, "cover-amazon", "enable", TRUE);
}
static void amazon_set_enabled(int enabled)
{
	cfg_set_single_value_as_int(config, "cover-amazon", "enable", enabled);
}

/**
 * Get priority 
 */
static int fetch_cover_priority(){
	return cfg_get_single_value_as_int_with_default(config, "cover-amazon", "priority", 80);
}
static void fetch_cover_priority_set(int priority)
{
    cfg_set_single_value_as_int(config, "cover-amazon", "priority", priority);
}

static int fetch_metadata(mpd_Song *song,MetaDataType type, char **path)
{
	gchar *url = NULL;

	if(song->artist == NULL || song->album == NULL)
	{
		return META_DATA_UNAVAILABLE;
	}
	if(type != META_ALBUM_ART && type != META_ALBUM_TXT) {
		return META_DATA_UNAVAILABLE;
	}
	/* Always fetch it. */
	fetch_cover_art(song,type, &url);
	if(url){
		*path = url;
		return META_DATA_AVAILABLE;
	}

	g_free(url);
	return META_DATA_UNAVAILABLE; 
}	

static amazon_song_info * amazon_song_info_new()
{
	amazon_song_info *asi = g_malloc(sizeof(amazon_song_info));
	asi->image_big = NULL;
	asi->image_medium = NULL;
	asi->image_small = NULL;
	asi->album_info = NULL;
	return asi;
}
static void amazon_song_info_free(amazon_song_info *asi)
{
	if(asi->image_big != NULL) g_free(asi->image_big);
	if(asi->image_medium != NULL) g_free(asi->image_medium);
	if(asi->image_small != NULL) g_free(asi->image_small);
	if(asi->album_info != NULL) g_free(asi->album_info);
	g_free(asi);
	return;
}

static xmlNodePtr get_first_node_by_name(xmlNodePtr xml, gchar *name) {
	if(xml) {
		xmlNodePtr c = xml->xmlChildrenNode;
		for(;c;c=c->next) {
			if(xmlStrEqual(c->name, (xmlChar *) name))
				return c;
		}
	}
	return NULL;
}

static amazon_song_info *__cover_art_xml_get_image(char *data,int size)
{
	xmlDocPtr doc = xmlParseMemory(data,size);
	if(doc)
	{
		xmlNodePtr root = xmlDocGetRootElement(doc);
		xmlNodePtr cur = get_first_node_by_name(root, "Items");
		amazon_song_info *asi = NULL; 
		if(cur) {
			cur = get_first_node_by_name(cur, "Item");
			if(cur) {
				xmlNodePtr child = NULL;
				asi = amazon_song_info_new();
				if((child = get_first_node_by_name(cur, "LargeImage"))) {
					xmlChar *temp = xmlNodeGetContent(get_first_node_by_name(child, "URL")); 
					/* copy it, so we can free it, and don't need xmlFree */
					asi->image_big = g_strdup((char *)temp);
					xmlFree(temp);
				}
				if((child = get_first_node_by_name(cur, "MediumImage"))){
					xmlChar *temp = xmlNodeGetContent(get_first_node_by_name(child, "URL"));
					asi->image_medium = g_strdup((char *)temp);
					xmlFree(temp);
				}
				if((child = get_first_node_by_name(cur, "SmallImage"))){
					xmlChar *temp = xmlNodeGetContent(get_first_node_by_name(child, "URL"));
					asi->image_small = g_strdup((char *)temp);
					xmlFree(temp);
				}	

				if((child = get_first_node_by_name(cur, "EditorialReviews"))) { 
					child = get_first_node_by_name(child, "EditorialReview");
					if(child) {
						xmlChar *temp = xmlNodeGetContent(get_first_node_by_name(child, "Content")); /* ugy, lazy */
						asi->album_info = g_strdup((char *)temp);
						xmlFree(temp);
					}
				}
			}
		}
		xmlFreeDoc(doc);
		return asi;
	}
	
	return NULL;
}


static int __fetch_metadata_amazon(mpd_Song *song, char *sfield, char *nartist, char *stype,char *nalbum,int type, char **url)
{

	gmpc_easy_download_struct data= {NULL, 0,-1, NULL, NULL};
	int found = 0;
	char furl[1024];
	int endpoint = cfg_get_single_value_as_int_with_default(config, "cover-amazon", "location", 0);
	char *endp = endpoints[endpoint][0];

	debug_printf(DEBUG_INFO, "search-type: %s\n", stype);
	snprintf(furl,1024, host,endp, AMAZONKEY,sfield, nartist, stype, nalbum);
	if(gmpc_easy_download(furl, &data))
	{
		amazon_song_info *asi = __cover_art_xml_get_image(data.data, data.size);
		gmpc_easy_download_clean(&data);
		if(asi) 
		{
			if(type&META_ALBUM_ART)
			{
				debug_printf(DEBUG_INFO, "Trying to fetch album art");
				gmpc_easy_download(asi->image_big, &data);
				if(data.size <= 900){
					gmpc_easy_download_clean(&data);
					gmpc_easy_download(asi->image_medium, &data);
					if(data.size <= 900){
						gmpc_easy_download_clean(&data);
						gmpc_easy_download(asi->image_small, &data);
						if(data.size <= 900)
						{
							gmpc_easy_download_clean(&data);
						}
					}
				}
				if(data.size){
					FILE *fp = NULL;
					gchar *imgpath = NULL;
                    imgpath = gmpc_get_metadata_filename(META_ALBUM_ART, song,"jpg"); 
					fp = g_fopen(imgpath, "wb");
					if(fp)
					{
						size_t size = fwrite(data.data, sizeof(char), data.size,fp);
                        if(size != data.size)
                        {
                            debug_printf(DEBUG_ERROR, "Failed to write complete file: %s\n", imgpath);
                        }
						*url = g_strdup(imgpath);
						found = 1;
						fclose(fp);
					}
					g_free(imgpath);
				}
				gmpc_easy_download_clean(&data);
				

			}
			else if(type&META_ALBUM_TXT)
			{
				debug_printf(DEBUG_INFO, "Trying to fetch album txt");
				if(asi->album_info)
				{
					FILE *fp;
					gchar *imgpath;

                    imgpath = gmpc_get_metadata_filename(META_ALBUM_TXT, song,"jpg"); 
                    fp = g_fopen(imgpath, "w");
					if(fp)
					{
						int j=0,depth=0;;
						*url = g_strdup(imgpath);
						/**
						 * Quick 'n Dirty html stripper
						 */
						for(j=0; j < strlen(asi->album_info);j++)
						{
							if((asi->album_info)[j]	 == '<') depth++;
							else if((asi->album_info)[j] == '>' && depth) depth--;
							else if(depth == 0)
							{
								fputc((asi->album_info)[j],fp);
							}
						}
						fclose(fp);
						found = 1;
					}

					g_free(imgpath);
				}
			}
			amazon_song_info_free(asi);
		}
	}

	return found;
}
static int fetch_cover_art(mpd_Song *song,int type, char **url) 
{
	int retval=0;
	gchar *artist = gmpc_easy_download_uri_escape(song->artist);
	gchar *album =  gmpc_easy_download_uri_escape(song->album);
    
    retval = __fetch_metadata_amazon(song,
                "Title",album,
                "Artist", artist, 
                type,url);
    if(!retval && song->performer)
    {
        gchar *performer = gmpc_easy_download_uri_escape(song->performer);
        retval = __fetch_metadata_amazon(
                song,
                "Title",album,
                "Performer", performer, 
                type,url);
        g_free(performer);
    }

	g_free(artist);
	g_free(album);
	return retval;
}

/**
 * Preferences
 */

static void amazon_cover_art_pref_destroy(GtkWidget *container)
{
	gtk_container_remove(GTK_CONTAINER(container), wp_pref_vbox);
}
static void amazon_cover_art_pref_selection_changed(GtkWidget *box)
{
	cfg_set_single_value_as_int(config, "cover-amazon", "location", gtk_combo_box_get_active(GTK_COMBO_BOX(box)));
}

static void amazon_cover_art_pref_construct(GtkWidget *container)
{
	GtkWidget *label = NULL;
	GtkWidget *selection = NULL;
	GtkWidget *hbox = NULL;
	int i;
	wp_pref_vbox = gtk_vbox_new(FALSE,6);


	/* Location */
	hbox = gtk_hbox_new(FALSE, 6);
	label = gtk_label_new("Location:");
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE,0);
	selection = gtk_combo_box_new_text();
	for(i=0;i<ENDPOINTS;i++)
		gtk_combo_box_append_text(GTK_COMBO_BOX(selection), endpoints[i][1]);
	gtk_combo_box_set_active(GTK_COMBO_BOX(selection),cfg_get_single_value_as_int_with_default(config, "cover-amazon", "location", 0));
	g_signal_connect(G_OBJECT(selection), "changed",G_CALLBACK(amazon_cover_art_pref_selection_changed), NULL);

	gtk_box_pack_start(GTK_BOX(hbox), selection, TRUE,TRUE,0);
	gtk_box_pack_start(GTK_BOX(wp_pref_vbox), hbox, FALSE, FALSE, 0);

	gtk_container_add(GTK_CONTAINER(container), wp_pref_vbox);
	gtk_widget_show_all(container);
}

gmpcPrefPlugin cam_pref = {
	.construct      = amazon_cover_art_pref_construct,
	.destroy        = amazon_cover_art_pref_destroy

};
gmpcMetaDataPlugin cam_cover = {
	.get_priority   = fetch_cover_priority,
    .set_priority   = fetch_cover_priority_set,
	.get_image      = fetch_metadata
};

int plugin_api_version = PLUGIN_API_VERSION;
gmpcPlugin plugin = {
	.name           = "Amazon Cover Fetcher",
	.version        = {PLUGIN_MAJOR_VERSION,PLUGIN_MINOR_VERSION,PLUGIN_MICRO_VERSION},
	.plugin_type    = GMPC_PLUGIN_META_DATA,
	.pref           = &cam_pref,
	.metadata       = &cam_cover,
	.get_enabled    = amazon_get_enabled,
	.set_enabled    = amazon_set_enabled
};
