#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libmpd/debug_printf.h>
#include <gmpc/plugin.h>
#include <gmpc/gmpc_easy_download.h>
#include <config.h>
#define AMAZONKEY "14TC04B24356BPHXW1R2"

static void init();
static char * host = "http://ecs.amazonaws.%s/onca/xml?Service=AWSECommerceService&Operation=ItemSearch&SearchIndex=Music&ResponseGroup=Images,EditorialReview&SubscriptionId=%s&Artist=%s&%s=%s";
static char* search_types[] = {"Title", "Keywords"};
static void amazon_cover_art_pref_construct(GtkWidget *container);
static void amazon_cover_art_pref_destroy(GtkWidget *container);
static int fetch_cover_art(mpd_Song *song,int type, char **url);
static int fetch_cover_priority();

static int shrink_string(gchar *string, int start, int end);

static int fetch_metadata(mpd_Song *song,MetaDataType type, char **path);

static void amazon_set_enabled(int enabled);
static int amazon_get_enabled();

#define ENDPOINTS 6
static char *endpoints[ENDPOINTS][2] = 
{ 
	{"com", 	"United States"},
	{"co.uk",	"United Kingdom"},
	{"jp",		"Japan"},
	{"fr",		"France"},
	{"ca", 		"Canada"},
	{"de", 		"Germany"}
};

static GtkWidget *wp_pref_vbox = NULL;
gmpcPrefPlugin cam_pref = {
	amazon_cover_art_pref_construct,
	amazon_cover_art_pref_destroy

};
typedef struct amazon_song_info {
	char *image_big;
	char *image_medium;
	char *image_small;
	char *album_info;

}amazon_song_info;

gmpcMetaDataPlugin cam_cover = {
	fetch_cover_priority,
	fetch_metadata
};

int plugin_api_version = PLUGIN_API_VERSION;
gmpcPlugin plugin = {
	"Amazon Cover Fetcher",
	{PLUGIN_MAJOR_VERSION,PLUGIN_MINOR_VERSION,PLUGIN_MICRO_VERSION},
	GMPC_PLUGIN_META_DATA,
	0,
	NULL, /* path  */
	init, /* init  */
        NULL,   /* Destroy */
	NULL, /* browser */
	NULL, /* status changed */
	NULL, /* connection changed */
	&cam_pref, /* preferences */
	&cam_cover, /** Metadata */
	amazon_get_enabled,
	amazon_set_enabled
};

/* Convert string to the wonderful % notation for url*/
static char * __cover_art_process_string(const gchar *string)
{
#define ACCEPTABLE(a) (((a) >= 'a' && (a) <= 'z') || ((a) >= 'A' && (a) <= 'Z') || ((a) >= '0' && (a) <= '9'))

	const gchar hex[16] = "0123456789ABCDEF";
	const gchar *p;
	gchar *q;
	gchar *result;
	int c;
	gint unacceptable = 0;
	const gchar *tmp_p;
	gchar *new_string;
	int depth = 0;
	int len;
	int i = 0;

	len = strlen(string);

	new_string = g_malloc(len + 1);

	/*	Get count of chars that will need to be converted to %
		and remove ([{}]) and everything between */
	for (p = string; *p != '\0'; p++) {
		c = (guchar) *p;

		if(c == '(' || c == '[' || c == '{') {
			depth++;
		} else if(c == ')' || c == ']' || c == '}') {
			depth--;
			if(depth < 0)
				depth = 0;
		} else if(depth == 0) {
			if (!ACCEPTABLE (c)) 
				unacceptable++;

			new_string[i] = c;

			i++;
		}
	}

	new_string[i] = '\0';

	len = strlen(new_string);

	/* remove double spaces from the string because removing ([{}])
		tends to create those */
	for(p = new_string + 1; *p != '\0'; p++) {
		c = (guchar) *p;
		if(c == ' ') {
			tmp_p = p - 1;
			if( *tmp_p == ' ') {
				len = shrink_string(new_string,  p - new_string, len);
				p--;
			}
		}
	}
	
	/* make sure first char isn't a space */
	if(new_string[0] == ' ')
		len = shrink_string(new_string, 0, len);

	/* make sure there isn't a trailing space*/
	if(new_string[len - 1] == ' ')
	len--;

	new_string[len] = '\0';

	result = g_malloc (len + unacceptable * 2 + 1);

	/*time to create the escaped string*/
	for (q = result, p = new_string; *p != '\0'; p++)
	{
		c = (guchar) *p;

		if (!ACCEPTABLE (c)) {
			*q++ = '%'; /* means hex coming */
			*q++ = hex[c >> 4];
			*q++ = hex[c & 15];
		}
		else
			*q++ = *p;
	}

	*q = '\0';

	g_free(new_string);

	return result;
}

/**
 * Get Set enabled
 */
static int amazon_get_enabled()
{
	return cfg_get_single_value_as_int_with_default(config, "cover-amazon", "enable", TRUE);
}
static void amazon_set_enabled(int enabled)
{
	cfg_set_single_value_as_int(config, "cover-amazon", "enable", enabled);
}

static void init()
{
	char *file = gmpc_get_covers_path(NULL);
	if(!g_file_test(file,G_FILE_TEST_EXISTS))
	{
		g_mkdir(file, 0755);
	}
	g_free(file);
}

static int fetch_cover_priority(){
	return cfg_get_single_value_as_int_with_default(config, "cover-amazon", "priority", 80);

}

static int fetch_metadata(mpd_Song *song,MetaDataType type, char **path)
{
	int j=2;
	gchar *url = NULL;
	gchar *filename;

	if(song->artist == NULL || song->album == NULL)
	{
		return META_DATA_UNAVAILABLE;
	}
	if(type != META_ALBUM_ART && type != META_ALBUM_TXT) {
		return META_DATA_UNAVAILABLE;
	}
	/* Always fetch it. */
	fetch_cover_art(song,type, &url);
	if(url){
		*path = url;
		return META_DATA_AVAILABLE;
	}

	g_free(url);
	return META_DATA_UNAVAILABLE; 
}	

static amazon_song_info * amazon_song_info_new()
{
	amazon_song_info *asi = g_malloc(sizeof(amazon_song_info));
	asi->image_big = NULL;
	asi->image_medium = NULL;
	asi->image_small = NULL;
	asi->album_info = NULL;
	return asi;
}
static void amazon_song_info_free(amazon_song_info *asi)
{
	if(asi->image_big != NULL) g_free(asi->image_big);
	if(asi->image_medium != NULL) g_free(asi->image_medium);
	if(asi->image_small != NULL) g_free(asi->image_small);
	if(asi->album_info != NULL) g_free(asi->album_info);
	g_free(asi);
	return;
}

static xmlNodePtr get_first_node_by_name(xmlNodePtr xml, gchar *name) {
	if(xml) {
		xmlNodePtr c = xml->xmlChildrenNode;
		for(;c;c=c->next) {
			if(xmlStrEqual(c->name, (xmlChar *) name))
				return c;
		}
	}
	return NULL;
}

static amazon_song_info *__cover_art_xml_get_image(char *data,int size)
{
	xmlDocPtr doc = xmlParseMemory(data,size);
	if(doc)
	{
		xmlNodePtr root = xmlDocGetRootElement(doc);
		xmlNodePtr cur = get_first_node_by_name(root, "Items");
		amazon_song_info *asi = NULL; 
		if(cur) {
			cur = get_first_node_by_name(cur, "Item");
			if(cur) {
				xmlNodePtr child = NULL;
				asi = amazon_song_info_new();
				if(child = get_first_node_by_name(cur, "LargeImage")) {
					xmlChar *temp = xmlNodeGetContent(get_first_node_by_name(child, "URL")); 
					/* copy it, so we can free it, and don't need xmlFree */
					asi->image_big = g_strdup((char *)temp);
					xmlFree(temp);
				}
				if(child = get_first_node_by_name(cur, "MediumImage")){
					xmlChar *temp = xmlNodeGetContent(get_first_node_by_name(child, "URL"));
					asi->image_medium = g_strdup((char *)temp);
					xmlFree(temp);
				}
				if(child = get_first_node_by_name(cur, "SmallImage")){
					xmlChar *temp = xmlNodeGetContent(get_first_node_by_name(child, "URL"));
					asi->image_small = g_strdup((char *)temp);
					xmlFree(temp);
				}	

				if (child = get_first_node_by_name(cur, "EditorialReviews")) { 
					child = get_first_node_by_name(child, "EditorialReview");
					if(child) {
						xmlChar *temp = xmlNodeGetContent(get_first_node_by_name(child, "Content")); /* ugy, lazy */
						asi->album_info = g_strdup((char *)temp);
						xmlFree(temp);
					}
				}
			}
		}
		xmlFreeDoc(doc);
		return asi;
	}
	xmlCleanupParser();
	return NULL;
}


static int __fetch_metadata_amazon(char *stype, char *nartist, char *nalbum,int type, char **url)
{

	gmpc_easy_download_struct data= {NULL, 0,-1, NULL, NULL};
	int found = 0;
	char furl[1024];
	int endpoint = cfg_get_single_value_as_int_with_default(config, "cover-amazon", "location", 0);
	char *endp = endpoints[endpoint][0];

	debug_printf(DEBUG_INFO, "search-type: %s\n", stype);
	snprintf(furl,1024, host,endp, AMAZONKEY, nartist, stype, nalbum);
	if(gmpc_easy_download(furl, &data))
	{
		amazon_song_info *asi = __cover_art_xml_get_image(data.data, data.size);
		gmpc_easy_download_clean(&data);
		if(asi) 
		{
			if(type&META_ALBUM_ART)
			{
				debug_printf(DEBUG_INFO, "Trying to fetch album art");
				gmpc_easy_download(asi->image_big, &data);
				if(data.size <= 900){
					gmpc_easy_download_clean(&data);
					gmpc_easy_download(asi->image_medium, &data);
					if(data.size <= 900){
						gmpc_easy_download_clean(&data);
						gmpc_easy_download(asi->image_small, &data);
						if(data.size <= 900)
						{
							gmpc_easy_download_clean(&data);
						}
					}
				}
				if(data.size){
					int i =0;
					FILE *fp = NULL;
					gchar *imgpath = NULL;
					gchar *filename = g_strdup_printf("%s-%s.jpg",nartist,nalbum);
					imgpath =  gmpc_get_covers_path(filename);
					g_free(filename);
					fp = fopen(imgpath, "wb");
					if(fp)
					{
						fwrite(data.data, sizeof(char), data.size,fp);
						*url = g_strdup(imgpath);
						found = 1;
						fclose(fp);
					}
					g_free(imgpath);
				}
				gmpc_easy_download_clean(&data);
				

			}
			else if(type&META_ALBUM_TXT)
			{
				debug_printf(DEBUG_INFO, "Trying to fetch album txt");
				if(asi->album_info)
				{
					FILE *fp;
					gchar *filename, *imgpath;
					filename = g_strdup_printf("%s-%s.albuminfo",nartist,nalbum);
					imgpath =  gmpc_get_covers_path(filename);
					g_free(filename);
					fp = fopen(imgpath, "w");
					if(fp)
					{
						int j=0,depth=0;;
						*url = g_strdup(imgpath);
						/**
						 * Quick 'n Dirty html stripper
						 */
						for(j=0; j < strlen(asi->album_info);j++)
						{
							if((asi->album_info)[j]	 == '<') depth++;
							else if((asi->album_info)[j] == '>' && depth) depth--;
							else if(depth == 0)
							{
								fputc((asi->album_info)[j],fp);
							}
						}
						fclose(fp);
						found = 1;
					}

					g_free(imgpath);
				}
			}
			amazon_song_info_free(asi);
		}
	}

	return found;
}
static int fetch_cover_art(mpd_Song *song,int type, char **url) 
{
	int retval=0,i=0;
	gchar *artist = __cover_art_process_string(song->artist);
	gchar *album =  __cover_art_process_string(song->album);

	while(!retval&&i<2)
		retval = __fetch_metadata_amazon(search_types[i++], artist, album,type,url);

	g_free(artist);
	g_free(album);
	return retval;
}

/**
 * Preferences
 */

static void amazon_cover_art_enable_toggle(GtkWidget *wid)
{
	int kk = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(wid));
	cfg_set_single_value_as_int(config, "cover-amazon", "enable", kk);
}
static void amazon_cover_art_pref_destroy(GtkWidget *container)
{
	gtk_container_remove(GTK_CONTAINER(container), wp_pref_vbox);
}
static void amazon_cover_art_pref_selection_changed(GtkWidget *box)
{
	cfg_set_single_value_as_int(config, "cover-amazon", "location", gtk_combo_box_get_active(GTK_COMBO_BOX(box)));
}

static void amazon_cover_art_pref_construct(GtkWidget *container)
{
	GtkWidget *enable_cg = gtk_check_button_new_with_mnemonic("_Enable amazon as cover art source");
	GtkWidget *label = NULL;
	GtkWidget *selection = NULL;
	GtkWidget *entry = NULL;
	GtkWidget *hbox = NULL;
	int i;
	wp_pref_vbox = gtk_vbox_new(FALSE,6);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_cg), 	
			cfg_get_single_value_as_int_with_default(config, "cover-amazon", "enable", TRUE));

	g_signal_connect(G_OBJECT(enable_cg), "toggled", G_CALLBACK(amazon_cover_art_enable_toggle), NULL);
	gtk_box_pack_start(GTK_BOX(wp_pref_vbox), enable_cg, FALSE, FALSE, 0);

	/* Location */
	hbox = gtk_hbox_new(FALSE, 6);
	label = gtk_label_new("Location:");
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE,0);
	selection = gtk_combo_box_new_text();
	for(i=0;i<ENDPOINTS;i++)
		gtk_combo_box_append_text(GTK_COMBO_BOX(selection), endpoints[i][1]);
	gtk_combo_box_set_active(GTK_COMBO_BOX(selection),cfg_get_single_value_as_int_with_default(config, "cover-amazon", "location", 0));
	g_signal_connect(G_OBJECT(selection), "changed",G_CALLBACK(amazon_cover_art_pref_selection_changed), NULL);

	gtk_box_pack_start(GTK_BOX(hbox), selection, TRUE,TRUE,0);
	gtk_box_pack_start(GTK_BOX(wp_pref_vbox), hbox, FALSE, FALSE, 0);

	gtk_container_add(GTK_CONTAINER(container), wp_pref_vbox);
	gtk_widget_show_all(container);
}

static int shrink_string(gchar *string, int start, int end)
{
	int i;
	
	for( i = start; i < end; i++)
		string[i] = string[i + 1];
		
	end--;
	
	return end;
}
