#include "SDL.h"

Uint32 get_pixel(SDL_Surface *surface, Sint32 Xcoord, Sint32 Ycoord) {  
	Uint8 r, g, b;  
	Uint8 *bits;  
	Uint8 bytespp;  
	Uint32 pixel;  
 
	/* Set the number of bytes per pixel to calculate the pixel offset... */  
	bytespp = surface->format->BytesPerPixel;  
 
        /*calculate the offset (pointer) to the screen->pixels array  
         * screen->pitch must be the x of the screen times bytes per pixel  
         */  
        bits = ((Uint8 *)surface->pixels) + Ycoord*surface->pitch +  
	Xcoord*bytespp;  
 
        /* We may now get the pixel */  
        switch(bytespp)  
          {  
          case 1:  
            (Uint8) pixel = *((Uint8 *) bits);  
            break;  
 
          case 2:  
            (Uint16) pixel = *((Uint16 *) bits);  
            break;  
 
          case 3: /* odd case, create offsets from screen data */  
            r = *((bits) + surface->format->Rshift / 8);  
            g = *((bits) + surface->format->Gshift / 8);  
            b = *((bits) + surface->format->Bshift / 8);  

            pixel = (r << surface->format->Rshift);
            pixel += (g << surface->format->Gshift);
            pixel += (b << surface->format->Bshift);  
            break;  
 
          case 4:  
            (Uint32) pixel = *((Uint32 *) bits);  
            break;  
          }  
	return pixel;  
} /* get_pixel() */

SDL_Surface *prep_image(char *file) { 
	SDL_Surface *image; 
	 
	image=SDL_LoadBMP(file); 
 
	if ( image == NULL ) { 
		fprintf(stderr, "Couldn't load %s: %s\n", file,	SDL_GetError()); 
		return(NULL); 
	} 
	 
	return(image); 
} 

void ClearScreen(SDL_Surface *surf) { 
	int i; 
	Uint8 *buffer;	 
 
	if ( SDL_MUSTLOCK(surf) ) { 
		if ( SDL_LockSurface(surf) < 0 ) 
			return; 
	} 
	buffer=(Uint8 *)surf->pixels; 
	memset(buffer,0,surf->w*surf->h*surf->format->BytesPerPixel); 
	 
	if ( SDL_MUSTLOCK(surf) ) { 
		SDL_UnlockSurface(surf); 
	} 
} 

void ClearScreen16(SDL_Surface *surf, Uint8 r, Uint8 g, Uint8 b) { 
	int i; 
	Uint16 *buffer;	 
	Uint16   pixel; 
 
	if ( SDL_MUSTLOCK(surf) ) { 
		if ( SDL_LockSurface(surf) < 0 ) 
			return; 
	} 

	pixel = ((r>>surf->format->Rloss)<<surf->format->Rshift) | 
		((g>>surf->format->Gloss)<<surf->format->Gshift) | 
		((b>>surf->format->Bloss)<<surf->format->Bshift); 

	buffer=(Uint16 *)surf->pixels; 
	for(i=0; i<surf->h*surf->w; i++)
		buffer[i]=pixel;
	 
	if ( SDL_MUSTLOCK(surf) ) { 
		SDL_UnlockSurface(surf); 
	} 
} 

void ClearScreenGrad(SDL_Surface *surf, Uint8 r, Uint8 g, Uint8 b,
	Uint8 r2, Uint8 g2, Uint8 b2) { 
	int i,j;
	float rt, gt, bt;
	float rd, gd, bd; 
	Uint16 *buffer;	 
	Uint16   pixel; 
 
	if ( SDL_MUSTLOCK(surf) ) { 
		if ( SDL_LockSurface(surf) < 0 ) 
			return; 
	} 

	rt=r;
	gt=g;
	bt=b;

	rd=((float)r2-(float)r)/(float)surf->h*2;
	gd=((float)g2-(float)g)/(float)surf->h*2;
	bd=((float)b2-(float)b)/(float)surf->h*2;

	pixel = (((Uint8)rt>>surf->format->Rloss)<<surf->format->Rshift) | 
		(((Uint8)gt>>surf->format->Gloss)<<surf->format->Gshift) | 
		(((Uint8)bt>>surf->format->Bloss)<<surf->format->Bshift); 

	buffer=(Uint16 *)surf->pixels; 

	for(j=0; j<surf->h/2; j++) {
		for(i=0; i<surf->w; i++)
			buffer[j*surf->w+i]=pixel;

		pixel = ((((Uint8)(rt+=rd))>>surf->format->Rloss)<<surf->format->Rshift) | 
			((((Uint8)(gt+=gd))>>surf->format->Gloss)<<surf->format->Gshift) | 
			((((Uint8)(bt+=bd))>>surf->format->Bloss)<<surf->format->Bshift); 
	}

	for(j=surf->h/2; j<surf->h; j++) {
		for(i=0; i<surf->w; i++)
			buffer[j*surf->w+i]=pixel;

		pixel = ((((Uint8)(rt-=rd))>>surf->format->Rloss)<<surf->format->Rshift) | 
			((((Uint8)(gt-=gd))>>surf->format->Gloss)<<surf->format->Gshift) | 
			((((Uint8)(bt-=bd))>>surf->format->Bloss)<<surf->format->Bshift); 
	}
 
	if ( SDL_MUSTLOCK(surf) ) { 
		SDL_UnlockSurface(surf); 
	} 
} 

void RS_Blit(SDL_Surface *src, SDL_Surface *dst, Sint16 x, Sint16 y) { 
	SDL_Rect dest; 
 
	dest.x = x; 
	dest.y = y; 
	dest.w = src->w; 
	dest.h = src->h; 
	SDL_BlitSurface(src, NULL, dst, &dest); 
}

void RS_Pset16(Sint32 x, Sint32 y, Uint8 r, Uint8 g, Uint8 b, SDL_Surface *surf) { 
	Uint16   pixel; 
	Uint8   *bits; 
 
	if((x>0)&&(x<surf->w)) {
	if((y>0)&&(y<surf->h)) {

	pixel = ((r>>surf->format->Rloss)<<surf->format->Rshift) | 
		((g>>surf->format->Gloss)<<surf->format->Gshift) | 
		((b>>surf->format->Bloss)<<surf->format->Bshift); 
	 
	if ( SDL_MUSTLOCK(surf) ) { 
		if ( SDL_LockSurface(surf) < 0 ) 
			return; 
	} 
 
	bits = ((Uint8 *)surf->pixels)+y*surf->pitch+(x<<1); 
 
	*((Uint16 *)(bits)) = (Uint16)pixel; 
 
	if ( SDL_MUSTLOCK(surf) ) { 
		SDL_UnlockSurface(surf); 
	} 

	}
	}
	return; 
}

void X_Pset16(Sint32 x, Sint32 y, Uint8 r, Uint8 g, Uint8 b, SDL_Surface *surf) { 
	/* Surface must be locked prior to calling */
	Uint16   pixel; 
	Uint8   *bits; 

	if((x>0)&&(x<surf->w)) {
	if((y>0)&&(y<surf->h)) {	

	pixel = ((r>>surf->format->Rloss)<<surf->format->Rshift) | 
		((g>>surf->format->Gloss)<<surf->format->Gshift) | 
		((b>>surf->format->Bloss)<<surf->format->Bshift); 
	 
 
	bits = ((Uint8 *)surf->pixels)+y*surf->pitch+(x<<1); 
 
	*((Uint16 *)(bits)) = (Uint16)pixel; 

	}
	}

	return; 
}
 
/* 
 * Draws a line in octant 0 or 3 ( |DeltaX| >= DeltaY ). 
 */ 
void Octant0(Sint32 X0, Sint32 Y0, Sint32 DeltaX, Sint32 DeltaY, char XDirection, Uint8 r, Uint8 g, Uint8 b, SDL_Surface *surf) 
{ 
   int DeltaYx2; 
   int DeltaYx2MinusDeltaXx2; 
   int ErrorTerm; 
 
   /* Set up initial error term and values used inside drawing loop */ 
   DeltaYx2 = DeltaY * 2; 
   DeltaYx2MinusDeltaXx2 = DeltaYx2 - (int) ( DeltaX * 2 ); 
   ErrorTerm = DeltaYx2 - (int) DeltaX; 
 
   /* Draw the line */ 
   RS_Pset16(X0, Y0, r, g, b, surf);		
   while ( DeltaX-- ) { 
      /* See if it's time to advance the Y coordinate */ 
      if ( ErrorTerm >= 0 ) { 
         /* Advance the Y coordinate & adjust the error term 
            back down */ 
         Y0++; 
         ErrorTerm += DeltaYx2MinusDeltaXx2; 
      } else { 
         /* Add to the error term */ 
         ErrorTerm += DeltaYx2; 
      } 
      X0 += XDirection;          /* advance the X coordinate */ 
      RS_Pset16(X0, Y0, r, g, b, surf);
   } 
} 
 
/* 
 * Draws a line in octant 1 or 2 ( |DeltaX| < DeltaY ). 
 */ 
void Octant1(Sint32 X0, Sint32 Y0, Sint32 DeltaX, Sint32 DeltaY, char XDirection, Uint8 r, Uint8 g, Uint8 b, SDL_Surface *surf) 
{ 
   int DeltaXx2; 
   int DeltaXx2MinusDeltaYx2; 
   int ErrorTerm; 
 
   /* Set up initial error term and values used inside drawing loop */ 
   DeltaXx2 = DeltaX * 2; 
   DeltaXx2MinusDeltaYx2 = DeltaXx2 - (int) ( DeltaY * 2 ); 
   ErrorTerm = DeltaXx2 - (int) DeltaY; 

   RS_Pset16(X0, Y0, r, g, b, surf); 
   while ( DeltaY-- ) { 
      /* See if it's time to advance the X coordinate */ 
      if ( ErrorTerm >= 0 ) { 
         /* Advance the X coordinate & adjust the error term 
            back down */ 
         X0 += XDirection; 
         ErrorTerm += DeltaXx2MinusDeltaYx2; 
      } else { 
         /* Add to the error term */ 
         ErrorTerm += DeltaXx2; 
      } 
      Y0++;                   /* advance the Y coordinate */ 
      RS_Pset16(X0, Y0, r, g, b, surf);
   } 
} 
 
/* 
 * Draws a line to a SDL surface
 */ 
void DrawLine(Sint32 X0, Sint32 Y0, Sint32 X1, Sint32 Y1, Uint8 r, Uint8 g, Uint8 b, SDL_Surface *surf) 
{ 
   Sint32 DeltaX, DeltaY; 
   Sint32 Temp; 
 
   /* Save half the line-drawing cases by swapping Y0 with Y1 
      and X0 with X1 if Y0 is greater than Y1. As a result, DeltaY 
      is always > 0, and only the octant 0-3 cases need to be 
      handled. */ 
   if ( Y0 > Y1 ) { 
      Temp = Y0; 
      Y0 = Y1; 
      Y1 = Temp; 
      Temp = X0; 
      X0 = X1; 
      X1 = Temp; 
   } 
 
   /* Handle as four separate cases, for the four octants in which 
      Y1 is greater than Y0 */ 
   DeltaX = X1 - X0;    /* calculate the length of the line 
                           in each coordinate */ 
   DeltaY = Y1 - Y0; 
   if ( DeltaX > 0 ) { 
      if ( DeltaX > DeltaY ) { 
         Octant0(X0, Y0, DeltaX, DeltaY, 1,r,g,b,surf); 
      } else { 
         Octant1(X0, Y0, DeltaX, DeltaY, 1,r,g,b,surf); 
      } 
   } else { 
      DeltaX = -DeltaX;             /* absolute value of DeltaX */ 
      if ( DeltaX > DeltaY ) { 
         Octant0(X0, Y0, DeltaX, DeltaY, -1,r,g,b,surf); 
      } else { 
         Octant1(X0, Y0, DeltaX, DeltaY, -1,r,g,b,surf); 
      } 
   } 
}

