/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|  4/22/2001  Solaris code contributed by Daisuke Yabuki <dxy@acm.org>
| 10/12/2000  NetBSD code contributed by Anthony Mallet
|		<anthony.mallet@useless-ficus.net> 
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

static time_t	base_uptime;

gulong	(*read_uptime)();

/* ====== System dependent interface ====================================== */


/* ----- Pick a system interface ----------------------------------------- */
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/sysctl.h>

static gint
setup_base_uptime(void)
	{
	static int		mib[] = { CTL_KERN, KERN_BOOTTIME };
	struct timeval		boottime;
	size_t			size = sizeof(boottime);
	time_t			now;

	base_uptime = (time_t) 0;
	if (sysctl(mib, 2, &boottime, &size, NULL, 0) != -1 &&
	    boottime.tv_sec != 0)
		{
		(void)time(&now);
		base_uptime = now - boottime.tv_sec;
		base_uptime += 30;
		}
	read_uptime = NULL;
	return (base_uptime == (time_t) 0) ? FALSE : TRUE;
	}
#elif defined(__linux__)

/* As pointed out by Joseph Garcia, calculating an uptime based on system
|  time has a fuzzy meaning for laptops since /proc/uptime does not include
|  time system has been sleeping.  So, read /proc/uptime always.
*/
static gulong
read_linux_uptime(void)
    {
	FILE			*f;
	gulong			l	= 0;

	if ((f = fopen("/proc/uptime", "r")) != NULL)
		{
		fscanf(f, "%lu", &l);
		fclose(f);
		}
    return l;
    }

static gint
setup_base_uptime(void)
    {
	read_uptime = read_linux_uptime;
	base_uptime = (time_t) (*read_uptime)();
    return (base_uptime == (time_t) 0) ? FALSE : TRUE;
    }
#elif defined(__solaris__)
#include <time.h>
#include <kstat.h>

static gint
setup_base_uptime(void) {
    time_t      boot, now;

    extern kstat_ctl_t *kc;
    kstat_t *ksp;
    kstat_named_t *knp;

    boot = 0;

    if (kstat_chain_update(kc) == -1) {
        perror("kstat_chain_update");
        return FALSE;
    }
    ksp = kstat_lookup(kc, "unix", -1, "system_misc");
    if (ksp && kstat_read(kc, ksp, NULL) >= 0) {
        knp = (kstat_named_t *)kstat_data_lookup(ksp, "boot_time");
        if (knp) { 
            boot = knp->value.ui32;
        }
    }
    if (time(&now) < 0)
        return FALSE;
    if (now <= boot) 
        return FALSE;

    base_uptime = now - boot;
    base_uptime += 30;

    read_uptime = NULL;
    return (base_uptime == (time_t) 0) ? FALSE : TRUE; 
}
#else
#include <glibtop/uptime.h>

static gint
setup_base_uptime(void)
    {
	glibtop_uptime	glt_uptime;

	base_uptime = (time_t) 0;
	glibtop_get_uptime (&glt_uptime);
	if (glt_uptime.flags & (1 << GLIBTOP_UPTIME_UPTIME))
		base_uptime = (time_t) glt_uptime.uptime;
	read_uptime = NULL;
    return (base_uptime == (time_t) 0) ? FALSE : TRUE;
    }
#endif


/* ======================================================================== */

Monitor			*mon_uptime;

static Panel	*uptime;

static Decal	*decal_uptime;

static Launcher	launch;

static gint		style_id;
static gboolean	uptime_enabled	= TRUE;

static gint
uptime_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	if (widget == uptime->drawing_area)
		{
		gdk_draw_pixmap(widget->window, GK.draw1_GC, uptime->pixmap,
				ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				ev->area.width, ev->area.height);
		}
	return FALSE;
	}

static void
draw_upminutes(gint minutes)
	{
	TextStyle	*ts;
	gint		w1, w2, w;
	gint		days, hours;
	gchar		buf1[16], buf2[16], *s;

	ts = gkrellm_meter_textstyle(style_id);
	hours = minutes / 60;
	minutes %= 60;
	days = hours / 24;
	hours %= 24;

	s = buf1;
	snprintf(buf1, sizeof(buf1), "%dd %2d:%02d", days, hours, minutes); 
	snprintf(buf2, sizeof(buf2), "%dd%2d:%02d", days, hours, minutes); 
	w = w1 = gdk_string_width(ts->font, buf1);
	if (w1 > decal_uptime->w)
		{
		if ((w2 = gdk_string_width(ts->font, buf2)) > decal_uptime->w)
			{
			ts = gkrellm_meter_alt_textstyle(style_id);
			w = gdk_string_width(ts->font, buf1);
			}
		else
			{
			s = buf2;
			w = w2;
			}
		}
	/* Last chance to fit it in.
	*/
	if (w > decal_uptime->w)
		{
		snprintf(buf1, sizeof(buf1), "%dd%2d:", days, hours);
		s = buf1; 
		}
	decal_uptime->x_off = (decal_uptime->w - w) / 2;
	if (decal_uptime->x_off < 0)
		decal_uptime->x_off = 0;

	decal_uptime->text_style.font = ts->font;
	gkrellm_draw_decal_text(uptime, decal_uptime, s, minutes);
	gkrellm_draw_panel_layers(uptime);
	}


static void
update_uptime(void)
	{
	gint	up_minutes;

	if (!uptime_enabled)
		return;

	/* Once every 10 seconds is default update period.
	*/
	if (GK.ten_second_tick || GK.up_minutes < 0)
		{
		if (read_uptime)
			up_minutes = (gint) ((*read_uptime)() / 60);
		else
			up_minutes = (gint)(time(0) - GK.start_time + base_uptime) / 60;
		if (GK.up_minutes != up_minutes)
				draw_upminutes(up_minutes);
		GK.up_minutes = up_minutes;
		}
	}

static void
create_uptime(GtkWidget *vbox, gint first_create)
	{
	static Style	uptime_style;
	TextStyle		*ts;
	gint			w,
					chart_width = gkrellm_chart_width();

	/* Give a dummy string to configure_panel() to get panel height right.
	|  then, null out the label string.
	*/
	if (first_create)
		uptime = gkrellm_panel_new0();

	uptime_style = *gkrellm_meter_style(style_id);
	uptime_style.label_position = LABEL_CENTER;

	ts = gkrellm_meter_textstyle(style_id);

	w = gdk_string_width(ts->font, "999d 23:99") + 2;
	if (w > chart_width - 2 * uptime_style.margin)
		w = chart_width - 2 * uptime_style.margin;

	decal_uptime = gkrellm_create_decal_text(uptime, "9d 12:99",
				ts, &uptime_style, -1, -1, w);
	decal_uptime->x = (chart_width - decal_uptime->w) / 2;

	gkrellm_panel_configure(uptime, NULL, &uptime_style);
	gkrellm_panel_create(vbox, mon_uptime, uptime);

	if (!uptime_enabled)
		gkrellm_panel_hide(uptime);

	if (first_create)
		gtk_signal_connect(GTK_OBJECT (uptime->drawing_area), "expose_event",
				(GtkSignalFunc) uptime_expose_event, NULL);
	gkrellm_setup_launcher(uptime, &launch, METER_PANEL_TYPE, 0);

	GK.up_minutes = -1;
	update_uptime();
	}


#define	UPTIME_CONFIG_KEYWORD	"uptime"

static void
save_uptime_config(FILE *f)
	{
	fprintf(f, "%s enable %d\n", UPTIME_CONFIG_KEYWORD, uptime_enabled);
	fprintf(f, "%s launch %s\n", UPTIME_CONFIG_KEYWORD, launch.command);
	fprintf(f, "%s tooltip %s\n", UPTIME_CONFIG_KEYWORD, launch.tooltip_comment);
	}

static void
load_uptime_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (strcmp(config, "enable") == 0)
			sscanf(item, "%d", &uptime_enabled);
		else if (strcmp(config, "launch") == 0)
			launch.command = g_strdup(item);
		else if (strcmp(config, "tooltip") == 0)
			launch.tooltip_comment = g_strdup(item);
		}
	}

/* --------------------------------------------------------------------- */
static GtkWidget	*uptime_enabled_button;
static GtkWidget	*launch_entry,
					*tooltip_entry;

static void
apply_uptime_config(void)
	{
	uptime_enabled = GTK_TOGGLE_BUTTON(uptime_enabled_button)->active;
	if (uptime_enabled)
		gkrellm_panel_show(uptime);
	else
		gkrellm_panel_hide(uptime);
	gkrellm_apply_launcher(&launch_entry, &tooltip_entry, uptime,
			&launch, gkrellm_launch_button_cb);
	}


static void
create_uptime_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*tabs, *table, *vbox, *vbox1;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* ---Options tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));

	vbox1 = gkrellm_framed_vbox(vbox, _("Options"),
			4, FALSE, 0, 2);
	gkrellm_check_button(vbox1, &uptime_enabled_button, uptime_enabled,
			FALSE, 10, _("Enable Uptime"));

	vbox1 = gkrellm_framed_vbox_end(vbox, _("Launch Commands"),
			4, FALSE, 0, 2);
	table = gkrellm_launcher_table_new(vbox1, 1);
	gkrellm_config_launcher(table, 0,  &launch_entry, &tooltip_entry,
				_("Uptime"), &launch);
	}


static Monitor	monitor_uptime =
	{
	N_("Uptime"),		/* Name, for config tab.	*/
	MON_UPTIME,			/* Id,  0 if a plugin		*/
	create_uptime,		/* The create function		*/
	update_uptime,		/* The update function		*/
	create_uptime_tab,	/* The config tab create function	*/
	apply_uptime_config, /* Apply the config function		*/

	save_uptime_config,	/* Save user conifg			*/
	load_uptime_config,	/* Load user config			*/
	UPTIME_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_uptime_monitor(void)
	{
	monitor_uptime.name = _(monitor_uptime.name);
	style_id = gkrellm_add_meter_style(&monitor_uptime, UPTIME_STYLE_NAME);
	mon_uptime = &monitor_uptime;
	if (setup_base_uptime())
		return &monitor_uptime;
	return NULL;
	}
