/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellm.h"

GdkImlibBorder	zero_border;


  /* Smooth the krell response with an exponential moving average.
  |  Eponential MA is quicker responding than pure moving average.
  |  a   =  2 / (period + 1)
  |  ema = ema + a * (reading - ema)
  |  Don't need floating point precision here, so do the int math in an
  |  order to minimize roundoff error and scale by 256 for some precision.
  */
static gint
exp_MA(Krell *k)
	{
	gint	ema, p, reading, round_up;

	/* First, help the krell settle to zero and full scale. This gives
	|  on/off fast response while the ema is smoothing in between.
	*/
	if (k->reading == 0 && k->last_reading == 0)
		return 0;
	if (k->reading >= k->full_scale && k->last_reading >= k->full_scale)
		return k->full_scale;
	if (k->last_reading == 0)	/* Fast liftoff as well */
		return k->reading;
	ema = k->ema << 8;
	p   = k->period + 1;		/* Don't scale this! */
	reading = (k->reading) << 8;

	ema = ema + 2 * reading / p - 2 * ema / p;
	round_up = ((ema & 0xff) > 127) ? 1 : 0;

	return (ema >> 8) + round_up;
	}

  /* If the Krell has moved, redraw its layer on its stencil.
  */
void
gkrellm_update_krell(Panel *p, Krell *k, gulong value)
	{
	gint	xnew, x_src, y_src, x_dst, w1, w2, w_overlap, d, frame;

	if (!p || !k || k->full_scale == 0)
		return;
	k->value = value;
	if (value < k->previous)		/* unsigned long overflow? */
		{
		k->previous = value;
		return;
		}
	if (!k->monotonic)
		k->previous = 0;
	k->reading = (gint) (value - k->previous) * k->full_scale_expand;
	if (k->reading > k->full_scale)
		k->reading = k->full_scale;

	k->ema = (k->period > 1) ? exp_MA(k) : k->reading;
	if (k->monotonic)
		k->previous = value;
	k->last_reading = k->reading;

	xnew = k->x0 + k->ema * k->w_scale / k->full_scale;

	if (xnew == k->x0 && k->reading)
		{
		xnew = k->x0 + 1;	/* Show something for all activity */
		k->ema = 1;
		}
	if (xnew == k->x_position)
		return;
	k->x_position = xnew;


	/* If the krell has depth, pick the frame to display as function of
	|  ema.  Depth == 2 means display frame 0 at xnew == 0, and frame 1
	|  everywhere else.  Depth > 2 means same thing for frame 0,
	|  diplay frame depth - 1 at xnew == full_scale, and display other
	|  frames in between at xnew proportional to ema/full_scale.
	*/
	d = k->depth;

	if (k->bar_mode)
		{
		frame = d - 1;
		y_src = k->h_frame * frame;
		x_src = k->x0;
		x_dst = k->x0;
		w_overlap = xnew - k->x0;
		}
	else
		{
		if (k->ema == 0 || xnew == k->x0) /* Krell can settle before ema*/
			frame = 0;
		else if (k->ema >= k->full_scale)
			frame = d - 1;
		else
			{
			if (d == 1)
				frame = 0;
			else if (d == 2)
				frame = 1;
			else
				frame = 1 + ((d - 2) * k->ema / k->full_scale);
			}
		y_src = k->h_frame * frame;

		x_src = k->x_hot - (xnew - k->x0);
		if (x_src < 0)
			x_src = 0;

		x_dst = xnew - k->x_hot;
		if (x_dst < k->x0)
			x_dst = k->x0;

		w1 = k->w - x_src;
		w2 = (k->x0 + k->w_scale + 1) - x_dst;
		w_overlap = (w2 > w1) ? w1 : w2;

		/* Clear the krell stencil bitmap and draw the mask.
		*/
		gdk_draw_rectangle(k->stencil, GK.bit0_GC, TRUE,
					0,0,  k->w, k->h_frame);
		if (k->mask)
			gdk_draw_pixmap(k->stencil, GK.bit1_GC, k->mask,
					x_src, y_src, x_dst, 0, w_overlap, k->h_frame);
		else
			gdk_draw_rectangle(k->stencil, GK.bit1_GC,
					TRUE, x_dst, 0, w_overlap, k->h_frame);
		}

	k->old_draw = k->draw;

	k->draw.x_src = x_src;
	k->draw.y_src = y_src;
	k->draw.x_dst = x_dst;
	k->draw.y_dst = k->y0;
	k->draw.w = w_overlap;
	k->draw.h = k->h_frame;

	k->modified = TRUE;
	}

void
gkrellm_draw_decal_on_chart(Chart *cp, Decal *d, gint x, gint y)
	{
	if (!cp || !d || d->state != DS_VISIBLE)
		return;
	d->x = x;
	d->y = y;
	gdk_gc_set_clip_mask(GK.text_GC, d->stencil);
	gdk_gc_set_clip_origin(GK.text_GC, x, y);
	gdk_draw_pixmap(cp->pixmap, GK.text_GC, d->pixmap,
				0, d->y_src, x, y, d->w, d->h);

	gdk_gc_set_clip_mask(GK.text_GC, NULL);
	gdk_gc_set_clip_origin(GK.text_GC, 0, 0);
	}

  /* Push decal pixmaps through their stencils onto a Panel expose pixmap
  */
static gboolean
push_decal_pixmaps(Panel *p, gboolean top_layer)
	{
	GList		*list;
	Decal		*d;
	gboolean	on_top, do_top_layer = FALSE;
	gboolean	restore_gc = FALSE;

	if (!p)
		return FALSE;
	for (list = p->decal; list; list = list->next)
		{
		d = (Decal *) list->data;
		on_top = (d->flags & DF_TOP_LAYER);
		if (on_top && !top_layer)
			do_top_layer = TRUE;

		if (   d->state != DS_VISIBLE
			|| (on_top && !top_layer) || (!on_top && top_layer)
		   )
			continue;
		gdk_gc_set_clip_mask(GK.text_GC, d->stencil);
		gdk_gc_set_clip_origin(GK.text_GC, d->x, d->y);
		gdk_draw_pixmap(p->pixmap, GK.text_GC, d->pixmap,
					0, d->y_src, d->x, d->y, d->w, d->h);
		restore_gc = TRUE;
		}
	if (restore_gc)
		{
		gdk_gc_set_clip_mask(GK.text_GC, NULL);
		gdk_gc_set_clip_origin(GK.text_GC, 0, 0);
		}

	return do_top_layer;
	}

  /* Push krell pixmaps through their stencils onto a Panel expose pixmap
  */
static void
push_krell_pixmaps(Panel *p)
	{
	GList		*list;
	Krell		*k;
	Draw_rec	*dr;
	gboolean	restore_clip_mask   = FALSE,
				restore_clip_origin = FALSE;

	if (!p)
		return;
	for (list = p->krell; list; list = list->next)
		{
		k = (Krell *) list->data;
		gdk_gc_set_clip_mask(GK.text_GC, k->stencil);
		if (k->y0 != 0 || restore_clip_origin)
			{
			gdk_gc_set_clip_origin(GK.text_GC, 0, k->y0);
			restore_clip_origin = TRUE;
			}
		dr = &k->draw;
		gdk_draw_pixmap(p->pixmap, GK.text_GC, k->pixmap,
					dr->x_src, dr->y_src, dr->x_dst, dr->y_dst, dr->w, dr->h);
		restore_clip_mask = TRUE;
		}
	if (restore_clip_mask)
		gdk_gc_set_clip_mask(GK.text_GC, NULL);
	if (restore_clip_origin)
		gdk_gc_set_clip_origin(GK.text_GC, 0, 0);
	}


/* ---- Deprecated function - name change ---- */
void
gkrellm_draw_layers(Panel *p)
	{
	gkrellm_draw_panel_layers(p);
	}

void
gkrellm_draw_panel_layers(Panel *p)
	{
	GdkDrawable	*drawable;
	GList		*list;
	Krell		*k;
	Decal		*d;			/* LED's or other graphic	*/
	Draw_rec	*dr;
	gboolean	do_top_layer_decals;

	if (!p)
		return;
	/* Restore background of panel where objects have changed.
	|  I should test for size of krell, and if large enough, just do this:
	|		gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
	|					0, 0,   0, 0,   p->w, p->h);
	|
	*/
	/* Restore background under old decal positions */
	for (list = p->decal; list; list = list->next)
		{
		d = (Decal *) list->data;
		if (d->modified)
			{
			if (d->state != DS_INVISIBLE)
				gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
						d->x, d->y,  d->x, d->y,   d->w, d->h);
			p->modified = TRUE;
			}
		}
	/* Restore background under old krell positions */
	for (list = p->krell; list; list = list->next)
		{
		k = (Krell *) list->data;
		if (k->modified)
			{
			dr = &k->old_draw;
			gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
					dr->x_dst, dr->y_dst, dr->x_dst, dr->y_dst, dr->w, dr->h);
			p->modified = TRUE;
			}
		}
	/* For each layer, push new layer image onto the expose pixmap.
	*/
	if (p->modified)
		{
		do_top_layer_decals = push_decal_pixmaps(p, FALSE);
		push_krell_pixmaps(p);
		if (do_top_layer_decals)
			push_decal_pixmaps(p, TRUE);

		/* Draw regions from the expose pixmap onto the screen.  Draw the
		|  Old krell region and the new krell region.
		|  I should test here for region sizes to minimize screen draws.
		*/
#if 0
		if (regions_to_draw_are_large_enough)	/* Blast expose pixmap */
			gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC, p->pixmap,
					0, 0,   0, 0,   p->w, p->h);
		else
#endif
		drawable = p->drawing_area->window;
		for (list = p->decal; list; list = list->next)
			{
			d = (Decal *) list->data;
			if (d->state != DS_INVISIBLE && drawable)
				gdk_draw_pixmap(drawable, GK.draw1_GC,
						p->pixmap, d->x, d->y,  d->x, d->y,   d->w, d->h);
			d->modified = FALSE;
			}
		for (list = p->krell; list; list = list->next)
			{
			k = (Krell *) list->data;
			if (drawable)
				{
				dr = &k->old_draw;
				gdk_draw_pixmap(drawable, GK.draw1_GC, p->pixmap,
					dr->x_dst, dr->y_dst, dr->x_dst, dr->y_dst, dr->w, dr->h);
				dr = &k->draw;
				gdk_draw_pixmap(drawable, GK.draw1_GC, p->pixmap,
					dr->x_dst, dr->y_dst, dr->x_dst, dr->y_dst, dr->w, dr->h);
				}
			k->modified = FALSE;
			}
		}
	p->modified = FALSE;
	}

/* ---- Deprecated function - name change ---- */
void
gkrellm_draw_layers_force(Panel *p)
	{
	gkrellm_draw_panel_layers_force(p);
	}

void
gkrellm_draw_panel_layers_force(Panel *p)
	{
	GList	*list;

	if (!p)
		return;
	for (list = p->decal; list; list = list->next)
		((Decal *) list->data)->modified = TRUE;
	for (list = p->krell; list; list = list->next)
		((Krell *) list->data)->modified = TRUE;
	gkrellm_draw_panel_layers(p);
	}


void
gkrellm_draw_decal_pixmap(Panel *p, Decal *d, gint index)
	{
	gint	y_src;

	if (!d)
		return;
	if (d->value != index)
		{
		/* Write the new decal mask onto the panel stencil
		*/
		y_src = index * d->h;
		d->y_src = y_src;
		if (d->mask)		/* Can be NULL if no transparency	*/
			gdk_draw_pixmap(d->stencil, GK.bit1_GC, d->mask,
					0, y_src, 0, 0, d->w, d->h);
		else	/* Fill decal extent with white	*/
			gdk_draw_rectangle(d->stencil, GK.bit1_GC,
					TRUE, 0, 0, d->w, d->h);
		d->modified = TRUE;
		}
	d->value = index;
	}

void
gkrellm_draw_decal_text(Panel *p, Decal *d, gchar *s, gint value)
	{
	TextStyle	*ts;
	gint		x, y;

	if (!s || !d || d->state == DS_INVISIBLE)
		return;
	if (d->value != value || value == -1)
		{
		ts = &d->text_style;
		x = d->x_off;
		y = d->y_baseline;
//		gdk_draw_pixmap(d->pixmap, GK.draw1_GC, p->bg_pixmap,
//				d->x, d->y,  0, 0,  d->w, d->h);
		gkrellm_draw_string(d->pixmap, ts, x, y, s);
		gdk_draw_rectangle(d->stencil, GK.bit0_GC, TRUE, 0, 0, d->w, d->h);
		gdk_draw_string(d->stencil, ts->font, GK.bit1_GC, x, y, s);
		if (ts->effect)
			gdk_draw_string(d->stencil, ts->font, GK.bit1_GC, x + 1, y + 1, s);
		d->modified = TRUE;
		}
	d->value = value;
	}

void
gkrellm_move_krell_yoff(Panel *p, Krell *k, gint y)
	{
	Draw_rec	*dr, dr_old;

	if (!k)
		return;
	dr_old = k->draw;
	k->y0 = k->draw.y_dst = y;
	dr = &dr_old;
	if (p && p->pixmap && p->bg_pixmap)
		{
		gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
				dr->x_dst, dr->y_dst, dr->x_dst, dr->y_dst, dr->w, dr->h);
		if (p->drawing_area->window)
			gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC,
					p->bg_pixmap,
					dr->x_dst, dr->y_dst, dr->x_dst, dr->y_dst, dr->w, dr->h);
		}
	k->modified = TRUE;
	}

void
gkrellm_destroy_krell(Krell *k)
	{
	if (k)
		{
		if (k->stencil)
			gdk_pixmap_unref(k->stencil);
		if (k->pixmap)
			gdk_imlib_free_pixmap(k->pixmap);
		g_free(k);
		}
	}

void
gkrellm_remove_krell(Panel *p, Krell *k)
	{
	Draw_rec	*dr;

	if (!p || !k || !g_list_find(p->krell, k))
		return;
	p->krell = g_list_remove(p->krell, k);
	dr = &k->draw;
	if (p->pixmap && p->bg_pixmap)
		{
		gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
			dr->x_dst, dr->y_dst, dr->x_dst, dr->y_dst, dr->w, dr->h);
		if (p->drawing_area->window)
			gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC,
				p->bg_pixmap,
				dr->x_dst, dr->y_dst, dr->x_dst, dr->y_dst, dr->w, dr->h);
		gkrellm_draw_panel_layers_force(p);
		}
	}

void
gkrellm_remove_and_destroy_krell(Panel *p, Krell *k)
	{
	gkrellm_remove_krell(p, k);
	gkrellm_destroy_krell(k);
	}

void
gkrellm_insert_krell(Panel *p, Krell *k, gboolean append)
	{
	if (!p || !k || g_list_find(p->krell, k))
		return;
	if (append)
		p->krell = g_list_append(p->krell, k);
	else
		p->krell = g_list_prepend(p->krell, k);
	gkrellm_draw_panel_layers_force(p);
	}

void
gkrellm_insert_krell_nth(Panel *p, Krell *k, gint n)
	{
	if (!p || !k || g_list_find(p->krell, k))
		return;
	p->krell = g_list_insert(p->krell, k, n);
	gkrellm_draw_panel_layers_force(p);
	}

void
gkrellm_destroy_krell_list(Panel *p)
	{
	GList	*list;

	if (!p)
		return;
	for (list = p->krell; list; list = list->next)
		gkrellm_destroy_krell((Krell *) list->data);
	if (p->krell)
		g_list_free(p->krell);
	p->krell = NULL;
	}

void
gkrellm_set_krell_margins(Panel *p, Krell *k, gint left, gint right)
	{
	Draw_rec	*dr;

	if (!k)
		return;
	if (left > GK.chart_width - 3)
		left = GK.chart_width - 3;
	k->x0 = left;
	k->w_scale  = GK.chart_width - k->x0 - right - 1;
	if (k->w_scale < 2)
		k->w_scale = 2;
	k->modified = TRUE;

	dr = &k->draw;
	if (p && p->pixmap && p->bg_pixmap)
		{
		gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
				dr->x_dst, dr->y_dst, dr->x_dst, dr->y_dst, dr->w, dr->h);
		if (p->drawing_area->window)
			gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC,
					p->bg_pixmap,
					dr->x_dst, dr->y_dst, dr->x_dst, dr->y_dst, dr->w, dr->h);

		/* Move the krell proportionally between the new limits
		*/
		k->x_position = -1;
		if (k->monotonic)
			k->previous -= k->value;
		gkrellm_update_krell(p, k, k->value);
		gkrellm_draw_panel_layers_force(p);
		}
	}

void
gkrellm_set_krell_full_scale(Krell *k, gint full_scale, gint expand)
	{
	if (!k)
		return;
	k->full_scale_expand = (expand <= 0 ) ? 1 : expand;
	k->full_scale = full_scale * k->full_scale_expand;
	}
	
Krell *
gkrellm_create_krell(Panel *p, GdkImlibImage *im, Style *style)
	{
	GtkWidget	*top_win = gkrellm_get_top_window();
	Krell		*k;
	gint		w, h, w_render;

	k = (Krell *) g_new0(Krell, 1);
	if (p)
		p->krell = g_list_append(p->krell, k);

	if (im == NULL || style == NULL)
		{
		printf(_("create_krell: NULL image or style\n"));
		exit(0);
		}
	k->x0 = 0;
	k->y0 = style->krell_yoff;
	if (k->y0 == -1)
		gkrellm_get_top_bottom_margins(style, &k->y0, NULL);

	k->w_scale  = GK.chart_width - k->x0 - 1;

	w = im->rgb_width;
	h = im->rgb_height;

	if (style->krell_x_hot < 0)
		style->krell_x_hot = w / 2;
	w_render = w;
	k->x_hot = style->krell_x_hot;
	k->bar_mode = FALSE;

	switch (style->krell_expand)
		{
		case KRELL_EXPAND_NONE:
			break;
		case KRELL_EXPAND_BAR_MODE:
			w_render = GK.chart_width;
			k->bar_mode = TRUE;
			break;
		case KRELL_EXPAND_LEFT:
			if (style->krell_x_hot > 0 && style->krell_x_hot < w)
				{
				w_render = GK.chart_width * w / style->krell_x_hot;
				k->x_hot = GK.chart_width - 1;
				}
			break;
		case KRELL_EXPAND_RIGHT:
			if (w > style->krell_x_hot)
				{
				w_render = GK.chart_width * w / (w - style->krell_x_hot);
				k->x_hot = style->krell_x_hot * w_render / w;
				}
			break;
		}

	gdk_imlib_render(im, w_render, h);

	k->pixmap = gdk_imlib_copy_image(im);
	k->mask	  = gdk_imlib_copy_mask(im);
	k->h = h;
	k->w = w_render;

	k->depth = style->krell_depth;
	if (k->depth < 1)
		k->depth = 1;
	k->h_frame = h / k->depth;

	if (k->h_frame == 0)	/* gkrellmrc error or old style theme */
		{
		k->h_frame = 1;
		k->depth = h / k->h_frame;
		}

	k->stencil = gdk_pixmap_new(top_win->window, GK.chart_width, k->h_frame,1);
	if (k->bar_mode)
		{	/* Stencil is never modified for bar mode, so draw it once */
		gdk_draw_rectangle(k->stencil, GK.bit0_GC, TRUE,
					0, 0,  k->w, k->h_frame);
		if (k->mask)
			gdk_draw_pixmap(k->stencil, GK.bit1_GC, k->mask,
					0, k->h_frame * (k->depth - 1), 0, 0, k->w, k->h_frame);
		else
			gdk_draw_rectangle(k->stencil, GK.bit1_GC, TRUE,
					0, 0, k->w, k->h_frame);
		}

	k->period = style->krell_ema_period;
	if (k->period >= 4 * GK.update_HZ)
		k->period = 4 * GK.update_HZ;

	k->x_position = -1;      /* Force initial draw   */
	k->full_scale_expand = 1;
	k->monotonic = TRUE;
	return k;
	}

void
gkrellm_monotonic_krell_values(Krell *k, gboolean mono)
	{
	if (k)
		k->monotonic = mono;
	}

/* -------------------------------------------------------------- */

static GList	*button_list;

static void
disconnect_panel_signals(Panel *p, DecalButton *b)
	{
	gtk_signal_disconnect(GTK_OBJECT(p->drawing_area), b->id_press);
	gtk_signal_disconnect(GTK_OBJECT(p->drawing_area), b->id_release);
	gtk_signal_disconnect(GTK_OBJECT(p->drawing_area), b->id_enter);
	gtk_signal_disconnect(GTK_OBJECT(p->drawing_area), b->id_leave);
	}

void
gkrellm_move_decal(Panel *p, Decal *d, gint x, gint y)
	{
	gint	x_old, y_old;

	if (!d)
		return;
	x_old = d->x;
	y_old = d->y;
	d->x = x;
	d->y = y;
	if (p && p->pixmap && p->bg_pixmap)
		{
		gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
					x_old, y_old,  x_old, y_old,   d->w, d->h);
		if (p->drawing_area->window)
			gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC,
					p->bg_pixmap, x_old, y_old,  x_old, y_old,   d->w, d->h);
		}
	d->modified = TRUE;
	}

void
gkrellm_decal_on_top_layer(Decal *d, gboolean top)
	{
	if (!d)
		return;
	if (top)
		d->flags |= DF_TOP_LAYER;
	else
		d->flags &= ~DF_TOP_LAYER;
	d->modified = TRUE;
	}

void
gkrellm_destroy_decal(Decal *d)
	{
	if (d)
		{
		if ((d->flags & DF_LOCAL_PIXMAPS) && d->pixmap)
			gdk_pixmap_unref(d->pixmap);
		if (d->stencil)
			gdk_pixmap_unref(d->stencil);
		g_free(d);
		}
	}

void
gkrellm_remove_decal(Panel *p, Decal *d)
	{
	if (!p || !d || !g_list_find(p->decal, d))
		return;
	p->decal = g_list_remove(p->decal, d);
	if (p->pixmap && p->bg_pixmap)
		{
		gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
				d->x, d->y,  d->x, d->y,   d->w, d->h);
		if (p->drawing_area->window)
			gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC,
					p->bg_pixmap, d->x, d->y,  d->x, d->y,   d->w, d->h);
		gkrellm_draw_panel_layers_force(p);
		}
	}

void
gkrellm_remove_and_destroy_decal(Panel *p, Decal *d)
	{
	gkrellm_remove_decal(p, d);
	gkrellm_destroy_decal(d);
	}

void
gkrellm_insert_decal(Panel *p, Decal *d, gboolean append)
	{
	if (!p || !d || g_list_find(p->decal, d))
		return;
	if (append)
		p->decal = g_list_append(p->decal, d);
	else
		p->decal = g_list_prepend(p->decal, d);
	gkrellm_draw_panel_layers_force(p);
	}

void
gkrellm_insert_decal_nth(Panel *p, Decal *d, gint n)
	{
	if (!p || !d || g_list_find(p->decal, d))
		return;
	p->decal = g_list_insert(p->decal, d, n);
	gkrellm_draw_panel_layers_force(p);
	}

void
gkrellm_destroy_decal_list(Panel *p)
	{
	DecalButton	*b;
	Decal		*d;
	GList		*list;

	if (!p)
		return;
	for (list = p->decal; list; list = list->next)
		{
		d = (Decal *) list->data;
		if ((b = gkrellm_decal_is_button(d)) != NULL)
			{
			disconnect_panel_signals(p, b);
			button_list = g_list_remove(button_list, b);
			g_free(b);
			}
		gkrellm_destroy_decal(d);
		}
	if (p->decal)
		g_list_free(p->decal);
	p->decal = NULL;
	}

Decal *
gkrellm_create_decal_pixmap(Panel *p, GdkPixmap *pixmap, GdkBitmap *mask,
			gint depth, Style *style, gint x, gint y)
	{
	GtkWidget	*top_win;
	Decal		*d;

	if (!pixmap)
		return NULL;
	top_win = gkrellm_get_top_window();
	d = (Decal *) g_new0(Decal, 1);
	if (p)
		p->decal = g_list_append(p->decal, d);

	d->w = ((GdkWindowPrivate *) pixmap)->width;
	d->h = ((GdkWindowPrivate *) pixmap)->height / depth;
	if (d->h == 0)
		d->h = 1;

	d->x = x;
	if (d->x < 0 && style)
		{
		if (style->label_position < 50)
			d->x = GK.chart_width - d->w - style->margin;
		else
			d->x = style->margin;
		}
	d->y = y;
	if (d->y < 0)
		gkrellm_get_top_bottom_margins(style, &d->y, NULL);

	d->pixmap = pixmap;
	d->mask   = mask;
	d->stencil = gdk_pixmap_new(top_win->window, d->w, d->h, 1);

	d->value = -1;		/* Force initial draw */
	d->flags = 0;
	d->state = DS_VISIBLE;
	return d;
	}

Decal *
gkrellm_create_decal_text(Panel *p, gchar *string, TextStyle *ts,
			Style *style, gint x, gint y, gint w)
	{
	GtkWidget	*top_win;
	Decal		*d;
	gint		ll, rr, ww, aa, dd;

	if (!ts || !string)
		return NULL;
	top_win = gkrellm_get_top_window();
	d = (Decal *) g_new0(Decal, 1);
	if (p)
		p->decal = g_list_append(p->decal, d);

	d->text_style = *ts;
	gdk_string_extents(d->text_style.font, string, &ll, &rr, &ww, &aa, &dd);

	d->y_baseline = aa;

	/* Setup size and position defaults.  Height is a function of the
	|  reference string passed in, but width defaults to full chart width
	|  minus borders unless caller gives a w > 0.  
	*/
	d->h = aa + dd + d->text_style.effect;
	if (style)
		{
		if (w < 0)
			d->w = GK.chart_width - 2 * style->margin;
		else if (w == 0)
			d->w = ww + d->text_style.effect;
		else
			d->w = w;

		d->x = (x >= 0) ? x : style->margin;

		d->y = y;
		if (d->y < 0)
			gkrellm_get_top_bottom_margins(style, &d->y, NULL);
		}
	else
		{
		if (w < 0)
			d->w = GK.chart_width;
		else if (w == 0)
			d->w = ww;
		else
			d->w = w;
		d->x = x;
		d->y = y;
		}
	if (d->w == 0)
		d->w = 1;
	d->pixmap = gdk_pixmap_new(top_win->window, d->w, d->h, -1);;
	d->mask   = NULL;
	d->stencil = gdk_pixmap_new(top_win->window, d->w, d->h, 1);

	d->value = -1;		/* Force initial draw */
	d->flags = DF_LOCAL_PIXMAPS;
	d->state = DS_VISIBLE;

	return d;
	}

void
gkrellm_make_decal_invisible(Panel *p, Decal *d)
	{
	if (!p || !d || d->state == DS_INVISIBLE)
		return;
	d->state = DS_ERASING;
	d->modified = TRUE;
	gkrellm_draw_panel_layers(p);
	d->state = DS_INVISIBLE;
	}

void
gkrellm_make_decal_visible(Panel *p, Decal *d)
	{
	if (!p || !d || d->state == DS_VISIBLE)
		return;
	d->state = DS_VISIBLE;
	d->modified = TRUE;
	gkrellm_draw_panel_layers(p);
	}

gint
gkrellm_is_decal_visible(Decal *d)
	{
	if (!d)
		return FALSE;
	return (d->state == DS_VISIBLE) ? TRUE : FALSE;
	}

/* ===================================================================== */
#define	OVERLAY_BUTTON 	1


gboolean
gkrellm_in_decal(Decal *d, GdkEventButton *ev)
	{
	if (!d || !ev)
		return FALSE;
	if (   ev->x >= d->x && ev->x < d->x + d->w
		&& ev->y >= d->y && ev->y < d->y + d->h
	   )
		return TRUE;
	return FALSE;
	}

static void
set_button_index(DecalButton *b, gint index, gint do_draw)
	{
	if (!b)
		return;
	gkrellm_draw_decal_pixmap(b->p, b->d, index);
	b->cur_index = index;
	if (do_draw)
		gkrellm_draw_panel_layers(b->p);
	}

void
gkrellm_set_button_sensitive(DecalButton *b, gboolean sensitive)
	{
	if (b)
		b->sensitive = sensitive;
	}

void
gkrellm_hide_button(DecalButton *b)
	{
	if (!b)
		return;
	b->sensitive = FALSE;
	gkrellm_make_decal_invisible(b->p, b->d);
	}

void
gkrellm_show_button(DecalButton *b)
	{
	b->sensitive = TRUE;
	gkrellm_make_decal_visible(b->p, b->d);
	}

  /* FIXME, button_press and button_release are called multiple times
  |  for each real event if there are multiple buttons in the panel.
  |  This is because I set signal handlers for each button.  Signal
  |  handler should be set at the panel level with ref counts.
  */
static gint
cb_decal_button_press(GtkWidget *widget, GdkEventButton *ev)
	{
	GList		*list;
	DecalButton	*b;

	for (list = button_list; list; list = list->next)
		{
		b = (DecalButton *) list->data;
		if (widget != b->p->drawing_area)
			continue;
		if (   b->sensitive && (*(b->cb_in_button))(b->d, ev)
			&& (   (b->cb_button_click && ev->button == 1)
				|| (b->cb_button_right_click && ev->button == 3)
			   )
		   )
			{
			if (b->cur_index != b->pressed_index)
				{
				b->saved_index = b->cur_index;
				set_button_index(b, b->pressed_index, 1);
				}
			break;
			}
		}
	return TRUE;
	}

static gint
cb_decal_button_release(GtkWidget *widget, GdkEventButton *ev)
	{
	GList		*list;
	DecalButton	*b;
	gboolean	in_button;

	for (list = button_list; list; list = list->next)
		{
		b = (DecalButton *) list->data;
		if (widget != b->p->drawing_area)
			continue;
		if (b->cur_index == b->pressed_index)
			{
			set_button_index(b, b->saved_index, 1);
			in_button = (*(b->cb_in_button))(b->d, ev);
			if (in_button && b->cb_button_click && ev->button == 1)
				(*(b->cb_button_click))(b, b->data);
			else if (in_button && b->cb_button_right_click && ev->button == 3)
				(*(b->cb_button_right_click))(b, b->right_data);
		 	}
		}
	return TRUE;
	}

static gint
cb_decal_button_leave(GtkWidget *widget, GdkEventButton *ev)
	{
	GList		*list;
	DecalButton	*b;

	for (list = button_list; list; list = list->next)
		{
		b = (DecalButton *) list->data;
		if (widget != b->p->drawing_area)
			continue;
		if (b->type == OVERLAY_BUTTON)
			{
			if (b->sensitive)
				{
				b->d->state = DS_ERASING;
				set_button_index(b, 0, 0);
				b->d->modified = TRUE;
				gkrellm_draw_panel_layers(b->p);
				b->d->state = DS_INVISIBLE;
				}
			}
		else
			if (b->cur_index == b->pressed_index)
				set_button_index(b, b->saved_index, 1);
		}
	return TRUE;
	}

static gint
cb_decal_button_enter(GtkWidget *widget, GdkEventButton *ev)
	{
	GList		*list;
	DecalButton	*b;

	for (list = button_list; list; list = list->next)
		{
		b = (DecalButton *) list->data;
		if (widget != b->p->drawing_area)
			continue;
		if (b->type != OVERLAY_BUTTON || ! b->sensitive)
			continue;
		b->d->state = DS_VISIBLE;
		set_button_index(b, 0, 0);
		b->d->modified = TRUE;
		gkrellm_draw_panel_layers(b->p);
		}
	return TRUE;
	}

void
gkrellm_set_decal_button_index(DecalButton *b, gint index)
	{
	if (!b)
		return;
	if (b->cur_index == b->pressed_index)
		b->saved_index = index;	/* Throw away old save */
	else
		set_button_index(b, index, 0);
	}

DecalButton *
gkrellm_decal_is_button(Decal *d)
	{
	DecalButton	*b;
	GList		*list;

	for (list = button_list; list; list = list->next)
		{
		b = (DecalButton *) list->data;
		if (b->d == d)
			return b;
		}
	return NULL;
	}

void
gkrellm_destroy_button(DecalButton *b)
	{
	Panel	*p;

	if (!b)
		return;
	p = b->p;
	gkrellm_destroy_decal(b->d);
	p->decal = g_list_remove(p->decal, b->d);

	disconnect_panel_signals(p, b);
	button_list = g_list_remove(button_list, b);
	g_free(b);
	}

  /* This is called only when cleaning up for a rebuild.  All monitors
  |  are expected to clean out their decal lists at this time, so I don't
  |  do any decal freeing here, just clean up the button list.
  */
void
destroy_decal_button_list(void)
	{
	GList		*list;
	Panel		*p;
	DecalButton	*b;

	for (list = button_list; list; list = list->next)
		{
		b = (DecalButton *) list->data;
		p = b->p;
		disconnect_panel_signals(p, b);
		g_free(b);
		}
	g_list_free(button_list);
	button_list = NULL;
	}

void
gkrellm_set_in_button_callback(DecalButton *b, gint (*func)())
	{
	if (b)
		b->cb_in_button = func;
	}

void
gkrellm_decal_button_connect(DecalButton *b, void (*func)(), void *data)
	{
	if (!b)
		return;
	b->data = data;
	b->cb_button_click = func;
	}

void
gkrellm_decal_button_right_connect(DecalButton *b, void (*func)(), void *data)
	{
	if (!b)
		return;
	b->right_data = data;
	b->cb_button_right_click = func;
	}

  /* Make an existing decal into a decal button.  The decal should already
  |  be created and be in the panel's decal list.
  */
DecalButton *
gkrellm_make_decal_button(Panel *p, Decal *d, void (*func)(), void *data,
			gint cur_index, gint pressed_index)
	{
	DecalButton	*b;

	if (!p || !d)
		return NULL;
	b = g_new0(DecalButton, 1);
	b->p = p;
	b->d = d;
	b->pressed_index = pressed_index;
	b->data = data;
	b->cb_button_click = func;
	b->cb_in_button = gkrellm_in_decal;
	b->sensitive = TRUE;

	b->id_press = gtk_signal_connect(GTK_OBJECT(p->drawing_area),
		"button_press_event", (GtkSignalFunc) cb_decal_button_press, NULL);
	b->id_release = gtk_signal_connect(GTK_OBJECT(p->drawing_area),
		"button_release_event", (GtkSignalFunc) cb_decal_button_release, NULL);
	b->id_enter = gtk_signal_connect(GTK_OBJECT(p->drawing_area),
		"enter_notify_event", (GtkSignalFunc) cb_decal_button_enter, NULL);
	b->id_leave = gtk_signal_connect(GTK_OBJECT(p->drawing_area),
		"leave_notify_event", (GtkSignalFunc) cb_decal_button_leave, NULL);

	set_button_index(b, cur_index, 0);
	button_list = g_list_append(button_list, b);
	return b;
	}


DecalButton *
gkrellm_make_overlay_button(Panel *p, void (*func)(), void *data,
			gint x, gint y, gint w, gint h,
			GdkImlibImage *normal_image, GdkImlibImage *pressed_image)
	{
	GtkWidget	*top_win;
	DecalButton	*b;
	Decal		*d;
	GdkPixmap	*pm, *pixmap;
	GdkBitmap	*m, *mask;
	gint		margin;

	if (!p)
		return NULL;
	top_win = gkrellm_get_top_window();
	margin = (p->style) ? p->style->margin : 0;
	if (x < 0)
		{
		w -= x;
		x = 0;
		}
	if (y < 0)
		{
		h -= y;
		y = 0;
		}
	if (x + w > p->w)
		w = p->w - x;
	if (y + h > p->h)
		h = p->h -y;
	if (h < 2)
		h = 2;
	if (w < 4)
		w = 4;

	pixmap = gdk_pixmap_new(top_win->window, w, 2 * h, -1);
	mask = gdk_pixmap_new(top_win->window, w, 2 * h, 1);

	if (normal_image && pressed_image)
		{
		gdk_imlib_render(normal_image, w, h);
		pm = gdk_imlib_copy_image(normal_image);
		m = gdk_imlib_copy_mask(normal_image);
		gdk_draw_pixmap(pixmap, GK.draw1_GC, pm, 0, 0, 0, 0, w, h);
		if (m)
			gdk_draw_pixmap(mask, GK.bit1_GC, m, 0, 0, 0, 0, w, h);
		else
			gdk_draw_rectangle(mask, GK.bit1_GC, TRUE, 0, 0, w, h);
		gdk_imlib_free_pixmap(pm);

		gdk_imlib_render(pressed_image, w, h);
		pm = gdk_imlib_copy_image(pressed_image);
		m = gdk_imlib_copy_mask(pressed_image);
		gdk_draw_pixmap(pixmap, GK.draw1_GC, pm, 0, 0, 0, h, w, h);
		if (m)
			gdk_draw_pixmap(mask, GK.bit1_GC, m, 0, 0, 0, h, w, h);
		else
			gdk_draw_rectangle(mask, GK.bit1_GC, TRUE, 0, h, w, h);
		gdk_imlib_free_pixmap(pm);
		}
	else	/* Make a default frame. */
		{
		GdkColor	gray0, gray1;
		GdkColormap	*cmap;

		cmap = gdk_colormap_get_system();
		gdk_color_parse("gray65", &gray0);
		gdk_color_parse("gray100", &gray1);
		gdk_color_alloc(cmap, &gray0);
		gdk_color_alloc(cmap, &gray1);

		gdk_gc_set_foreground(GK.draw1_GC, &gray1);
		gdk_draw_line(pixmap, GK.draw1_GC, 0, 0, w - 1, 0);		
		gdk_draw_line(pixmap, GK.draw1_GC, 0, 0, 0, h - 1);		
		gdk_draw_line(pixmap, GK.draw1_GC, 0, 2 * h - 1, w - 1, 2 * h - 1);	
		gdk_draw_line(pixmap, GK.draw1_GC, w - 1, 2 * h - 1, w - 1, h);		

		gdk_gc_set_foreground(GK.draw1_GC, &gray0);
		gdk_draw_line(pixmap, GK.draw1_GC, 0, h - 1, w - 1, h - 1);		
		gdk_draw_line(pixmap, GK.draw1_GC, w - 1, h - 1, w - 1, 0);		
		gdk_draw_line(pixmap, GK.draw1_GC, 0, h, w - 1, h);		
		gdk_draw_line(pixmap, GK.draw1_GC, 0, h, 0, 2 * h - 1);		

		gdk_draw_rectangle(mask, GK.bit1_GC, TRUE, 0, 0, w, 2 * h);		
		gdk_draw_rectangle(mask, GK.bit0_GC, TRUE, 1, 1, w - 2, h - 2);
		gdk_draw_rectangle(mask, GK.bit0_GC, TRUE, 1, h + 1, w - 2, h - 2);

		gdk_colormap_free_colors(cmap, &gray0, 1);
		gdk_colormap_free_colors(cmap, &gray1, 1);
		}

	d = gkrellm_create_decal_pixmap(p, pixmap, mask, 2, NULL, x, y);
	d->flags |= DF_LOCAL_PIXMAPS;
	d->state = DS_INVISIBLE;

	b = gkrellm_make_decal_button(p, d, func, data, 0, 1);
	b->type = OVERLAY_BUTTON;

	return b;
	}

DecalButton *
gkrellm_put_decal_in_panel_button(Panel *p, Decal *d,
								void (*func)(), void *data)
	{
	DecalButton		*b;
	GdkImlibBorder	*border;
	gint			x, y, w, h;

	if (!p || !d)
		return NULL;
	border = &GK.button_panel_border;
	x = d->x - border->left;
	y = d->y - border->top;
	w = d->w + border->left + border->right;
	h = d->h + border->top + border->bottom;
	b = gkrellm_make_overlay_button(p, func, data, x, y, w, h,
				GK.button_panel_out_image, GK.button_panel_in_image);
	return b;
	}

DecalButton *
gkrellm_put_decal_in_meter_button(Panel *p, Decal *d, void (*func)(), void *data)
	{
	DecalButton		*b;
	GdkImlibBorder	*border;
	gint			x, y, w, h;

	if (!p || !d)
		return NULL;
	border = &GK.button_meter_border;
	x = d->x - border->left;
	y = d->y - border->top;
	w = d->w + border->left + border->right;
	h = d->h + border->top + border->bottom;
	b = gkrellm_make_overlay_button(p, func, data, x, y, w, h,
				GK.button_meter_out_image, GK.button_meter_in_image);
	return b;
	}


static DecalButton *
put_label_in_button(Panel *p, void (*func)(), void *data, gint type, gint pad)
	{
	DecalButton		*b;
	Label			*lbl;
	GdkImlibImage	*normal_image, *pressed_image;
	GdkImlibBorder	*style_border, *fr_border;
	gint			x, y, w, h;

	if (!p)
		return NULL;
	fr_border = (type == METER_PANEL_TYPE) ? &GK.button_meter_border
										: &GK.button_panel_border;
	normal_image = (type == METER_PANEL_TYPE)
				? GK.button_meter_out_image : GK.button_panel_out_image;
	pressed_image = (type == METER_PANEL_TYPE)
				? GK.button_meter_in_image : GK.button_panel_in_image;

	/* If no panel label, put the whole panel in the button.
	*/
	if ((lbl = p->label) == NULL || lbl->string == NULL || lbl->position < 0)
		{
		style_border = p->style ? &p->style->border : &zero_border;
		x = 0 + (p->style) ? p->style->margin : 0;
		y = 0 + style_border->top - fr_border->top;
		w = p->w - x -  ((p->style) ? p->style->margin : 0);
		h = p->h - y - style_border->bottom + fr_border->bottom;
		}
	else
		{
		x = lbl->x_panel - fr_border->left - pad;
		y = lbl->y_baseline - lbl->ascent - fr_border->top;
		w = lbl->width + fr_border->left + fr_border->right + 2 * pad;
		h = lbl->ascent + lbl->descent + p->textstyle->effect
				+ fr_border->top + fr_border->bottom;
		}
	b = gkrellm_make_overlay_button(p, func, data, x, y, w, h,
					normal_image, pressed_image);
	return b;
	}


DecalButton *
gkrellm_put_label_in_panel_button(Panel *p, void (*func)(), void *data,
				gint pad)
	{
	DecalButton		*b;

	b = put_label_in_button(p, func, data, CHART_PANEL_TYPE, pad);
	return b;
	}


DecalButton *
gkrellm_put_label_in_meter_button(Panel *p, void (*func)(), void *data,
				gint pad)
	{
	DecalButton		*b;

	b = put_label_in_button(p, func, data, METER_PANEL_TYPE, pad);
	return b;
	}



  /* As of 0.9.6 I want to move away from these last four routines.
  |  For now I'll keep them around to avoid breaking plugins.
  */
Decal *
gkrellm_create_text_decal(Panel *p, gchar *string, TextStyle *ts,
			Style *style, gint x, gint y, gint w)
	{
	return gkrellm_create_decal_text(p, string, ts, style, x, y, w);
	}

Decal *
gkrellm_create_pixmap_decal(Panel *p, GdkPixmap *pixmap, GdkBitmap *mask,
			gint depth, Style *style)
	{
	return gkrellm_create_decal_pixmap(p, pixmap, mask, depth, style, -1, -1);
	}

void
gkrellm_draw_pixmap_decal(Panel *p, Decal *d, gint index)
	{
	gkrellm_draw_decal_pixmap(p, d, index);
	}

void
gkrellm_draw_text_decal(Panel *p, Decal *d, gchar *str, gint value)
	{
	gkrellm_draw_decal_text(p, d, str, value);
	}
