/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|  4/22/2001  Solaris code contributed by Daisuke Yabuki <dxy@acm.org>
|  1/23/2001  Hajimu UMEMOTO improvements with read_freebsd_cpu() and
|               register_freebsd_cpus().
| 10/12/2000  NetBSD code contributed by Anthony Mallet <metall@ficus.yi.org>
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include "cpu_disk.h"


GList	*cpu_mon_list;

void	(*read_system_cpu_info)();

static gint		n_cpus;
gint			n_smp_cpus;


/* ====== System dependent interface ====================================== */

/* ----- FreeBSD ---------------------------------------------------- */

#if defined(__FreeBSD__) || defined(__OpenBSD__)
#include <sys/dkstat.h>
#include <kvm.h>

static struct nlist nl[] = {
#define N_CP_TIME	0
	{ "_cp_time" },
	{ "" }
};

extern	kvm_t	*kvmd;

#if defined(__FreeBSD__)
#include <sys/param.h>
#include <sys/sysctl.h>

static int	oid_cp_time[CTL_MAXNAME + 2];
static size_t	oid_cp_time_len = sizeof(oid_cp_time);
static gint	have_cp_time;

void
read_freebsd_cpu()
	{
	static gint	data_read_tick = -1;
	long		cp_time[CPUSTATES];
	int		len = sizeof(cp_time);
	CpuMon		*cpu;

	if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
		return;
	data_read_tick = GK.timer_ticks;

	if (have_cp_time)
		{
		if (sysctl(oid_cp_time, oid_cp_time_len,
			   cp_time, &len, 0, 0) < 0)
			return;
		}
	else
		{
		if (kvmd == NULL)
			return;
		if (nl[0].n_type == 0)
			if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
				return;
		if (kvm_read(kvmd, nl[N_CP_TIME].n_value, (char *)&cp_time,
			     sizeof(cp_time)) != sizeof(cp_time))
			return;
		}

	/* Currently, SMP is not supported */
	cpu = (CpuMon *) cpu_mon_list->data;
	cpu->user = cp_time[CP_USER];
	cpu->nice = cp_time[CP_NICE];
	cpu->sys = cp_time[CP_SYS];
	cpu->idle = cp_time[CP_IDLE];
	}

static void register_glibtop_cpus();

static void
register_freebsd_cpus()
	{
	static int	oid_name2oid[2] = { 0, 3 };
	static char	*name = "kern.cp_time";

	register_glibtop_cpus();

	if (sysctl(oid_name2oid, 2, oid_cp_time, &oid_cp_time_len,
		   (void *)name, strlen(name)) < 0)
		return;
	oid_cp_time_len /= sizeof(int);
	++have_cp_time;
	}
#else
void
read_openbsd_cpu()
	{
	long		cp_time[CPUSTATES];
	CpuMon		*cpu;
	static gint	data_read_tick	= -1;

	if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
		return;
	data_read_tick = GK.timer_ticks;

	if (kvmd == NULL)
		return;
	if (nl[0].n_type == 0)
		if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
			return;
	if (kvm_read(kvmd, nl[N_CP_TIME].n_value,
		     (char *)&cp_time, sizeof(cp_time)) != sizeof(cp_time))
		return;

	/* Currently, SMP is not supported */
	cpu = (CpuMon *) cpu_mon_list->data;
	cpu->user = cp_time[CP_USER];
	cpu->nice = cp_time[CP_NICE];
	cpu->sys = cp_time[CP_SYS];
	cpu->idle = cp_time[CP_IDLE];
	}
#endif
#endif


/* ----- Linux ---------------------------------------------------- */
#if defined(__linux__)
#define	PROC_STAT_FILE	"/proc/stat"

static gulong	swapin,
				swapout;

  /* CPU, and Disk monitors call this in their update routine. 
  |  Whoever calls it first will read the data for everyone.
  |
  | /proc/stat has cpu entries like:
  |		cpu		total_user	total_nice	total_sys	total_idle
  |		cpu0	cpu0_user	cpu0_nice	cpu0_sys	cpu0_idle
  |			...
  |		cpuN	cpuN_user	cpuN_nice	cpuN_sys	cpuN_idle
  |  where ticks for cpu are jiffies * smp_num_cpus
  |  and ticks for cpu[i] are jiffies (1/CLK_TCK)
  */
void
read_proc_stat(void)
	{
	FILE		*f;
	GList		*list;
	CpuMon		*cpu;
	DiskMon		*disk;
	gchar		*item, *arg;
	gchar		buf[1024];
	static gint	data_read_tick	= -1;

	if (   data_read_tick == GK.timer_ticks		/* Only one read per tick */
		|| (f = fopen(PROC_STAT_FILE, "r")) == NULL
	   )
		return;
	data_read_tick = GK.timer_ticks;

	composite_disk->rblk = 0;
	composite_disk->wblk = 0;
	while ((fgets(buf, sizeof(buf), f)) != NULL)
		{
		item = strtok(buf, " \t\n");
		arg = strtok(NULL, "\n");
		if (item == NULL || arg == NULL)
			continue;
		if (item[0] == 'c' && item[1] == 'p')
			{
			for (list = cpu_mon_list; list; list = list->next)
				{
				cpu = (CpuMon *) list->data;
				if (strcmp(cpu->name, item) == 0)
					{
					sscanf(arg,"%lu %lu %lu %lu", &cpu->user, &cpu->nice,
								&cpu->sys, &cpu->idle);
					break;
					}
				}
			}
		else if (strcmp("disk_rblk", item) == 0)	/* Pre kernel 2.4 format */
			{
			for (list = disk_mon_list->next; list; list = list->next)
				{
				disk = (DiskMon *) list->data;
				disk->rblk = strtoul(arg, &arg, 0);
				composite_disk->rblk += disk->rblk;
				}
			}
		else if (strcmp("disk_wblk", item) == 0)	/* Pre kernel 2.4 format */
			{
			for (list = disk_mon_list->next; list; list = list->next)
				{
				disk = (DiskMon *) list->data;
				disk->wblk = strtoul (arg, &arg, 0);
				composite_disk->wblk += disk->wblk;
				}
			}
		/* Read swap data for the meminfo monitor
		*/
		else if (strcmp("swap", item) == 0)
			sscanf(arg, "%lu %lu", &swapin, &swapout);

		else if (strcmp("disk_io:", item) == 0)	/* Kernel 2.4 format */
			{
			gint	major, i_disk, n;
			gulong	rblk, wblk, rb1, rb2, wb1, wb2;
			DiskMon	*disk;

			item = strtok(arg, " \t\n");
			while (item)
				{
				/* disk_io lines in 2.4.x kernels have had 2 formats */
				n = sscanf(item, "(%d,%d):(%*d,%lu,%lu,%lu,%lu)",
						&major, &i_disk, &rb1, &rb2, &wb1, &wb2);
				if (n == 6)	/* patched as of 2.4.0-test1-ac9 */
					{		/* (major,disk):(total_io,rio,rblk,wio,wblk) */
					rblk = rb2;
					wblk = wb2;
					}
				else	/* 2.3.99-pre8 to 2.4.0-testX */
					{		/* (major,disk):(rio,rblk,wio,wblk) */
					rblk = rb1;
					wblk = wb1;
					}
				if ((disk = lookup_disk_by_device(major, i_disk)) != NULL)
					{
					disk->rblk = rblk;
					disk->wblk = wblk;
					}
				composite_disk->rblk += rblk;
				composite_disk->wblk += wblk;
				item = strtok(NULL, " \t\n");
				}
			break;
			}
		}
	fclose(f);
	}

void
read_stat_swap(gulong *sin, gulong *sout)
	{
	*sin = swapin;
	*sout = swapout;
	}

static void
register_stat_cpus(void)
	{
	FILE	*f;
	CpuMon	*cpu;
	GList	*list;
	gchar	buf[1024], *s;

	if ((f = fopen(PROC_STAT_FILE, "r")) == NULL)
		return;

	while (fgets(buf, sizeof(buf), f))
		{
		if (strncmp(buf, "cpu", 3) != 0)
			continue;
		s = strtok(buf, " \t\n");
		cpu = g_new0(CpuMon, 1);
		cpu->name = g_strdup(s);
		cpu_mon_list = g_list_append(cpu_mon_list, cpu);
		++n_cpus;
		}
	fclose(f);

	/* There can be cpu and cpu0 only (single cpu machines with kernel 2.4
	|  or 2.2 compiled for SMP), and in this case, eliminate cpu0.
	*/
	n_smp_cpus = n_cpus - 1;
	if (n_smp_cpus == 1)	/* cpu and cpu0 but no cpu1 (not really SMP)	*/
		{
		n_smp_cpus = 0;
		n_cpus = 1;
		list = cpu_mon_list->next;	/* cpu0 */
		cpu = (CpuMon *) list->data;
		g_free(cpu->name);
		g_free(cpu);
		cpu_mon_list = g_list_remove_link(cpu_mon_list, list);
		}
	if (n_smp_cpus)			/* The first one is the composite cpu	*/
		((CpuMon *) cpu_mon_list->data)->is_composite = TRUE;
	}
#endif	/* __linux__ */


/* ----- NetBSD --------------------------------------------------------- */

#if defined(__NetBSD__)

#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/sched.h>

void
read_netbsd_cpu()
{
   static gint data_read_tick = -1;
   static int mib[] = { CTL_KERN, KERN_CP_TIME };
   u_int64_t cp_time[CPUSTATES];
   int len = sizeof(cp_time);
   CpuMon *cpu;

   if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
      return;
   data_read_tick = GK.timer_ticks;

   if (sysctl(mib, 2, cp_time, &len, NULL, 0) < 0) return;

   /* Currently, SMP is not supported */
   cpu = (CpuMon *) cpu_mon_list->data;
   cpu->user = cp_time[CP_USER];
   cpu->nice = cp_time[CP_NICE];
   cpu->sys = cp_time[CP_SYS];
   cpu->idle = cp_time[CP_IDLE];
}

#endif /* __NetBSD__ */


/* ----- Solaris ------------------------------------------------------ */

#if defined(__solaris__)

#include <kstat.h>
#include <sys/sysinfo.h>

void
read_solaris_cpu()
{
    CpuMon *cpu;
    GList *list;
    extern kstat_ctl_t *kc;
    kstat_t *ksp;
    cpu_stat_t cs;
    gulong composite_user, composite_nice, composite_sys, composite_idle;
    gchar *cpuid;
#if 0
    static gint data_read_tick = -1;

    if (data_read_tick == GK.timer_ticks)        /* Only one read per tick */
        return;
    data_read_tick = GK.timer_ticks;
#else
    /* one read per tick seems to make older Ultra machines too busy */
    if (!GK.second_tick)
        return;
#endif

    composite_user = composite_nice = composite_sys = composite_idle = 0;

    if (kstat_chain_update(kc) == -1) {
        perror("kstat_chain_update");
        return;
    }

    for (ksp = kc->kc_chain; ksp; ksp = ksp->ks_next) {
        if (strcmp(ksp->ks_module, "cpu_stat")) 
            continue;
        if (kstat_read(kc, ksp, &cs) == -1) {
            perror("kstat_read");
            continue;
        }
        for (list = cpu_mon_list; list; list = list->next) {
            cpu = (CpuMon *) list->data;
            cpuid = g_strdup_printf("cpu%d", ksp->ks_instance);
            if (strcmp(cpu->name, cpuid)) {
                g_free(cpuid);
                continue;
            }
            g_free(cpuid);

            cpu->user = cs.cpu_sysinfo.cpu[CPU_USER];
            cpu->nice = cs.cpu_sysinfo.cpu[CPU_WAIT];
            cpu->sys  = cs.cpu_sysinfo.cpu[CPU_KERNEL];
            cpu->idle = cs.cpu_sysinfo.cpu[CPU_IDLE];

            if (n_smp_cpus) {
                composite_user += cpu->user;
                composite_nice += cpu->nice;
                composite_sys  += cpu->sys;
                composite_idle += cpu->idle;
            }
        } 
    }

    /* dump composite data onto the first element (composite element) */
    if (n_smp_cpus) {
        ((CpuMon *) cpu_mon_list->data)->user = composite_user;
        ((CpuMon *) cpu_mon_list->data)->nice = composite_nice;
        ((CpuMon *) cpu_mon_list->data)->sys  = composite_sys;
        ((CpuMon *) cpu_mon_list->data)->idle = composite_idle;
    }

}

/*
 * note: on some SPARC systems, you can monitor temperature of CPUs 
 * with kstat (unix::temperature:[min/max/state/trend...])
 */

static void 
register_solaris_cpus() {
    CpuMon *cpu;
    extern kstat_ctl_t *kc;
    kstat_t *ksp;

    n_cpus = n_smp_cpus = 0;

    if(kstat_chain_update(kc) == -1) {
        perror("kstat_chain_update");
        return;
    } 

    for (ksp = kc->kc_chain; ksp; ksp = ksp->ks_next) {
        if (strcmp(ksp->ks_module, "cpu_stat")) 
            continue;
        if (kstat_read(kc, ksp, NULL) != -1) {
            cpu = g_new0(CpuMon, 1);
            cpu->name = g_strdup_printf("cpu%d", ksp->ks_instance);
            cpu_mon_list = g_list_append(cpu_mon_list, cpu);
            n_cpus++;
        }
    }

    if (n_cpus > 1) 
	n_smp_cpus++;

    /* create composite cpu entry and add it at the head of the list */
    if (n_smp_cpus) {
        cpu = g_new0(CpuMon, 1);
        cpu->name = g_strdup("cpu"); 
        cpu->is_composite = TRUE;
        cpu_mon_list = g_list_insert(cpu_mon_list, cpu, 0);
    }

}

#endif /* __solaris__ */

/* ----- Others ------------------------------------------------------- */
#if defined(USE_LIBGTOP)

#include <glibtop/cpu.h>

static void
read_glibtop_cpu(void)
	{
	CpuMon		*cpu;
	GList		*list;
	glibtop_cpu	glt_cpu;

	for (list = cpu_mon_list; list; list = list->next)	/* Only one! */
		{
		glibtop_get_cpu(&glt_cpu);
		cpu = (CpuMon *) list->data;
		cpu->user = (gulong) glt_cpu.user;
		cpu->nice = (gulong) glt_cpu.nice;
		cpu->sys  = (gulong) glt_cpu.sys;
		cpu->idle = (gulong) glt_cpu.idle;
		}
	}
#endif


#if !defined(__linux__) && !defined(__solaris__)
static void
register_glibtop_cpus(void)
	{
	CpuMon	*cpu;

	cpu = g_new0(CpuMon, 1);
	cpu->name = g_strdup("cpu");
	cpu_mon_list = g_list_append(cpu_mon_list, cpu);
	n_cpus = 1;
	n_smp_cpus = 0;
	}
#endif

/* ----- Pick a system interface ----------------------------------------- */

static gint
setup_cpu_interface(void)
	{
#if defined(__FreeBSD__)
	register_freebsd_cpus();
	read_system_cpu_info = read_freebsd_cpu;
#elif defined(__linux__)
	register_stat_cpus();
	read_system_cpu_info = read_proc_stat;
#elif defined(__NetBSD__)
	register_glibtop_cpus();
	read_system_cpu_info = read_netbsd_cpu;
#elif defined(__OpenBSD__)
	register_glibtop_cpus();
	read_system_cpu_info = read_openbsd_cpu;
#elif defined(__solaris__)
	register_solaris_cpus();   
	read_system_cpu_info = read_solaris_cpu;
#else
	register_glibtop_cpus();
	read_system_cpu_info = read_glibtop_cpu;
#endif
	return TRUE;
	}

/* ======================================================================== */
/* Exporting CPU data for plugins */

gint
gkrellm_smp_cpus(void)
	{
	return n_smp_cpus;
	}

gboolean
gkrellm_cpu_stats(gint n, gulong *user, gulong *nice,
			gulong *sys, gulong *idle)
	{
	GList	*list;
	CpuMon	*cpu;

	list = g_list_nth(cpu_mon_list, n);
	if (!list)
		return FALSE;
	cpu = (CpuMon *) list->data;
	if (user)
		*user = cpu->user;
	if (nice)
		*nice = cpu->nice;
	if (sys)
		*sys = cpu->sys;
	if (idle)
		*idle = cpu->idle;
	return TRUE;
	}

/* ======================================================================== */

  /* Values for smp_mode - must be same order as buttons in config */
#define	SMP_REAL_MODE				0
#define	SMP_COMPOSITE_MODE			1
#define	SMP_COMPOSITE_AND_REAL_MODE	2

static gint		smp_mode;
static Monitor	*mon_cpu;

static gboolean	cpu_enabled  = TRUE;
static gint		omit_nice_mode;
static gint		ascent;
static gint		style_id;
static gint		mapped_sensors;
static gint		sensor_separate_mode;


static gchar    *text_format;

static void
format_chart_text(CpuMon *cpu, gchar *buf, gint size)
	{
	Chart	*cp;
	gchar	c, *s;
	gint	len, sys, user, nice, total, t;

	--size;
	*buf = '\0';
	cp = cpu->chart;
	sys = gkrellm_get_current_chartdata(cpu->sys_cd);
	user = gkrellm_get_current_chartdata(cpu->user_cd);
	nice = gkrellm_get_current_chartdata(cpu->nice_cd);
	total = sys + user;
	if (!omit_nice_mode && !gkrellm_get_chartdata_hide(cpu->nice_cd))
		total += nice;
	for (s = text_format; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			t = -1;
			if ((c = *(s + 1)) == 'T')
				t = total;
			else if (c == 's')
				t = sys;
			else if (c == 'u')
				t = user;
			else if (c == 'n')
				t = nice;
			else if (c == 'L')
				len = snprintf(buf, size, "%s", cp->panel->label->string);
			else
				{
				*buf = *s;
				if (size > 1)
					{
					*(buf + 1) = *(s + 1);
					++len;
					}
				}
			if (t >= 0)
				{
				t = ((200 * t / cpu->total_diff) + 1) / 2;
				if (t > 100)
					t = 100;
				len = snprintf(buf, size, "%d%%", t);
				}
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';	
	}

static void
draw_cpu_extra(CpuMon *cpu)
	{
	Chart	*cp  = cpu->chart;
	gchar	buf[128];

	if (!cp || cpu->total_diff <= 0)
		return;
	format_chart_text(cpu, buf, sizeof(buf));
	gkrellm_draw_chart_text(cp, style_id, buf);
	}

static void
refresh_cpu_chart(CpuMon *cpu)
	{
	Chart	*cp		= cpu->chart;

	gkrellm_draw_chartdata(cp);
	if (cpu->extra_info)
		draw_cpu_extra(cpu);
	gkrellm_draw_chart_to_screen(cp);
	}

static void
draw_sensor_decals(CpuMon *cpu)
	{
	Panel	*p		= cpu->chart->panel;
	gchar	*name	= cpu->name;
	gchar	units;
	gfloat	t, f;
	gint	toggle;

	if (   sensor_separate_mode
		&& (mapped_sensors & SENSOR_TEMPERATURE)
		&& (mapped_sensors & SENSOR_FAN)
	   )
		{
		gkrellm_sensor_read_temperature(name, &t, &units);
		gkrellm_sensor_draw_temperature_decal(p, cpu->sensor_decal, t, units);
		gkrellm_sensor_read_fan(name, &f);
		gkrellm_sensor_draw_fan_decal(p, cpu->fan_decal, f);
		}
	else
		{
		toggle = time_now & 2;
		if (   (mapped_sensors & SENSOR_FAN)
			&& (toggle || !(mapped_sensors & SENSOR_TEMPERATURE))
		   )
			{
			gkrellm_sensor_read_fan(name, &f);
			gkrellm_sensor_draw_fan_decal(p, cpu->sensor_decal, f);
			}
		else if (   (mapped_sensors & SENSOR_TEMPERATURE)
				 && (!toggle || !(mapped_sensors & SENSOR_FAN))
				)
			{
			gkrellm_sensor_read_temperature(name, &t, &units);
			gkrellm_sensor_draw_temperature_decal(p, cpu->sensor_decal, t,
							units);
			}
		}
	}

static void
update_cpu(void)
	{
	GList		*list;
	CpuMon		*cpu;
	Chart		*cp;
	Panel		*p;
	Krell		*krell;
	gulong		total, krell_value;

	GK.cpu_sys_activity = 0;
	(*read_system_cpu_info)();

	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;
		if (!cpu->enabled)
			continue;
		cp = cpu->chart;
		p = cp->panel;
		if (smp_mode == SMP_REAL_MODE)
			GK.cpu_sys_activity += (int)(cpu->sys - cpu->sys_cd->previous);
		else if (list == cpu_mon_list)	/* Use composite cpu values */
			GK.cpu_sys_activity = (int)(cpu->sys - cpu->sys_cd->previous);
		total = cpu->user + cpu->nice + cpu->sys + cpu->idle;
		if (GK.second_tick)
			{
			cpu->total_diff = (gint) (total - cpu->previous_total_sec);
			cpu->previous_total_sec = total;
			gkrellm_store_chartdata(cp, total, cpu->sys, cpu->user, cpu->nice);
			refresh_cpu_chart(cpu);
			}
		if (   (GK.two_second_tick && !sensor_separate_mode)
			|| (GK.five_second_tick && sensor_separate_mode)
		   )
			draw_sensor_decals(cpu);

		krell = KRELL(cp->panel);
		gkrellm_set_krell_full_scale(krell,
					(gint) (total - cpu->previous_total), 10);
		if (cpu->previous_total > 0)
			{
			krell_value = cpu->sys + cpu->user;
			if (!omit_nice_mode && !gkrellm_get_chartdata_hide(cpu->nice_cd))
				krell_value += cpu->nice;
			gkrellm_update_krell(p, krell, krell_value);
			gkrellm_draw_panel_layers(p);
			}
		cpu->previous_total = total;
		}
	}


static gint
cpu_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GList		*list;
	Chart		*cp;
	GdkPixmap	*pixmap	= NULL;

	for (list = cpu_mon_list; list; list = list->next)
		{
		cp = ((CpuMon *) list->data)->chart;
		if (widget == cp->drawing_area)
			pixmap = cp->pixmap;
		else if (widget == cp->panel->drawing_area)
			pixmap = cp->panel->pixmap;
		if (pixmap)
			{
			gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
			break;
			}
		}
	return FALSE;
	}


static gint
cb_cpu_extra(GtkWidget *widget, GdkEventButton *ev)
	{
	GList	*list;
	CpuMon	*cpu;
	Chart	*cp;

	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;
		cp = cpu->chart;
		if (widget != cpu->chart->drawing_area)
			continue;
		if (ev->button == 1 && ev->type == GDK_BUTTON_PRESS)
			{
			cpu->extra_info = !cpu->extra_info;
			refresh_cpu_chart(cpu);
			gkrellm_config_modified();
			}
		else if (   ev->button == 3
				 || (ev->button == 1 && ev->type == GDK_2BUTTON_PRESS)
				)
			gkrellm_chartconfig_window_create(cpu->chart);
		break;
		}
	return TRUE;
	}

static void
setup_cpu_scaling(ChartConfig *cf)
	{
	gint	grids;

	grids = gkrellm_get_chartconfig_fixed_grids(cf);
	if (!grids)
		grids = FULL_SCALE_GRIDS;

	gkrellm_set_chartconfig_grid_resolution(cf,
				CPU_TICKS_PER_SECOND / grids);
	}

static gboolean
enable_cpu_visibility(CpuMon *cpu)
	{
	gint	enabled = cpu_enabled;

	if (n_smp_cpus > 0)
		{
		if (   (cpu->is_composite && smp_mode == SMP_REAL_MODE)
			|| (! cpu->is_composite && smp_mode == SMP_COMPOSITE_MODE)
		   )
			enabled = FALSE;
		}
	return gkrellm_chart_enable_visibility(cpu->chart, enabled, &cpu->enabled);
	}


static void
cb_cpu_temp_alert_trigger(Alert *alert, CpuMon *cpu)
	{
	AlertDecal	*ad;
	Decal		*d;

	if (alert && cpu && cpu->chart)
		{
		ad = &alert->ad;
		d = cpu->sensor_decal;
		if (d)
			{
			ad->x = d->x - 1;
			ad->y = d->y - 1;
			ad->w = d->w + 2;
			ad->h = d->h + 2;
			gkrellm_render_default_alert_decal(alert);
			}
		alert->panel = cpu->chart->panel;
		}
	}

static void
cb_cpu_fan_alert_trigger(Alert *alert, CpuMon *cpu)
	{
	AlertDecal	*ad;
	Decal		*d;

	if (alert && cpu && cpu->chart)
		{
		ad = &alert->ad;
		if (sensor_separate_mode)
			d = cpu->fan_decal;
		else
			d = cpu->sensor_decal;
		if (d)
			{
			ad->x = d->x - 1;
			ad->y = d->y - 1;
			ad->w = d->w + 2;
			ad->h = d->h + 2;
			gkrellm_render_default_alert_decal(alert);
			}
		alert->panel = cpu->chart->panel;
		}
	}

  /* How to decide when to make the sensor_decal and fan_decal visible.
  |  The sensor_decal can show temp values, fan values, or alternating
  |  temp/fan values.  The fan_decal only shows fan values when non
  |  alternating mode is selected. The sensors are mapped in sensors.c
  |
  |   Sensor and fan decal display truth table:
  |   |-----decals visible----||--sensors mapped--|   separate  |
  |	  |sensor_decal  fan_decal||   temp     fan   |    mode     |
  |   |-----------------------||--------------------------------|
  |   |     0           0     ||     0       0          0       |
  |   |     1           0     ||     1       0          0       |
  |   |     1           0     ||     0       1          0       |
  |   |     1           0     ||     1       1          0       |
  |   |     0           0     ||     0       0          1       |
  |   |     1           0     ||     1       0          1       |
  |   |     1           0     ||     0       1          1       |
  |   |     1           1     ||     1       1          1       |
  |   |----------------------------------------------------------
  */
static gint
adjust_sensors_display(CpuMon *cpu, gint force)
	{
	Panel	*p;
	Decal	*ds, *df;
	Alert	*alert;
	gint	position	= 0;

	ds = cpu->sensor_decal;
	df = cpu->fan_decal;
	if (!ds || !df)
		return FALSE;
	/* The test for state change is decal state vs success at reading
	|  a temperature.
	*/
	p = cpu->chart->panel;
	mapped_sensors = 0;
	if (!GK.demo)
		{
		gkrellm_mapped_sensor_alert_connect(cpu->name,
				SENSOR_TEMPERATURE, cb_cpu_temp_alert_trigger, cpu);
		gkrellm_mapped_sensor_alert_connect(cpu->name,
				SENSOR_FAN, cb_cpu_fan_alert_trigger, cpu);
		}

	/* If a fan alert is triggered, turn it off in case fan decal being used
	|  is changed.  The alert will just retrigger at next fan update.
	*/
	alert = gkrellm_mapped_sensor_alert(cpu->name, SENSOR_FAN);
	gkrellm_reset_alert(alert);

	if (get_mapped_sensor(cpu->name, SENSOR_TEMPERATURE) || GK.demo)
		mapped_sensors |= SENSOR_TEMPERATURE;
	if (get_mapped_sensor(cpu->name, SENSOR_FAN) || GK.demo)
		mapped_sensors |= SENSOR_FAN;

	if (mapped_sensors & (SENSOR_TEMPERATURE | SENSOR_FAN))
		{
		if (! gkrellm_is_decal_visible(ds) || force)
			gkrellm_make_decal_visible(p, ds);
		position = 0;
		}
	else
		{
		if (gkrellm_is_decal_visible(ds) || force)
			gkrellm_make_decal_invisible(p, ds);
		position = cpu->save_label_position;
		}
	if (   (mapped_sensors & SENSOR_FAN)
		&& (mapped_sensors & SENSOR_TEMPERATURE)
		&& sensor_separate_mode
	   )
		{
		if (! gkrellm_is_decal_visible(df) || force)
			gkrellm_make_decal_visible(p, df);
		position = -1;
		}
	else
		{
		if (gkrellm_is_decal_visible(df) || force)
			gkrellm_make_decal_invisible(p, df);
		}
	if (position != p->label->position || force)
		{
		if (cpu->save_label_position >= 0)	/* Reassign position only if the */
			p->label->position = position;	/* original label was visible.   */
		gkrellm_draw_panel_label(p, gkrellm_bg_panel_image(style_id));
		draw_sensor_decals(cpu);
		gkrellm_draw_panel_layers(p);
		return TRUE;
		}
	return FALSE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button == 3)
		gkrellm_open_config_window(mon_cpu);
	return TRUE;
	}

static GdkImlibImage	*nice_data_image,
						*nice_data_grid_image;
static GdkPixmap		*nice_data_pixmap,
						*nice_data_grid_pixmap;
static GdkColor			nice_color,
						nice_grid_color;
static gchar			*nice_color_string,
						*nice_grid_color_string;

static void
load_nice_data_images(void)
	{
	g_free(nice_color_string);
	g_free(nice_grid_color_string);
	nice_color_string = gkrellm_get_gkrellmrc_string("cpu_nice_color");
	nice_grid_color_string =
				gkrellm_get_gkrellmrc_string("cpu_nice_grid_color");
	map_color_string(nice_color_string, &nice_color);
	map_color_string(nice_grid_color_string, &nice_grid_color);
		
	if (nice_data_image)
		gdk_imlib_kill_image(nice_data_image);
	if (nice_data_grid_image)
		gdk_imlib_kill_image(nice_data_grid_image);
	nice_data_image = nice_data_grid_image = NULL;

	gkrellm_free_pixmap(&nice_data_pixmap);
	gkrellm_free_pixmap(&nice_data_grid_pixmap);

	gkrellm_load_image("nice", NULL, &nice_data_image, CPU_STYLE_NAME);
	gkrellm_load_image("nice_grid", NULL, &nice_data_grid_image,
					CPU_STYLE_NAME);
	}

static void
render_nice_data_pixmaps(void)
	{
	GList		*list;
	CpuMon		*cpu;
	gint		h_max;

	gkrellm_render_data_grid_pixmap(nice_data_grid_image,
				&nice_data_grid_pixmap, &nice_grid_color);

	h_max = 2;
	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;
		if (cpu->chart && (cpu->chart->h > h_max))
			h_max = cpu->chart->h;
		}
	gkrellm_render_data_pixmap(nice_data_image,
				&nice_data_pixmap, &nice_color, h_max);
	}

  /* I want cpu labels in upper case.
  */
static gchar	mapname[5] ;

static void
create_cpu(GtkWidget *vbox, gint first_create)
	{
	CpuMon		*cpu;
	Chart		*cp;
	Panel		*p;
	Style		*style;
	GList		*list;

	load_nice_data_images();
	ascent = 0;
	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;

		/* Default to all cpu charts visible.  Correct this as last step.
		*/
		if (first_create)
			{
			/* don't really need the cpu->vbox unless I start destroying...
			*/
			cpu->vbox = gtk_vbox_new(FALSE, 0);
			gtk_container_add(GTK_CONTAINER(vbox), cpu->vbox);
			gtk_widget_show(cpu->vbox);
			cpu->chart = gkrellm_chart_new0();
			cpu->chart->panel = gkrellm_panel_new0();
			cpu->enabled = TRUE;
			}
		cp = cpu->chart;
		p = cp->panel;

		style = gkrellm_panel_style(style_id);
		gkrellm_create_krell(p, gkrellm_krell_panel_image(style_id), style);

		gkrellm_chart_create(cpu->vbox, mon_cpu, cp, &cpu->chart_config);
		gkrellm_set_draw_chart_function(cp, refresh_cpu_chart, cpu);
		cpu->sys_cd = gkrellm_add_default_chartdata(cp, _("sys time"));
		cpu->user_cd = gkrellm_add_default_chartdata(cp, _("user time"));
		if (   (nice_data_image && nice_data_grid_image)
			|| (nice_color_string && nice_grid_color_string)
		   )
			{
			render_nice_data_pixmaps();
			cpu->nice_cd = gkrellm_add_chartdata(cp, &nice_data_pixmap,
						nice_data_grid_pixmap, _("nice time"));
			}
		else
			cpu->nice_cd = gkrellm_add_default_chartdata(cp, _("nice time"));
		gkrellm_set_chartdata_flags(cpu->nice_cd, CHARTDATA_ALLOW_HIDE);

		/* Since there is a constant Max value of the chart (CPU_TICKS/SEC)
		|  I control the grid resolution when fixed grids change.
		|  So don't call gkrellm_grid_resolution_adjustment() so there won't
		|  be a grid resolution part in the chart config.  Also, make sure
		|  auto grid res is off.
		*/
		gkrellm_set_chartconfig_auto_grid_resolution(cp->config, FALSE);
		gkrellm_chartconfig_fixed_grids_connect(cp->config,
									setup_cpu_scaling, NULL);
		setup_cpu_scaling(cp->config);

		if (! (!strcmp(cpu->name, "cpu") && cpu->is_composite))
			sensor_create_decals(p, style_id,
					&cpu->sensor_decal, &cpu->fan_decal);

		sprintf( mapname, _("CPU%c"), (cpu->name)[3] );
		gkrellm_panel_configure(p, mapname, style);
		gkrellm_panel_create(cpu->vbox, mon_cpu, p);

		cpu->save_label_position = p->label->position;
		if (cpu->sensor_decal)
			adjust_sensors_display(cpu, TRUE);

		gkrellm_alloc_chartdata(cp);
		enable_cpu_visibility(cpu);

		if (first_create)
			{
			gtk_signal_connect(GTK_OBJECT (cp->drawing_area), "expose_event",
					(GtkSignalFunc) cpu_expose_event, NULL);
			gtk_signal_connect(GTK_OBJECT (p->drawing_area), "expose_event",
					(GtkSignalFunc) cpu_expose_event, NULL);

			gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
					"button_press_event", (GtkSignalFunc) cb_cpu_extra, NULL);
			gtk_signal_connect(GTK_OBJECT(p->drawing_area),
					"button_press_event", (GtkSignalFunc) cb_panel_press,NULL);
			}
		else
			refresh_cpu_chart(cpu);
		gkrellm_setup_launcher(p, &cpu->launch, CHART_PANEL_TYPE, 4);
		}
	}


/* ------------------------------------------------------------------ */
#define	CPU_CONFIG_KEYWORD	"cpu"

static GtkWidget	*cpu_enabled_button,
					*text_format_combo;
static GtkWidget	*smp_button[3];
static GtkWidget	*omit_nice_button;
static GtkWidget	*sensor_separate_button;


static void
save_cpu_config(FILE *f)
	{
	GList	*list;
	CpuMon	*cpu;

	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;
		if (*(cpu->launch.command) != '\0')
			fprintf(f, "%s launch %s %s\n", CPU_CONFIG_KEYWORD,
						cpu->name, cpu->launch.command);
		if (*(cpu->launch.tooltip_comment) != '\0')
			fprintf(f, "%s tooltip_comment %s %s\n", CPU_CONFIG_KEYWORD,
						cpu->name, cpu->launch.tooltip_comment);
		fprintf(f, "%s extra_info %s %d\n", CPU_CONFIG_KEYWORD,
					cpu->name, cpu->extra_info);
		gkrellm_save_chartconfig(f, cpu->chart_config,
					CPU_CONFIG_KEYWORD, cpu->name);
		}
	fprintf(f, "%s enable %d\n", CPU_CONFIG_KEYWORD, cpu_enabled);
	fprintf(f, "%s smp_mode %d\n", CPU_CONFIG_KEYWORD, smp_mode);
	fprintf(f, "%s omit_nice_mode %d\n", CPU_CONFIG_KEYWORD, omit_nice_mode);
	fprintf(f, "%s sensor_mode %d\n", CPU_CONFIG_KEYWORD,
			sensor_separate_mode);
	fprintf(f, "%s text_format %s\n", CPU_CONFIG_KEYWORD, text_format);
	}

static void
load_cpu_config(gchar *arg)
	{
	GList	*list;
	CpuMon	*cpu;
	gchar	config[32], item[CFG_BUFSIZE],
			cpu_name[32], command[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (!strcmp(config, "enable"))
			sscanf(item, "%d", &cpu_enabled);
		else if (!strcmp(config, "smp_mode"))
			sscanf(item, "%d\n", &smp_mode);
		else if (!strcmp(config, "omit_nice_mode"))
			sscanf(item, "%d\n", &omit_nice_mode);
		else if (!strcmp(config, "sensor_mode"))
			sscanf(item, "%d\n", &sensor_separate_mode);
		else if (!strcmp(config, "text_format"))
			gkrellm_dup_string(&text_format, item);
		else if (!strcmp(config, GKRELLM_CHARTCONFIG_KEYWORD))
			{
			sscanf(item, "%31s %[^\n]", cpu_name, command);
			for (list = cpu_mon_list; list; list = list->next)
				{
				cpu = (CpuMon *) list->data;
				if (strcmp(cpu->name, cpu_name) == 0)
					gkrellm_load_chartconfig(&cpu->chart_config, command, 3);
				}
			}
		else if (!strcmp(config, "extra_info"))
			{
			sscanf(item, "%31s %[^\n]", cpu_name, command);
			for (list = cpu_mon_list; list; list = list->next)
				{
				cpu = (CpuMon *) list->data;
				if (strcmp(cpu->name, cpu_name) == 0)
					sscanf(command, "%d\n", &cpu->extra_info);
				}
			}
		else if (!strcmp(config, "launch"))
			{
			sscanf(item, "%31s %[^\n]", cpu_name, command);
			for (list = cpu_mon_list; list; list = list->next)
				{
				cpu = (CpuMon *) list->data;
				if (strcmp(cpu->name, cpu_name) == 0)
					cpu->launch.command = g_strdup(command);
				}
			}
		else if (!strcmp(config, "tooltip_comment"))
			{
			sscanf(item, "%31s %[^\n]", cpu_name, command);
			for (list = cpu_mon_list; list; list = list->next)
				{
				cpu = (CpuMon *) list->data;
				if (strcmp(cpu->name, cpu_name) == 0)
					cpu->launch.tooltip_comment = g_strdup(command);
				}
			}
		}
	}


static void
apply_cpu_config(void)
	{
	GList	*list;
	CpuMon	*cpu;
	gchar	*s;
	gint	i;

	s = gkrellm_entry_get_text(&(GTK_COMBO(text_format_combo)->entry));
	gkrellm_dup_string(&text_format, s);

	cpu_enabled = GTK_TOGGLE_BUTTON(cpu_enabled_button)->active;
	if (n_smp_cpus > 0)
		for (i = 0; i < 3; ++i)
			if (GTK_TOGGLE_BUTTON(smp_button[i])->active)
				smp_mode = i;
	omit_nice_mode = GTK_TOGGLE_BUTTON(omit_nice_button)->active;
	if (sensor_separate_button)
		sensor_separate_mode =
				GTK_TOGGLE_BUTTON(sensor_separate_button)->active;

	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;
		if (enable_cpu_visibility(cpu) && cpu->enabled)
			gkrellm_reset_and_draw_chart(cpu->chart);
		gkrellm_apply_launcher(&cpu->launch_entry, &cpu->tooltip_entry,
					cpu->chart->panel, &cpu->launch, gkrellm_launch_button_cb);
		if (adjust_sensors_display(cpu, FALSE) && cpu->launch.button)
			{
			gkrellm_destroy_button(cpu->launch.button);
			cpu->launch.button = 
				gkrellm_put_label_in_panel_button(cpu->chart->panel,
					gkrellm_launch_button_cb, &cpu->launch, cpu->launch.pad);
			}
		}
	}


#define	DEFAULT_TEXT_FORMAT	"$T"

static gchar	*cpu_info_text[] =
{
N_("<b>Chart Labels\n"),
N_("Substitution variables for the format string for chart labels:\n"),
N_("\t$L    the CPU label\n"),
N_("\t$T    total sys + user {+ nice} time percent usage\n"),
N_("\t$s    sys time percent usage\n"),
N_("\t$u    user time percent usage\n"),
N_("\t$n    nice time percent usage\n"),
};

static void
create_cpu_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*tabs;
	GtkWidget	*button;
	GtkWidget	*hbox, *vbox, *vbox1;
	GtkWidget	*text;
	GtkWidget	*table;
	GSList		*group;
	GList		*list;
	CpuMon		*cpu;
	gchar		buf[128];
	gint		i;


	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* -- Options tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Options"));
	gkrellm_check_button(vbox, &cpu_enabled_button, cpu_enabled,
			FALSE, 10, _("Enable CPU"));
	gkrellm_check_button(vbox, &omit_nice_button, omit_nice_mode, FALSE, 0,
		_("Exclude nice CPU time from krell even if nice is shown on chart"));
	if (gkrellm_sensors_available())
		gkrellm_check_button(vbox, &sensor_separate_button,
					sensor_separate_mode, FALSE, 0,
		_("Draw fan and temperature values separately (not alternating)."));
	if (n_smp_cpus > 0)
		{
		vbox1 = gkrellm_framed_vbox(vbox, _("SMP Charts Select"), 4, FALSE,
				0, 2);
		hbox = gtk_hbox_new (FALSE, 3);
		gtk_container_add(GTK_CONTAINER(vbox1), hbox);

		button = gtk_radio_button_new_with_label(NULL, _("Real CPUs."));
		gtk_box_pack_start(GTK_BOX (hbox), button, TRUE, TRUE, 0);
		smp_button[0] = button;
		group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));

		button = gtk_radio_button_new_with_label(group, _("Composite CPU."));
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
		group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
		smp_button[1] = button;

		button = gtk_radio_button_new_with_label(group, _("Composite and real"));
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
		smp_button[2] = button;

		button = smp_button[smp_mode];
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
		}

/* -- Setup tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));
	vbox1 = gkrellm_framed_vbox(vbox, _("Format String for Chart Labels"),
			 4, FALSE, 0, 2);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, TRUE, 5);
	text_format_combo = gtk_combo_new();
	gtk_widget_set_usize (GTK_WIDGET(text_format_combo), 350, 0);
	gtk_box_pack_start(GTK_BOX(hbox), text_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, DEFAULT_TEXT_FORMAT);
	list = g_list_append(list, _("\\fu \\.$u\\n\\fs \\.$s"));
	list = g_list_append(list, _("\\ww\\D2\\f\\au\\.$u\\D1\\f\\as\\.$s"));
	list = g_list_append(list, _("\\ww\\D3\\f\\au\\.$u\\D0\\f\\as\\.$s"));
	list = g_list_append(list,
		"\\ww\\C\\f$L\\D5\\f\\an\\.$n\\D2\\f\\au\\.$u\\D1\\f\\as\\.$s");
	gtk_combo_set_popdown_strings(GTK_COMBO(text_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(text_format_combo)->entry),
			text_format);
	
	vbox1 = gkrellm_framed_vbox_end(vbox, _("Launch Commands"), 4, TRUE, 0,2);
	vbox1 = gkrellm_scrolled_vbox(vbox1, NULL,
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	table = gkrellm_launcher_table_new(vbox1, n_cpus);
	for (i = 0, list = cpu_mon_list; list; list = list->next, ++i)
		{
		cpu = (CpuMon *) list->data;
		snprintf(buf, sizeof(buf), _("%s"), cpu->name);
		gkrellm_config_launcher(table, i,  &cpu->launch_entry,
				&cpu->tooltip_entry, buf, &cpu->launch);
		}

/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(cpu_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(cpu_info_text[i]));

	}

static Monitor	monitor_cpu =
	{
	N_("CPU"),				/* Name, for config tab.	*/
	MON_CPU,			/* Id,  0 if a plugin		*/
	create_cpu,			/* The create function		*/
	update_cpu,			/* The update function		*/
	create_cpu_tab,		/* The config tab create function	*/
	apply_cpu_config,	/* Apply the config function		*/

	save_cpu_config,	/* Save user conifg			*/
	load_cpu_config,	/* Load user config			*/
	CPU_CONFIG_KEYWORD,	/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_cpu_monitor(void)
	{
	GList	*list;

	monitor_cpu.name = _(monitor_cpu.name);
	style_id = gkrellm_add_chart_style(&monitor_cpu, CPU_STYLE_NAME);
	text_format = g_strdup(DEFAULT_TEXT_FORMAT);

	mon_cpu = &monitor_cpu;

	if (setup_cpu_interface())
		{
		for (list = cpu_mon_list; list; list = list->next)
			((CpuMon *)(list->data))->extra_info = TRUE;
		return &monitor_cpu;
		}
	return NULL;
	}

