/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* 12/22/2000 Patch from Dan Maas <dmaas@dcine.com>
|		Added code to support hostname qualified config files so a remote
|		machine with a shared home directory can have independent configs.
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include <unistd.h> /* needed for gethostname() */

#include	"pixmaps/frame_top.xpm"
#include	"pixmaps/frame_bottom.xpm"
#include	"pixmaps/frame_left.xpm"
#include	"pixmaps/frame_right.xpm"

#include	"pixmaps/button_panel_out.xpm"
#include	"pixmaps/button_panel_in.xpm"
#include	"pixmaps/button_meter_out.xpm"
#include	"pixmaps/button_meter_in.xpm"

#include	"pixmaps/bg_chart.xpm"
#include	"pixmaps/bg_grid.xpm"
#include	"pixmaps/bg_panel.xpm"
#include	"pixmaps/bg_separator.xpm"

#include	"pixmaps/bg_meter.xpm"

#include	"pixmaps/bg_slider_panel.xpm"
#include	"pixmaps/bg_slider_meter.xpm"

#include	"pixmaps/decal_alarm.xpm"
#include	"pixmaps/decal_warn.xpm"
//#include	"pixmaps/krell_alarm.xpm"
//#include	"pixmaps/krell_warn.xpm"

  /* These data images are used only for the default theme
  */
#include	"pixmaps/data_in.xpm"
#include	"pixmaps/data_in_grid.xpm"
#include	"pixmaps/data_out.xpm"
#include	"pixmaps/data_out_grid.xpm"

#include	"pixmaps/decal_misc.xpm"

#include	"pixmaps/krell_panel.xpm"
#include	"pixmaps/krell_meter.xpm"
#include	"pixmaps/krell_slider.xpm"
#include	"pixmaps/krell_mini.xpm"

  /* Theme images for builtin monitors.
  */
#include    "pixmaps/cal/bg_panel.xpm"
#include    "pixmaps/clock/bg_panel.xpm"
#include	"pixmaps/cpu/nice.xpm"
#include	"pixmaps/cpu/nice_grid.xpm"
#include    "pixmaps/fs/bg_panel.xpm"
#include    "pixmaps/fs/bg_panel_1.xpm"
#include    "pixmaps/fs/bg_panel_2.xpm"
#include	"pixmaps/fs/spacer_top.xpm"
#include	"pixmaps/fs/spacer_bottom.xpm"
#include	"pixmaps/host/bg_panel.xpm"
#ifdef BSD
#include	"pixmaps/mail/krell_mail_daemon.xpm"
#else
#include	"pixmaps/mail/krell_mail.xpm"
#endif
#include	"pixmaps/mem/bg_panel.xpm"
#include	"pixmaps/mem/krell.xpm"
#include    "pixmaps/sensors/bg_panel.xpm"
#include    "pixmaps/sensors/bg_panel_1.xpm"
#include    "pixmaps/sensors/bg_panel_2.xpm"
#include	"pixmaps/swap/bg_panel.xpm"
#include	"pixmaps/swap/krell.xpm"
#include	"pixmaps/uptime/bg_panel.xpm"
#include	"pixmaps/timer/bg_panel.xpm"

  /* Default theme images for various plugins
  */
#include	"pixmaps/gkrellmms/bg_scroll.xpm"
#include	"pixmaps/gkrellmms/bg_panel.xpm"
#include	"pixmaps/gkrellmms/bg_panel_1.xpm"
#include	"pixmaps/gkrellmms/bg_panel_2.xpm"
#include	"pixmaps/gkrellmms/spacer_top.xpm"
#include	"pixmaps/gkrellmms/spacer_bottom.xpm"

#include	"pixmaps/timers/bg_panel.xpm"
#include	"pixmaps/timers/bg_panel_1.xpm"
#include	"pixmaps/timers/bg_panel_2.xpm"
#include	"pixmaps/timers/spacer_top.xpm"
#include	"pixmaps/timers/spacer_bottom.xpm"

#include	"pixmaps/volume/bg_panel.xpm"
#include	"pixmaps/volume/bg_panel_1.xpm"
#include	"pixmaps/volume/bg_panel_2.xpm"
#include	"pixmaps/volume/spacer_top.xpm"
#include	"pixmaps/volume/spacer_bottom.xpm"

#include	"pixmaps/pmu/bg_panel.xpm"
#include	"pixmaps/pmu/bg_panel_1.xpm"
#include	"pixmaps/pmu/bg_panel_2.xpm"
#include	"pixmaps/pmu/spacer_top.xpm"
#include	"pixmaps/pmu/spacer_bottom.xpm"


static gchar	*image_type[] =
	{
	".png", ".jpg", ".xpm", ".gif"
	};

gchar *
gkrellm_theme_file_exists(char *name, gchar *subdir)
	{
	gint			i;
	static gchar	*path;
	struct stat		st;

	if (gkrellm_using_default_theme())
		return NULL;
	if (path)
		g_free(path);
	if (GK.theme_alternative > 0)
		{
		for (i = 0; i < sizeof(image_type) / sizeof(char *); ++i)
			{
			if (subdir)
				path = g_strdup_printf("%s/%s/%s_%d%s", GK.theme_path, subdir,
						name, GK.theme_alternative, image_type[i]);
			else
				path = g_strdup_printf("%s/%s_%d%s", GK.theme_path,
						name, GK.theme_alternative, image_type[i]);
			if (   stat(path, &st) == 0
				&& (S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))
			   )
				return path;
			}
		}
	for (i = 0; i < sizeof(image_type) / sizeof(char *); ++i)
		{
		if (subdir)
			path = g_strdup_printf("%s/%s/%s%s", GK.theme_path, subdir, name,
					image_type[i]);
		else
			path = g_strdup_printf("%s/%s%s", GK.theme_path, name,
					image_type[i]);
		if (   stat(path, &st) == 0
			&& (S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))
		   )
			return path;
		g_free(path);
		path = NULL;
		}
	return NULL;
	}

static void
set_border(GdkImlibBorder *border, char *string)
	{
	if (!border)
		return;
	border->left = 0;
	border->right = 0;
	border->top = 0;
	border->bottom = 0;
	if (string == NULL)
		return;
	sscanf(string, "%d,%d,%d,%d", &border->left, &border->right,
				&border->top, &border->bottom);
	}

static void
set_margins(Style *style, char *string)
	{
	Margin	*margins;

	GK.use_top_bottom_margins = TRUE;
	if (!style)
		return;
	margins = &style->margins;
	margins->left = 0;
	margins->right = 0;
	margins->top = 0;
	margins->bottom = 0;
	if (string == NULL)
		return;
	sscanf(string, "%d,%d,%d,%d", &margins->left, &margins->right,
				&margins->top, &margins->bottom);
	style->margin = margins->left;	/* Compatibility */
	}


static void
assign_font(Style *style, gchar *fontname, gint AorB)
	{
	TextStyle	*ts;

	ts = (AorB == SET_TEXTFONT_A) ? &style->label_tsA : &style->label_tsB;

	if (strcmp(fontname, "large_font") == 0)
		ts->font_seed = &GK.large_font;
	else if (strcmp(fontname, "normal_font") == 0)
		ts->font_seed = &GK.normal_font;
	else if (strcmp(fontname, "small_font") == 0)
		ts->font_seed = &GK.small_font;
	}

static void
assign_textcolor(Style *style, gchar *arg, gint AorB)
	{
	TextStyle	*ts;
	gchar		*values, *s;
	gchar		*color, *shadowcolor, *effect;

	values = g_strconcat(arg, NULL);

	color = cut_quoted_string(values, &s);
	shadowcolor = cut_quoted_string(s, &s);
	effect = cut_quoted_string(s, &s);
	if (*color == '\0' || *shadowcolor == '\0' || *effect == '\0')
		{
		printf(_("Bad textcolor line %s\n"), arg);
		g_free(values);
		return;
		}
	ts = (AorB == SET_TEXTCOLOR_A) ? &style->label_tsA : &style->label_tsB;
	map_color_string(color, &(ts->color));
	map_color_string(shadowcolor, &(ts->shadow_color));
	ts->effect = effect_string_value(effect);
	g_free(values);
	}

void
gkrellm_set_style_krell_values(Style *s, gint yoff, gint depth, gint x_hot,
				gint expand, gint left_margin, gint right_margin)
	{
	if (!s)
		return;
	s->krell_yoff = yoff;
	s->krell_depth = depth;
	s->krell_x_hot = x_hot;
	s->krell_expand = expand;
	s->krell_left_margin = left_margin;
	s->krell_right_margin = right_margin;
	}

void
gkrellm_set_krell_expand(Style *style, gchar *value)
	{
	gint	expand	= KRELL_EXPAND_NONE;

	if (!style)
		return;
	if (value)
		{
		if (!strcmp(value, "left"))
			expand = KRELL_EXPAND_LEFT;
		else if (!strcmp(value, "left-scaled"))
			expand = KRELL_EXPAND_LEFT_SCALED;
		else if (!strcmp(value, "right"))
			expand = KRELL_EXPAND_RIGHT;
		else if (!strcmp(value, "right-scaled"))
			expand = KRELL_EXPAND_RIGHT_SCALED;
		else if (!strcmp(value, "bar-mode"))
			expand = KRELL_EXPAND_BAR_MODE;
		else if (!strcmp(value, "bar-mode-scaled"))
			expand = KRELL_EXPAND_BAR_MODE_SCALED;
		}
	style->krell_expand = expand;
	}

static void
assign_style_entry(Style *style, gchar *value, gint entry_flag)
	{
	if (entry_flag == SET_KRELL_YOFF)
		style->krell_yoff = atoi(value);
	else if (entry_flag == SET_KRELL_EXPAND)
		gkrellm_set_krell_expand(style, value);
	else if (entry_flag == SET_KRELL_X_HOT)
		style->krell_x_hot = atoi(value);
	else if (entry_flag == SET_KRELL_EMA_PERIOD)
		style->krell_ema_period = atoi(value);
	else if (entry_flag == SET_KRELL_DEPTH)
		style->krell_depth = atoi(value);
	else if (entry_flag == SET_KRELL_LEFT_MARGIN)
		style->krell_left_margin = atoi(value);
	else if (entry_flag == SET_KRELL_RIGHT_MARGIN)
		style->krell_right_margin = atoi(value);
	else if (entry_flag == SET_LABEL_POSITION)
		{
		if (strcmp(value, "center") == 0)
			style->label_position = GKRELLM_LABEL_CENTER;
		else if (isdigit(*value))
			style->label_position = atoi(value);
		else
			style->label_position = GKRELLM_LABEL_NONE;
		}
	else if (entry_flag == SET_MARGIN)	/* Deprecated as of 1.2.9 */
		{
		style->margin = atoi(value);
		style->margins.left = style->margin;
		style->margins.right = style->margin;
		}
	else if (entry_flag == SET_TOP_MARGIN)
		{
		style->margins.top = atoi(value);
		GK.use_top_bottom_margins = TRUE;	/* Allow themes to adapt. */
		}
	else if (entry_flag == SET_BOTTOM_MARGIN)
		{
		style->margins.bottom = atoi(value);
		GK.use_top_bottom_margins = TRUE;	/* Allow themes to adapt. */
		}
	else if (entry_flag == SET_LEFT_MARGIN)
		style->margins.left = atoi(value);
	else if (entry_flag == SET_RIGHT_MARGIN)
		style->margins.right = atoi(value);
	else if (entry_flag == SET_TRANSPARENCY)
		style->transparency = atoi(value);
	else if (entry_flag == SET_TEXTCOLOR_A)
		assign_textcolor(style, value, SET_TEXTCOLOR_A);
	else if (entry_flag == SET_TEXTCOLOR_B)
		assign_textcolor(style, value, SET_TEXTCOLOR_B);
	else if (entry_flag == SET_TEXTFONT_A)
		assign_font(style, value, SET_TEXTFONT_A);
	else if (entry_flag == SET_TEXTFONT_B)
		assign_font(style, value, SET_TEXTFONT_B);
	else if (entry_flag == SET_BORDER)
		set_border(&style->border, value);
	else if (entry_flag == SET_MARGINS)
		set_margins(style, value);
	}


static void
assign_style(gchar *debug_name, GList *style_list, gint index,
				gchar *arg, gint entry_value, gint override)
	{
	Style	*style;
	GList	*list;

	style = (Style *) g_list_nth_data(style_list, index);
	if (!style)
		return;

	/* If this is not an override assignment and this entry has already had
	|  an override assignment, then we do not assign.
	*/
	if (! override && (style->override & BIT(entry_value)))
		return;
	if (override)
		style->override |= BIT(entry_value);
	assign_style_entry(style, arg, entry_value);
	if (index > 0)	/* Theme has custom setting for this style */
		style->themed = TRUE;
	if (index++ == 0)		/* style == style_list */
		{
		if (override)
			printf("Bad override on DEFAULT: %s %s %d\n",
					debug_name, arg, entry_value);
		for (list = style_list->next; list; list = list->next, ++index)
			{
			style = (Style *) list->data;
			if (style && !(style->override & BIT(entry_value)))
				assign_style_entry(style, arg, entry_value);
			}
		}
	}

#if 0
static void
assign_chart_style(gint index, gchar *arg, gint entry_value, gint override)
	{
	assign_style("StyleChart", GK.chart_style_list, index, arg,
				entry_value, override);
	}

static void
assign_panel_style(gint index, gchar *arg, gint entry_value, gint override)
	{
	assign_style("StylePanel", GK.panel_style_list, index, arg,
				entry_value, override);
	}
#endif

static void
assign_meter_style(gint index, gchar *arg, gint entry_value, gint override)
	{
	assign_style("StyleMeter", GK.meter_style_list, index, arg,
				entry_value, override);
	}

static void
assign_custom_style(gchar *debug_name, GList *style_list, gint index,
				gchar *arg, gint entry_value, gchar *custom_name)
	{
	Style	*style, *custom_style;
	gint	i;

	if ((i = string_position_in_list(GK.custom_name_list, custom_name)) < 0)
		{
		if ((style = (Style *) g_list_nth_data(style_list, index)) == NULL)
			return;
		custom_style = gkrellm_copy_style(style);
		custom_style->themed = TRUE;
		GK.custom_name_list = g_list_append(GK.custom_name_list,
				g_strdup(custom_name));
		GK.custom_style_list = g_list_append(GK.custom_style_list,
				custom_style);
		}
	else
		custom_style = (Style *) g_list_nth_data(GK.custom_style_list, i);

//printf("assign_custom_style(%s, %s, %d, %s) %d\n",
//debug_name, custom_name, entry_value, arg, i);
	assign_style_entry(custom_style, arg, entry_value);
	}

static struct string_map
	{
	gchar	*string;
	gint	value;
	}
	entry_map[] =
	{
	{ "krell_yoff",			SET_KRELL_YOFF },
	{ "krell_expand",		SET_KRELL_EXPAND },
	{ "krell_x_hot",		SET_KRELL_X_HOT },
	{ "krell_ema_period",	SET_KRELL_EMA_PERIOD },
	{ "krell_depth",		SET_KRELL_DEPTH },
	{ "krell_left_margin",	SET_KRELL_LEFT_MARGIN },
	{ "krell_right_margin",	SET_KRELL_RIGHT_MARGIN },
	{ "label_position",		SET_LABEL_POSITION },
	{ "margin",				SET_MARGIN },		/* Deprecated */
	{ "margins",			SET_MARGINS },
	{ "left_margin",		SET_LEFT_MARGIN },
	{ "right_margin",		SET_RIGHT_MARGIN },
	{ "top_margin",			SET_TOP_MARGIN },
	{ "bottom_margin",		SET_BOTTOM_MARGIN },
	{ "textcolor",			SET_TEXTCOLOR_A },
	{ "alt_textcolor",		SET_TEXTCOLOR_B },
	{ "font",				SET_TEXTFONT_A },
	{ "alt_font",			SET_TEXTFONT_B },
	{ "border",				SET_BORDER },
	{ "transparency",		SET_TRANSPARENCY },
	};


static gint
get_entry_value(gchar *entry)
	{
	struct string_map	*sm;

	for (sm = &entry_map[0];
		sm < &entry_map[sizeof(entry_map) / sizeof(struct string_map)]; ++sm)
		if (!strcmp(entry, sm->string))
			return sm->value;
	return -1;
	}

static void
assign_gkrellmrc_style(gchar *source_line, gchar *area, gchar *string)
	{
	GList	*style_list = NULL, *name_list = NULL;
	gchar	*s;
	gchar	*arg = NULL, *mon_name = NULL, *custom_name = NULL, *entry = NULL;
	gint	index, entry_value, override;

	/* string starts out in format "*.yyy arg" or "foo.yyy arg"
	*/
	mon_name = strtok(string, " \t=:");	/* "*.yyy" or "foo.yyy" */
	if (mon_name && (arg = strtok(NULL, "\n")) != NULL)	/* arg is "arg" part */
		{
		while (*arg == ' ' || *arg == '\t' || *arg == '=' || *arg == ':')
			++arg;
		entry = strrchr(mon_name, '.');
		if (entry)
			*entry++ = '\0';
		if ((s = strchr(mon_name, '.')) != NULL)
			{
			custom_name = g_strdup(mon_name);
			*s = '\0';
			}
		}
	if (!mon_name || !entry || !*entry || !arg)
		{
		printf("StyleXXX ?: %s\n", source_line);
		g_free(custom_name);
		return;
		}
	override = TRUE;
	entry_value = get_entry_value(entry);
	if (!strcmp(area, "StyleChart"))
		{
		name_list = GK.chart_name_list;
		style_list = GK.chart_style_list;
		}
	else if (!strcmp(area, "StylePanel"))
		{
		name_list = GK.chart_name_list;
		style_list = GK.panel_style_list;
		}
	else if (!strcmp(area, "StyleMeter"))
		{
		name_list = GK.meter_name_list;
		style_list = GK.meter_style_list;
		}
	else
		{
		printf("StyleXXX ?: %s\n", source_line);
		g_free(custom_name);
		return;
		}
	index = string_position_in_list(name_list, mon_name);
	if (index == DEFAULT_STYLE_ID)
		override = FALSE;

	if (entry_value >= 0 && index >= 0)
		{
		if (custom_name)
			assign_custom_style(area, style_list, index, arg, entry_value,
					custom_name);
		else
			assign_style(area, style_list, index, arg, entry_value, override);
		}
	g_free(custom_name);
	}

gint
gkrellm_add_chart_style(Monitor *mon, gchar *name)
	{
	Style		*panel_style, *chart_style;
	gint		id;
	static gint	style_id;

	if (!name)
		return 0;
	id = style_id++;
	chart_style = gkrellm_style_new0();
	panel_style = gkrellm_style_new0();
	if (mon)
		{
		if (mon->privat == NULL)
			mon->privat = g_new0(MonPrivate, 1);
		mon->privat->panel_style = panel_style;
		mon->privat->chart_style = chart_style;
		mon->privat->style_name = name;
		mon->privat->style_type = CHART_PANEL_TYPE;
		mon->privat->style_id = id;
		}
	GK.chart_name_list = g_list_append(GK.chart_name_list, (gchar *) name);
	GK.chart_style_list = g_list_append(GK.chart_style_list, chart_style);
	GK.panel_style_list = g_list_append(GK.panel_style_list, panel_style);
	GK.bg_chart_image_list = g_list_append(GK.bg_chart_image_list, NULL);
	GK.bg_grid_image_list = g_list_append(GK.bg_grid_image_list, NULL);
	GK.bg_panel_image_list = g_list_append(GK.bg_panel_image_list, NULL);
	GK.krell_panel_image_list = g_list_append(GK.krell_panel_image_list, NULL);
	return id;
	}

gint
gkrellm_add_meter_style(Monitor *mon, gchar *name)
	{
	Style		*style;
	gint		id;
	static gint	style_id;

	if (!name)
		return 0;
	id = style_id++;
	style = gkrellm_style_new0();
	if (mon)
		{
		if (mon->privat == NULL)
			mon->privat = g_new0(MonPrivate, 1);
		mon->privat->panel_style = style;
		mon->privat->style_name = name;
		mon->privat->style_type = METER_PANEL_TYPE;
		mon->privat->style_id = id;
		}
	GK.meter_name_list = g_list_append(GK.meter_name_list, (gchar *) name);
	GK.meter_style_list = g_list_append(GK.meter_style_list, style);
	GK.bg_meter_image_list = g_list_append(GK.bg_meter_image_list, NULL);
	GK.krell_meter_image_list = g_list_append(GK.krell_meter_image_list, NULL);
	return id;
	}


static void
set_image_borders_in_list(GList *st_list, GList *im_list, GList *nm_list)
	{
	Style			*style;
	GdkImlibImage	*image;

	for ( ; st_list && im_list && nm_list;
			st_list = st_list->next, im_list = im_list->next,
			nm_list = nm_list->next)
		{
		style = (Style *) st_list->data;
		image = (GdkImlibImage *) im_list->data;
		if (style && image)
			gdk_imlib_set_image_border(image, &style->border);
		}
	}

static void
setup_images(void)
	{
	gdk_imlib_set_image_border(GK.frame_top_image, &GK.frame_top_border);
	gdk_imlib_set_image_border(GK.frame_bottom_image, &GK.frame_bottom_border);
	if (GK.frame_left_width == 0)
		GK.frame_left_width = GK.frame_left_image->rgb_width;
	gdk_imlib_set_image_border(GK.frame_left_image, &GK.frame_left_border);
	if (GK.frame_right_width == 0)
		GK.frame_right_width = GK.frame_right_image->rgb_width;
	gdk_imlib_set_image_border(GK.frame_right_image, &GK.frame_right_border);

	gdk_imlib_set_image_border(GK.button_panel_out_image,
						&GK.button_panel_border);
	gdk_imlib_set_image_border(GK.button_panel_in_image,
						&GK.button_panel_border);

	gdk_imlib_set_image_border(GK.button_meter_out_image,
						&GK.button_meter_border);
	gdk_imlib_set_image_border(GK.button_meter_in_image,
						&GK.button_meter_border);

	set_image_borders_in_list(GK.chart_style_list, GK.bg_chart_image_list,
				GK.chart_name_list);
	set_image_borders_in_list(GK.panel_style_list, GK.bg_panel_image_list,
				GK.chart_name_list);
	set_image_borders_in_list(GK.meter_style_list, GK.bg_meter_image_list,
				GK.meter_name_list);

	gkrellm_render_to_pixmap(GK.decal_misc_image, &GK.decal_misc_pixmap,
					&GK.decal_misc_mask, 0, 0);
	}


typedef struct
	{
	gchar			*name;
	gchar			**xpm;
	GdkImlibImage	**im;
	GList			**image_list;
	gchar			*name_in_list;
	}
	ImageTable;

static ImageTable base_theme_images[]	=
	{
	/* Images in this table which have a non NULL _xpm default form the
	|  minimal set of required images for a complete theme change.
	|  If there is a NULL xpm, the image will be somehow constructed to
	|  a default image in the code.
	*/
{ "frame_top",		frame_top_xpm,		&GK.frame_top_image,	NULL, NULL},
{ "frame_bottom",	frame_bottom_xpm,	&GK.frame_bottom_image,	NULL, NULL},
{ "frame_left",  	frame_left_xpm,		&GK.frame_left_image,	NULL, NULL},
{ "frame_right", 	frame_right_xpm,	&GK.frame_right_image,	NULL, NULL},

{ "button_panel_out",	NULL,	&GK.button_panel_out_image,		NULL, NULL},
{ "button_panel_in",	NULL,	&GK.button_panel_in_image,		NULL, NULL},
{ "button_meter_out",	NULL,	&GK.button_meter_out_image,		NULL, NULL},
{ "button_meter_in",	NULL,	&GK.button_meter_in_image,		NULL, NULL},

{ "bg_chart",	 	bg_chart_xpm, NULL,	&GK.bg_chart_image_list,	"*"	},
{ "bg_grid", 		bg_grid_xpm, NULL, &GK.bg_grid_image_list,		"*"},
{ "bg_panel",		bg_panel_xpm, NULL, &GK.bg_panel_image_list,	"*" },
{ "bg_meter",		bg_meter_xpm, NULL, &GK.bg_meter_image_list, 	"*" },

{ "bg_slider_panel", NULL, &GK.bg_slider_image[CHART_PANEL_TYPE], NULL, NULL},
{ "bg_slider_meter", NULL, &GK.bg_slider_image[METER_PANEL_TYPE], NULL, NULL},

{ "decal_alarm",	decal_alarm_xpm,	&GK.decal_alarm_image, NULL, NULL},
{ "decal_warn",		decal_warn_xpm,		&GK.decal_warn_image, NULL,	NULL},

{ "decal_misc",		decal_misc_xpm,		&GK.decal_misc_image, NULL, NULL},

{ "data_in",	 	NULL,		&GK.data_in_image,			NULL, 	NULL},
{ "data_in_grid", 	NULL,		&GK.data_in_grid_image,		NULL,	NULL},
{ "data_out",	 	NULL,		&GK.data_out_image,			NULL,	NULL},
{ "data_out_grid", 	NULL,		&GK.data_out_grid_image,	NULL,	NULL},

{ "bg_separator", NULL,		&GK.bg_separator_image,			NULL,	NULL},

{ "krell_panel",	krell_panel_xpm, NULL, &GK.krell_panel_image_list, "*"},
{ "krell_meter",	krell_meter_xpm, NULL, &GK.krell_meter_image_list, "*"},
{ "krell_mail", krell_mail_xpm, NULL,
								&GK.krell_meter_image_list, MAIL_STYLE_NAME },

{ "krell_slider",	krell_slider_xpm,	&GK.krell_slider_image,	NULL,	NULL},
{ "krell_mini", 	krell_mini_xpm,		&GK.krell_mini_image,	NULL,	NULL}
	};


static ImageTable	default_theme_images[] =
	{
{ NULL, button_panel_out_xpm, &GK.button_panel_out_image,NULL, NULL},
{ NULL, button_panel_in_xpm, &GK.button_panel_in_image, NULL, NULL},
{ NULL, button_meter_out_xpm,&GK.button_meter_out_image, NULL, NULL},
{ NULL, button_meter_in_xpm, &GK.button_meter_in_image, NULL, NULL},

{ NULL,	bg_separator_xpm,	&GK.bg_separator_image, NULL, NULL},

{ NULL,	bg_panel_cal_xpm, NULL, &GK.bg_meter_image_list, CAL_STYLE_NAME},
{ NULL,	bg_panel_clock_xpm, NULL, &GK.bg_meter_image_list, CLOCK_STYLE_NAME},
{ NULL, bg_panel_mem_xpm, NULL, &GK.bg_meter_image_list, MEM_STYLE_NAME},
{ NULL,	bg_panel_host_xpm, NULL, &GK.bg_meter_image_list, HOST_STYLE_NAME},
{ NULL, bg_panel_uptime_xpm, NULL, &GK.bg_meter_image_list, UPTIME_STYLE_NAME},
{ NULL, bg_panel_timer_xpm, NULL, &GK.bg_meter_image_list, TIMER_STYLE_NAME},
{ NULL, bg_panel_swap_xpm, NULL, &GK.bg_meter_image_list, SWAP_STYLE_NAME},

{ NULL, bg_slider_panel_xpm, &GK.bg_slider_image[CHART_PANEL_TYPE], NULL,NULL},
{ NULL, bg_slider_meter_xpm, &GK.bg_slider_image[METER_PANEL_TYPE], NULL,NULL},

{ NULL,	data_in_xpm,		&GK.data_in_image,		NULL, NULL},
{ NULL,	data_in_grid_xpm,	&GK.data_in_grid_image,	NULL, NULL},
{ NULL,	data_out_xpm,		&GK.data_out_image,		NULL, NULL},
{ NULL,	data_out_grid_xpm,	&GK.data_out_grid_image, NULL, NULL},

{ NULL,	krell_mem_xpm, NULL, &GK.krell_meter_image_list, MEM_STYLE_NAME},
{ NULL,	krell_swap_xpm, NULL, &GK.krell_meter_image_list, SWAP_STYLE_NAME},
	};

static ImageTable	default_theme_alt0_images[] =
	{
{ NULL,	bg_panel_fs_xpm, NULL, &GK.bg_meter_image_list, FS_STYLE_NAME},
{ NULL,	bg_panel_sensors_xpm, NULL,&GK.bg_meter_image_list, "sensors" },

/* Plugins */
{ NULL,	bg_panel_timers_xpm,	NULL, &GK.bg_meter_image_list, "timers"},
{ NULL,	bg_panel_volume_xpm,	NULL, &GK.bg_meter_image_list, "volume"},
{ NULL,	bg_panel_gkrellmms_xpm,	NULL, &GK.bg_meter_image_list, "gkrellmms"},
{ NULL,	bg_panel_pmu_xpm,		NULL, &GK.bg_meter_image_list, "pmu"},
	};

static ImageTable	default_theme_alt1_images[] =
	{
{ NULL,	bg_panel_fs_1_xpm, NULL, &GK.bg_meter_image_list, FS_STYLE_NAME},
{ NULL,	bg_panel_sensors_1_xpm, NULL, &GK.bg_meter_image_list,
			"sensors"},

/* Plugins */
{ NULL,	bg_panel_timers_1_xpm,	NULL, &GK.bg_meter_image_list, "timers"},
{ NULL,	bg_panel_volume_1_xpm,	NULL, &GK.bg_meter_image_list, "volume"},
{ NULL,	bg_panel_gkrellmms_1_xpm, NULL, &GK.bg_meter_image_list, "gkrellmms"},
{ NULL,	bg_panel_pmu_1_xpm,		NULL, &GK.bg_meter_image_list, "pmu"},
	};

static ImageTable	default_theme_alt2_images[] =
	{
{ NULL,	bg_panel_fs_2_xpm, NULL, &GK.bg_meter_image_list, FS_STYLE_NAME},
{ NULL,	bg_panel_sensors_2_xpm, NULL, &GK.bg_meter_image_list,
			"sensors"},

/* Plugins */
{ NULL,	bg_panel_timers_2_xpm,	NULL, &GK.bg_meter_image_list, "timers"},
{ NULL,	bg_panel_volume_2_xpm,	NULL, &GK.bg_meter_image_list, "volume"},
{ NULL,	bg_panel_gkrellmms_2_xpm, NULL, &GK.bg_meter_image_list, "gkrellmms"},
{ NULL,	bg_panel_pmu_2_xpm,		NULL, &GK.bg_meter_image_list, "pmu"},
	};


  /* Need a trap to look for extra custom and extension images I've made for
  |  the default theme.
  */
static GdkImlibImage *
default_theme_extension_image(gchar *name, gchar *subdir)
	{
	GdkImlibImage	*im		= NULL;

	if (!strcmp(subdir, "gkrellmms") && !strcmp(name, "bg_scroll"))
		im = gdk_imlib_create_image_from_xpm_data(bg_scroll_gkrellmms_xpm);
	else if (!strcmp(subdir, CPU_STYLE_NAME) && !strcmp(name, "nice"))
		im = gdk_imlib_create_image_from_xpm_data(nice_xpm);
	else if (!strcmp(subdir, CPU_STYLE_NAME) && !strcmp(name,"nice_grid"))
		im = gdk_imlib_create_image_from_xpm_data(nice_grid_xpm);
	return im;
	}

gboolean
gkrellm_load_image(gchar *name, gchar **xpm, GdkImlibImage **image,
		gchar *subdir)
	{
	GdkImlibImage	*im		= NULL;
	gchar			*fname;

	if (gkrellm_using_default_theme() && name && subdir)
		im = default_theme_extension_image(name, subdir);
	else if (name && (fname = gkrellm_theme_file_exists(name, subdir)) != NULL)
		{
		name = fname;
		im = gdk_imlib_load_image(fname);
		if (im == NULL)
			printf(_("  Cannot load file image: %s\n"), fname);
		}
	if (im == NULL && xpm)
		{
		im = gdk_imlib_create_image_from_xpm_data(xpm);
		if (im == NULL)
			printf(_("  Cannot load xpm: %s\n"), name);
		}
	if (im && image)
		{
		if (*image)
			gdk_imlib_kill_image(*image);
		*image = im;
		}
	return (im ? TRUE : FALSE);
    }


static void
load_from_image_list(gchar *name, GList *image_list, gint index, gchar *subdir)
	{
	GList			*list;
	GdkImlibImage	**im;

	list = g_list_nth(image_list, index);
	if (list)
		{
		im = (GdkImlibImage **) (&list->data);
		gkrellm_load_image(name, NULL, im, subdir);
		}
	else
		printf("Bad index %d for image list (meter/panel problem?)\n", index);
	}

static void
load_monitor_specific_images(void)
	{
	Monitor	*mon;
	GList	*list;
	gchar	*subdir;
	gint	i;

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if ((subdir = mon->privat->style_name) == NULL)
			continue;
		i = mon->privat->style_id;
		if (mon->privat->style_type == CHART_PANEL_TYPE)
			{
			load_from_image_list("bg_chart", GK.bg_chart_image_list, i,subdir);
			load_from_image_list("bg_grid", GK.bg_grid_image_list, i, subdir);
			load_from_image_list("bg_panel", GK.bg_panel_image_list, i,subdir);
			load_from_image_list("krell", GK.krell_panel_image_list, i,subdir);
			}
		else
			{
			load_from_image_list("krell", GK.krell_meter_image_list, i,subdir);
			load_from_image_list("bg_panel", GK.bg_meter_image_list, i,subdir);
			load_from_image_list("bg_meter", GK.bg_meter_image_list, i,subdir);
			}
		gkrellm_load_image("spacer_top", NULL,
					&mon->privat->spacer_top_image, subdir);
		gkrellm_load_image("spacer_bottom", NULL,
					&mon->privat->spacer_bottom_image, subdir);
		}
	}

static void
assign_gkrellmrc_spacer_height(gchar *source_line, gchar *area, gchar *string)
	{
	Monitor	*mon;
	gchar	style_name[32];
	gint	h;

	if (sscanf(string, "%32s %d", style_name, &h) != 2)
		return;
	if ((mon = lookup_monitor_from_style_name(style_name)) == NULL)
		return;
	if (!strcmp(area, "spacer_top_height"))
		mon->privat->spacer_top_height = h;
	else if (!strcmp(area, "spacer_bottom_height"))
		mon->privat->spacer_bottom_height = h;
	}


  /* Yeoww! I have to do something about separate chart/meter lists.
  */
static GList *
lookup_image_from_name(GList *image_list, gchar *name)
	{
	GList	*n_list, *i_list;

	for (n_list = GK.chart_name_list, i_list = image_list;
			n_list && i_list; n_list = n_list->next, i_list = i_list->next)
		if (!strcmp(name, (gchar *) n_list->data))
			return i_list;
	for (n_list = GK.meter_name_list, i_list = image_list;
			n_list && i_list; n_list = n_list->next, i_list = i_list->next)
		if (!strcmp(name, (gchar *) n_list->data))
			return i_list;
	return NULL;
	}

static void
load_image_table(ImageTable *ti, gint n_images, gchar *subdir)
	{
	GdkImlibImage	**im;
	GList			*list;
	gint			i;

	for (i = 0; i < n_images; ++i, ++ti)
		{
		if (ti->image_list)
			{
/*			list = g_list_nth(*(ti->image_list), ti->list_index); */
			list = lookup_image_from_name(*(ti->image_list), ti->name_in_list);
			if (list)
				{
				im = (GdkImlibImage **) (&list->data);
				gkrellm_load_image(ti->name, ti->xpm, im, subdir);
				}
			}
		else
			gkrellm_load_image(ti->name, ti->xpm, ti->im, subdir);
		}
    }

  /* When loading a new theme, required base level images are not cleaned
  |  so the program will not crash if a theme does not have all images yet.
  |  It will just look funny.  But all optional base level images are cleaned
  |  so they will not carry over to the new theme.  There are no optional
  |  base level images in the image_lists.
  */
static void
clean_base_image_table(void)
	{
	ImageTable		*ti;
	gint			i;

	ti = &base_theme_images[0];
	for (i = 0; i < sizeof(base_theme_images) / sizeof(ImageTable); ++i, ++ti)
		if (ti->xpm == NULL && ti->im && *(ti->im))	/* Is an optional image */
			{
			gdk_imlib_kill_image(*(ti->im));
			*(ti->im) = NULL;
			}
	}

static void
kill_image(GdkImlibImage **im)
	{
	if (im && *im)
		{
		gdk_imlib_kill_image(*im);
		*im = NULL;
		}
	}

static void
kill_image_list(GList *list, GList *name_list, gchar *debug_name)
	{
	for ( ; list; list = list->next, name_list = name_list->next)
		kill_image((GdkImlibImage **) (&list->data));
	}

static void
kill_monitor_specific_images(void)
	{
	Monitor	*mon;
	GList	*list;

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		kill_image(&mon->privat->bg_chart_image);
		kill_image(&mon->privat->bg_panel_image);
		kill_image(&mon->privat->bg_grid_image);
		kill_image(&mon->privat->krell_image);
		kill_image(&mon->privat->spacer_top_image);
		kill_image(&mon->privat->spacer_bottom_image);	
		}
	}

void
load_theme_images(void)
	{
	Monitor	*mon;
	gint	n_base, n_default;

	/* Free up all custom images from old theme.
	*/
	kill_image_list(GK.bg_chart_image_list, GK.chart_name_list, "bg_chart");
	kill_image_list(GK.bg_grid_image_list, GK.chart_name_list, "bg_grid");
	kill_image_list(GK.bg_panel_image_list, GK.chart_name_list, "bg_panel");
	kill_image_list(GK.bg_meter_image_list, GK.meter_name_list, "bg_meter");
	kill_image_list(GK.krell_panel_image_list, GK.chart_name_list, "krell_panel");
	kill_image_list(GK.krell_meter_image_list, GK.meter_name_list, "krell_meter");
	kill_monitor_specific_images();

	clean_base_image_table();

	/* This loads the base images in the top level of the theme directory.
	|  For backward compatibility, it also loads monitor specific name
	|  qualified images in the top level directory.  The new way is for
	|  monitor specific images to be in subdirectories, loaded below.
	*/
	n_base = sizeof(base_theme_images) / sizeof(ImageTable);
	load_image_table(&base_theme_images[0], n_base, NULL);

	if (gkrellm_using_default_theme())
		{
		n_default = sizeof(default_theme_images) / sizeof(ImageTable);
		load_image_table(&default_theme_images[0], n_default, NULL);

		if (GK.theme_alternative == 2 || GK.theme_alternative == 5)
			{
			n_default = sizeof(default_theme_alt2_images) / sizeof(ImageTable);
			load_image_table(&default_theme_alt2_images[0], n_default, NULL);
			}
		else if (GK.theme_alternative == 1 || GK.theme_alternative == 4)
			{
			n_default = sizeof(default_theme_alt1_images) / sizeof(ImageTable);
			load_image_table(&default_theme_alt1_images[0], n_default, NULL);
			}
		else
			{
			n_default = sizeof(default_theme_alt0_images) / sizeof(ImageTable);
			load_image_table(&default_theme_alt0_images[0], n_default, NULL);
			}

		if ((mon = lookup_monitor_from_style_name("timers")) != NULL)
			{
			gkrellm_load_image(NULL, spacer_top_timers_xpm,
						&mon->privat->spacer_top_image, NULL);
			gkrellm_load_image(NULL, spacer_bottom_timers_xpm,
						&mon->privat->spacer_bottom_image, NULL);
			}
		if ((mon = lookup_monitor_from_style_name("volume")) != NULL)
			{
			gkrellm_load_image(NULL, spacer_top_volume_xpm,
						&mon->privat->spacer_top_image, NULL);
			gkrellm_load_image(NULL, spacer_bottom_volume_xpm,
						&mon->privat->spacer_bottom_image, NULL);
			}
		if ((mon = lookup_monitor_from_style_name("gkrellmms")) != NULL)
			{
			gkrellm_load_image(NULL, spacer_top_gkrellmms_xpm,
						&mon->privat->spacer_top_image, NULL);
			gkrellm_load_image(NULL, spacer_bottom_gkrellmms_xpm,
						&mon->privat->spacer_bottom_image, NULL);
			}
		if ((mon = lookup_monitor_from_style_name("pmu")) != NULL)
			{
			gkrellm_load_image(NULL, spacer_top_pmu_xpm,
						&mon->privat->spacer_top_image, NULL);
			gkrellm_load_image(NULL, spacer_bottom_pmu_xpm,
						&mon->privat->spacer_bottom_image, NULL);
			}
		if ((mon = lookup_monitor_from_style_name(FS_STYLE_NAME)) != NULL)
			{
			gkrellm_load_image(NULL, spacer_top_fs_xpm,
						&mon->privat->spacer_top_image, NULL);
			gkrellm_load_image(NULL, spacer_bottom_fs_xpm,
						&mon->privat->spacer_bottom_image, NULL);
			}
		}
	else
		{
		load_monitor_specific_images();
		}
	setup_images();
	}

  /* Borders for things that are not primary background parts of a monitor,
  |  and so are not set by a style line.
  */
static gchar
			*frame_top_border,
			*frame_bottom_border,
			*frame_left_border,
			*frame_right_border,
			*button_panel_border,
			*button_meter_border,
            *bg_slider_panel_border,
            *bg_slider_meter_border,
			*krell_slider_expand;

gint		krell_slider_depth,
			krell_slider_x_hot;

static struct	_config
	{
	gchar	*option;
	gint	*value;
	gchar	**arg;
	gint	minimum;
	}
	theme_config []	=
	{
	{"author",				NULL,		NULL,			-100 },

	{"theme_alternatives",	&GK.theme_n_alternatives,	NULL,		0  },

	{"frame_top_height",	&GK.frame_top_height,		NULL,		0  },
	{"frame_bottom_height",	&GK.frame_bottom_height,	NULL,		0  },
	{"frame_left_width",	&GK.frame_left_width,		NULL,		0  },
	{"frame_right_width",	&GK.frame_right_width,		NULL,		0  },
	{"chart_width_ref",		&GK.chart_width_ref,		NULL,		30 },
	{"chart_height_min",	&GK.chart_height_min,		NULL,		2 },
	{"chart_height_max",	&GK.chart_height_max,		NULL,		20 },
	{"bg_separator_height", &GK.bg_separator_height,	NULL,		0  },
	{"allow_scaling",		&GK.allow_scaling,			NULL,		0 },

	{"rx_led_x",			&GK.rx_led_x,				NULL,		-99 },
	{"rx_led_y",			&GK.rx_led_y,				NULL,		0   },
	{"tx_led_x",			&GK.tx_led_x,				NULL,		-99 },
	{"tx_led_y",			&GK.tx_led_y,				NULL,		0   },

	/* These two are handled as a service for mail.c because of historical
	|  reasons.  They should be set with set_integer in the gkrellmrc.
	*/
	{"decal_mail_frames",	&GK.decal_mail_frames,	NULL,			1  },
	{"decal_mail_delay",	&GK.decal_mail_delay,	NULL,			1  },

	{"decal_alarm_frames",	&GK.decal_alarm_frames,	NULL,			1  },
	{"decal_warn_frames",	&GK.decal_warn_frames,	NULL,			1  },

	{"large_font",			NULL,		&GK.large_font_string,		-100 },
	{"normal_font",			NULL,		&GK.normal_font_string,		-100 },
	{"small_font",			NULL,		&GK.small_font_string,		-100 },

	{"chart_in_color",		NULL,		&GK.chart_in_color,		-100 },
	{"chart_in_color_grid",	NULL,		&GK.chart_in_color_grid,	-100 },
	{"chart_out_color",		NULL,		&GK.chart_out_color,		-100 },
	{"chart_out_color_grid",NULL,		&GK.chart_out_color_grid,	-100 },

	{"bg_grid_mode",		&GK.bg_grid_mode,		NULL,			0  },

    {"frame_top_border",	NULL,		&frame_top_border,			-100 },
    {"frame_bottom_border",	NULL,		&frame_bottom_border,		-100 },
    {"frame_left_border",	NULL,		&frame_left_border,			-100 },
    {"frame_right_border",	NULL,		&frame_right_border,		-100 },
    {"button_panel_border", NULL,		&button_panel_border,		-100 },
    {"button_meter_border", NULL,		&button_meter_border,		-100 },

    {"bg_slider_panel_border", NULL,	&bg_slider_panel_border,	-100 },
    {"bg_slider_meter_border", NULL,	&bg_slider_meter_border,	-100 },

	{"krell_slider_depth",	&krell_slider_depth,	NULL,			1  },
	{"krell_slider_x_hot",	&krell_slider_x_hot,	NULL,			-1  },
	{"krell_slider_expand",	NULL, 		&krell_slider_expand,		-1  },
	};


  /* Handle borders set in gkrellmrc which are not set by a style line.
  */
static void
cleanup_gkrellmrc(void)
	{
	if (GK.font_override)
		{
		gkrellm_dup_string(&GK.large_font_string, get_large_font_override());
		gkrellm_dup_string(&GK.normal_font_string, get_normal_font_override());
		gkrellm_dup_string(&GK.small_font_string, get_small_font_override());
		}
	set_border(&GK.frame_top_border, frame_top_border);
	set_border(&GK.frame_bottom_border, frame_bottom_border);
	set_border(&GK.frame_left_border, frame_left_border);
	set_border(&GK.frame_right_border, frame_right_border);

	set_border(&GK.button_panel_border, button_panel_border);
	set_border(&GK.button_meter_border, button_meter_border);
	set_border(&GK.bg_slider_border[CHART_PANEL_TYPE], bg_slider_panel_border);
	set_border(&GK.bg_slider_border[METER_PANEL_TYPE], bg_slider_meter_border);

	GK.krell_slider_style->krell_x_hot  = krell_slider_x_hot;
	GK.krell_slider_style->krell_depth  = krell_slider_depth;
	gkrellm_set_krell_expand(GK.krell_slider_style, krell_slider_expand);
	}

static GList	*gkrellmrc_border_list,
				*gkrellmrc_integer_list,
				*gkrellmrc_string_list;


static GdkImlibBorder	zero_border;

gboolean
gkrellm_set_image_border(gchar *image_name, GdkImlibImage *image, Style *style)
	{
	static GdkImlibBorder	b;
	GList					*list;
	gchar					name[64], border_string[32];
	gchar					*s, *r;

	if (style)
		style->border = zero_border;
	if (!image || !image_name)
		return FALSE;
	for (list = gkrellmrc_border_list; list; list = list->next)
		{
		s = list->data;
		if ((r = strchr(s, '=')) != NULL)
			*r = ' ';
		sscanf(s, "%64s %32s", name, border_string);
		if (!strcmp(name, image_name))
			{
			set_border(&b, border_string);
			gdk_imlib_set_image_border(image, &b);
			if (style)
				style->border = b;
			return TRUE;
			}
		}
	return FALSE;
	}

gboolean
gkrellm_get_gkrellmrc_image_border(gchar *image_name, GdkImlibImage *image,
				GdkImlibBorder *border)
	{
	GdkImlibBorder	b;
	GList			*list;
	gchar			name[64], border_string[32];
	gchar			*s, *r;

	if (!image || !image_name)
		return FALSE;
	for (list = gkrellmrc_border_list; list; list = list->next)
		{
		s = list->data;
		if ((r = strchr(s, '=')) != NULL)
			*r = ' ';
		sscanf(s, "%64s %32s", name, border_string);
		if (!strcmp(name, image_name))
			{
			set_border(&b, border_string);
			gdk_imlib_set_image_border(image, &b);
			if (border)
				*border = b;
			return TRUE;
			}
		}
	return FALSE;
	}

gboolean
gkrellm_get_gkrellmrc_integer(gchar *int_name, gint *result)
	{
	GList		*list;
	gchar		name[64], string[64];
	gchar		*s, *r;
	gboolean	found = FALSE;

	if (!int_name || !result)
		return FALSE;
	for (list = gkrellmrc_integer_list; list; list = list->next)
		{
		s = list->data;
		if ((r = strchr(s, '=')) != NULL)
			*r = ' ';
		sscanf(s, "%64s %64s", name, string);
		if (!strcmp(name, int_name) && sscanf(string, "%d", result) == 1)
			found = TRUE;
		}
	return found;
	}

gchar *
gkrellm_get_gkrellmrc_string(gchar *string_name)
	{
	GList	*list;
	gchar	name[64], string[CFG_BUFSIZE];
	gchar	*s, *r;

	if (!string_name)
		return NULL;
	for (list = gkrellmrc_string_list; list; list = list->next)
		{
		s = list->data;
		if ((r = strchr(s, '=')) != NULL)
			*r = ' ';
		sscanf(s, "%64s %[^\n]", name, string);
		if (!strcmp(name, string_name))
			{
			if ((s = cut_quoted_string(string, NULL)) != NULL)
				return g_strdup(s);
			break;
			}
		}
	return NULL;
	}

static void
parse_config_line(gchar *line, struct _config *cf, gint size)
	{
	GList			*list;
	Monitor			*mon;
	struct _config	*pc;
	gchar			*s, *arg;
	gint			i, val;
	gchar			buf[CFG_BUFSIZE], pbuf[CFG_BUFSIZE];

	strncpy(buf, line, CFG_BUFSIZE);	/* strtok() is destructive */
	buf[CFG_BUFSIZE - 1] = '\0';
	s = strtok(buf, " \t:=\n");
	if (s == NULL || *s == '#' || *s == '\n' || *s == '\0')
		return;

	/* Special case diversions.
	*/
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (!mon->config_keyword || strcmp(mon->config_keyword, s))
			continue;
		arg = strtok(NULL, "\n");
		if (mon->load_user_config && mon->privat->enabled)
			(*(mon->load_user_config))(arg);
		return;
		}
	if (strncmp(s, "Style", 5) == 0)	/* StyleChart ... */
		{
		arg = strtok(NULL, "\n");
		assign_gkrellmrc_style(line, s, arg);
		return;
		}
	if (strncmp(s, "spacer_", 7) == 0)
		{
		arg = strtok(NULL, "\n");
		assign_gkrellmrc_spacer_height(line, s, arg);
		return;
		}
	if (strcmp(s, "set_image_border") == 0)
		{
		arg = strtok(NULL, "\n");
		gkrellmrc_border_list = g_list_append(gkrellmrc_border_list,
				g_strdup(arg));
		return;
		}
	if (strcmp(s, "set_integer") == 0)
		{
		arg = strtok(NULL, "\n");
		gkrellmrc_integer_list = g_list_append(gkrellmrc_integer_list,
				g_strdup(arg));
		return;
		}
	if (strcmp(s, "set_string") == 0)
		{
		arg = strtok(NULL, "\n");
		gkrellmrc_string_list = g_list_append(gkrellmrc_string_list,
				g_strdup(arg));
		return;
		}
	arg = strtok(NULL, " \t:=\n");

	/* Must allow '#' char for arguments - color setting.
	*/
	if (arg == NULL || *arg == '\n' || *arg == '\0')
		{
		printf(_("Incomplete config line:\n    %s\n"), line);
		return;
		}
	if (isdigit(*arg) || *arg == '-')
		val = atoi(arg);
	else
		val = 0;
	if (strcmp(arg, "off") == 0 || strcmp(arg, "false") == 0)
		val = 0;
	if (strcmp(arg, "on") == 0 || strcmp(arg, "true") == 0)
		val = 1;

	if (!cf)
		return;
	for (i = 0; i < size; ++i)
		{
		pc = cf + i;
		if (strcmp(pc->option, s) == 0)
			{
			if (pc->value)
				*(pc->value) = val;
			if (pc->arg)
				{
				/* String args can have quotes around them.
				*/
				if (*arg == '"')
					{
					strcpy(pbuf, arg + 1);
					arg = strtok(NULL, "#\n");	/* Get rest of line */
					if (arg)
						{
						strcat(pbuf, " ");
						strcat(pbuf, arg);
						}
					arg = strchr(pbuf, (int) '"');
					if (arg)
						*arg = '\0';
					gkrellm_dup_string(pc->arg, pbuf);
					}
				else
					gkrellm_dup_string(pc->arg, arg);
				}
			if (pc->minimum > -100 && val < pc->minimum)
				*(pc->value) = pc->minimum;
			break;
			}
		}
	}

static void
clear_style_list(GList *list, GList *name_list)
	{
	Style	*style;

	for ( ; list; list = list->next, name_list = name_list->next)
		{
		style = (Style *) list->data;
		if (style)
			memset(style, 0, sizeof(Style));
		}
	}

static gchar	*base_theme[]	=
	{
	"StyleChart *.border = 0,0,0,0",
	"StyleChart *.font = normal_font",
	"StyleChart *.alt_font = small_font",
	"StyleChart *.textcolor = #efd097 #000000 shadow",
	"StyleChart *.alt_textcolor = #c0c0c0 #181818 shadow",

	"StylePanel *.border = 0,0,2,1",
	"StylePanel *.font = normal_font",
	"StylePanel *.alt_font = normal_font",
	"StylePanel *.textcolor = white #000000 shadow",
	"StylePanel *.alt_textcolor = #d8e0c8 #000000 shadow",
	"StylePanel *.label_position = 50",
	"StylePanel *.margins = 1,1,2,1",
	"StylePanel *.krell_yoff = 0",
	"StylePanel *.krell_x_hot = 3",
	"StylePanel *.krell_depth = 4",
	"StylePanel *.krell_expand = 0",
	"StylePanel *.krell_ema_period = 3",

	"StyleMeter *.border = 3,3,3,2",
	"StyleMeter *.font = normal_font",
	"StyleMeter *.alt_font = small_font",
	"StyleMeter *.textcolor = #ffeac4 #000000 shadow",
	"StyleMeter *.alt_textcolor = wheat #000000 shadow",
	"StyleMeter *.label_position = 50",
	"StyleMeter *.margins = 2,2,2,2",
	"StyleMeter *.krell_yoff = 1",
	"StyleMeter *.krell_expand = 0",
	"StyleMeter *.krell_x_hot = -1",
	"StyleMeter *.krell_depth = 1",
	"StyleMeter *.krell_ema_period = 1",

	/* These have an override effect */
	"StyleMeter apm.bottom_margin = 2",
	"StyleMeter mail.krell_depth = 15",
	"StyleMeter mail.krell_yoff = 0",
	"StyleMeter mail.krell_x_hot = -1",
	"StyleMeter mail.krell_expand = 0",
	"StyleMeter mail.label_position = 70",
	"StyleChart net.alt_font = small_font",
	};


static gchar	*default_fonts[] =
	{
	"-b&h-lucida-bold-r-normal-*-*-120-*-*-*-*-iso8859-*",		/* large  */
	"-b&h-lucida-medium-r-normal-*-*-100-*-*-*-*-iso8859-*",		/* normal */
	"-adobe-helvetica-medium-r-normal-*-*-80-*-*-*-*-iso8859-*"	/* small  */
	};

static void
init_theme(void)
	{
	Monitor	*mon;
	GList	*list;
	Style	*style;
	gint	style_id;
	gint	i;

	GK.theme_n_alternatives = 0;
	GK.frame_top_height = 0;	/* use image height. */
	GK.frame_bottom_height = 0;	/* use image height. */
	GK.frame_left_width = 0;
	GK.frame_right_width = 0;
	GK.chart_height_min = 5;
	GK.chart_height_max = 200;
	GK.chart_width_ref = 60;
	GK.bg_separator_height = 2;
	GK.allow_scaling = TRUE;
	GK.bg_grid_mode = GRID_MODE_NORMAL;

	GK.decal_mail_frames = 18;
	GK.decal_mail_delay = 1;

	GK.decal_alarm_frames = 10;
	GK.decal_warn_frames = 10;

	GK.rx_led_x = 2;
	GK.rx_led_y = 6;
	GK.tx_led_x = -2;
	GK.tx_led_y = 6;

	gkrellm_dup_string(&GK.large_font_string, default_fonts[0]);
	gkrellm_dup_string(&GK.normal_font_string, default_fonts[1]);
	gkrellm_dup_string(&GK.small_font_string, default_fonts[2]);

	gkrellm_dup_string(&GK.chart_in_color, "#10d3d3");
	gkrellm_dup_string(&GK.chart_in_color_grid, "#00a3a3");
	gkrellm_dup_string(&GK.chart_out_color, "#f4ac4a");
	gkrellm_dup_string(&GK.chart_out_color_grid, "#b47c20");


	/* Setup the default styles.  Substyles may be set in gkrellmrc.  If
	|  they are not, then they will be set to point to the default after
	|  parsing the gkrellmrc file.
	*/
	clear_style_list(GK.chart_style_list, GK.chart_name_list);
	clear_style_list(GK.panel_style_list, GK.chart_name_list);
	clear_style_list(GK.meter_style_list, GK.meter_name_list);
	free_glist_and_data(&GK.custom_name_list);
	free_glist_and_data(&GK.custom_style_list);

	for (i = 0; i < sizeof(base_theme) / sizeof(gchar *); ++i)
		parse_config_line(base_theme[i], &theme_config[0],
					sizeof(theme_config) / sizeof (struct _config));

	/* Allow themes to transition to using top/bottom margins. */
	GK.use_top_bottom_margins = FALSE;

	/* I set some base theme parameters with no override.  The idea is if
	|  a theme does not bother to set anything, these will remain in effect,
	|  but if the theme sets any "*" settings, they can wipe these out.
	|  This is probably a mistake, I am contributing to theme author
	|  laziness and should move these to the default theme.
	*/
	style_id = gkrellm_lookup_meter_style_id(APM_STYLE_NAME);
	assign_meter_style(style_id, "3,3,2,2", SET_BORDER, 0);

	style_id = gkrellm_lookup_meter_style_id(CAL_STYLE_NAME);
	assign_meter_style(style_id,  "small_font", SET_TEXTFONT_A, 0);
	assign_meter_style(style_id,  "large_font", SET_TEXTFONT_B, 0);

	style_id = gkrellm_lookup_meter_style_id(CLOCK_STYLE_NAME);
	assign_meter_style(style_id,"large_font", SET_TEXTFONT_A, 0);

	style_id = gkrellm_lookup_meter_style_id(FS_STYLE_NAME);
	assign_meter_style(style_id, "0", SET_LABEL_POSITION, 0);

	if (GK.krell_slider_style)
		g_free(GK.krell_slider_style);
	GK.krell_slider_style = gkrellm_style_new0();
	style = (Style *) GK.meter_style_list->data;
	*GK.krell_slider_style = *style;

	if (GK.krell_mini_style)
		g_free(GK.krell_mini_style);
	GK.krell_mini_style = gkrellm_style_new0();
	*GK.krell_mini_style = *style;

	gkrellm_dup_string(&frame_top_border, "0,0,0,0");
	gkrellm_dup_string(&frame_bottom_border, "0,0,0,0");
	gkrellm_dup_string(&frame_left_border, "0,0,0,0");
	gkrellm_dup_string(&frame_right_border, "0,0,0,0");
	gkrellm_dup_string(&button_panel_border, "2,2,2,2");
	gkrellm_dup_string(&button_meter_border, "2,2,2,2");

	gkrellm_dup_string(&bg_slider_panel_border, "1,1,1,1");
	gkrellm_dup_string(&bg_slider_meter_border, "1,1,1,1");

	krell_slider_x_hot = -1;
	krell_slider_depth = 6;
	gkrellm_dup_string(&krell_slider_expand, "none");

	free_glist_and_data(&gkrellmrc_border_list);
	free_glist_and_data(&gkrellmrc_integer_list);
	free_glist_and_data(&gkrellmrc_string_list);
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		mon->privat->spacer_top_height = 0;
		mon->privat->spacer_bottom_height = 0;
		}
	}

static gchar	*default_theme[]	=
	{
	"frame_left_border = 0,0,42,16",
	"frame_right_border = 0,0,42,16",

	"StyleChart *.textcolor = #efd097 #000000 shadow",
	"StyleChart *.alt_textcolor = #a8e4e2 #000000 shadow",

	"StylePanel *.textcolor = white #000000 shadow",
	"StylePanel *.alt_textcolor = #60fff0 #000000 shadow",

	"StyleMeter *.border = 4,4,3,2",
	"StyleMeter *.margins = 2,2,2,2",
	"StyleMeter *.krell_yoff = 1",

	"StyleMeter *.textcolor = #c8e4e2 #000000 shadow",
	"StyleMeter *.alt_textcolor = #e8e4d2 #000000 shadow",

	"StyleMeter cal.textcolor = #c8e4e2 #000000 shadow",

	"StyleMeter clock.textcolor = #e8e4d2 #000000 shadow",
	"StyleMeter clock.alt_textcolor = #c8e4e2 #000000 shadow",

	"StyleMeter sensors.textcolor = #60fff0 #000000 shadow",
	"StyleMeter sensors.alt_textcolor = #c8e4e2 #000000 shadow",

	"StyleMeter fs.border = 4,4,3,1",
	"StyleMeter fs.bottom_margin = 0",
	"StyleMeter fs.label_position = 0",
	"StyleMeter fs.alt_font = normal_font",

	"StyleMeter fs.alt_textcolor = #c8e4e2 #000000 shadow",
	"StyleMeter mem.alt_textcolor = #c8e4e2 #000000 shadow",
	"StyleMeter swap.alt_textcolor = #c8e4e2 #000000 shadow",


	"StyleMeter host.textcolor = #c8d4d2 #000000 shadow",
	"StyleMeter host.top_margin = 1",
	"StyleMeter host.bottom_margin = 1",

	"StyleMeter mail.alt_textcolor = #ffc0c0 #000000 shadow",

	"StyleMeter mem.krell_yoff = 0",
	"StyleMeter mem.alt_font = normal_font",
	"StyleMeter mem.top_margin = 3",
	"StyleMeter swap.top_margin = 3",
	"StyleMeter swap.bottom_margin = 1",

	"StyleMeter swap.krell_yoff = -2",		/* Bottom justify */
	"StyleMeter swap.alt_font = normal_font",

	"StyleMeter sensors.alt_textcolor = #d8e0c8 #000000 shadow",
	"StyleMeter sensors.top_margin = 3",
	"StyleMeter sensors.bottom_margin = 3",
	"set_image_border sensors_bg_volt 1,1,1,1",

	"StyleMeter timer.textcolor = #e8e4d2 #000000 shadow",
	"StyleMeter timer.alt_textcolor = #c8e4e2 #000000 shadow",
	"StyleMeter timer.margins = 1,1,1,2",
	"set_image_border timer_bg_timer 1,1,2,2",

	"StyleMeter uptime.textcolor = #e8e4d2 #000000 shadow",
	"StyleMeter uptime.border = 3,3,2,2",
	"StyleMeter uptime.bottom_margin = 1",


	/* ---- plugins ---- */

	"spacer_top_height pmu 3",
	"spacer_bottom_height pmu 2",

	/* GKrellMMS scroll bar panel */
	"spacer_top_height gkrellmms 3",
	"spacer_bottom_height gkrellmms 3",
	"set_image_border gkrellmms_bg_scroll 3,3,2,2",
	"set_integer gkrellmms_scroll_margin 2",
	"set_integer gkrellmms_scroll_top_margin 2",
	"set_integer gkrellmms_scroll_bottom_margin 2",
	/* GKrellMMS time bar and button bar panels */
	"StyleMeter gkrellmms.alt_textcolor = black #dcdccc shadow",
	"StyleMeter gkrellmms.margins = 2,2,2,0",
	"StyleMeter gkrellmms.krell_yoff = 0",

	/* Timers panels */
	"spacer_top_height timers 3",
	"spacer_bottom_height timers 3",
	"StyleMeter timers.border = 6,6,2,2",
	"StyleMeter timers.font = large_font",
	"StyleMeter timers.textcolor = #d8e4d2 #000000 shadow",
	"StyleMeter timers.alt_textcolor = #c8e4e2 #000000 shadow",

	/* All volume panels */
	"spacer_top_height volume 3",
	"spacer_bottom_height volume 3",
	"StyleMeter volume.label_position = 0",
	"StyleMeter volume.border = 26,3,7,1",
	"StyleMeter volume.top_margin = 1",
	"StyleMeter volume.bottom_margin = 0",
	};

static gchar	*default_theme_1[]	=
	{
	"StyleChart *.textcolor #efd097 #000000 shadow",
	"StyleChart *.alt_textcolor #e4e4e2 #000000 shadow",
	"StylePanel *.textcolor white #000000 shadow",
	"StylePanel *.alt_textcolor #f0f080 #000000 shadow",
	"StyleMeter *.textcolor = #f2f4d8 #000000 shadow",
	"StyleMeter *.alt_textcolor #e8e4b2 #000000 shadow",
	"StyleMeter cal.textcolor #f2f4d8 #000000 shadow",
	"StyleMeter clock.textcolor #e8e4b2 #000000 shadow",
	"StyleMeter clock.alt_textcolor #f2f4d8 #000000 shadow",
	"StyleMeter fs.alt_textcolor = #f2f4d8 #000000 shadow",
	"StyleMeter host.textcolor #b8c4c2 #000000 shadow",
	"StyleMeter mail.alt_textcolor #e0c0c0 #000000 shadow",
	"StyleMeter mem.alt_textcolor = #f2f4d8 #000000 shadow",
	"StyleMeter swap.alt_textcolor = #f2f4d8 #000000 shadow",
	"StyleMeter sensors.textcolor = #f0f080 #000000 shadow",
	"StyleMeter sensors.alt_textcolor = #f2f4d8 #000000 shadow",
	"StyleMeter timer.textcolor #e8e4b2 #000000 shadow",
	"StyleMeter timer.alt_textcolor #f2f4d8 #000000 shadow",
	"StyleMeter uptime.textcolor #e8e4b2 #000000 shadow",
	"StyleMeter gkrellmms.alt_textcolor = black #dcdccc shadow",
	"StyleMeter timers.textcolor #d2d8c0 #000000 shadow",
	"StyleMeter timers.alt_textcolor = #f2f4d8 #000000 shadow",
	};

static gchar	*default_theme_2[]	=
	{
	"StyleChart *.textcolor #f8b080 #000000 shadow",
	"StyleChart *.alt_textcolor #f0e8f0 #000000 shadow",
	"StylePanel *.textcolor white #000000 shadow",
	"StylePanel *.alt_textcolor #f0d0f0 #000000 shadow",
	"StyleMeter *.textcolor = #f0e8f0 #000000 shadow",
	"StyleMeter *.alt_textcolor #f0c0a0 #000000 shadow",
	"StyleMeter cal.textcolor #f0e8f0 #000000 shadow",
	"StyleMeter clock.textcolor #f0c0a0 #000000 shadow",
	"StyleMeter clock.alt_textcolor #f0e8f0 #000000 shadow",
	"StyleMeter fs.alt_textcolor = #f0e8f0 #000000 shadow",
	"StyleMeter host.textcolor #b8c4c2 #000000 shadow",
	"StyleMeter mail.alt_textcolor #e0c0c0 #000000 shadow",
	"StyleMeter mem.alt_textcolor = #f0e8f0 #000000 shadow",
	"StyleMeter swap.alt_textcolor = #f0e8f0 #000000 shadow",
	"StyleMeter sensors.textcolor = #f0d0f0 #000000 shadow",
	"StyleMeter sensors.alt_textcolor = #f0e8f0 #000000 shadow",
	"StyleMeter timer.textcolor #f0c0a0 #000000 shadow",
	"StyleMeter timer.alt_textcolor #f0e8f0 #000000 shadow",
	"StyleMeter uptime.textcolor #f0c0a0 #000000 shadow",
	"StyleMeter gkrellmms.alt_textcolor = black #dcdccc shadow",
	"StyleMeter timers.textcolor #f0d0b0 #000000 shadow",
	"StyleMeter timers.alt_textcolor = #f0e8f0 #000000 shadow",
	};

static void
default_theme_config(void)
	{
	gint	i;

	GK.theme_n_alternatives = 5;
	if (GK.theme_alternative > GK.theme_n_alternatives)
		GK.theme_alternative = GK.theme_n_alternatives;

	for (i = 0; i < sizeof(default_theme) / sizeof(gchar *); ++i)
		parse_config_line(default_theme[i], &theme_config[0],
					sizeof(theme_config) / sizeof (struct _config));

	if (GK.theme_alternative == 1 || GK.theme_alternative == 4)
		{
		for (i = 0; i < sizeof(default_theme_1) / sizeof(gchar *); ++i)
			parse_config_line(default_theme_1[i], &theme_config[0],
						sizeof(theme_config) / sizeof (struct _config));
		}
	if (GK.theme_alternative == 2 || GK.theme_alternative == 5)
		{
		for (i = 0; i < sizeof(default_theme_2) / sizeof(gchar *); ++i)
			parse_config_line(default_theme_2[i], &theme_config[0],
						sizeof(theme_config) / sizeof (struct _config));
		}
	if (GK.theme_alternative > 2)
		parse_config_line("StyleChart *.transparency = 1", NULL, 0);
	}


static void
parse_gkrellmrc(gint alternative)
	{
	FILE	*f;
	gchar	*path, lbuf[16];
	gchar	buf[CFG_BUFSIZE];

	lbuf[0] = '\0';
	if (alternative > 0)
		snprintf(lbuf, sizeof(lbuf), "_%d", alternative);
	path = g_strdup_printf("%s/%s%s", GK.theme_path, GKRELLMRC, lbuf);
	if ((f = fopen(path, "r")) != NULL)
		{
		while (fgets(buf, sizeof(buf), f))
			parse_config_line(buf, &theme_config[0],
					sizeof(theme_config) / sizeof (struct _config));
		fclose(f);
		}
	g_free(path);
	}

gboolean
gkrellm_using_default_theme(void)
	{
	return (*(GK.theme_path) == '\0') ? TRUE : FALSE;
	}

void
gkrellmrc_theme_config(void)
	{
	init_theme();
	load_theme_config();

	if (gkrellm_using_default_theme())
		default_theme_config();
	else
		{
		parse_gkrellmrc(0);
		if (GK.theme_alternative > GK.theme_n_alternatives)
			GK.theme_alternative = GK.theme_n_alternatives;
		if (GK.theme_alternative > 0)
			parse_gkrellmrc(GK.theme_alternative);
		}
	cleanup_gkrellmrc();
	set_theme_alternatives_label();
	}


/* --------------------------------------------------------------*/

struct	_config	user_config[] =
	{
	{"enable_hostname",	&GK.enable_hostname,	NULL,				0  },
	{"hostname_short",	&GK.hostname_short,		NULL,				0  },
	{"enable_sysname",	&GK.enable_system_name,	NULL,				0  },

	{"track_gtk_rcfiles", &GK.track_gtk_rcfiles, NULL,				0  },
	{"large_info_font", &GK.large_info_font,	NULL,				0  },
	{"save_position",	&GK.save_position,		NULL,				0  },

	{"chart_width",		&GK.chart_width,			NULL,			0  },

	{"update_HZ",		&GK.update_HZ,			NULL,				0  },
	};


void
read_user_config(void)
	{
	FILE	*f;
	gchar	*config;
	gchar	buf[CFG_BUFSIZE];

	GK.enable_hostname = TRUE;
	GK.hostname_short = FALSE;
	GK.enable_system_name = FALSE;
	GK.chart_width = 60;
	GK.update_HZ = 10;

	config = gkrellm_make_config_file_name(gkrellm_homedir(),
					GKRELLM_USER_CONFIG);
	f = fopen(config, "r");
	g_free(config);
	if (f)
		{
		while (fgets(buf, sizeof(buf), f))
			parse_config_line(buf, &user_config[0],
						sizeof(user_config) / sizeof (struct _config));
		fclose(f);
		}
	if (GK.chart_width < CHART_WIDTH_MIN)
		GK.chart_width = CHART_WIDTH_MIN;
	if (GK.chart_width > CHART_WIDTH_MAX)
		GK.chart_width = CHART_WIDTH_MAX;
	}

void
gkrellm_config_modified(void)
	{
	if (GK.no_config)
		return;
	GK.config_modified = TRUE;
	}


gchar *
gkrellm_make_config_file_name(gchar *dir, gchar *config)
	{
	gchar	hostname[256],
			*fname,
			*s		= NULL;

	if (GK.config_suffix || GK.found_host_config || GK.force_host_config)
		{
		if (GK.config_suffix)
			s = GK.config_suffix;
		else if (!gethostname(hostname, 256))
			s = hostname;
		if (s)
			s = g_strdup_printf("%s-%s", config, s);
		}
	if (!s)
		s = g_strdup(config);

	if (dir)
		{
		fname = g_strdup_printf("%s/%s", dir, s);
		g_free(s);
		}
	else
		fname = s;
	return fname;
	}

void
save_user_config(void)
	{
	FILE	*f;
	GList	*list;
	Monitor	*mon;
	gint	i;
	mode_t	mode;
	gchar	*config;

	if (GK.demo)
		return;
	config = gkrellm_make_config_file_name(gkrellm_homedir(),
								GKRELLM_USER_CONFIG);
	f = fopen(config, "w");
	g_free(config);
	if (f == NULL)
		{
		printf(_("Cannot open config file %s for writing.\n"), config);
		return;
		}
	fprintf(f,
		"### GKrellM user config.  Auto written, do not edit (usually) ###\n");
	fprintf(f, "### Version %d.%d.%d ###\n",
			GKRELLM_VERSION_MAJOR, GKRELLM_VERSION_MINOR, GKRELLM_VERSION_REV);
	for (i = 0; i < sizeof(user_config) / sizeof(struct _config); ++i)
		{
		if (user_config[i].value)
			fprintf(f, "%s %d\n", user_config[i].option,
							*(user_config[i].value));
		else if (user_config[i].arg)	/* Put quotes around strings */
			fprintf(f, "%s \"%s\"\n",user_config[i].option,
						*(user_config[i].arg));
		}
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (mon->save_user_config && mon->privat->enabled)
			(*(mon->save_user_config))(f);
		}
#if defined (S_IRUSR)
	mode = (S_IRUSR | S_IWUSR);
#elif defined (S_IREAD)
	mode = (S_IREAD | S_IWRITE);
#else
	mode = 0600;
#endif
	fchmod(fileno(f), mode);
	fclose(f);

	GK.config_modified = 0;
	}

