/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|  1/23/2001 Hajimu UMEMOTO merged Free/Net/Open BSD code into one
|			read_bsd_net_data().
| 10/12/2000  NetBSD code contributed by Anthony Mallet <metall@ficus.yi.org>
|  7/30/2000  Patch from Yuuki NINOMIYA <gm@debian.or.jp>
|	          Added seconds to the timer button online time.
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO <ume@mahoroba.org>
|
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

typedef struct
	{
	gchar		*name;
	GtkWidget	*vbox;			/* Handle for hidding */
	GtkWidget	*spin_button,	/* For user config settings. */
				*enable_button,
				*force_button,
				*label_entry;
	Chart		*chart;

	Decal		*rxled,
				*txled;

	/* All known net interfaces are in the net_mon_list.  Only interfaces
	|  which are UP and are config enabled will actually have a monitor
	|  created, unless the interface is locked to the timer button.
	|  A locked interface always has a monitor created for it regardless
	|  of the config enabled state but it will have a chart that may not be
	|  visible if the interface is DOWN (ppp) or the connect state is
	|  hangup (ippp).
	*/
	gint		created;			/* True if monitor is created		*/
	gint		locked;				/* True if locked to timer button	*/
	gint		chart_is_visible;	/* True if chart is visible			*/
	gint		force_net_up;		/* Experimental feature				*/

	gint		config_temp;
	gint		extra_info;
	gchar		*label;
	gulong		oldrx,
				oldtx;
	gulong		cur;
	Launcher	launch;
	GtkWidget	*launch_entry,
				*tooltip_entry;

	/* (*(sync_net_interfaces))() needs to set this  */
	gint		state;
	gint		old_state;

	/* (*(read_net_data))() needs to fill in this data  */
	gint		rxtx_units;
	gulong		rx,
				tx;
	}
	NetMon;


#define	TIMER_TYPE_NONE		0
#define	TIMER_TYPE_PPP		1
#define	TIMER_TYPE_IPPP		2

typedef struct
	{
	gchar	*name;
	gint	type;
	}
	TimerType;

static GList	*net_mon_list;


static void	(*read_net_data)();
static void	(*sync_net_interfaces)();
static gint	(*is_ISDN_online)();

#define NET_DOWN	0
#define NET_UP		1

#define	NET_UNITS_PACKETS	0
#define	NET_UNITS_BYTES		1


/* ====== System dependent interface ====================================== */
#define	PPP_LOCK_FILE		"LCK..modem"


/* -------- FreeBSD / NetBSD / OpenBSD ------------------------------------ */
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/sysctl.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/route.h>

#define	LOCK_DIRECTORY		"/var/spool/lock"

static TimerType	timer_defaults[] =
	{
	{"tun0", TIMER_TYPE_PPP },
	{"ppp0", TIMER_TYPE_PPP }
	};

static int	mib[] = { CTL_NET, PF_ROUTE, 0, 0, NET_RT_IFLIST, 0 };
static char	*buf;
static int	alloc;

void
read_bsd_net_data()
	{
	GList			*list;
	NetMon			*net;
	struct if_msghdr	*ifm, *nextifm;
	struct sockaddr_dl	*sdl;
	char			*lim, *next;
	size_t			needed;
	gchar			s[32];

	if (sysctl(mib, 6, NULL, &needed, NULL, 0) < 0)
		return;
	if (alloc < needed)
		{
		if (buf != NULL)
			free(buf);
		buf = malloc(needed);
		if (buf == NULL)
			return;
		alloc = needed;
		}

	if (sysctl(mib, 6, buf, &needed, NULL, 0) < 0)
		return;
	lim = buf + needed;

	next = buf;
	while (next < lim)
		{
		ifm = (struct if_msghdr *)next;
		if (ifm->ifm_type != RTM_IFINFO)
			return;
		next += ifm->ifm_msglen;

		while (next < lim)
			{
			nextifm = (struct if_msghdr *)next;
			if (nextifm->ifm_type != RTM_NEWADDR)
				break;
			next += nextifm->ifm_msglen;
			}

		if (ifm->ifm_flags & IFF_UP)
			{
			sdl = (struct sockaddr_dl *)(ifm + 1);
			if (sdl->sdl_family != AF_LINK)
				continue;
			strncpy(s, sdl->sdl_data, sdl->sdl_nlen);
			s[sdl->sdl_nlen] = '\0';

			for (list = net_mon_list; list; list = list->next)
				{
				net = (NetMon *) list->data;
				if (strcmp(net->name, s) == 0)
					{
					net->rx = ifm->ifm_data.ifi_ibytes;
					net->tx = ifm->ifm_data.ifi_obytes;
					net->rxtx_units = NET_UNITS_BYTES;
					break;
					}
				}
			}
		}
	}

static void
sync_bsd_net_interfaces()
	{
	GList			*list;
	NetMon			*net;
	struct if_msghdr	*ifm, *nextifm;
	struct sockaddr_dl	*sdl;
	char			*lim, *next;
	size_t			needed;
	gchar			s[32];

	for (list = net_mon_list; list; list = list->next)
		{
		net = (NetMon *) list->data;
		net->old_state = net->state;
		net->state = NET_DOWN;
		}

	if (sysctl(mib, 6, NULL, &needed, NULL, 0) < 0)
		return;
	if (alloc < needed)
		{
		if (buf != NULL)
			free(buf);
		buf = malloc(needed);
		if (buf == NULL)
			return;
		alloc = needed;
		}

	if (sysctl(mib, 6, buf, &needed, NULL, 0) < 0)
		return;
	lim = buf + needed;

	next = buf;
	while (next < lim)
		{
		ifm = (struct if_msghdr *)next;
		if (ifm->ifm_type != RTM_IFINFO)
			return;

		next += ifm->ifm_msglen;

		while (next < lim)
			{
			nextifm = (struct if_msghdr *)next;
			if (nextifm->ifm_type != RTM_NEWADDR)
				break;
			next += nextifm->ifm_msglen;
			}

		if (ifm->ifm_flags & IFF_UP)
			{
			sdl = (struct sockaddr_dl *)(ifm + 1);
			if (sdl->sdl_family != AF_LINK)
				continue;
			strncpy(s, sdl->sdl_data, sdl->sdl_nlen);
			s[sdl->sdl_nlen] = '\0';
#if 0
			/* Should we skip lo0 and dummy0?  Actually,
                        |  those who don't want to see it can disable
                        |  it from config panel.
			*/
			if (!strncmp(s, "lo", 2) || !strncmp(s, "dummy", 5))
				continue;
#endif
			for (list = net_mon_list; list; list = list->next)
				{
				net = (NetMon *) list->data;
				if (strcmp(net->name, s) == 0)
					{
					net->state = NET_UP;
					break;
					}
				}
			if (list == NULL)
				{
				net = g_new0(NetMon, 1);
				net->name = g_strdup(s);
				net_mon_list = g_list_append(net_mon_list, net);
				net->state = NET_UP;
				net->old_state = NET_DOWN;
				}
			}
		}
	}
#endif


/* -------- Linux --------------------------------------------------------- */
#if defined(__linux__)

/* See linux/net/ipv4/route.c (or ipv6) for /proc/net/route code */
/* See linux/net/core/dev.c for /proc/net/dev code */

#define	PROC_NET_DEV_FILE	"/proc/net/dev"
#define	PROC_NET_ROUTE_FILE	"/proc/net/route"
#define	LOCK_DIRECTORY		"/var/lock"

static TimerType	timer_defaults[] =
	{
	{"ppp0", TIMER_TYPE_PPP },
	{"ippp0", TIMER_TYPE_IPPP }
	};

static gint			rx_bytes_index,
					tx_bytes_index,
					rx_packets_index,
					tx_packets_index;


  /* I read both the bytes (kernel 2.2.x) and packets (all kernels).  Some
  |  net drivers for 2.2.x do not update the bytes counters.
  */
static void
read_linux_net_data()
	{
	FILE	*f;
	GList	*list;
	NetMon	*net;
	gchar	buf[512];
	gchar	*s, *s1;
	gint	i;
	gulong	rx_packets	= 0,
			tx_packets	= 0;

	if ((f = fopen(PROC_NET_DEV_FILE, "r")) == NULL)
		return;
	fgets(buf, sizeof(buf), f);		/* Waste first 2 lines */
	fgets(buf, sizeof(buf), f);
	while (fgets(buf, sizeof(buf), f))
		{
		/* Virtual net interfaces have a colon in the name, and a colon seps
		|  the name from data, + there might be no space between data and name!
		|  Eg. this is possible -> eth2:0:11249029    0 ...
		|  So, replace the colon that seps data from the name with a space.
		*/
		s = strchr(buf, (int) ':');
		if (s)
			{
			s1 = strchr(s + 1, (int) ':');
			if (s1)
				*s1 = ' ';
			else
				*s = ' ';
			}
		if ((s = strtok(buf, " \t\n")) == NULL)	/* Get name of interface */
			{
			fclose(f);
			return;
			}
		for (list = net_mon_list; list; list = list->next)
			{
			net = (NetMon *) list->data;
			if (strcmp(net->name, s))
				continue;
			net->rx = net->tx = 0;
			for (i = 1; s; ++i)
				{
				if ((s = strtok(NULL, " \t\n")) == NULL)
					break;
				if (i == rx_bytes_index)
					net->rx = strtoul(s, NULL, 0);
				else if (i == tx_bytes_index)
					net->tx = strtoul(s, NULL, 0);
				else if (i == rx_packets_index)
					rx_packets = strtoul(s, NULL, 0);
				else if (i == tx_packets_index)
					tx_packets = strtoul(s, NULL, 0);
				if (i > tx_bytes_index && i > tx_packets_index)
					break;
				}
			if (net->rx == 0 && net->tx == 0)
				{
				net->rxtx_units = NET_UNITS_PACKETS;
				net->rx = rx_packets;
				net->tx = tx_packets;
				}
			else
				net->rxtx_units = NET_UNITS_BYTES;
			break;
			}
		}
	fclose(f);
	}

static void
sync_linux_net_interfaces()
	{
	FILE		*f;
	GList		*list;
	NetMon		*net;
	gchar		*s;
	gchar		buf[512];

	for (list = net_mon_list; list; list = list->next)
		{
		net = (NetMon *) list->data;
		net->old_state = net->state;
		net->state = NET_DOWN;
		}
	if ((f = fopen(PROC_NET_ROUTE_FILE, "r")) != NULL)
		{
		fgets(buf, sizeof(buf), f);		/* Waste the first line */
		while (fgets(buf, sizeof(buf), f))
			{
			if (   ((s = strtok(buf, " \t\n")) == NULL)
				|| strncmp(s, "lo", 2) == 0
				|| strncmp(s, "dummy", 5) == 0
				|| (*s == '*' && *(s+1) == '\0')
			   )
				continue;
			for (list = net_mon_list; list; list = list->next)
				{
				net = (NetMon *) list->data;
				if (strcmp(net->name, s) == 0)
					{
					net->state = NET_UP;
					break;
					}
				}
			if (list == NULL)
				{
				net = g_new0(NetMon, 1);
				net->name = g_strdup(s);
				net_mon_list = g_list_append(net_mon_list, net);
				net->state = NET_UP;
				net->old_state = NET_DOWN;
				}
			}
		fclose(f);
		}
	}


static gint
is_linux_ISDN_online()
	{
	FILE	*f = 0;
	char	buffer[512], *p, *end;
	int		i;

	if (   (f = fopen ("/dev/isdninfo", "r")) == NULL
		&& (f = fopen("/dev/isdn/isdninfo", "r")) == NULL
	   )
		{
		if (GK.debug_level & DEBUG_NET)
			printf("is_linux_ISDN_online: no /dev/isdninfo?\n");
		return FALSE;
		}
	 for (i = 0; i < 5; i++)
		{
		if (fgets (buffer, BUFSIZ, f) == NULL)
			{
			fclose (f);
			return FALSE;
			}
		}
	fclose (f);
	if (strncmp (buffer, "flags:", 6))
		return FALSE;

	p = buffer+6;
	while (*p)
		{
		if (isspace (*p))
			{
			p++;
			continue;
			}
		for (end = p; *end && !isspace (*end); end++)
			;
		if (*end == '\0')
			break;
		else
			*end = 0;
		if (!strcmp (p, "?") || !strcmp (p, "0"))
			{
			p = end+1;
			continue;
			}
		return TRUE;	/* ISDN is online */
		}
	return FALSE;	/* ISND is off line */
	}

static const char	*delim	= " :|\t\n";

static void
get_io_indices()
	{
	FILE	*f;
	gchar	*s;
	gchar	buf[184];
	gint	i;

	if ((f = fopen(PROC_NET_DEV_FILE, "r")))
		{
		fgets(buf, sizeof(buf), f);		/* Waste the first line.	*/
		fgets(buf, sizeof(buf), f);		/* Look for "units" in this line */
		s = strtok(buf, delim);
		for (i = 0; s; ++i)
			{
			if (strcmp(s, "bytes") == 0)
				{
				if (rx_bytes_index == 0)
					rx_bytes_index = i;
				else
					tx_bytes_index = i;
				}
			if (strcmp(s, "packets") == 0)
				{
				if (rx_packets_index == 0)
					rx_packets_index = i;
				else
					tx_packets_index = i;
				}
			s = strtok(NULL, delim);
			}
		fclose(f);
		}
	if (GK.debug_level & DEBUG_NET)
		printf(_("rx_bytes=%d tx_bytes=%d rx_packets=%d tx_packets=%d\n"),
			rx_bytes_index, tx_bytes_index, rx_packets_index, tx_packets_index);
	}
#endif	/* __linux__ */


/* ----- Others ------------------------------------------------------- */
#if defined(USE_LIBGTOP)

#include <glibtop/netload.h>
#include <glibtop/ppp.h>

#define	LOCK_DIRECTORY		"/var/lock"

static TimerType	timer_defaults[] =
	{
	{"ppp0", TIMER_TYPE_PPP },
	{"ippp0", TIMER_TYPE_IPPP }
	};


static void
read_glibtop_net_data()
	{
	glibtop_netload	netload;
	GList			*list;
	NetMon			*net;
	gulong			rx_packets, tx_packets;

	for (list = net_mon_list; list; list = list->next)
		{
		net = (NetMon *) list->data;
		glibtop_get_netload (&netload, net->name);
		rx_packets = (gulong) netload.packets_in;
		tx_packets = (gulong) netload.packets_out;
		net->rx = (gulong) netload.bytes_in;
		net->tx = (gulong) netload.bytes_out;
		if (net->rx == 0 && net->tx == 0)
			{
			net->rxtx_units = NET_UNITS_PACKETS;
			net->rx = rx_packets;
			net->tx = tx_packets;
			}
		else
			net->rxtx_units = NET_UNITS_BYTES;
		}
	}

  /* Make an example list.  GKrellM expects to be able to automatically
  |  detect any UP net interface.
  */
static gchar	*glt_net_names[] =
	{ "eth0", "ppp0", "eth1", "eth2", "ippp0", "plip0" };

#define	GLT_IF_UP(flags)	(flags & (1 << GLIBTOP_IF_FLAGS_UP))

static void
sync_glibtop_net_interfaces()
	{
	glibtop_netload	netload;
	GList			*list;
	NetMon			*net;
	gint			i;

	for (list = net_mon_list; list; list = list->next)
		{
		net = (NetMon *) list->data;
		net->old_state = net->state;
		net->state = NET_DOWN;
		}
	for (i = 0; i < sizeof (glt_net_names) / sizeof (gchar *); ++i)
		{
		glibtop_get_netload (&netload, glt_net_names[i]);
		for (list = net_mon_list; list; list = list->next)
			{
			net = (NetMon *) list->data;
			if (strcmp(net->name, glt_net_names[i]) == 0)
				{
				if (GLT_IF_UP(netload.if_flags))
					net->state = NET_UP;
				break;
				}
			}
		if (list == NULL && GLT_IF_UP(netload.if_flags))
			{
			net = g_new0(NetMon, 1);
			net->name = g_strdup(glt_net_names[i]);
			net_mon_list = g_list_append(net_mon_list, net);
			net->state = NET_UP;
			net->old_state = NET_DOWN;
			}
		}
	}

static gint
is_glibtop_ISDN_online()
	{
	glibtop_ppp	isdn;

	glibtop_get_ppp(&isdn, 0  /* Reads /dev/isdninfo */);
	if (isdn.state == GLIBTOP_PPP_STATE_ONLINE)
		return TRUE;
	return FALSE;
	}
#endif

/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_net_interface()
    {
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
	read_net_data = read_bsd_net_data;
	sync_net_interfaces = sync_bsd_net_interfaces;
	is_ISDN_online = NULL;
#elif defined(__linux__)
	get_io_indices();
	read_net_data = read_linux_net_data;
	sync_net_interfaces = sync_linux_net_interfaces;
	is_ISDN_online = is_linux_ISDN_online;
#else
	read_net_data = read_glibtop_net_data;
	sync_net_interfaces = sync_glibtop_net_interfaces;
	is_ISDN_online = is_glibtop_ISDN_online;
#endif
    return TRUE;
    }

/* ======================================================================== */

  /* States for the timer button are indexes to the corresponding
  |  timer button decal frame shown.
  */
#define	TB_NORMAL		0
#define	TB_PRESSED		1
#define	TB_STANDBY		2
#define	TB_ON			3
#define	N_TB_DECALS		4

#define	RX_LED	0
#define	TX_LED	1

#define	RX_OFF	0
#define	RX_ON	1
#define	TX_OFF	2
#define	TX_ON	3
#define	N_LEDS	4



typedef struct
	{
	gchar	*name;
	gint	resolution;
	gshort	enabled;
	gint	extra_info;
	gint	force_net_up;
	gchar	*label;
	gchar	*command;
	gchar	*tooltip;
	}
	NetConfig;

static GList	*net_config_list;


static NetMon		*net_timed;		/* Monitor linked to timer button  */

Panel				*timer_panel;		/* For the timer and button	*/

static GtkWidget	*net_vbox;		/* Where all net monitors live */
static GtkWidget	*dynamic_net_vbox;
static GtkWidget	*timer_vbox;

static GdkImlibImage *bg_timer_image;
static GdkPixmap	*decal_net_led_pixmap;
static GdkBitmap	*decal_net_led_mask;

static GdkPixmap	*decal_timer_button_pixmap;
static GdkBitmap	*decal_timer_button_mask;

static Decal		*time_decal,
					*seconds_decal,
					*button_decal;
static gchar		*timer_on_command;
static gchar		*timer_off_command;
static gchar		*timer_button_iface;
static gint			timer_button_enabled;
static gint			last_time = -1;


static gint			timer_button_type,
					timer_button_state,
					timer_button_old_state,
					last_timer_command;
static gint			timer_seconds,
					x_off_minutes;

static gboolean		wide_timer;

static gint			check_connect_state;

static gint			ascent,
					ascent_alt;

static time_t		net_timer0;

static gint			net_style_id,
					timer_style_id;

  /* A feature accessible only by editing the user_config.  Some have
  |  a need to monitor interfaces that are not routed, but I'm inclined
  |  to not make this automatically available from the GUI config.
  |  But the capability can be seeded by hand and it should work at least
  |  for unrouted Linux interfaces that report data in /proc/net/dev.
  |  Edit or add this line to the user_config:  net allow_forcing 1
  |  And add interface lines if necessary: net iface name_of_not_routed_net
  */
static gint			allow_forcing;


static void
draw_led(NetMon *net, int rxtx, int led_index)
	{
	Panel		*p;
	Decal		*led;

	p = net->chart->panel;
	if (rxtx == RX_LED)
		led = net->rxled;
	else
		led = net->txled;

	gkrellm_draw_decal_pixmap(p, led, led_index);
	}

static void
draw_timer(Panel *p, gint seconds, gint force)
	{
	gint		minutes, hours;
	gchar		buf[32], buf_sec[16];

	last_time = seconds;
	hours = seconds / 60 / 60;
	minutes = (seconds / 60) % 60;
	seconds = seconds % 60;

	snprintf(buf_sec, sizeof(buf_sec), "%02d", seconds);
	if (! wide_timer && hours == 0 && timer_seconds)
		{
		snprintf(buf, sizeof(buf), "%2d", minutes);
		time_decal->x_off = x_off_minutes;
		}
	else
		{
		snprintf(buf, sizeof(buf), "%2d:%02d", hours, minutes);
		time_decal->x_off = 0;
		}
	gkrellm_draw_decal_text(p, time_decal, buf, force ? -1 : minutes);

	if (wide_timer || (hours == 0 && timer_seconds))
		gkrellm_draw_decal_text(p, seconds_decal, buf_sec,
					force ? -1 : seconds);
	else
		gkrellm_draw_decal_text(p, seconds_decal, "", 0);

	gkrellm_draw_layers(timer_panel);
	}


static gint
get_connect_state()
	{
	struct stat	st;
	gchar		buf[256];
	gint		state	= TB_NORMAL;
	static gint	old_state;

	switch (timer_button_type)
		{
		case TIMER_TYPE_NONE:
			break;
		case TIMER_TYPE_PPP:
			if (net_timed->state == NET_UP)
				state = TB_ON;
			else
				{
				snprintf(buf, sizeof(buf),
					 "%s/%s", LOCK_DIRECTORY, PPP_LOCK_FILE);
				if (stat(buf, &st) == 0)
					state = TB_STANDBY;
				else
					{
					/* If lock file is ttySx, then user can make a link:
					|  ln -s ~/.gkrellm/LCK..modem LOCK_DIRECTORY/LCK..ttySx
					*/
					snprintf(buf, sizeof(buf), "%s/%s/%s",
								gkrellm_homedir(), GKRELLM_DIR, PPP_LOCK_FILE);
					if (stat(buf, &st) == 0)
						state = TB_STANDBY;
					}
				}
			break;
		case TIMER_TYPE_IPPP:
			if (is_ISDN_online && (*is_ISDN_online)())
				state = (net_timed->state == NET_UP) ? TB_ON : TB_STANDBY;
			break;
		}
	if ((GK.debug_level & DEBUG_TIMER) && state != old_state)
		printf(_("get_connect_state changed from %d to %d  (check=%d)\n"),
				old_state, state, check_connect_state);
	old_state = state;
	return state;
	}

static time_t
get_connect_time(void)
	{
	struct stat	st;
	gchar		buf[256];
	time_t		t	= 0;

	switch (timer_button_type)
		{
		case TIMER_TYPE_NONE:
			break;
		case TIMER_TYPE_PPP:
			snprintf(buf, sizeof(buf), "/var/run/%s.pid", timer_button_iface);
			if (stat(buf, &st) == 0)
				t = st.st_mtime;
			break;
		case TIMER_TYPE_IPPP:
			break;
		}
	return t;
	}

static void
set_timer_button_state(gint decal_state)
	{
	timer_button_old_state = timer_button_state;
	timer_button_state = decal_state;
	gkrellm_draw_decal_pixmap(timer_panel, button_decal, decal_state);
	gkrellm_draw_layers(timer_panel);

	if (   (GK.debug_level & DEBUG_TIMER)
		&& timer_button_state != timer_button_old_state
	   )
		printf(_("set_timer_button_state from %d to %d (check=%d)\n"),
			timer_button_old_state, timer_button_state, check_connect_state);
	}

static void
update_timer_button_monitor()
	{
	if (timer_button_state == TB_PRESSED)
		return;
	if (   (GK.debug_level & DEBUG_TIMER)
		&& timer_button_type != TIMER_TYPE_NONE
		&& net_timed->state != net_timed->old_state
	   )
		printf(_("update_timer_button net_timed old_state=%d new_state=%d\n"),
			   net_timed->old_state, net_timed->state);
	switch (timer_button_type)
		{
		case TIMER_TYPE_NONE:
			if (timer_button_state == TB_ON)
				draw_timer(timer_panel, (int) (time(0) - net_timer0), 0);
			break;

		case TIMER_TYPE_PPP:
			if (net_timed->state == NET_UP)
				{
				set_timer_button_state(TB_ON);
				check_connect_state = FALSE;
				if (net_timed->old_state == NET_DOWN)
					time(&net_timer0);  /* New session just started */
				}
			else if (net_timed->old_state == NET_UP)
				set_timer_button_state(TB_NORMAL);
			if (check_connect_state)
				set_timer_button_state(get_connect_state());

			if (net_timed->state == NET_UP)
				draw_timer(timer_panel, (int) (time(0) - net_timer0), 0);
			break;

		case TIMER_TYPE_IPPP:
			/* get all isdn status from get_connect_state because the
			|  net_timed->state can be UP even with isdn line not connected.
			*/
			set_timer_button_state(get_connect_state());
			if (   timer_button_state != TB_NORMAL
				&& timer_button_old_state == TB_NORMAL
			   )
				time(&net_timer0);  /* New session just started */
			if (timer_button_state != TB_NORMAL)
				draw_timer(timer_panel, (int) (time(0) - net_timer0), 0);
			break;
		}
	}

static void
stale_pppd_files_debug()
	{
	struct stat st;
	gchar	buf[256];

	snprintf(buf, sizeof(buf), "/var/run/%s.pid", timer_button_iface);
	if (stat(buf, &st) == 0 && net_timed->state == NET_DOWN)
		printf(_("  **** Stale pppd pppX.pid file detected!\n"));
	}

static gint
in_button(Decal *d, GdkEventButton *ev)
	{
	if (ev->x > d->x && ev->x <= d->x + d->w)
		return TRUE;
	return FALSE;
	}

static gint	save_tb_state;

static void
cb_timer_button_press(GtkWidget *widget, GdkEventButton *ev)
	{
	if (! in_button(DECAL(timer_panel), ev))
		return;
	if (timer_button_state != TB_PRESSED)		/* button bounce? */
		save_tb_state = timer_button_state;
	set_timer_button_state(TB_PRESSED);
	}

static void
cb_timer_button_release(GtkWidget *widget, GdkEventButton *ev)
	{
	gint	tstate, timer_command;

	if (timer_button_state != TB_PRESSED)
		return;
	set_timer_button_state(save_tb_state);	
	if (! in_button(DECAL(timer_panel), ev))
		return;

	if (GK.debug_level & DEBUG_TIMER)
		printf(_("%d: button release\n"), (gint) time_now);
	switch (timer_button_type)
		{
		case TIMER_TYPE_NONE:
			if (timer_button_state == TB_NORMAL)
				{
				if (*timer_on_command != '\0')
					system(timer_on_command);
				set_timer_button_state(TB_ON);
				time(&net_timer0);
				}
			else
				{
				if (*timer_off_command != '\0')
					system(timer_off_command);
				set_timer_button_state(TB_NORMAL);
				}
			break;

		case TIMER_TYPE_PPP:
		case TIMER_TYPE_IPPP:
			check_connect_state = TRUE;
			tstate = get_connect_state();
			if (GK.debug_level & DEBUG_TIMER)
				stale_pppd_files_debug();
			if (tstate == TB_NORMAL)
				timer_command = ON;
			else if (tstate == TB_ON)
				timer_command = OFF;
			else /* tstate == TB_STANDBY */
				{
				/* For some, pppd is leaving stale LCK..modem (and ppp0.pid)
				|  files which can fool gkrellm.  So the question is, do I
				|  launch off or on command here?  Since I can't trust
				|  TB_STANDBY to mean pppd is running, I'll just base it on
				|  state info.
				*/
				if (last_timer_command == ON)
					{
					timer_command = OFF;
					if (timer_button_type == TIMER_TYPE_PPP)
						set_timer_button_state(TB_NORMAL);
					check_connect_state = FALSE;
					draw_led(net_timed, RX_LED, RX_OFF);	/* Noise */
					draw_led(net_timed, TX_LED, TX_OFF);
					}
				else
					timer_command = ON;
				}
			if (GK.debug_level & DEBUG_TIMER)
				{
				printf(_("Timer button (connect_state=%d last_comand=%d) %s\n"),
					tstate, last_timer_command,
					timer_command ? timer_on_command : timer_off_command);
				}
			if (timer_command == ON && *timer_on_command != '\0')
					system(timer_on_command);
			if (timer_command == OFF && *timer_off_command != '\0')
					system(timer_off_command);
			last_timer_command = timer_command;
			break;
		}
	}


  /* A locked net (locked to the timer button) always has a panel
  |  visible, and the Chart visibility is toggled based on net up/down state.
  */
static void
sync_chart_visibility(NetMon *net)
	{
	gint	state;

	if (! net->created || ! net->locked)
		{
		printf(_("Bogus net logic: %s created=%d locked=%d vis=%d\n"), net->name,
				net->created, net->locked, net->chart_is_visible);
		return;
		}
	/* For ippp, the route may always be up, so base a NET_UP state on the
	|  route being alive and the line status being connected (timer button
	|  state)
	*/
	if (timer_button_type == TIMER_TYPE_IPPP)
		state = (net->state == NET_UP && timer_button_state != TB_NORMAL)
				? NET_UP : NET_DOWN;
	else
		state = net->state;
	switch (state)
		{
		case NET_DOWN:
			if (net->chart_is_visible == TRUE)
				{
				if (GK.debug_level & DEBUG_NET)
					printf(_("Setting visibility of %s to %d\n"),
							net->name, net->state);
				gtk_widget_hide(net->chart->hbox);
				gkrellm_monitor_height_adjust( - net->chart->h);
				gkrellm_pack_side_frames();	/* Change in monitor area height */
				}
			net->chart_is_visible = FALSE;
			break;
		case NET_UP:
			if (net->chart_is_visible == FALSE)
				{
				if (GK.debug_level & DEBUG_NET)
					printf(_("Setting visibility of %s to %d\n"),
								net->name, net->state);
				gtk_widget_show(net->chart->hbox);
				gkrellm_monitor_height_adjust(net->chart->h);
				gkrellm_pack_side_frames();
				}
			net->chart_is_visible = TRUE;
			break;
		}
	}


static NetConfig *
lookup_net_config(gchar *name)
	{
	GList		*list;
	NetConfig	*nc;

	for (list = net_config_list; list; list = list->next)
		{
		nc = (NetConfig *) list->data;
		if (strcmp(name, nc->name) == 0)
			return nc;
		}
	return NULL;
	}

static gint
net_is_config_enabled(NetMon *net)
	{
	NetConfig	*nc;

	if ((nc = lookup_net_config(net->name)))
		return nc->enabled;
	return TRUE;		/* No config entry, so net cannot be disabled */
	}

  /* The net config list has every net interface ever detected and never
  |  removes entries.  So a config for a net persists even if that
  |  net is down whenever config files are rewritten.
  */
static NetConfig *
store_net_config(gchar *name, gint resolution, gint enable, gint extra,
			gint force, gchar *label, gchar *command, gchar *tooltip)
	{
	NetConfig	*nc;

	if ((nc = lookup_net_config(name)) != NULL)
		{
		if (resolution > 0)
			nc->resolution = resolution;
		nc->enabled = enable;
		nc->extra_info = extra;
		nc->force_net_up = force;
		gkrellm_dup_string(&nc->label, label);
		gkrellm_dup_string(&nc->command, command);
		gkrellm_dup_string(&nc->tooltip, tooltip);
		}
	else
		{
		nc = g_new0(NetConfig, 1);
		nc->name = g_strdup(name);
		nc->resolution = resolution;
		nc->enabled = enable;
		nc->extra_info = extra;
		nc->force_net_up = force;
		nc->label = g_strdup(label);
		nc->command = g_strdup(command);
		nc->tooltip = g_strdup(tooltip);
		net_config_list = g_list_append(net_config_list, nc);
		}
	return nc;
	}


static size_abbrev_table	net_bytes_abbrev[]	=
	{
	{ KB_SIZE(1),		1,				"%.0f" },
	{ KB_SIZE(20),		KB_SIZE(1),		"%.1fK" },
	{ MB_SIZE(1),		KB_SIZE(1),		"%.0fK" },
	{ MB_SIZE(20),		MB_SIZE(1),		"%.1fM" },
	{ GB_SIZE(1),		MB_SIZE(1),		"%.0fM" },
	{ GB_SIZE(20),		GB_SIZE(1),		"%.1fG" },
	{ TB_SIZE(1),		GB_SIZE(1),		"%.0fG" },
	{ TB_SIZE(20),		TB_SIZE(1),		"%.1fT" },
	{ TB_SIZE(1000),	TB_SIZE(1),		"%.0fT" }
	};

static void
draw_net_extra(NetMon *net, unsigned long l)
	{
	Chart		*cp;
	TextStyle	*ts, *ts_alt;
	gchar		*s, buf[32];
	gint		n, x, y;

	cp = net->chart;
	ts = gkrellm_chart_textstyle(net_style_id);
	ts_alt = gkrellm_chart_alt_textstyle(net_style_id);
	if (ascent == 0)
		ascent = gdk_char_height(ts->font, '8');
	if (ascent_alt == 0)
		ascent_alt = gdk_char_height(ts_alt->font, 'A');

	format_size_abbrev(buf, sizeof(buf), (gfloat) l, &net_bytes_abbrev[0],
				sizeof(net_bytes_abbrev) / sizeof(size_abbrev_table));
	y = 2;
	x = 4;
	if (net->label && *net->label != '\0')
		{
		s = net->label;
		if (*s == '.')		/* Leading '.' means center the label */
			{
			++s;
			n = gdk_string_width(ts_alt->font, s);
			x = (UC.chart_width - n) / 2;
			if (x < 0)
				x = 0;
			}
		y += 2 + ascent_alt;
		gkrellm_draw_chart_label(cp, ts_alt, x, y, s);
		}
	y += 2 + ascent;
	gkrellm_draw_chart_label(cp, ts, 4, y, buf);
	}



/* On exposure event, redraw the screen from the backing pixmap
*/
static gint
net_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GList		*list;
	NetMon		*net;
	GdkPixmap	*pixmap	= NULL;

	if (timer_panel->drawing_area == widget)
		pixmap = timer_panel->pixmap;
	else
		for (list = net_mon_list; list; list = list->next)
			{
			net = (NetMon *) list->data;
			if (!net->chart || !net->chart->panel)	/* A disabled iface */
				continue;
			if (net->chart->drawing_area == widget)
				pixmap = net->chart->pixmap;
			else if (net->chart->panel->drawing_area == widget)
				pixmap = net->chart->panel->pixmap;
			if (pixmap)
				break;
			}
	if (pixmap)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
			ev->area.x, ev->area.y, ev->area.x, ev->area.y,
			ev->area.width, ev->area.height);
	return FALSE;
	}

static gint
map_x(gint x, gint width)
	{
	gint	xnew;

	xnew = x;
	if (GK.allow_scaling && GK.chart_width_ref != UC.chart_width)
		xnew = xnew * UC.chart_width / GK.chart_width_ref;
	if (x < 0)
		xnew += UC.chart_width - width;
	return xnew;
	}

static gint
set_scale_default(Chart *cp)
	{
	gchar	*s 	= cp->name;
	gint	res;

	if (! strncmp(s, "ppp", 3))
		res = 2000;
#if defined(__FreeBSD__)
	else if (! strncmp(s, "tun", 3))
		res = 2000;
#endif
	else if (! strncmp(s, "plip", 3) || ! strncmp(s, "ippp", 4))
		res = 5000;
	else if (! strncmp(s, "eth", 3))
		res = 20000;
	else
		res = 10000;
	return res;
	}

static void
setup_net_scaling(Chart *cp, gint resolution)
	{
	gint	grids;

	grids = UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS;
	cp->scale_min = resolution;
	KRELL(cp->panel)->full_scale = resolution * grids / UC.update_HZ;
	cp->scale_max = 0;		/* Force chart rescale */
	}

  /* Each monitor in the net_mon_list will have a chart/panel allocated only
  |  if it was config enabled or locked to the timer button.
  */
static void
destroy_net_monitor(NetMon *net)
	{
	Chart	*cp		= net->chart;

	if (net->locked)
		return;
	if (net->launch.button)
		gkrellm_destroy_button(net->launch.button);
	net->launch.button = NULL;
	if (net->created)		/* Charts and such were allocated */
		{
		gkrellm_monitor_height_adjust( - cp->panel->h);
		if (net->chart_is_visible)
			gkrellm_monitor_height_adjust( - cp->h);
		gkrellm_destroy_panel(cp->panel);
		net->launch.tooltip = NULL;
		g_free(cp->panel->textstyle);
		g_free(cp->panel);
		gkrellm_destroy_chart(cp);
		g_free(cp);
		gtk_widget_destroy(net->vbox);
		net->chart = NULL;
		net->vbox = NULL;
		}
	net->state = NET_DOWN;
	net->created = FALSE;
	}

static void
refresh_net_chart(NetMon *net)
	{
	gkrellm_draw_chart(net->chart);
	if (net->extra_info)
		draw_net_extra(net, net->cur);
	net->chart->need_redraw = FALSE;
	}


static gint
cb_net_extra(GtkWidget *widget, GdkEventButton *event)
	{
	GList		*list;
	NetMon		*net;
	NetConfig	*nc;

	if (event->button == 1)
		for (list = net_mon_list; list; list = list->next)
			{
			net = (NetMon *) list->data;
			if (net->created == FALSE)
				continue;
			if (net->chart->drawing_area == widget)
				{
				net->extra_info = 1 - net->extra_info;
				if ((nc = lookup_net_config(net->name)) != NULL)
					nc->extra_info = net->extra_info;
				gkrellm_config_modified();
				}
			refresh_net_chart(net);
			}
	return TRUE;
	}

static void
create_net_monitor(GtkWidget *vbox, NetMon *net, gint first_create)
	{
	Style		*style;
	Chart		*cp;
	Panel		*p;
	NetConfig	*nc;
	gint		n;

	if (first_create)
		{
		net->vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(vbox), net->vbox);

		net->chart = gkrellm_chart_new0();
		net->chart->panel = gkrellm_panel_new0();
		net->chart->panel->textstyle = gkrellm_textstyle_new0();
		}
	else
		{
		gkrellm_destroy_decal_list(net->chart->panel);
		gkrellm_destroy_krell_list(net->chart->panel);
		}
	cp = net->chart;
	p = cp->panel;
	cp->name = net->name;

	/* Net monitors may not be alive when config files are written.
	|  So, I keep a record of every net monitor ever alive and save
	|  the config (resolution) for it.
	*/
	if ((nc = lookup_net_config(net->name)) == NULL)
		{
		n = set_scale_default(cp);
		nc = store_net_config(net->name, n, TRUE, TRUE, FALSE, "", "", "");
		}
	net->label = g_strdup(nc->label);

	cp->h = UC.chart_height[MON_NET];
	gkrellm_create_chart(net->vbox, cp, net_style_id);

	style = gkrellm_panel_style(net_style_id);
	gkrellm_create_krell(p, gkrellm_krell_panel_image(net_style_id), style);

	net->rxled = gkrellm_create_decal_pixmap(p, decal_net_led_pixmap,
				decal_net_led_mask, N_LEDS, style, 0, GK.rx_led_y);
	net->rxled->x = map_x(GK.rx_led_x, net->rxled->w);

	net->txled = gkrellm_create_decal_pixmap(p, decal_net_led_pixmap,
				decal_net_led_mask, N_LEDS, style, 0, GK.tx_led_y);
	net->txled->x = map_x(GK.tx_led_x, net->txled->w);

	*(p->textstyle) = *gkrellm_panel_textstyle(net_style_id);

	if (strlen(cp->name) > 5)
		p->textstyle->font = gkrellm_panel_alt_textstyle(net_style_id)->font;

	gkrellm_configure_panel(p, cp->name, style);
	gkrellm_create_panel(net->vbox, p, gkrellm_bg_panel_image(net_style_id));
	gkrellm_monitor_height_adjust(cp->h + p->h);

	setup_net_scaling(cp, nc->resolution);
	gkrellm_alloc_chart_data(cp);

	if (first_create)
		{
		net->extra_info = nc->extra_info;
		net->force_net_up = nc->force_net_up;
		gtk_signal_connect(GTK_OBJECT (cp->drawing_area), "expose_event",
				(GtkSignalFunc) net_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT (p->drawing_area), "expose_event",
				(GtkSignalFunc) net_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_net_extra, NULL);
		gtk_widget_show_all(net->vbox);
		}
	else
		refresh_net_chart(net);

	gkrellm_setup_launcher(p, &net->launch, CHART_PANEL_TYPE, 0);

	if (net->state == NET_DOWN)	/* Could be for a t/but linked iface */
		{
		net->chart_is_visible = FALSE;
		gtk_widget_hide(cp->hbox);
		gkrellm_monitor_height_adjust( - cp->h);
		}
	else
		net->chart_is_visible = TRUE;

	draw_led(net, RX_LED, RX_OFF);
	draw_led(net, TX_LED, TX_OFF);

	net->created = TRUE;
	}

#define SEC_PAD	0

static void
create_net_timer(GtkWidget *vbox, gint first_create)
	{
	Panel			*p;
	Style			*style;
	GdkImlibBorder	*tb;
	gint			top_margin, bot_margin;
	gint			x, y, w, w1, h, w_avail;

	if (first_create)
		{
		timer_panel = gkrellm_panel_new0();
		}
	else
		{
		gkrellm_destroy_decal_list(timer_panel);
		}
	p = timer_panel;

	style = gkrellm_meter_style(timer_style_id);
	tb = &GK.bg_timer_border;

	button_decal = gkrellm_create_decal_pixmap(p, decal_timer_button_pixmap,
			decal_timer_button_mask, N_TB_DECALS, style, -1, -1);
	button_decal->x = gkrellm_chart_width() - style->margin - button_decal->w;

	p->textstyle = gkrellm_meter_textstyle(timer_style_id);

	if (timer_seconds)
		seconds_decal = gkrellm_create_decal_text(p, "00",
				gkrellm_meter_alt_textstyle(timer_style_id), style, -1, -1, 0);
	else
		seconds_decal = NULL;

	time_decal = gkrellm_create_decal_text(p, "00:00", p->textstyle,
					style, -1, -1, 0);
	gkrellm_configure_panel(p, NULL, style);

	w = time_decal->w + (seconds_decal ? (seconds_decal->w + SEC_PAD) : 0);
	w_avail = button_decal->x - style->margin;
	if (GK.bg_timer_image)
		w_avail -= tb->left + tb->right;
	wide_timer = (w > w_avail || ! timer_seconds) ? FALSE : TRUE;

	/* Some special work needed here if there is a bg_timer.  I have so
	|  far the time_decal and button_decal fitting inside margins of a h_panel.
	|  If I add bg_timer borders to time_decal height and that ends up higher
	|  than button_decal, I will need to grow the h_panel.
	*/
	w = time_decal->w + ((wide_timer && seconds_decal) ? seconds_decal->w : 0);
	h = time_decal->h;

	if ((bg_timer_image = GK.bg_timer_image) != NULL)
		{
		gkrellm_get_top_bottom_margins(style, &top_margin, &bot_margin);
		w += tb->left + tb->right + SEC_PAD;
		h += tb->top + tb->bottom;
		time_decal->y += tb->top;
		time_decal->x += tb->left;
		if (h > button_decal->h)	/* Need to grow the height ? */
			{
			p->label->h_panel = h + top_margin + bot_margin;
			button_decal->y += (h - button_decal->h) / 2;
			}
		else	/* button_decal->y is OK */
			time_decal->y += (button_decal->h - h) / 2;
		}
	else
		{
		/* time_decal and button_decal are initially at same y = top_margin
		*/	
		if (time_decal->h > button_decal->h)
			button_decal->y += (time_decal->h - button_decal->h) / 2;
		else
			time_decal->y += (button_decal->h - time_decal->h) / 2;
		}
	if (seconds_decal)
		{
		/* If no wide timer, overlay the seconds decal on the time_decal.
		*/
		seconds_decal->x = time_decal->x + time_decal->w + SEC_PAD;
		if (! wide_timer)
			{
			w1 = gdk_string_measure(time_decal->text_style.font, "00:") - 2;
			x = (w1 - seconds_decal->w) / 2;
			if (x < 0)
				x = 0;
			seconds_decal->x -= seconds_decal->w + x + 1;
			x_off_minutes = (seconds_decal->x - time_decal->x - w1) / 2;
			if (x_off_minutes < 0)
				x_off_minutes = 0;
			}
		seconds_decal->y = time_decal->y + time_decal->y_baseline
							- seconds_decal->y_baseline;
		}
	gkrellm_create_panel(vbox, p, gkrellm_bg_meter_image(timer_style_id));
	gkrellm_monitor_height_adjust(p->h);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT (p->drawing_area), "expose_event",
				(GtkSignalFunc) net_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),"button_release_event",
				(GtkSignalFunc) cb_timer_button_release, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),"leave_notify_event",
				(GtkSignalFunc) cb_timer_button_release, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),"button_press_event",
				(GtkSignalFunc) cb_timer_button_press, NULL);
		}

	if (bg_timer_image)
		{
		x = time_decal->x - tb->left;
		y = time_decal->y - tb->top;
		gdk_imlib_paste_image(bg_timer_image, p->pixmap, x, y, w, h);
		gdk_imlib_paste_image(bg_timer_image, p->bg_pixmap, x, y, w, h);
		}

	if (! timer_button_enabled)
		{
		gkrellm_monitor_height_adjust( - timer_panel->h);
		gtk_widget_hide(timer_panel->hbox);
		}
	set_timer_button_state(timer_button_state);
	}

  /* Read /proc/net/dev at full speed to run the LEDs, but only update the
  |  charts once per second.
  */
static void
update_net()
	{
	GList		*list;
	NetMon		*net;
	gulong		l;

	(*read_net_data)();
	for (list = net_mon_list; list; list = list->next)
		{
		net = (NetMon *) list->data;
		if (net->created == FALSE)
			continue;
		if (net->state != NET_DOWN)
			{
			if (net->rx > net->oldrx)
				draw_led(net, RX_LED, RX_ON);
			else
				draw_led(net, RX_LED, RX_OFF);
			if (net->tx > net->oldtx)
				draw_led(net, TX_LED, TX_ON);
			else
				draw_led(net, TX_LED, TX_OFF);
			}
		net->oldrx = net->rx;
		net->oldtx = net->tx;
		if (GK.second_tick)
			{
			if (net->chart->primed)
				net->cur = (net->rx - net->chart->prevIn)
						+ (net->tx - net->chart->prevOut);
			gkrellm_store_chart_data(net->chart, net->tx, net->rx, 0);
			refresh_net_chart(net);
			}
		if (net->chart->need_redraw)
			refresh_net_chart(net);
		/* I don't care if this unsigned long add overflows because I
		|  care only about differences from one update to the next.
		*/
		gkrellm_update_krell(net->chart->panel, KRELL(net->chart->panel),
					net->tx + net->rx);
		gkrellm_draw_layers(net->chart->panel);
		}
	if (GK.second_tick)
		{
		l = 0;
		(*sync_net_interfaces)();
		for (list = net_mon_list; list; list = list->next)
			{
			net = (NetMon *) list->data;
			if (allow_forcing && net->force_net_up)
				net->state = NET_UP;
			if (   net->state == NET_UP && net->old_state == NET_DOWN
				&& !net->locked && net_is_config_enabled(net)
			   )
				{
				create_net_monitor(dynamic_net_vbox, net, TRUE);
				gkrellm_pack_side_frames();
				}
			else if (net->state == NET_DOWN && !net->locked && net->created)
				{
				destroy_net_monitor(net);
				gkrellm_pack_side_frames();
				}
			if (net->locked)
				sync_chart_visibility(net);
			}
		update_timer_button_monitor();
		if (net_timed && net_timed->state == NET_DOWN)
			{
			draw_led(net_timed, RX_LED, RX_OFF);
			draw_led(net_timed, TX_LED, TX_OFF);
			}
		}
	}

static NetMon	*
lookup_net(gchar *name)
	{
	NetMon	*net;
	GList	*list;

	for (list = net_mon_list; list; list = list->next)
		{
		net = (NetMon *) list->data;
		if (strcmp(net->name, name) == 0)
			return net;
		}
	return NULL;
	}

static void
create_timed_monitor(gchar *name)
	{
	GList	*list;
	NetMon	*net;
	time_t	t;
	gint	i;

	net_timed = NULL;
	timer_button_type = TIMER_TYPE_NONE;
	time(&net_timer0);

	if (   ! timer_button_iface || *timer_button_iface == '\0'
		|| *timer_button_iface == ' ' || ! strcmp(timer_button_iface, "none")
	   )
		return;

	/* Making a timed mon out of one that is already up?  It needs to be
	|  moved to a different vbox, so destroy and create dance.
	*/
	for (list = net_mon_list; list; list = list->next)
		{
		net = (NetMon *) list->data;
		if (strcmp(net->name, timer_button_iface) == 0)
			{
			destroy_net_monitor(net);
			break;
			}
		}
	if ((net = lookup_net(timer_button_iface)) == NULL)
		{
		net = g_new0(NetMon, 1);
		net->name = g_strdup(timer_button_iface);
		net_mon_list = g_list_append(net_mon_list, net);
		}
	net->locked = TRUE;
	create_net_monitor(net_vbox, net, TRUE);
	net_timed = net;
	timer_button_type = timer_defaults[0].type;
	for (i = 0; i < sizeof(timer_defaults) / sizeof(TimerType); ++i)
		{
		if (strncmp(timer_button_iface, timer_defaults[i].name,
						strlen(timer_defaults[i].name) - 1) == 0)
			{
			timer_button_type = timer_defaults[i].type;
			break;
			}
		}
	t = get_connect_time();
	if (t > 0)
		net_timer0 = t;
	}

#if 0
/* With 0.10.0, the way for each monitor to load extra images is to check
|  its own theme directory and not rely on config.c to do the loading.  But
|  for 0.9,X compatibility, this will not be enforced for a long while.
|  But when I do, this routine becomes active.
*/
#include    "pixmaps/bg_meter_timerbutton.xpm"
#include    "pixmaps/bg_timer.xpm"
#include    "pixmaps/decal_net_leds.xpm"
#include    "pixmaps/decal_timer_button.xpm"

static void
load_net_extra_images()
	{
	gchar	**xpm;
	gint	w;

	/* Check for theme_dir/net/decal_net_leds.png.
	*/
	gkrellm_load_image("decal_net_leds", decal_net_leds_xpm,
			&decal_net_led_image, NET_STYLE_NAME);
	w = GK.decal_net_led_image->rgb_width;
	if (GK.allow_scaling && GK.chart_width_ref != UC.chart_width)
		w = w * UC.chart_width / GK.chart_width_ref;
	gkrellm_render_to_pixmap(decal_net_led_image, &decal_net_led_pixmap,
			&decal_net_led_mask, w, 0);

	/* Check for theme_dir/net/decal_timer_button.png
	*/
	gkrellm_load_image("decal_timer_button", decal_timer_button_xpm,
			&decal_timer_button_image, TIMER_STYLE_NAME);
	gkrellm_render_to_pixmap(decal_timer_button_image,
			&decal_timer_button_pixmap, &decal_timer_button_mask, 0, 0);

	/* Check for theme_dir/net/bg_timer.png.  It is an extra image with
	|  no default - so bg_timer_image may end up NULL.
	*/
	xpm = (gkrellm_using_default_theme()) ? NULL : bg_timer_xpm;
	gkrellm_load_image("bg_timer", xpm,
				&bg_timer_image, TIMER_STYLE_NAME))
	}
#endif

static void
load_net_extra_images()
	{
	gint	w;

	/* Check for theme_dir/net/decal_net_leds.png.  All these image loads
	|  leave the image pointer alone if no image is found so previous image
	|  loads in config.c will be preserved.
	*/
	gkrellm_load_image("decal_net_leds", NULL,
			&GK.decal_net_led_image, NET_STYLE_NAME);
	w = GK.decal_net_led_image->rgb_width;
	if (GK.allow_scaling && GK.chart_width_ref != UC.chart_width)
		w = w * UC.chart_width / GK.chart_width_ref;
	gkrellm_render_to_pixmap(GK.decal_net_led_image, &decal_net_led_pixmap,
			&decal_net_led_mask, w, 0);

	/* Check for theme_dir/net/decal_timer_button.png
	*/
	gkrellm_load_image("decal_timer_button", NULL,
			&GK.decal_timer_button_image, TIMER_STYLE_NAME);
	gkrellm_render_to_pixmap(GK.decal_timer_button_image,
			&decal_timer_button_pixmap, &decal_timer_button_mask, 0, 0);

	/* Check for theme_dir/net/bg_timer.png.  It is an extra image with
	|  no default - so GK.bg_timer_image may end up NULL.
	*/
	if (gkrellm_load_image("bg_timer", NULL,
					&GK.bg_timer_image, TIMER_STYLE_NAME))
		gdk_imlib_set_image_border(GK.bg_timer_image, &GK.bg_timer_border);
	}

static void
create_net(GtkWidget *vbox, gint first_create)
	{
	GList	*list;
	NetMon	*net;

	load_net_extra_images();

	/* Make a couple of vboxes here so I can control the net layout.
	|  I want interface linked to the timer button  to go last.
	*/
	if (first_create)
		{
		dynamic_net_vbox = gtk_vbox_new (FALSE, 0);
		gtk_container_add (GTK_CONTAINER (vbox), dynamic_net_vbox);
		gtk_widget_show(dynamic_net_vbox);

		net_vbox = gtk_vbox_new (FALSE, 0);
		gtk_container_add (GTK_CONTAINER (vbox), net_vbox);
		gtk_widget_show(net_vbox);
		create_timed_monitor(timer_button_iface);

		(*(sync_net_interfaces))();
		for (list = net_mon_list; list; list = list->next)
			{
			net = (NetMon *) list->data;
			if (   ! net->locked && net->state == NET_UP
				&& net_is_config_enabled(net)
			   )
				create_net_monitor(dynamic_net_vbox, net, first_create);
			}
		}
	else
		for (list = net_mon_list; list; list = list->next)
			{
			net = (NetMon *) list->data;
			if (   (net->state == NET_UP && net_is_config_enabled(net))
				|| net->locked
			   )
				create_net_monitor(NULL, net, first_create);
			}
	timer_vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), timer_vbox);
	gtk_widget_show(timer_vbox);
	create_net_timer(timer_vbox, first_create);
	ascent = 0;
	ascent_alt = 0;
	if (first_create)
		draw_timer(timer_panel, (int) (time(0) - net_timer0), 1);
	else
		draw_timer(timer_panel, last_time , 1);
	gkrellm_draw_layers(timer_panel);
	}


#define	NET_CONFIG_KEYWORD	"net"

static void
save_net_config(FILE *f)
	{
	GList		*list;
	NetConfig	*nc;
	gchar		*s, buf[64];

	fprintf(f, "%s allow_forcing %d\n", NET_CONFIG_KEYWORD, allow_forcing);
	for (list = net_config_list; list; list = list->next)
		{
		nc = (NetConfig *) list->data;
		if (*nc->label == '\0')
			strcpy(buf, "0");
		else
			strcpy(buf, nc->label);
		for (s = buf; *s != '\0'; ++s)
			if (*s == ' ' || *s == '\t')
				*s = '\\';
		fprintf(f, "%s iface %s %d %d %s %d %d\n", NET_CONFIG_KEYWORD,
				nc->name, nc->resolution, nc->enabled, buf,
				nc->extra_info, nc->force_net_up);
		fprintf(f, "%s launch %s\n", NET_CONFIG_KEYWORD, nc->command);
		fprintf(f, "%s tooltip %s\n", NET_CONFIG_KEYWORD, nc->tooltip);
		}
	fprintf(f, "%s timer_enabled %d\n", NET_CONFIG_KEYWORD,
						timer_button_enabled);
	fprintf(f, "%s timer_seconds %d\n", NET_CONFIG_KEYWORD, timer_seconds);
	fprintf(f, "%s timer_iface %s\n", NET_CONFIG_KEYWORD, timer_button_iface);
	fprintf(f, "%s timer_on %s\n", NET_CONFIG_KEYWORD, timer_on_command);
	fprintf(f, "%s timer_off %s\n", NET_CONFIG_KEYWORD, timer_off_command);
	}

static void
load_net_config(gchar *arg)
	{
	static NetConfig	*nc_prev;
	static NetMon		*net_prev;
	NetMon				*net;
	gchar				*s, name[32], label[32];
	gchar				config[32], item[CFG_BUFSIZE];
	gint				resolution	= 20000,
						enable		= TRUE,
						extra		= 0,
						force		= 0;
	gint				n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (GK.debug_level & (DEBUG_NET | DEBUG_TIMER))
			printf(_("load config: %s %s\n"), config, item);
		if (net_prev && !strcmp(config, "launch"))
			{
			gkrellm_dup_string(&net_prev->launch.command, item);
			gkrellm_dup_string(&nc_prev->command, item);
			}
		else if (net_prev && !strcmp(config, "tooltip"))
			{
			gkrellm_dup_string(&net_prev->launch.tooltip_comment, item);
			gkrellm_dup_string(&nc_prev->tooltip, item);
			}
		else if (strcmp(config, "iface") == 0)
			{
			sscanf(item, "%31s %d %d %31s %d %d",
						name, &resolution, &enable, label, &extra, &force);
			if (!strcmp(label, "0"))
				label[0] = '\0';
			for (s = label; *s != '\0'; ++s)
				if (*s == '\\')
					*s = ' ';
			nc_prev = store_net_config(name, resolution, enable, extra, force,
							label, "", "");
			net = g_new0(NetMon, 1);
			net->name = g_strdup(name);
			net_mon_list = g_list_append(net_mon_list, net);
			net->state = NET_DOWN;
			if (allow_forcing)
				net->force_net_up = force;
			net->launch.command = g_strdup("");
			net->launch.tooltip_comment = g_strdup("");
			net_prev = net;
			}
		else if (strcmp(config, "timer_enabled") == 0 && ! GK.demo)
			sscanf(item, "%d", &timer_button_enabled);
		else if (strcmp(config, "timer_seconds") == 0)
			sscanf(item, "%d", &timer_seconds);
		else if (strcmp(config, "timer_iface") == 0)
			{
			if (GK.demo && strcmp(item, "none") == 0)
				gkrellm_dup_string(&timer_button_iface, "ppp0");
			else
				gkrellm_dup_string(&timer_button_iface, item);
			}
		else if (strcmp(config, "timer_on") == 0)
			gkrellm_dup_string(&timer_on_command, item);
		else if (strcmp(config, "timer_off") == 0)
			gkrellm_dup_string(&timer_off_command, item);
		else if (strcmp(config, "allow_forcing") == 0)
			sscanf(item, "%d", &allow_forcing);
		}
	}


/* -------------- User config interface --------------------- */

#define STEP	1

static GtkWidget	*pon_combo,
					*poff_combo,
					*timer_button_iface_combo;

static GtkWidget	*enable_net_timer_button;
static GtkWidget	*net_timer_seconds_button;

static gint	net_res_map []	=
	{
	5,
	10, 20, 50,
	100, 200, 500,
	1000, 2000, 5000,
	10000, 20000, 50000,
	100000, 200000, 500000,
	1000000, 2000000, 5000000,
	10000000
	};

gint
map_1_2_5(gint value, gint *table, gint size)
	{
	gint	i;

	for (i = 0; i < size; ++i)
		{
/*printf("  mapping[%d] value=%d table=%d\n", i, value, table[i]); */
		if (value == table[i])
			return table[i];
		else if (value == table[i] - STEP)
			return table[i - 1];
		else if (value == table[i] + STEP)
			return table[i + 1];
		}
	return value;
	}

static void
cb_net_resolution(GtkWidget *widget, GtkSpinButton *spin)
	{
	GList		*list;
	NetMon		*net;
	gint		res;

	for (list = net_mon_list; list; list = list->next)
		{
		net = (NetMon *) list->data;
		if (spin == GTK_SPIN_BUTTON(net->spin_button))
			{
			res = gtk_spin_button_get_value_as_int(spin);
			if (res != net->config_temp)	/* Avoid recursion */
				{
				res = map_1_2_5(res, net_res_map,
						sizeof(net_res_map)/sizeof(int));
				net->config_temp = res;
				gtk_spin_button_set_value(spin, (gfloat) res);
				if (net->created)
					{
					setup_net_scaling(net->chart, res);
					gkrellm_draw_chart(net->chart);
					}
				}
			break;
			}
		}
	}



static gchar	*net_info_text[] =
{
#if defined(__linux__)

N_("Routed net interfaces are automatically detected and charted.\n"
	"Data is plotted with units of bytes/sec if possible, however,\n"
	"units of packets/sec are used for kernel 2.0 and some 2.2\n"
	"nic drivers which do not report bytes.\n\n"),

N_("<b>Timer Button\n"),
N_("The timer button may be linked to a ppp or ippp net interface,\n"
	"and shows an off, standby, or on state by a distinctive (color\n"
	"or shape) icon.\n"),

N_("<b>\tppp: "),
N_("Standby state is while the modem phone line is locked while\n"
	"\tppp is connecting, and the on state is the ppp link connected.\n"
	"\tThe phone line lock is determined by the existence of the modem\n"
	"\tlock file /var/lock/LCK..modem.  If your pppd setup does not\n"
	"\tuse /dev/modem, then you can configure an alternative with:\n"),
N_("<i>\t\tln  -s  /var/lock/LCK..ttySx   ~/.gkrellm/LCK..modem\n"),
N_("\twhere ttySx is the tty device your modem uses.  The ppp on\n"
	"\tstate is detected by the existence of /var/run/pppX.pid and\n"
	"\tthe time stamp of this file is the base for the on line time.\n"),

N_("<b>\n\tippp: "),
N_("The timer button standby state is not applicable to isdn\n"
	"\tinterfaces that are always routed. The on state is isdn on line\n"
	"\twhile the ippp interface is routed.  The on line timer is reset\n"
	"\twhen the isdn interface transitions from a hangup to an on line\n"
	"\tstate\n\n"),

N_("For both ppp and ippp timer button links, the panel area of the\n"
	"interface is always shown and the chart appears when the interface\n"
	"is routed with the phone link connected or on line.\n" 
	"The timer button Start Command must run in the background and\n"
	"this is automatically the default for most ppp logon scripts. One\n"
	"exception, if you use wvdial it needs to be explicitly backgrounded:\n"),

N_("<i>\t\twvdail &\n"),

N_("and the ppp logoff Stop Command in this case could be:\n"),

N_("<i>\t\tskill -c wvdial\n"),

N_("Otherwise do not append the \"&\" on more common ppp Start\n"
	"Command entries.\n"
	"If the timer button is not linked to a net interface, then it can\n"
	"be used as a push on / push off timer\n\n"),

N_("<b>Chart Labels\n"),
N_("If you enter an identifying label for a net chart, you can make it\n"
	"easy to see who the interface is connected to.  This is useful if\n"
	"there are several net interfaces or remote GKrellMs running.  A\n"
	"label beginning with a period is centered\n\n"
	"See General->Info for help on setting Net chart resolutions.\n")

#else

#if defined(__FreeBSD__)
/* **************************
	I could use someone running FreeBSD to write up a better Net Info
	section and mail it to me.
**************************** */
N_("Routed net interfaces are automatically detected and charted.\n"
	"The timer button may be linked to a tun net interface, and\n"
	"shows an off, standby, or on state by a distinctive (color\n"
	"or shape) icon.\n"
	"tun: Standby state is while the modem phone line is locked while\n"
	"\tconnecting, and the on state is the tun link connected.\n"
	"\tThe phone line lock is determined by the existence of the modem\n"
	"\tlock file /var/spool/lock/LCK..modem.  If your setup does not\n"
	"\tuse /dev/modem, then you can configure an alternative with:\n"
	"\t\tln  -s  /var/lock/LCK..ttySx   ~/.gkrellm/LCK..modem\n"
	"\twhere ttySx is the tty device your modem uses.  The tun on\n"
	"\tstate is detected by the existence of /var/run/pppX.pid and\n"
	"\tthe time stamp of this file is the base for the on line time.\n"
	"If the timer button is not linked to a net interface, then it can\n"
	"be used as a push on / push off timer\n\n"

	"If you enter an identifying label for a net chart, you can make it\n"
	"easy to see who is connected to who.  This is useful if there are\n"
	"several net interfaces or remote GKrellMs running.  A label\n\n"
	"beginning with a period is centered\n\n"

	"See General->Info for help on setting Net chart resolutions.\n"),
#endif
""
#endif
}
;

static void
create_net_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*table;
	GtkWidget		*vbox, *vbox1;
	GtkWidget		*hbox;
	GtkWidget		*label;
	GtkWidget		*separator;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	GList			*list;
	NetMon			*net;
	NetConfig		*nc;
	gchar			buf[256];
	gint			i;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

	vbox = gkrellm_create_tab(tabs, _("Timer Button"));

	gkrellm_check_button(vbox, &enable_net_timer_button, timer_button_enabled,
				TRUE, 0, _("Enable Net timer button"));
	gkrellm_check_button(vbox, &net_timer_seconds_button, timer_seconds,
				TRUE, 0, _("Show seconds in Net timer"));
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	timer_button_iface_combo = gtk_combo_new();
	gtk_box_pack_start(GTK_BOX(hbox), timer_button_iface_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, _("none"));
	for (i = 0; i < sizeof(timer_defaults) / sizeof(TimerType); ++i)
		list = g_list_append(list, timer_defaults[i].name);

	gtk_combo_set_popdown_strings( GTK_COMBO(timer_button_iface_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(timer_button_iface_combo)->entry),
						timer_button_iface);
	label = gtk_label_new(_("Interface to link to the timer button"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);


	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	vbox1 = gtk_vbox_new (FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(vbox1), 3);
	gtk_container_add(GTK_CONTAINER(hbox), vbox1);
	label = gtk_label_new(_("Start Command"));
	gtk_box_pack_start (GTK_BOX (vbox1), label, FALSE, TRUE, 0);
	pon_combo = gtk_combo_new();
	gtk_box_pack_start (GTK_BOX (vbox1), pon_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, "pon");
	gtk_combo_set_popdown_strings( GTK_COMBO(pon_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(pon_combo)->entry),
						timer_on_command);

	vbox1 = gtk_vbox_new (FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(vbox1), 3);
	gtk_container_add(GTK_CONTAINER(hbox), vbox1);
	label = gtk_label_new(_("Stop Command"));
	gtk_box_pack_start (GTK_BOX (vbox1), label, FALSE, TRUE, 0);
	poff_combo = gtk_combo_new();
	gtk_box_pack_start (GTK_BOX (vbox1), poff_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, "poff");
	gtk_combo_set_popdown_strings( GTK_COMBO(poff_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(poff_combo)->entry),
						timer_off_command);

	for (list = net_mon_list; list; list = list->next)
		{
		net = (NetMon *) list->data;
		nc = lookup_net_config(net->name);
		net->config_temp = nc->resolution;

		vbox = gkrellm_create_tab(tabs, net->name);

		if (net->locked)
			nc->enabled = TRUE;
		snprintf(buf, sizeof(buf), _("Enable %s"), net->name);
		gkrellm_check_button(vbox, &net->enable_button, nc->enabled,
						TRUE, 0, buf);
		if (allow_forcing)
			gkrellm_check_button(vbox, &net->force_button, nc->force_net_up,
						TRUE, 0,
		_("Force chart to be always shown even if interface is not routed."));
		separator = gtk_hseparator_new();
		gtk_box_pack_start(GTK_BOX(vbox), separator, TRUE, TRUE, 0);

		snprintf(buf, sizeof(buf), _("Chart resolution in %s/sec per grid"),
							(net->rxtx_units == NET_UNITS_BYTES) ?
							_("bytes") : _("packets"));
		gkrellm_spin_button(vbox, &net->spin_button, (gfloat) nc->resolution,
			5.0, 10000000.0, (gfloat) STEP, (gfloat) STEP, 0, 85,
			cb_net_resolution, NULL, FALSE, buf);



		hbox = gtk_hbox_new (FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
		net->label_entry = gtk_entry_new_with_max_length(16);
		gtk_widget_set_usize(net->label_entry, 70, 0);
		gtk_entry_set_text(GTK_ENTRY(net->label_entry), nc->label);
		gtk_box_pack_start (GTK_BOX (hbox), net->label_entry, FALSE, TRUE, 2);
		label = gtk_label_new(_("Optional label for this interface."));
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 4);

		if (net->created)
			net->config_temp = net->chart->scale_min;

		table = gkrellm_launcher_table_new(vbox, 1);
		gkrellm_config_launcher(table, 0,  &net->launch_entry,
						&net->tooltip_entry, net->name, &net->launch);
		}

/* --Info tab */
	if (*net_info_text[0] != '\0')
		{
		vbox = gkrellm_create_tab(tabs, _("Info"));
		scrolled = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
		text = gtk_text_new(NULL, NULL);
		for (i = 0; i < sizeof(net_info_text)/sizeof(gchar *); ++i)
			gkrellm_add_info_text_string(text, _(net_info_text[i]));
		gtk_text_set_editable(GTK_TEXT(text), FALSE);
		gtk_container_add(GTK_CONTAINER(scrolled), text);
		}
	}

  /* Apply net config tab settings to the net UC and, for the net apply
  |  we also apply to the net_config[].
  */
static void
apply_net_config()
	{
	GList		*list;
	NetMon		*net;
	Chart		*cp;
	gint		enable;
	gint		old_timer_seconds;
	gchar		*s0, *s1;

	enable = GTK_TOGGLE_BUTTON(enable_net_timer_button)->active;
	gkrellm_enable_visibility(enable, &timer_button_enabled, timer_panel->hbox,
					timer_panel->h);

	old_timer_seconds = timer_seconds;
	timer_seconds = GTK_TOGGLE_BUTTON(net_timer_seconds_button)->active;
	if (timer_seconds != old_timer_seconds)
		{
		if (timer_button_enabled)
			gkrellm_monitor_height_adjust( - timer_panel->h);
		create_net_timer(timer_vbox, FALSE);
		draw_timer(timer_panel, last_time , 1);
		}

	for (list = net_mon_list; list; list = list->next)
		{
		/* The chart resolution already has been updated in the callback. 
		|  Here I need to update the net_config entries so they
		|  will be saved.
		*/
		net = (NetMon *) list->data;
		cp = net->chart;

		if (net->enable_button)
			{
			enable = GTK_TOGGLE_BUTTON(net->enable_button)->active;
			if (net->locked && !enable)		/* Cannot disable a locked iface */
				{
				enable = TRUE;
				gtk_toggle_button_set_active(
						GTK_TOGGLE_BUTTON(net->enable_button), TRUE);
				s0 = g_strdup_printf(
					_("The net interface %s cannot be disabled\n"
					"unless it is unlinked from the Timer Button\n"
					"or the Timer Button is disabled.\n"), net->name);
				gkrellm_message_window(_("GKrellM Config Error"), s0, NULL);
				g_free(s0);
				}
			}
		else
			enable = TRUE;
		if (allow_forcing && net->force_button)
			net->force_net_up = GTK_TOGGLE_BUTTON(net->force_button)->active;
		if (net->launch_entry && net->created)
			gkrellm_apply_launcher(&net->launch_entry, &net->tooltip_entry,
					net->chart->panel, &net->launch, gkrellm_launch_button_cb);
		gkrellm_dup_string(&net->label,
						gkrellm_entry_get_text(&net->label_entry));
		store_net_config(net->name, net->config_temp, enable,
				net->extra_info, net->force_net_up,
				net->label, net->launch.command, net->launch.tooltip_comment);

		/* If enabling a previously disable iface, just set the state to
		|  net down.  If net was up, then it will be detected as NET_UP
		|  and a chart will be created. (it is now config enabled)
		*/
		if (enable && ! net->created)		/* No chart exists */
			net->state = NET_DOWN;

		/* If disabling and net was UP, I destroy the monitor.
		|  net config now has it disabled so no chart will be created.
		*/
		else if (! enable && net->created)
			destroy_net_monitor(net);
		else if (net->created)
			setup_net_scaling(cp, cp->scale_min);	/* In case grid change */
		}
	if (!timer_button_enabled)
		gtk_entry_set_text(
				GTK_ENTRY(GTK_COMBO(timer_button_iface_combo)->entry), "none");
	s0 = gkrellm_entry_get_text(&(GTK_COMBO(timer_button_iface_combo)->entry));
	if (*s0 == '\0')
		s0 = "none";
	if (strcmp(s0, timer_button_iface))
		{
		gkrellm_dup_string(&timer_button_iface, s0);
		if (net_timed)
			{
			net_timed->locked = FALSE;
			destroy_net_monitor(net_timed);
			}
		create_timed_monitor(timer_button_iface);
		draw_timer(timer_panel, (int) (time(0) - net_timer0), 1);
		gkrellm_draw_layers(timer_panel);
		}
	s0 = gkrellm_entry_get_text(&(GTK_COMBO(pon_combo)->entry));
	s1 = gkrellm_entry_get_text(&(GTK_COMBO(poff_combo)->entry));
	gkrellm_dup_string(&timer_on_command, s0);
	gkrellm_dup_string(&timer_off_command, s1);
	}




static Monitor	monitor_net =
	{
	N_("Net"),				/* Name, for config tab.	*/
	MON_NET,			/* Id,  0 if a plugin		*/
	create_net,			/* The create function		*/
	update_net,			/* The update function		*/
	create_net_tab,		/* The config tab create function	*/
	apply_net_config,	/* Apply the config function		*/

	save_net_config,	/* Save user conifg			*/
	load_net_config,	/* Load user config			*/
	NET_CONFIG_KEYWORD,	/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_net_monitor(void)
	{
	time(&net_timer0);
	timer_button_iface = g_strdup(timer_defaults[0].name);
	timer_button_type = timer_defaults[0].type;
	timer_button_enabled = TRUE;
	timer_seconds = FALSE;
	timer_on_command = g_strdup("");
	timer_off_command = g_strdup("");

    monitor_net.name = _(monitor_net.name);
	net_style_id = gkrellm_add_chart_style(&monitor_net, NET_STYLE_NAME);
	if (setup_net_interface())
		return &monitor_net;
	return NULL;
	}


static Monitor	monitor_timer =
	{
	NULL,				/* Name, for config tab.	*/
	MON_TIMER,			/* Id,  0 if a plugin		*/
	NULL,				/* The create function		*/
	NULL,				/* The update function		*/
	NULL,				/* The config tab create function	*/
	NULL,				/* Apply the config function		*/
	
	NULL,				/* Save user conifg			*/
	NULL,				/* Load user config			*/
	NULL,				/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_timer_monitor(void)
	{
	timer_style_id = gkrellm_add_meter_style(&monitor_timer, TIMER_STYLE_NAME);
	return &monitor_timer;
	}

