/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
 
/*
|  1/29/2001 Hajimu UMEMOTO patched better networking code and IPv6 support
|			into the new TCP connection lookup.  Also improved
|			read_freebsd_tcp_port_data() with 1/23 patch.
| 10/12/2000 NetBSD code contributed by Anthony Mallet
|			<metall@ficus.yi.org>
| 2/25/2000	FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
|             He also added IPv6 support for FreeBSD and Linux.
*/


#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include "gkrellm_threads.h"

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

/* Conditionals for INET6 and u_int32_t from
|  Hideaki YOSHIFUJI <yoshfuji@ecei.tohoku.ac.jp>
*/
/* Most INET6 for Linux needs libc 2.1, but sockaddr_storage is not in 2.1.1 */
#if defined(__linux__)
#if defined(__GLIBC__) && ((__GLIBC__>2) || (__GLIBC__==2 && __GLIBC_MINOR__>=2))
#define INET6
#endif
#if !defined(__GLIBC__) || __GLIBC__<2 || (__GLIBC__==2 && __GLIBC_MINOR__<1)
#define uint32_t        u_int32_t
#endif
#endif


#if defined(__FreeBSD__)
#include <osreldate.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/sysctl.h>
#include <net/route.h>
#if defined(__KAME__) && !defined(INET6)
#define INET6
#endif
#if defined(INET6)
#include <netinet6/ip6_var.h>
#endif
#include <netinet/in_pcb.h>
#include <netinet/ip_var.h>
#if defined(INET6)
#include <netinet/ip6.h>
#endif
#include <netinet/tcp.h>
#include <netinet/tcpip.h>
#include <netinet/tcp_seq.h>
#include <netinet/tcp_fsm.h>
#include <netinet/tcp_timer.h>
#include <netinet/tcp_var.h>

#if __FreeBSD_version < 300000
#if defined(INET6)
#include <netinet6/tcp6.h>
#include <netinet6/in6_pcb.h>
#include <netinet6/tcp6_timer.h>
#include <netinet6/tcp6_var.h>
#endif

#include <kvm.h>
#include <nlist.h>

static struct nlist nl[] = {
#define	N_TCB		0
	{ "_tcb" },
#if defined(INET6)
#define	N_TCB6		1
	{ "_tcb6" },
#endif
	{ "" },
};

extern	kvm_t	*kvmd;
extern	char	errbuf[];
#endif
#endif

typedef struct
	{
	GtkWidget	*vbox;

	gchar		*name;
	gint		idx;
	Chart		*chart;
	Chart		chart_minute;
	Chart		chart_hour;
	gint		extra_info;
	gint		height;
	Launcher	launch;
	GtkWidget	*launch_entry,
				*tooltip_entry;
	GtkWidget	*launch_table;

	gint		scale_min_minute;
	gint		scale_min_hour;

	gshort		*mark_data;		/* Draw marks if hits for any second */
	gint		mark_position,
				mark_prev_hits;

	Decal		*list_decal;
	DecalButton	*list_button;
	GString		*connection_string;
	GList		*tcp_save_list;
	gboolean	busy;
	gboolean	connection_string_event;

	gchar		*label0;
	gint		active0;
	gint		prev_active0;
	gulong		hits0;
	gboolean	data0_is_range;
	gulong		port0_0,
				port0_1;

	gchar		*label1;
	gint		active1;
	gint		prev_active1;
	gulong		hits1;
	gboolean	data1_is_range;
	gulong		port1_0,
				port1_1;
	}
	InetMon;


  /* Values for state.
  */
#define	TCP_DEAD	0
#define	TCP_ALIVE	1

typedef struct
	{
	gint			state;
	gint			family;
	gint			local_port;
	struct in_addr	remote_addr;
#if defined(INET6)
	struct in6_addr	remote_addr6;
#endif
	gint			remote_port;
	gint			new_hit;
	}
	ActiveTCP;

static GList		*inet_mon_list;

static GList		*active_tcp_list,
					*free_tcp_list;


/* ====== System dependent interface ====================================== */

void (*read_tcp_port_data)();

#if defined(__linux__) || defined(__FreeBSD__) || defined(__NetBSD__) \
	|| defined(__OpenBSD__)
static ActiveTCP *
tcp_alloc()
	{
	ActiveTCP	*tcp;

	if (free_tcp_list)
		{
		tcp = free_tcp_list->data;
		free_tcp_list = g_list_remove(free_tcp_list, tcp);
		}
	else
		tcp = g_new0(ActiveTCP, 1);
	return tcp;
	}

static ActiveTCP *
log_active_port(ActiveTCP *tcp)
	{
	GList		*list;
	ActiveTCP	*active_tcp, *new_tcp;
	gchar		*ap, *aap;
	gint		slen;

	for (list = active_tcp_list; list; list = list->next)
		{
		active_tcp = (ActiveTCP *) (list->data);
		if (tcp->family == AF_INET)
			{
			ap = (char *)&tcp->remote_addr;
			aap = (char *)&active_tcp->remote_addr;
			slen = sizeof(struct in_addr);
			}
#if defined(INET6)
		else if (tcp->family == AF_INET6)
			{
			ap = (char *)&tcp->remote_addr6;
			aap = (char *)&active_tcp->remote_addr6;
			slen = sizeof(struct in6_addr);
			}
#endif
		else
			return 0;
		if (   memcmp(aap, ap, slen) == 0
			&& active_tcp->remote_port == tcp->remote_port
			&& active_tcp->local_port == tcp->local_port
		   )
			{
			active_tcp->state = TCP_ALIVE;
			return active_tcp;	/* Old hit still alive, not a new hit	*/
			}
		}
	tcp->state = TCP_ALIVE;
	tcp->new_hit = 1;
	new_tcp = tcp_alloc();
	*new_tcp = *tcp;
	active_tcp_list = g_list_prepend(active_tcp_list, (gpointer) new_tcp);
	return new_tcp;		/* A new hit	*/
	}

static void
log_tcp_port_data(ActiveTCP *tcp)
	{
	GList		*list;
	InetMon		*in;
	ActiveTCP	*active_tcp = NULL;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		if (   (!in->data0_is_range
				&& (   in->port0_0 == tcp->local_port
					|| in->port0_1 == tcp->local_port))
			|| (in->data0_is_range
				&& tcp->local_port >= in->port0_0
				&& tcp->local_port <= in->port0_1)
		   )
			{
			++in->active0;
			active_tcp = log_active_port(tcp);
			in->hits0 += active_tcp->new_hit;
			}
		if (   (!in->data1_is_range
				&& (   in->port1_0 == tcp->local_port
					|| in->port1_1 == tcp->local_port))
			|| (in->data1_is_range
				&& tcp->local_port >= in->port1_0
				&& tcp->local_port <= in->port1_1)
		   )
			{
			++in->active1;
			active_tcp = log_active_port(tcp);
			in->hits1 += active_tcp->new_hit;
			}
		}
	/* Defer setting new hit to 0 until here so multiple inet charts can
	|  monitor the same port number.  The above active_port will be the
	|  same if found for both data0 and data1.
	*/
	if (active_tcp)
		active_tcp->new_hit = 0;
	}

#endif	/* __LINUX__ || __FreeBSD__ || __NetBSD__ || __OpenBSD__ */



/* ----- FreeBSD ------------------------------------------------------- */

#if defined(__FreeBSD__)
#if __FreeBSD_version < 300000
static void
read_freebsd_tcp_port_data()
	{
	ActiveTCP	tcp;
	gint		tcp_status;
	struct		inpcbhead head;
	struct		inpcb inpcb, *next;
#if defined(INET6)
	struct		in6pcb in6pcb, *prev6, *next6;
	u_long		off = nl[N_TCB6].n_value;
#endif

	if (kvmd == NULL)
		return;
	if (nl[0].n_type == 0)
		if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
			return;
	if (kvm_read(kvmd, nl[N_TCB].n_value, (char *)&head,
		     sizeof(struct inpcbhead)) != sizeof(struct inpcbhead))
		return;

	for (next = head.lh_first; next != NULL;
	     next = inpcb.inp_list.le_next)
		{
		struct tcpcb tcpcb;

		if (kvm_read(kvmd, (u_long)next, (char *)&inpcb,
			     sizeof(inpcb)) != sizeof(inpcb))
			return;
		if (kvm_read(kvmd, (u_long)inpcb.inp_ppcb, (char *)&tcpcb,
			     sizeof(tcpcb)) != sizeof(tcpcb))
			return;
		tcp.local_port = ntohs(inpcb.inp_lport);
		tcp.remote_addr.s_addr = inpcb.inp_faddr.s_addr;
		tcp.remote_port = ntohs(inpcb.inp_fport);
		tcp_status = (tcpcb.t_state == TCPS_ESTABLISHED);
		tcp.family = AF_INET;
		if (tcp_status == TCP_ALIVE)
			log_tcp_port_data(&tcp);
		}
#if defined(INET6)
	if (kvm_read(kvmd, off, (char *)&in6pcb,
		         sizeof(struct in6pcb)) != sizeof(struct in6pcb))
		return;
	prev6 = (struct in6pcb *)off;
	if (in6pcb.in6p_next == (struct in6pcb *)off)
		return;
	while (in6pcb.in6p_next != (struct in6pcb *)off)
		{
		struct tcp6cb tcp6cb;

		next6 = in6pcb.in6p_next;
		if (kvm_read(kvmd, (u_long)next6, (char *)&in6pcb,
			     sizeof(in6pcb)) != sizeof(in6pcb))
			return;
		if (in6pcb.in6p_prev != prev6)
			break;
		if (kvm_read(kvmd, (u_long)in6pcb.in6p_ppcb, (char *)&tcp6cb,
			     sizeof(tcp6cb)) != sizeof(tcp6cb))
			return;
		tcp.local_port = ntohs(in6pcb.in6p_lport);
		memcpy(&tcp.remote_addr6, &in6pcb.in6p_faddr,
		       sizeof(struct in6_addr));
		tcp.remote_port = ntohs(in6pcb.in6p_fport);
		tcp_status = (tcp6cb.t_state == TCPS_ESTABLISHED);
		tcp.family = AF_INET6;
		if (tcp_status == TCP_ALIVE)
			log_tcp_port_data(&tcp);
		prev6 = next6;
		}
#endif
	}
#else
static void
read_freebsd_tcp_port_data()
	{
	static char	*name = "net.inet.tcp.pcblist";
	static int	oid_name2oid[2] = { 0, 3 };
	static int	oid_pcblist[CTL_MAXNAME + 2];
	static size_t	oid_pcblist_len = sizeof(oid_pcblist);
	static gint	initialized = 0;
	ActiveTCP	tcp;
	gint		tcp_status;
	struct xinpgen	*xig, *oxig;
	gchar		*buf;
	gint		len = 0;

	if (!initialized)
		{
		if (sysctl(oid_name2oid, 2, oid_pcblist, &oid_pcblist_len,
			   (void *)name, strlen(name)) < 0)
			return;
		oid_pcblist_len /= sizeof(int);
		++initialized;
		}
	if (sysctl(oid_pcblist, oid_pcblist_len, 0, &len, 0, 0) < 0)
		return;
	if ((buf = malloc(len)) == 0)
		return;
	if (sysctl(oid_pcblist, oid_pcblist_len, buf, &len, 0, 0) >= 0)
		{
		oxig = xig = (struct xinpgen *)buf;
		for (xig = (struct xinpgen *)((char *)xig + xig->xig_len);
		     xig->xig_len > sizeof(struct xinpgen);
		     xig = (struct xinpgen *)((char *)xig + xig->xig_len))
			{
			struct tcpcb *tp = &((struct xtcpcb *)xig)->xt_tp;
			struct inpcb *inp = &((struct xtcpcb *)xig)->xt_inp;
			struct xsocket *so = &((struct xtcpcb *)xig)->xt_socket;

			/* Ignore sockets for protocols other than tcp. */
			if (so->xso_protocol != IPPROTO_TCP)
				return;

			/* Ignore PCBs which were freed during copyout. */
			if (inp->inp_gencnt > oxig->xig_gen)
				return;

#if defined(INET6)
			if (inp->inp_vflag & INP_IPV4)
				{
				tcp.remote_addr.s_addr = inp->inp_faddr.s_addr;
				tcp.family = AF_INET;
				}
			else if (inp->inp_vflag & INP_IPV6)
				{
					memcpy(&tcp.remote_addr6,
					       &inp->in6p_faddr,
					       sizeof(struct in6_addr));
					tcp.family = AF_INET6;
				}
			else
				continue;
#else
			tcp.remote_addr.s_addr = inp->inp_faddr.s_addr;
			tcp.family = AF_INET;
#endif
			tcp.remote_port = ntohs(inp->inp_fport);
			tcp.local_port = ntohs(inp->inp_lport);
			tcp_status = (tp->t_state == TCPS_ESTABLISHED);
			if (tcp_status == TCP_ALIVE)
				log_tcp_port_data(&tcp);
			}
		}
	free(buf);
	}
#endif
#endif


/* ----- Linux -------------------------------------------------------- */
#if defined(__linux__)

#define	PROC_NET_TCP_FILE	"/proc/net/tcp"
#if defined(INET6)
#define	PROC_NET_TCP6_FILE	"/proc/net/tcp6"
#endif

static void
read_linux_tcp_port_data()
	{
	FILE		*f;
	ActiveTCP	tcp;
	gchar		buf[512];
	gint		tcp_status;
	gulong		addr;

	if ((f = fopen(PROC_NET_TCP_FILE, "r")) != NULL)
		{
		fgets(buf, sizeof(buf), f);		/* Waste first line */

		while (fgets(buf, sizeof(buf), f))
			{
			sscanf(buf, "%*d: %*x:%x %lx:%x %x", &tcp.local_port,
						&addr, &tcp.remote_port, &tcp_status);
			tcp.remote_addr.s_addr = (uint32_t) addr;
			tcp.family = AF_INET;
			if (tcp_status != TCP_ALIVE)
				continue;
			log_tcp_port_data(&tcp);
			}
		fclose(f);
		}
#if defined(INET6)
	if ((f = fopen(PROC_NET_TCP6_FILE, "r")) != NULL)
		{
		fgets(buf, sizeof(buf), f);		/* Waste first line */
		while (fgets(buf, sizeof(buf), f))
			{
			sscanf(buf, "%*d: %*x:%x %8x%8x%8x%8x:%x %x",
			       &tcp.local_port,
			       &tcp.remote_addr6.s6_addr32[0],
			       &tcp.remote_addr6.s6_addr32[1],
			       &tcp.remote_addr6.s6_addr32[2],
			       &tcp.remote_addr6.s6_addr32[3],
			       &tcp.remote_port, &tcp_status);
			tcp.family = AF_INET6;
			if (tcp_status != TCP_ALIVE)
				continue;
			log_tcp_port_data(&tcp);
			}
		fclose(f);
		}
#endif	/* INET6 */
	}
#endif	/* __linux__ */


/* ----- NetBSD --------------------------------------------------------- */

#if defined(__NetBSD__) || defined(__OpenBSD__) /* NO IPv6 SUPPORT YET */

#include <net/route.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/in_pcb.h>

#include <netinet/ip_var.h>
#include <netinet/tcp.h>
#include <netinet/tcpip.h>
#include <netinet/tcp_seq.h>
#include <netinet/tcp_fsm.h>
#include <netinet/tcp_timer.h>
#include <netinet/tcp_var.h>

#include <kvm.h>

static struct nlist nl[] = {
#define X_TCBTABLE    0
   { "_tcbtable" },
   { NULL }
};

extern	kvm_t	*kvmd;


void
read_netbsd_tcp_port_data()
{
   ActiveTCP tcp;
   gint	tcp_status;
   struct inpcbtable table;
   struct inpcb inpcb, *next;
   struct tcpcb tcpcb;

   if (kvmd == NULL) return;
   if (nl[0].n_type == 0)
      if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)	return;

   if (kvm_read(kvmd, nl[X_TCBTABLE].n_value, (char *)&table,
		sizeof(struct inpcbtable)) != sizeof(struct inpcbtable))
      return;

   next = table.inpt_queue.cqh_first;
   while(next != NULL) {
      
      if (kvm_read(kvmd, (u_long)next,
		   (char *)&inpcb, sizeof(inpcb)) == sizeof(inpcb) &&
	  kvm_read(kvmd, (u_long)inpcb.inp_ppcb,
		   (char *)&tcpcb, sizeof(tcpcb)) == sizeof(tcpcb)) {

	 tcp.local_port = ntohs(inpcb.inp_lport);
	 tcp.remote_addr.s_addr = inpcb.inp_faddr.s_addr;
	 tcp.remote_port = ntohs(inpcb.inp_fport);
	 tcp_status = (tcpcb.t_state == TCPS_ESTABLISHED);
	 tcp.family = AF_INET;
	 if (tcp_status == TCP_ALIVE)
	    log_tcp_port_data(&tcp);
      }

      next = inpcb.inp_queue.cqe_next;
   }
}

#endif /* __NetBSD__ || __OpenBSD__ */


/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_inet_interface()
    {
	gint	available = FALSE;

#if defined(__FreeBSD__)
	read_tcp_port_data = read_freebsd_tcp_port_data;
	available = TRUE;
#endif

#if defined(__linux__)
	read_tcp_port_data = read_linux_tcp_port_data;
	available = TRUE;
#endif

#if defined(__NetBSD__) || defined(__OpenBSD__)
	read_tcp_port_data = read_netbsd_tcp_port_data;
	available = TRUE;
#endif

    return available;
    }

/* ======================================================================== */


static GtkWidget	*inet_vbox;
static GdkImage		*grid;

static gint			n_inet_monitors;
static gint			e_ascent;
static gint			i_ascent;

static gint			style_id;

static void
draw_inet_extra(InetMon *in)
	{
	TextStyle	*ts, *ts_alt;
	gint		x0, x1, y;
	gchar		buf[16];

	y = 2;
	x0 = 4;
	ts = gkrellm_chart_textstyle(style_id);
	ts_alt = gkrellm_chart_alt_textstyle(style_id);

	if (e_ascent == 0)
		e_ascent = gdk_char_height(ts->font, '8');
	if (in->label0 && in->label0[0] != '\0')
		{
		snprintf(buf, sizeof(buf), "%d", in->active0);
		y += 2 + e_ascent;
		x0 = gkrellm_draw_chart_label(in->chart, ts, x0, y, buf);
		gkrellm_draw_chart_label(in->chart, ts_alt, x0 + 4, y,
					in->label0);
		}

	if (in->label1 && in->label1[0] != '\0')
		{
		snprintf(buf, sizeof(buf), "%d", in->active1);
		y += 2 + e_ascent;
		x1 = gkrellm_draw_chart_label(in->chart, ts, 4, y, buf);
		if (x1 < x0)
			x1 = x0;
		gkrellm_draw_chart_label(in->chart, ts_alt, x1 + 4, y,
						in->label1);
		}
	}

  /* Use the reserved area below the main chart to draw marks if any
  |  hits in second intervals.
  */
static void
draw_inet_mark_data(InetMon *in, gint minute_mark)
	{
	Chart	*cp;
	gint	hits, x, y, n;

	cp = in->chart;
	in->mark_position = (in->mark_position + 1) % cp->w;
	if (minute_mark)
		{
		in->mark_data[in->mark_position] = -1;	/* minute flag in the data */
		return;
		}
	hits = in->hits0 + in->hits1;
	in->mark_data[in->mark_position] = hits - in->mark_prev_hits;
	in->mark_prev_hits = hits;

	/* Clear out the area and redraw the marks.
	*/
	y = cp->h - cp->y;
	gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cp->bg_pixmap,
			0, y,  0, y,  cp->w, cp->y);
	gdk_gc_set_foreground(GK.draw1_GC, &GK.out_color);
	gdk_gc_set_foreground(GK.draw2_GC, &GK.in_color);
	gdk_gc_set_foreground(GK.draw3_GC, &GK.white_color);
	for (n = 0; n < cp->w; ++n)
		{
		x = (in->mark_position + n + 1) % cp->w;
		if (in->mark_data[x] > 0)
			gdk_draw_line(cp->pixmap, GK.draw1_GC,
						cp->x + n, cp->h - 1, cp->x + n, y);
		else if (in->mark_data[x] == -1)	/* Minute tick	*/
			gdk_draw_line(cp->pixmap, GK.draw3_GC,
						cp->x + n, cp->h - 1, cp->x + n, y);
		}
	gdk_draw_pixmap(cp->drawing_area->window, GK.draw1_GC, cp->pixmap,
			0, y,  0, y,  cp->w, cp->y);
	}

static gchar	*weekday_char[]	= { N_("S"), N_("M"), N_("T"), N_("W"), N_("T"), N_("F"), N_("S") };

static void
draw_inet_chart(InetMon *in)
	{
	Chart		*cp;
	TextStyle	*ts;
	GdkColor	tmp_color;
	guint32		pixel0, pixel1;
	gint		y0, h4, h, n, wday, hour, minute;

	/* Each draw of the hour chart must do a total redraw because the
	|  in/out_data_bitmaps are scrolled when the minute inet chart data
	|  is stored.  I need separate bitmaps for hour/minute XXX.
	*/
	cp = in->chart;
	if (cp == &in->chart_hour || GK.hour_tick)
		cp->scale_max = 0;

	gkrellm_draw_chart(cp);

	y0 = cp->h - cp->y;
	h4 = y0 / 4;
	h = ((GdkWindowPrivate *) in->chart->grid_pixmap)->height;
	if (grid == NULL)
		grid = gdk_image_get(in->chart->grid_pixmap, 0, 0, UC.chart_width, h);
	ts = gkrellm_chart_alt_textstyle(style_id);
	if (i_ascent == 0)
		i_ascent = gdk_char_height(ts->font, '8');

	gdk_gc_set_foreground(GK.draw3_GC, &GK.white_color);
	if (cp == &in->chart_hour)
		{
		hour = current_tm.tm_hour;
		wday = current_tm.tm_wday;
		for (n = cp->w - 1; n >= 0; --n)
			{
			/* When hour ticked to 0, 23rd hour data was stored and a slot
			|  was skipped.
			*/
			if (hour == 0)	/* Draw day mark at midnight	*/
				{
				pixel0 = gdk_image_get_pixel(grid, cp->x + n, 0);
				tmp_color.pixel = pixel0;
				gdk_gc_set_foreground(GK.draw3_GC, &tmp_color);
				gdk_draw_line(cp->pixmap, GK.draw3_GC,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, 3);
				gdk_draw_line(cp->drawing_area->window, GK.draw3_GC,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, 3);
				if (h > 1)
					{
					pixel1 = gdk_image_get_pixel(grid, cp->x + n, 1);
					tmp_color.pixel = pixel1;
					gdk_gc_set_foreground(GK.draw3_GC, &tmp_color);
					gdk_draw_line(cp->pixmap, GK.draw3_GC,
							cp->x + n, y0 - 3, cp->x + n, 3);
					gdk_draw_line(cp->drawing_area->window, GK.draw3_GC,
							cp->x + n, y0 - 3, cp->x + n, 3);
					}
				}
			if (hour == 1 && n < cp->w - 5)
				gkrellm_draw_chart_label(in->chart, ts,
						cp->x + n, i_ascent + 3, _(weekday_char[wday]));
			if (--hour < 0)
				{
				hour = 24;		/* Extra hour for skipped slot	*/
				if (--wday < 0)
					wday = 6;
				}
			}
		}
	if (cp == &in->chart_minute)
		{
		minute = current_tm.tm_min;
		for (n = cp->w - 1; n >= 0; --n)
			{
			/* When minute ticked to 0, 59 minute data was stored and a slot
			|  was skipped.
			*/
			if (minute == 0)	/* Draw hour mark	*/
				{
				pixel0 = gdk_image_get_pixel(grid, cp->x + n, 0);
				pixel1 = gdk_image_get_pixel(grid, cp->x + n, 1);

				tmp_color.pixel = pixel0;
				gdk_gc_set_foreground(GK.draw3_GC, &tmp_color);
				gdk_draw_line(cp->pixmap, GK.draw3_GC,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, y0 - h4);
				gdk_draw_line(cp->drawing_area->window, GK.draw3_GC,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, y0 - h4);

				tmp_color.pixel = pixel1;
				gdk_gc_set_foreground(GK.draw3_GC, &tmp_color);
				gdk_draw_line(cp->pixmap, GK.draw3_GC,
						cp->x + n, y0 - 3, cp->x + n, y0 - h4);
				gdk_draw_line(cp->drawing_area->window, GK.draw3_GC,
						cp->x + n, y0 - 3, cp->x + n, y0 - h4);
				}
			if (--minute < 0)
				minute = 60;	/* extra minute for skipped slot */
			}
		}
	if (in->extra_info)
		draw_inet_extra(in);
	in->prev_active0 = in->active0;
	in->prev_active1 = in->active1;
	}

static void
update_inet(void)
	{
	InetMon		*in;
	Chart		*cp;
	ActiveTCP	*tcp;
	GList		*list;
	gchar		buf[32];
	gint		i;

	if (inet_mon_list == NULL)
		return;

	if (GK.second_tick || GK.minute_tick || GK.hour_tick)
		{
		for (list = inet_mon_list; list; list = list->next)
			{
			in = (InetMon *) list->data;
			in->active0 = 0;
			in->active1 = 0;
			}
		/* Assume all connections are dead, then read_tcp_port_data() will set
		|  still alive ones back to alive.  Then I can prune really dead ones.
		*/
		for (list = active_tcp_list; list; list = list->next)
			{
			tcp = (ActiveTCP *)(list->data);
			tcp->state = TCP_DEAD;
			}

		(*read_tcp_port_data)();

		for (list = active_tcp_list; list; )
			{
			tcp = (ActiveTCP *)(list->data);
			if (tcp->state == TCP_DEAD)
				{
				if (list == active_tcp_list)
					active_tcp_list = active_tcp_list->next;
				list = g_list_remove(list, tcp);
				free_tcp_list = g_list_prepend(free_tcp_list, tcp);
				}
			else
				list = list->next;
			}
		}

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		cp = in->chart;
		if (GK.hour_tick)
			{
			gkrellm_store_chart_data(&in->chart_hour, in->hits1, in->hits0, 0);

			if (GK.day_tick)	/* Make room for vertical day grid */
				{
				gkrellm_store_chart_data(&in->chart_hour, in->hits1,
								in->hits0, 0);
				gkrellm_store_chart_data(&in->chart_hour, in->hits1,
								in->hits0, 0);
				}
			if (cp == &in->chart_hour)
				draw_inet_chart(in);
			}
		if (GK.minute_tick)
			{
			gkrellm_store_chart_data(&in->chart_minute, in->hits1,
							in->hits0, 0);

			if (GK.hour_tick)	/* Make room for vertical hour grid */
				{
				gkrellm_store_chart_data(&in->chart_minute, in->hits1,
								in->hits0, 0);
				gkrellm_store_chart_data(&in->chart_minute, in->hits1,
								in->hits0, 0);
				}
			if (cp == &in->chart_minute)
				draw_inet_chart(in);
			draw_inet_mark_data(in, 1);
			}
		else if (   GK.second_tick
				 && (   in->prev_active0 != in->active0
					 || in->prev_active1 != in->active1
					)
				)
			draw_inet_chart(in);	/* Just to update extra info draw */

		if (in->chart_minute.need_redraw)
			{
			draw_inet_chart(in);
			in->chart_minute.need_redraw = FALSE;
			}
		if (GK.second_tick)
			draw_inet_mark_data(in, 0);

		if (in->busy && in->list_button->cur_index == D_MISC_BUTTON_OUT)
			i = D_MISC_BUTTON_ON;
		else
			i = D_MISC_BUTTON_OUT;
		gkrellm_set_decal_button_index(in->list_button, i);

		gkrellm_update_krell(in->chart_minute.panel,
					KRELL(in->chart_minute.panel), in->hits0 + in->hits1);
		gkrellm_draw_layers(in->chart_minute.panel);

		if (in->connection_string_event)
			{
			snprintf(buf, sizeof(buf), _("%s Connections"), in->name);
			gkrellm_message_window(buf, in->connection_string->str, NULL);
			in->connection_string_event = FALSE;
			in->busy = FALSE;
			}
		}
	}

static gboolean
tcp_port_is_monitored(ActiveTCP *tcp, gboolean range, gulong p0, gulong p1)
	{
	if (   (!range && (p0 == tcp->local_port || p1 == tcp->local_port))
		|| ( range && tcp->local_port >= p0 && tcp->local_port <= p1)
	   )
		return TRUE;
	return FALSE;
	}


static void
get_connection_string_thread(void *data)
	{
	InetMon			*in	= (InetMon *) data;
	GList			*list;
	ActiveTCP		*tcp;
#if defined(INET6)
	struct sockaddr_storage	ss;
	struct sockaddr_in	*sin;
	struct sockaddr_in6	*sin6;
	gint			salen, flag = 0;
	gchar			hbuf[NI_MAXHOST];
	gchar			buf[NI_MAXHOST + 10];
#else
	struct hostent	*hostent;
	gchar			buf[64];
#endif

	if (in->connection_string)
		in->connection_string = g_string_truncate(in->connection_string, 0);
	else
		in->connection_string = g_string_new("");
	for (list = in->tcp_save_list; list; list = list->next)
		{
		tcp = (ActiveTCP *) list->data;
#if defined(INET6)
		memset(&ss, 0, sizeof(ss));
		switch (tcp->family)
			{
			case AF_INET:
				sin = (struct sockaddr_in *)&ss;
				salen = sizeof(struct sockaddr_in);
				memcpy(&sin->sin_addr,
						&tcp->remote_addr, salen);
#if defined(SIN6_LEN)
				sin->sin_len = salen;
#endif
				sin->sin_family = tcp->family;
				break;
			case AF_INET6:
				sin6 = (struct sockaddr_in6 *)&ss;
				salen = sizeof(struct sockaddr_in6);
				memcpy(&sin6->sin6_addr,
						&tcp->remote_addr6, salen);
#if defined(SIN6_LEN)
				sin6->sin6_len = salen;
#endif
				sin6->sin6_family = tcp->family;
				/* XXX: We should mention about
				|  scope, too. */
				break;
			default:
				continue;
			}
		if (getnameinfo((struct sockaddr *)&ss, salen,
					hbuf, sizeof(hbuf), NULL, 0, flag))
			continue;
		snprintf(buf, sizeof(buf), "%6d:  %s\n",
					tcp->local_port, hbuf);
#else
		hostent = gethostbyaddr((char *) &tcp->remote_addr,
					sizeof(struct in_addr), AF_INET);
		if (hostent)
			snprintf(buf, sizeof(buf),
				 "%6d:  %s\n", tcp->local_port,
				 hostent->h_name);
		else
			snprintf(buf, sizeof(buf),
				 "%6d:  %s\n", tcp->local_port,
				 inet_ntoa(tcp->remote_addr));
#endif
		g_string_append(in->connection_string, buf);
		}
	if (in->connection_string->len == 0)
		g_string_append(in->connection_string, _("No current connections."));
	in->connection_string_event = TRUE;
	free_glist_and_data(&in->tcp_save_list);
	}

static void
cb_list_button(DecalButton *button)
    {
	InetMon			*in	= (InetMon *) button->data;
	GList			*list;
	ActiveTCP		*tcp, *tcp_save;
	gpointer		thread_id;

	if (in->busy)
		return;
	in->busy = TRUE;

	/* Save a snapshot of active connections so I don't have to worry about
	|  the active_tcp_list changing while in the thread.
	*/
	for (list = active_tcp_list; list; list = list->next)
		{
		tcp = (ActiveTCP *) list->data;
		if (   tcp_port_is_monitored(tcp, in->data0_is_range,
						in->port0_0, in->port0_1)
			|| tcp_port_is_monitored(tcp, in->data1_is_range,
						in->port1_0, in->port1_1)
		   )
			{
			tcp_save = g_new0(ActiveTCP, 1);
			*tcp_save = *tcp;
			in->tcp_save_list = g_list_append(in->tcp_save_list, tcp_save);
			}
		}
	thread_id = new_thread(get_connection_string_thread, in);
	}

  /* Real pixmaps are always in chart_minute even if chart_hour is displayed.
  */
static gint
inet_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	InetMon		*in;
	GList		*list;
	Chart		*cp;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		cp = &in->chart_minute;
		if (widget == cp->panel->drawing_area)
			gdk_draw_pixmap(widget->window,GK.draw1_GC, cp->panel->pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
		if (widget == cp->drawing_area)
			{
			gdk_draw_pixmap(widget->window, GK.draw1_GC, cp->pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
			if (in->extra_info)
				draw_inet_extra(in);
			}
		}
	return FALSE;
	}

static gint
cb_inet_extra(GtkWidget *widget, GdkEventButton *event)
	{
	InetMon		*in;
	GList		*list;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		if (widget != in->chart->drawing_area)
			continue;
		if (event->button == 1)
			{
			in->extra_info = 1 - in->extra_info;
			in->prev_active0 = -1;	/* Force a redraw */
			draw_inet_chart(in);
			gkrellm_config_modified();
			}
		if (event->button == 2 || event->button == 3)
			{
			if (in->chart == &in->chart_minute)
				in->chart = &in->chart_hour;
			else
				in->chart = &in->chart_minute;
			in->chart->scale_max = 0;			/* Force a rescale */
			draw_inet_chart(in);
			}
		}
	return TRUE;
	}

static void
create_inet_monitor(GtkWidget *vbox1, InetMon *in, gint index,
			gint first_create)
	{
	GtkWidget	*vbox;
	Chart		*cp, *cp_hour;
	Panel		*p;
	Style		*style;
	gchar		buf[16];

	if (first_create)
		{
		vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(vbox1), vbox);
		in->vbox = vbox;

		snprintf(buf, sizeof(buf), _("inet%d"), index);
		in->name = g_strdup(buf);
		in->chart_minute.panel = gkrellm_panel_new0();
		in->chart_hour.panel = gkrellm_panel_new0();	/* XXX */
		in->chart = &in->chart_minute;
		}
	else
		{
		vbox = in->vbox;
		gkrellm_destroy_decal_list(in->chart_minute.panel);
		gkrellm_destroy_krell_list(in->chart_minute.panel);
		}
	cp = &in->chart_minute;
	cp_hour = &in->chart_hour;
	cp->name = in->name;
	p = cp->panel;
	in->idx = index;
	++n_inet_monitors;

	cp->scale_min = in->scale_min_minute;
	cp_hour->scale_min = in->scale_min_hour;
	cp->y = 3;

	style = gkrellm_panel_style(style_id);
	if (style->label_position == 50 && gkrellm_chart_width() < 80)
		style->label_position = 40;		/* Not a kludge, an adjustment! */
	in->list_decal = gkrellm_create_decal_pixmap(p,
				gkrellm_decal_misc_pixmap(), gkrellm_decal_misc_mask(),
				N_MISC_DECALS, style, -1, -1);

	gkrellm_create_krell(p, gkrellm_krell_panel_image(style_id), style);

	/* Inet krells are not related to chart scale_max.  Just give a constant
	|  full scale of 5.
	*/
	KRELL(cp->panel)->full_scale = 5;

	cp->h = UC.chart_height[MON_INET];
	gkrellm_create_chart(vbox, cp, style_id);

	p->textstyle = gkrellm_panel_textstyle(style_id);

	gkrellm_configure_panel(p, cp->name, style);
	gkrellm_create_panel(vbox, p, gkrellm_bg_panel_image(style_id));
	in->height = cp->h + p->h;
	gkrellm_monitor_height_adjust(in->height);

	in->list_button = gkrellm_make_decal_button(p, in->list_decal,
			cb_list_button, in, D_MISC_BUTTON_OUT, D_MISC_BUTTON_IN);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT (cp->drawing_area), "expose_event",
				(GtkSignalFunc) inet_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),"button_press_event",
				(GtkSignalFunc) cb_inet_extra, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),"expose_event",
				(GtkSignalFunc) inet_expose_event, NULL);
		gtk_widget_show_all(vbox);
		}
	gkrellm_setup_launcher(p, &in->launch, CHART_PANEL_TYPE, 4);

	gkrellm_alloc_chart_data(cp);

	/* Setup the hour chart.  Panel and krell are in the minute chart, so
	|  The chart_hour is going to be a skeleton chart with only stuff
	|  needed for plotting the hourly data.  The same pixmaps are used for
	|  minute or hour draws.
	*/
	cp_hour->name = cp->name;
	cp_hour->drawing_area = cp->drawing_area;
	cp_hour->pixmap = cp->pixmap;
	cp_hour->in_grided_pixmap = cp->in_grided_pixmap;
	cp_hour->out_grided_pixmap = cp->out_grided_pixmap;
	cp_hour->in_data_bitmap = cp->in_data_bitmap;
	cp_hour->out_data_bitmap = cp->out_data_bitmap;

	cp_hour->bg_pixmap = cp->bg_pixmap;
	cp_hour->bg_grided_pixmap = cp->bg_grided_pixmap;
	cp_hour->grid_pixmap = cp->grid_pixmap;
	cp_hour->x = cp->x;
	cp_hour->y = cp->y;
	cp_hour->w = cp->w;
	cp_hour->h = cp->h;
	if (in->mark_data)
		g_free(in->mark_data);
	in->mark_data = g_new0(gshort, cp->w);
	gkrellm_alloc_chart_data(cp_hour);

	/* Don't want to waste an hour priming the pump, and don't need to
	|  because data always starts at zero.
	*/
	in->chart_hour.scale_max = 0;			/* Force a rescale */
	in->chart_hour.primed = TRUE;

	in->chart_minute.scale_max = 0;			/* Force a rescale */
	in->chart_minute.primed = TRUE;
	if (! first_create)
		draw_inet_chart(in);
	}

static void
destroy_inet_monitor(InetMon *in)
	{
	in->chart = &in->chart_minute;	/* Everything was allocate here */
	g_free(in->name);
	g_free(in->label0);
	g_free(in->label1);
	if (in->launch.command)
		g_free(in->launch.command);
	if (in->launch.button)
		gkrellm_destroy_button(in->launch.button);
	gkrellm_monitor_height_adjust(-in->height);
	gkrellm_destroy_panel(in->chart->panel);
	g_free(in->chart->panel);
	g_free(in->chart_hour.panel);
    in->launch.tooltip = NULL;
	g_free(in->chart_hour.pDataIn);
	g_free(in->chart_hour.pDataOut);
	g_free(in->mark_data);
	gkrellm_destroy_chart(in->chart);
	gtk_widget_destroy(in->vbox);
	g_free(in);
	--n_inet_monitors;
	}

#define	INET_CONFIG_KEYWORD		"inet"

static void
save_inet_config(FILE *f)
	{
	GList	*list;
	InetMon	*in;
	gchar	*l0, *l1;
	gint	i;

	if (GK.demo)
		return;
	for (i = 0, list = inet_mon_list; list; list = list->next, ++i)
		{
		in = (InetMon *) list->data;
		l0 = (in->label0[0] == '\0') ? "NONE" : in->label0;
		l1 = (in->label1[0] == '\0') ? "NONE" : in->label1;
		fprintf(f, "%s %s %lu %lu %s %lu %lu %d %d %d %d %d\n",
				INET_CONFIG_KEYWORD,
				l0, in->port0_0, in->port0_1,
				l1, in->port1_0, in->port1_1,
				in->scale_min_minute, in->scale_min_hour, in->extra_info,
				in->data0_is_range, in->data1_is_range);
		fprintf(f, "%s launch %s\n", INET_CONFIG_KEYWORD, in->launch.command);
		fprintf(f, "%s tooltip_comment %s\n", INET_CONFIG_KEYWORD,
					in->launch.tooltip_comment);
		}
	}

static void
fix_port_order(InetMon *in)
	{
	gulong	tmp;

	if (in->data0_is_range)
		{
		if (in->port0_1 < in->port0_0)
			{
			tmp = in->port0_1;
			in->port0_1 = in->port0_0;
			in->port0_0 = tmp;
			}
		}
	if (in->data1_is_range)
		{
		if (in->port1_1 < in->port1_0)
			{
			tmp = in->port1_1;
			in->port1_1 = in->port1_0;
			in->port1_0 = tmp;
			}
		}
	}

static void
load_inet_config(gchar *arg)
	{
	static InetMon	*in_prev;
	InetMon			*in;
	gchar			config[32], item[CFG_BUFSIZE];
	gchar			label0[16], label1[16];
	gint			n;

	if ((n = sscanf(arg, "%31s %[^\n]", config, item)) != 2)
		return;
	if (in_prev && !strcmp(config, "launch"))
		gkrellm_dup_string(&in_prev->launch.command, item);
	else if (in_prev && !strcmp(config, "tooltip_comment"))
		gkrellm_dup_string(&in_prev->launch.tooltip_comment, item);
	else
		{
		in = g_new0(InetMon, 1);
		sscanf(arg, "%15s %lu %lu %15s %lu %lu %d %d %d %d %d",
				label0, &in->port0_0, &in->port0_1,
				label1, &in->port1_0, &in->port1_1,
				&in->scale_min_minute, &in->scale_min_hour, &in->extra_info,
				&in->data0_is_range, &in->data1_is_range);
		if (strcmp(label0, "NONE") == 0)
			{
			label0[0] = '\0';
			in->port0_0 = 0;
			in->port0_1 = 0;
			}
		if (strcmp(label1, "NONE") == 0)
			{
			label1[0] = '\0';
			in->port1_0 = 0;
			in->port1_1 = 0;
			}
		in->label0 = g_strdup(label0);
		in->label1 = g_strdup(label1);
		inet_mon_list = g_list_append(inet_mon_list, in);
		in_prev = in;
		fix_port_order(in);
		}
	}


static void
create_inet(GtkWidget *vbox, gint first_create)
	{
	GList		*list;
	gint		i;

	if (first_create)
		{
		inet_vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(vbox), inet_vbox);
		gtk_widget_show(inet_vbox);
		}
	if (inet_mon_list == NULL && GK.demo)
		load_inet_config("http 25 8080 ftp 21 0 2 2");
	if (grid)
		{
		gdk_image_destroy(grid);
		grid = NULL;
		}
	e_ascent = 0;
	i_ascent = 0;
	n_inet_monitors = 0;
	for (i = 0, list = inet_mon_list; list; ++i, list = list->next)
		create_inet_monitor(inet_vbox, (InetMon *)list->data, i, first_create);
	}



/* --------------------------------------------------------------------- */

  /* Read saved inet data (from a previous gkrellm process).  Return the
  |  number of missing data slots (skew).
  */
static gint
read_inet_data(Chart *cp, FILE *f, gint minute_chart,
			gint min, gint hour, gint yday, gint width)
	{
	gchar	data[64];
	gint	n, in, out, cur_slot, skew, day;

	day = current_tm.tm_yday - yday;

	/* Check for new years wrap around. I don't handle leap year here, will
	|  get some email, then be safe for four more years...
	*/
	if (day < 0)
		day = current_tm.tm_yday + ((yday < 365) ? 365 - yday : 0);

	cur_slot = day * 24 + current_tm.tm_hour;
	n = hour;
	if (minute_chart)
		{
		cur_slot = cur_slot * 60 + current_tm.tm_min;
		n = n * 60 + min;
		}
	skew = cur_slot - n;
	if (GK.debug_level & DEBUG_MISC)
		printf(_("read_inet_data() %d cur_slot=%d skew=%d\n"), minute_chart,
					cur_slot, skew);

	for (n = 0; n < width; ++n)
		{
		if (fgets(data, sizeof(data), f) == NULL)
			break;

		if (skew >= cp->w)	/* All stored data is off the chart	*/
			continue;

		/* Use chart data storing routines to load in data so I don't care
		|  if current chart width is less or greater than stored data width.
		|  Charts will circular buff fill until data runs out.
		*/
		sscanf(data, "%d %d", &in, &out);
		cp->prevIn  = 0;
		cp->prevOut = 0;
		gkrellm_store_chart_data(cp, out, in, 0);
		}
	/* Need to store zero data for time slots not in read data to bring
	|  the chart up to date wrt current time.  As in update_inet() I need
	|  to skip slots for hour or minute ticks.
	|  Warning: skew can be negative if quit gkrellm, change system clock
	|  to earlier time, then restart gkrellm.
	*/
	if ((n = skew) < cp->w)		/* Do this only if some data was stored  */
		{
		while (n-- > 0)
			{
			gkrellm_store_chart_data(cp, out, in, 0);
			if (minute_chart && min++ == 0)
				{
				gkrellm_store_chart_data(cp, out, in, 0);
				gkrellm_store_chart_data(cp, out, in, 0);
				if (min == 60)
					min = 0;
				}
			else if (!minute_chart && hour++ == 0)
				{
				gkrellm_store_chart_data(cp, out, in, 0);
				gkrellm_store_chart_data(cp, out, in, 0);
				if (hour == 24)
					hour = 0;
				}
			}
		}
	return skew;
	}

static void
write_inet_data(Chart *cp, FILE *f)
	{
	gchar	buf[64];
	gint	n, x;

	for (n = 0; n < cp->w; ++n)
		{
		x = computed_index(cp, n);
		snprintf(buf, sizeof(buf), "%d %d\n", cp->pDataIn[x], cp->pDataOut[x]);
		fputs(buf, f);
		}
	}

static gchar *
make_inet_data_fname(InetMon *in)
	{
	static gchar	idata_fname[256];
	gchar			c_sep0, c_sep1;

	c_sep0 = in->data0_is_range ? '-': '_';
	c_sep1 = in->data1_is_range ? '-': '_';
	snprintf(idata_fname, sizeof(idata_fname), "%s/%s/inet_%ld%c%ld_%ld%c%ld",
		gkrellm_homedir(), GKRELLM_DATA_DIR,
		in->port0_0, c_sep0, in->port0_1, in->port1_0, c_sep1, in->port1_1);
	return idata_fname;
	}

void
save_inet_data()
	{
	FILE		*f;
	GList		*list;
	InetMon		*in;
	gchar		*fname, buf[256];

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		fname = make_inet_data_fname(in);
		if ((f = fopen(fname, "w")) == NULL)
			continue;

		fputs("minute hour yday width\n", f);
		snprintf(buf, sizeof(buf), "%d %d %d %d\n", current_tm.tm_min,
				current_tm.tm_hour, current_tm.tm_yday, in->chart_minute.w);
		fputs(buf, f);

		/* Save any accumulated hits which have not been stored into the
		|  chart data array, and then save the chart data.
		*/
		fputs("hits0 hits1 min.prev0 min.prev1 hr.prev0 hr.prev1\n", f);
		snprintf(buf, sizeof(buf), "%ld %ld %ld %ld %ld %ld\n",
					in->hits0, in->hits1,
					in->chart_minute.prevIn, in->chart_minute.prevOut,
					in->chart_hour.prevIn, in->chart_hour.prevOut);
		fputs(buf, f);
		write_inet_data(&in->chart_minute, f);
		write_inet_data(&in->chart_hour, f);
		fclose(f);
		}
	}

void
load_inet_data()
	{
	FILE	*f;
	GList	*list;
	InetMon	*in;
	gchar	buf[96], *fname;
	gint	min, hour, yday, len, skew;
	gulong	min_prevIn, min_prevOut, hr_prevIn, hr_prevOut;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		fname = make_inet_data_fname(in);
		if ((f = fopen(fname, "r")) == NULL)
			{
			gkrellm_reset_chart_data(&in->chart_minute);
			gkrellm_reset_chart_data(&in->chart_hour);
			draw_inet_chart(in);
			continue;
			}
		if (GK.debug_level & DEBUG_MISC)
			printf(_("Loading %s\n"), fname);
		fgets(buf, sizeof(buf), f);		/* Comment line */
		fgets(buf, sizeof(buf), f);
		sscanf(buf, "%d %d %d %d", &min, &hour, &yday, &len);
		fgets(buf, sizeof(buf), f);		/* Comment line */
		fgets(buf, sizeof(buf), f);
		sscanf(buf, "%ld %ld %ld %ld %ld %ld", &in->hits0, &in->hits1,
					&min_prevIn, &min_prevOut, &hr_prevIn, &hr_prevOut);

		skew = read_inet_data(&in->chart_minute, f, 1, min, hour, yday, len);
		if (skew > 0)  /* Current minute slot is different from saved */
			{
			in->chart_minute.prevIn = in->hits0;
			in->chart_minute.prevOut = in->hits1;
			}
		else  /* saved hit0 and hit1 data is valid for current min slot */
			{
			in->chart_minute.prevIn = min_prevIn;
			in->chart_minute.prevOut = min_prevOut;
			}

		skew = read_inet_data(&in->chart_hour, f, 0, min, hour, yday, len);
		if (skew > 0)  /* Current hour slot is different from saved */
			{
			in->chart_hour.prevIn = in->hits0;
			in->chart_hour.prevOut = in->hits1;
			}
		else  /* saved hit0 and hit1 data is valid for current hour slot */
			{
			in->chart_hour.prevIn = hr_prevIn;
			in->chart_hour.prevOut = hr_prevOut;
			}
		fclose(f);
		in->chart_minute.scale_max = 0;
		in->chart_hour.scale_max = 0;
		draw_inet_chart(in);
		}
	}

/* --------------------------------------------------------------------- */

static GtkWidget	*inet_clist;
static gint			selected_row,
					inet_list_modified;


static GtkWidget	*label0_entry,
					*label1_entry;
static GtkWidget	*port0_0_entry,
					*port0_1_entry,
					*port1_0_entry,
					*port1_1_entry;

static GtkWidget	*launch_vbox,
					*minute_spinbutton,
					*hour_spinbutton;

static GtkWidget	*data0_range_button,
					*data1_range_button;

static gint		min_res_temp,
				hr_res_temp;

static gint	inet_map[] =
	{
	2, 5,
	10, 20, 50,
	100, 200, 500,
	1000, 2000, 5000,
	10000, 20000, 50000
	};


static void
cb_inet_resolution(GtkWidget *widget, GtkSpinButton *spin)
	{
	gchar	smbuf[16], shbuf[16];
	gint	sm, sh;
	gint	value, *ref;

	value = gtk_spin_button_get_value_as_int(spin);
	if (spin == GTK_SPIN_BUTTON(minute_spinbutton))
		ref = &min_res_temp;
	else
		ref = &hr_res_temp;
	if (value != *ref)		/* Avoid recursion */
		{
		value = map_1_2_5(value, inet_map, sizeof(inet_map)/sizeof(int));
		*ref = value;
		gtk_spin_button_set_value(spin, (gfloat) value);

		if (selected_row >= 0)
			{
			sm = gtk_spin_button_get_value_as_int(
					GTK_SPIN_BUTTON(minute_spinbutton));
			sh = gtk_spin_button_get_value_as_int(
					GTK_SPIN_BUTTON(hour_spinbutton));
			snprintf(smbuf, sizeof(smbuf), "%d", sm);
			snprintf(shbuf, sizeof(shbuf), "%d", sh);
			gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 8, smbuf);
			gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 9, shbuf);
			inet_list_modified = TRUE;
#if 0
			/* If I can correspond selected row to an existing monitor, go
			|  ahead and let the charts track??
			*/
			in->chart_hour.scale_min = sm;
			in->chart_minute.scale_min = sh;
			in->chart->scale_max = 0;
			draw_inet_chart(in);
#endif
			}
		}
	}

static void
reset_inet_entries()
	{
	gtk_entry_set_text(GTK_ENTRY(label0_entry), "");
	gtk_entry_set_text(GTK_ENTRY(port0_0_entry), "0");
	gtk_entry_set_text(GTK_ENTRY(port0_1_entry), "0");
	gtk_entry_set_text(GTK_ENTRY(label1_entry), "");
	gtk_entry_set_text(GTK_ENTRY(port1_0_entry), "0");
	gtk_entry_set_text(GTK_ENTRY(port1_1_entry), "0");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data0_range_button), 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data1_range_button), 0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(minute_spinbutton), 2);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(hour_spinbutton), 2);
	min_res_temp = 0;
	hr_res_temp = 0;
	}

static void
cb_inet_clist_selected(GtkWidget *clist, gint row, gint column,
		GdkEventButton *bevent, gpointer data)
	{
	gchar	*s;

	gtk_clist_get_text(GTK_CLIST(clist), row, 0, &s);
	gtk_entry_set_text(GTK_ENTRY(label0_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 1, &s);
	gtk_entry_set_text(GTK_ENTRY(port0_0_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 2, &s);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data0_range_button),
			!strcmp(s, "-") ? TRUE : FALSE);
	gtk_clist_get_text(GTK_CLIST(clist), row, 3, &s);
	gtk_entry_set_text(GTK_ENTRY(port0_1_entry), s);

	gtk_clist_get_text(GTK_CLIST(clist), row, 5, &s);
	gtk_entry_set_text(GTK_ENTRY(label1_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 6, &s);
	gtk_entry_set_text(GTK_ENTRY(port1_0_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 7, &s);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data1_range_button),
			!strcmp(s, "-") ? TRUE : FALSE);
	gtk_clist_get_text(GTK_CLIST(clist), row, 8, &s);
	gtk_entry_set_text(GTK_ENTRY(port1_1_entry), s);

	gtk_clist_get_text(GTK_CLIST(clist), row, 10, &s);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(minute_spinbutton), atoi(s));
	gtk_clist_get_text(GTK_CLIST(clist), row, 11, &s);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(hour_spinbutton), atoi(s));

	selected_row = row;
	}

static void
cb_inet_clist_unselected(GtkWidget *clist, gint row, gint column,
		GdkEventButton *bevent, gpointer data)
	{
	selected_row = -1;
	reset_inet_entries();
	}

static void
cb_enter_inet(GtkWidget *widget, gpointer data)
	{
	gchar	*buf[13], smbuf[16], shbuf[16];
	gint	sm, sh, data0_is_range, data1_is_range;

	data0_is_range = GTK_TOGGLE_BUTTON(data0_range_button)->active;
	data1_is_range = GTK_TOGGLE_BUTTON(data1_range_button)->active;
	buf[0] = gkrellm_entry_get_text(&label0_entry);
	buf[1] = gkrellm_entry_get_text(&port0_0_entry);
	buf[2] = data0_is_range ? "-" : ",";
	buf[3] = gkrellm_entry_get_text(&port0_1_entry);
	buf[4] = "";
	buf[5] = gkrellm_entry_get_text(&label1_entry);
	buf[6] = gkrellm_entry_get_text(&port1_0_entry);
	buf[7] = data1_is_range ? "-" : ",";
	buf[8] = gkrellm_entry_get_text(&port1_1_entry);
	buf[9] = "";
	sm = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(minute_spinbutton));
	sh = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(hour_spinbutton));
	snprintf(smbuf, sizeof(smbuf), "%d", sm);
	snprintf(shbuf, sizeof(shbuf), "%d", sh);
	buf[10] = smbuf;
	buf[11] = shbuf;
	buf[12] = NULL;

	/* Validate the values
	*/
	if (   (*buf[0] == '\0' && *buf[5] == '\0')
		|| ((*buf[0] != '\0' && atoi(buf[1]) == 0 && atoi(buf[3]) == 0))
		|| ((*buf[5] != '\0' && atoi(buf[6]) == 0 && atoi(buf[8]) == 0))
	   )
		return;

	if (selected_row >= 0)
		{
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 0, buf[0]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 1, buf[1]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 2, buf[2]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 3, buf[3]);

		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 5, buf[5]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 6, buf[6]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 7, buf[7]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 8, buf[8]);

		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 10, buf[10]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 11, buf[11]);
		gtk_clist_unselect_row(GTK_CLIST(inet_clist), selected_row, 0);
		selected_row = -1;
		}
	else
		gtk_clist_append(GTK_CLIST(inet_clist), buf);
	reset_inet_entries();
	inet_list_modified = TRUE;
	}

static void
cb_delete_inet(GtkWidget *widget, gpointer data)
	{
	reset_inet_entries();
	if (selected_row >= 0)
		{
		gtk_clist_remove(GTK_CLIST(inet_clist), selected_row);
		inet_list_modified = TRUE;
		selected_row = -1;
		}
	}

static void
add_launch_entry(GtkWidget *vbox, InetMon *in)
	{
	GtkWidget	*table;

	table = gkrellm_launcher_table_new(vbox, 1);
	gkrellm_config_launcher(table, 0,  &in->launch_entry, &in->tooltip_entry,
				in->name, &in->launch);
	in->launch_table = table;
	gtk_widget_show_all(table);
	}

static void
apply_inet_config()
	{
	InetMon	*in;
	GList	*list, *new_inet_list, *old_inet_list;
	gchar	*s;
	gint	row, need_new_monitor;


	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		if (in->launch_entry)
			gkrellm_apply_launcher(&in->launch_entry, &in->tooltip_entry,
					in->chart->panel, &in->launch, gkrellm_launch_button_cb);
		}

	if (! inet_list_modified)
		{	/* Rescale and redraw in case grid mode has changed */
		for (list = inet_mon_list; list; list = list->next)
			{
			in = (InetMon *) list->data;
			in->chart_minute.scale_max = 0;
			in->chart_hour.scale_max = 0;
			draw_inet_chart(in);
			}
		return;
		}

	/* Just save all data and then later read it back in.  This avoids
	|  complicated detecting of name changes while ports the same, moving
	|  a inet down or up slots, etc.  Data is lost only if a port number
	|  for a monitor is changed.
	*/
	save_inet_data();
	new_inet_list = NULL;
	old_inet_list = inet_mon_list;
	for (row = 0; row < GTK_CLIST(inet_clist)->rows; ++row)
		{
		if (old_inet_list)
			{
			in = (InetMon *) old_inet_list->data;
			g_free(in->label0);
			g_free(in->label1);
			old_inet_list = old_inet_list->next;
			need_new_monitor = FALSE;
			}
		else
			{
			in = g_new0(InetMon, 1);
			need_new_monitor = TRUE;
			}
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 0, &s);
		in->label0 = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 1, &s);
		in->port0_0 = atoi(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 2, &s);
		in->data0_is_range = (!strcmp(s, "-") ? TRUE : FALSE);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 3, &s);
		in->port0_1 = atoi(s);

		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 5, &s);
		in->label1 = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 6, &s);
		in->port1_0 = atoi(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 7, &s);
		in->data1_is_range = (!strcmp(s, "-") ? TRUE : FALSE);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 8, &s);
		in->port1_1 = atoi(s);

		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 10, &s);
		in->scale_min_minute = atoi(s);
		in->chart_minute.scale_min = in->scale_min_minute;
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 11, &s);
		in->scale_min_hour = atoi(s);
		in->chart_hour.scale_min = in->scale_min_hour;

		new_inet_list = g_list_append(new_inet_list, in);
		fix_port_order(in);
		if (need_new_monitor)
			{
			create_inet_monitor(inet_vbox, in, n_inet_monitors, TRUE);
			draw_inet_chart(in);
			add_launch_entry(launch_vbox, in);
			}
		in->extra_info = TRUE;
		}
	while (old_inet_list)	/* Number of inet monitors went down. */
		{
		in = (InetMon *) old_inet_list->data;
		gtk_widget_destroy(in->launch_table);
		destroy_inet_monitor(in);
		old_inet_list = old_inet_list->next;
		}
	while (inet_mon_list)	/* Destroy the old glist */
		inet_mon_list = g_list_remove(inet_mon_list, inet_mon_list->data);
	inet_mon_list = new_inet_list;
	gkrellm_pack_side_frames();
	load_inet_data();
	inet_list_modified = FALSE;
	}

static gchar	*inet_info_text[] =
{
N_("Inet charts show historical TCP port hits on a minute or hourly\n"
	"chart. Below the chart there is a strip where marks are drawn for\n"
	"port hits in second intervals.   The inet krell has a full scale\n"
	"value of 5 hits and samples once per second.  The extra info\n"
	"display shows current TCP port connections.\n\n"
	"For each internet monitor, you can specify two labeled data sets with\n"
	"one or two non-zero port numbers entered for each data set.  Two\n"
	"ports are allowed because some internet ports are related and you\n"
	"might want to group them.  Check /etc/services for port numbers.\n\n"
	"For example, if you created an inet monitor:\n"),

N_("<i>\thttp 80 8080   ftp 21\n"),

N_("Http hits on the standard http port 80 and www web caching service\n"
	"on port 8080 are combined and plotted in the \"in\" color.  Ftp hits\n"
	"on the single ftp port 21 are plotted in the \"out\" color.\n\n"),

N_("If the range button is checked, then all port numbers between Port0 and\n"
   "Port1 are monitored and included in the plot.\n\n"),

N_("Set chart grid resolutions for the minute and hour charts with the\n"
	"hits/minute and hits/hour spin buttons.  Chart data is saved when\n"
	"GKrellM exits and loaded at startup.\n\n"),


N_("<b>Mouse Button Actions:\n"),
N_("<b>\tLeft "),
N_("click on an inet chart to toggle the extra info display of\n"
	"\t\tcurrent TCP port connections.\n"),
N_("<b>\tMiddle or Right "),
N_("click on an inet chart to toggle hour/minute charts.\n")
};


static gchar	*inet_titles[12] =
	{
	N_("Label"), N_("Port0"), "     ", N_("Port1"), "          ",
	N_("Label"), N_("Port0"), "     ", N_("Port1"), "          ",
		N_("Hits/min"), N_("Hits/hr")
	};

static void
create_inet_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*table;
	GtkWidget		*hbox, *vbox;
	GtkWidget		*separator;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	GtkWidget		*label;
	GtkWidget		*button;
	GList			*list;
	InetMon			*in;
	gchar			*buf[13];
	gchar			p00[16], p01[16], p10[16], p11[16], resm[16], resh[16];
	gint			i;
	
	inet_list_modified = FALSE;
	selected_row = -1;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

	vbox = gkrellm_create_tab(tabs, _("Ports"));

	table = gtk_table_new(7, 6, FALSE /*homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 3);

	label = gtk_label_new(_("Data0"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 3, 0, 1);
	separator = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 0, 3, 1, 2);
	label = gtk_label_new(_("Data1"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 4, 7, 0, 1);
	separator = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 4, 7, 1, 2);

	separator = gtk_vseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 3, 4, 0, 6);

	label = gtk_label_new(_("Label"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	label = gtk_label_new(_("Port0"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 2, 3);
	label = gtk_label_new(_("Port1"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 2, 3);
	label = gtk_label_new(_("Label"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 4, 5, 2, 3);
	label = gtk_label_new(_("Port0"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 5, 6, 2, 3);
	label = gtk_label_new(_("Port1"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 6, 7, 2, 3);

	label0_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(label0_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), label0_entry, 0, 1, 3, 4);
	port0_0_entry = gtk_entry_new_with_max_length(5);
	gtk_widget_set_usize(port0_0_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port0_0_entry, 1, 2, 3, 4);
	port0_1_entry = gtk_entry_new_with_max_length(5);
	gtk_widget_set_usize(port0_1_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port0_1_entry, 2, 3, 3, 4);

	label1_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(label1_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), label1_entry, 4, 5, 3, 4);
	port1_0_entry = gtk_entry_new_with_max_length(5);
	gtk_widget_set_usize(port1_0_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port1_0_entry, 5, 6, 3, 4);
	port1_1_entry = gtk_entry_new_with_max_length(5);
	gtk_widget_set_usize(port1_1_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port1_1_entry, 6, 7, 3, 4);

	hbox = gtk_hbox_new(FALSE, 2);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 3, 4, 5);
	gkrellm_check_button(hbox, &data0_range_button, 0, TRUE, 0,
		_("Port0 - Port1 is a range"));

	hbox = gtk_hbox_new(FALSE, 2);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 4, 7, 4, 5);
	gkrellm_check_button(hbox, &data1_range_button, 0, TRUE, 0,
		_("Port0 - Port1 is a range"));

	separator = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 0, 7, 5, 6);

	hbox = gtk_hbox_new(FALSE, 3);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);

	gkrellm_spin_button(hbox, &minute_spinbutton, 2.0, 2.0, 100000.0, 1.0, 5.0,
			0, 50, cb_inet_resolution, NULL, FALSE, _("Hits/minute"));

	gkrellm_spin_button(hbox, &hour_spinbutton, 2.0, 2.0, 100000.0, 1.0, 5.0,
			0, 50, cb_inet_resolution, NULL, FALSE, _("Hits/hour"));

	button = gtk_button_new_with_label(_("Enter"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) cb_enter_inet, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);

	button = gtk_button_new_with_label(_("Delete"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) cb_delete_inet, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 2);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
    for(i = 0; i < 12; i++)
		inet_titles[i] = _(inet_titles[i]);
	inet_clist = gtk_clist_new_with_titles(12, inet_titles);
	gtk_clist_set_shadow_type (GTK_CLIST(inet_clist), GTK_SHADOW_OUT);
	gtk_clist_set_column_width (GTK_CLIST(inet_clist), 0, 32);
	gtk_signal_connect (GTK_OBJECT(inet_clist), "select_row",
			(GtkSignalFunc) cb_inet_clist_selected, NULL);
	gtk_signal_connect (GTK_OBJECT(inet_clist), "unselect_row",
			(GtkSignalFunc) cb_inet_clist_unselected, NULL);
	gtk_container_add (GTK_CONTAINER (scrolled), inet_clist);
	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		snprintf(p00, sizeof(p00), "%d", (int) in->port0_0);
		snprintf(p01, sizeof(p01), "%d", (int) in->port0_1);
		snprintf(p10, sizeof(p10), "%d", (int) in->port1_0);
		snprintf(p11, sizeof(p11), "%d", (int) in->port1_1);
		snprintf(resm, sizeof(resm), "%d", in->scale_min_minute);
		snprintf(resh, sizeof(resh), "%d", in->scale_min_hour);
		buf[0] = in->label0;
		buf[1] = p00;
		buf[2] = in->data0_is_range ? "-" : ",";
		buf[3] = p01;
		buf[4] = "";
		buf[5] = in->label1;
		buf[6] = p10;
		buf[7] = in->data1_is_range ? "-" : ",";
		buf[8] = p11;
		buf[9] = "";
		buf[10] = resm;
		buf[11] = resh;
		buf[12] = NULL;
		gtk_clist_append(GTK_CLIST(inet_clist), buf);
		}

/* --Launch tab */
	launch_vbox = gkrellm_create_tab(tabs, _("Launch"));
	insert_expanded_filler(launch_vbox);
	for (i = 0, list = inet_mon_list; list; list = list->next, ++i)
		{
		in = (InetMon *) list->data;
		add_launch_entry(launch_vbox, in);
		}

/* --Info tab */
	vbox = gkrellm_create_tab(tabs, _("Info"));
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	text = gtk_text_new(NULL, NULL);
	for (i = 0; i < sizeof(inet_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(inet_info_text[i]));
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_container_add(GTK_CONTAINER(scrolled), text);

	reset_inet_entries();
	}



static Monitor	monitor_inet =
	{
	N_("Internet"),			/* Name, for config tab.	*/
	MON_INET,			/* Id,  0 if a plugin		*/
	create_inet,		/* The create function		*/
	update_inet,		/* The update function		*/
	create_inet_tab,	/* The config tab create function	*/
	apply_inet_config,	/* Apply the config function		*/

	save_inet_config,	/* Save user conifg			*/
	load_inet_config,	/* Load user config			*/
	"inet",				/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_inet_monitor(void)
	{
	monitor_inet.name = _(monitor_inet.name);
	style_id = gkrellm_add_chart_style(&monitor_inet, INET_STYLE_NAME);
	if (setup_inet_interface())
		return &monitor_inet;
	return FALSE;
	}
