/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* 10/12/2000	NetBSD code contributed by Anthony Mallet
|			<metall@ficus.yi.org>
| 2/25/2000	FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/


#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include "cpu_disk.h"

#define	DEFAULT_RESOLUTION	200

void	(*read_system_disk_info)();

static DiskMon	*lookup_disk_by_name(gchar *);

gint		n_disks;
gboolean	using_DiskN_names;
GList		*disk_mon_list;



/* ====== System dependent interface ====================================== */

/* ----- FreeBSD ----------------------------------------------------- */

#if defined(__FreeBSD__)
#include <osreldate.h>
#include <sys/dkstat.h>
#if __FreeBSD_version >= 300000
#include <devstat.h>
static struct statinfo	statinfo_cur;
#else
#include <kvm.h>
static struct nlist nl[] = {
#define N_DK_NDRIVE	0
	{ "_dk_ndrive" },
#define N_DK_XFER	1
	{ "_dk_xfer" },
	{ "" }
};

extern	kvm_t	*kvmd;
#endif

static DiskMon *
add_disk_device(gint major, gint minor)
	{
	return NULL;
	}

#if __FreeBSD_version < 300000
static void
read_freebsd_disk()
	{
	static gint	data_read_tick = -1;
	int		ndevs;
	long		*cur_dk_xfer;
	int		dn;
	GList		*list;
	DiskMon		*disk;
   
	if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
		return;
	data_read_tick = GK.timer_ticks;

	if (kvmd == NULL)
		return;
	if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
		return;
	(void) kvm_read(kvmd, nl[N_DK_NDRIVE].n_value,
			(char *)&ndevs, sizeof(ndevs));
	if (ndevs <= 0)
		return;
	if ((cur_dk_xfer = calloc(ndevs, sizeof(long))) == NULL)
		return;
	if (kvm_read(kvmd, nl[N_DK_XFER].n_value, (char *)cur_dk_xfer,
		     ndevs * sizeof(long)) == ndevs * sizeof(long))
		{
		composite_disk->rblk = 0;
		composite_disk->wblk = 0;
		dn = 0;
		for (list = disk_mon_list->next; list; list = list->next)
			{
			if (dn >= ndevs)
				break;
			disk = (DiskMon *) list->data;
			disk->rblk = 0;
			disk->wblk = cur_dk_xfer[dn];
			composite_disk->wblk += disk->wblk;
			++dn;
			}
		}
	free(cur_dk_xfer);
	}

static void
register_disks(gboolean re_checking)
	{
	gint	i;
	DiskMon	*disk;

	if (re_checking)
		return;
	if (kvmd == NULL)
		return;
	if (kvm_nlist(kvmd, nl) >= 0 && nl[0].n_type != 0)
		(void) kvm_read(kvmd, nl[N_DK_NDRIVE].n_value,
				(char *)&n_disks, sizeof(n_disks));
	for (i = 0; i < n_disks; ++i)
		{
		disk = g_new0(DiskMon, 1);
		disk_mon_list = g_list_append(disk_mon_list, disk);
		disk->name = g_strdup_printf("%s%c", _("Disk"), 'A' + i);
		disk->resolution = DEFAULT_RESOLUTION;
		}
	using_DiskN_names = TRUE;
	}
#else
static void
read_freebsd_disk()
	{
	static gint		data_read_tick = -1;
	int			ndevs;
	int			num_selected;
	int			num_selections;
	int			maxshowdevs = 10;
	struct device_selection	*dev_select = NULL;
	long			select_generation;
	int			dn;
	DiskMon			*disk;

	if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
		return;
	data_read_tick = GK.timer_ticks;

	if (getdevs(&statinfo_cur) < 0)
		return;
	ndevs = statinfo_cur.dinfo->numdevs;
	if (selectdevs(&dev_select, &num_selected, &num_selections,
		       &select_generation, statinfo_cur.dinfo->generation,
		       statinfo_cur.dinfo->devices, ndevs,
		       NULL, 0, NULL, 0,
		       DS_SELECT_ONLY, maxshowdevs, 1) >= 0)
		{
		composite_disk->rblk = 0;
		composite_disk->wblk = 0;
		for (dn = 0; dn < ndevs; ++dn)
			{
			int		di;
			struct devstat	*dev;
			int		block_size;
			int		blocks_read, blocks_written;

			di = dev_select[dn].position;
			dev = &statinfo_cur.dinfo->devices[di];
			block_size = (dev->block_size > 0)
				   ? dev->block_size : 512;
			blocks_read = dev->bytes_read / block_size;
			blocks_written = dev->bytes_written / block_size;

			if ((disk = lookup_disk_by_device(dev->device_number,
							  0)) != NULL)
				{
				disk->rblk = blocks_read;
				disk->wblk = blocks_written;
				}
			composite_disk->rblk += blocks_read;
			composite_disk->wblk += blocks_written;
			}
		free(dev_select);
		}
	}

static void
register_disks(gboolean re_checking)
	{
	DiskMon			*disk;
	GList			*list;
	static gint		initted = FALSE;
	int			num_selected;
	int			num_selections;
	int			maxshowdevs = 10;
	struct device_selection	*dev_select = NULL;
	long			select_generation;
	gint			dn, di, i;
	struct devstat		*dev;

	if (!initted)
		{
		n_disks = getnumdevs();
		bzero(&statinfo_cur, sizeof(statinfo_cur));
		statinfo_cur.dinfo = (struct devinfo *)malloc(sizeof(struct devinfo));
		bzero(statinfo_cur.dinfo, sizeof(struct devinfo));
		initted = TRUE;
		}
	if (getdevs(&statinfo_cur) < 0)
		return;
	n_disks = statinfo_cur.dinfo->numdevs;
	if (selectdevs(&dev_select, &num_selected, &num_selections,
		       &select_generation,
		       statinfo_cur.dinfo->generation,
		       statinfo_cur.dinfo->devices, n_disks,
		       NULL, 0, NULL, 0,
		       DS_SELECT_ONLY, maxshowdevs, 1) < 0)
		return;
	for (dn = 0; dn < n_disks; ++dn)
		{
		di = dev_select[dn].position;
		dev = &statinfo_cur.dinfo->devices[di];
		if (lookup_disk_by_device(dev->device_number, 0) != NULL)
			continue;
		disk = g_new0(DiskMon, 1);
		disk->name = g_strdup_printf("%s%d", dev->device_name,
					     dev->unit_number);
		disk->resolution = DEFAULT_RESOLUTION;
		disk->major = dev->device_number;
		disk->minor = 0;
		disk->order = dev->device_type + 1;	/* Skip the composite disk */
		i = 1;
		for (list = disk_mon_list->next; list; list = list->next, ++i)
			if (disk->order < ((DiskMon *) list->data)->order)
				break;
		disk_mon_list = g_list_insert(disk_mon_list, disk, i);
		}
	free(dev_select);
	using_DiskN_names = TRUE;
	}
#endif
#endif


/* ----- Linux ----------------------------------------------------- */

/* See array.c (2.2.x) or proc_misc.c (2.3.99+) in /usr/src/linux/fs/proc
|  to see how info for disks is limited to 4 devices.
|  See drive_stat_acct() in in /usr/src/linux/drivers/block/ll_rw_blk.c
|  to see how device numbers for ide and scsi are mapped into one of the
|  four disk devices reported in /proc.
|  The devices currently reported are DAC960_MAJOR (55), SCSI_DISK0_MAJOR (8),
|  IDE0_MAJOR (3), and IDE1_MAJOR (22)
*/

#if defined(__linux__)
#define	PROC_STAT_FILE	"/proc/stat"

#include <linux/major.h>
#if ! defined (SCSI_DISK0_MAJOR)
#define SCSI_DISK0_MAJOR	8
#endif
#if ! defined (MD_MAJOR)
#define MD_MAJOR	9
#endif

/* The read_proc_stat() routine is in cpu.c		*/

struct _disk_name_map
	{
	gchar	*name;
	gint	major;
	gchar	suffix_base;
	};

  /* Disk charts will appear in GKrellm in the same order as this table.
  */
struct _disk_name_map	disk_name_map[] =
	{
	{"hd",	IDE0_MAJOR,			'a' },		/* 3:  hda, hdb */
	{"hd",	IDE1_MAJOR,			'c' },		/* 22: hdc, hdd */
	{"hd",	IDE2_MAJOR,			'e' },		/* 33: hde, hdf */
	{"hd",	IDE3_MAJOR,			'g' },		/* 34: hdg, hdh */
	{"sd",	SCSI_DISK0_MAJOR,	'a' },		/* 8:  sda-sdh */
	{"sg",	SCSI_GENERIC_MAJOR,	'0' },		/* 21: sg0-sg16 */
	{"scd",	SCSI_CDROM_MAJOR,	'0' },		/* 11: scd0-scd16 */
	{"md",	MD_MAJOR,			'0' },		/* 9:  md0-md3 */
	{"fd",	FLOPPY_MAJOR,		'0' },		/* 2:  fd0-fd3  */
	};

static DiskMon *
add_disk_device(gint major, gint i_disk)
	{
	struct _disk_name_map	*dm	= NULL;
	DiskMon	*disk;
	GList	*list;
	gint	i;
	gchar	buf[32], suffix;

	for (i = 0; i < sizeof(disk_name_map) / sizeof(struct _disk_name_map); ++i)
		{
		if (major == disk_name_map[i].major)
			{
			dm = &disk_name_map[i];
			break;
			}
		}
	if (dm)
		{
		suffix = dm->suffix_base + i_disk;
		sprintf(buf, "%s%c", dm->name, suffix);
		if ((disk = lookup_disk_by_name(buf)) == NULL)
			{
			disk = g_new0(DiskMon, 1);
			disk->name = g_strdup(buf);
			disk->resolution = DEFAULT_RESOLUTION;
			disk->major = major;
			disk->minor = i_disk;
			disk->order = i + 1;	/* Skip the composite disk */
			i = 1;
			for (list = disk_mon_list->next; list; list = list->next, ++i)
				if (disk->order < ((DiskMon *) list->data)->order)
					break;
			disk_mon_list = g_list_insert(disk_mon_list, disk, i);
			++n_disks;
			}
		return disk;
		}
	return NULL;
	}

  /* Linux kernel 2.2 has 4 disk fields or 8 if my patch has been applied.
  |  Kernel 2.4 has variable number of fields.  Fields are added as drives
  |  are mounted and generate I/O.  re_checking is FALSE if this is the first
  |  call to register_disks().
  */
static void
register_disks(gboolean re_checking)
	{
	FILE	*f;
	DiskMon	*disk;
	gchar	buf[1024], *item, *s;
	gchar	c	= 'A';
	gint	major, minor;

	if ((f = fopen(PROC_STAT_FILE, "r")) == NULL)
		return;
	while (fgets(buf, sizeof(buf), f))
		{
		item = strtok(buf, " \t\n");
		if (!strncmp(item, "disk_io:", 8))	/* Kernel 2.4 */
			{
			while ((s = strtok(NULL, " \t\n")) != NULL)
				if (sscanf(s, "(%d,%d)", &major, &minor) == 2)
					add_disk_device(major, minor);
			break;
			}
		if (re_checking)	/* Can't have added fields for kernel 2.2 */
			continue;
		if (!strncmp(item, "disk_rblk", 9))	/* Kernel 2.2, count fields */
			{
			using_DiskN_names = TRUE;
			while ((s = strtok(NULL, " \t\n")) != NULL)
				{
				disk = g_new0(DiskMon, 1);
				disk_mon_list = g_list_append(disk_mon_list, disk);
				disk->name = g_strdup_printf("%s%c", _("Disk"), c);
				disk->resolution = DEFAULT_RESOLUTION;
				++c;
				++n_disks;
				}
			break;
			}
		}
	fclose(f);
	}
#endif	/* __linux__ */


/* ----- NetBSD --------------------------------------------------------- */

#if defined(__NetBSD__) || defined(__OpenBSD__)

#include <sys/dkstat.h>
#include <sys/disk.h>
#include <kvm.h>

static struct nlist nl[] = {
#define X_DISK_COUNT    0
   { "_disk_count" },      /* number of disks */
#define X_DISKLIST      1
   { "_disklist" },        /* TAILQ of disks */
   { NULL },
};

static struct disk *dkdisks;	/* kernel disk list head */

extern	kvm_t	*kvmd;

DiskMon *
add_disk_device(gint major, gint minor)
{
   /* should something be done there ? */
   return NULL;
}

void
read_netbsd_disk()
{
   static gint data_read_tick = -1;
   GList *list;
   DiskMon *disk;
   struct disk d, *p;
   
   if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
      return;
   data_read_tick = GK.timer_ticks;

   if (kvmd == NULL) return;
   if (n_disks <= 0) return;		/* computed by register_disks() */
   if (nl[0].n_type == 0) return;	/* ditto. */

   composite_disk->rblk = 0;
   composite_disk->wblk = 0;

   p = dkdisks;
   for(list = disk_mon_list->next; list; list = list->next) {
      if (kvm_read(kvmd, (u_long)p, &d, sizeof(d)) == sizeof(d)) {
	 
	 /* It seems impossible to get the read and write transfers
	  * separately. It's just a matter of choice to put the total in
	  * the rblk member but I like the blue color so much :).
	  *
	  * If someone as an idea for the wblk value... It though of the
	  * d.xfer value (number of transfers) but there a scaling problem:
	  * d.xfer is usually much more smaller thant d.dk_bytes... */

	 disk = (DiskMon *) list->data;
	 disk->rblk = d.dk_bytes / 512;	/* block size hardcoded :
					 * d.dk_byteshift is always 0 ?? */
	 disk->wblk = 0;

	 composite_disk->rblk += disk->rblk;
	 composite_disk->wblk += disk->wblk; /* useless, isn't it ? :-D */
      }
      p = d.dk_link.tqe_next;
   }
}

void
register_disks(gboolean re_checking)
{
   gint	i;
   DiskMon *disk;
   struct disklist_head head;
   struct disk d, *p;
   char buf[20];

   if (re_checking) return;
   if (kvmd == NULL) return;

   /* get disk count */
   if (kvm_nlist(kvmd, nl) >= 0 && nl[0].n_type != 0)
      if (kvm_read(kvmd, nl[X_DISK_COUNT].n_value,
		   (char *)&n_disks, sizeof(n_disks)) != sizeof(n_disks))
	 n_disks = 0;

   /* get first disk */
   if (n_disks > 0) {
      if (kvm_read(kvmd, nl[X_DISKLIST].n_value, 
		   &head, sizeof(head)) != sizeof(head))
	 n_disks = 0;

      dkdisks = head.tqh_first;
   }

   /* get disk names */
   for (i = 0, p = dkdisks; i < n_disks; i++) {
      disk = g_new0(DiskMon, 1);
      disk_mon_list = g_list_append(disk_mon_list, disk);

      if (kvm_read(kvmd, (u_long)p, &d, sizeof(d)) != sizeof(d) ||
	  kvm_read(kvmd, (u_long)d.dk_name, buf, sizeof(buf)) != sizeof(buf))
	 /* fallback to default name if kvm_read failed */
	 disk->name = g_strdup_printf("%s%c", _("Disk"), 'A' + i);
      else
	 disk->name = strdup(buf);

      disk->resolution = DEFAULT_RESOLUTION;

      p = d.dk_link.tqe_next;
   }

   using_DiskN_names = TRUE;
}

#endif /* __NetBSD__ || __OpenBSD__ */


/* ----- Other Systems -------------- */

#if defined(USE_LIBGTOP)

/* libgtop can't read disk stats */

static DiskMon *
add_disk_device(gint major, gint minor)
	{
	return NULL;
	}

static void
register_disks(gint x)
	{
	}
#endif

/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_disk_interface()
	{
	gint	available = FALSE;

#if defined(__FreeBSD__)
	register_disks(FALSE);
	read_system_disk_info = read_freebsd_disk;	/* above */
	available = TRUE;
#endif

#if defined(__linux__)
	register_disks(FALSE);
	read_system_disk_info = read_proc_stat;		/* In stat_cpu.c */
	available = TRUE;
#endif

#if defined(__NetBSD__) || defined(__OpenBSD__)
	register_disks(FALSE);
	read_system_disk_info = read_netbsd_disk;	/* above */
	available = TRUE;
#endif

	return available;
	}

/* ======================================================================== */

DiskMon			*composite_disk;
static gint		ascent;
static gint		style_id;

DiskMon *
lookup_disk_by_device(gint major, gint minor)
	{
	DiskMon	*disk;
	GList	*list;

	for (list = disk_mon_list->next; list; list = list->next)
		{
		disk = (DiskMon * ) list->data;
		if (disk->major == major && disk->minor == minor)
			return disk;
		}
	return NULL;
	}

static DiskMon *
lookup_disk_by_name(gchar *name)
	{
	DiskMon	*disk;
	GList	*list;

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon * ) list->data;
		if (!strcmp(name, disk->name))
			return disk;
		}
	return NULL;
	}


static void
draw_disk_extra(Chart *cp, unsigned long l)
	{
	TextStyle	*ts;
	gchar		buf[32];

	ts = gkrellm_chart_textstyle(style_id);
	if (ascent == 0)
		ascent = gdk_char_height(ts->font, '8');
	snprintf(buf, sizeof(buf), "%d", (gint) l);
	gkrellm_draw_chart_label(cp, ts, 4, 4 + ascent, buf);
	}

static void
refresh_disk_chart(DiskMon *disk)
	{
	Chart	*cp		= disk->chart;

	gkrellm_draw_chart(cp);
	if (disk->extra_info)
		draw_disk_extra(cp, disk->cur);
	cp->need_redraw = FALSE;
	}

static void
update_disk()
	{
	GList		*list;
	DiskMon		*disk;
	Chart		*cp;

	if (n_disks == 0)
		return;
	(*read_system_disk_info)();

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if ((cp = disk->chart) == NULL)		/* or disk->enabled FALSE */
			continue;
		if (GK.second_tick)
			{
			if (cp->primed)
				disk->cur = (disk->rblk - cp->prevIn)
							+ (disk->wblk - cp->prevOut);
			gkrellm_store_chart_data(cp, disk->wblk, disk->rblk, 0);
			refresh_disk_chart(disk);
			}
		if (cp->need_redraw)
			refresh_disk_chart(disk);
		gkrellm_update_krell(cp->panel, KRELL(cp->panel),
										disk->wblk + disk->rblk);
		gkrellm_draw_layers(cp->panel);
		}
	}


static gint
disk_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GList		*list;
	Chart		*cp;
	GdkPixmap	*pixmap	= NULL;

	for (list = disk_mon_list; list; list = list->next)
		{
		if ((cp = ((DiskMon *) list->data)->chart) == NULL)
			continue;
		if (widget == cp->drawing_area)
			pixmap = cp->pixmap;
		else if (widget == cp->panel->drawing_area)
			pixmap = cp->panel->pixmap;
		if (pixmap)
			{
			gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
				  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				  ev->area.width, ev->area.height);
			break;
			}
		}
	return FALSE;
	}

static gint
cb_disk_extra(GtkWidget *widget, GdkEventButton *event)
	{
	GList	*list;
	DiskMon	*disk;

	if (event->button == 1)
		{
		for (list = disk_mon_list; list; list = list->next)
			{
			disk = (DiskMon *) list->data;
			if (disk->enabled && widget == disk->chart->drawing_area)
				{
				disk->extra_info = 1 - disk->extra_info;
				refresh_disk_chart(disk);
				gkrellm_config_modified();
				}
			}
		}
	return TRUE;
	}

static void
setup_disk_scaling(DiskMon *disk)
	{
	Chart	*cp		= disk->chart;
	gint	grids;

	grids = UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS;
	cp->scale_min = disk->resolution;
	KRELL(cp->panel)->full_scale = cp->scale_min * grids / UC.update_HZ;
	cp->scale_max = 0;
	}

  /* Destroy everything in a DiskMon structure except for the vbox which
  |  is preserved so disk ordering will be maintained.  Compare this to
  |  destroying an InetMon where really everything is destroyed including
  |  the InetMon structure.  Here the DiskMon structure is not destroyed.
  */
static void
destroy_disk_monitor(DiskMon *disk)
	{
	Chart	*cp;

	if (disk->launch.button)
		gkrellm_destroy_button(disk->launch.button);
	disk->launch.button = NULL;
	gkrellm_dup_string(&disk->launch.command, "");
	gkrellm_dup_string(&disk->launch.tooltip_comment, "");
	cp = disk->chart;
	g_free(cp->name);
	gkrellm_monitor_height_adjust( - disk->height);
	gkrellm_destroy_panel(cp->panel);
	disk->launch.tooltip = NULL;
	g_free(cp->panel);
	gkrellm_destroy_chart(cp);
	g_free(cp);
	disk->chart = NULL;
	disk->enabled = FALSE;
	}

static void
create_disk_monitor(DiskMon *disk, gint first_create)
	{
	Panel		*p;
	Style		*style;
	Chart		*cp;

	if (first_create)
		{
		disk->chart = gkrellm_chart_new0();
		disk->chart->panel = gkrellm_panel_new0();
		}
	else
		{
		gkrellm_destroy_decal_list(disk->chart->panel);
		gkrellm_destroy_krell_list(disk->chart->panel);
		}
	cp = disk->chart;
	p = cp->panel;

	gkrellm_dup_string(&cp->name, disk->name);

	style = gkrellm_panel_style(style_id);
	gkrellm_create_krell(p, gkrellm_krell_panel_image(style_id), style);
	setup_disk_scaling(disk);

	cp->h = UC.chart_height[MON_DISK]; 
	gkrellm_create_chart(disk->vbox, cp, style_id);

	p->textstyle = gkrellm_panel_textstyle(style_id);

	gkrellm_configure_panel(p, cp->name, style);
	gkrellm_create_panel(disk->vbox, p, gkrellm_bg_panel_image(style_id));
	disk->height = cp->h + p->h;
	gkrellm_monitor_height_adjust(disk->height);
	disk->enabled = TRUE;

	gkrellm_alloc_chart_data(cp);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "expose_event",
					(GtkSignalFunc) disk_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT (p->drawing_area), "expose_event",
					(GtkSignalFunc) disk_expose_event, NULL);

		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "button_press_event",
					(GtkSignalFunc) cb_disk_extra, NULL);
		}
	else
		refresh_disk_chart(disk);

	gkrellm_configure_tooltip(p, &disk->launch);
	if (*(disk->launch.command) != '\0')
		disk->launch.button = gkrellm_put_label_in_panel_button(p,
				gkrellm_launch_button_cb, &disk->launch, disk->launch.pad);
	}


static GtkWidget	*disk_vbox;

static void
create_disk(GtkWidget *vbox, gint first_create)
	{
	GList	*list;
	DiskMon	*disk;

	ascent = 0;
	disk_vbox = vbox;
	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (first_create)
			{
			disk->vbox = gtk_vbox_new(FALSE, 0);
			gtk_container_add(GTK_CONTAINER(vbox), disk->vbox);
			if (disk->enabled)
				gtk_widget_show(disk->vbox);
			}
		gkrellm_setup_launcher(NULL, &disk->launch, CHART_PANEL_TYPE, 4);
		if (disk->enabled)
			create_disk_monitor(disk, first_create);
		}
	}

  /* Kernel 2.4 will not show a disk until it has I/O, and some systems
  |  may dynamically add a drive.
  */
static void
check_for_new_disks()
	{
	GList	*list;
	DiskMon	*disk;

	register_disks(TRUE);

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (disk->vbox == NULL)
			{
			disk->vbox = gtk_vbox_new(FALSE, 0);
			gtk_container_add(GTK_CONTAINER(disk_vbox), disk->vbox);
			gkrellm_setup_launcher(NULL, &disk->launch, CHART_PANEL_TYPE, 4);
			}
		}
	}

/* --------------------------------------------------------------------	*/
#define	DISK_CONFIG_KEYWORD	"disk"

static GtkWidget	*launch_vbox;

static gint     disk_map[] =
	{ 10, 20, 50, 100, 200, 500, 1000, 2000, 5000, 10000, 20000, 50000 };

static gint		disk_temp;

static void
cb_disk_resolution(GtkWidget *widget, GtkSpinButton *spin)
	{
	gint    value;

	value = gtk_spin_button_get_value_as_int(spin);
	if (value != disk_temp) /* Avoid recursion */
		{
		value = map_1_2_5(value, disk_map, sizeof(disk_map) / sizeof(int));
		disk_temp = value;
		gtk_spin_button_set_value(spin, (gfloat) value);
		}
	}


static void
save_disk_config(FILE *f)
	{
	GList	*list;
	DiskMon	*disk;

	if (n_disks == 0)
		return;
	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (!using_DiskN_names && disk != composite_disk)
			fprintf(f, "%s enable (%d,%d) %d %d %d\n", DISK_CONFIG_KEYWORD,
					disk->major, disk->minor, disk->enabled,
					disk->resolution, disk->extra_info);
		else
			fprintf(f, "%s enable %s %d %d %d\n", DISK_CONFIG_KEYWORD,
					disk->name, disk->enabled,
					disk->resolution, disk->extra_info);
		if (*(disk->launch.command) != '\0')
			fprintf(f, "%s launch %s %s\n", DISK_CONFIG_KEYWORD,
						disk->name, disk->launch.command);
		if (*(disk->launch.tooltip_comment) != '\0')
			fprintf(f, "%s tooltip_comment %s %s\n", DISK_CONFIG_KEYWORD,
						disk->name, disk->launch.tooltip_comment);
		}
	}

static void
load_disk_config(gchar *arg)
	{
	DiskMon	*disk;
	gchar	config[32], item[CFG_BUFSIZE],
			disk_name[32], command[CFG_BUFSIZE];
	gint	major, minor, res, n, extra;

	if (n_disks == 0)
		return;
	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (strcmp(config, "composite") == 0)	/* XXX Obsolete >= 1.0.8 */
			sscanf(item, "%d %d %d", &composite_disk->enabled,
					&composite_disk->resolution, &composite_disk->extra_info);
		else if (strcmp(config, "enable") == 0)
			{
			n = FALSE;
			res = DEFAULT_RESOLUTION;
			extra = 0;
			disk = NULL;
			if (item[0] == '(')
				{	/* Avoid user_config pathology (eg kernel 2.4 -> 2.2) */
				if (   !using_DiskN_names
					&& sscanf(item, "(%d,%d) %d %d %d",
						&major, &minor, &n, &res, &extra) > 1
				   )
					disk = add_disk_device(major, minor);
				}
			else
				{
				sscanf(item, "%31s %d %d %d", disk_name, &n, &res, &extra);
				disk = lookup_disk_by_name(disk_name);
				}
			if (disk)
				{
				disk->enabled = n;
				disk->resolution = res;
				disk->extra_info = extra;
				}
			}
		else if (strcmp(config, "launch") == 0)
			{
			sscanf(item, "%31s %[^\n]", disk_name, command);
			if ((disk = lookup_disk_by_name(disk_name)) != NULL)
				disk->launch.command = g_strdup(command);
			}
		else if (strcmp(config, "tooltip_comment") == 0)
			{
			sscanf(item, "%31s %[^\n]", disk_name, command);
			if ((disk = lookup_disk_by_name(disk_name)) != NULL)
				disk->launch.tooltip_comment = g_strdup(command);
			}
		}
	}

static void
add_launch_entry(GtkWidget *vbox, DiskMon *disk)
	{
	GtkWidget   *table;

	table = gkrellm_launcher_table_new(vbox, 1);
	gkrellm_config_launcher(table, 0,  &disk->launch_entry,
				&disk->tooltip_entry, disk->name,
				&disk->launch);
	disk->launch_table = table;
	gtk_widget_show_all(table);
    }

static void
apply_disk_config()
	{
	GtkSpinButton	*spin;
	GList			*list;
	DiskMon			*disk;

	if (n_disks == 0)
		return;

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		spin = GTK_SPIN_BUTTON(disk->spin_button);
		disk->resolution = gtk_spin_button_get_value_as_int(spin);
		
		if (disk->launch_table)		/* Not NULL only if enabled */
			{
			gkrellm_apply_launcher(&disk->launch_entry, &disk->tooltip_entry,
				disk->chart->panel, &disk->launch, gkrellm_launch_button_cb);
			gtk_widget_destroy(disk->launch_table);
			}
		disk->launch_table = NULL;
		}
	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		disk->new_disk = GTK_TOGGLE_BUTTON(disk->enable_button)->active;
		if (disk->new_disk)
			add_launch_entry(launch_vbox, disk);
		if (disk->new_disk && ! disk->enabled)
			{
			gtk_widget_show(disk->vbox);
			create_disk_monitor(disk, TRUE);
			disk->chart->primed = FALSE;
			}
		else if (! disk->new_disk && disk->enabled)
			{
			gtk_widget_hide(disk->vbox);
			destroy_disk_monitor(disk);
			}
		if (disk->chart)
			setup_disk_scaling(disk);
		disk->enabled = disk->new_disk;
		}
	}


static gchar	*disk_info_text[] =
{
#if defined (__linux__)
N_("<b>Kernel 2.2.x or 2.0.x\n"),
N_("The kernel reports accumulated disk I/O stats in /proc/stat for\n"
	"disk drives /dev/hda through /dev/hdd and /dev/sda through /dev/sdd.\n"
	"The letter suffix a-d for these devices correspond to charts DiskA\n"
	"through DiskD.\n"
	"When both an IDE and SCSI drive have a common minor number (eg hda\n"
	"and sda), the disk I/O stats for the two drives are combined.  So, if\n"
	"you have an IDE /dev/hda and a SCSI drive which you wish to monitor\n"
	"separately, then the SCSI drive should not be configured as /dev/sda.\n"
	"SCSI drives with minor numbers greater than 3 (eg sde, sdf, ...) do\n"
	"not have I/O stats reported in /proc/stat\n\n"),

N_("<b>Kernel 2.4.x\n"),
N_("Data for IDE and SCSI drives is not combined as in kernel 2.2, and\n"
	"disk charts may be selected by device name instead of generic DiskN\n"
	"names.")
#else
""
#endif
};


static void
create_disk_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*table;
	GtkWidget		*tabs;
	GtkWidget		*vbox, *vbox1, *hbox;
	GtkWidget		*label;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	GtkWidget		*sep;
	GList			*list;
	DiskMon			*disk;
	gint			i;

	check_for_new_disks();
	if (n_disks == 0)
		return;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* --Setup tab */
	vbox = gkrellm_create_tab(tabs, _("Setup"));

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);

	vbox1 = gtk_vbox_new(FALSE, 3);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled),vbox1);
	table = gtk_table_new (n_disks + 2, 4, FALSE /*homogeneous*/);
	gtk_box_pack_start(GTK_BOX(vbox1), table, FALSE, FALSE, 2);

	label = gtk_label_new(_("Disk Name"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	label = gtk_label_new(_("Chart Resolution"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);
	sep = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), sep, 0, 3, 1, 2);

	for (i = 2, list = disk_mon_list; list; list = list->next, ++i)
		{
		disk = (DiskMon *) list->data;
		disk->enable_button = gtk_check_button_new_with_label(disk->name);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(disk->enable_button),
				disk->enabled);
		gtk_table_attach_defaults(GTK_TABLE(table), disk->enable_button,
				0, 1, i, i + 1);

		hbox = gtk_hbox_new(TRUE, 0);
		gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, i, i+1);
		gkrellm_spin_button(hbox, &disk->spin_button,
			(gfloat) disk->resolution, 10.0, 50000.0, 1.0, 1.0, 0, 80,
			cb_disk_resolution, NULL, FALSE, NULL);

		if (i == 2)
			{
			label = gtk_label_new(
					_("Composite chart combines data for all disks"));
			gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
			gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 2, 3);
			}

		}

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), sep, FALSE, FALSE, 2);

	label = gtk_label_new(
		_("Chart resolutions are disk I/O blocks/sec per grid"));
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 5);

/* --Launch tab */
	vbox = gkrellm_create_tab(tabs, _("Launch"));
/*	insert_expanded_filler(vbox); */

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	launch_vbox = gtk_vbox_new(FALSE, 0);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled),
			launch_vbox);
	for (i = 0, list = disk_mon_list; list; list = list->next, ++i)
		{
		disk = (DiskMon *) list->data;
		if (disk->enabled)
			add_launch_entry(launch_vbox, disk);
		}

/* --Info tab */
	if (*disk_info_text[0] != '\0')
		{
		vbox = gkrellm_create_tab(tabs, _("Info"));
		scrolled = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
		text = gtk_text_new(NULL, NULL);
		for (i = 0; i < sizeof(disk_info_text)/sizeof(gchar *); ++i)
			gkrellm_add_info_text_string(text, _(disk_info_text[i]));
		gtk_text_set_editable(GTK_TEXT(text), FALSE);
		gtk_container_add(GTK_CONTAINER(scrolled), text);
		}
	}

static Monitor	monitor_disk =
	{
	N_("Disk"),				/* Name, for config tab.	*/
	MON_DISK,			/* Id,  0 if a plugin		*/
	create_disk,		/* The create function		*/
	update_disk,				/* The update function		*/
	create_disk_tab,	/* The config tab create function	*/
	apply_disk_config,	/* Apply the config function		*/

	save_disk_config,	/* Save user conifg			*/
	load_disk_config,	/* Load user config			*/
	DISK_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_disk_monitor(void)
	{
	DiskMon	*disk;
	GList	*list;

	monitor_disk.name = _(monitor_disk.name);

	disk = g_new0(DiskMon, 1);
	disk_mon_list = g_list_append(disk_mon_list, disk);
	disk->name = g_strdup(_("Disk"));
	disk->resolution = DEFAULT_RESOLUTION;
	composite_disk = disk;

	if (setup_disk_interface())
		{
		for (list = disk_mon_list->next; list; list = list->next)
			((DiskMon *) list->data)->enabled = FALSE;
		if (n_disks > 0)
			composite_disk->enabled = TRUE;
		style_id = gkrellm_add_chart_style(&monitor_disk, DISK_STYLE_NAME);
		return &monitor_disk;
		}
	return NULL;
	}
