/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|  1/23/2001 Hajimu UMEMOTO added get_bufspace(), improved
|				read_freebsd_meminfo().
| 10/12/2000 NetBSD code contributed by Anthony Mallet
|			<metall@ficus.yi.org>
|  5/27/2000 Patch from William Carrel <william.a@carrel.org> and 
|		Hajimu UMEMOTO ume@mahoroba.org to cleanup FreeBSD swap code.
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

  /* The meminfo monitor is a hybrid of chart and meter types.
  |  There is a swap_chart without a panel, and a mem and swap meter.
  |  The swap meter has both a standard meter styled krell (krell_used)
  |  and a chart panel styled krell (krell_delta).
  |  There are two monitors created here, mem and swap meters.  Because
  |  this was the original setup and many themes have customized based
  |  on this, the swap_chart is sort of retrofitted on.  I put it on
  |  top of the mem/swap meter pair for looks and it uses the default
  |  chart style (ie cannot be custom themed).  So from the point of
  |  view of themers, there are still only mem and swap meter monitors.
  */
  /* Also, enabling is handled differently for the chart vs meters.  The
  |  chart is created/destroyed when enabled/disabled while the meters
  |  just toggle hbox visibility when enabled/disabled.  The reason is ...
  |  ooops, I'm out of time ...
  */
typedef struct
	{
	Panel		*panel;
	Krell		*krell_used,	/* Meter styled, shows fraction used	*/
				*krell_delta;	/* Chart styled, for page in/out deltas */
	gchar		*label,
				*data_format;	/* Format string for scrolling text		*/
	gint		x_label;
	Decal		*decal_label;
	gboolean	label_is_data,
				restore_label;	/* Helper to know when to toggle data fmt off*/
	gint		style_id;
	gint		enabled;
	Launcher	launch;

	/* (*(read_system_meminfo))() fills in this data
	*/
	gulong	total,		/* Total memory or swap in system */
			used,		/* Amount of memory (calulated) or swap used  */
			x_used,		/* Raw kernel value, not used by swap monitor */
			free,		/* Not used by swap	monitor */
			shared,		/* Not used by swap	monitor */
			buffers,	/* Not used by swap	monitor */
			cached;		/* Not used by swap	monitor */
	}
	MeminfoMeter;

MeminfoMeter	mem,
				swap;

typedef struct
	{
	GtkWidget	*vbox;			/* Holds the chart */
	Chart		*chart;
	gint		enabled;
	gint		resolution;
	gint		extra_info;
	gulong		cur;

	/* (*(read_system_meminfo))() fills in this data
	*/
	gulong		page_in,
				page_out;
	}
	MeminfoChart;

MeminfoChart	swap_chart;


void	(*read_system_meminfo)();

gint force_meminfo_update();


/* ====== System dependent interface ====================================== */

/* ------- FreeBSD ----------------------------------------------------- */

#if defined(__FreeBSD__)
#include <osreldate.h>
#include <kvm.h>
#include <limits.h>
#include <sys/conf.h>
#if __FreeBSD_version < 400000
#include <sys/rlist.h>
#endif
#include <sys/vmmeter.h>
#include <sys/sysctl.h>
#include <vm/vm_param.h>

static struct nlist nl[] = {
#define N_CNT		0
	{ "_cnt" },
#if __FreeBSD_version < 400000
#define VM_SWAPLIST	1
	{ "_swaplist" },
#define VM_SWDEVT	2
	{ "_swdevt" },
#define VM_NSWAP	3
	{ "_nswap" },
#define VM_NSWDEV	4
	{ "_nswdev" },
#define VM_DMMAX	5
	{ "_dmmax" },
#if  __FreeBSD_version < 300000
#define N_BUFSPACE	6
	{ "_bufspace" },
#endif
#endif
	{ "" }
};

extern	kvm_t	*kvmd;
extern	char	errbuf[];

static int
swapmode(long *retavail, long *retfree)
	{
	int used, avail;
#if  __FreeBSD_version >= 400000
	static int psize = -1;
	struct kvm_swap kvmswap;
#else
	char *header;
	int hlen, nswap, nswdev, dmmax;
	int i, div, nfree, npfree;
	struct swdevt *sw;
	long blocksize, *perdev;
	u_long ptr;
	struct rlist head;
#  if __FreeBSD_version >= 220000
	struct rlisthdr swaplist;
#  else 
	struct rlist *swaplist;
#  endif
	struct rlist *swapptr;
#endif

	/*
	 * Counter for error messages. If we reach the limit,
	 * stop reading information from swap devices and
	 * return zero. This prevent endless 'bad address'
	 * messages.
	 */
	static int warning = 10;

	if (warning <= 0)
		{
		/* a single warning */
		if (!warning)
	    		{
			warning--;
			fprintf(stderr, "Too much errors, stop reading swap devices ...\n");
			}
		return(0);
		}
	warning--;		/* decrease counter, see end of function */

	if (kvmd == NULL)
		return(0);
#if  __FreeBSD_version >= 400000
	if (kvm_getswapinfo(kvmd, &kvmswap, 1, 0) < 0)
		{
		fprintf(stderr, "kvm_getswapinfo failed\n");
		return(0);
		}

	if (psize < 0)
	    psize = getpagesize();
	*retavail = avail = (quad_t)kvmswap.ksw_total * psize;
	used = (quad_t)kvmswap.ksw_used * psize;
	*retfree = avail - used;
#else
	if (kvm_read(kvmd, nl[VM_NSWAP].n_value,
		     &nswap, sizeof(nswap)) != sizeof(nswap))
		return(0);
	if (!nswap)
		{
		fprintf(stderr, "No swap space available\n");
		return(0);
		}

	if (kvm_read(kvmd, nl[VM_NSWDEV].n_value,
		     &nswdev, sizeof(nswdev)) != sizeof(nswdev))
		return(0);
	if (kvm_read(kvmd, nl[VM_DMMAX].n_value,
		     &dmmax, sizeof(dmmax)) != sizeof(dmmax))
		return(0);
	if (kvm_read(kvmd, nl[VM_SWAPLIST].n_value,
		     &swaplist, sizeof(swaplist)) != sizeof(swaplist))
		return(0);

	if ((sw = (struct swdevt *)malloc(nswdev * sizeof(*sw))) == NULL ||
	    (perdev = (long *)malloc(nswdev * sizeof(*perdev))) == NULL)
		{
		perror("malloc");
		exit(1);
		}
	if (kvm_read(kvmd, nl[VM_SWDEVT].n_value,
		     &ptr, sizeof ptr) != sizeof ptr)
		return(0);
	if (kvm_read(kvmd, ptr,
		     sw, nswdev * sizeof(*sw)) != nswdev * sizeof(*sw))
		return(0);

	/* Count up swap space. */
	nfree = 0;
	memset(perdev, 0, nswdev * sizeof(*perdev));
#if  __FreeBSD_version >= 220000
	swapptr = swaplist.rlh_list;
	while (swapptr)
#else
	while (swaplist)
#endif
		{
		int	top, bottom, next_block;
#if  __FreeBSD_version >= 220000
		if (kvm_read(kvmd, (u_long)swapptr, &head,
			     sizeof(struct rlist)) != sizeof(struct rlist))
			return (0);
#else
		if (kvm_read(kvmd, (u_long)swaplist, &head,
			     sizeof(struct rlist)) != sizeof(struct rlist))
			return (0);
#endif

		top = head.rl_end;
		bottom = head.rl_start;

		nfree += top - bottom + 1;

		/*
		 * Swap space is split up among the configured disks.
		 *
		 * For interleaved swap devices, the first dmmax blocks
		 * of swap space some from the first disk, the next dmmax
		 * blocks from the next, and so on up to nswap blocks.
		 *
		 * The list of free space joins adjacent free blocks,
		 * ignoring device boundries.  If we want to keep track
		 * of this information per device, we'll just have to
		 * extract it ourselves.
		 */

		while (top / dmmax != bottom / dmmax)
			{
			next_block = ((bottom + dmmax) / dmmax);
			perdev[(bottom / dmmax) % nswdev] +=
				next_block * dmmax - bottom;
			bottom = next_block * dmmax;
			}
		perdev[(bottom / dmmax) % nswdev] +=
			top - bottom + 1;

#if  __FreeBSD_version >= 220000
		swapptr = head.rl_next;
#else
		swaplist = head.rl_next;
#endif
		}

	header = getbsize(&hlen, &blocksize);
	div = blocksize / 512;
	avail = npfree = 0;
	for (i = 0; i < nswdev; i++)
		{
		int xsize, xfree;

		/*
		 * Don't report statistics for partitions which have not
		 * yet been activated via swapon(8).
		 */

		xsize = sw[i].sw_nblks;
		xfree = perdev[i];
		used = xsize - xfree;
		npfree++;
		avail += xsize;
		}

	/* 
	 * If only one partition has been set up via swapon(8), we don't
	 * need to bother with totals.
	 */
	*retavail = avail << 9;
	*retfree = nfree << 9;
	used = avail - nfree;
	free(sw); free(perdev);
#endif /* __FreeBSD_version >= 400000 */

	/* increase counter, no errors occurs */
	warning++; 

	return  (int)(((double)used / (double)avail * 100.0) + 0.5);
	}

static int
get_bufspace(unsigned long *bufspacep)
	{
#if  __FreeBSD_version < 300000
	u_int	bufspace;

	if (kvm_read(kvmd, nl[N_BUFSPACE].n_value, (char *)&bufspace,
		     sizeof(bufspace)) != sizeof(bufspace))
		return 0;
#else
	static char	*name = "vfs.bufspace";
	static int	oid_name2oid[2] = { 0, 3 };
	static int	oid_bufspace[CTL_MAXNAME + 2];
	static size_t	oid_bufspace_len = sizeof(oid_bufspace);
	static gint	initialized = 0;
	u_int		bufspace;
	size_t		bufspace_len = sizeof(bufspace);

	if (!initialized)
		{
		if (sysctl(oid_name2oid, 2, oid_bufspace,
			   &oid_bufspace_len, (void *)name, strlen(name)) < 0)
			return 0;
		oid_bufspace_len /= sizeof(int);
		++initialized;
		}

	if (sysctl(oid_bufspace, oid_bufspace_len,
		   &bufspace, &bufspace_len, 0, 0) < 0)
		return 0;
#endif
	*bufspacep = bufspace;
	return 1;	
	}

#if __FreeBSD_version >= 410000
struct mibtab {
    char	*name;
    int		oid[CTL_MAXNAME + 2];
    size_t	oid_len;
    u_int	value;
    size_t	value_len;
};

static struct mibtab mibs[] = {
#define MIB_V_PAGE_COUNT	0
    { "vm.stats.vm.v_page_count" },
#define MIB_V_FREE_COUNT	1
    { "vm.stats.vm.v_free_count" },
#define MIB_V_WIRE_COUNT	2
    { "vm.stats.vm.v_wire_count" },
#define MIB_V_ACTIVE_COUNT	3
    { "vm.stats.vm.v_active_count" },
#define MIB_V_INACTIVE_COUNT	4
    { "vm.stats.vm.v_inactive_count" },
#define MIB_V_CACHE_COUNT	5
    { "vm.stats.vm.v_cache_count" },
#define MIB_V_SWAPPGSIN		6
    { "vm.stats.vm.v_swappgsin" },
#define MIB_V_SWAPPGSOUT	7
    { "vm.stats.vm.v_swappgsout" },
    { NULL }
};

#define	PROC_MEMINFO_FILE	"/compat/linux/proc/meminfo"
#endif

static void
read_freebsd_meminfo()
	{
	static gint	psize, pshift = 0;
	static gint	first_time_done = 0;
	static gint	swappgsin = -1;
	static gint	swappgsout = -1;
	gint		dpagein, dpageout;
	struct vmmeter	sum;
	unsigned long	buffers;
	struct vmtotal	vmt;
	size_t		length_vmt = sizeof(vmt);
	static int	oid_vmt[] = { CTL_VM, VM_METER };
#if __FreeBSD_version >= 410000
	static int	oid_name2oid[2] = { 0, 3 };
	gint		i;
	FILE		*f;
	gchar		buf[160];
#endif

	/* Collecting meminfo data is expensive under FreeBSD, so
	|  take extra precautions to minimize reading it.
	*/
	if (!GK.ten_second_tick && !force_meminfo_update())
		return;

	if (pshift == 0)
		{
		for (psize = getpagesize(); psize > 1; psize >>= 1)
			pshift++;
		}

	if (kvmd == NULL)
		{
#if __FreeBSD_version >= 410000
		if (!first_time_done)
			{
			for (i = 0; mibs[i].name; ++i)
				{
				mibs[i].oid_len = sizeof(mibs[i].oid);
				if (sysctl(oid_name2oid, 2, mibs[i].oid,
					   &mibs[i].oid_len,
					   (void *)mibs[i].name,
					   strlen(mibs[i].name)) < 0)
				 	return;
				mibs[i].oid_len /= sizeof(int);
				mibs[i].value_len = sizeof(mibs[i].value);
				}
			++first_time_done;
			}
		for (i = 0; mibs[i].name; ++i)
			if (sysctl(mibs[i].oid, mibs[i].oid_len,
				   &mibs[i].value,
				   &mibs[i].value_len, 0, 0) < 0)
				return;
		mem.total = (mibs[MIB_V_PAGE_COUNT].value -
			     mibs[MIB_V_WIRE_COUNT].value) << pshift;
		mem.x_used = (mibs[MIB_V_ACTIVE_COUNT].value +
			      mibs[MIB_V_INACTIVE_COUNT].value) << pshift;
		mem.free = mibs[MIB_V_FREE_COUNT].value << pshift;
		if (sysctl(oid_vmt, 2, &vmt, &length_vmt, NULL, 0) == 0)
			mem.shared = vmt.t_rmshr << pshift;
		get_bufspace(&mem.buffers);
		mem.cached = mibs[MIB_V_CACHE_COUNT].value << pshift;
		mem.used = mem.x_used - mem.buffers - mem.cached;

		swap_chart.page_in = mibs[MIB_V_SWAPPGSIN].value;
		swap_chart.page_out = mibs[MIB_V_SWAPPGSOUT].value;

		/* Try linprocfs for swap info */
		if ((f = fopen(PROC_MEMINFO_FILE, "r")) == NULL)
			return;
		/* total: used: free: shared: buffers: cached: */
		while ((fgets(buf, sizeof(buf), f)) != NULL)
			{
			if (strncmp(buf, "Swap:", 5) == 0)
				{
				sscanf(buf, "Swap: %lu %lu",
				       &swap.total, &swap.used);
				break;
				}
			}
		fclose(f);
#endif
		return;
		}

	if (nl[0].n_type == 0)
		if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
			return;
	if (kvm_read(kvmd, nl[N_CNT].n_value, (char *)&sum,
		     sizeof(sum)) != sizeof(sum))
		return;

	mem.total = (sum.v_page_count - sum.v_wire_count) << pshift;
	mem.x_used = (sum.v_active_count + sum.v_inactive_count) << pshift;
	mem.free = sum.v_free_count << pshift;
	if (sysctl(oid_vmt, 2, &vmt, &length_vmt, NULL, 0) == 0)
		mem.shared = vmt.t_rmshr << pshift;
	get_bufspace(&mem.buffers);
	mem.cached = sum.v_cache_count << pshift;
	if (swappgsin < 0)
		{
		dpagein = 0;
		dpageout = 0;
		}
	else
		{
		dpagein = (sum.v_swappgsin - swappgsin) << (pshift - 10);
		dpageout = (sum.v_swappgsout - swappgsout) << (pshift - 10);
		}
	swappgsin = sum.v_swappgsin;
	swappgsout = sum.v_swappgsout;

	if (dpagein > 0 || dpageout > 0 || first_time_done == 0)
		{
		swapmode(&swap.total, &swap.used);
		swap.used = swap.total - swap.used;
		}
	first_time_done = 1;
	mem.used = mem.x_used - mem.buffers - mem.cached;
	swap_chart.page_in = swappgsin;
	swap_chart.page_out = swappgsout;
	}
#endif


/* ------- Linux ----------------------------------------------------- */
#if defined(__linux__)

#define	PROC_MEMINFO_FILE	"/proc/meminfo"


static void
read_proc_meminfo()
	{
	FILE		*f;
	gchar		buf[160];

	/* Reading /proc/meminfo is expensive under Linux kernel 2.2, so
	|  take extra precautions to minimize reading it.  If kernel 2.4
	|  improves this situation, I will later speed this up for it.
	*/
	if (GK.ten_second_tick || force_meminfo_update())
		{
		if ((f = fopen(PROC_MEMINFO_FILE, "r")) != NULL)
			{		/* total:    used:    free:  shared: buffers:  cached: */
			while ((fgets(buf, sizeof(buf), f)) != NULL)
				{
				if (strncmp(buf, "Mem:", 4) == 0)
					sscanf(buf,"Mem: %lu %lu %lu %lu %lu %lu",
						&mem.total, &mem.x_used, &mem.free,
						&mem.shared, &mem.buffers, &mem.cached);

				if (strncmp(buf, "Swap:", 5) == 0)
					{
					sscanf(buf,"Swap: %lu %lu", &swap.total,
						&swap.used);
					break;
					}
				}
			mem.used = mem.x_used - mem.buffers - mem.cached;
			fclose(f);
			}
		}
	/* swap page in/out data is in /proc/stat and this is read in cpu.c
	|  at full speed where all /proc/stat data is collected.
	|  Just go get what is there.
	*/
	read_stat_swap(&swap_chart.page_in, &swap_chart.page_out);
	}
#endif	/* __linux__ */


/* --------- NetBSD ----------------------------------------------------- */

#if defined(__NetBSD__) || defined(__OpenBSD__)

#include <sys/vmmeter.h>
#include <sys/sysctl.h>
#include <uvm/uvm_extern.h>
#include <kvm.h>

static struct nlist nl[] = {
#define X_UVM_EXP    0
   { "_uvmexp" },
   { NULL }
};

extern	kvm_t	*kvmd;

void
read_netbsd_meminfo()
{
   static int mib[] = { CTL_VM, VM_METER };
   static int pgout, pgin;
   struct vmtotal vmt;
   struct uvmexp uvmexp;
   int len;

   if (kvmd == NULL) return;

   /* get the name list if it's not done yet */
   if (nl[0].n_type == 0) kvm_nlist(kvmd, nl);

   if (nl[0].n_type != 0)
      if (kvm_read(kvmd, nl[X_UVM_EXP].n_value,
		   &uvmexp, sizeof(uvmexp)) != sizeof(uvmexp))
	 memset(&uvmexp, 0, sizeof(uvmexp));

   if (sysctl(mib, 2, &vmt, &len, NULL, 0) < 0)
      memset(&vmt, 0, sizeof(vmt));

   mem.total = (uvmexp.npages - uvmexp.wired) << uvmexp.pageshift;

   /* not sure of what must be computed */
   mem.x_used = (uvmexp.active + uvmexp.inactive) << uvmexp.pageshift;
   mem.free = uvmexp.free << uvmexp.pageshift;
   mem.shared = vmt.t_rmshr << uvmexp.pageshift;

   /* want to see only this in the chat. this could be changed */
   mem.used = uvmexp.active << uvmexp.pageshift;

   /* don't know how to get those values */
   mem.buffers = 0;
   mem.cached = 0;

   /* show only the pages located on the disk and not in memory */
   swap.total = uvmexp.swpages << uvmexp.pageshift;
   swap.used = uvmexp.swpgonly << uvmexp.pageshift;

   /* For page in/out operations, uvmexp struct doesn't seem to be reliable */

   /* if the number of swapped pages that are in memory (inuse - only) is
    * greater that the previous value (pgin), we count this a "page in" */
   if (uvmexp.swpginuse - uvmexp.swpgonly > pgin)
      swap_chart.page_in += uvmexp.swpginuse - uvmexp.swpgonly - pgin;
   pgin = uvmexp.swpginuse - uvmexp.swpgonly;

   /* same for page out */
   if (uvmexp.swpgonly > pgout)
      swap_chart.page_out += uvmexp.swpgonly - pgout;
   pgout = uvmexp.swpgonly;
}

#endif /* __NetBSD__ || __OpenBSD__ */


/* --------- Others --------------------------------------------------- */
#if defined(USE_LIBGTOP)

#include <glibtop/mem.h>
#include <glibtop/swap.h>

static void
read_glibtop_meminfo()
	{
	glibtop_mem		glt_mem;
	glibtop_swap	glt_swap;

	/* I don't know if collecting meminfo data is expensive for other
	|  systems as it is on Linux, so slow down to once/sec just in case.
    */
	if (! GK.second_tick)
		return;

	glibtop_get_mem (&glt_mem);
	mem.total   = (gulong) glt_mem.total;
	mem.x_used    = (gulong) glt_mem.used;
	mem.free    = (gulong) glt_mem.free;
	mem.shared  = (gulong) glt_mem.shared;
	mem.buffers = (gulong) glt_mem.buffer;
	mem.cached  = (gulong) glt_mem.cached;

	/* Not sure why, but glibtop has a used memory calculation:
	|  glt_mem.used = mem.total - mem.free - mem.shared
	|			- mem.buffers;
	|  while the free command calculates a "used" the way I have here:
	*/
	mem.used = mem.x_used - mem.buffers - mem.cached;

	glibtop_get_swap (&glt_swap);
	swap.total = (gulong) glt_swap.total;
	swap.used  = (gulong) glt_swap.used;
	if (glt_swap.flags
				& ((1 << GLIBTOP_SWAP_PAGEIN) + (1 << GLIBTOP_SWAP_PAGEOUT)))
		{
		swap_chart.page_in = (gulong) glt_swap.pagein;
		swap_chart.page_out = (gulong) glt_swap.pageout;
		}
	else
		{
		swap_chart.page_in = 0;
		swap_chart.page_out = 0;
		}
	}
#endif


/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_meminfo_interface()
	{
#if defined(__FreeBSD__)
	read_system_meminfo = read_freebsd_meminfo;
#elif defined(__linux__)
	read_system_meminfo = read_proc_meminfo;
#elif defined(__NetBSD__) || defined(__OpenBSD__)
	read_system_meminfo = read_netbsd_meminfo;
#else
	read_system_meminfo = read_glibtop_meminfo;
#endif
	return TRUE;
	}

/* ======================================================================== */


#define	DEFAULT_FORMAT	_("$t - $f free")

static gint			x_scroll,
					x_mon_motion,
					x_moved,
					ascent;

static MeminfoMeter		*mon_in_motion;

static gint			chart_height	= 20;

#define MEG(x)	(((x) + (2 << 19)) >> 20)


  /* Updating the /proc/meminfo meters can be expensive in Linux 2.2,
  |  so I do some dynamic adjustments on how often I do the updates.
  |  I increase the update rate if system activity is detected.
  |  This is an effort to get good meter response and to
  |  not contribute to cpu chart activity during quiet times, ie maintain
  |  a good S/N where I'm the noise.
  */
#define PIPE_SIZE	3
static gint		mem_pipe[PIPE_SIZE],
				swap_pipe[PIPE_SIZE];

gint
force_meminfo_update()
	{
	gint	i;

	if (GK.second_tick)
		{
		for (i = 1; i < PIPE_SIZE; ++i)
			if (mem_pipe[i] || swap_pipe[i])
				return TRUE;
		if (GK.cpu_sys_activity > 3)
			return TRUE;
		}
	return FALSE;
	}

static gint
format_meminfo_data(MeminfoMeter *mm, gchar *buf, gint size)
	{
	gulong	t, u, f;
	gchar	*s, *format, *label;
	gint	len;

	--size;
	*buf = '\0';
	label = mm->label;
	t = MEG(mm->total);
	u = MEG(mm->used);
	f = t - u;
	format = mm->data_format;

	for (s = format; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			switch(*(s + 1))
				{
				case 'l':
					len = snprintf(buf, size, "%s", label);
					break;
				case 't':
					len = snprintf(buf, size, "%ldM", t);
					break;
				case 'u':
					len = snprintf(buf, size, "%ldM", u);
					break;
				case 'U':
					if (t > 0)
						len = snprintf(buf, size, "%ld%%", 100 * u / t);
					break;
				case 'f':
					len = snprintf(buf, size, "%ldM", f);
					break;
				case 'F':
					if (t > 0)
						len = snprintf(buf, size, "%ld%%", 100 * f / t);
					break;
				case 's':
					if (mm == &mem)
						len = snprintf(buf, size, "%ldM", MEG(mem.shared));
					break;
				case 'b':
					if (mm == &mem)
						len = snprintf(buf, size, "%ldM",MEG(mem.buffers));
					break;
				case 'c':
					if (mm == &mem)
						len = snprintf(buf, size, "%ldM", MEG(mem.cached));
					break;
				default:
					*buf = *s;
					if (size > 1)
						{
						*(buf + 1) = *(s + 1);
						++len;
						}
					break;
				}
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';
	return u;		/* A way to know if decal text changed. */
	}

static gint
draw_decal_label(MeminfoMeter *mm, gint draw_to_screen)
	{
	Decal		*d;
	TextStyle	ts_save;
	gchar		buf[128];
	gint		changed;
	gint		w	= 0;

	d = mm->decal_label;
	if (! mm->label_is_data)
		{
		d->x_off = mm->x_label;
		gkrellm_draw_decal_text(mm->panel, d, mm->label, 0);
		}
    else
        {
		ts_save = d->text_style;
		d->text_style = *gkrellm_meter_alt_textstyle(mm->style_id);

		changed = format_meminfo_data(mm, buf, sizeof(buf));
		w = gdk_string_width(d->text_style.font, buf);

		if (w  > d->w)
			{
			d->x_off = d->w / 3 - x_scroll;
			changed = x_scroll + 1;
			}
		else
			d->x_off = 0;

		/* Draw the decal, actual draws occur only if "changed" is different
		|  from last call.  If scrolling, will be so each time, if not
		|  will be so as format_meminfo_data() return value changes.
		*/
		gkrellm_draw_decal_text(mm->panel, d, buf, changed);
		d->text_style = ts_save;
		}
	if (draw_to_screen)
		gkrellm_draw_layers(mm->panel);
	return w;
	}


static void
record_activity(gint *pipe, gint modified)
	{
	gint	i;

	for (i = PIPE_SIZE - 1; i > 0; --i)
		pipe[i] = pipe[i-1];
	pipe[0] = modified;
	}

static void
draw_extra(MeminfoChart *mc)
    {
    TextStyle   *ts;
    gchar       buf[32];
	gint		n;

    ts = gkrellm_chart_textstyle(DEFAULT_STYLE_ID);
    if (ascent == 0)
		ascent = gdk_char_height(ts->font, '8');
	n = (gint) mc->cur;
	if (n < 1000)
		snprintf(buf, sizeof(buf), "%d", n);
	else if (n < 19950)
		{
		n += 50;
		snprintf(buf, sizeof(buf), "%d.%dK", n / 1000, (n % 1000) / 100);
		}
	else
		{
		n += 500;
		snprintf(buf, sizeof(buf), "%dK", n / 1000);
		}
    gkrellm_draw_chart_label(mc->chart, ts, 4, 4 + ascent, buf);
    }


static void
refresh_chart(MeminfoChart *mc)
	{
	if (mc->chart)
		{
		gkrellm_draw_chart(mc->chart);
		if (mc->extra_info)
			draw_extra(mc);
		mc->chart->need_redraw = FALSE;
		}
	}

static gint
cb_extra(GtkWidget *widget, GdkEventButton *event, gpointer data)
	{
	MeminfoChart	*mc	= (MeminfoChart *) data;

	if (event->button == 1)
		{
		mc->extra_info = 1 - mc->extra_info;
		gkrellm_config_modified();
		}
	refresh_chart(mc);
	return TRUE;
	}

static void
update_meminfo(void)
	{
	Chart	*cp;
	gulong	u;
	gint	w_scroll, w;

	if (! (mem.enabled || swap.enabled || swap_chart.enabled))
		return;
	(*read_system_meminfo)();

	if (GK.second_tick)
		{
		MeminfoChart	*mc	= &swap_chart;

		if ((cp = mc->chart) != NULL && GK.second_tick)
			{
			if (cp->primed)
				mc->cur = (mc->page_in - cp->prevIn)
							+ (mc->page_out - cp->prevOut);
			gkrellm_store_chart_data(cp, mc->page_out, mc->page_in, 0);
			refresh_chart(mc);
			}
		}
	if (mem.enabled)
		{
		mem.krell_used->full_scale = (gint) (mem.total >> 12);
		mem.krell_used->previous = 0;
		u = mem.used >> 12;
		if (mem.label_is_data && mon_in_motion && x_moved)
			u = 0;
		gkrellm_update_krell(mem.panel, mem.krell_used, u);
		record_activity(mem_pipe, mem.krell_used->modified);
		}
	if (swap.enabled)
		{
		swap.krell_used->full_scale = (gint) (swap.total >> 12);
		swap.krell_used->previous = 0;
		u = swap.used >> 12;
		if (swap.label_is_data && mon_in_motion && x_moved)
			u = 0;
		gkrellm_update_krell(swap.panel, swap.krell_used, u);
		record_activity(swap_pipe, swap.krell_used->modified);
		}
	if (swap.krell_delta && swap.enabled)
		gkrellm_update_krell(swap.panel, swap.krell_delta,
				swap_chart.page_in + swap_chart.page_out);
	if (swap_chart.chart && swap_chart.chart->need_redraw)
		refresh_chart(&swap_chart);

	w = w_scroll = 0;
	if (mem.label_is_data && mon_in_motion != &mem && mem.enabled)
		w_scroll = draw_decal_label(&mem, 1);
	if (swap.label_is_data && mon_in_motion != &swap && swap.enabled)
		{
		if ((w = draw_decal_label(&swap, 1)) > w_scroll)
			w_scroll = w;
		}
	if (!mon_in_motion)
		{
		if (w_scroll > mem.decal_label->w)
			x_scroll = (x_scroll + ((UC.update_HZ < 7) ? 2 : 1))
						% (w_scroll - mem.decal_label->w / 3);
		else
			x_scroll = 0;
		}
	gkrellm_draw_layers(mem.panel);
	gkrellm_draw_layers(swap.panel);
	}

static gint
meminfo_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GdkPixmap	*pixmap	= NULL;

	if (widget == mem.panel->drawing_area)
		pixmap = mem.panel->pixmap;
	else if (widget == swap.panel->drawing_area)
		pixmap = swap.panel->pixmap;
	else if (swap_chart.chart && widget == swap_chart.chart->drawing_area)
		pixmap = swap_chart.chart->pixmap;
	if (pixmap)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
				ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				ev->area.width, ev->area.height);
	return FALSE;
	}

static gint
cb_panel_release(GtkWidget *widget, GdkEventButton *ev)
	{
	if (mon_in_motion)
		{
		if (mon_in_motion->restore_label)
			{
			if (mon_in_motion->label_is_data)
				gkrellm_config_modified();
			mon_in_motion->label_is_data = FALSE;
			draw_decal_label(mon_in_motion, 1);
			}
		mon_in_motion->restore_label = TRUE;
		}
	mon_in_motion = NULL;
	x_moved = FALSE;
	return TRUE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
    {
    if (ev->button != 2 && ev->button != 3)
		return TRUE;
	if (widget == mem.panel->drawing_area)
		mon_in_motion = &mem;
	else if (widget == swap.panel->drawing_area)
		mon_in_motion = &swap;
	else
		return TRUE;
	if (! mon_in_motion->label_is_data)
		{
		mon_in_motion->label_is_data = TRUE;
		mon_in_motion->restore_label = FALSE;
		gkrellm_config_modified();
		}
	x_mon_motion = ev->x;
	draw_decal_label(mon_in_motion, 1);
	x_moved = FALSE;
	return TRUE;
	}

static gint
cb_panel_motion(GtkWidget *widget, GdkEventButton *ev)
	{
	GdkModifierType	state;
	Decal			*d;
	GdkFont			*font;
	gchar			buf[128];
	gint			w, x_delta;

	state = ev->state;
	if (   ! mon_in_motion
		|| ! (state & (GDK_BUTTON2_MASK | GDK_BUTTON3_MASK))
		|| ! mon_in_motion->label_is_data
	   )
		{
		mon_in_motion = NULL;
		return TRUE;
		}
	d = mon_in_motion->decal_label;
	font = gkrellm_meter_alt_textstyle(mon_in_motion->style_id)->font;

	format_meminfo_data(mon_in_motion, buf, sizeof(buf));
	w = gdk_string_width(font, buf);
	if (w > d->w)
		{
		x_delta = ev->x - x_mon_motion;
		x_mon_motion = ev->x;
		d->x_off += x_delta;
		if (d->x_off < -w)
			d->x_off = -w;
		if (d->x_off > d->w)
			d->x_off = d->w;
		x_scroll = d->w / 3 - d->x_off;
		if (mem.label_is_data)
			draw_decal_label(&mem, 1);
		if (swap.label_is_data)
			draw_decal_label(&swap, 1);
		mon_in_motion->restore_label = FALSE;
		}
	x_moved = TRUE;
	return TRUE;
	}

static void
setup_scaling(MeminfoChart *mc, MeminfoMeter *mm)
	{
	Chart	*cp	= mc->chart;
	gint	grids;

	grids = UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS;
	if (cp)
		{
		cp->scale_min = mc->resolution;
		cp->scale_max = 0;		/* Force a chart redraw */
		}
	if (mm->krell_delta)
		mm->krell_delta->full_scale =
					mc->resolution * grids / UC.update_HZ;
	}

static void
destroy_chart(MeminfoChart *mc)
	{
	if (! mc->chart)
		return;
	gkrellm_monitor_height_adjust( - mc->chart->h);
	gkrellm_destroy_chart(mc->chart);
	g_free(mc->chart);
	mc->chart = NULL;
	mc->enabled = FALSE;
	gtk_widget_hide(mc->vbox);
	}

static void
create_chart(MeminfoChart *mc, gint first_create)
	{
	Chart		*cp;

	if (first_create)
		mc->chart = gkrellm_chart_new0();

	if ((cp = mc->chart) == NULL)
		return;
	cp->h = chart_height;
	gkrellm_create_chart(mc->vbox, cp, DEFAULT_STYLE_ID);
	gkrellm_monitor_height_adjust(cp->h);
	gkrellm_alloc_chart_data(cp);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "expose_event",
				(GtkSignalFunc) meminfo_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_extra, mc);
		gtk_widget_show(mc->vbox);
		}
	else
		refresh_chart(mc);	/* Avoid second lag at theme/size switches */
	mc->enabled = TRUE;
	ascent = 0;
	}

static void
create_panel(GtkWidget *vbox, MeminfoMeter *mm, gint first_create)
	{
	Panel		*p;
	TextStyle	*ts;
	Style		*style;

	if (first_create)
		mm->panel = gkrellm_panel_new0();
	else
		{
		gkrellm_destroy_decal_list(mm->panel);
		gkrellm_destroy_krell_list(mm->panel);
		}
	p = mm->panel;

	if (mm == &swap)
		{
		style = gkrellm_panel_style(DEFAULT_STYLE_ID);
		mm->krell_delta = gkrellm_create_krell(p,
					gkrellm_krell_panel_image(DEFAULT_STYLE_ID), style);
		}

	style = gkrellm_meter_style(mm->style_id);
	mm->krell_used = gkrellm_create_krell(p,
				gkrellm_krell_meter_image(mm->style_id), style);
	p->textstyle = gkrellm_meter_textstyle(mm->style_id);
	mm->decal_label = gkrellm_create_decal_text(p,
			(mm == &swap) ? "Sp8" : "M8", p->textstyle, style, -1, -1, -1);
	gkrellm_configure_panel(p, NULL, style);
	gkrellm_create_panel(vbox, p, gkrellm_bg_meter_image(mm->style_id));
	ts = &mm->decal_label->text_style;
	mm->x_label = x_label_position(p->label->position, mm->decal_label->w,
				gdk_string_width(ts->font, mm->label), 0);
	draw_decal_label(mm, 0);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT (p->drawing_area),
				"expose_event", (GtkSignalFunc) meminfo_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
			"button_press_event", (GtkSignalFunc) cb_panel_press, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
			"button_release_event", (GtkSignalFunc) cb_panel_release, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"motion_notify_event", (GtkSignalFunc) cb_panel_motion, NULL);
		}
	gkrellm_setup_launcher(p, &mm->launch, METER_PANEL_TYPE, 4);

	if (mm->enabled)
		gkrellm_monitor_height_adjust(p->h);
	else
		gtk_widget_hide(p->hbox);
	}

  /* The mem and swap meters are a unit (for looks and because some themes
  |  assume that), so I stick the swap_chart just above the mem meter.
  |  This code assumes mem is stacked on top of swap (ie mem created first)
  */
static void
create_mem(GtkWidget *vbox, gint first_create)
	{
	if (first_create)
		{
		mem.label = _("Mem");

		swap_chart.vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(vbox), swap_chart.vbox);
		}
	create_panel(vbox, &mem, first_create);
	}

static void
create_swap(GtkWidget *vbox, gint first_create)
	{
	if (first_create)
		swap.label = _("Swap");

	if (swap_chart.enabled)
		create_chart(&swap_chart, first_create);

	create_panel(vbox, &swap, first_create);
	setup_scaling(&swap_chart, &swap);
	}

#define	MEM_CONFIG_KEYWORD	"meminfo"

static void
save_meminfo_config(FILE *f)
	{
	fprintf(f, "%s mem_meter %d %d\n", MEM_CONFIG_KEYWORD,
				mem.enabled, mem.label_is_data);
	fprintf(f, "%s swap_meter %d %d\n", MEM_CONFIG_KEYWORD,
				swap.enabled, swap.label_is_data);
	fprintf(f, "%s swap_chart %d %d %d\n", MEM_CONFIG_KEYWORD,
			swap_chart.enabled, swap_chart.resolution, swap_chart.extra_info);
	fprintf(f, "%s chart_height %d\n", MEM_CONFIG_KEYWORD, chart_height);

	fprintf(f, "%s mem_launch %s\n", MEM_CONFIG_KEYWORD,
				mem.launch.command);
	fprintf(f, "%s mem_tooltip %s\n", MEM_CONFIG_KEYWORD,
				mem.launch.tooltip_comment);
	fprintf(f, "%s mem_data_format %s\n", MEM_CONFIG_KEYWORD,mem.data_format);

	fprintf(f, "%s swap_launch %s\n", MEM_CONFIG_KEYWORD,
				swap.launch.command);
	fprintf(f, "%s swap_tooltip %s\n", MEM_CONFIG_KEYWORD,
				swap.launch.tooltip_comment);
	fprintf(f, "%s swap_data_format %s\n", MEM_CONFIG_KEYWORD,
				swap.data_format);
	}

static void
load_meminfo_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (strcmp(config, "mem_meter") == 0)
			{
			sscanf(item, "%d %d", &mem.enabled, &mem.label_is_data);
			if (mem.label_is_data)
				mem.restore_label = TRUE;
			}
		else if (strcmp(config, "swap_meter") == 0)
			{
			sscanf(item, "%d %d", &swap.enabled, &swap.label_is_data);
			if (swap.label_is_data)
				swap.restore_label = TRUE;
			}
		else if (strcmp(config, "swap_chart") == 0)
			sscanf(item, "%d %d %d", &swap_chart.enabled,
			&swap_chart.resolution, &swap_chart.extra_info);
		else if (strcmp(config, "chart_height") == 0)
			sscanf(item, "%d", &chart_height);
		else if (strcmp(config, "mem_launch") == 0)
			mem.launch.command = g_strdup(item);
		else if (strcmp(config, "mem_tooltip") == 0)
			mem.launch.tooltip_comment = g_strdup(item);
		else if (strcmp(config, "mem_data_format") == 0)
			gkrellm_dup_string(&mem.data_format, item);
		else if (strcmp(config, "swap_launch") == 0)
			swap.launch.command = g_strdup(item);
		else if (strcmp(config, "swap_tooltip") == 0)
			swap.launch.tooltip_comment = g_strdup(item);
		else if (strcmp(config, "swap_data_format") == 0)
			gkrellm_dup_string(&swap.data_format, item);
		}
	}

/* --------------------------------------------------------------------- */
static GtkWidget	*mem_meter_enable_button,
					*swap_meter_enable_button,
					*swap_chart_enable_button;

static GtkWidget	*swap_chart_spin_button,
					*height_spin_button;

static GtkWidget	*mem_launch_entry,
					*mem_tooltip_entry,
					*swap_launch_entry,
					*swap_tooltip_entry;

static GtkWidget	*mem_format_combo,
					*swap_format_combo;

static void
apply_meminfo_config()
	{
	gchar	*s;
	gint	new_enabled, new_height;

	s = gkrellm_entry_get_text(&(GTK_COMBO(mem_format_combo)->entry));
	gkrellm_dup_string(&mem.data_format, s);
	s = gkrellm_entry_get_text(&(GTK_COMBO(swap_format_combo)->entry));
	gkrellm_dup_string(&swap.data_format, s);

	gkrellm_apply_launcher(&mem_launch_entry, &mem_tooltip_entry, mem.panel,
				&mem.launch, gkrellm_launch_button_cb);
	gkrellm_apply_launcher(&swap_launch_entry, &swap_tooltip_entry, swap.panel,
				&swap.launch, gkrellm_launch_button_cb);

	new_enabled = GTK_TOGGLE_BUTTON(mem_meter_enable_button)->active;
	gkrellm_enable_visibility(new_enabled, &mem.enabled,
					mem.panel->hbox, mem.panel->h);
	new_enabled = GTK_TOGGLE_BUTTON(swap_meter_enable_button)->active;
	gkrellm_enable_visibility(new_enabled, &swap.enabled,
					swap.panel->hbox, swap.panel->h);

	new_height = gtk_spin_button_get_value_as_int(
				GTK_SPIN_BUTTON(height_spin_button));
	if (new_height != chart_height)
		{
		if (swap_chart.enabled)
			destroy_chart(&swap_chart);
		}
	chart_height = new_height;

	swap_chart.resolution = gtk_spin_button_get_value_as_int(
				GTK_SPIN_BUTTON(swap_chart_spin_button));
	new_enabled = GTK_TOGGLE_BUTTON(swap_chart_enable_button)->active;
	if (new_enabled && ! swap_chart.enabled)
		create_chart(&swap_chart, TRUE);
	else if (! new_enabled && swap_chart.enabled)
		destroy_chart(&swap_chart);

	setup_scaling(&swap_chart, &swap);
	}

static gint     res_map[] =
	{ 20, 50, 100, 200, 500, 1000, 2000, 5000, 10000, 20000, 50000, 100000 };

static void
cb_resolution(GtkWidget *widget, GtkSpinButton *spin)
	{
	gint    	value;
	static gint	temp;

	value = gtk_spin_button_get_value_as_int(spin);
	if (value != temp) /* Avoid recursion */
		{
		value = map_1_2_5(value, res_map, sizeof(res_map) / sizeof(gint));
		temp = value;
		gtk_spin_button_set_value(spin, (gfloat) value);
		}
	}

static void
create_meminfo_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox;
	GtkWidget		*table;
	GtkWidget		*hbox;
	GtkWidget		*label;
	GtkWidget		*sep;
	GList			*list;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* --Setup Tab */
	vbox = gkrellm_create_tab(tabs, _("Setup"));

    gkrellm_check_button(vbox, &swap_chart_enable_button, swap_chart.enabled,
				 TRUE, 0, _("Enable swap pages in/out chart"));
    gkrellm_check_button(vbox, &mem_meter_enable_button, mem.enabled,
				TRUE, 0, _("Enable memory meter"));
    gkrellm_check_button(vbox, &swap_meter_enable_button, swap.enabled,
				TRUE, 0, _("Enable swap meter"));

	gkrellm_spin_button(vbox, &swap_chart_spin_button,
			(gfloat) swap_chart.resolution,
			20.0, 100000.0, 1.0, 1.0, 0, 60, cb_resolution, NULL,
			FALSE, _("Swap chart resolution in pages/sec per grid"));

	gkrellm_spin_button(vbox, &height_spin_button, (gfloat) chart_height,
			10.0, (gfloat) GK.max_chart_height, 1.0, 5.0, 0, 60, NULL, NULL,
			FALSE, _("Swap chart height"));

/* --Options Tab */
	vbox = gkrellm_create_tab(tabs, _("Options"));
	label = gtk_label_new(
_("A middle or right click in the Mem or Swap panel toggles a display of\n"
"memory or swap capacities in a format controlled by a user supplied format\n"
"string.  For memory and swap, $t is total MiB, $u is used MiB, $f is free MiB,\n"
"$U is used %, $F is free %, and $l is the panel label.\n"
"For memory only, $s is shared MiB, $b is buffered MiB, and $c is cached MiB.\n"
"MiB is a binary megabyte (2^20)."));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 2);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	mem_format_combo = gtk_combo_new();
	gtk_box_pack_start(GTK_BOX(hbox), mem_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, DEFAULT_FORMAT);
	list = g_list_append(list, _("$t - $u used"));
	list = g_list_append(list, _("$t - $U"));
	list = g_list_append(list, _("$t - $u used  $s sh  $b bf  $c ca"));
	gtk_combo_set_popdown_strings(GTK_COMBO(mem_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(mem_format_combo)->entry),
			mem.data_format);
	label = gtk_label_new(_("Format string for Mem data display"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	swap_format_combo = gtk_combo_new();
	gtk_box_pack_start(GTK_BOX(hbox), swap_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, DEFAULT_FORMAT);
	list = g_list_append(list, _("$t - $u used"));
	list = g_list_append(list, _("$t - $U"));
	gtk_combo_set_popdown_strings(GTK_COMBO(swap_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(swap_format_combo)->entry),
			swap.data_format);
	label = gtk_label_new(_("Format string for Swap data display"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), sep, TRUE, TRUE, 0);

	table = gkrellm_launcher_table_new(vbox, 2);
	gkrellm_config_launcher(table, 0,  &mem_launch_entry, &mem_tooltip_entry,
		_("Mem"), &(mem.launch));
	gkrellm_config_launcher(table, 1,  &swap_launch_entry, &swap_tooltip_entry,
		_("Swap"), &(swap.launch));
	}


  /* The meminfo monitor is a bit of a hybrid.  To provide for easy theming,
  |  the mem, swap, and swap_chart monitors are created as separate monitors,
  |  but they all have several common routines (update, config, ...).  Where
  |  a common routine is used, it is entered in only one of the Monitor
  |  structures, and NULL is entered in the others.
  */
static Monitor	mon_mem =
	{
	N_("Memory"),			/* Name, for config tab.	*/
	MON_MEM,					/* Id,  0 if a plugin		*/
	create_mem,			/* The create function		*/
	update_meminfo,		/* The update function		*/
	create_meminfo_tab, /* The config tab create function	*/
	apply_meminfo_config, /* Apply the config function		*/

	save_meminfo_config, /* Save user conifg			*/
	load_meminfo_config, /* Load user config			*/
	MEM_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_mem_monitor(void)
	{
	mon_mem.name = _(mon_mem.name);
	mem.style_id = gkrellm_add_meter_style(&mon_mem, MEM_STYLE_NAME);
	mem.data_format = g_strdup(DEFAULT_FORMAT);
	mem.enabled = TRUE;

	if (setup_meminfo_interface())
		return &mon_mem;
	return NULL;
	}

static Monitor	mon_swap =
	{
	NULL,			/* Name, for config tab. Done in mon_mem*/
	MON_SWAP,		/* Id,  0 if a plugin		*/
	create_swap,	/* The create function		*/
	NULL,			/* The update function		*/
	NULL,			/* The config tab create function	*/
	NULL,			/* Apply the config function		*/
	NULL,			/* Save user conifg			*/
	NULL,			/* Load user config			*/
	NULL,			/* config keyword			*/
	NULL,			/* Undef 2	*/
	NULL,			/* Undef 1	*/
	NULL,			/* Undef 0	*/
	0,				/* insert_before_id - place plugin before this mon */
	NULL,			/* Handle if a plugin, filled in by GKrellM		*/
	NULL			/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_swap_monitor(void)
	{
	swap.style_id = gkrellm_add_meter_style(&mon_swap, SWAP_STYLE_NAME);
	swap.data_format = g_strdup(DEFAULT_FORMAT);
	swap.enabled = TRUE;

	/* See comments above about swap_chart wrt the meters.
	*/
	swap_chart.resolution = 1000;
	swap_chart.enabled = FALSE;

	if (setup_meminfo_interface())
		return &mon_swap;
	return NULL;
	}
