/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellm.h"


/* ----------------------------------------------------------------------- */

void
gkrellm_clear_chart(Chart *pC)
	{
	gdk_draw_pixmap(pC->pixmap, GK.draw1_GC, pC->bg_pixmap,
						0, 0,	0, 0,   pC->w, pC->h);
	gdk_draw_pixmap(pC->drawing_area->window, GK.draw1_GC, pC->pixmap,
						0, 0,	0, 0,   pC->w, pC->h);
	}

gint
computed_index(Chart *pC, gint i)
	{
	gint	x;
	
	x = (pC->position + i + 1) % pC->w;
	return x;
	}


  /* Avoid runaway grid drawing for badly set grid resolutions.  This
  |  clips chart drawing to the number of GRID_LIMIT lines.  It may
  |  look like a bug when triggered, but flags that you should change the
  |  grid resolution.
  */
#define	GRID_LIMIT	24
#define	CHART_MARGIN	1

static gint
chart_ycoord(Chart *pC, gint val)
	{
	glong	dy, t;
	
	if (pC->scale_max <= 0)
		pC->scale_max = 1;
	dy = (long) (pC->h - pC->y - (2 * CHART_MARGIN));
	t = ((long) val * dy / pC->scale_max);
	if (t < 0L)
		t = 0L;
	if (t >= dy)
		t = dy - 1;
	t += CHART_MARGIN;

	/* Map 0,0 of window upper left to lower left for my charts.
	|  Account for offset (if any) of the chart into its pixmap (pC->y).
	*/
	return pC->h - 1 - (int) t - pC->y;
	}

static gint
scale_chart(Chart *pC)
	{
	glong		scalemax;
	gint		y, y0;
	gint		i0;

	/* maxval is dynamic, so compute a scalemax and compare to last
	|  pC->scale_max.  Redraw grided background if different.
	*/
	if (UC.fixed_scale)
		scalemax = pC->scale_min * UC.fixed_scale;
	else	/* Auto scale to pC->maxval */
		{
		if (pC->maxval == 0)
			scalemax = pC->scale_min;
		else
			scalemax = ((pC->maxval - 1) / pC->scale_min + 1) * pC->scale_min;
		}
	if (scalemax > GRID_LIMIT * pC->scale_min)
		scalemax = GRID_LIMIT * pC->scale_min;
	if (scalemax != pC->scale_max)
		{
		i0 = 0;		/* Will draw all lines */
		pC->scale_max = scalemax;

		/* Redraw the background and data pixmaps with new grid lines.
		*/
		gdk_draw_pixmap(pC->bg_grided_pixmap, GK.draw1_GC, pC->bg_pixmap,
						0, 0,	0, 0,   pC->w, pC->h);
		gdk_draw_pixmap(pC->in_grided_pixmap, GK.draw1_GC, GK.data_in_pixmap,
						0, 0,	0, 0,   pC->w, pC->h);
		gdk_draw_pixmap(pC->out_grided_pixmap, GK.draw1_GC, GK.data_out_pixmap,
						0, 0,	0, 0,   pC->w, pC->h);
		for (y = 0; y <= scalemax; y += pC->scale_min)
			{
			if (   GK.bg_grid_mode == GRID_MODE_RESTRAINED
				&& ((y == 0 && pC->y == 0) || y == scalemax)
			   )
				continue;
			y0 = chart_ycoord(pC, y);
			gdk_draw_pixmap(pC->bg_grided_pixmap, GK.draw1_GC,
					pC->grid_pixmap, 0, 0, pC->x, y0, pC->w,
					((GdkWindowPrivate *) pC->bg_grided_pixmap)->height);
			gdk_draw_pixmap(pC->in_grided_pixmap, GK.draw1_GC,
					GK.data_in_grid_pixmap, 0, 0, pC->x, y0, pC->w,
					((GdkWindowPrivate *) GK.data_in_grid_pixmap)->height);
			gdk_draw_pixmap(pC->out_grided_pixmap, GK.draw1_GC,
					GK.data_out_grid_pixmap, 0, 0, pC->x, y0, pC->w,
					((GdkWindowPrivate *) GK.data_out_grid_pixmap)->height);
			}
		}
	else
		{
		i0 = pC->w - 1;		/* Will draw the last line only */
		}
	/* Clear the area of the data bitmaps that data will be drawn into
	*/
	gdk_draw_rectangle(pC->in_data_bitmap, GK.bit0_GC, TRUE,
				i0, 0, pC->w - i0, pC->h);
	gdk_draw_rectangle(pC->out_data_bitmap, GK.bit0_GC, TRUE,
				i0, 0, pC->w - i0, pC->h);
	return i0;
	}

  /* Special case chart draw for the Proc monitor. Out is load, In is processes
  */
void
draw_proc_chart(Chart *pC)
	{
	gint		n, x, y, y0, y1, y2;
	gint		i0;
	static gint	y_prev;		/* OK since there is only one proc chart */

	i0 = scale_chart(pC);
	for (n = i0; n < pC->w; ++n)
		{
		x = computed_index(pC, n);
		y0 = chart_ycoord(pC, 0);
		y1 = chart_ycoord(pC, (int) pC->pDataOut[x]);
		y2 = chart_ycoord(pC, (int) pC->pDataIn[x]);
		if (n == 0)
			y_prev = y1;

		if (pC->pDataOut[x] > 0)
			{
			if (UC.proc_load_bar_graph)
				gdk_draw_line(pC->out_data_bitmap, GK.bit1_GC,
								pC->x + n, y0, pC->x + n, y1);
			else
				{
				gdk_draw_line(pC->out_data_bitmap, GK.bit1_GC,
								pC->x + n, y_prev, pC->x + n, y1);
				if (y1 < y0)
					gdk_draw_point(pC->out_data_bitmap, GK.bit1_GC,
								pC->x + n, y1 + 1);
				}
			}
		y_prev = y1;

		if (pC->pDataIn[x] > 0)
			gdk_draw_line(pC->in_data_bitmap, GK.bit1_GC,
					pC->x + n, y0, pC->x + n, y2);
		}
	/* Redraw the expose pixmap. First copy the bg_grided_pixmap to it, then
	|  push the in/out grided pixmaps through the in/out data bitmaps.
	*/
	gdk_draw_pixmap(pC->pixmap, GK.draw1_GC, pC->bg_grided_pixmap,
						0, 0,	0, 0,   pC->w, pC->h);
	if (pC->out_maxval > 0)
		{
		y = chart_ycoord(pC, pC->out_maxval);
		gdk_gc_set_clip_mask(GK.draw1_GC, pC->out_data_bitmap);
		gdk_draw_pixmap(pC->pixmap, GK.draw1_GC, pC->out_grided_pixmap,
						0, y,  0, y,  pC->w, pC->h - y);
		}
	if (pC->in_maxval > 0)
		{
		y = chart_ycoord(pC, pC->in_maxval);
		gdk_gc_set_clip_mask(GK.draw1_GC, pC->in_data_bitmap);
		gdk_draw_pixmap(pC->pixmap, GK.draw1_GC, pC->in_grided_pixmap,
						0, y,  0, y,  pC->w, pC->h - y);
		}
	gdk_gc_set_clip_mask(GK.draw1_GC, NULL);

	/* Draw the expose pixmap onto the screen.
	*/
	gdk_draw_pixmap(pC->drawing_area->window, GK.draw1_GC, pC->pixmap,
						0, 0,	0, 0,   pC->w, pC->h);
	}


  /* The generic chart draw.  Out/In data are bar graphs.  Draw by pushing
  |  grided in/out images through data bitmaps.  Unless the chart is rescaled,
  |  this allows me to scroll the bitmaps left, draw one line, and then push
  |  in/out images through the bitmaps.  This optimizes for reduced X request
  |  traffic, very useful when GKrellM is networked.
  */
void
gkrellm_draw_chart(Chart *pC)
	{
	gint		n, x, y, y0, y1, y2;
	gint		i0;

	if (pC->pDataOut == NULL || pC->pDataIn == NULL)
		return;
	i0 = scale_chart(pC);

	/* Draw data onto the bitmaps.  Will either draw one line if chart
	|  was scrolled or all lines if the chart was rescaled.
	*/
	for (n = i0; n < pC->w; ++n)
		{
		x = computed_index(pC, n);
		y0 = chart_ycoord(pC, 0);
		y1 = chart_ycoord(pC, pC->pDataOut[x]);
		y2 = chart_ycoord(pC, pC->pDataOut[x] + pC->pDataIn[x]);
		if (pC->pDataOut[x] > 0)
			gdk_draw_line(pC->out_data_bitmap, GK.bit1_GC, n, y0, n, y1);
		if (pC->pDataIn[x] > 0)
			gdk_draw_line(pC->in_data_bitmap, GK.bit1_GC, n, y1, n, y2);
		}

	/* Redraw the expose pixmap. First copy the bg_grided_pixmap to it, then
	|  push the in/out grided pixmaps through the in/out data bitmaps.
	*/
	gdk_draw_pixmap(pC->pixmap, GK.draw1_GC, pC->bg_grided_pixmap,
						0, 0,	0, 0,   pC->w, pC->h);
	if (pC->maxval > 0)
		{
		y = chart_ycoord(pC, pC->maxval);
		gdk_gc_set_clip_mask(GK.draw1_GC, pC->in_data_bitmap);
		gdk_draw_pixmap(pC->pixmap, GK.draw1_GC, pC->in_grided_pixmap,
						0, y,  0, y,  pC->w, pC->h - y);
		}
	if (pC->out_maxval > 0)
		{
		y = chart_ycoord(pC, pC->out_maxval);
		gdk_gc_set_clip_mask(GK.draw1_GC, pC->out_data_bitmap);
		gdk_draw_pixmap(pC->pixmap, GK.draw1_GC, pC->out_grided_pixmap,
						0, y,  0, y,  pC->w, pC->h - y);
		}
	gdk_gc_set_clip_mask(GK.draw1_GC, NULL);

	/* Draw the expose pixmap onto the screen.
	*/
	gdk_draw_pixmap(pC->drawing_area->window, GK.draw1_GC, pC->pixmap,
						0, 0,	0, 0,   pC->w, pC->h);
	}

static void
scroll_data_bitmaps(Chart *pC)
	{
	gdk_draw_pixmap(pC->in_data_bitmap, GK.bit1_GC, pC->in_data_bitmap,
					1, 0, 0, 0, pC->w - 1, pC->h);
	gdk_draw_pixmap(pC->out_data_bitmap, GK.bit1_GC, pC->out_data_bitmap,
					1, 0, 0, 0, pC->w - 1, pC->h);
	}


void
gkrellm_alloc_chart_data(Chart *pC)
	{
	gint	*pNewOut, *pNewIn;
	size_t	w;

	w = (size_t) pC->w;

	if (pC->alloc_width != w)
		{
		pNewOut = (gint *) g_new0(gint, w);
		pNewIn  = (gint *) g_new0(gint, w);
		pC->alloc_width = w;

		if (pC->pDataOut)
			g_free(pC->pDataOut);
		if (pC->pDataIn)
			g_free(pC->pDataIn);
		pC->pDataOut = pNewOut;
		pC->pDataIn = pNewIn;
		pC->position = pC->w - 1;
		pC->maxval = 0;
		}
	pC->scale_max = 0;
	pC->type = 0;
	}

  /* Process chart is done differently.
  */
void
store_proc_chart_data(Chart *pC, gint load, gint processes)
	{
	gint			i, n, discard;

	if (pC->pDataOut == NULL || pC->pDataIn == NULL)
		return;
	if (load < pC->prevOut || processes < pC->prevIn)
		pC->primed = FALSE;

	if (pC->primed)
		{
		/* Increment position in circular buffer and remember the data
		| value to be thrown out.
		*/
		pC->position = (pC->position + 1) % pC->w;
		n = pC->position;

		discard = pC->pDataOut[n];
		if (!UC.proc_clip_processes && discard < pC->pDataIn[n])
			discard = pC->pDataIn[n];

		pC->pDataOut[n] = load - (int) pC->prevOut;
		pC->pDataIn[n] = processes - (int) pC->prevIn;

		/* If throwing out old maxval, scan data to assign new maxval.
		*/
		if (discard >= pC->maxval)
			{
			pC->maxval = 0;
			for (i = 0; i < pC->w; ++i)
				{
				if (pC->pDataOut[i] > pC->maxval)
				    pC->maxval = pC->pDataOut[i];
				if (!UC.proc_clip_processes && pC->pDataIn[i] > pC->maxval)
					pC->maxval = pC->pDataIn[i];
				if (pC->pDataIn[i] > pC->in_maxval)
					pC->in_maxval = pC->pDataIn[i];
				if (pC->pDataOut[i] > pC->out_maxval)
					pC->out_maxval = pC->pDataOut[i];
				}
			}
		if (pC->in_maxval < pC->pDataIn[n])
			pC->in_maxval = pC->pDataIn[n];
		if (pC->out_maxval < pC->pDataOut[n])
			pC->out_maxval = pC->pDataOut[n];
		if (pC->maxval < pC->pDataOut[n])
			pC->maxval = pC->pDataOut[n];
		if ( !UC.proc_clip_processes && pC->pDataIn[n] > pC->maxval)
			pC->maxval = pC->pDataIn[n];

		scroll_data_bitmaps(pC);
		}
	pC->prevOut = load;
	pC->prevIn = processes;
	pC->primed = TRUE;
	}

  /* If total is given, scale the stored data to range between 0 and
  |  5x the scale_min.  For cpu data which I plot as a percent usage.
  */
void
gkrellm_store_chart_data(Chart *pC, unsigned long out, unsigned long in,
								unsigned long total)
	{
	unsigned long	out_diff, in_diff, total_diff, range;
	gint			i, n;
	gint			discard, discard_in, discard_out;

	if (pC->pDataOut == NULL || pC->pDataIn == NULL)
		return;

	/* In case data (eg ppp0 interface going down/up?) is ever reset to zero
	|  or ethx interface wraps around unsigned long size, etc
	*/
	if (out < pC->prevOut || in < pC->prevIn)
		pC->primed = FALSE;

	if (pC->primed)
		{
		/* Increment position in circular buffer and remember the data
		|  value to be thrown out.
		*/
		pC->position = (pC->position + 1) % pC->w;
		n = pC->position;

		discard_in = pC->pDataIn[n];
		discard_out = pC->pDataOut[n];
		discard = discard_in + discard_out;

		out_diff = out - pC->prevOut;
		in_diff = in - pC->prevIn;
		if (total)
			{
			range = (UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS)
						 * pC->scale_min;
			if ((total_diff = total - pC->prevTotal) > 0)
				{
				out_diff = ((out_diff * range * 2 / total_diff) + 1) / 2;
				in_diff = ((in_diff * range * 2 / total_diff) + 1) / 2;
				}
			if (out_diff + in_diff > range)
				{
				if (out_diff > in_diff)
					out_diff = range - in_diff;
				else
					in_diff = range - out_diff;
				}
			}
		pC->pDataOut[n] = (gint) out_diff;
		pC->pDataIn[n] = (gint) in_diff;

		/* If throwing out old maxval, scan data to assign new maxval.
		*/
		if (   discard >= pC->maxval
			|| discard_in >= pC->in_maxval || discard_out >= pC->out_maxval
		   )
			{
			pC->maxval = pC->in_maxval = pC->out_maxval = 0;
			for (i = 0; i < pC->w; ++i)
				{
				if (pC->pDataOut[i] + pC->pDataIn[i] > pC->maxval)
				    pC->maxval = pC->pDataOut[i] + pC->pDataIn[i];
				if (pC->pDataIn[i] > pC->in_maxval)
					pC->in_maxval = pC->pDataIn[i];
				if (pC->pDataOut[i] > pC->out_maxval)
					pC->out_maxval = pC->pDataOut[i];
				}
			}
		if (pC->maxval < (gint) (out_diff + in_diff))
			pC->maxval = (gint) (out_diff + in_diff);
		if (pC->in_maxval < (gint) in_diff)
			pC->in_maxval = (gint) in_diff;
		if (pC->out_maxval < (gint) out_diff)
			pC->out_maxval = (gint) out_diff;

		scroll_data_bitmaps(pC);
		}
	pC->prevOut = out;
	pC->prevIn = in;
	pC->prevTotal = total;
	pC->primed = TRUE;
	}

void
gkrellm_reset_chart_data(Chart *pC)
	{
	gint	i;

	pC->maxval = 0;
	pC->scale_max = 0;
	pC->position = pC->w - 1;
	pC->primed = FALSE;

	for (i = 0; i < pC->w; ++i)
		{
		pC->pDataOut[i] = 0;
		pC->pDataIn[i] = 0;
		}
	}

void
gkrellm_reset_chart(Chart *pC)
	{
	gkrellm_reset_chart_data(pC);
	if (pC->type == 1)
		draw_proc_chart(pC);
	else
		gkrellm_draw_chart(pC);
	}
