/*
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
| 10/12/2000 NetBSD code contributed by Anthony Mallet
|			<metall@ficus.yi.org>
| 3/6/2000	Patch from Kazuhisa TAKEI <takei@vinelinux.org> added a toggle
|			to show percentage time left on apm panel.
| 2/25/2000	FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"


  /* Structure that (*(read_apm_data))() must fill in
  */
struct
	{
	gint	battery_is_available;
	gint	ac_is_on_line;
	gint	battery_is_charging;
	gint	battery_percentage;
	gint	battery_time_left;		/* In minutes	*/
	gint	battery_display_percent;
	}
	sys_apm;

static gint		enable_apm;

void    (*read_apm_data)();


/* ====== System dependent interface ====================================== */


#define	L_NO_BATTERY	0x80
#define	L_ON_LINE		1
#define	L_CHARGING		3
#define L_UNKNOWN		0xFF

/* ------- FreeBSD ------------------------------------------------------- */

#if defined(__FreeBSD__)
#include <osreldate.h>
#include <machine/apm_bios.h>
#define	APMDEV		"/dev/apm"

static void
read_freebsd_apm_data()
	{
	int		f, r;
	struct apm_info	info;

	if ((f = open(APMDEV, O_RDONLY)) == -1)
		return;
	r = ioctl(f, APMIO_GETINFO, &info);
	close(f);
	if (r == -1)
		return;
	sys_apm.battery_is_available = (info.ai_batt_stat != L_UNKNOWN);
	sys_apm.ac_is_on_line = (info.ai_acline == L_ON_LINE) ? TRUE : FALSE;
	sys_apm.battery_is_charging = (info.ai_batt_stat == L_CHARGING) ? TRUE : FALSE;
	sys_apm.battery_percentage = info.ai_batt_life;
#if defined(APM_GETCAPABILITIES)
	sys_apm.battery_time_left = info.ai_batt_time / 60;
#else
	sys_apm.battery_time_left = -1;
#endif
	}
#endif


/* ------- Linux ------------------------------------------------------- */
/* ----- see /usr/src/linux/arch/i386/kernel/apm.c ----- */
#if defined(__linux__)

#define	PROC_APM_FILE	"/proc/apm"

static void
read_linux_apm_data()
	{
	FILE	*f;
	gchar	buf[128];
	gint	ac_line_status,
			battery_status,
			flag,
			percentage,
			time;
	gchar	units[32];

	if ((f = fopen(PROC_APM_FILE, "r")) == NULL)
		return;
	fgets(buf, sizeof(buf), f);
	fclose(f);

	sscanf(buf, "%*s %*d.%*d %*x %x %x %x %d%% %d %s\n", &ac_line_status,
			&battery_status, &flag, &percentage, &time, units);

	if ((flag & L_NO_BATTERY) == 0 && battery_status != L_UNKNOWN)
		sys_apm.battery_is_available = TRUE;
	else
		sys_apm.battery_is_available = FALSE;

	sys_apm.ac_is_on_line = (ac_line_status == L_ON_LINE) ? TRUE : FALSE;
	sys_apm.battery_is_charging= (battery_status == L_CHARGING) ? TRUE : FALSE;
	sys_apm.battery_percentage = percentage;
	sys_apm.battery_time_left = time;
	if (strcmp(units, "sec") == 0)
		sys_apm.battery_time_left /= 60;
	}
#endif	/* __linux__ */


/* ------- NetBSD ------------------------------------------------------- */

#if (defined(__NetBSD__) || defined(__OpenBSD__)) && defined(__i386__)

#include <machine/apmvar.h>
#define	APMDEV		"/dev/apm"

void
read_netbsd_apm_data()
{
   int f, r;
   struct apm_power_info info;

   if (!GK.second_tick)	/* No need to reap apm more often */
      return;

   if ((f = open(APMDEV, O_RDONLY)) == -1) return;
   r = ioctl(f, APM_IOC_GETPOWER, &info);
   close(f);
   if (r == -1) return;

   sys_apm.battery_is_available = (info.battery_state != APM_BATT_UNKNOWN);
   sys_apm.ac_is_on_line = (info.ac_state == APM_AC_ON) ? TRUE : FALSE;
   sys_apm.battery_is_charging = 
      (info.battery_state == APM_BATT_CHARGING) ? TRUE : FALSE;
   sys_apm.battery_percentage = info.battery_life;
   sys_apm.battery_time_left = info.minutes_left;
}

#endif /* __NetBSD__ || __OpenBSD__ */


/* ------- Demo ------------------------------------------------------- */
  /* Themers need to be able to see the apm monitor.
  */
static void
read_apm_demo()
	{
	static gint	bump;

	sys_apm.battery_is_available = TRUE;

	if (GK.two_second_tick)
		bump = ++bump % 20;

	sys_apm.ac_is_on_line = bump & 2;
	sys_apm.battery_is_charging = FALSE;
	sys_apm.battery_percentage = sys_apm.ac_is_on_line ? 90 : 10;
	sys_apm.battery_time_left = sys_apm.ac_is_on_line ? 225 : 1;
	}


/* ----- Pick a system interface ----------------------------------------- */

static gint
setup_apm_interface()
	{
	gint	available	= FALSE;

	if (GK.demo)
		{
		read_apm_data = read_apm_demo;
		return TRUE;
		}

#if defined(__FreeBSD__)
	read_apm_data = read_freebsd_apm_data;
	available = TRUE;
#endif

#if defined(__linux__)
	read_apm_data = read_linux_apm_data;
	available = TRUE;
#endif

#if (defined(__NetBSD__) || defined(__OpenBSD__)) && defined(__i386__)
	read_apm_data = read_netbsd_apm_data;
	available = TRUE;
#endif

	return available;
	}



/* ======================================================================== */

static Panel	*apm;

static Krell	*apm_krell;
static Decal	*power_source_decal;
static Decal	*minutes_left_decal;

static Launcher	launch;

static gint		apm_visible;

static gint		style_id;

static void
draw_time_left_decal(gint force)
	{
	Decal		*d;
	gchar		buf[16];
	gint		t, x, w;
	static gint	charging = -1;

	d = minutes_left_decal;
	if (charging != sys_apm.battery_is_charging)
		force = TRUE;
	charging = sys_apm.battery_is_charging;

	t = sys_apm.battery_time_left;
	if (charging)
		buf[0] = '\0';
	else if (sys_apm.battery_display_percent)
		{
		t = sys_apm.battery_percentage;
		snprintf(buf, sizeof(buf), "%d %s", t, "%");
		}
	else
		snprintf(buf, sizeof(buf), "%d %s", t, "min");

	w = gdk_string_measure(d->text_style.font, buf);
	x = (d->w - w) / 2;
	if (x < 0)
		x = 0;
	d->x_off = x;
	gkrellm_draw_decal_text(apm, d, buf, force ? -1 : t);
	}


static void
update_apm()
	{
	gint			t, decal;
	static gboolean	warning;

	if (enable_apm == FALSE)
		return;
	if (GK.two_second_tick || warning)
		{
		(*read_apm_data)();

		warning = FALSE;
		t = sys_apm.battery_display_percent ? sys_apm.battery_percentage
							: sys_apm.battery_time_left;
		if (sys_apm.ac_is_on_line)
			decal = D_MISC_AC;
		else
			{
			decal = D_MISC_BATTERY;
			if (t < 20)
				{
				warning = TRUE;
				if (t <= 1 || !(GK.timer_ticks % t))
					decal = D_MISC_BATTERY_WARN;
				}
			}
		gkrellm_draw_decal_pixmap(apm, power_source_decal, decal);
		draw_time_left_decal(FALSE);

		apm_krell->full_scale = 100;
		apm_krell->previous = 0;
		gkrellm_update_krell(apm, apm_krell, sys_apm.battery_percentage);
		gkrellm_draw_layers(apm);
		}
	}


static gint
apm_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	if (widget == apm->drawing_area)
		{
		gdk_draw_pixmap(widget->window,
				widget->style->fg_gc[GTK_WIDGET_STATE(widget)], apm->pixmap,
				ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				ev->area.width, ev->area.height);
		}
	return FALSE;
	}

static gint
apm_display_change (GtkWidget *widget, GdkEventButton *event)
	{
	if (event->button == 2 || event->button == 3)
		{
		sys_apm.battery_display_percent = 1 - sys_apm.battery_display_percent;
		draw_time_left_decal(TRUE);
		gkrellm_draw_layers(apm);
		}
	return TRUE;
	}

static void
create_apm(GtkWidget *vbox, gint first_create)
	{
	TextStyle	*ts;
	Style		*style;
	gint		x, w;

	if (first_create)
		{
		apm = gkrellm_panel_new0();
		}
	else
		{
		gkrellm_destroy_decal_list(apm);
		gkrellm_destroy_krell_list(apm);
		}

	style = gkrellm_meter_style(style_id);


	power_source_decal = gkrellm_create_decal_pixmap(apm, GK.decal_misc_pixmap,
			GK.decal_misc_mask, N_MISC_DECALS, style, style->margin + 2, -1);

	/* The default text decal has width of chart_width - margins and
	|  x position at left margin, so calculate the override values.
	|  Cannot override w after the create.
	*/
	ts = gkrellm_meter_textstyle(style_id);
	x = style->margin + power_source_decal->w + 6;
	w = UC.chart_width - x - style->margin;
	minutes_left_decal = gkrellm_create_decal_text(apm, "100", ts,
						style, x, -1, w);	/* -1 means use y default */

	apm_krell = gkrellm_create_krell(apm, gkrellm_krell_meter_image(style_id),
								style);

	gkrellm_configure_panel(apm, NULL, style);
	gkrellm_create_panel(vbox, apm, gkrellm_bg_meter_image(style_id));

	/* Center the decals with respect to each other.
	*/
	if (power_source_decal->h > minutes_left_decal->h)
		minutes_left_decal->y +=
					(power_source_decal->h - minutes_left_decal->h) / 2;
	else
		power_source_decal->y +=
					(minutes_left_decal->h - power_source_decal->h) / 2;

	/* Some laptops apparently have probs reading /proc/apm. So avoid it
	|  unless explicitely enabled.
	*/
	if (GK.demo)
		enable_apm = TRUE;
	if (enable_apm)
		{
		(*(read_apm_data))();
		if (! sys_apm.battery_is_available)
			gtk_widget_hide(apm->hbox);
		else
			{
			apm_visible = TRUE;
			gkrellm_monitor_height_adjust(apm->h);
			}
		}
	else
		gtk_widget_hide(apm->hbox);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT (apm->drawing_area), "expose_event",
				(GtkSignalFunc) apm_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT (apm->drawing_area), "button_press_event",
				(GtkSignalFunc) apm_display_change, NULL);
         }
	gkrellm_setup_launcher(apm, &launch, METER_PANEL_TYPE, 0);
	}




#define	APM_CONFIG_KEYWORD	"apm"

static void
save_apm_config(FILE *f)
	{
	fprintf(f, "%s enable %d\n", APM_CONFIG_KEYWORD, enable_apm);
	fprintf(f, "%s launch1 %s\n", APM_CONFIG_KEYWORD, launch.command);
	fprintf(f, "%s tooltip_comment %s\n",
			APM_CONFIG_KEYWORD, launch.tooltip_comment);
	}

static void
load_apm_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];

	if (sscanf(arg, "%31s %[^\n]", config, item) == 2)
		{
		if (strcmp(config, "enable") == 0)
			sscanf(item, "%d", &enable_apm);
		else if (strcmp(config, "launch1") == 0)
			launch.command = g_strdup(item);
		else if (strcmp(config, "tooltip_comment") == 0)
			launch.tooltip_comment = g_strdup(item);
		}
	}

static GtkWidget	*enable_apm_button;
static GtkWidget	*launch_entry,
					*tooltip_entry;

static void 
apply_apm_config()
	{
	gint	new;

	new = GTK_TOGGLE_BUTTON(enable_apm_button)->active;
	if (enable_apm == FALSE && new == TRUE)
		{
		(*read_apm_data)();
		if (! sys_apm.battery_is_available)
			{
			gkrellm_config_message_window(_("APM Config Error"),
				_("No battery available."), NULL);
 			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_apm_button),
					FALSE);
			return;
			}
		}
	enable_apm = new;
	gkrellm_enable_visibility(enable_apm, &apm_visible, apm->hbox, apm->h);
	gkrellm_apply_launcher(&launch_entry, &tooltip_entry, apm,
				&launch, gkrellm_launch_button_cb);
	}

static void
create_apm_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*table;

	gkrellm_check_button(tab_vbox, &enable_apm_button, enable_apm, FALSE, 10,
			_("Enable APM"));
	insert_expanded_filler(tab_vbox);
	table = gkrellm_launcher_table_new(tab_vbox, 1);
	gkrellm_config_launcher(table, 0,  &launch_entry, &tooltip_entry, 
					_("APM"), &launch);
	}

static Monitor	monitor_apm =
	{
	N_("APM"),			/* Name, for config tab.	*/
	MON_APM,			/* Id,  0 if a plugin		*/
	create_apm,			/* The create function		*/
	update_apm,			/* The update function		*/
	create_apm_tab,		/* The config tab create function	*/
	apply_apm_config,	/* Apply the config function		*/

	save_apm_config,	/* Save user conifg			*/
	load_apm_config,	/* Load user config			*/
	APM_CONFIG_KEYWORD,	/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_apm_monitor(void)
	{
	enable_apm = FALSE;			/* Some people have probs reading /proc/apm */
	monitor_apm.name=_(monitor_apm.name);
	style_id = gkrellm_add_meter_style(&monitor_apm, APM_STYLE_NAME);
    if (setup_apm_interface())
		return &monitor_apm;
	return NULL;
	}

