/******************************************
 *
 * $GAMGI/src/chem/gamgi_chem_polytope.c
 *
 * Copyright (C) 2008 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

/*****************************************************
 *                   Bibliography                    *
 *                                                   *
 * Rourke J., Computational Geometry in C,           *
 * Cambridge University Press (1998)                 *
 *                                                   *
 * Berg M., Kreveld M., Overmars M., Schwarzkopf O., *
 * Computational Geometry, Springer Verlag (2000)    *
 *****************************************************/

#include "gamgi_engine.h"
#include "gamgi_mesa.h"
#include "gamgi_math.h"
#include "gamgi_global.h"

#include "gamgi_engine_create.h"
#include "gamgi_engine_start.h"
#include "gamgi_engine_link.h"
#include "gamgi_engine_dlist.h"
#include "gamgi_engine_list.h"
#include "gamgi_math_vector.h"

#define GAMGI_CHEM_POLYTOPE_WIDTH 0.1

#define GAMGI_CHEM_POLYTOPE_FACES_START  20
#define GAMGI_CHEM_POLYTOPE_FACES_STEP   10

#define GAMGI_CHEM_POLYTOPE_EDGES_START  20
#define GAMGI_CHEM_POLYTOPE_EDGES_STEP   10

#define GAMGI_CHEM_POLYTOPE_VERTICES_START  20
#define GAMGI_CHEM_POLYTOPE_VERTICES_STEP   10

#define GAMGI_CHEM_POLYTOPE_FACE_START   20
#define GAMGI_CHEM_POLYTOPE_FACE_STEP    10

#define GAMGI_CHEM_POLYTOPE_VERTEX_START 20
#define GAMGI_CHEM_POLYTOPE_VERTEX_STEP  10

#define GAMGI_CHEM_POLYTOPE_LOOPS_START  20
#define GAMGI_CHEM_POLYTOPE_LOOPS_STEP   10

#define GAMGI_CHEM_POLYTOPE_REMOVE 4
#define GAMGI_CHEM_POLYTOPE_MODIFY 3
#define GAMGI_CHEM_POLYTOPE_KEEP   2
#define GAMGI_CHEM_POLYTOPE_USE    1

#define GAMGI_CHEM_POLYTOPE_TOLERANCE_LENGTH 1.0E-3
#define GAMGI_CHEM_POLYTOPE_TOLERANCE_AREA   1.0E-4
#define GAMGI_CHEM_POLYTOPE_TOLERANCE_VOLUME 1.0E-5

#define GAMGI_CHEM_POLYTOPE_TOLERANCE_COPLANAR 1.0E-4
#define GAMGI_CHEM_POLYTOPE_TOLERANCE_VISIBLE  1.0E-4

typedef struct _gamgi_polytope {

gamgi_object *parent;
gamgi_dlist *atoms;
int n_atoms;

gamgi_slist **cells;
double width_x, width_y, width_z;
int n_cells;

gamgi_atom *atom;
gamgi_bool global;

int element;
int number;
double max;

gamgi_enum style;
float *color;

/***************************************************************
 *                          arrays format                      *
 *                                                             *
 * int **vertices: vertex 1, ..., vertex n                     *
 *                                                             *
 * int **edges: edges 1, ..., edges n                          *
 *                                                             *
 * int **faces: face 1, ..., face n                            *
 *                                                             *
 * int *vertex: total, state, number of edges, number          *
 * of conflicts, face 1, offset 1, ..., face n, offset n       *
 *                                                             *
 * int *edge: total, state, face 1, face 2, vertex 1, vertex 2 *
 *                                                             *
 * int *face: total, state, number of conflicts, vertex 1,     *
 * offset 1, ..., vertex n, offset n, number of edges, edge 1, * 
 * before 1 next 1, ..., edge n, before n, next n              *
 ***************************************************************/

int **vertices;
int n_vertices;

int **edges;
int n_edges;

int **faces;
int n_faces;

/*******************************************
 *         shell neighbour data            *
 * length = array with neighbour lengths   *
 * neighbours = array with neighbour atoms *
 * n_neighbours = array total positions    *
 * n_vertices = array used positions       *
 *******************************************/

gamgi_atom **neighbours;
int n_neighbours;
double *length;

} gamgi_polytope;

static int *static_export_points (gamgi_group *group, gamgi_polytope *polytope)
{
gamgi_atom **neighbours;
gamgi_atom *atom;
double *points;
int **vertices;
int *vertex, *index;
int n_vertices;
int n, i;

vertices = polytope->vertices;
n_vertices = polytope->n_vertices;
neighbours = polytope->neighbours;

/******************************************
 * build the index array to redirect the  *
 * used vertices to their final positions *
 ******************************************/

n = 0;
index = (int *) malloc (n_vertices * sizeof (int));

for (i = 0; i < n_vertices; i++)
  {
  vertex = vertices[i];
  if (vertex[2] > 0) index[i] = n++;
  }

/******************************************************
 * build the points array that will contain the       *
 * positions of all the vertices of the polyhedron    *
 * contained in the group object linked to the parent *
 ******************************************************/

group->n_points = n;
points = (double *) malloc (3 * n * sizeof (double));
group->points = points;

for (i = 0; i < n_vertices; i++)
  {
  vertex = vertices[i];
  if (vertex[2] > 0)
    {
    /*****************
     * save vertices *
     *****************/

    n = index[i];
    atom = neighbours[i];
    points[3 * n + 0] = atom->position[0];
    points[3 * n + 1] = atom->position[1];
    points[3 * n + 2] = atom->position[2];

    /********************
     * calculate center *
     ********************/

    gamgi_math_vector_add (group->center, atom->position, group->center);
    }
  }

/********************
 * calculate center *
 ********************/

if (group->n_points > 0)
  gamgi_math_vector_scale (group->center, 
  group->center, 1.0 / group->n_points);

return index;
}

static void static_export_loops (int *index, 
gamgi_group *group, gamgi_polytope *polytope)
{
int *loops;
int **faces, **edges;
int *face, *edge;
int v, e, e_start;
int offset, next;
int n_faces, n_conflicts, n_loops;
int total, i, j;

edges = polytope->edges;
faces = polytope->faces;
n_faces = polytope->n_faces;

n_loops = GAMGI_CHEM_POLYTOPE_LOOPS_START;
loops = (int *) malloc (n_loops * sizeof (int));
loops[0] = 0;

total = 1;
offset = 1;
for (i = 0; i < n_faces; i++)
  {
  /********************
   * find valid faces *
   ********************/

  face = faces[i];
  if (face[1] == GAMGI_CHEM_POLYTOPE_USE) continue;
  loops[0]++;

  /*************************
   * find first valid edge *
   *************************/

  n_conflicts = face[2];
  for (j = 4 + 2 * n_conflicts; j < face[0] - 2; j += 3)
    if (face[j] >= 0) break;

  /**********************************
   * if needed increase loops array *
   **********************************/

  if (++total > n_loops)
    {
    n_loops += GAMGI_CHEM_POLYTOPE_LOOPS_STEP;
    loops = (int *) realloc (loops, n_loops * sizeof (int));
    }

  loops[offset] = 0;
  e_start = face[j];
  next = face[j + 2];
  do
    {
    e = face[next];
    next = face[next + 2];

    edge = edges[e];
    if (edge[2] == i) v = edge[4];
    else v = edge[5];

    /****************************************************
     * redirect valid vertices to final array positions *
     ****************************************************/

    v = index[v];

    /**********************************
     * if needed increase loops array *
     **********************************/

    if (++total > n_loops)
      {
      n_loops += GAMGI_CHEM_POLYTOPE_LOOPS_STEP;
      loops = (int *) realloc (loops, n_loops * sizeof (int));
      }

    loops[offset]++;
    loops[offset + loops[offset]] = v;
    } while (e != e_start);

  offset += loops[offset] + 1;
  }

/**********************************
 * set final size for loops array *
 **********************************/

group->n_loops = total;
group->loops = (int *) realloc (loops, total * sizeof (int));
}

static void static_export (gamgi_polytope *polytope)
{
gamgi_group *group;
int *index;

/*********************************************
 * create container group and link to parent *
 *********************************************/

group = gamgi_engine_create_group ();
gamgi_engine_start_group (group);
strcpy (group->object.name, "Polytope");
gamgi_engine_link_object_object (GAMGI_CAST_OBJECT group, polytope->parent);

/*******************************
 * export points, loops arrays *
 *******************************/

index = static_export_points (group, polytope);
static_export_loops (index, group, polytope);

/**************
 * set visual *
 **************/

if (polytope->style == GAMGI_MESA_SOLID) group->faces = TRUE;

group->red = polytope->color[0];
group->green = polytope->color[1];
group->blue = polytope->color[2];

free (index);
}

static int static_before (int offset, int f, gamgi_polytope *polytope)
{
int *face, *edge;
int e;

face = polytope->faces[f];
offset = face[offset + 1];

e = face[offset];
edge = polytope->edges[e];

/***********************************************************
 * if the previous edge has face_hidden as first face      *
 * then get its first vertex, otherwise get the second     *
 * (this is the vertex farther away from the visible edge) *
 ***********************************************************/

if (edge[2] == f) return edge[4];

return edge[5];
}

static int static_dot (int v1, int v2,
double tolerance, gamgi_polytope *polytope)
{
gamgi_atom **neighbours;
gamgi_atom *a1, *a2;
double u[3];
double dot;

neighbours = polytope->neighbours;

a1 = neighbours[v1];
a2 = neighbours[v2];

gamgi_math_vector_sub (a2->position, a1->position, u);
dot = gamgi_math_vector_dot (u, u);

if (dot > tolerance) return 1;

return 0;
}

static int static_cross (int v1, int v2, int v3,
double tolerance, gamgi_polytope *polytope)
{
gamgi_atom **neighbours;
gamgi_atom *a1, *a2, *a3;
double u[3], v[3], w[3];
double dot;

neighbours = polytope->neighbours;

a1 = neighbours[v1];
a2 = neighbours[v2];
a3 = neighbours[v3];

gamgi_math_vector_sub (a2->position, a1->position, u);
gamgi_math_vector_sub (a3->position, a1->position, v);
gamgi_math_vector_cross (u, v, w);
dot = gamgi_math_vector_dot (w, w);

if (dot > tolerance) return 1;

return 0;
}

static int static_mix (int v1, int v2, int v3, int v4, 
double tolerance, gamgi_polytope *polytope)
{
gamgi_atom **neighbours;
gamgi_atom *a1, *a2, *a3, *a4;
double u[3], v[3], w[3];
double mix;

neighbours = polytope->neighbours;

a1 = neighbours[v1];
a2 = neighbours[v2];
a3 = neighbours[v3];
a4 = neighbours[v4];

gamgi_math_vector_sub (a2->position, a1->position, u);
gamgi_math_vector_sub (a3->position, a1->position, v);
gamgi_math_vector_sub (a4->position, a1->position, w);
mix = gamgi_math_vector_mix (u, v, w);

if (mix > tolerance) return 1;

if (mix < -tolerance) return -1;

return 0;
}

static void static_vertex_start (int v, gamgi_polytope *polytope)
{
int *vertex;
int i;

vertex = polytope->vertices[v];

vertex[1] = -1;
vertex[2] = 0;
vertex[3] = 0;
for (i = 4; i < vertex[0]; i++) vertex[i] = -1;
}

static int *static_vertex_create (int v, gamgi_polytope *polytope)
{
int *vertex;
int total;

/*****************************************
 * create vertex array, with 0 conflicts *
 *****************************************/

total = GAMGI_CHEM_POLYTOPE_VERTEX_START;
vertex = (int *) malloc (total * sizeof (int));
vertex[0] = total;

polytope->vertices[v] = vertex;
static_vertex_start (v, polytope);

return vertex;
}

static int static_vertex_conflict_recreate (int v, gamgi_polytope *polytope)
{
int **vertices;
int *vertex;
int i, j;

vertices = polytope->vertices;

/******************************************
 * look for a old face conflict position  *
 *                                        *
 * the first position for conflicts is 4  *
 ******************************************/

vertex = vertices[v];

for (i = 4; i < vertex[0] - 1; i+= 2)
  if (vertex[i] < 0) return i;

/**********************************************************
 * if that is not available increase and initialize array *
 **********************************************************/

vertex[0] += GAMGI_CHEM_POLYTOPE_VERTEX_STEP;
vertex =  (int *) realloc (vertex, vertex[0] * sizeof (int));
vertices[v] = vertex;

for (j = i; j < vertex[0]; j++) vertex[i] = -1;

return i;
}

static void static_edge_start (int e, gamgi_polytope *polytope)
{
int *edge;

/*****************************************************
 * this face is no longer in use and can be recycled *
 *****************************************************/

edge = polytope->edges[e];
edge[1] = GAMGI_CHEM_POLYTOPE_USE;
}

static void static_edge_remove (int e, gamgi_polytope *polytope)
{
int *edge, *vertex;
int v;

edge = polytope->edges[e];

/*********************************************************
 * for both vertices decrease counting (number of edges) *
 *********************************************************/

v = edge[4];
vertex = polytope->vertices[v];
vertex[2]--;

v = edge[5];
vertex = polytope->vertices[v];
vertex[2]--;

/*****************************************************
 * this face is no longer in use and can be recycled *
 *****************************************************/

static_edge_start (e, polytope);
}

static void static_edges_remove (int v_new, gamgi_polytope *polytope)
{
int **faces, **edges;
int *vertex_new, *face_visible, *edge_visible;
int f_visible, e_visible;
int i, j, n_conflicts;

faces = polytope->faces;
edges = polytope->edges;

/*******************************************************************
 * use the vertex conflict data to remove all doubly visible edges *
 *******************************************************************/

vertex_new = polytope->vertices[v_new];

for (i = 4; i < vertex_new[0] - 1; i += 2)
  {
  if (vertex_new[i] < 0) continue;

  f_visible = vertex_new[i];
  face_visible = faces[f_visible];

  n_conflicts = face_visible[2];
  for (j = 4 + 2 * n_conflicts; j < face_visible[0] - 2; j += 3)
    {
    e_visible = face_visible[4 + 2*n_conflicts + j];
    if (e_visible < 0) continue;

    edge_visible = edges[e_visible];

    if (edge_visible[1] == GAMGI_CHEM_POLYTOPE_REMOVE)
      static_edge_remove (e_visible, polytope);
    else
      edge_visible[1] = GAMGI_CHEM_POLYTOPE_KEEP;
    }
  }

}

static int *static_edge_create (int e, gamgi_polytope *polytope)
{
int *edge;
int total;

/**************************************************
 * create edge array, with 2 faces and 2 vertices *
 **************************************************/

total = 6;
edge = (int *) malloc (total * sizeof (int));
edge[0] = total;

polytope->edges[e] = edge;
static_edge_start (e, polytope);

return edge;
}

static int static_edges_create (gamgi_polytope *polytope)
{
int **edges;
int *edge;
int n_edges, i, j;

edges = polytope->edges;
n_edges = polytope->n_edges;

for (i = 0; i < n_edges; i++)
  {
  /*************************
   * reuse old edges array *
   *************************/

  edge = edges[i];
  if (edge[1] == GAMGI_CHEM_POLYTOPE_USE)
    {
    edge[1] = GAMGI_CHEM_POLYTOPE_KEEP;
    return i;
    }
  }

/************************
 * increase edges array *
 ************************/

n_edges += GAMGI_CHEM_POLYTOPE_EDGES_STEP;
polytope->n_edges = n_edges;

edges = (int **) realloc (edges, n_edges * sizeof (int *));
polytope->edges = edges;

for (j = i; j < n_edges; j++)
  {
  edge = static_edge_create (j, polytope);
  edges[j] = edge;
  }

edge = edges[i];
edge[1] = GAMGI_CHEM_POLYTOPE_KEEP;

return i;
}

static void static_edges_mark (int v_new, gamgi_polytope *polytope)
{
int **faces, **edges, **vertices;
int *face_start, *face_end, *face_visible;
int *vertex_new, *edge_visible;
int f_start, f_end, f_visible, e_visible;
int i, j;

faces = polytope->faces;
edges = polytope->edges;
vertices = polytope->vertices;

/*****************************************************************
 * use the vertex conflict data to scan all the visible faces,   *
 * to get all the edges that are susceptible of changes, without *
 * scanning the whole set of edges                               *
 *                                                               *
 * the edges in visible faces are either in the boundary of the  *
 * visible region (only one face is visible), forming new faces  *
 * with the new vertex, or are inside the visible region (both   *
 * faces are visible), and must be removed.                      *
 *****************************************************************/

vertex_new = vertices[v_new];

for (i = 4; i < vertex_new[0] - 1; i += 2)
  {
  if (vertex_new[i] < 0) continue;

  f_visible = vertex_new[i];
  face_visible = faces[f_visible];

  for (j = 4 + 2 * face_visible[2]; j < face_visible[0] - 2; j += 3)
    {
    if (face_visible[j] < 0) continue;

    e_visible = face_visible[j];
    edge_visible = edges[e_visible];
    edge_visible[1] = GAMGI_CHEM_POLYTOPE_MODIFY;

    f_start = edge_visible[2];
    face_start = faces[f_start];
    f_end = edge_visible[3];
    face_end = faces[f_end];

    if (face_start[1] == GAMGI_CHEM_POLYTOPE_REMOVE &&
    face_end[1] == GAMGI_CHEM_POLYTOPE_REMOVE)
      edge_visible[1] = GAMGI_CHEM_POLYTOPE_REMOVE;
    }
  }

}

static void static_face_start (int f, gamgi_polytope *polytope)
{
int *face;
int i;

face = polytope->faces[f];

/*****************************************************
 * this face is no longer in use and can be recycled *
 *****************************************************/

face[1] = GAMGI_CHEM_POLYTOPE_USE;
face[2] = 0;
for (i = 3; i < face[0]; i++) face[i] = -1;
}

static void static_face_remove (int f, gamgi_polytope *polytope)
{
int **vertices;
int *vertex, *face;
int v, i, offset;

vertices = polytope->vertices;

face = polytope->faces[f];

/******************************************
 * remove all conflicts for this face     *
 * from vertices before removing the face *
 ******************************************/

for (i = 3; i < 3 + 2 * face[2] - 1; i += 2)
  {
  v = face[i + 0];
  offset = face[i + 1];
  vertex = vertices[v];
  vertex[offset] = -1;
  }

/*****************************************************
 * this face is no longer in use and can be recycled *
 *****************************************************/

static_face_start (f, polytope);
}

static void static_faces_remove (int v_new, gamgi_polytope *polytope)
{
int *vertex_new;
int f_visible, i;

/************************************************************
 * use the vertex conflict data to remove all visible faces *
 ************************************************************/

vertex_new = polytope->vertices[v_new];

for (i = 4; i < vertex_new[0] - 1; i += 2)
  {
  if (vertex_new[i] < 0) continue;

  f_visible = vertex_new[i];
  static_face_remove (f_visible, polytope);
  }
}

static void static_face_clean (int f, gamgi_polytope *polytope)
{
int **edges;
int *face, *edge, *edge_next;
int before, next, next_next;
int e, e_next, i;

edges = polytope->edges;

/*****************************
 * use old edge if it exists *
 *****************************/

face = polytope->faces[f];

for (i = 3 + 2 * face[2] - 2; i < face[0] - 2; i += 3)
  {
  e = face[i];
  if (e < 0) continue;

  edge = edges[e];
  if (edge[1] == GAMGI_CHEM_POLYTOPE_KEEP && edge[2] == edge[3])
    {
    /*******************************************************
     * each edge has three positions, the first is the     *
     * edge number, the second is the offset of the edge   *
     * before and the third is the offset of the edge next *
     *                                                     *
     * the edges are orientated counter-clockwise in a     *
     * circular way: positions i+1 and i+2 indicate        *
     * the offset for the edge before and next             *
     *                                                     *
     * mark e and e_next to be removed later. This in      *
     * turn will remove the vertex between e and e_next    *
     *******************************************************/

    before = face[i + 1];
    next = face[i + 2];

    next_next = face[next + 2];
    face[before + 2] = next_next;
    face[next_next + 1] = before;

    edge[1] = GAMGI_CHEM_POLYTOPE_REMOVE;
    e_next = face[next];
    edge_next = edges[e_next];
    edge_next[1] = GAMGI_CHEM_POLYTOPE_REMOVE;
    }
  }

face[1] = GAMGI_CHEM_POLYTOPE_KEEP;
}

static int static_face_edge_recreate (int f, gamgi_polytope *polytope)
{
int **faces, **edges;
int *face, *edge;
int e, i, j;

faces = polytope->faces;
edges = polytope->edges;

face = faces[f];

/*****************************
 * use old edge if it exists *
 *****************************/

for (i = 4 + 2 * face[2]; i < face[0] - 2; i += 3)
  {
  e = face[i];
  if (e < 0) return i;

  edge = edges[e];
  if (edge[1] == GAMGI_CHEM_POLYTOPE_USE) return i;
  }

/**********************************************************
 * if that is not available increase and initialize array *
 **********************************************************/

face[0] += GAMGI_CHEM_POLYTOPE_FACE_STEP;
face =  (int *) realloc (face, face[0] * sizeof (int));
faces[f] = face;

for (j = i; j < face[0]; j++) face[j] = -1;

return i;
}

static int static_face_conflict_recreate (int f, gamgi_polytope *polytope)
{
int **faces;
int *face;
int i, j;

faces = polytope->faces;
face = faces[f];

/******************************************
 * look for a old face conflict position  *
 *                                        *
 * the first position for conflicts is 3  *
 *                                        *
 * the last 10 positions are reserved for *
 * n_edges and 3 edges, for this new face *
 ******************************************/

for (i = 3; i < face[0] - 11; i+= 2)
  if (face[i] < 0) return i;

/*******************************************
 * if that is not available increase array *
 *******************************************/

face[0] += GAMGI_CHEM_POLYTOPE_FACE_STEP;
face = (int *) realloc (face, face[0] * sizeof (int));
faces[f] = face;

for (j = i; j < face[0]; j++) face[j] = -1;

return i;
}

static void static_conflict_add (int v1, int v2, int v3, 
int v, int f, int *n_conflicts, gamgi_polytope *polytope)
{
int *face, *vertex;
int offset_f, offset_v;
int mix;

/***************************************************
 * v1, v2, v3 must be orientated counter-clockwise *
 * when seen from the outside of the polytope!     *
 ***************************************************/

mix = static_mix (v1, v2, v3, v, 
GAMGI_CHEM_POLYTOPE_TOLERANCE_VISIBLE, polytope);
if (mix > 0)
  {
  /*************************************************
   * vertex v is visible: add to list of conflicts *
   *************************************************/

  offset_v = static_vertex_conflict_recreate (v, polytope);
  offset_f = static_face_conflict_recreate (f, polytope);

  face = polytope->faces[f];
  face[offset_f + 0] = v;
  face[offset_f + 1] = offset_v;

  vertex = polytope->vertices[v];
  vertex[offset_v + 0] = f;
  vertex[offset_v + 1] = offset_f;

  *n_conflicts += 1;
  }
}

static int static_conflict_face (int f_old, int f, 
int v_start, int v_end, int v_new, gamgi_polytope *polytope)
{
int **faces, **vertices;
int *face, *face_old;
int n_conflicts;
int i, v;

faces = polytope->faces;
vertices = polytope->vertices;

face_old = faces[f_old];
face = faces[f];

n_conflicts = 0;
for (i = 3; i < 3 + 2 * face_old[2] - 1; i += 2)
  {
  v = face_old[i];

  /***************************************************************
   * vertices supplied to static_visible must be orientated      *
   * counter-clockwise when seen from the outside. As v_start,   *
   * v_end are orientated for the first face (face hidden), they *
   * must be reversed for the second face (face new). The third  *
   * vertex (the new vertex) of this triangular face comes next. *
   ***************************************************************/

  static_conflict_add (v_end, v_start, v_new, v, f, &n_conflicts, polytope);
  }

return n_conflicts;
}

static int static_conflict_recreate (int f, int e, int v, gamgi_polytope *polytope)
{
int *edge;
int f_start, f_end;
int v_start, v_end;
int n_conflicts;

edge = polytope->edges[e];

/************************************************
 * vertex conflicts to new face must come from  *
 * vertex conflicts with faces adjacent to edge *
 ************************************************/

n_conflicts = 0;
f_start = edge[2];
f_end = edge[3];

v_start = edge[4];
v_end = edge[5];

n_conflicts += static_conflict_face (f, f_start, v_start, v_end, v, polytope);
n_conflicts += static_conflict_face (f, f_end, v_start, v_end, v, polytope);

return n_conflicts;
}

static void static_conflict_create (gamgi_polytope *polytope)
{
int **faces, **vertices;
int *face;
int n_conflicts, n_faces, n_vertices;
int v1, v2, v3;
int f, v;

faces = polytope->faces;
n_faces = 4;

vertices = polytope->vertices;
n_vertices = polytope->n_vertices;

for (f = 0; f < n_faces; f++)
  {
  face = faces[f];

  /* add vertex function here */

  n_conflicts = 0;
  for (v = 4; v < n_vertices; v++)
    {
    /***************************************************************
     * vertices supplied to static_visible must be orientated      *
     * counter-clockwise when seen from the outside. As v_start,   *
     * v_end are orientated for the first face (face hidden), they *
     * must be reversed for the second face (face new). The third  *
     * vertex (the new vertex) of this triangular face comes next. *
     ***************************************************************/

    static_conflict_add (v1, v2, v3, v, f, &n_conflicts, polytope);
    }

  face[2] = n_conflicts;
  }

}

static int *static_face_create (int f, gamgi_polytope *polytope)
{
int *face;
int total;

/***************************************************
 * create face array, with 3 edges and 0 conflicts *
 ***************************************************/

total = GAMGI_CHEM_POLYTOPE_FACE_START;
face = (int *) malloc (total * sizeof (int));
face[0] = total;

polytope->faces[f] = face;
static_face_start (f, polytope);

return face;
}

static int static_faces_create (gamgi_polytope *polytope)
{
int **faces;
int *face;
int n_faces;
int i, j;

faces = polytope->faces;
n_faces = polytope->n_faces;

for (i = 0; i < n_faces; i++)
  {
  /*************************
   * reuse old faces array *
   *************************/

  face = faces[i];
  if (face[1] == GAMGI_CHEM_POLYTOPE_USE)
    {
    face[1] = GAMGI_CHEM_POLYTOPE_KEEP;
    return i;
    }
  }

/************************
 * increase faces array *
 ************************/

n_faces += GAMGI_CHEM_POLYTOPE_FACES_STEP;
polytope->n_faces = n_faces;

faces = (int **) realloc (faces, n_faces * sizeof (int *));
polytope->faces = faces;

for (j = i; j < n_faces; j++)
  {
  face = static_face_create (j, polytope);
  faces[j] = face;
  }

face = faces[i];
face[1] = GAMGI_CHEM_POLYTOPE_KEEP;

return i;
}

static void static_faces_mark (int v_new, gamgi_polytope *polytope)
{
int **faces, **vertices;
int *vertex_new, *face_visible;
int f_visible;
int i;

faces = polytope->faces;
vertices = polytope->vertices;

/*******************************************
 * signal all faces that should be removed *
 *******************************************/

vertex_new = vertices[v_new];

for (i = 4; i < vertex_new[0] - 1; i += 2)
  {
  f_visible = vertex_new[i];
  if (f_visible < 0) continue;

  face_visible = faces[f_visible];
  face_visible[2] = GAMGI_CHEM_POLYTOPE_REMOVE;
  }

}

static void static_face_build (int v_new, int v_start, int v_end,
int e_visible, gamgi_polytope *polytope)
{
int **faces, **edges, **vertices;
int *face_new, *vertex_new;
int *vertex_start, *vertex_end;
int *edge_start, *edge_end;
int f_new, e_start, e_end;
int n_conflicts, offset;

faces = polytope->faces;
edges = polytope->edges;
vertices = polytope->vertices;

vertex_new = vertices[v_new];
vertex_start = vertices[v_start];
vertex_end = vertices[v_end];

/******************************************************
 * Create new face, with 3 vertices and 3 edges,      *
 * counter-clockwise: for each edge, the vertices are *
 * orientated counter-clockwise for the first face,   *
 * for the second face the vertices must be reversed  *
 ******************************************************/
 
f_new = static_faces_create (polytope);
n_conflicts = static_conflict_recreate (f_new, e_visible, v_new, polytope);

face_new = faces[f_new];
face_new[2] = n_conflicts;
offset = 3 + 2 * n_conflicts;

face_new[offset + 0] = 3;
face_new[offset + 1] = e_visible;
face_new[offset + 2] = offset + 7;
face_new[offset + 3] = offset + 4;

if (vertex_start[1] < 0)
  {
  /******************************************************
   * create new edge, pointing to edges before and next *
   ******************************************************/

  e_start = static_edges_create (polytope);
  edge_start = edges[e_start];
  face_new[offset + 4] = e_start;
  face_new[offset + 5] = offset + 1;
  face_new[offset + 6] = offset + 7;
  edge_start[2] = f_new;
  edge_start[3] = -1;
  edge_start[4] = v_start;
  edge_start[5] = v_new;

  /**************************************************
   * increase vertex counting (its number of edges) *
   **************************************************/

  vertex_start[2]++;
  vertex_new[2]++;

  /****************************
   * point vertex to new edge *
   ****************************/

  vertex_start[1] = e_start;
  }
else
  {
  /***************************
   * the edge already exists *
   ***************************/

  e_start = vertex_start[1];
  edge_start = edges[e_start];
  face_new[offset + 4] = e_start;
  face_new[offset + 5] = offset + 1;
  face_new[offset + 6] = offset + 7;
  edge_start[3] = f_new;

  /******************************
   * reset vertex edge pointing *
   ******************************/

  vertex_start[1] = -1;
  }

if (vertex_end[1] < 0)
  {
  /******************************************************
   * create new edge, pointing to edges before and next *
   ******************************************************/

  e_end = static_edges_create (polytope);
  edge_end = edges[e_end];
  face_new[offset + 7] = e_end;
  face_new[offset + 8] = offset + 4;
  face_new[offset + 9] = offset + 1;
  edge_end[2] = f_new;
  edge_end[3] = -1;
  edge_end[4] = v_new;
  edge_end[5] = v_end;

  /**************************************************
   * increase vertex counting (its number of edges) *
   **************************************************/

  vertex_new[2]++;
  vertex_end[2]++;

  /****************************
   * point vertex to new edge *
   ****************************/

  vertex_end[1] = e_end;
  }
else
  {
  /***************************
   * the edge already exists *
   ***************************/

  e_end = vertex_end[1];
  edge_end = edges[e_end];
  face_new[offset + 7] = e_end;
  face_new[offset + 8] = offset + 4;
  face_new[offset + 9] = offset + 1;
  edge_end[4] = f_new;

  /******************************
   * reset vertex edge pointing *
   ******************************/

  vertex_end[1] = -1;
  }

}

static void static_face_merge (int v_new, int v_start, int v_end,
int e_visible, int before, int next, gamgi_polytope *polytope)
{
int **faces, **edges, **vertices;
int *face_hidden, *vertex_new;
int *vertex_start, *vertex_end;
int *edge_start, *edge_end, *edge_visible;
int f_hidden;
int e_start, e_end;
int start, end;

faces = polytope->faces;
edges = polytope->edges;
vertices = polytope->vertices;

vertex_new = vertices[v_new];
vertex_start = vertices[v_start];
vertex_end = vertices[v_end];

edge_visible = edges[e_visible];
f_hidden = edge_visible[2];
face_hidden = faces[f_hidden];

/***********************************************************
 * This is the same face as face_hidden: 1) add vertex_new *
 * to face_hidden, in counter-clockwise direction.         *
 *                                                         *
 * 2) mark face_hidden to be checked again in the end,     *
 * looking for common edges. This happens when v_new is    *
 * coplanar with more than one face, in which case         *
 * redundant edges and vertices must be removed.           *
 ***********************************************************/

face_hidden[2] = GAMGI_CHEM_POLYTOPE_MODIFY;

if (vertex_start[1] < 0)
  {
  /*******************
   * create new edge *
   *******************/

  e_start = static_edges_create (polytope);
  edge_start = edges[e_start];

  start = static_face_edge_recreate (f_hidden, polytope);
  face_hidden[start] = e_start;

  edge_start[2] = f_hidden;
  edge_start[3] = -1;
  edge_start[4] = v_start;
  edge_start[5] = v_new;

  /**************************************************
   * increase vertex counting (its number of edges) *
   **************************************************/

  vertex_start[2]++;
  vertex_new[2]++;

  /****************************
   * point vertex to new edge *
   ****************************/

  vertex_start[1] = e_start;
  }
else
  {
  /***************************
   * the edge already exists *
   ***************************/

  e_start = vertex_start[1];
  edge_start = edges[e_start];
  edge_start[3] = f_hidden;

  start = static_face_edge_recreate (f_hidden, polytope);
  face_hidden[start] = e_start;

  /******************************
   * reset vertex edge pointing *
   ******************************/

  vertex_start[1] = -1;
  }

if (vertex_end[1] < 0)
  {
  /*******************
   * create new edge *
   *******************/

  e_end = static_edges_create (polytope);
  edge_end = edges[e_end];

  end = static_face_edge_recreate (f_hidden, polytope);
  face_hidden[end] = e_end;

  edge_end[2] =  f_hidden;
  edge_end[3] =  -1;
  edge_end[4] = v_new;
  edge_end[5] = v_end;

  /**************************************************
   * increase vertex counting (its number of edges) *
   **************************************************/

  vertex_new[2]++;
  vertex_end[2]++;

  /****************************
   * point vertex to new edge *
   ****************************/

  vertex_end[1] = e_end;
  }
else
  {
  /***************************
   * the edge already exists *
   ***************************/

  e_end = vertex_end[1];
  edge_end = edges[e_end];
  edge_end[3] = f_hidden;

  end = static_face_edge_recreate (f_hidden, polytope);
  face_hidden[end] = e_end;

  /******************************
   * reset vertex edge pointing *
   ******************************/

  vertex_end[1] = -1;
  }

/*************************************************
 * update before and next pointers to all edges, *
 * ordered as before -> start -> end -> next,    *
 * to keep all edges orientated counter-clocwise *
 *************************************************/

face_hidden[start + 1] = before;
face_hidden[start + 2] = e_end;
face_hidden[end + 1] = e_start;
face_hidden[end + 2] = next;

face_hidden[before + 2] = start;
face_hidden[next + 1] = end;
 
edge_visible[1] = GAMGI_CHEM_POLYTOPE_REMOVE;
}

static void static_build_vertex (int v_new, gamgi_polytope *polytope)
{
int **faces, **edges, **vertices;
int *vertex_new, *edge_visible, *face_visible, *face_edge;
int v_before, v_start, v_end, e_visible, f_visible, f_edge;
int offset, before, next;
int i, j, aux;
int mix;

faces = polytope->faces;
edges = polytope->edges;
vertices = polytope->vertices;

static_faces_mark (v_new, polytope);
static_edges_mark (v_new, polytope);

/*****************************************************************
 * use again the vertex conflict data to scan all the            *
 * visible faces, to get all the edges that are susceptible      *
 * of changes, without scanning the whole set of edges           *
 *                                                               *
 * build a new face for each edge in the boundary of the visible *
 * region (only one face is visible), linking its two vertices   *
 * with the new vertex,                                          *
 *****************************************************************/

for (i = 4; i < vertex_new[0] - 1; i += 2)
  {
  if (vertex_new[i] < 0) continue;

  f_visible = vertex_new[i];
  face_visible = faces[f_visible];

  for (j = 4 + 2 * face_visible[2]; j < face_visible[0] - 2; j += 3)
    {
    if (face_visible[j] < 0) continue;

    e_visible = face_visible[j];
    edge_visible = edges[e_visible];

    before = face_visible[j + 1];
    next = face_visible[j + 2];

    /*************************************************
     * possible cases: 1) both faces are visible,    *
     * and edge will be removed only in the end;     *
     * 2) new coplanar edges have been added to face *
     * and should be ignored; 3) only one face is    *
     * visible and the second face must be created   *
     *************************************************/

    if (edge_visible[1] == GAMGI_CHEM_POLYTOPE_REMOVE) continue;

    if (edge_visible[1] == GAMGI_CHEM_POLYTOPE_KEEP) continue;

    /****************************************************
     * edge format: face1 face2 vertex1 vertex2 where   *
     * vertex1 vertex2 are orientated counter-clockwise *
     * for face1 (and clockwise for face2)              *
     ****************************************************/

    f_edge = edge_visible[2];
    face_edge = faces[f_edge];
    if (face_edge[1] == GAMGI_CHEM_POLYTOPE_REMOVE)
      {
      /************************************************************ 
       * The face that survives, face_hidden, should be first     *
       * and vertices should be ordered according to face_hidden. *
       * When face_hidden is not first, swap faces and vertices.  *
       ************************************************************/

      aux = edge_visible[2];
      edge_visible[2] = edge_visible[3];
      edge_visible[3] = aux;

      aux = edge_visible[4];
      edge_visible[4] = edge_visible[5];
      edge_visible[5] = aux;
      }

    /************************************************
     * get vertices in visible edge and edge before *
     ************************************************/

    v_start = edge_visible[4];
    v_end = edge_visible[5];
    v_before = static_before (offset, edge_visible[2], polytope);

    mix = static_mix (v_before, v_start, v_end, v_new,
    GAMGI_CHEM_POLYTOPE_TOLERANCE_COPLANAR, polytope);
    if (mix == 0)
      static_face_merge (v_new, v_start, v_end, e_visible, before, next, polytope);
    else
      static_face_build (v_new, v_start, v_end, e_visible, polytope);
    }
  }

/***********************************************************
 * use the vertex conflict data to scan all visible faces, *
 * looking for edges, vertices to remove in the same face  *
 ***********************************************************/

for (i = 4; i < vertex_new[0] - 1; i += 2)
  {
  f_visible = vertex_new[i];
  face_visible = faces[f_visible];
  if (face_visible[2] == GAMGI_CHEM_POLYTOPE_MODIFY)
    static_face_clean (f_visible, polytope);
  }

static_edges_remove (v_new, polytope);
static_faces_remove (v_new, polytope);
}

static gamgi_bool static_inside_local (int x, int y, int z, gamgi_polytope *polytope)
{
double min_x, min_y, min_z, min;

/***********************************************************
 * x,y,z = number of cells from central to neighbour cell: *
 * n_x - cell_x = x,  n_y - cell_y = y,  n_z - cell_z = z  *
 ***********************************************************/

if (x < 0) x = -x;
if (y < 0) y = -y;
if (z < 0) z = -z;

x = x - 1; if (x < 0) x = 0;
y = y - 1; if (y < 0) y = 0;
z = z - 1; if (z < 0) z = 0;

min_x = x * polytope->width_x;
min_y = y * polytope->width_y;
min_z = z * polytope->width_z;

min = min_x * min_x + min_y * min_y + min_z * min_z;

/************************************************
 * radius contains the radius distance OR the   *
 * distance for the last neighbour found so far *
 ************************************************/

if (min > polytope->max) return FALSE;

return TRUE;
}

static gamgi_bool static_inside_global (int shell, gamgi_polytope *polytope)
{
double min_x, min_y, min_z;
double max = polytope->max;

min_x = (shell - 1) * polytope->width_x;
min_y = (shell - 1) * polytope->width_y;
min_z = (shell - 1) * polytope->width_z;

/************************************************
 * radius contains the radius distance OR the   *
 * distance for the last neighbour found so far *
 ************************************************/

if (min_x > max && min_y > max && min_z > max) return FALSE;

return TRUE;
}

void static_shell_find (int n_x, int n_y, int n_z, gamgi_polytope *polytope)
{
gamgi_atom **neighbours;
gamgi_atom *atom;
gamgi_slist *slist;
double *length;
double x, y, z, d;
int n_neighbours, n_vertices, n;
int index, i;

n = polytope->n_cells;

n_vertices = polytope->n_vertices;
n_neighbours = polytope->n_neighbours;
neighbours = polytope->neighbours;
length = polytope->length;

index = n_z * n * n + n_y * n + n_x;
for (slist = polytope->cells[index]; slist != NULL; slist = slist->next)
  {
  atom = GAMGI_CAST_ATOM slist->data;
  if (atom->element != polytope->element || atom == polytope->atom) continue;

  x = atom->position[0] - polytope->atom->position[0];
  y = atom->position[1] - polytope->atom->position[1];
  z = atom->position[2] - polytope->atom->position[2];
  d = x * x + y * y + z * z;

  /********************************************
   * add atom to last position in arrays:     *
   * shell_d = array with neighbour distances *
   * shell_a = array with neighbour atoms     *
   * shell_n = number of shell neighbours     *
   ********************************************/

  /******************************************************************
   * when search is determined by distance, max is fixed            *
   * when search is determined by number, max decreases with search *
   ******************************************************************/

  if (d > polytope->max) continue;

  /**************************************************************
   * increase number of neighbours, when determined by distance *
   **************************************************************/

  if (polytope->number == 0)
    {
    n_vertices++;
    if (n_vertices > n_neighbours)
      {
      n_neighbours += GAMGI_CHEM_POLYTOPE_VERTICES_STEP;
      length = (double *) realloc (length, n_neighbours * sizeof (double));
      neighbours = (gamgi_atom **) realloc (neighbours, n_neighbours * sizeof (gamgi_atom *));
      }
    }

  length[n_vertices - 1] = d;
  neighbours[n_vertices - 1] = atom;

  /***************************************************************
   * set atom in the right order in arrays according to distance *
   ***************************************************************/

  for (i = n_vertices - 1; i > 0; i--)
    {
    if (length[i] > length[i - 1]) break;

    d = length[i - 1];
    length[i - 1] = length[i];
    length[i] = d;

    atom = neighbours[i - 1];
    neighbours[i - 1] = neighbours[i];
    neighbours[i] = atom;
    }

  /*********************************************************
   * save last neighbour found so far in max, so radius    *
   * defines the maximum distance that is is worth to scan *
   *********************************************************/

  if (polytope->number > 0) polytope->max = length[n_vertices - 1];
  }

polytope->n_vertices = n_vertices;
polytope->n_neighbours = n_neighbours;
polytope->neighbours = neighbours;
polytope->length = length;
}

static void static_raw_x0 (int cell_x, int cell_y, int cell_z, 
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i, j;

n = polytope->n_cells;

/****************************************
 * find neighbours in cells in -x faces *
 ****************************************/

n_x = -shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_z = i + cell_z;
  if (n_z < 0 || n_z > n - 1) continue;

  for (j = -shell + 1; j <= shell - 1; j++)
    {
    n_y = j + cell_y;
    if (n_y < 0 || n_y > n - 1) continue;

    if (static_inside_local (-shell, j, i, polytope) == TRUE)
      static_shell_find (n_x, n_y, n_z, polytope);
    }
  }                                              
}

static void static_raw_x1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i, j;

n = polytope->n_cells;

/****************************************
 * find neighbours in cells in +x faces *
 ****************************************/

n_x = shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_z = i + cell_z;
  if (n_z < 0 || n_z > n - 1) continue;

  for (j = -shell + 1; j <= shell - 1; j++)
    {
    n_y = j + cell_y;
    if (n_y < 0 || n_y > n - 1) continue;

    if (static_inside_local (shell, j, i, polytope) == TRUE)
      static_shell_find (n_x, n_y, n_z, polytope);
    }
  }
}

static void static_raw_y0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i, j;

n = polytope->n_cells;

/****************************************
 * find neighbours in cells in -y faces *
 ****************************************/

n_y = -shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_z = i + cell_z;
  if (n_z < 0 || n_z > n - 1) continue;

  for (j = -shell + 1; j <= shell - 1; j++)
    {
    n_x = j + cell_x;
    if (n_x < 0 || n_x > n - 1) continue;

    if (static_inside_local (j, -shell, i, polytope) == TRUE)
      static_shell_find (n_x, n_y, n_z, polytope);
    }
  }
}

static void static_raw_y1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i, j;

n = polytope->n_cells;

/****************************************
 * find neighbours in cells in +y faces *
 ****************************************/

n_y = shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_z = i + cell_z;
  if (n_z < 0 || n_z > n - 1) continue;

  for (j = -shell + 1; j <= shell - 1; j++)
    {
    n_x = j + cell_x;
    if (n_x < 0 || n_x > n - 1) continue;

    if (static_inside_local (j, shell, i, polytope) == TRUE)
      static_shell_find (n_x, n_y, n_z, polytope);
    }
  }
}

static void static_raw_z0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i, j;

n = polytope->n_cells;

/****************************************
 * find neighbours in cells in -z faces *
 ****************************************/

n_z = -shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_y = i + cell_y;
  if (n_y < 0 || n_y > n - 1) continue;

  for (j = -shell + 1; j <= shell - 1; j++)
    {
    n_x = j + cell_x;
    if (n_x < 0 || n_x > n - 1) continue;

    if (static_inside_local (j, i, -shell, polytope) == TRUE)
      static_shell_find (n_x, n_y, n_z, polytope);
    }
  }
}

static void static_raw_z1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i, j;

n = polytope->n_cells;

/****************************************
 * find neighbours in cells in +z faces *
 ****************************************/

n_z = shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_y = i + cell_y;
  if (n_y < 0 || n_y > n - 1) continue;

  for (j = -shell + 1; j <= shell - 1; j++)
    {
    n_x = j + cell_x;
    if (n_x < 0 || n_x > n - 1) continue;

    if (static_inside_local (j, i, shell, polytope) == TRUE)
      static_shell_find (n_x, n_y, n_z, polytope);
    }
  }
}

static void static_raw_x0y0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along x0y0 edges *
 *********************************************/

n_x = -shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = -shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_z = i + cell_z;
  if (n_z < 0 || n_z > n - 1) continue;

  if (static_inside_local (-shell, -shell, i, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_x1y0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along x0y0 edges *
 *********************************************/

n_x = shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = -shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_z = i + cell_z;
  if (n_z < 0 || n_z > n - 1) continue;

  if (static_inside_local (shell, -shell, i, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_x0y1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along x0y0 edges *
 *********************************************/

n_x = -shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_z = i + cell_z;
  if (n_z < 0 || n_z > n - 1) continue;

  if (static_inside_local (-shell, shell, i, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_x1y1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int i;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along x0y0 edges *
 *********************************************/

n_x = shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

for (i = -shell + 1; i <= shell - 1; i++)
  {
  n_z = i + cell_z;
  if (n_z < 0 || n_z > n - 1) continue;

  if (static_inside_local (shell, shell, i, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_x0z0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int j;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along x0z0 edges *
 *********************************************/

n_x = -shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_z = -shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (j = -shell + 1; j <= shell - 1; j++)
  {
  n_y = j + cell_y;
  if (n_y < 0 || n_y > n - 1) continue;

  if (static_inside_local (-shell, j, -shell, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_x1z0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int j;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along x1z0 edges *
 *********************************************/

n_x = shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_z = -shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (j = -shell + 1; j <= shell - 1; j++)
  {
  n_y = j + cell_y;
  if (n_y < 0 || n_y > n - 1) continue;

  if (static_inside_local (shell, j, -shell, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_x0z1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int j;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along x0z1 edges *
 *********************************************/

n_x = -shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_z = shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (j = -shell + 1; j <= shell - 1; j++)
  {
  n_y = j + cell_y;
  if (n_y < 0 || n_y > n - 1) continue;

  if (static_inside_local (-shell, j, shell, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_x1z1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int j;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along x1z1 edges *
 *********************************************/

n_x = shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_z = shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (j = -shell + 1; j <= shell - 1; j++)
  {
  n_y = j + cell_y;
  if (n_y < 0 || n_y > n - 1) continue;

  if (static_inside_local (shell, j, shell, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_y0z0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int k;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along y0z0 edges *
 *********************************************/

n_y = -shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = -shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (k = -shell + 1; k <= shell - 1; k++)
  {
  n_x = k + cell_x;
  if (n_x < 0 || n_x > n - 1) continue;

  if (static_inside_local (k, -shell, -shell, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_y1z0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int k;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along y1z0 edges *
 *********************************************/

n_y = shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = -shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (k = -shell + 1; k <= shell - 1; k++)
  {
  n_x = k + cell_x;
  if (n_x < 0 || n_x > n - 1) continue;

  if (static_inside_local (k, shell, -shell, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_y0z1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int k;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along y0z1 edges *
 *********************************************/

n_y = -shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (k = -shell + 1; k <= shell - 1; k++)
  {
  n_x = k + cell_x;
  if (n_x < 0 || n_x > n - 1) continue;

  if (static_inside_local (k, -shell, shell, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_y1z1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;
int k;

n = polytope->n_cells;

/*********************************************
 * find neighbours in cells along y1z1 edges *
 *********************************************/

n_y = shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

for (k = -shell + 1; k <= shell - 1; k++)
  {
  n_x = k + cell_x;
  if (n_x < 0 || n_x > n - 1) continue;

  if (static_inside_local (k, shell, shell, polytope) == TRUE)
    static_shell_find (n_x, n_y, n_z, polytope);
  }
}

static void static_raw_x0y0z0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;

n = polytope->n_cells;

/**************************************************
 * find neighbours in cells along x0y0z0 vertices *
 **************************************************/

n_x = -shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = -shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = -shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

if (static_inside_local (-shell, -shell, -shell, polytope) == TRUE)
  static_shell_find (n_x, n_y, n_z, polytope);
}

static void static_raw_x1y0z0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;

n = polytope->n_cells;

/**************************************************
 * find neighbours in cells along x1y0z0 vertices *
 **************************************************/

n_x = shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = -shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = -shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

if (static_inside_local (shell, -shell, -shell, polytope) == TRUE)
  static_shell_find (n_x, n_y, n_z, polytope);
}

static void static_raw_x0y1z0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;

n = polytope->n_cells;

/**************************************************
 * find neighbours in cells along x0y1z0 vertices *
 **************************************************/

n_x = -shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = -shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

if (static_inside_local (-shell, shell, -shell, polytope) == TRUE)
  static_shell_find (n_x, n_y, n_z, polytope);
}

static void static_raw_x1y1z0 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;

n = polytope->n_cells;

/**************************************************
 * find neighbours in cells along x1y1z0 vertices *
 **************************************************/

n_x = shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = -shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

if (static_inside_local (shell, shell, -shell, polytope) == TRUE)
  static_shell_find (n_x, n_y, n_z, polytope);
}

static void static_raw_x0y0z1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;

n = polytope->n_cells;

/**************************************************
 * find neighbours in cells along x0y0z1 vertices *
 **************************************************/

n_x = -shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = -shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

if (static_inside_local (-shell, -shell, shell, polytope) == TRUE)
  static_shell_find (n_x, n_y, n_z, polytope);
}

static void static_raw_x1y0z1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;

n = polytope->n_cells;

/**************************************************
 * find neighbours in cells along x1y0z1 vertices *
 **************************************************/

n_x = shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = -shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

if (static_inside_local (shell, -shell, shell, polytope) == TRUE)
  static_shell_find (n_x, n_y, n_z, polytope);
}

static void static_raw_x0y1z1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;

n = polytope->n_cells;

/**************************************************
 * find neighbours in cells along x0y1z1 vertices *
 **************************************************/

n_x = -shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

if (static_inside_local (-shell, shell, shell, polytope) == TRUE)
  static_shell_find (n_x, n_y, n_z, polytope);
}

static void static_raw_x1y1z1 (int cell_x, int cell_y, int cell_z,
int shell, gamgi_polytope *polytope)
{
int n, n_x, n_y, n_z;

n = polytope->n_cells;

/**************************************************
 * find neighbours in cells along x1y1z1 vertices *
 **************************************************/

n_x = shell + cell_x;
if (n_x < 0 || n_x > n - 1) return;

n_y = shell + cell_y;
if (n_y < 0 || n_y > n - 1) return;

n_z = shell + cell_z;
if (n_z < 0 || n_z > n - 1) return;

if (static_inside_local (shell, shell, shell, polytope) == TRUE)
  static_shell_find (n_x, n_y, n_z, polytope);
}

static void static_reset_shell (gamgi_polytope *polytope)
{
int i;

/***************************************
 * start shell data to find neighbours *
 ***************************************/

if (polytope->number > 0)
  {
  polytope->n_vertices = polytope->n_neighbours;

  polytope->max = DBL_MAX;
  for (i = 0; i < polytope->n_neighbours; i++)
    polytope->length[i] = DBL_MAX;
  }
else polytope->n_vertices = 0;
}

static void static_reset_arrays (gamgi_polytope *polytope)
{
int n_faces, n_edges, n_vertices;
int i;

/****************
 * reset arrays *
 ****************/

n_edges = polytope->n_edges;
for (i = 0; i < n_edges; i++)
  static_edge_start (i, polytope);

n_vertices = polytope->n_vertices;
for (i = 0; i < n_vertices; i++)
  static_vertex_start (i, polytope);

n_faces = polytope->n_faces;
for (i = 0; i < n_faces; i++)
  static_face_start (i, polytope);
}

static gamgi_bool static_build_vertices (int cell_x, int cell_y, int cell_z,
gamgi_slist *start, gamgi_atom *atom, gamgi_polytope *polytope)
{
int shell, shell_max;
int n, min, max;

/***********************************************************
 * go around cells (starting with central cell) and create *
 * vertices array, with closer vertices, in reverse order, *
 * so the vertices more far away are handled first         *
 ***********************************************************/

static_shell_find (cell_x, cell_y, cell_z, polytope);

n = polytope->n_cells;

min = cell_x;
if (cell_y < min) min = cell_y;
if (cell_z < min) min = cell_z;

max = cell_x;
if (cell_y > max) max = cell_y;
if (cell_z > max) max = cell_z;

shell_max = max;
if (shell_max < n - 1 - min) shell_max = n - 1 - min;
for (shell = 1; shell <= shell_max; shell++)
  {
  if (static_inside_global (shell, polytope) == FALSE) break;
    
  /*********
   * faces *
   *********/

  static_raw_x0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x1 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_y0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_y1 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_z0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_z1 (cell_x, cell_y, cell_z, shell, polytope);

  /*********
   * edges *
   *********/

  static_raw_x0y0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x1y0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x0y1 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x1y1 (cell_x, cell_y, cell_z, shell, polytope);

  static_raw_x0z0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x1z0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x0z1 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x1z1 (cell_x, cell_y, cell_z, shell, polytope);

  static_raw_y0z0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_y1z0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_y0z1 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_y1z1 (cell_x, cell_y, cell_z, shell, polytope);

  /************
   * vertices *
   ************/

  static_raw_x0y0z0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x1y0z0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x0y1z0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x1y1z0 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x0y0z1 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x1y0z1 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x0y1z1 (cell_x, cell_y, cell_z, shell, polytope);
  static_raw_x1y1z1 (cell_x, cell_y, cell_z, shell, polytope);
  }

/************************************
 * Not enough neighbours were found *
 ************************************/

if (polytope->n_vertices < 2 || 
polytope->length[polytope->n_vertices - 1] == DBL_MAX) return FALSE;

return TRUE;
}

static int static_build_vicinity (gamgi_polytope *polytope)
{
gamgi_atom **neighbours;
gamgi_atom *atom;
double *length;
int dot, cross, mix;
int n, i;

/************************************************************************
 *      at this stage it is guaranteed that at least two neighbours     *
 *      exist, but they might be too close to even define a line!       *
 *                                                                      *
 * At this stage neighbours are ordered by distance, so the last one    *
 * will define a valid vertex in the final convex hull. Moreover, atoms *
 * more far away will likely contribute more to the final convex hull.  *
 ************************************************************************/

neighbours = polytope->neighbours;
n = polytope->n_vertices;
length = polytope->length;

/**************************
 * look for second vertex *
 **************************/

for (i = 2; i <= n; i++)
  {
  atom =  polytope->neighbours[n - i];
  dot = static_dot (n - 1, n - i,
  GAMGI_CHEM_POLYTOPE_TOLERANCE_LENGTH, polytope);
  if (dot > 0)
    {
    /*******************************************************
     * valid length: swap valid atom with second last atom *
     *******************************************************/

    atom = neighbours[n - i];
    neighbours[n - i] = neighbours[n - 2];
    neighbours[n - 2] = atom;
    break;
    }
  }
if (i == n + 1) return 1;

/*************************
 * look for third vertex *
 *************************/

for (i = 3; i <= n; i++)
  {
  atom =  polytope->neighbours[n - i];
  cross = static_cross (n - 1, n - 2, n - i,
  GAMGI_CHEM_POLYTOPE_TOLERANCE_AREA, polytope);
  if (cross > 0)
    {
    /****************************************************
     * valid area: swap valid atom with third last atom *
     ****************************************************/

    atom = neighbours[n - i];
    neighbours[n - i] = neighbours[n - 3];
    neighbours[n - 3] = atom;
    break;
    }
  }
if (i == n + 1) return 2;

/**************************
 * look for fourth vertex *
 **************************/

for (i = 4; i <= n; i++)
  {
  atom =  polytope->neighbours[n - i];
  mix = static_mix (n - 1, n - 2, n - 3, n - i, 
  GAMGI_CHEM_POLYTOPE_TOLERANCE_VOLUME, polytope); 
  if (mix != 0)
    {
    /*******************************************************
     * valid volume: swap valid atom with fourth last atom *
     *******************************************************/

    atom = neighbours[n - i];
    neighbours[n - i] = neighbours[n - 4];
    neighbours[n - 4] = atom;

    if (mix < 0)
      { 
      /**************************************************************
       * swap atoms 2 and 3, to make it easy to build a tetrahedron *
       * with all faces orientated counter-clockwise to the outside *
       **************************************************************/

      atom = neighbours[n - 3];
      neighbours[n - 3] = neighbours[n - 2];
      neighbours[n - 2] = atom;
      }
    break;
    }
  }
if (i == n + 1) return 3;

/*****************************************************
 * four valid vertices were found: build tetrahedron *
 *****************************************************/

return 4;
}

static void static_swap (gamgi_polytope *polytope)
{
gamgi_atom **neighbours;
gamgi_atom *atom;
int i, n_vertices, half;

neighbours = polytope->neighbours;
n_vertices = polytope->n_vertices;
half = n_vertices / 2;

/************************************************
 * swap atoms, so the last atom (the farthest   *
 * atom) becomes the first one and the first    *
 * atom (the closest atom) becomes the last one *
 *                                              *
 * When n_vertices is even, half of the atoms   *
 * wap with the other half. When n_vertices     *
 * is odd, the central atom does not swap.      *
 ************************************************/

for (i = 0; i < half; i++)
  {
  atom = neighbours[i];
  neighbours[i] = neighbours[n_vertices - 1 - i];
  neighbours[n_vertices - 1 - i] = atom;
  }
}

static void static_build_line (gamgi_polytope *polytope)
{
printf ("line\n");
}

static void static_build_polygon (gamgi_polytope *polytope)
{
printf ("triangle\n");
}

static void static_build_tetrahedron (gamgi_polytope *polytope)
{
int **faces, **edges, **vertices;
int *vertex, *edge, *face;

faces = polytope->faces;
edges = polytope->edges;
vertices = polytope->vertices;

/*****************************************************************************
 * build a tetrahedron orientated counter-clockwise, with theses rules:      *
 *                                                                           *
 * set vertices v0, v1, v2, v3, so (v1 - v0) X (v2 - v0) . (v3 - v0) > 0     *
 *                                                                           *
 * set faces f0, f1, f2, f3, so the face number is the excluded vertex       *
 *                                                                           *
 * set edges e0, e1, e2, e3, e4, e5, with the vertex increasing combination: *
 * e0: v0v1  e1: v0v2  e2: v0v3  e3: v1v2  e4: v1v3  e5: v2v3                *
 *****************************************************************************/

/***************************
 * add vertices 0, 1, 2, 3 *
 ***************************/

vertex = vertices[0];
vertex[2] = 3;

vertex = vertices[1];
vertex[2] = 3;

vertex = vertices[2];
vertex[2] = 3;

vertex = vertices[3];
vertex[2] = 3;

/************************************************************
 *                add edges 0, 1, 2, 3, 4, 5                *
 *                                                          *
 * write first the face containing this vertex sequence     *
 * and then the face containing the reverse vertex sequence *
 *                                                          *
 * e0: f2 f3 v0 v1     e2: f1 f2 v0 v3      e4: f2 f0 v1 v3 *
 * e1: f3 f1 v0 v2     e3: f0 f3 v1 v2      e5: f0 f1 v2 v3 *
 ************************************************************/

edge = edges[0];
edge[1] = GAMGI_CHEM_POLYTOPE_KEEP;
edge[2] = 2;
edge[3] = 3;
edge[4] = 0;
edge[5] = 1;

edge = edges[1];
edge[1] = GAMGI_CHEM_POLYTOPE_KEEP;
edge[2] = 3;
edge[3] = 1;
edge[4] = 0;
edge[5] = 2;

edge = edges[2];
edge[1] = GAMGI_CHEM_POLYTOPE_KEEP;
edge[2] = 1;
edge[3] = 2;
edge[4] = 0;
edge[5] = 3;

edge = edges[3];
edge[1] = GAMGI_CHEM_POLYTOPE_KEEP;
edge[2] = 0;
edge[3] = 3;
edge[4] = 1;
edge[5] = 2;

edge = edges[4];
edge[1] = GAMGI_CHEM_POLYTOPE_KEEP;
edge[2] = 2;
edge[3] = 0;
edge[4] = 1;
edge[5] = 3;

edge = edges[5];
edge[1] = GAMGI_CHEM_POLYTOPE_KEEP;
edge[2] = 0;
edge[3] = 1;
edge[4] = 2;
edge[5] = 3;

/************************
 * add faces 0, 1, 2, 3 *
 ************************/

face = faces[0];
face[1] = GAMGI_CHEM_POLYTOPE_KEEP;
face[3] = 3;

face[4] = 3;
face[5] = 10;
face[6] = 7;

face[7] = 5;
face[8] = 4;
face[9] = 10;

face[10] = 4;
face[11] = 7;
face[12] = 4;

face = faces[1];
face[1] = GAMGI_CHEM_POLYTOPE_KEEP;
face[3] = 3;

face[4] = 1;
face[5] = 10;
face[6] = 7;

face[7] = 2;
face[8] = 4;
face[9] = 10;

face[10] = 5;
face[11] = 7;
face[12] = 4;

face = faces[2];
face[1] = GAMGI_CHEM_POLYTOPE_KEEP;
face[3] = 3;

face[4] = 0;
face[5] = 10;
face[6] = 7;

face[7] = 4;
face[8] = 4;
face[9] = 10;

face[10] = 2;
face[11] = 7;
face[12] = 4;

face = faces[3];
face[1] = GAMGI_CHEM_POLYTOPE_KEEP;
face[3] = 3;

face[4] = 0;
face[5] = 10;
face[6] = 7;

face[7] = 1;
face[8] = 4;
face[9] = 10;

face[10] = 3;
face[11] = 7;
face[12] = 4;
}

static void static_build_polyhedron (gamgi_polytope *polytope)
{
/**************************************************************************
 * when applying the incremental algorithm, it should be faster to handle *
 * the farthest atoms first, as part of the other atoms may fall inside   *
 * and be removed without requiring expensive polyhedra recalculations    *
 **************************************************************************/

static_swap (polytope);

static_build_tetrahedron (polytope);

/*
static_conflict_create (polytope);

for (i = 4; i < polytope->n_vertices; i++)
  static_build_vertex (i, polytope);
*/

static_export (polytope);
}

static gamgi_bool static_build_central (int cell_x, int cell_y, int cell_z, 
gamgi_slist *start, gamgi_atom *atom, gamgi_polytope *polytope)
{
int independent;

/****************
 * central atom *
 ****************/
 
polytope->atom = atom;

/*************************************
 * reset shell data, building arrays *
 *************************************/

static_reset_shell (polytope);
static_reset_arrays (polytope);

if (static_build_vertices (cell_x, cell_y, cell_z, 
start, atom, polytope) == FALSE) return FALSE;

independent = static_build_vicinity (polytope);
if (independent < 2) return FALSE;

if (independent == 2) static_build_line (polytope);
if (independent == 3) static_build_polygon (polytope);
if (independent == 4) static_build_polyhedron (polytope);

return TRUE;
}

static gamgi_bool static_build (gamgi_polytope *polytope)
{
gamgi_atom *atom;
gamgi_slist **cells;
gamgi_slist *slist, *start;
int cell_x, cell_y, cell_z;
int n_cells, offset;

cells = polytope->cells;
n_cells = polytope->n_cells;

for (cell_z = 0; cell_z < n_cells; cell_z++)
  {
  for (cell_y = 0; cell_y < n_cells; cell_y++)
    {
    for (cell_x = 0; cell_x < n_cells; cell_x++)
      {
      offset = cell_z * n_cells * n_cells + cell_y * n_cells + cell_x;
      start = cells[offset];
      for (slist = start; slist != NULL; slist = slist->next)
        {
        /*****************************************************
         * build polyhedron from distance-ordered neighbours *
         *****************************************************/

        atom = GAMGI_CAST_ATOM slist->data;
        if (atom->element == polytope->atom->element && 
        (polytope->global == TRUE || atom == polytope->atom) &&
        static_build_central (cell_x, cell_y, cell_z, start, 
        atom, polytope) == FALSE) return FALSE;
        }

      }
    }
  }

return TRUE;
}

static void static_start_arrays (gamgi_polytope *polytope)
{
int i;

polytope->n_faces = GAMGI_CHEM_POLYTOPE_FACES_START;
polytope->faces = (int **) malloc (polytope->n_faces * sizeof (int *));
for (i = 0; i < polytope->n_faces; i++)
  static_face_create (i, polytope);

polytope->n_edges = GAMGI_CHEM_POLYTOPE_EDGES_START;
polytope->edges = (int **) malloc (polytope->n_edges * sizeof (int *));
for (i = 0; i < polytope->n_edges; i++)
  static_edge_create (i, polytope);

polytope->n_vertices = GAMGI_CHEM_POLYTOPE_VERTICES_START;
polytope->vertices = (int **) malloc (polytope->n_vertices * sizeof (int *));
for (i = 0; i < polytope->n_vertices; i++)
  static_vertex_create (i, polytope);
}

static void static_end_arrays (gamgi_polytope *polytope)
{
int i;

for (i = 0; i < polytope->n_faces; i++) free (polytope->faces[i]);
free (polytope->faces);

for (i = 0; i < polytope->n_edges; i++) free (polytope->edges[i]);
free (polytope->edges);

for (i = 0; i < polytope->n_vertices; i++) free (polytope->vertices[i]);
free (polytope->vertices);
}

static void static_start_cells (gamgi_polytope *polytope)
{
gamgi_atom *atom;
gamgi_slist **cells;
gamgi_dlist *dlist;
double min_x, max_x;
double min_y, max_y;
double min_z, max_z; 
double width_x, width_y, width_z;
double r_cells, min;
int cell_x, cell_y, cell_z;
int central, neighbour;
int offset;
int n;

central = polytope->atom->element;
neighbour = polytope->element;

/**************************************************
 * determine: 1) system dimensions; 2) number of  *
 * atoms; 3) maximum radius (radical tesselation) *
 **************************************************/

min_x = DBL_MAX; max_x = -DBL_MAX;
min_y = DBL_MAX; max_y = -DBL_MAX;
min_z = DBL_MAX; max_z = -DBL_MAX;

n = 0;
for (dlist = polytope->atoms; dlist != NULL; dlist = dlist->next)
  {
  /*****************************************************
   * give a different, sequential, number to each atom *
   *                                                   *
   *     find minimum and maximum cell boundaries      *
   *                                                   *
   *     find maximum radius (radical tesselation)     *
   *****************************************************/

  atom = GAMGI_CAST_ATOM dlist->data;
  if (atom->element != central && atom->element != neighbour) continue;
  n++;

  if (atom->position[0] < min_x) min_x = atom->position[0];
  if (atom->position[0] > max_x) max_x = atom->position[0];

  if (atom->position[1] < min_y) min_y = atom->position[1];
  if (atom->position[1] > max_y) max_y = atom->position[1];

  if (atom->position[2] < min_z) min_z = atom->position[2];
  if (atom->position[2] > max_z) max_z = atom->position[2];
  }
polytope->n_atoms = n;

/************************************************************
 * determine number of cells on each direction:             *
 * on average, each cell has approximately one atom.        *
 *                                                          *
 * allocate array of cells:                                 *
 * each cell points to a linked list of atoms in that cell. *
 ************************************************************/

if (modf (pow (n, 1/3.0), &r_cells) > 0.5) r_cells++;
polytope->n_cells = n = r_cells;

cells = (gamgi_slist **) calloc (pow (n, 3), sizeof (gamgi_slist *));
polytope->cells = cells;

/***********************************************************
 * determine cell dimensions: when the width is too small  *
 * (for example when the atoms have the same z coordinate) *
 * we need to enlarge the cell dimensions symmetricaly, to *
 * guarantee that the distance from the seeds at old min,  *
 * max coordinates to the new border planes is the same    *
 ***********************************************************/

min = GAMGI_CHEM_POLYTOPE_WIDTH;

if (max_x - min_x < min)
  {
  min_x = (min_x + max_x - min) / 2.0;
  max_x = (min_x + max_x + min) / 2.0;
  }
width_x = (max_x - min_x) / n;

if (max_y - min_y < min)
  {
  min_y = (min_y + max_y - min) / 2.0;
  max_y = (min_y + max_y + min) / 2.0;
  }
width_y = (max_y - min_y) / n;

if (max_z - min_z < min)
  {
  min_z = (min_z + max_z - min) / 2.0;
  max_z = (min_z + max_z + min) / 2.0;
  }
width_z = (max_z - min_z) / n;

polytope->width_x = width_x;
polytope->width_y = width_y;
polytope->width_z = width_z;

/********************************************
 * 1) find cell where each atom belongs     *
 * 2) add atom to linked list for that cell *
 ********************************************/

for (dlist = polytope->atoms; dlist != NULL; dlist = dlist->next)
  {
  atom = GAMGI_CAST_ATOM dlist->data; 
  if (atom->element != central && atom->element != neighbour) continue;

  cell_x = floor ((atom->position[0] - min_x) / width_x);
  if (cell_x == n) cell_x--;
  cell_y = floor ((atom->position[1] - min_y) / width_y);
  if (cell_y == n) cell_y--;
  cell_z = floor ((atom->position[2] - min_z) / width_z);
  if (cell_z == n) cell_z--;

  offset = cell_z * n * n + cell_y * n + cell_x;
  cells[offset] = gamgi_engine_slist_add_start (cells[offset]);
  cells[offset]->data = atom;
  }

}

static void static_end_cells (gamgi_polytope *polytope)
{
gamgi_slist **cells;
gamgi_slist *slist;
int n_cells;
int cell_x, cell_y, cell_z;
int offset;

/********************************************
 * remove lists of nodes, one for each cell *
 ********************************************/

cells = polytope->cells;
n_cells = polytope->n_cells;
for (cell_z = 0; cell_z < n_cells; cell_z++)
  {
  for (cell_y = 0; cell_y < n_cells; cell_y++)
    {
    for (cell_x = 0; cell_x < n_cells; cell_x++)
      {
      offset = cell_z * n_cells * n_cells + cell_y * n_cells + cell_x;
      slist = cells[offset];

      while (slist != NULL)
        slist = gamgi_engine_slist_remove_start (slist);
      }
    }
  }

}

static void static_start_shell (gamgi_polytope *polytope)
{
int n;

/*********************************************************
 * find a fixed number of neighbours around central atom *
 * or find all neighbours within a given distance        *
 *********************************************************/

if (polytope->number > 0)
  n = polytope->number;
else
  n = GAMGI_CHEM_POLYTOPE_VERTICES_START;

polytope->neighbours = (gamgi_atom **) malloc (n * sizeof (gamgi_atom *));
polytope->length = (double *) malloc (n * sizeof (double));

polytope->n_neighbours = n;
}

static void static_end_shell (gamgi_polytope *polytope)
{
free (polytope->neighbours);
free (polytope->length);
}

static gamgi_polytope *static_start (gamgi_object *parent, gamgi_atom *atom, 
gamgi_bool global, int element, int number, double radius, 
gamgi_enum style, float *color)
{
gamgi_polytope *polytope;
gamgi_dlist *atoms;

/*****************************************
 * allocate atom list and main structure *
 *****************************************/

atoms = gamgi_engine_dlist_atom_object (parent, NULL);
if (atoms == NULL) return NULL;

polytope = (gamgi_polytope *) malloc (sizeof (gamgi_polytope));
polytope->atoms = atoms;

/*******************
 * save input data *
 *******************/

polytope->parent = parent;

polytope->atom = atom;
polytope->global = global;

polytope->element = element;
polytope->number = number;
polytope->max = radius * radius;

polytope->style = style;
polytope->color = color;

/**********************************************************
 * allocate atoms in cells with lists, for atom searching *
 *                                                        *
 * allocate arrays faces,edges,vertices to build polytope *
 *                                                        *
 *        allocate arrays to find shell neighbours        *
 **********************************************************/

static_start_cells (polytope);
static_start_arrays (polytope);
static_start_shell (polytope);

return polytope;
}

static void static_end (gamgi_polytope *polytope)
{
gamgi_dlist *dlist;

/**************************
 * free arrays and shells *
 **************************/

static_end_shell (polytope);
static_end_arrays (polytope);
static_end_cells (polytope);

/**************************
 * reset atoms, free list *
 **************************/

dlist = polytope->atoms;
while (dlist != NULL)
  {
  (GAMGI_CAST_ATOM dlist->data)->mark = 0;
  dlist = gamgi_engine_dlist_remove_start (dlist);
  }

free (polytope);
}

gamgi_bool gamgi_chem_polytope (gamgi_object *parent, gamgi_atom *atom,
gamgi_bool global, int element, int number, double radius, 
gamgi_enum style, float *color, GtkWidget *text)
{
gamgi_polytope *polytope;
gamgi_bool valid;

polytope = static_start (parent, atom, global,
element, number, radius, style, color);
if (polytope == NULL) return FALSE;

valid = static_build (polytope);
static_end (polytope);

return valid;
}
