/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "glade.h"
#include "prefs.h"
#include "prefs_utils.h"
#include "prefs_mime.h"
#include "mozilla_prefs.h"
#include "mozilla.h"
#include "history.h"
#include "spinner.h"
#include "embed.h"
#include "gtkmozembed.h"
#include "mime.h"
#include "gfilepicker.h"
#include "mozilla_i18n.h"
#include "window.h"
#include "toolbar.h"
#include "bookmarks.h"
#include "state.h"
#include "eel-gconf-extensions.h"
#include "dialog.h"
#include "toolbar_editor.h"
#include "misc_gui.h"

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-util.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtklist.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkinvisible.h>
#include <libgnomeui/gnome-icon-list.h>
#include <libgnomeui/gnome-font-picker.h>
#include <libgnomeui/gnome-color-picker.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-file-entry.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
/* from nsICache.h */
enum
{
	STORE_ANYWHERE	      = 0,
	STORE_IN_MEMORY	      = 1,
	STORE_ON_DISK	      = 2,
	STORE_ON_DISK_AS_FILE = 3
};

/* local types */

typedef enum
{
	PG_AUTOAPPLY,
	PG_PROXY,
	PG_ALL
} PrefGroup;

typedef struct
{ 
	gchar const *name;
	PreferencesPageId page;
	gchar const *icon;
} PreferencesPanelButton;

typedef struct
{ 
	gchar const *name;
	PreferencesPanelButton const *icons;
} PreferencesSidebarItem;

typedef enum
{
	PT_TOGGLEBUTTON,
	PT_RADIOBUTTON,
	PT_SPINBUTTON,
	PT_COLOR,
	PT_OPTIONMENU,
	PT_ENTRY
} PrefType;

typedef struct 
{
	gchar *widgetname;
	gchar *prefname;
	PrefType type;
	char **sg;
	PrefGroup group;
} PrefInfo;

typedef struct 
{
	PrefInfo *info;
	GtkWidget *widget;
} PrefQueueItem;

/* a row in the css manager */
typedef struct
{
	gchar *filename;
	gboolean is_default;
} CSSManagerRow;

static void
prefs_lookup_widgets (GladeXML *gxml);
static void
prefs_disconnect_signals (void);
static void
prefs_connect_signals (void);
static void
prefs_add_to_queue (GtkWidget *widget, PrefInfo *pi);
static void
prefs_apply_queue (PrefGroup group);
static void 
prefs_togglebutton_clicked_cb (GtkWidget *widget, PrefInfo *pi);
static void 
prefs_radiobutton_clicked_cb (GtkWidget *widget,  PrefInfo *pi);
static void 
prefs_spinbutton_changed_cb (GtkWidget *widget,  PrefInfo *pi);
static void 
prefs_color_changed_cb (GtkWidget *widget, char *dummy);
static void 
prefs_entry_changed_cb (GtkWidget *widget,  PrefInfo *pi);
static void 
prefs_optionmenu_selected_cb (GtkWidget *widget,  PrefInfo *pi);
static void prefs_set_dialog_to_config ();
static void
prefs_set_group_sensitivity (GtkWidget *widget, PrefType type, char **sg);
static void
prefs_set_widget_from_config (GtkWidget *widget, char *pref, PrefType type, char **sg);
static void prefs_help (void);
static void prefs_build_dialog (void);
static void
prefs_build_sidebar (void);
static void 
prefs_sidebar_select_panel_cb (GtkWidget *button);
static void prefs_sidebar_select_page_cb (GtkWidget *button);
static void
prefs_fonts_language_init (void);
static void
prefs_theme_selector_init (void);
static void
theme_selector_fill_themelist_from_dir (GnomeIconList *theme_list, gchar *base);
static void
free_theme_selector (GnomeIconList *list);
static void
prefs_custom_css_set_config_from_list (void);
static void
prefs_custom_css_set_list_from_config (void);

/* Callbacks */
void
prefs_switch_page_cb (GtkWidget *notebook, 
		      GtkNotebookPage *notebook_page,
		      guint page_num,
		      PreferencesDialog *pd);
void
prefs_ok_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void
prefs_proxy_apply_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
gboolean
prefs_close_cb (GnomeDialog *dialog, PreferencesDialog *pd);
void
prefs_help_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void
prefs_clear_disk_cache_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void
prefs_clear_memory_cache_button_clicked_cb (GtkButton *button, 
					    PreferencesDialog *pd);
void
prefs_homepage_my_portal_button_clicked_cb (GtkButton *button, 
					    PreferencesDialog *pd);
void
prefs_homepage_current_button_clicked_cb (GtkButton *button, 
					  PreferencesDialog *pd);
void
prefs_lang_list_selection_changed_cb (GtkList *list, PreferencesDialog *pd);
void
prefs_lang_add_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void
prefs_lang_remove_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void
prefs_lang_up_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void
prefs_lang_down_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void
prefs_clear_history_button_clicked_cb (GtkButton *button,
				       PreferencesDialog *pd);
void
prefs_clear_history_question_cb (gint reply, gpointer data);
void
prefs_browse_clicked_cb (GnomeFileEntry *fileentry, PreferencesDialog *pd);
void
prefs_font_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd);
void
prefs_color_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd);
void 
prefs_spinner_select_icon_cb (GnomeIconList *gil, gint num, 
			      GdkEvent *event, PreferencesDialog *pd);
void
prefs_theme_select_icon_cb (GnomeIconList *list, gint row, GdkEvent *event, 
			    PreferencesDialog *pd);
void
preferences_switch_page_cb (GtkWidget *notebook, 
			    GtkNotebookPage *notebook_page,
			    guint page_num,
			    PreferencesDialog *pd);
void
prefs_encoding_activated_cb (GtkWidget *button, 
			     PreferencesDialog *pd);
void
prefs_proxy_auto_url_reload_cb (GtkButton *button,
				PreferencesDialog *pd);
void
custom_css_clist_select_row_cb (GtkCList *clist, gint row, gint column, 
				GdkEventButton *event, 
				PreferencesDialog *pd);
void
custom_css_clist_unselect_row_cb (GtkCList *clist, gint row, gint column, 
				  GdkEventButton *event, 
				  PreferencesDialog *pd);
void
custom_css_clist_key_press_cb (GtkWidget *w, GdkEventKey *event,
			       PreferencesDialog *pd);
void
custom_css_add_button_clicked_cb (GtkWidget *button,
				  PreferencesDialog *pd);
void
custom_css_remove_button_clicked_cb (GtkWidget *button,
				     PreferencesDialog *pd);
void
custom_css_default_check_toggled_cb (GtkToggleButton *button,
				     PreferencesDialog *pd);
static void 
prefs_serif_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd);
static void 
prefs_monospace_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd);
static 
void prefs_sansserif_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd);
static void 
prefs_cursive_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd);
static void 
prefs_fantasy_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd);
static void 
prefs_minsize_font_changed_cb (GtkWidget *widget, PreferencesDialog *pd);
gint 
prefs_minfont_spinbutton_timeout_cb (int info[]);
gint 
prefs_spinbutton_timeout_cb (PrefQueueItem *item);
void
prefs_certsmanager_button_clicked_cb (GtkWidget *widget, PreferencesDialog *pd);

/* local variables */

/* FIXME */
static GtkWidget *page_buttons[PREFS_PAGE_COUNT];
static GtkWidget *current_panel = NULL;
static GList *prefs_queue = NULL;

#define SPIN_DELAY 0.20

GTimer *spin_timer = NULL;
PrefQueueItem spin_item;

GTimer *minfont_spin_timer = NULL;
int minfont_info[2];

PreferencesDialog *pd = NULL;

#define END_STRING "_"

char *source_sg [] =
{ 
	"external_source_viewer_entry",
	NULL
};

char *cookies_sg [] =
{
	NULL,
	NULL,
	"!warn_cookie_checkbutton",
	NULL,
	END_STRING
};

char *javascript_sg [] =
{
	"statusbar_rewrite_check",
	NULL,
	END_STRING
};

char *shorten_sg [] =
{
	"tabbed_shorten_spin2",
	NULL,
	"tabbed_shorten_spin",
	"tabbed_separator_check",
	"tabbed_vowel_check",
	"tabbed_prefix_check",
	NULL,
	END_STRING
};

char *download_sg [] =
{
	NULL,
	NULL,
	"download_command_combo",
	"download_run_in_terminal",
	NULL,
	END_STRING
};

char *proxy_sg [] = 
{
	NULL,
	"http_proxy_port_spin",
	"http_proxy_entry",
	"ftp_proxy_port_spin",
	"ftp_proxy_entry",
	"ssl_proxy_port_spin",
	"ssl_proxy_entry",
	"socks_proxy_entry",
	"socks_proxy_port_spin",
	"socksv4_radiobutton",
	"socksv5_radiobutton",
	"no_proxies_for_entry",
	NULL,
	"proxy_auto_url_entry",
	"proxy_auto_reload_button",
	NULL,
	END_STRING
};

static const
PrefInfo lookup_table[] =
{
	{ "tabbed_always_show_check", CONF_TABS_TABBED_ALWAYS_SHOW, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY},
	{ "always_save_session_check", CONF_GENERAL_ALWAYS_SAVE_SESSION, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY},
	{ "allow_popups_checkbutton", CONF_FILTERING_ALLOW_POPUPS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "download_ask_dir", CONF_DOWNLOADING_ASK_DIR, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "download_run_in_terminal", CONF_DOWNLOADING_EXTERNAL_TERMINAL, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_enable_check", CONF_TABS_TABBED, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_popups_enable_check", CONF_TABS_TABBED_POPUPS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_autojump_check", CONF_TABS_TABBED_AUTOJUMP, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_closebutton_check", CONF_TABS_TABBED_CLOSEBUTTON, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_closebutton_insensitive_check", CONF_TABS_TABBED_CLOSEBUTTON_INSENSITIVE, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_separator_check", CONF_TABS_TABBED_SEPARATOR, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_vowel_check", CONF_TABS_TABBED_VOWEL, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_prefix_check", CONF_TABS_TABBED_PREFIX, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "toolbar_share_main_toolbar", CONF_TOOLBAR_URL_LOCATION, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "show_toolbars_in_fullscreen_check", CONF_WINDOWS_FS_SHOW_TOOLBARS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "show_bookmarks_in_fullscreen_check", CONF_WINDOWS_FS_SHOW_BOOKMARKS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "show_menubar_in_fullscreen_check", CONF_WINDOWS_FS_SHOW_MENUBAR, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "show_statusbar_in_fullscreen_check", CONF_WINDOWS_FS_SHOW_STATUSBAR, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "show_toolbars_check", CONF_WINDOWS_SHOW_TOOLBARS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "show_bookmarks_check", CONF_WINDOWS_SHOW_BOOKMARKS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "show_menubar_check", CONF_WINDOWS_SHOW_MENUBAR, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "show_statusbar_check", CONF_WINDOWS_SHOW_STATUSBAR, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "use_own_fonts_checkbutton", CONF_RENDERING_USE_OWN_FONTS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "use_own_colors_checkbutton", CONF_RENDERING_USE_OWN_COLORS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "underline_links_checkbutton", CONF_RENDERING_UNDERLINE_LINKS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "smart_bm_history_check", CONF_BOOKMARKS_HISTORY, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "smart_bm_clear_check", CONF_BOOKMARKS_CLEAR, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "autocompletion_enabled_checkbutton", CONF_HISTORY_AUTOCOMP_ENABLE, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "autocompletion_show_alternatives_auto_checkbutton", CONF_COMPLETION_SHOW_LIST_AUTO, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "use_external_source_viewer_check", CONF_PROGRAMS_USE_EXTERNAL_SOURCE_VIEWER, PT_TOGGLEBUTTON, source_sg, PG_AUTOAPPLY },
	{ "use_internal_ghelp_check", CONF_GHELP_USE_GALEON, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "java_checkbutton", CONF_FILTERING_JAVA_ENABLED, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "javascript_checkbutton", CONF_FILTERING_JAVASCRIPT_ENABLED, PT_TOGGLEBUTTON, javascript_sg, PG_AUTOAPPLY },
	{ "statusbar_rewrite_check", CONF_FILTERING_STATUSBAR_REWRITE, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "proxy_keepalive_checkbutton", CONF_NETWORK_PROXY_KA, PT_TOGGLEBUTTON, NULL, PG_PROXY },
	{ "warn_cookie_checkbutton", CONF_PERSISTENT_COOKIE_WARN, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "session_cookie_checkbutton", CONF_PERSISTENT_COOKIE_LIFETIME, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "passwords_save_checkbutton", CONF_PERSISTENT_PASSWORDS_SAVE, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_insert_new_tabs_check", CONF_TABS_TABBED_INSERT_NEW_TABS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "ssl2_checkbutton", CONF_SECURITY_SSL2, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "ssl3_checkbutton", CONF_SECURITY_SSL3, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },
	{ "tls_checkbutton", CONF_SECURITY_TLS, PT_TOGGLEBUTTON, NULL, PG_AUTOAPPLY },

	{ "startup_show_homepage", CONF_GENERAL_STARTPAGE_TYPE, PT_RADIOBUTTON, NULL, PG_AUTOAPPLY },
	{ "newpage_show_homepage", CONF_GENERAL_NEWPAGE_TYPE, PT_RADIOBUTTON, NULL, PG_AUTOAPPLY },
	{ "bookmarks_use_page_title", CONF_BOOKMARKS_TITLE, PT_RADIOBUTTON, NULL, PG_AUTOAPPLY },
	{ "bookmarks_show_actions_top", CONF_BOOKMARKS_XITEMS, PT_RADIOBUTTON, NULL, PG_AUTOAPPLY },
	{ "download_use_builtin", CONF_DOWNLOADING_DOWNLOADER, PT_RADIOBUTTON, download_sg, PG_AUTOAPPLY },
	{ "shorten_target_radiobutton", CONF_TABS_TABBED_SHORTEN_STYLE, PT_RADIOBUTTON, shorten_sg, PG_AUTOAPPLY },
	{ "show_dock_none_radiobutton", CONF_WINDOWS_SHOW_DOCK, PT_RADIOBUTTON, NULL, PG_AUTOAPPLY },
	{ "download_ftp_use_builtin", CONF_DOWNLOADING_EXTERNAL_FTP, PT_RADIOBUTTON, NULL, PG_AUTOAPPLY },
	{ "images_always_load", CONF_FILTERING_IMAGE_LOADING_TYPE, PT_RADIOBUTTON, NULL, PG_AUTOAPPLY },
	{ "cookies_always_accept", CONF_PERSISTENT_COOKIES_BEHAVIOR, PT_RADIOBUTTON, cookies_sg, PG_AUTOAPPLY },
	{ "proxy_none_radiobutton", CONF_NETWORK_PROXY_MODE, PT_RADIOBUTTON, proxy_sg, PG_PROXY },
	{ "socksv4_radiobutton", CONF_NETWORK_SOCKS_PROXY_VERSION, PT_RADIOBUTTON, NULL, PG_PROXY },

	{ "tabbed_shorten_spin", CONF_TABS_TABBED_SHORTEN_CHARS, PT_SPINBUTTON, NULL, PG_AUTOAPPLY },
	{ "tabbed_shorten_spin2", CONF_TABS_TABBED_SHORTEN_POINTS, PT_SPINBUTTON, NULL, PG_AUTOAPPLY },
	{ "mouse_wheel_spin_button", CONF_MOUSE_WHEEL_STEP_SIZE, PT_SPINBUTTON, NULL, PG_AUTOAPPLY },
	{ "http_proxy_port_spin", CONF_NETWORK_HTTP_PROXY_PORT, PT_SPINBUTTON, NULL, PG_PROXY },
	{ "ssl_proxy_port_spin", CONF_NETWORK_SSL_PROXY_PORT, PT_SPINBUTTON, NULL, PG_PROXY },
	{ "ftp_proxy_port_spin", CONF_NETWORK_FTP_PROXY_PORT, PT_SPINBUTTON, NULL, PG_PROXY },
	{ "socks_proxy_port_spin", CONF_NETWORK_SOCKS_PROXY_PORT, PT_SPINBUTTON, NULL, PG_PROXY },
	{ "disk_cache_entry", CONF_NETWORK_DISK_CACHE, PT_SPINBUTTON, NULL, PG_AUTOAPPLY },
	{ "mem_cache_entry", CONF_NETWORK_MEMORY_CACHE, PT_SPINBUTTON, NULL, PG_AUTOAPPLY },
	{ "history_expire_spin", CONF_HISTORY_EXPIRE, PT_SPINBUTTON, NULL, PG_AUTOAPPLY },

	{ "background_color", CONF_RENDERING_BG_COLOR, PT_COLOR, NULL, PG_AUTOAPPLY },
	{ "text_color", CONF_RENDERING_TEXT_COLOR, PT_COLOR, NULL, PG_AUTOAPPLY },
	{ "unvisited_link_color", CONF_RENDERING_UNVISITED_LINKS, PT_COLOR, NULL, PG_AUTOAPPLY },
	{ "visited_link_color", CONF_RENDERING_VISITED_LINKS, PT_COLOR, NULL, PG_AUTOAPPLY },
	{ "tabbed_loading_color", CONF_TABS_TABBED_LOADING_COLOR, PT_COLOR, NULL, PG_AUTOAPPLY },
	{ "tabbed_new_color", CONF_TABS_TABBED_NEW_COLOR, PT_COLOR, NULL, PG_AUTOAPPLY },

	{ "tabbed_position_optionmenu", CONF_TABS_TABBED_POSITION, PT_OPTIONMENU, NULL, PG_AUTOAPPLY },
	{ "toolbar_style_optionmenu", CONF_TOOLBAR_STYLE, PT_OPTIONMENU, NULL, PG_AUTOAPPLY },
	{ "default_font_optionmenu", CONF_RENDERING_DEFAULT_FONT, PT_OPTIONMENU, NULL, PG_AUTOAPPLY },
	{ "middle_button_optionmenu", CONF_MOUSE_MMB_ACTION, PT_OPTIONMENU, NULL, PG_AUTOAPPLY },
	{ "right_button_optionmenu", CONF_MOUSE_RMB_ACTION, PT_OPTIONMENU, NULL, PG_AUTOAPPLY },
	{ "cache_compare_optionmenu", CONF_NETWORK_CACHE_COMPARE, PT_OPTIONMENU, NULL, PG_AUTOAPPLY },
	{ "autodetect_charset_optionmenu", CONF_LANGUAGE_AUTODETECT_CHARSET, PT_OPTIONMENU, NULL, PG_AUTOAPPLY },
	{ "http_version_optionmenu", CONF_NETWORK_HTTP_VERSION, PT_OPTIONMENU, NULL, PG_AUTOAPPLY },

	{ "startpage_entry", CONF_GENERAL_HOMEPAGE, PT_ENTRY, NULL, PG_AUTOAPPLY },
	{ "download_dir_entry", CONF_DOWNLOADING_DIR, PT_ENTRY, NULL, PG_AUTOAPPLY },
	{ "download_command_entry", CONF_DOWNLOADING_EXTERNAL_COMMAND, PT_ENTRY, NULL, PG_AUTOAPPLY },
	{ "spinner_dir_entry", CONF_TOOLBAR_SPINNER_DIR, PT_ENTRY, NULL, PG_AUTOAPPLY },
	{ "theme_dir_entry", CONF_TOOLBAR_THEME_DIR, PT_ENTRY, NULL, PG_AUTOAPPLY },
	{ "external_source_viewer_entry", CONF_PROGRAMS_EXTERNAL_SOURCE_VIEWER, PT_ENTRY, NULL, PG_AUTOAPPLY },
	{ "legacy_mailer_entry", CONF_PROGRAMS_MAILER, PT_ENTRY, NULL, PG_AUTOAPPLY },
	{ "proxy_auto_url_entry", CONF_NETWORK_PROXY_AUTO_URL, PT_ENTRY, NULL, PG_PROXY },
	{ "http_proxy_entry", CONF_NETWORK_HTTP_PROXY, PT_ENTRY, NULL, PG_PROXY },
	{ "ssl_proxy_entry", CONF_NETWORK_SSL_PROXY, PT_ENTRY, NULL, PG_PROXY },
	{ "ftp_proxy_entry", CONF_NETWORK_FTP_PROXY, PT_ENTRY, NULL, PG_PROXY },
	{ "socks_proxy_entry", CONF_NETWORK_SOCKS_PROXY, PT_ENTRY, NULL, PG_PROXY },
	{ "no_proxies_for_entry", CONF_NETWORK_NO_PROXIES_FOR, PT_ENTRY, NULL, PG_PROXY },
	{ "user_agent_entry", CONF_NETWORK_USER_AGENT, PT_ENTRY, NULL, PG_AUTOAPPLY },
	{ "default_charset_entry", CONF_LANGUAGE_DEFAULT_CHARSET, PT_ENTRY, NULL, PG_AUTOAPPLY },

	{ NULL, NULL }
};

static void
prefs_lookup_widgets (GladeXML *gxml)
{
        gint i;
        GtkWidget *value;
        WidgetLookup lookup[] =
        {
		{ "notebook", &(pd->notebook) },
		{ "sidebar",  &(pd->sidebar) },
		{ "themes_page", &(pd->themes_page) },
		{ "custom_css_clist", &(pd->custom_css_clist) },
		{ "custom_css_remove_button", &(pd->custom_css_remove_button) },
		{ "custom_css_default_check", &(pd->custom_css_default_check) },
		{ "serif_font_fpicker", &(pd->serif_font_fpicker) },
		{ "sansserif_font_fpicker", &(pd->sansserif_font_fpicker) },
		{ "cursive_font_fpicker", &(pd->cursive_font_fpicker) },
		{ "fantasy_font_fpicker", &(pd->fantasy_font_fpicker) },
		{ "monospace_font_fpicker", &(pd->monospace_font_fpicker) },
		{ "minimum_font_size_spin", &(pd->minimum_font_size_spin) },
		{ "default_charset_combo", &(pd->default_charset_combo) },
		{ "lang_encoding", &(pd->lang_encoding_optionmenu) },
		{ "theme_dir_entry", &(pd->theme_dir_entry) },
		{ "theme_iconlist", &(pd->theme_iconlist) },
		{ "mime_action_frame", &(pd->mime_action_frame) },
		{ "startpage_entry",  &(pd->startpage_entry) },

		{ "lang_list", &(pd->lang_list) },
		{ "lang_remove_button", &(pd->lang_remove_button) },
		{ "lang_up_button", &(pd->lang_up_button) },
		{ "lang_down_button", &(pd->lang_down_button) },

		{ "spinner_dir_entry", &(pd->spinner_dir_entry) },
		{ "spinner_iconlist", &(pd->spinner_iconlist) },

		/* mime types */
		{ "mime_list", &(pd->mime_list) },
		{ "mime_type_label", &(pd->mime_type_label) },
		{ "mime_type_entry", &(pd->mime_type_entry) },
		{ "mime_save_to_disk", &(pd->mime_save_to_disk)	},
		{ "mime_helper_frame", &(pd->mime_helper_frame)	},
		{ "mime_helper_entry", &(pd->mime_helper_entry)	},
		{ "mime_always_use", &(pd->mime_always_use) },
		{ "mime_terminal", &(pd->mime_terminal) },
		{ "mime_url_helper", &(pd->mime_url_helper) },
		{ "mime_delete_button",	&(pd->mime_delete_button) },

                /* toolbars editor */
		{ "available_clist", &(pd->available_clist) },
		{ "current_clist", &(pd->current_clist) },
		{ "edit_toolbars_add_button", &(pd->add_button) },
		{ "edit_toolbars_remove_button", &(pd->remove_button) },
		{ "edit_toolbars_up_button", &(pd->up_button) },
		{ "edit_toolbars_down_button", &(pd->down_button) },

		{ "lang_entry", &(pd->lang_entry) },
		{ "proxy_auto_url_entry", &(pd->proxy_auto_url_entry) },
		{ "proxy_auto_reload_button", &(pd->proxy_auto_reload_button) },

		/* fonts */
		{ "serif_font_fpicker",  &(pd->serif_font_picker) },
		{ "sansserif_font_fpicker",  &(pd->sansserif_font_picker) },
		{ "cursive_font_fpicker",  &(pd->cursive_font_picker) },
		{ "fantasy_font_fpicker",  &(pd->fantasy_font_picker) },
		{ "monospace_font_fpicker",  &(pd->monospace_font_picker) },
		{ "minimum_font_size_spin",  &(pd->minsize_spinbutton) },

		{ NULL, NULL }
	};

        g_return_if_fail (gxml != NULL);

	lookup_widgets (gxml, lookup);

	for (i = 0; lookup_table[i].widgetname != NULL; i++);
	pd->widgets = (GtkWidget **) g_new0 (GtkWidget *, i);

        /* lookup all of the widgets */
        for (i = 0; lookup_table[i].widgetname != NULL; i++)
        {
		value = glade_xml_get_widget (gxml, lookup_table[i].widgetname);
		pd->widgets[i] = value;
        }
}

static void
prefs_disconnect_signals (void)
{
	int i;
	GSList *list;

	for (i = 0; lookup_table[i].widgetname != NULL; i++)
	{
		switch (lookup_table[i].type)
		{
		case PT_TOGGLEBUTTON:
			gtk_signal_disconnect_by_func 
				(GTK_OBJECT (pd->widgets[i]), 
				 (prefs_togglebutton_clicked_cb), 
				 (gpointer)&lookup_table[i]);
			break;
		case PT_RADIOBUTTON:
			list = gtk_radio_button_group 
				(GTK_RADIO_BUTTON(pd->widgets[i]));
			for (; list != NULL; list = list->next)
			{
				gtk_signal_disconnect_by_func 
					(GTK_OBJECT (list->data), 
					 (prefs_radiobutton_clicked_cb), 
					 (gpointer)&lookup_table[i]);
			}
			break;
		case PT_SPINBUTTON:
			gtk_signal_disconnect_by_func 
				(GTK_OBJECT (pd->widgets[i]),
				 (prefs_spinbutton_changed_cb), 
				 (gpointer)&lookup_table[i]);
			break;
		case PT_COLOR:
			gtk_signal_disconnect_by_func 
				(GTK_OBJECT (pd->widgets[i]),
				 prefs_color_changed_cb, 
				 NULL);
			break;
		case PT_OPTIONMENU:
			gtk_signal_disconnect_by_func
				(GTK_OBJECT (GTK_OPTION_MENU(pd->widgets[i])->menu), 
				 prefs_optionmenu_selected_cb, 
				 (gpointer)&lookup_table[i]);
			break;
		case PT_ENTRY:
			gtk_signal_disconnect_by_func 
				(GTK_OBJECT (pd->widgets[i]),
				 (prefs_entry_changed_cb), 
				 (gpointer)&lookup_table[i]);
			break;
		}
	}
}

static void
prefs_connect_signals ()
{
	int i;
	GSList *list;

	for (i = 0; lookup_table[i].widgetname != NULL; i++)
	{
		switch (lookup_table[i].type)
		{
		case PT_TOGGLEBUTTON:
			gtk_signal_connect (GTK_OBJECT (pd->widgets[i]), "clicked", 
					    (prefs_togglebutton_clicked_cb), 
					    (gpointer)&lookup_table[i]);
			break;
		case PT_RADIOBUTTON:
			list = gtk_radio_button_group 
				(GTK_RADIO_BUTTON(pd->widgets[i]));
			for (; list != NULL; list = list->next)
			{

				gtk_signal_connect 
					(GTK_OBJECT (list->data), "clicked", 
					 (prefs_radiobutton_clicked_cb), 
					 (gpointer)&lookup_table[i]);
			}
			break;
		case PT_SPINBUTTON:
			gtk_signal_connect (GTK_OBJECT (pd->widgets[i]), "changed", 
					    (prefs_spinbutton_changed_cb), 
					    (gpointer)&lookup_table[i]);
			break;
		case PT_COLOR:
			gtk_object_set_data (GTK_OBJECT(pd->widgets[i]), 
					     "prefname", (gpointer)&lookup_table[i]);
			gtk_signal_connect (GTK_OBJECT (pd->widgets[i]), "color_set", 
					    prefs_color_changed_cb, 
					    NULL);
			break;
		case PT_OPTIONMENU:
			gtk_object_set_data (GTK_OBJECT(GTK_OPTION_MENU(pd->widgets[i])->menu), 
					     "optionmenu", pd->widgets[i]);  
			gtk_signal_connect (GTK_OBJECT (GTK_OPTION_MENU(pd->widgets[i])->menu), 
					    "selection_done", 
					    prefs_optionmenu_selected_cb, 
					    (gpointer)&lookup_table[i]);
			break;
		case PT_ENTRY:
			gtk_signal_connect (GTK_OBJECT (pd->widgets[i]), "changed", 
					    (prefs_entry_changed_cb), 
					    (gpointer)&lookup_table[i]);
			break;
		}
	}
}

static void
prefs_add_to_queue (GtkWidget *widget, PrefInfo *pi)
{
	PrefQueueItem *item; 
	GList *list = prefs_queue;

	for (; list != NULL; list = list->next)
	{
		PrefInfo *oldpi = (PrefInfo*)list->data;

		if (strcmp(pi->prefname, oldpi->prefname) == 0)
		{
			return;
		}
	}

	item = g_new0 (PrefQueueItem, 1);
	item->widget = widget;
	item->info = pi;
	prefs_queue = g_list_append (prefs_queue, item);
}

static void
prefs_apply_queue (PrefGroup group)
{
	GList *list = prefs_queue;

	while (list != NULL)
	{
		PrefQueueItem *item = (PrefQueueItem *)list->data;

		if ( group == PG_ALL || group == item->info->group)
		{
			switch (item->info->type)
			{
			case PT_TOGGLEBUTTON:
				pu_set_config_from_togglebutton 
					(item->widget, 
					 item->info->prefname);
				break;
			case PT_RADIOBUTTON:
				pu_set_config_from_radiobuttongroup 
					(item->widget, 
					 item->info->prefname);
				break;
			case PT_SPINBUTTON:
				pu_set_config_from_spin_button 
					(item->widget, 
					 item->info->prefname);
				break;
			case PT_COLOR:
				pu_set_config_from_color 
					(item->widget, 
					 item->info->prefname);
				break;
			case PT_OPTIONMENU:
				pu_set_config_from_optionmenu 
					(item->widget, 
					 item->info->prefname);
				break;
			case PT_ENTRY:
				pu_set_config_from_editable 
					(item->widget,  
					 item->info->prefname);
				break;
			}

			list = g_list_remove (list, list->data);
		}
		else
		{
			list = list->next;
		}
	}

	prefs_queue = g_list_first (list);
}

static void 
prefs_togglebutton_clicked_cb (GtkWidget *widget, PrefInfo *pi)
{
	if (pi->group == PG_AUTOAPPLY)
	{
		pu_set_config_from_togglebutton (widget, pi->prefname);
	}
	else
	{
		prefs_add_to_queue (widget, pi);
	}
	prefs_set_group_sensitivity (widget, pi->type, pi->sg);
}

static void 
prefs_radiobutton_clicked_cb (GtkWidget *widget,  PrefInfo *pi)
{
	if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(widget)))
	{
		return;
	}

	if (pi->group == PG_AUTOAPPLY)
	{
		pu_set_config_from_radiobuttongroup (widget, pi->prefname);
	}
	else
	{
		prefs_add_to_queue (widget, pi);
	}

	prefs_set_group_sensitivity (widget, pi->type, pi->sg);
}

static void 
prefs_spinbutton_changed_cb (GtkWidget *widget,  PrefInfo *pi)
{
	if (pi->group == PG_AUTOAPPLY)
	{
		spin_item.widget = widget;
		spin_item.info = pi;

		/* destroy any existing timer */
		if (spin_timer != NULL)
		{
			g_timer_destroy (spin_timer);
		}

		/* start the new one */
		spin_timer = g_timer_new();
		g_timer_start (spin_timer);
		g_timeout_add (50, (GSourceFunc) prefs_spinbutton_timeout_cb,
			       &spin_item);
	}
	else
	{
		prefs_add_to_queue (widget, pi);
	}
}

gint 
prefs_spinbutton_timeout_cb (PrefQueueItem *item)
{
        /* timer still valid? */
        if (spin_timer == NULL)
        {
                return FALSE;
        }

        /* okay, we're ready to set */
        if (g_timer_elapsed (spin_timer, NULL) >= SPIN_DELAY)
        {
                /* kill off the timer */
                g_timer_destroy (spin_timer);
                spin_timer = NULL;

                /* set */
                pu_set_config_from_spin_button (item->widget, 
						item->info->prefname);

                /* done now */
                return FALSE;
        }

        /* call me again */
        return TRUE;
}

static void 
prefs_color_changed_cb (GtkWidget *widget, char *dummy)
{
	PrefInfo *pi;

	pi = gtk_object_get_data (GTK_OBJECT(widget), "prefname");

	if (pi->group == PG_AUTOAPPLY)
	{
		pu_set_config_from_color (widget,  pi->prefname);
	}
	else
	{
		prefs_add_to_queue (widget, pi);
	}
}

static void 
prefs_entry_changed_cb (GtkWidget *widget,  PrefInfo *pi)
{
	if (pi->group == PG_AUTOAPPLY)
	{
		pu_set_config_from_editable (widget,  pi->prefname);
	}
	else
	{
		prefs_add_to_queue (widget, pi);
	}
}

static void 
prefs_optionmenu_selected_cb (GtkWidget *widget,  PrefInfo *pi)
{
	GtkWidget *optionmenu;

	optionmenu = gtk_object_get_data (GTK_OBJECT(widget), "optionmenu");

	if (pi->group == PG_AUTOAPPLY)
	{
		pu_set_config_from_optionmenu (optionmenu, pi->prefname);
	}
	else
	{
		prefs_add_to_queue (optionmenu, pi);
	}
}

static void
prefs_set_dialog_to_config (void)
{
	int i;

	for (i = 0; lookup_table[i].widgetname != NULL; i++)
	{
		prefs_set_widget_from_config (pd->widgets[i], 
					      lookup_table[i].prefname,
					      lookup_table[i].type,
					      lookup_table[i].sg);
	}

	/* initialize css list */
	prefs_custom_css_set_list_from_config ();
	pu_set_list_from_config (pd->lang_list,
				 CONF_RENDERING_LANGUAGE); 

	/* initialise toolbar editor */
        toolbar_editor_init ();

	/* initialise theme selector */
        prefs_theme_selector_init ();

        /* process mime types */
        prefs_mime_set_dialog_to_config ();
}

static void
prefs_set_group_sensitivity (GtkWidget *widget, PrefType type, char **sg)
{
	gboolean value;
	int index;
	GtkWidget *swidget;
	int i = 0, k = 0;

	if (!sg) return;

	switch (type)
	{
	case PT_TOGGLEBUTTON:
		value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(widget));
		for (i=0; sg[i] != NULL; i++)
		{
			swidget = glade_lookup_widget (widget, sg[i]);
			gtk_widget_set_sensitive (swidget, value);
		}
		break;	
	case PT_RADIOBUTTON:
		index = misc_gui_gtk_radio_button_get (
						GTK_RADIO_BUTTON (widget));
		do
		{
			value = (index==k);
			for (; sg[i] != NULL; i++)
			{
				gchar *name;
				gboolean invert = FALSE;

				if (sg[i][0] == '!')
				{
					invert = TRUE;
					name = sg[i] + 1;
				}
				else name = sg[i];

				swidget = glade_lookup_widget (widget, name);
				gtk_widget_set_sensitive (swidget,
							  value ^ invert);
			}
			i++;
			k++;
		}
		while (!sg[i] || strcmp (END_STRING, sg[i]));
		break;

	case PT_SPINBUTTON:
	case PT_COLOR:
	case PT_OPTIONMENU:
	case PT_ENTRY:
		break;
	}
}

static void
prefs_set_widget_from_config (GtkWidget *widget, char *pref, PrefType type, char **sg)
{
	switch (type)
	{
	case PT_TOGGLEBUTTON:
		pu_set_togglebutton_from_config (widget, pref);
		prefs_set_group_sensitivity (widget, type, sg);
		break;
	case PT_RADIOBUTTON:
		pu_set_radiobuttongroup_from_config (widget, pref);
		prefs_set_group_sensitivity (widget, type, sg);
		break;
	case PT_SPINBUTTON:
		pu_set_spin_button_from_config (widget, pref);
		break;
	case PT_COLOR:
		pu_set_color_from_config (widget, pref);
		break;
	case PT_ENTRY:
		pu_set_editable_from_config (widget, pref);
		break;
	case PT_OPTIONMENU:
		pu_set_optionmenu_from_config (widget, pref);
		break;
	default:
		g_error("Invalid preference type");
	}
}

/**
 * prefs_open: open the prefs dialog
 */
void
prefs_open (GaleonWindow *window)
{
	/* dialog available? */
	if (pd == NULL)
	{
		/* nope, build it */
		prefs_build_dialog ();
	}
	else if (GTK_WIDGET_VISIBLE (pd->dialog))
	{
		/* set the parent window */
		pd->parent_window = window;

		/* already visible, raise to top */
		gdk_window_raise (GTK_WIDGET (
			pd->dialog)->window);
		return;
	}
	
	/* get the saved geometry */
	state_load_window_state (GTK_WIDGET (pd->dialog), 
				 "prefs_dialog");

	/* initialize fonts data structure */
	prefs_fonts_language_init ();

	spinner_fill_iconlist (GNOME_ICON_LIST (pd->spinner_iconlist));

	/* set values in dialog to stored values in prefs */
	prefs_set_dialog_to_config ();

	/* connect signals */
	prefs_connect_signals ();

	/* set the parent window */
	pd->parent_window = window;

	/* display it */
	gtk_widget_show (GTK_WIDGET (pd->dialog));
	dialog_set_parent (GTK_WIDGET (pd->dialog), 
			   window->wmain);
}

/**
 * prefs_show_page: open preferences at a particular page
 */
void
prefs_show_page (GaleonWindow *window, PreferencesPageId id)
{
	GtkWidget *panel;

	/* make sure the window is open */
	prefs_open (window);

	/* switch to appropiate panel */
	panel = GTK_WIDGET (gtk_object_get_data 
			   (GTK_OBJECT (page_buttons[id]), "panel"));
	if (current_panel != NULL && current_panel != panel)
	{
		gtk_widget_hide (current_panel);
	}
	if (current_panel != panel)
	{
		gtk_widget_show (GTK_WIDGET (current_panel = panel));
	}

	/* click appropiate button in order to load page */
	gtk_button_clicked (GTK_BUTTON (page_buttons[id]));
}

/**
 * prefs_close: close the preferences dialog window (hides it), after
 * saving all the prefs that need to be saved
 */
static void
prefs_close (void)
{
	/* apply queued prefs */
	prefs_apply_queue (PG_ALL);

        /* save the toolbar state */
        toolbar_editor_save ();

	/* save css list */
	prefs_custom_css_set_config_from_list ();

	/* save the mime state */
	mime_db_save ();

	state_save_window_state (GTK_WIDGET (pd->dialog)->window, 
				 "prefs_dialog");

	/* suggest a sync */
	eel_gconf_suggest_sync ();      

	/* This unselect is necessary because for some reason, gtk is
	 * unselecting the row without sending the unselect signal when the
	 * window is hidden.  -Daniel */
	/* Do the same with the language list -Yanko */
	gtk_clist_unselect_all (GTK_CLIST (pd->mime_list));
	gtk_list_unselect_all (GTK_LIST (pd->lang_list));

	pd->parent_window = NULL;

	gtk_widget_hide (GTK_WIDGET (pd->dialog));

	prefs_disconnect_signals ();
}

/**
 * prefs_load: load Galeon preferences needed at startup
 */
void
prefs_load (void)
{
	mime_db_load ();

	/* Reflect gconf proxy prefs to mozilla, and explictly
	   load autoconf_url if appropriate */

	mozilla_prefs_set_proxy ();
}

/**
 * prefs_help: show the prefs Galeon manual page   
 */
static void
prefs_help (void)
{
	GnomeHelpMenuEntry pref_help = { "galeon-manual", "preferences.html" };
	gchar *file = NULL;

	/* Try to find the preferences help page.  If we fail, use ghelp
	   to load the index, instead */
	file = gnome_help_file_find_file("galeon-manual", "preferences.html");

	if (file)
	{
	        gnome_help_display (NULL, &pref_help);
	}
	else
	{
		gnome_help_goto (NULL, "ghelp:galeon-manual");
	}

	g_free(file);
}

static void
prefs_build_dialog (void)
{
	GList *li;
	GtkWidget *menu;

	/* clear up any existing display, makes the app seem more responsive */
	while (gtk_events_pending ())
	{
		gtk_main_iteration ();
	}

	/* not been generated yet... */
	pd = g_new0 (PreferencesDialog, 1);

	/* build the widgets */
	pd->gxml = glade_widget_new ("preferences.glade", 
				     "preferences_dialog", 
				     &(pd->dialog), 
				     pd);
		
	/* lookup all the widgets */
	prefs_lookup_widgets (pd->gxml);

	/* build the tree */
	prefs_build_sidebar ();
	
	/* get the charsets from mozilla if they are still empty */
	if (sorted_charset_titles == NULL)
	{
		mozilla_get_charsets (&charsets, &sorted_charset_titles);
	}
	
	/* initialize the default charset combo */
	gtk_combo_set_popdown_strings (GTK_COMBO (pd->default_charset_combo), 
				       sorted_charset_titles);

	/* attach the font language optionmenu event */
	menu = GTK_OPTION_MENU(pd->lang_encoding_optionmenu)->menu;
	li = GTK_MENU_SHELL(menu)->children;
	for (; li != NULL; li = li->next)
	{
		gtk_signal_connect 
			(GTK_OBJECT (li->data), "activate",
			 GTK_SIGNAL_FUNC(prefs_encoding_activated_cb), pd);
	} 

	/* connect fonts signals */
	gtk_signal_connect 
		(GTK_OBJECT (pd->serif_font_picker), "font_set",
		 GTK_SIGNAL_FUNC(prefs_serif_font_set_cb), pd);
	gtk_signal_connect 
		(GTK_OBJECT (pd->sansserif_font_picker), "font_set",
		 GTK_SIGNAL_FUNC(prefs_sansserif_font_set_cb), pd);
	gtk_signal_connect 
		(GTK_OBJECT (pd->cursive_font_picker), "font_set",
		 GTK_SIGNAL_FUNC(prefs_cursive_font_set_cb), pd);
	gtk_signal_connect 
		(GTK_OBJECT (pd->fantasy_font_picker), "font_set",
		 GTK_SIGNAL_FUNC(prefs_fantasy_font_set_cb), pd);
	gtk_signal_connect 
		(GTK_OBJECT (pd->monospace_font_picker), "font_set",
		 GTK_SIGNAL_FUNC(prefs_monospace_font_set_cb), pd);
	gtk_signal_connect 
		(GTK_OBJECT (pd->minsize_spinbutton), "changed", 
		 prefs_minsize_font_changed_cb, 
		 pd);
}

/**
 * prefs_destroy: free the prefs dialog (called at exit)
 */
void
prefs_destroy (void)
{
	/* if the prefs dialog is open, close it (this will also sync
	 * changes) */
	if (pd && GTK_WIDGET_VISIBLE (pd->dialog))
	{
		prefs_close ();
	}

	if (pd)
	{
		gtk_object_unref (GTK_OBJECT (pd->gxml));
		g_free (pd->widgets);
	}
}

/**
 * preferences_build_sidebar: build the sidebar of the preferences dialog
 */
static void
prefs_build_sidebar (void)
{
	PreferencesPanelButton browsing_icons[] =
	{
		{ _("General"),         PREFS_PAGE_ID_BROWSING_GENERAL,
					SHARE_DIR "/general.png" },
		{ _("Bookmarks"),       PREFS_PAGE_ID_BROWSING_BOOKMARKS,
					SHARE_DIR "/bookmarks.png" },
		{ _("History"),         PREFS_PAGE_ID_BROWSING_HISTORY,
					SHARE_DIR "/history.png" },
		{ NULL } /* terminator */
        };
	PreferencesPanelButton user_interface_icons[] =
	{
		{ _("Tabs"),            PREFS_PAGE_ID_USER_INTERFACE_TABS,
					SHARE_DIR "/tabs.png" },
		{ _("Windows"),         PREFS_PAGE_ID_USER_INTERFACE_WINDOWS,
					SHARE_DIR "/windows.png" },
		{ _("Toolbars"),        PREFS_PAGE_ID_USER_INTERFACE_TOOLBARS,
					SHARE_DIR "/toolbars.png" },
		{ _("Mouse"),           PREFS_PAGE_ID_USER_INTERFACE_MOUSE,
					SHARE_DIR "/mouse.png" },
		{ NULL } /* terminator */
        };
	PreferencesPanelButton handlers_icons[] =
	{
		{ _("Downloading"),     PREFS_PAGE_ID_HANDLERS_DOWNLOADING,
					SHARE_DIR "/download.png" },
		{ _("Programs"),        PREFS_PAGE_ID_HANDLERS_PROGRAMS,
					SHARE_DIR "/programs.png" },
		{ _("MIME Types"),      PREFS_PAGE_ID_HANDLERS_MIME_TYPES, 
					SHARE_DIR "/mime.png" },
		{ NULL } /* terminator */
	};
	PreferencesPanelButton rendering_icons[] =
	{
		{ _("Fonts/Colors"),    PREFS_PAGE_ID_RENDERING_FONTS_COLORS,
					SHARE_DIR "/fonts.png" },
		{ _("Languages"),       PREFS_PAGE_ID_RENDERING_LANGUAGES,
					SHARE_DIR "/babelfish.png" },
		{ NULL } /* terminator */
	};
	PreferencesPanelButton advanced_icons[] =
	{
		{ _("Network"),         PREFS_PAGE_ID_ADVANCED_NETWORK,
					SHARE_DIR "/network.png" },
		{ _("Filtering"),       PREFS_PAGE_ID_ADVANCED_FILTERING,
					SHARE_DIR "/stop.png" },
		{ _("Persistent Data"), PREFS_PAGE_ID_ADVANCED_PERSISTENT_DATA,
					SHARE_DIR "/harddrive.png" },
		{ _("Security"),        PREFS_PAGE_ID_ADVANCED_SECURITY,
					SHARE_DIR "/security.png" },
		{ NULL } /* terminator */
	};
	PreferencesSidebarItem sidebar[] = 
	{
		{  _("Browsing"),        browsing_icons       },
		{  _("User Interface"),  user_interface_icons },
		{  _("Handlers"),        handlers_icons       },
		{  _("Rendering"),       rendering_icons      },
		{  _("Advanced"),        advanced_icons       },
		{ NULL } /* terminator */
	};

	PreferencesSidebarItem *item;

	g_return_if_fail (pd != NULL);
	g_return_if_fail (pd->sidebar != NULL);

	for (item = sidebar; NULL != item->name; ++item)
	{
		GtkWidget *button, *panel_button, *panel;
		PreferencesPanelButton const *icon;
		GSList *button_group = NULL;

		panel_button = gtk_button_new_with_label (item->name);
		panel = gtk_vbox_new (FALSE, 0);

		gtk_object_set_data (GTK_OBJECT (panel_button), "panel", 
				     panel);
		gtk_signal_connect (GTK_OBJECT (panel_button), "clicked",
				    prefs_sidebar_select_panel_cb, pd);
		gtk_box_pack_start (GTK_BOX (pd->sidebar), panel_button,
				    FALSE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (pd->sidebar), panel,
				    TRUE, TRUE, 2);

		if (sidebar == item) 
		{
			gtk_button_clicked (GTK_BUTTON (panel_button));
		}
		gtk_widget_show (panel_button);

		/* invisible dummy button -> nothing selected */
		button = gtk_radio_button_new (button_group);
		button_group = gtk_radio_button_group 
			(GTK_RADIO_BUTTON (button));
		gtk_box_pack_start (GTK_BOX (panel), button, FALSE, TRUE, 0);
		gtk_button_clicked (GTK_BUTTON (button));

		/* fill the panel */
		for (icon = item->icons; NULL != icon->name; ++icon)
		{
			GtkWidget *widget, *vbox;
			PixmapData *icon_data;
			gint id = icon->page;

			page_buttons[id] = gtk_radio_button_new (button_group);
			gtk_button_set_relief (GTK_BUTTON (page_buttons[id]),
					       GTK_RELIEF_NONE);
			gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON 
					    (page_buttons[id]), FALSE);
			button_group = gtk_radio_button_group
						(GTK_RADIO_BUTTON (
						page_buttons[id]));
			gtk_object_set_data (GTK_OBJECT (page_buttons[id]), 
					    "page_id", GINT_TO_POINTER (id));
			gtk_object_set_data (GTK_OBJECT (page_buttons[id]), 
					    "panel", panel);
			gtk_signal_connect (GTK_OBJECT (page_buttons[id]), 
					    "clicked",
					    prefs_sidebar_select_page_cb,
					    pd);
			gtk_box_pack_start (GTK_BOX (panel), page_buttons[id],
					    FALSE, TRUE, 0);
			gtk_widget_show (page_buttons[id]);


			vbox = gtk_vbox_new (FALSE, 0);
			gtk_container_add (GTK_CONTAINER (page_buttons[id]), 
					    vbox);
			gtk_widget_show (vbox);


			icon_data = misc_gui_pixmap_data_new_from_file (
							icon->icon, FALSE);

			if (NULL != icon_data)
			{
				widget = gtk_pixmap_new (icon_data->pixmap,
							 icon_data->mask);
				gtk_box_pack_start (GTK_BOX (vbox), widget,
						    FALSE, FALSE, 0);
				gtk_widget_show (widget);

				g_free (icon_data);
			}

			widget = gtk_label_new (icon->name);
			gtk_misc_set_alignment (GTK_MISC (widget), .5, .5);
			gtk_label_set_line_wrap (GTK_LABEL (widget), FALSE);
			gtk_box_pack_start (GTK_BOX (vbox), widget,
					    FALSE, TRUE, 0);
			gtk_widget_show (widget);
		}

		gtk_object_set_data (GTK_OBJECT (panel_button), 
				     "group", button_group);
	}
}

/**
 * preferences_sidebar_select_panel: show the selected (clicked) panel
 * or hide it when active already
 */
static void 
prefs_sidebar_select_panel_cb (GtkWidget *button)
{
	GtkWidget *panel = GTK_WIDGET
		(gtk_object_get_data (GTK_OBJECT (button), "panel"));
	GSList *group;
	int page_id = 0;

	if (NULL != current_panel)
	{
		gtk_widget_hide (current_panel);
	}

	if (current_panel != panel)
	{
		gtk_widget_show (current_panel = panel);

		group = (GSList *)(gtk_object_get_data 
				   (GTK_OBJECT (button), "group"));

		for (; group != NULL ; group = g_slist_next(group))
		{
			if (gtk_toggle_button_get_active 
			    (GTK_TOGGLE_BUTTON (group->data)))
			{
				page_id = GPOINTER_TO_INT 
					(gtk_object_get_data
					 (GTK_OBJECT (group->data), 
					  "page_id"));
			}
		}
	}
	else
		current_panel = NULL;

	gtk_notebook_set_page (GTK_NOTEBOOK (pd->notebook), page_id);
}

/**
 * prefs_sidebar_select_page: activate the selected (clicked) page
 */
static void prefs_sidebar_select_page_cb (GtkWidget *button)
{
	int page_id = GPOINTER_TO_INT
		(gtk_object_get_data (GTK_OBJECT (button), "page_id"));
	gtk_notebook_set_page (GTK_NOTEBOOK (pd->notebook), page_id);
}

static void
prefs_fonts_language_init (void)
{
	int i = pu_get_int_from_optionmenu (pd->lang_encoding_optionmenu);
	LanguageFontInfo fonts;

	mozilla_prefs_get_fonts (&fonts, i);
		
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->serif_font_fpicker),
		 fonts.serif);
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->sansserif_font_fpicker),
		 fonts.sansserif);
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->cursive_font_fpicker),
		 fonts.cursive);
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->fantasy_font_fpicker),
		 fonts.fantasy);
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->monospace_font_fpicker),
		 fonts.monospace);
	gtk_spin_button_set_value 
		(GTK_SPIN_BUTTON(pd->minimum_font_size_spin),
		 fonts.min_size);

	g_free (fonts.serif);
	g_free (fonts.sansserif);
	g_free (fonts.cursive);
	g_free (fonts.fantasy);
	g_free (fonts.monospace);
}

/**
 * prefs_theme_selector_init: Initialize theme selector
 */
static void
prefs_theme_selector_init (void)
{
	GnomeIconList *list = GNOME_ICON_LIST (pd->theme_iconlist);
	gchar *home_theme_dir, *home_nautilus_theme_dir;
	gchar *pref_theme_path = NULL;
	g_assert (list!=NULL);

	gnome_icon_list_freeze (list);

	/* clear the list out */
	free_theme_selector (list);
	gnome_icon_list_clear (list);

	/* fill list */
	gnome_icon_list_append (list, SHARE_DIR "/gnome_icons.png", 
				_("GNOME default"));
	gnome_icon_list_set_icon_data (list, 0, g_strdup (""));

	gnome_icon_list_append (list, SHARE_DIR "/nautilus_icons.png", 
				_("Nautilus theme"));
	gnome_icon_list_set_icon_data (list, 1, g_strdup ("NAUTILUS_THEME"));

	/* load all the galeon themes */
	theme_selector_fill_themelist_from_dir (list, SHARE_DIR "/themes");
	home_theme_dir =
		g_strconcat (g_get_home_dir (), "/.galeon/themes", NULL);
	theme_selector_fill_themelist_from_dir (list, home_theme_dir);
	g_free (home_theme_dir);

	/* load all the nautilus themes */
	theme_selector_fill_themelist_from_dir (list, GNOME_PIXMAPS_DIR 
						      "/nautilus");
	home_nautilus_theme_dir =
		g_strconcat (g_get_home_dir (), "/.nautilus/themes", NULL);
	theme_selector_fill_themelist_from_dir (list, home_nautilus_theme_dir);
	g_free (home_nautilus_theme_dir);

	/* select theme */
	pref_theme_path = eel_gconf_get_string (CONF_TOOLBAR_THEME_DIR);
	if (!pref_theme_path || strcmp (pref_theme_path, "") == 0)
	{
		gnome_icon_list_select_icon (list, 0);
	}
	else if (pref_theme_path && strcmp (pref_theme_path,
					    "NAUTILUS_THEME") == 0)
	{
		gnome_icon_list_select_icon (list, 1);
	}
	else if (pref_theme_path)
	{
		int i;

		/* look through all the icons for the one with the matching
		 * path */
		for (i = 2; i < list->icons; i++)
		{
			if (!strcmp (pref_theme_path,
				     gnome_icon_list_get_icon_data (list, i)))
			{
				gnome_icon_list_select_icon (list, i);
				break;
			}
		}
	}

	if (pref_theme_path) g_free (pref_theme_path);

	gnome_icon_list_thaw (list);
}

/**
 * theme_selector_fill_themelist_from_dir: Fill themelist with available themes
 * starting from base
 */
static void
theme_selector_fill_themelist_from_dir (GnomeIconList *theme_list, gchar *base)
{
	gchar *icon;
	gchar *pref_theme_path; 	
	GList *node;
	GList *list = NULL;
	GnomeVFSResult rc;

	pref_theme_path = eel_gconf_get_string (CONF_TOOLBAR_THEME_DIR);
	rc = gnome_vfs_directory_list_load
		(&list, base, (GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
			       GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE |
			       GNOME_VFS_FILE_INFO_FOLLOW_LINKS), NULL);

	for (node = list; node; node = node->next)
	{
		GnomeVFSFileInfo *info;
		gchar *path;

		info = node->data;
		if (info->name[0] == '.')
			continue;
		if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY)
			continue;

		path = g_concat_dir_and_file (base, info->name);

		/* Check to see if there is a Refresh.png */
		icon = g_concat_dir_and_file (path, "Forward.png");
		if (g_file_exists (icon))
		{
			gint index = gnome_icon_list_append (theme_list, icon,
							     g_basename (path));
			gnome_icon_list_set_icon_data (theme_list, index,
						       g_strdup (path));

			/* Select the icon configured in prefs */
			if (pref_theme_path && 
			    strcmp (pref_theme_path, path) == 0)
			{
				gnome_icon_list_select_icon (theme_list, 
							     index);
			}
		}

		g_free (path);
		g_free (icon);
	}

	gnome_vfs_file_info_list_free (list);
	g_free (pref_theme_path);
}

/**
 * free_themes_list: Free any existing themes list.
 */
static void
free_theme_selector (GnomeIconList *list)
{
	gint i;

	for (i = 0; i < list->icons; i++)
	{
		g_free (gnome_icon_list_get_icon_data (list, i));
	}
}

/**
 * prefs_custom_css_set_config_from_list: apply stylesheet changes
 */
static void
prefs_custom_css_set_config_from_list (void)
{
	GtkCList *clist = GTK_CLIST (pd->custom_css_clist);
	GSList *config = NULL, *defaults = NULL;
	gint i;

	for (i = 0; i < clist->rows; i++)
	{
		CSSManagerRow *item = gtk_clist_get_row_data (clist, i);
		config = g_slist_append (config, item->filename);
		if (item->is_default)
		{
			defaults = g_slist_append (defaults, item->filename);
		}
	}	

	eel_gconf_set_string_list (CONF_RENDERING_STYLESHEETS,
				   config);
	eel_gconf_set_string_list (CONF_RENDERING_DEFAULT_STYLESHEETS,
				   defaults);
	
	g_slist_free (config);
	g_slist_free (defaults);
}

/**
 * prefs_custom_css_set_list_from_config: load stylesheet configuration
 */
static void
prefs_custom_css_set_list_from_config (void)
{
	GtkCList *clist = GTK_CLIST (pd->custom_css_clist);
	GSList *config, *defaults, *l;
	gint i;

	config = eel_gconf_get_string_list
		(CONF_RENDERING_STYLESHEETS);
	defaults = eel_gconf_get_string_list
		(CONF_RENDERING_DEFAULT_STYLESHEETS);
	
	gtk_clist_freeze (clist);
	
	/* clean out what is still there */
	for (i = clist->rows - 1; i >= 0; i--)
	{
		CSSManagerRow *item = gtk_clist_get_row_data (clist, i);
		g_free (item->filename);
		g_free (item);
	}

	gtk_clist_clear (clist);

	gtk_widget_set_sensitive (pd->custom_css_remove_button, FALSE);
	gtk_widget_set_sensitive (pd->custom_css_default_check, FALSE);

	gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON (pd->custom_css_default_check), FALSE);
	
	/* load */
	for (l = config; l; l = g_slist_next (l))
	{
		CSSManagerRow *r = g_new0 (CSSManagerRow, 1);
		gchar *s = (gchar *) l->data;
		gchar *row[1];
		gint rownum;

		r->filename = g_strdup (s);
		r->is_default = (g_slist_find_custom (defaults, s,
				(GCompareFunc) strcmp) != NULL);
		
		row[0] = s;

		rownum = gtk_clist_append (clist, row);
		gtk_clist_set_row_data (clist, rownum, r);
	}

	gtk_clist_set_reorderable (clist, TRUE);
	gtk_clist_thaw (clist);
	
	/* cleanup */
	g_slist_foreach (config, (GFunc) g_free, NULL);
	g_slist_free (config);
	g_slist_foreach (defaults, (GFunc) g_free, NULL);
	g_slist_free (defaults);
}

gboolean
prefs_galeon_used_by_gnome_for_protocol (const char *protocol)
{
	gboolean uses_galeon;

	gchar *path = g_strconcat ("/Gnome/URL Handlers/", protocol,
				   "-show", NULL);
	gchar *handler = gnome_config_get_string (path);
	g_free (path);

	if (handler == NULL)
	{
		handler = gnome_config_get_string 
			("/Gnome/URL Handlers/default-show");
	}
	if (handler == NULL)
	{
		uses_galeon = FALSE;
	}
	else
	{
		uses_galeon = (strstr (handler, "galeon") != NULL);
		free (handler);
	}

	return uses_galeon;
}

/**
 * prefs_ok_button_clicked_cb: preferences dialog ok button clicked
 */
void
prefs_ok_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	prefs_close ();
}

void
prefs_proxy_apply_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	prefs_apply_queue (PG_PROXY);
}

/**
 * prefs_close_cb: dialog has been closed
 */
gboolean
prefs_close_cb (GnomeDialog *dialog, PreferencesDialog *pd)
{
	prefs_close ();	
	return TRUE;
}

void
prefs_help_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
        prefs_help ();
}

/**
 * prefs_clear_disk_cache_button_clicked_cb: "Clear disk cache" clicked
 */
void
prefs_clear_disk_cache_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	mozilla_clear_cache (STORE_ON_DISK);
}

/**
 * prefs_clear_memory_cache_button_clicked_cb: "Clear memory cache" clicked
 */
void
prefs_clear_memory_cache_button_clicked_cb (GtkButton *button, 
					    PreferencesDialog *pd)
{
	mozilla_clear_cache (STORE_IN_MEMORY);
}

/**
 * prefs_homepage_my_portal_button_clicked_cb: set homepage URL to "My Portal"
 */
void
prefs_homepage_my_portal_button_clicked_cb (GtkButton *button, 
					    PreferencesDialog *pd)
{
	gtk_entry_set_text (GTK_ENTRY (pd->startpage_entry), MYPORTAL_URL);
}


/**
 * prefs_homepage_current_button_clicked_cb: set homepage URL to current page
 */
void
prefs_homepage_current_button_clicked_cb (GtkButton *button, 
					  PreferencesDialog *pd)
{
	GaleonWindow *window;
	GaleonEmbed *embed;
	
	/* if the parent window is no longer open, use the first window we
	 * find instead */
	window = pd->parent_window;
	if (!window || window->magic != GALEON_WINDOW_MAGIC)
		window = all_windows->data;

	/* get active embed */
	embed = window->active_embed;
	return_if_not_embed (embed);

	/* use embed's location, or about:blank if no location is loaded */
	if (embed->location && embed->location[0] != '\0')
		gtk_entry_set_text (GTK_ENTRY (pd->startpage_entry),
				    embed->location);
	else gtk_entry_set_text (GTK_ENTRY (pd->startpage_entry),
				 "about:blank");
}

void
prefs_lang_add_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GtkWidget *entry = pd->lang_entry;
	GList *items = NULL;
	GtkWidget *list_item;
	char *entry_text;

	entry_text = gtk_editable_get_chars (GTK_EDITABLE (entry),0,-1);
	list_item = gtk_list_item_new_with_label (entry_text);

	items = g_list_append (items, list_item);
	gtk_list_append_items (list, items);
	gtk_widget_show (list_item);

	gtk_list_select_child (list, list_item);

	pu_set_config_from_list (pd->lang_list, CONF_RENDERING_LANGUAGE);
}

void
prefs_lang_list_selection_changed_cb (GtkList *list, PreferencesDialog *pd)
{
	gpointer first;
	gpointer last;
	gboolean remove_sensitive;
	gboolean up_sensitive;
	gboolean down_sensitive;

	first = g_list_first (list->children)->data;
	last = g_list_last (list->children)->data;

	remove_sensitive = (list->selection != NULL);
	up_sensitive = (remove_sensitive &&
			g_list_find (list->selection, first) == NULL);
	down_sensitive = (remove_sensitive &&
			  g_list_find (list->selection, last) == NULL);

	gtk_widget_set_sensitive (pd->lang_remove_button,	
			          remove_sensitive);
	gtk_widget_set_sensitive (pd->lang_up_button, 	up_sensitive);
	gtk_widget_set_sensitive (pd->lang_down_button,	down_sensitive);
}

void
prefs_lang_remove_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GList *selected;

	if (list->selection)
	{
		selected = g_list_copy (list->selection);
		gtk_list_remove_items (list, selected); 
	}

	pu_set_config_from_list (pd->lang_list, CONF_RENDERING_LANGUAGE);
}

void
prefs_lang_up_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GList *selected;
	int pos;

	if (list->selection)
	{
		selected = g_list_copy (list->selection);
		pos = gtk_list_child_position (list, 
					       GTK_WIDGET (selected->data));
		if (pos!=0)
		{
			gtk_list_remove_items_no_unref (list, selected); 
			pos--;
			gtk_list_insert_items (list, selected, pos);
			gtk_list_select_item (list, pos);
		}
	}

	pu_set_config_from_list (pd->lang_list, CONF_RENDERING_LANGUAGE);
}

void
prefs_lang_down_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GList *selected;
	int pos;

	if (list->selection)
	{
		selected = g_list_copy (list->selection);
		pos = gtk_list_child_position (list, 
					       GTK_WIDGET (selected->data));
		gtk_list_remove_items_no_unref (list, selected); 
		pos++;
		gtk_list_insert_items (list, selected, pos);
		gtk_list_select_item (list, pos);
	}
	
	pu_set_config_from_list (pd->lang_list, CONF_RENDERING_LANGUAGE);
}

void
prefs_clear_history_button_clicked_cb (GtkButton *button,
				       PreferencesDialog *pd)
{
	GtkWidget *dialog;
	
	dialog = gnome_question_dialog_modal_parented
		(_("This will delete all items stored in your history.\n"
		   "Are you sure you want to do this?"),
		 (GnomeReplyCallback)prefs_clear_history_question_cb, pd,
		 GTK_WINDOW (pd->dialog));

	dialog_set_parent (dialog, pd->dialog);
	gnome_dialog_run (GNOME_DIALOG (dialog));
}

void
prefs_clear_history_question_cb (gint reply, gpointer data)
{
	if (reply)
	{
		return;
	}

	/* clear the hash table */
	history_clear ();
}

void
prefs_browse_clicked_cb (GnomeFileEntry *fileentry, PreferencesDialog *pd)
{
	if (fileentry->fsw != NULL)
	{
		dialog_set_parent (fileentry->fsw, pd->dialog);
	}
}

void
prefs_font_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd)
{
	dialog_set_parent (GNOME_FONT_PICKER (widget)->font_dialog,
			   pd->dialog);
}

void
prefs_color_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd)
{
	dialog_set_parent (GNOME_COLOR_PICKER (widget)->cs_dialog,
			   pd->dialog);
}

void 
prefs_spinner_select_icon_cb (GnomeIconList *gil, gint num, 
			      GdkEvent *event, PreferencesDialog *pd)
{
	const gchar *path;

	path = spinner_get_path_from_index (num);
	gtk_entry_set_text (GTK_ENTRY (pd->spinner_dir_entry), path);
}

void
prefs_certsmanager_button_clicked_cb (GtkWidget *widget, PreferencesDialog *pd)
{
	GaleonEmbed *embed = NULL;
	GaleonWindow *window;

	embed = embed_create_in_window_chrome (NULL, NULL, NULL,
					       EMBED_CREATE_RAISE_WINDOW |
					       EMBED_CREATE_DONT_SHOW,
					       GTK_MOZ_EMBED_FLAG_OPENASCHROME);
	window = embed->parent_window;
	return_if_not_window (window);

	window->is_popup = TRUE;
	gtk_widget_set_usize (GTK_WIDGET (embed->mozembed), 
			      500, 400);
	window->set_size = TRUE;

	embed_set_visibility (embed, TRUE);
	embed_load_url(embed, "chrome://pippki/content/certManager.xul");
}

void
prefs_theme_select_icon_cb (GnomeIconList *list, gint row, GdkEvent *event, 
			    PreferencesDialog *pd)
{
	const gchar *path;

	path = gnome_icon_list_get_icon_data (list, row);
	gtk_entry_set_text (GTK_ENTRY (pd->theme_dir_entry), path);
}

void
prefs_encoding_activated_cb (GtkWidget *button, 
			     PreferencesDialog *pd)
{
	prefs_fonts_language_init ();
}

void
prefs_proxy_auto_url_reload_cb (GtkButton *button,
				PreferencesDialog *pd)
{
	gchar *url = gtk_entry_get_text (GTK_ENTRY(pd->proxy_auto_url_entry));
	mozilla_reload_proxy_autoconfiguration (url);
}

/**
 * custom_css_clist_select_row_cb: row in the stylesheet list selected
 */
void
custom_css_clist_select_row_cb (GtkCList *clist, gint row, gint column, 
				GdkEventButton *event, 
				PreferencesDialog *pd)
{
	CSSManagerRow *sel = gtk_clist_get_row_data (clist, row);

	gtk_widget_set_sensitive (pd->custom_css_remove_button, TRUE);
	gtk_widget_set_sensitive (pd->custom_css_default_check, TRUE);

	gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON (pd->custom_css_default_check),
		 sel->is_default);
}

/**
 * custom_css_clist_unselect_row_cb: row in the stylesheet list deselected
 */
void
custom_css_clist_unselect_row_cb (GtkCList *clist, gint row, gint column, 
				  GdkEventButton *event, 
				  PreferencesDialog *pd)
{
	if (clist->selection != NULL) return;

	gtk_widget_set_sensitive (pd->custom_css_remove_button, FALSE);
	gtk_widget_set_sensitive (pd->custom_css_default_check, FALSE);

	gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON (pd->custom_css_default_check), FALSE);
}

/**
 * custom_css_clist_key_press_cb: key pressed in the stylesheet list
 */
void
custom_css_clist_key_press_cb (GtkWidget *w, GdkEventKey *event,
			       PreferencesDialog *pd)
{
	if (event->state == 0 &&
	   (event->keyval == GDK_Delete || event->keyval == GDK_KP_Delete))
	{
		custom_css_remove_button_clicked_cb (w, pd);
	}
}

/**
 * custom_css_add_button_clicked_cb: add button clicked in the stylesheet
 * manager
 */
void
custom_css_add_button_clicked_cb (GtkWidget *button,
				  PreferencesDialog *pd)
{
	GtkCList *clist = GTK_CLIST (pd->custom_css_clist);
	gboolean ret;
	gchar *file = NULL;
	
	ret = show_file_picker (pd->dialog, _("Select a file to add"),
				g_get_home_dir (), NULL, modeOpen, &file,
				NULL, NULL, NULL);

	if (ret == TRUE)
	{
		CSSManagerRow *r;
		gchar *row[1];
		gint rownum, i;

		/* check for duplicates */
		for (i = 0; i < clist->rows; i++)
		{
			CSSManagerRow *item = gtk_clist_get_row_data (clist, i);

			if (strcmp (file, item->filename) == 0)
			{
				g_free (file);
				return;
			}
		}

		r = g_new0 (CSSManagerRow, 1);
		r->filename = g_strdup (file);
		r->is_default = FALSE;
	
		row[0] = file;

		rownum = gtk_clist_append (clist, row);
		gtk_clist_set_row_data (clist, rownum, r);
	}

	g_free (file);
}

/**
 * custom_css_remove_button_clicked_cb: remove button clicked in the stylesheet
 * manager
 */
void
custom_css_remove_button_clicked_cb (GtkWidget *button,
				     PreferencesDialog *pd)
{
	GtkCList *clist = GTK_CLIST (pd->custom_css_clist);
	GList *l, *remove = NULL;
	
	gtk_clist_freeze (clist);
	
	for (l = clist->selection; l; l = g_list_next (l))
	{
		gint row = GPOINTER_TO_INT (l->data);
		remove = g_list_append (remove,
					gtk_clist_get_row_data (clist, row)); 
	}

	for (l = remove; l; l = g_list_next (l))
	{
		CSSManagerRow *sel = (CSSManagerRow *) l->data; 
		gint row = gtk_clist_find_row_from_data (clist, l->data);
		gtk_clist_remove (clist, row);
		g_free (sel->filename);
		g_free (sel);
	}

	gtk_clist_thaw (clist);

	g_list_free (remove);
}

/**
 * custom_css_default_check_toggled_cb: default check in stylesheet manager
 * toggled
 */
void
custom_css_default_check_toggled_cb (GtkToggleButton *button,
				     PreferencesDialog *pd)
{
	GtkCList *clist = GTK_CLIST (pd->custom_css_clist);
	GList *l;

	for (l = clist->selection; l; l = g_list_next (l))
	{
		gint row = GPOINTER_TO_INT (l->data);
		CSSManagerRow *sel = gtk_clist_get_row_data (clist, row);
		sel->is_default = button->active;
	}

}

static void 
prefs_serif_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd)
{
	int language = pu_get_int_from_optionmenu (pd->lang_encoding_optionmenu);
	int variable_size;

	variable_size = mozilla_prefs_set_font 
		(font_name, "font.name.serif", 
		 language);

	mozilla_prefs_set_font_size 
		("font.size.variable", 
		 variable_size, language);
}

static void 
prefs_monospace_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd)
{
	int language = pu_get_int_from_optionmenu (pd->lang_encoding_optionmenu);
	int fixed_size;

	fixed_size = mozilla_prefs_set_font 
		(font_name, "font.name.monospace", 
		 language);

	mozilla_prefs_set_font_size 
		("font.size.fixed", 
		 fixed_size, language);
}

static void 
prefs_sansserif_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd)
{
	int language = pu_get_int_from_optionmenu (pd->lang_encoding_optionmenu);
	mozilla_prefs_set_font (font_name, "font.name.sans-serif", 
				language);
}

static void 
prefs_cursive_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd)
{
	int language = pu_get_int_from_optionmenu (pd->lang_encoding_optionmenu);
	mozilla_prefs_set_font (font_name, "font.name.cursive", 
				language);
}

static void 
prefs_fantasy_font_set_cb (GnomeFontPicker *gfp, gchar *font_name, PreferencesDialog *pd)
{
	int language = pu_get_int_from_optionmenu (pd->lang_encoding_optionmenu);
	mozilla_prefs_set_font (font_name, "font.name.fantasy", 
				language);
}

static void 
prefs_minsize_font_changed_cb (GtkWidget *widget, PreferencesDialog *pd)
{
	int language = pu_get_int_from_optionmenu (pd->lang_encoding_optionmenu);
	int min_size;

	min_size = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(widget));

	minfont_info[0] = min_size;
	minfont_info[1] = language;

	/* destroy any existing timer */
	if (minfont_spin_timer != NULL)
	{
		g_timer_destroy (minfont_spin_timer);
	}

	/* start the new one */
	minfont_spin_timer = g_timer_new();
	g_timer_start (minfont_spin_timer);
	g_timeout_add (50, (GSourceFunc) prefs_minfont_spinbutton_timeout_cb, 
		       minfont_info);
}

gint 
prefs_minfont_spinbutton_timeout_cb (int info[])
{
        /* timer still valid? */
        if (minfont_spin_timer == NULL)
        {
                return FALSE;
        }

        /* okay, we're ready to set */
        if (g_timer_elapsed (minfont_spin_timer, NULL) >= SPIN_DELAY)
        {
                /* kill off the timer */
                g_timer_destroy (minfont_spin_timer);
                minfont_spin_timer = NULL;

                /* set */
		mozilla_prefs_set_font_size ("font.min-size.variable", 
					     info[0], info[1]);
		mozilla_prefs_set_font_size ("font.min-size.fixed", 
					     info[0], info[1]);

                /* done now */
                return FALSE;
        }

        /* call me again */
        return TRUE;
}


