/*
 *  Original Cookie Manager code
 *  Copyright (C) 2001 Daniel Stodden <stodden@in.tum.de>
 *  
 *  Updates for new nsICookieManager interface
 *  Password Handling
 *  General Cleanup
 *  Copyright (C) 2001 Philip Langdale <philipl@mail.utexas.edu>
 *
 *  Complete rewrite
 *  Copyright (C) 2001 Jorn Baayen <jorn@nl.linux.org>
 *
 *  GTK+based persistent data manager ( cookies, passwords, image blocks )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "window.h"
#include "mozilla.h"
#include "glade.h"
#include "dialog.h"
#include "persistent_data_manager.h"

#include <string.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkentry.h>
#include <gdk/gdkkeysyms.h>
#include <libgnome/libgnome.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog.h>

/* local variables */
static PDMDialog *pd = NULL;

static GList *cookie_list;
static GList *removed_cookie_list;
static GList *blocked_cookie_list;

static GList *cookie_site_list;
static GList *removed_cookie_site_list;

static GList *image_site_list;
static GList *removed_image_site_list;

static GList *password_list;
static GList *removed_password_list;

static GList *password_site_list;
static GList *removed_password_site_list;

/* local function prototypes */
static void load_cookie (void);
static void load_cookie_sites (void);
static void load_image_sites (void);
static void load_password (void);
static void load_password_sites (void);

static PDMDialog*
pdm_create_manager (GaleonWindow *window)
{
	GladeXML *gxml;

	/* use existing dialog if available */
	if (pd != NULL) return pd;

	/* build dialog */
	pd = g_new0 (PDMDialog, 1);

	gxml = glade_widget_new ("galeon.glade", "pdm_dialog",
				 &(pd->dialog), pd);

	pd->notebook = glade_xml_get_widget (gxml, "notebook");
	pd->cookie_clist = glade_xml_get_widget (gxml, "cookie_clist");
	pd->cookie_remove_button =
		glade_xml_get_widget (gxml, "cookie_remove");
	pd->cookie_remove_and_block_button =
		glade_xml_get_widget (gxml, "cookie_block");
	pd->cookie_remove_all_button =
		glade_xml_get_widget (gxml, "cookie_remove_all");
	pd->cookie_name =
		glade_xml_get_widget (gxml, "cookie_name");
	pd->cookie_info =
		glade_xml_get_widget (gxml, "cookie_info");
	pd->cookie_host =
		glade_xml_get_widget (gxml, "cookie_host");
	pd->cookie_secure =
		glade_xml_get_widget (gxml, "cookie_secure");
	pd->cookie_path =
		glade_xml_get_widget (gxml, "cookie_path");
	pd->cookie_expires =
		glade_xml_get_widget (gxml, "cookie_expires");

	pd->cookie_site_clist =
		glade_xml_get_widget (gxml, "cookie_site_clist");
	pd->cookie_site_remove_button =
		glade_xml_get_widget (gxml, "cookie_site_remove");
	pd->cookie_site_remove_all_button =
		glade_xml_get_widget (gxml, "cookie_site_remove_all");

	pd->image_site_clist =
		glade_xml_get_widget (gxml, "image_site_clist");
	pd->image_site_remove_button =
		glade_xml_get_widget (gxml, "image_site_remove");
	pd->image_site_remove_all_button =
		glade_xml_get_widget (gxml, "image_site_remove_all");
	
	pd->password_clist =
		glade_xml_get_widget (gxml, "password_clist");
	pd->password_remove_button =
		glade_xml_get_widget (gxml, "password_remove");
	pd->password_remove_all_button =
		glade_xml_get_widget (gxml, "password_remove_all");

	pd->password_site_clist =
		glade_xml_get_widget (gxml, "password_site_clist");
	pd->password_site_remove_button =
		glade_xml_get_widget (gxml, "password_site_remove");
	pd->password_site_remove_all_button =
		glade_xml_get_widget (gxml, "password_site_remove_all");

	gtk_object_destroy (GTK_OBJECT (gxml));

	/* load lists */
	cookie_list = mozilla_list_cookies ();
	removed_cookie_list = NULL;
	blocked_cookie_list = NULL;
	cookie_site_list = mozilla_get_permissions (COOKIEPERMISSION);
	removed_cookie_site_list = NULL;
	image_site_list = mozilla_get_permissions (IMAGEPERMISSION);
	removed_image_site_list = NULL;
	password_list = mozilla_list_passwords (PASSWORD_PASSWORD);
	removed_password_list = NULL;
	password_site_list = mozilla_list_passwords (PASSWORD_REJECT);
	removed_password_site_list = NULL;

	/* read in the stored data */
	load_cookie ();
	load_cookie_sites ();
	load_image_sites ();
	load_password ();
	load_password_sites ();

	return pd;
}

void
pdm_show_page (GaleonWindow *window, PDMPage page)
{
	/* make sure the dialog has been created */
	pdm_create_manager (window);

	gtk_notebook_set_page (GTK_NOTEBOOK (pd->notebook), page);

	if (window != NULL)
	{
		dialog_set_parent (pd->dialog, window->wmain);
	}

	gtk_widget_show_all (pd->dialog);

	return;
}

void
pdm_apply (void)
{
	/* do queued cookie site actions */
	/* we need to do this BEFORE doing the cookie actions, because
	 * of the remove&block cookie hack. (Sites can be removed and
	 * readded later. So wen eed to remove first) */
	mozilla_remove_permissions (removed_cookie_site_list, COOKIEPERMISSION);
	g_list_free (removed_cookie_site_list);
	removed_cookie_site_list = NULL;

	/* do queued cookie actions (should be before cookie sites, because
	 * if we removed & blocked a cookie, and then removed that site again
	 * it needs to be blocked beofre we can remove the permission again )*/
	mozilla_remove_cookies (removed_cookie_list, FALSE);
	mozilla_remove_cookies (blocked_cookie_list, TRUE);
	g_list_free (removed_cookie_list);
	g_list_free (blocked_cookie_list);
	removed_cookie_list = NULL;
	blocked_cookie_list = NULL;

	/* do queued image site actions */
	mozilla_remove_permissions (removed_image_site_list, IMAGEPERMISSION);
	g_list_free (removed_image_site_list);
	removed_image_site_list = NULL;

	/* do queued password actions */
	mozilla_remove_passwords (removed_password_list, PASSWORD_PASSWORD);
	g_list_free (removed_password_list);
	removed_password_list = NULL;

	/* do queued password sites actions */
	mozilla_remove_passwords (removed_password_site_list, PASSWORD_REJECT);
	g_list_free (removed_password_site_list);
	removed_password_site_list = NULL;
}

void
pdm_close (void)
{
	GList *l;

	/* destroy dialog (the actual GnomeDialog will be closed by
	 * persistent_data_manager_dialog_close_cb, which called us) */
	g_free (pd);
	pd = NULL;

	/* cleanup cookies */
	for (l = cookie_list; l; l = g_list_next (l))
	{
		Cookie *c = (Cookie *) l->data;
		g_free (c->base.type);
		g_free (c->base.domain);
		g_free (c->name);
		g_free (c->value);
		g_free (c->path);
		g_free (c->secure);
		g_free (c->expire);
		g_free (c);
	}
	g_list_free (cookie_list);
	g_list_free (removed_cookie_list);
	g_list_free (blocked_cookie_list);

	/* cleanup cookie sites */
	for (l = cookie_site_list; l; l = g_list_next (l))
	{
		BlockedHost *b = (BlockedHost *) l->data;
		g_free (b->type);
		g_free (b->domain);
		g_free (b);
	}
	g_list_free (cookie_site_list);
	g_list_free (removed_cookie_site_list);

	/* cleanup image sites */
	for (l = image_site_list; l; l = g_list_next (l))
	{
		BlockedHost *b = (BlockedHost *) l->data;
		g_free (b->type);
		g_free (b->domain);
		g_free (b);
	}
	g_list_free (image_site_list);
	g_list_free (removed_image_site_list);

	/* cleanup passwords */
	for (l = password_list; l; l = g_list_next (l))
	{
		Password *p = (Password *) l->data;
		g_free (p->host);
		g_free (p->username);
		g_free (p);
	}
	g_list_free (password_list);
	g_list_free (removed_password_list);

	/* cleanup password sites */
	for (l = password_site_list; l; l = g_list_next (l))
	{
		Password *p = (Password *) l->data;
		g_free (p->host);
		g_free (p->username);
		g_free (p);
	}
	g_list_free (password_site_list);
	g_list_free (removed_password_site_list);
}

void
pdm_update_cookie_view (Cookie *c)
{
	gtk_widget_set_sensitive (pd->cookie_remove_button,
		GTK_CLIST (pd->cookie_clist)->selection != NULL);
	gtk_widget_set_sensitive (pd->cookie_remove_and_block_button,
		GTK_CLIST (pd->cookie_clist)->selection != NULL);

	/* update cookie info */
	if (c != NULL)
	{
		gtk_entry_set_text (GTK_ENTRY (pd->cookie_name),
				    c->name);
		gtk_entry_set_text (GTK_ENTRY (pd->cookie_info),
				    c->value);
		gtk_entry_set_text (GTK_ENTRY (pd->cookie_host),
				    c->base.domain);
		gtk_entry_set_text (GTK_ENTRY (pd->cookie_path),
				    c->path);
		gtk_entry_set_text (GTK_ENTRY (pd->cookie_secure),
				    c->secure);
		gtk_entry_set_text (GTK_ENTRY (pd->cookie_expires),
				    c->expire);
	}
	else
	{
		gtk_editable_delete_text
			(GTK_EDITABLE (pd->cookie_name), 0, -1);
		gtk_editable_delete_text
			(GTK_EDITABLE (pd->cookie_info), 0, -1);
		gtk_editable_delete_text
			(GTK_EDITABLE (pd->cookie_host), 0, -1);
		gtk_editable_delete_text
			(GTK_EDITABLE (pd->cookie_path), 0, -1);
		gtk_editable_delete_text
			(GTK_EDITABLE (pd->cookie_secure), 0, -1);
		gtk_editable_delete_text
			(GTK_EDITABLE (pd->cookie_expires), 0, -1);
	}
}

void
pdm_remove_cookie (Cookie *c, gboolean block)
{
	gint row = gtk_clist_find_row_from_data
		(GTK_CLIST (pd->cookie_clist), c);
	gtk_clist_remove (GTK_CLIST (pd->cookie_clist), row);
	if (block)
	{
		/* append a fake entry to the blocked list */
		GtkCList *clist = GTK_CLIST (pd->cookie_site_clist);
		BlockedHost *h = g_new0 (BlockedHost, 1);
		BlockedHost *bh = NULL;
		gchar *row[2];
		gint rownum, i;

		gtk_clist_freeze (clist);
		
		/* remove any matching existing entry from the clist */
		for (i = clist->rows - 1; i >= 0; i--)
		{
			bh = (BlockedHost *) gtk_clist_get_row_data (clist, i);
			if (strcmp (bh->domain, c->base.domain) == 0)
			{
				gint r = gtk_clist_find_row_from_data
					(clist, bh);
				gtk_clist_remove (clist, r);
				break;
			}
		}

		/* append to blocked sits clist */
		h->domain = g_strdup (c->base.domain);
		h->type = g_strdup (_("Blocked"));

		row[0] = h->domain;
		row[1] = h->type;

		rownum = gtk_clist_append (clist, row);
		gtk_clist_set_row_data (clist, rownum, h);
		gtk_clist_sort (clist);
		gtk_clist_thaw (clist);
		
		/* append to remove & block list */
		blocked_cookie_list = g_list_append (blocked_cookie_list, c);
	}
	else
	{
		removed_cookie_list = g_list_append (removed_cookie_list, c);
	}

	gtk_clist_select_row (GTK_CLIST (pd->cookie_clist), row, 0);
	gtk_widget_set_sensitive (pd->cookie_remove_all_button,
				  GTK_CLIST (pd->cookie_clist)->rows > 0);
}

void
pdm_update_cookie_sites_view (BlockedHost *c)
{
	gtk_widget_set_sensitive (pd->cookie_site_remove_button,
		GTK_CLIST (pd->cookie_site_clist)->selection != NULL);
}

void
pdm_remove_cookie_site (BlockedHost *c)
{
	gint row = gtk_clist_find_row_from_data
		(GTK_CLIST (pd->cookie_site_clist), c);
	gtk_clist_remove (GTK_CLIST (pd->cookie_site_clist), row);
	gtk_clist_select_row (GTK_CLIST (pd->cookie_site_clist), row, 0);
	removed_cookie_site_list = g_list_append (removed_cookie_site_list, c);

	gtk_widget_set_sensitive (pd->cookie_site_remove_all_button,
				  GTK_CLIST (pd->cookie_site_clist)->rows > 0);
}

void
pdm_update_image_sites_view (BlockedHost *c)
{
	gtk_widget_set_sensitive (pd->image_site_remove_button,
		GTK_CLIST (pd->image_site_clist)->selection != NULL);
}

void
pdm_remove_image_site (BlockedHost *c)
{
	gint row = gtk_clist_find_row_from_data
		(GTK_CLIST (pd->image_site_clist), c);
	gtk_clist_remove (GTK_CLIST (pd->image_site_clist), row);
	gtk_clist_select_row (GTK_CLIST (pd->image_site_clist), row, 0);
	removed_image_site_list = g_list_append (removed_image_site_list, c);

	gtk_widget_set_sensitive (pd->image_site_remove_all_button,
				  GTK_CLIST (pd->image_site_clist)->rows > 0);
}

void
pdm_update_password_view (Password *p)
{

	gtk_widget_set_sensitive (pd->password_remove_button,
		GTK_CLIST (pd->password_clist)->selection != NULL);
}

void
pdm_remove_password (Password *p)
{
	gint row = gtk_clist_find_row_from_data
		(GTK_CLIST (pd->password_clist), p);
	gtk_clist_remove (GTK_CLIST (pd->password_clist), row);
	gtk_clist_select_row (GTK_CLIST (pd->password_clist), row, 0);
	removed_password_list = g_list_append (removed_password_list, p);

	gtk_widget_set_sensitive (pd->password_remove_all_button,
				  GTK_CLIST (pd->password_clist)->rows > 0);
}

void
pdm_update_password_sites_view (Password *p)
{
	gtk_widget_set_sensitive (pd->password_site_remove_button,
		GTK_CLIST (pd->password_site_clist)->selection != NULL);
}

void
pdm_remove_password_site (Password *p)
{
	gint row = gtk_clist_find_row_from_data
		(GTK_CLIST (pd->password_site_clist), p);
	gtk_clist_remove (GTK_CLIST (pd->password_site_clist), row);
	gtk_clist_select_row (GTK_CLIST (pd->password_site_clist), row, 0);
	removed_password_site_list =
		g_list_append (removed_password_site_list, p);
	
	gtk_widget_set_sensitive (pd->password_site_remove_all_button,
			GTK_CLIST (pd->password_site_clist)->rows > 0);
}

/* local functions */
static void
load_cookie ()
{
	GList *l;

	gtk_clist_freeze (GTK_CLIST (pd->cookie_clist));

	for (l = cookie_list; l; l = g_list_next (l))
	{
		Cookie *c = (Cookie *) l->data;
		gchar *row[2];
		gint rownum;

		row[0] = c->base.domain;
		row[1] = c->name;

		rownum = gtk_clist_append (GTK_CLIST (pd->cookie_clist), row);
		gtk_clist_set_row_data (GTK_CLIST (pd->cookie_clist),
					rownum, c);
	}

	gtk_clist_sort (GTK_CLIST (pd->cookie_clist));
	gtk_clist_thaw (GTK_CLIST (pd->cookie_clist));

	gtk_widget_set_sensitive (pd->cookie_remove_all_button,
				  GTK_CLIST (pd->cookie_clist)->rows > 0);
}

static void
load_cookie_sites ()
{
	GList *l;

	gtk_clist_freeze (GTK_CLIST (pd->cookie_site_clist));

	for (l = cookie_site_list; l; l = g_list_next (l))
	{
		BlockedHost *c = (BlockedHost *) l->data;
		gchar *row[2];
		gint rownum;

		row[0] = c->domain;
		row[1] = c->type;

		rownum = gtk_clist_append
			(GTK_CLIST (pd->cookie_site_clist), row);
		gtk_clist_set_row_data (GTK_CLIST (pd->cookie_site_clist),
					rownum, c);
	}

	gtk_clist_sort (GTK_CLIST (pd->cookie_site_clist));
	gtk_clist_thaw (GTK_CLIST (pd->cookie_site_clist));

	gtk_widget_set_sensitive (pd->cookie_site_remove_all_button,
				  GTK_CLIST (pd->cookie_site_clist)->rows > 0);
}

static void
load_image_sites ()
{
	GList *l;

	gtk_clist_freeze (GTK_CLIST (pd->image_site_clist));

	for (l = image_site_list; l; l = g_list_next (l))
	{
		BlockedHost *c = (BlockedHost *) l->data;
		gchar *row[2];
		gint rownum;

		row[0] = c->domain;
		row[1] = c->type;

		rownum = gtk_clist_append
			(GTK_CLIST (pd->image_site_clist), row);
		gtk_clist_set_row_data (GTK_CLIST (pd->image_site_clist),
					rownum, c);
	}

	gtk_clist_sort (GTK_CLIST (pd->image_site_clist));	
	gtk_clist_thaw (GTK_CLIST (pd->image_site_clist));

	gtk_widget_set_sensitive (pd->image_site_remove_all_button,
				  GTK_CLIST (pd->image_site_clist)->rows > 0);
}

static void
load_password ()
{
	GList *l;

	gtk_clist_freeze (GTK_CLIST (pd->password_clist));

	for (l = password_list; l; l = g_list_next (l))
	{
		Password *c = (Password *) l->data;
		gchar *row[2];
		gint rownum;

		row[0] = c->host;
		row[1] = c->username;

		rownum = gtk_clist_append
			(GTK_CLIST (pd->password_clist), row);
		gtk_clist_set_row_data (GTK_CLIST (pd->password_clist),
					rownum, c);
	}

	gtk_clist_sort (GTK_CLIST (pd->password_clist));
	gtk_clist_thaw (GTK_CLIST (pd->password_clist));

	gtk_widget_set_sensitive (pd->password_remove_all_button,
				  GTK_CLIST (pd->password_clist)->rows > 0);
}

static void
load_password_sites ()
{
	GList *l;

	gtk_clist_freeze (GTK_CLIST (pd->password_site_clist));

	for (l = password_site_list; l; l = g_list_next (l))
	{
		Password *c = (Password *) l->data;
		gchar *row[2];
		gint rownum;

		row[0] = c->host;
		row[1] = g_strdup (_("Blocked"));

		rownum = gtk_clist_append
			(GTK_CLIST (pd->password_site_clist), row);
		gtk_clist_set_row_data (GTK_CLIST (pd->password_site_clist),
					rownum, c);

		g_free (row[1]);
	}

	gtk_clist_sort (GTK_CLIST (pd->password_site_clist));
	gtk_clist_thaw (GTK_CLIST (pd->password_site_clist));
	
	gtk_widget_set_sensitive (pd->password_site_remove_all_button,
			GTK_CLIST (pd->password_site_clist)->rows > 0);
}
