/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "prefs.h"

#include <algorithm>

#ifdef __GNUC__

#if (__GNUC__ < 3)

#ifndef STRING_HEADER
#define STRING_HEADER <string>
#endif

#include STRING_HEADER

#else
#include <string>

#endif

#else
#include <string>
#endif

#include <list>
#include <dirent.h>
#include <sys/stat.h>
#include <libgen.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-i18n.h>

#include "nsXPIDLString.h"
#include "nsIIOService.h"
#include "nsIServiceManager.h"
#include "nsCOMPtr.h"
#include "nsIURI.h"
#include "nsNetCID.h"
#include "nsIStorageStream.h"
#include "nsNetUtil.h"
#include "nsIExternalProtocolService.h"
#include "nsCExternalHandlerService.h"
#include "nsIPlatformCharset.h"

#include "BaseProtocolHandler.h"
#include "TOCProtocolHandler.h"

#ifndef HAVE_SCANDIR
#include "scandir.h"
#endif

using namespace std;

#define RENDER(str) \
        oStream->Write (str, strlen(str), &bytesWriten);
#define RENDER_NS_STR(str) \
	oStream->Write (str.get(), str.Length(), &bytesWriten);
#define RENDER_STL_STR(str) \
	oStream->Write (str.c_str(), str.size(), &bytesWriten);

static void ParseEnvPath(const nsACString &path, list<string> &dirs);
static int gHelpSelect (const struct dirent *dirEntry);
static int gnomeHelpSelect (const struct dirent *dirEntry);
static void RenderContentType (nsIOutputStream *oStream, PRUint32 &bytesWriten);

/* Implementation file */
NS_IMPL_ISUPPORTS1 (GTOCProtocolHandler, nsIProtocolHandler)

/* nsIChannel newChannel (in nsIURI aURI); */
NS_IMETHODIMP GTOCProtocolHandler::NewChannel(nsIURI *aURI,
					      nsIChannel **_retval)
{
	nsresult rv;
	mURI = aURI;

	rv = aURI->GetPath (mDocType);
	if (NS_FAILED(rv)) return rv;

	rv = CreatePage ();
	NS_IF_ADDREF (*_retval = mChannel);
	return rv;
}

NS_METHOD GTOCProtocolHandler::CreatePage (void)
{
	nsresult rv;

	/* open the rendering stream */
	rv = NS_NewStorageStream(16384, (PRUint32)-1, getter_AddRefs(mStream));
	if (NS_FAILED(rv)) return rv;

	/* render appropriate content */
	if (mDocType.Equals("info"))
		rv = CreateInfoPage ();
	else if (mDocType.Equals("man"))
		rv = CreateManPage();
	else if (mDocType.Equals("ghelp"))
		rv = CreateHelpPage ("ghelp", gHelpSelect);
	else if (mDocType.Equals("gnome-help"))
		rv = CreateHelpPage ("gnome-help", gnomeHelpSelect);
	else
		rv = CreateTOCPage();
	if (NS_FAILED(rv)) return rv;

	/* finish the rendering */
	nsCOMPtr<nsIInputStream> iStream;
	rv = mStream->NewInputStream(0, getter_AddRefs(iStream));
	if (NS_FAILED(rv)) return rv;

#if MOZILLA_SNAPSHOT > 4
	rv = NS_NewInputStreamChannel(getter_AddRefs(mChannel), mURI,
				      iStream, NS_LITERAL_CSTRING("text/html"),
				      NS_LITERAL_CSTRING("utf-8"));
#else
	PRUint32 size;
	rv = mStream->GetLength(&size);
	if (NS_FAILED(rv)) return rv;

	rv = NS_NewInputStreamChannel(getter_AddRefs(mChannel), mURI,
				      iStream, NS_LITERAL_CSTRING("text/html"),
				      NS_LITERAL_CSTRING("utf-8"), size);
#endif
	if (NS_FAILED(rv)) return rv;

	return rv;
}

NS_METHOD GTOCProtocolHandler::CreateTOCPage (void)
{
	nsresult rv;

	nsCOMPtr<nsIOutputStream> oStream;
	rv = mStream->GetOutputStream(0, getter_AddRefs(oStream));
	if (NS_FAILED(rv)) return rv;

	PRUint32 bytesWriten;
	
	RENDER ("<html><head>\n");
	RENDER ("<link rel=\"stylesheet\" href=\"file:");
	RENDER (SHARE_DIR "/toc.css");
	RENDER ("\" type=\"text/css\">\n");
	RENDER ("<title>");
	RENDER (_("GNOME Help Index"));
	RENDER ("</title></head>\n");

	RenderContentType(oStream, bytesWriten);

	RENDER("</head>\n<body>\n");

	RENDER ("<h3><strong>Table of contents</strong></h3>");

	RENDER ("<a href=\"ghelp:gnome-users-guide\">");
	RENDER (_("GNOME User's Guide"));
	RENDER ("</a>\n<br>\n");

	RENDER ("<a href=\"toc:man\">");
	RENDER (_("Man Pages"));
	RENDER ("</a>\n<br>\n");

	RENDER ("<a href=\"toc:info\">");
	RENDER (_("Info Pages"));
	RENDER ("</a>\n<br>\n");

	RENDER ("<a href=\"toc:ghelp\">");
	RENDER (_("HTML GNOME Documents"));
	RENDER ("</a>\n<br>\n");

	RENDER ("<a href=\"toc:gnome-help\">");
	RENDER (_("SGML GNOME Documents"));
	RENDER ("</a>\n<br>\n");

	RENDER ("</body></html>\n");

	return NS_OK;
}

NS_METHOD GTOCProtocolHandler::CreateInfoPage (void)
{
	nsresult rv;

	const nsCAutoString infopath(NS_LITERAL_CSTRING("/usr/info:/usr/local/info:") +
				     nsDependentCString(g_getenv("INFOPATH")));

	const nsCAutoString infoEnv(NS_LITERAL_CSTRING("INFOPATH=") + infopath);
	char *auxEnv = strdup(infoEnv.get());
	putenv (auxEnv);

	list<string> dirs;
	ParseEnvPath (infopath, dirs);

	if (dirs.size()>0)
	{
		dirs.sort ();
		dirs.unique ();
		dirs.erase(find(dirs.begin(), dirs.end(), "/usr/info"));
		dirs.erase(find(dirs.begin(), dirs.end(), "/usr/local/info"));
	}
	dirs.push_front("/usr/local/info");
	dirs.push_front("/usr/info");

	nsCOMPtr<nsIOutputStream> oStream;
	rv = mStream->GetOutputStream(0, getter_AddRefs(oStream));
	if (NS_FAILED(rv)) return rv;

	PRUint32 bytesWriten;
	
	RENDER ("<html><head>\n");
	RENDER ("<link rel=\"stylesheet\" href=\"file:");
	RENDER (SHARE_DIR "/toc.css");
	RENDER ("\" type=\"text/css\">\n");
	RENDER ("<title>");
	RENDER (_("GNOME"));
	RENDER (" ");
	RENDER (_("Info"));
	RENDER (":");
	RENDER (_("pages"));
	RENDER ("</title></head>\n");

	RenderContentType(oStream, bytesWriten);

	RENDER("</head>\n<body>\n");
	RENDER ("<h3><strong>Info pages</strong></h3>\n");

	for (list<string>::iterator j = dirs.begin() ;
	     j != dirs.end() ; j++)
	{
		string dirname(*j);
		if (dirname.size() == 0) continue;

		RENDER ("<a href=\"");
		RENDER ("info:");
		RENDER_STL_STR (dirname);
		struct stat buf;
		if (!stat((dirname+".gz").c_str(), &buf))
		{
			RENDER ("/dir.gz\">");
		}
		else
		{
			RENDER ("/dir\">");
		}
		RENDER_STL_STR (dirname);
		RENDER ("</a>\n<br>\n");
	}

	RENDER ("\n</body></html>\n");

	return NS_OK;
}

NS_METHOD GTOCProtocolHandler::CreateManPage (void)
{
	nsresult rv;

	list<string> dirs;
	ParseEnvPath (nsDependentCString(g_getenv("MANPATH")), dirs);

	if (dirs.size()>0)
	{
		dirs.sort ();
		dirs.unique ();

		list<string>::iterator i;
		i = find(dirs.begin(), dirs.end(), "/usr/man");
		if(i != dirs.end()) dirs.erase(i);
		i = find(dirs.begin(), dirs.end(), "/usr/local/man");
		if(i != dirs.end()) dirs.erase(i);
	}
	dirs.push_front("/usr/local/man");
	dirs.push_front("/usr/man");

	nsCOMPtr<nsIOutputStream> oStream;
	rv = mStream->GetOutputStream(0, getter_AddRefs(oStream));
	if (NS_FAILED(rv)) return rv;

	PRUint32 bytesWriten;
	
	RENDER ("<html><head>\n");
	RENDER ("<link rel=\"stylesheet\" href=\"file:");
	RENDER (SHARE_DIR "/toc.css");
	RENDER ("\" type=\"text/css\">\n");
	RENDER ("<title>");
	RENDER (_("GNOME"));
	RENDER (" ");
	RENDER (_("Man"));
	RENDER (":");
	RENDER (_("pages"));
	RENDER ("</title></head>\n");

	RenderContentType(oStream, bytesWriten);

	RENDER("</head>\n<body>\n");
	RENDER ("<h3><strong>Man pages</strong></h3>");

	for (int i = 1 ; i <= 9 ; i++)
	{
		RENDER ("<h3><strong>");
		char *tmp;
		if (i != 9)
			tmp = g_strdup_printf ("man%d", i);
		else
			tmp = g_strdup ("mann");
		RENDER (tmp);
		RENDER ("</strong></h3>");

		for (list<string>::iterator j = dirs.begin() ;
		     j != dirs.end() ; j++)
		{
			string dir = (*j) + "/";
			dir += tmp;

			struct dirent **namelist;
			int n = scandir (dir.c_str(), &namelist, NULL,
				 alphasort);
			if (n<0) continue;

			RENDER ("<h4>");
			RENDER (dir.c_str());
			RENDER ("</h4>");

			for (int k=0 ; k<n ; k++)
			{
				nsCAutoString fileName (namelist[k]->d_name);
				PRInt32 n = fileName.Find (".");
				if (n == -1) continue;
				nsCAutoString baseName;
				n = fileName.Left (baseName, n);
				if (baseName.Length() == 0) continue;
				RENDER ("<a href=\"");
				RENDER ("man:");
				RENDER_NS_STR (baseName);
				RENDER ("\">");
				RENDER_NS_STR (baseName);
				RENDER ("</a>\n");
				g_free (namelist[k]);
			}
			g_free (namelist);
		}
		g_free (tmp);
	}

	RENDER ("</body></html>\n");

	return NS_OK;
}

NS_METHOD GTOCProtocolHandler::CreateHelpPage (const char *type,
					       int (*select)(const struct dirent *))
{
	nsresult rv;

	struct dirent **namelist;
	int n = scandir (SHARE_DIR"/../gnome/help", &namelist, select,
			 alphasort);
	if (n<0) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIOutputStream> oStream;
	rv = mStream->GetOutputStream(0, getter_AddRefs(oStream));
	if (NS_FAILED(rv)) return rv;

	PRUint32 bytesWriten;
	
	RENDER ("<html><head>\n");
	RENDER ("<link rel=\"stylesheet\" href=\"file:");
	RENDER (SHARE_DIR "/toc.css");
	RENDER ("\" type=\"text/css\">\n");
	RENDER ("<title>");
	RENDER (_("GNOME"));
	RENDER (" ");
	RENDER (type);
	RENDER (":");
	RENDER (_("pages"));
	RENDER ("</title></head>\n");

	RenderContentType(oStream, bytesWriten);

	RENDER("</head>\n<body>\n");
	RENDER ("<h3><strong>Help pages</strong></h3>");

	for (int i=0 ; i<n ; i++)
	{
		RENDER ("<a href=\"");
		RENDER (type);
		RENDER (":");
		RENDER (namelist[i]->d_name);
		RENDER ("\">");
		RENDER (namelist[i]->d_name);
		RENDER ("</a>\n<br>\n");
		free (namelist[i]);
	}
	free (namelist);

	RENDER ("</body></html>\n");

	return NS_OK;
}

void ParseEnvPath(const nsACString &aPath, list<string> &dirs)
{
	const nsCString path(aPath);
	if (path.Length() != 0)
	{
		PRInt32 n = 0, m = path.Find (":"), t;
		nsCAutoString item;
		if (m != -1)
		{
			path.Left(item, m);
			dirs.push_back(item.get());
			n = m;
			m = path.Find(":",PR_FALSE,m+1);
			while (m!= -1)
			{
				t = path.Mid(item, n+1, m-n-1);
				dirs.push_back(item.get());
				n = m;
				m = path.Find(":",PR_FALSE,n+1);
			}
			PRInt32 end = path.Length();
			if(n+1 < end)
			{
				path.Mid(item, n+1, end);
				dirs.push_back(item.get());
			}
		}
		else
		{
			dirs.push_back(path.get());
		}
	}
}

int gHelpSelect (const struct dirent *dirEntry)
{
	char *helpPath = gnome_help_file_find_file (
					const_cast<char *>(dirEntry->d_name),
					"index.html");
	int ret;
	if (helpPath) ret = 1;
	else ret = 0;
	g_free (helpPath);
	return ret;
}

int gnomeHelpSelect (const struct dirent *dirEntry)
{
	nsCAutoString fileName(nsDependentCString(dirEntry->d_name) +
			       NS_LITERAL_CSTRING(".sgml"));

	char *helpPath = gnome_help_file_find_file (
					const_cast<char *>(dirEntry->d_name),
					const_cast<char *>(fileName.get()));
	int ret;
	if (helpPath) ret = 1;
	else ret = 0;
	g_free (helpPath);
	return ret;
}

void RenderContentType (nsIOutputStream *oStream, PRUint32 &bytesWriten)
{
	nsresult rv;
	nsCOMPtr<nsIPlatformCharset> pc =
		do_GetService(NS_PLATFORMCHARSET_CONTRACTID, &rv);
	g_return_if_fail (NS_SUCCEEDED(rv));
	
#if MOZILLA_SNAPSHOT > 9
	nsCAutoString platformCharset;
#else
	nsAutoString platformCharset;
#endif
	rv = pc->GetCharset(kPlatformCharsetSel_Menu, platformCharset);
	if (!platformCharset.IsEmpty())
	{
		RENDER ("<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=");
#if MOZILLA_SNAPSHOT > 9
		RENDER (platformCharset.get());
#else
		RENDER (NS_ConvertUCS2toUTF8(platformCharset).get());
#endif
		RENDER ("\">");
	}
}

////////////////////////////////////////////////////////////////////////////////


