/*
 *  Copyright (C) 2002 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* GSidebar: Only a single instance of this class should be present at 
 * any one time. It handles incoming sidebar sites from mozilla through
 * GSidebarProxy instances as explained in SidebarProxy.cpp. In turn
 * it creates and destroys instances of SidebarDock, which is the actual
 * Gtk sidebar implementation. The single instance of GSidebar is reference
 * counted by each instance of GSidebarProxy and by each Gtk sidebar dock.
 * Thus, it will only go out of scope when there are no visible docks and
 * no mozilla DOMWindows holding a nsCOMPtr<nsISidebar>
 */

#include "embed.h"
#include "galeon.h"
#include "glade.h"
#include "mozilla.h"
#include "window.h"
#include "eel-gconf-extensions.h"
#include <gnome-xml/parser.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/xmlmemory.h>

#include <gtkmozembed.h>
#include <nsString.h>
#include <nsCRT.h>
#include <nsIProgrammingLanguage.h>

#include "SideBar.h"

#include <utility>

using namespace std;

extern "C" {
#include <libgnome/libgnome.h>
}

GSidebar *GlobalSidebar(nsnull);

static gboolean FreeSidebarHashTableEntry(gpointer key, gpointer val, gpointer data);
static void SaveHashTableEntry(gpointer key, gpointer value, gpointer xmlData);

extern "C"
gint sidebar_dom_mouse_click_cb (GtkMozEmbed *dummy, gpointer dom_event,
                                 GaleonEmbed *embed);

/* Implementation file */
NS_IMPL_ISUPPORTS1(GSidebar, nsISidebar)

GSidebar::GSidebar() : mComboList(nsnull),
		       mComboListEmpty(PR_TRUE),
		       mUrlTable(nsnull),
		       mDocks(nsnull),
		       mLastDock(nsnull),
		       mRefCount(0)
{
	NS_INIT_ISUPPORTS();
	mUrlTable = g_hash_table_new(g_str_hash, g_str_equal);
	LoadSidebarPages();
}

GSidebar::~GSidebar()
{
	g_assert(mRefCount == 0);
	g_assert(!g_list_length(mDocks));

	SaveSidebarPages();

	/*Notice that we *do not* free the strings in the list as they
	 *are the same strings as contained in the hash table.
	 */
	g_list_free(mComboList);

	g_hash_table_foreach_remove(mUrlTable,
				    (GHRFunc) FreeSidebarHashTableEntry,
				    NULL);
	g_hash_table_destroy(mUrlTable);

	GlobalSidebar = nsnull;
}

//------------------------------------------------------------------------------
//nsISidebar Impl.
//------------------------------------------------------------------------------

#if MOZILLA_SNAPSHOT < 12
/* void setWindow (in nsIDOMWindowInternal aWindow); */
NS_IMETHODIMP GSidebar::SetWindow(nsIDOMWindowInternal *aWindow)
{
	nsresult rv;
	nsCOMPtr<nsIDOMWindow> aDOMWindow =
		do_QueryInterface(aWindow, &rv);
	if(NS_FAILED(rv) || !aDOMWindow) return NS_ERROR_FAILURE;

	GaleonWindow *aParent(nsnull);
	aParent = mozilla_find_galeon_parent(aDOMWindow);
	if(!aParent) return NS_ERROR_FAILURE;

	return SetWindow(aParent, PR_TRUE);
}
#endif

/* void addPanel (in wstring aTitle, in string aContentURL, in string aCustomizeURL); */
NS_IMETHODIMP GSidebar::AddPanel(const PRUnichar *aTitle,
				 const char *aContentURL,
				 const char *aCustomizeURL)
{
	gchar *localeTitle = mozilla_unicode_to_locale(aTitle);

	gchar *oldURL = static_cast<gchar *>(g_hash_table_lookup(mUrlTable,
							         localeTitle));
	if(oldURL)
	{
		if(mLastDock && GTK_IS_ENTRY(mLastDock))
		{
			gtk_entry_set_text(GTK_ENTRY(mLastDock->mComboEntry),
					   localeTitle);
		}
		g_free(localeTitle);
	}
	else
	{
		if(mComboList && mComboListEmpty)
		{
			gpointer nullString = mComboList->data;
			mComboList = g_list_remove(mComboList, mComboList->data);
			g_free(nullString);
		}

		/* Note that the localeTitle string is shared between the
		 * list and the hash table. This is fine because there will
		 * never be a case where the string should not be in both.
		 * It also saves on storage space and has has other advantages
		 */
		mComboList = g_list_append(mComboList, localeTitle);
		mComboListEmpty = PR_FALSE;
		g_hash_table_insert(mUrlTable, localeTitle,
				    g_strdup(aContentURL));

#define dock(x) (static_cast<SideBarDock *>(x->data))
		for(GList *i = mDocks ; i ; i = g_list_next(i))
		{
			if(dock(i)->mParent->dock_type != DOCK_SIDEBAR)
				dock(i)->ShowSidebar();
			gtk_combo_set_popdown_strings(GTK_COMBO(dock(i)->mComboBox),
						      mComboList);
		}
#undef dock

		//Load the new sidebar url into the last updated dock
		//This is almost certain to be the one attached to the
		//window in which the user clicked on the sidebar installtion
		//link. It's certainly better than updating all open docks.
		if(mLastDock && GTK_IS_ENTRY(mLastDock))
		{
			gtk_entry_set_text(GTK_ENTRY(mLastDock->mComboEntry),
					   localeTitle);
		}
	}

	return NS_OK;
}

/* void addPersistentPanel (in wstring aTitle, in string aContentURL, in string aCustomizeURL); */
NS_IMETHODIMP GSidebar::AddPersistentPanel(const PRUnichar *aTitle,
					   const char *aContentURL,
					   const char *aCustomizeURL)
{
	g_print("GSidebar::AddPersistentPanel\n");
	return NS_ERROR_NOT_IMPLEMENTED;
}

/* void addSearchEngine (in string engineURL, in string iconURL, in wstring suggestedTitle, in wstring suggestedCategory); */
NS_IMETHODIMP GSidebar::AddSearchEngine(const char *engineURL,
					const char *iconURL,
					const PRUnichar *suggestedTitle,
					const PRUnichar *suggestedCategory)
{
	g_print("GSidebar::AddSearchEngine\n");
	return NS_ERROR_NOT_IMPLEMENTED;
}

void GSidebar::RemoveDock(SideBarDock *aDock)
{
	if(mLastDock == aDock) mLastDock = nsnull;
	mDocks = g_list_remove(mDocks, aDock);
	delete aDock;
}

//------------------------------------------------------------------------------
//GSidebar public functions
//------------------------------------------------------------------------------

NS_METHOD GSidebar::SetWindow(GaleonWindow *aParent, PRBool aCalledByMoz)
{

	bool exists(false);
	mLastDock = nsnull;
	for(GList *i = mDocks ; i ; i = g_list_next(i))
		if(static_cast<SideBarDock *>(i->data)->mParent == aParent)
		{
			exists = true;
			mLastDock = static_cast<SideBarDock *>(i->data);
			break;
		}
	if(!exists && aCalledByMoz == PR_TRUE)
	{
		mLastDock = nsnull;
		return NS_ERROR_FAILURE;
	}

	if(!exists)
	{
		mLastDock = new SideBarDock(this, aParent);
		mDocks = g_list_prepend(mDocks, mLastDock);
	}

	if(mLastDock->mParent->dock_type != DOCK_SIDEBAR)
		mLastDock->ShowSidebar();

	return NS_OK;
}

void GSidebar::RemovePage(const char *desc)
{
	if(!desc || !strlen(desc)) return;

	gchar *url = static_cast<gchar *>(g_hash_table_lookup (mUrlTable,
					  desc));

	/* Here is one of the advantages of the shared string. We can
	 * use desc to lookup the url to free it but we can't free the
	 * localeTitle. It has the same contents as desc but is a physically
	 * different string. Getting a pointer to the original title from
	 * the hash table would be almost impossible, but we can get that
	 * pointer from the list, which is how we free it.
	 */
	if(url)// && strlen(url))
	{
		g_hash_table_remove(mUrlTable, desc);
		g_free(url);
	}

	for(GList *j = mComboList ; j ; j = g_list_next(j))
		if(g_strcasecmp(desc, static_cast<gchar *>(j->data)) == 0)
		{
			gpointer localTitle = j->data;
			mComboList = g_list_remove(mComboList, j->data);
			g_free(localTitle);
			break;
		}
	if(!g_list_length(mComboList))
	{
		mComboList = g_list_append(mComboList,g_strdup(""));
		mComboListEmpty = PR_TRUE;
	}

#define dock(x) (static_cast<SideBarDock *>(x->data))
		for(GList *i = mDocks ; i ; i = g_list_next(i))
		{
			gtk_combo_set_popdown_strings(GTK_COMBO(dock(i)->mComboBox),
						      mComboList);
			gtk_entry_set_text(GTK_ENTRY(dock(i)->mComboEntry),
					   "");
		}
#undef dock
}

//------------------------------------------------------------------------------
//GSidebar private functions
//------------------------------------------------------------------------------

void GSidebar::SaveSidebarPages(void)
{
	xmlNodePtr rootNode;
	xmlDocPtr doc;

	gchar *filename = g_concat_dir_and_file(g_get_home_dir(),
						".galeon/sidebars.xml");

	/* version doesn't really make sense, but... */
        doc = xmlNewDoc (reinterpret_cast<const xmlChar*>("1.0"));

	/* create and set the root node for the session */
        rootNode = xmlNewDocNode (doc, NULL, reinterpret_cast<const xmlChar*>("sidebars"), NULL);
        xmlDocSetRootElement (doc, rootNode);

	/* iterate through all the windows */
	pair<xmlDocPtr, xmlNodePtr> xmlPtrs(doc, rootNode);
	g_hash_table_foreach(mUrlTable, SaveHashTableEntry, &xmlPtrs);

	/* save it all out to disk */

        xmlSaveFile (filename, doc);
	xmlFreeDoc (doc);
	g_free (filename);
}

void GSidebar::LoadSidebarPages(void)
{
	xmlDocPtr doc;

	gchar *filename = g_concat_dir_and_file(g_get_home_dir(),
						".galeon/sidebars.xml");

	/* check the file exists */
	if (!(g_file_exists (filename)))
	{
		g_free (filename);
		return;
	}

	/* load the file */
	doc = xmlParseFile (filename);
	g_free (filename);
	if (doc == NULL) 
	{
		g_warning ("Unable to parse `%s', no sidebars loaded.",
			   filename);
		return;
	}

	/* iterate over sidebar pages in document */
	if(doc->root->childs) mComboListEmpty = PR_FALSE;
        for (xmlNodePtr page = doc->root->childs;
             page != NULL; page = page->next)
	{
		xmlChar *desc, *url;
		desc = xmlGetProp (page,
				   reinterpret_cast<const xmlChar*>("description"));
		url = xmlGetProp (page,
				  reinterpret_cast<const xmlChar*>("url"));

		g_hash_table_insert(mUrlTable,
				    g_strdup(reinterpret_cast<const char *>(desc)),
				    g_strdup(reinterpret_cast<const char *>(url)));

		mComboList = g_list_append(mComboList,
					   g_strdup(reinterpret_cast<const char *>(desc)));

		if (desc) xmlFree (desc);
		if (url) xmlFree (url);
	}

	xmlFreeDoc (doc);
}

//------------------------------------------------------------------------------
//SideBarDock Impl.
//------------------------------------------------------------------------------

SideBarDock::SideBarDock(GSidebar *aMozSidebar,
			 GaleonWindow *aParent) : mMozSidebar(aMozSidebar),
						  mParent(aParent),
						  mSidebar(nsnull),
						  mSidebarEmbed(nsnull),
						  mComboBox(nsnull),
						  mComboEntry(nsnull),
						  mCloseButton(nsnull),
						  mVBox(nsnull)
{
	GladeXML *gxml;

	/* widgets to be looked up */
	WidgetLookup lookup_table[] =
	{
		{ "sidebar_combo_box",    &mComboBox    },
		{ "sidebar_combo_entry",  &mComboEntry  },
		{ "sidebar_close_button", &mCloseButton },
		{ "sidebar_vbox",         &mVBox        },
		{ NULL, NULL } /* terminator, must be last */
	};

	/* build the widgets */
	gxml = glade_widget_new("galeon.glade", "mozilla_sidebar",
				&mSidebar, this);
        lookup_widgets(gxml, lookup_table);
        gtk_object_unref(GTK_OBJECT (gxml));

	gtk_button_set_relief(GTK_BUTTON(mCloseButton), GTK_RELIEF_NONE);

	mSidebarEmbed = embed_create_no_window();
	mSidebarEmbed->parent_window = mParent;

	gtk_signal_connect_while_alive (GTK_OBJECT(mSidebarEmbed->mozembed),
                                        "dom_mouse_click",
                                        GTK_SIGNAL_FUNC(sidebar_dom_mouse_click_cb),
					mSidebarEmbed,
					GTK_OBJECT(mSidebarEmbed->mozembed));

	gtk_box_pack_end_defaults(GTK_BOX(mVBox), mSidebarEmbed->mozembed);

	mMozSidebar->IncRefCount();
}

SideBarDock::~SideBarDock()
{
	mMozSidebar->DecRefCount();
}

void SideBarDock::ShowSidebar(void)
{
	gtk_combo_set_popdown_strings(GTK_COMBO(mComboBox),
				      mMozSidebar->mComboList);
	window_dock(mParent, mSidebar, CONF_STATE_SIDEBAR_DOCK_WIDTH);
	mParent->dock_type = DOCK_SIDEBAR;
	embed_set_visibility(mSidebarEmbed, TRUE);
}

void SideBarDock::LoadUrlFromDesc(gchar *desc)
{
	if(!desc || !strlen(desc)) return;
	gchar *url = static_cast<gchar *>(g_hash_table_lookup (mMozSidebar->mUrlTable,
					  desc));

	if(url && strlen(url))
		embed_load_url(mSidebarEmbed, url);		
}

//------------------------------------------------------------------------------
//Hash table free helper
//------------------------------------------------------------------------------

static inline
gboolean FreeSidebarHashTableEntry(gpointer key, gpointer val, gpointer data)
{
	g_free(key);
	g_free(val);
	return TRUE;
}

void SaveHashTableEntry(gpointer key, gpointer value, gpointer xmlData)
{
	xmlDocPtr doc = static_cast<pair<xmlDocPtr, xmlNodePtr>*>(xmlData)->first;
	xmlNodePtr rootNode = static_cast<pair<xmlDocPtr, xmlNodePtr>*>(xmlData)->second;

	/* make a new XML node */
	xmlNodePtr pageNode = xmlNewDocNode (doc, NULL,
					     reinterpret_cast<const xmlChar*>("page"),
					     NULL);

	/* fill out fields */
	xmlSetProp (pageNode, reinterpret_cast<const xmlChar*>("description"),
		    reinterpret_cast<xmlChar *>(key));
	xmlSetProp (pageNode, reinterpret_cast<const xmlChar*>("url"),
		    reinterpret_cast<xmlChar *>(value));

	xmlAddChild (rootNode, pageNode);
}

//------------------------------------------------------------------------------
//GTK Callbacks
//------------------------------------------------------------------------------

void sidebar_combo_entry_changed_cb (GtkEditable *entry,
				     SideBarDock *sbd)
{
	sbd->LoadUrlFromDesc(gtk_entry_get_text(GTK_ENTRY(entry)));
}

void sidebar_close_button_clicked_cb (GtkButton *button, 
				      SideBarDock *sbd)
{
	return_if_not_window(sbd->mParent);

	//Placing this code here avoids round-tripping through the
	//mozembed disconnect callback.
	gtk_widget_hide(sbd->mSidebarEmbed->mozembed);
	gtk_container_remove(GTK_CONTAINER(sbd->mVBox), sbd->mSidebarEmbed->mozembed);
	sbd->mParent->dock_type = DOCK_NONE;

	window_undock(sbd->mParent);
}

void sidebar_delete_current_page_cb (GtkButton *button,
				     SideBarDock *sbd)
{
	sbd->mMozSidebar->RemovePage(gtk_entry_get_text(GTK_ENTRY(sbd->mComboEntry)));
}

// See window.c - window_undock() for the rationale behind this
void sidebar_disconnect_mozembed_cb (GtkWidget *dock,
				     gchar *msg,
				     SideBarDock *sbd)
{
	if(g_strcasecmp(msg, "disconnect-mozembed") == 0)
	{
		gtk_widget_hide(sbd->mSidebarEmbed->mozembed);
		gtk_container_remove(GTK_CONTAINER(sbd->mVBox),
				     sbd->mSidebarEmbed->mozembed);
		sbd->mParent->dock_type = DOCK_NONE;
	}
}

void sidebar_destroy_cb (GtkWidget *dock,
			 SideBarDock *sbd)
{
	embed_close(sbd->mSidebarEmbed);
	sbd->mMozSidebar->RemoveDock(sbd);
}

gint sidebar_dom_mouse_click_cb (GtkMozEmbed *dummy, gpointer dom_event,
                                 GaleonEmbed *embed)
{
	GaleonWindow *window;
	WrapperMouseEventInfo *info;
	gboolean handled = FALSE;

	return_val_if_not_sane_embed (embed, FALSE);
	window = embed->parent_window;

	info = g_new0 (WrapperMouseEventInfo,1);
	if (!mozilla_get_mouse_event_info (embed, dom_event, info)) 
	{
		mozilla_free_context_info_sub(&info->ctx);
		g_free(info);
		mozilla_pop_target_document (embed);
		return FALSE;
	}

	if((info->button == 0) && (info->ctx.context & CONTEXT_LINK))
	{
		if(nsCRT::strcmp(info->ctx.linktarget, "_content") == 0)
		{
			embed_load_url(window->active_embed, info->ctx.link);
			handled = TRUE;
		}
		else if(nsCRT::strcmp(info->ctx.linktarget, "_self") == 0)
		{
			handled = FALSE;
		}
		else
		{
			gboolean inNewWindow;
			inNewWindow = !eel_gconf_get_boolean(CONF_TABS_TABBED);
			embed_create_after_embed(window->active_embed,
						 inNewWindow,
						 info->ctx.link,
						 EMBED_CREATE_FORCE_JUMP);
			handled = TRUE;
		}			       
	}
	return handled;
}
